/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.lemminx.uriresolver;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.io.MoreFiles;
import com.google.common.io.RecursiveDeleteOption;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URI;
import java.net.URL;
import java.net.URLConnection;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.eclipse.lemminx.uriresolver.CacheResourceDownloadedException;
import org.eclipse.lemminx.uriresolver.CacheResourceDownloadingException;
import org.eclipse.lemminx.uriresolver.InvalidURIException;
import org.eclipse.lemminx.utils.ExceptionUtils;
import org.eclipse.lemminx.utils.FilesUtils;
import org.eclipse.lemminx.utils.StringUtils;
import org.eclipse.lemminx.utils.platform.Platform;

public class CacheResourcesManager {
    private static final String USER_AGENT_KEY = "User-Agent";
    private static final String USER_AGENT_VALUE = "LemMinX/" + Platform.getVersion().getVersionNumber() + " (" + Platform.getOS().getName() + " " + Platform.getOS().getVersion() + ")";
    private final Cache<String, CacheResourceDownloadedException> unavailableURICache;
    private final Cache<String, Boolean> forceDownloadExternalResources;
    private static final String CACHE_PATH = "cache";
    private static final Logger LOGGER = Logger.getLogger(CacheResourcesManager.class.getName());
    private static final Path TEMP_DOWNLOAD_DIR;
    private final Map<String, CompletableFuture<Path>> resourcesLoading = new HashMap<String, CompletableFuture<Path>>();
    private boolean useCache;
    private boolean downloadExternalResources;
    private final Set<String> protocolsForCache = new HashSet<String>();

    public CacheResourcesManager() {
        this(CacheBuilder.newBuilder().maximumSize(100L).expireAfterWrite(30L, TimeUnit.SECONDS).build());
    }

    CacheResourcesManager(Cache<String, CacheResourceDownloadedException> cache) {
        this.unavailableURICache = cache;
        this.forceDownloadExternalResources = CacheBuilder.newBuilder().maximumSize(100L).expireAfterWrite(30L, TimeUnit.SECONDS).build();
        this.addDefaultProtocolsForCache();
        this.setDownloadExternalResources(true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Path getResource(String resourceURI) throws IOException {
        Path resourceCachePath = CacheResourcesManager.getResourceCachePath(resourceURI);
        if (Files.exists(resourceCachePath, new LinkOption[0])) {
            return resourceCachePath;
        }
        if (!this.isDownloadExternalResources() && !this.isForceDownloadExternalResource(resourceURI)) {
            throw new CacheResourceDownloadingException(resourceURI, resourceCachePath, CacheResourceDownloadingException.CacheResourceDownloadingError.DOWNLOAD_DISABLED, null, null);
        }
        if (!FilesUtils.isIncludedInDeployedPath(resourceCachePath)) {
            throw new CacheResourceDownloadingException(resourceURI, resourceCachePath, CacheResourceDownloadingException.CacheResourceDownloadingError.RESOURCE_NOT_IN_DEPLOYED_PATH, null, null);
        }
        CacheResourceDownloadedException cacheException = this.unavailableURICache.getIfPresent(resourceURI);
        if (cacheException != null) {
            throw cacheException;
        }
        CompletableFuture<Path> f = null;
        Map<String, CompletableFuture<Path>> map = this.resourcesLoading;
        synchronized (map) {
            if (this.resourcesLoading.containsKey(resourceURI)) {
                CompletableFuture<Path> future = this.resourcesLoading.get(resourceURI);
                throw new CacheResourceDownloadingException(resourceURI, resourceCachePath, CacheResourceDownloadingException.CacheResourceDownloadingError.RESOURCE_LOADING, future, null);
            }
            f = this.downloadResource(resourceURI, resourceCachePath);
            this.resourcesLoading.put(resourceURI, f);
        }
        if (f.getNow(null) == null) {
            throw new CacheResourceDownloadingException(resourceURI, resourceCachePath, CacheResourceDownloadingException.CacheResourceDownloadingError.RESOURCE_LOADING, f, null);
        }
        return resourceCachePath;
    }

    private CompletableFuture<Path> downloadResource(String resourceURI, Path resourceCachePath) {
        return CompletableFuture.supplyAsync(() -> {
            long start = System.currentTimeMillis();
            URLConnection conn = null;
            try {
                Object actualURI = resourceURI;
                URL url = new URL((String)actualURI);
                String originalProtocol = url.getProtocol();
                if (!this.protocolsForCache.contains(CacheResourcesManager.formatProtocol(originalProtocol))) {
                    throw new InvalidURIException(resourceURI, InvalidURIException.InvalidURIError.UNSUPPORTED_PROTOCOL, originalProtocol);
                }
                boolean isOriginalRequestSecure = this.isSecure(originalProtocol);
                LOGGER.info("Downloading " + resourceURI + " to " + resourceCachePath + "...");
                conn = url.openConnection();
                conn.setRequestProperty(USER_AGENT_KEY, USER_AGENT_VALUE);
                for (int allowedRedirects = 5; conn.getHeaderField("Location") != null && allowedRedirects > 0; --allowedRedirects) {
                    actualURI = conn.getHeaderField("Location");
                    url = new URL((String)actualURI);
                    String protocol = url.getProtocol();
                    if (!this.protocolsForCache.contains(CacheResourcesManager.formatProtocol(protocol))) {
                        throw new InvalidURIException(url.toString(), InvalidURIException.InvalidURIError.UNSUPPORTED_PROTOCOL, protocol);
                    }
                    if (isOriginalRequestSecure && !this.isSecure(protocol)) {
                        throw new InvalidURIException(resourceURI, InvalidURIException.InvalidURIError.INSECURE_REDIRECTION, url.toString());
                    }
                    conn = url.openConnection();
                    conn.setRequestProperty(USER_AGENT_KEY, USER_AGENT_VALUE);
                }
                Path path = Files.createTempFile(TEMP_DOWNLOAD_DIR, resourceCachePath.getFileName().toString(), ".lemminx", new FileAttribute[0]);
                try (ReadableByteChannel rbc = Channels.newChannel(conn.getInputStream());
                     FileOutputStream fos = new FileOutputStream(path.toFile());){
                    fos.getChannel().transferFrom(rbc, 0L, Long.MAX_VALUE);
                }
                Path dir = resourceCachePath.getParent();
                if (!Files.exists(dir, new LinkOption[0])) {
                    Files.createDirectories(dir, new FileAttribute[0]);
                }
                Files.move(path, resourceCachePath, new CopyOption[0]);
                long elapsed = System.currentTimeMillis() - start;
                LOGGER.info("Downloaded " + resourceURI + " to " + resourceCachePath + " in " + elapsed + "ms");
            }
            catch (Exception e) {
                Throwable rootCause = ExceptionUtils.getRootCause(e);
                String error = "[" + rootCause.getClass().getTypeName() + "] " + rootCause.getMessage();
                LOGGER.log(Level.SEVERE, "Error while downloading " + resourceURI + " to " + resourceCachePath + " : " + error);
                String httpResponseCode = CacheResourcesManager.getHttpResponseCode(conn);
                if (httpResponseCode != null) {
                    error = error + " with code: " + httpResponseCode;
                }
                CacheResourceDownloadedException cacheException = new CacheResourceDownloadedException(resourceURI, resourceCachePath, error, e);
                this.unavailableURICache.put(resourceURI, cacheException);
                throw cacheException;
            }
            finally {
                Map<String, CompletableFuture<Path>> map = this.resourcesLoading;
                synchronized (map) {
                    this.resourcesLoading.remove(resourceURI);
                }
                if (conn != null && conn instanceof HttpURLConnection) {
                    ((HttpURLConnection)conn).disconnect();
                }
            }
            return resourceCachePath;
        });
    }

    private static String getHttpResponseCode(URLConnection conn) {
        if (conn != null && conn instanceof HttpURLConnection) {
            try {
                HttpURLConnection httpConn = (HttpURLConnection)conn;
                return String.valueOf(httpConn.getResponseCode()) + " " + httpConn.getResponseMessage();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        return null;
    }

    private boolean isSecure(String protocol) {
        return "https".equals(protocol);
    }

    public static Path getResourceCachePath(String resourceURI) throws IOException {
        URI uri = null;
        try {
            uri = URI.create(resourceURI);
        }
        catch (Exception e) {
            throw new InvalidURIException(resourceURI, InvalidURIException.InvalidURIError.ILLEGAL_SYNTAX, (Throwable)e);
        }
        return CacheResourcesManager.getResourceCachePath(uri);
    }

    public static Path getResourceCachePath(URI uri) throws IOException {
        URI normalizedUri = uri.normalize();
        if (normalizedUri.getPath().contains("/../")) {
            throw new InvalidURIException(uri.toString(), InvalidURIException.InvalidURIError.INVALID_PATH, new String[0]);
        }
        Path resourceCachePath = normalizedUri.getPort() > 0 ? Paths.get(CACHE_PATH, normalizedUri.getScheme(), normalizedUri.getHost(), String.valueOf(normalizedUri.getPort()), normalizedUri.getPath()) : Paths.get(CACHE_PATH, normalizedUri.getScheme(), normalizedUri.getHost(), normalizedUri.getPath());
        return FilesUtils.getDeployedPath(resourceCachePath);
    }

    public static Path getResourceCachePath(ResourceToDeploy resource) throws IOException {
        Path outFile = resource.getDeployedPath();
        if (!outFile.toFile().exists()) {
            try (InputStream in = CacheResourcesManager.class.getResourceAsStream(resource.getResourceFromClasspath());){
                FilesUtils.saveToFile(in, outFile);
            }
        }
        return outFile;
    }

    public boolean canUseCache(String url) {
        return this.isUseCache() && this.isUseCacheFor(url);
    }

    public void setUseCache(boolean useCache) {
        this.useCache = useCache;
    }

    public boolean isUseCache() {
        return this.useCache;
    }

    public boolean isDownloadExternalResources() {
        return this.downloadExternalResources;
    }

    public void setDownloadExternalResources(boolean downloadExternalResources) {
        this.downloadExternalResources = downloadExternalResources;
    }

    public void evictCache() throws IOException {
        Path cachePath = FilesUtils.getDeployedPath(Paths.get(CACHE_PATH, new String[0]));
        if (Files.exists(cachePath, new LinkOption[0])) {
            MoreFiles.deleteDirectoryContents(cachePath, RecursiveDeleteOption.ALLOW_INSECURE);
        }
    }

    public void addProtocolForCache(String protocol) {
        this.protocolsForCache.add(CacheResourcesManager.formatProtocol(protocol));
    }

    public void removeProtocolForCache(String protocol) {
        this.protocolsForCache.remove(CacheResourcesManager.formatProtocol(protocol));
    }

    private static String formatProtocol(String protocol) {
        if (!protocol.endsWith(":")) {
            return protocol + ":";
        }
        return protocol;
    }

    private boolean isUseCacheFor(String url) {
        if (StringUtils.isEmpty(url)) {
            return false;
        }
        for (String protocol : this.protocolsForCache) {
            if (!url.startsWith(protocol)) continue;
            return true;
        }
        return false;
    }

    private void addDefaultProtocolsForCache() {
        this.addProtocolForCache("http");
        this.addProtocolForCache("https");
        this.addProtocolForCache("ftp");
    }

    public void forceDownloadExternalResource(String url) {
        this.forceDownloadExternalResources.put(url, Boolean.TRUE);
    }

    private boolean isForceDownloadExternalResource(String url) {
        return this.forceDownloadExternalResources.getIfPresent(url) != null;
    }

    static {
        Path tempDownloadDir = null;
        try {
            tempDownloadDir = Files.createTempDirectory("lemminx-temp", new FileAttribute[0]);
        }
        catch (Exception exception) {
            // empty catch block
        }
        TEMP_DOWNLOAD_DIR = tempDownloadDir;
    }

    public static class ResourceToDeploy {
        private final Path resourceCachePath;
        private final String resourceFromClasspath;

        public ResourceToDeploy(String resourceURI, String resourceFromClasspath) {
            this(URI.create(resourceURI), resourceFromClasspath);
        }

        public ResourceToDeploy(URI resourceURI, String resourceFromClasspath) {
            this.resourceCachePath = Paths.get(CacheResourcesManager.CACHE_PATH, resourceURI.getScheme(), resourceURI.getHost(), resourceURI.getPath());
            this.resourceFromClasspath = resourceFromClasspath.startsWith("/") ? resourceFromClasspath : "/" + resourceFromClasspath;
        }

        public Path getDeployedPath() throws IOException {
            return FilesUtils.getDeployedPath(this.resourceCachePath);
        }

        public String getResourceFromClasspath() {
            return this.resourceFromClasspath;
        }
    }

    class ResourceInfo {
        String resourceURI;
        CompletableFuture<Path> future;

        ResourceInfo() {
        }
    }
}

