;;; GNUTLS-EXTRA --- Guile bindings for GnuTLS-EXTRA.
;;; Copyright (C) 2007  Free Software Foundation
;;;
;;; GNUTLS-EXTRA is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 3 of the License, or
;;; (at your option) any later version.
;;;
;;; GNUTLS-EXTRA is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNUTLS-EXTRA; if not, write to the Free Software
;;; Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301,
;;; USA.

;;; Written by Ludovic Courts <ludo@chbouib.org>.


;;;
;;; Test session establishment using OpenPGP certificate authentication.
;;;

(use-modules (gnutls)
             (gnutls extra)
             (srfi srfi-4))


;; TLS session settings.
(define %protos  (list protocol/tls-1.0))
(define %certs   (list certificate-type/openpgp))
(define %ciphers (list cipher/null cipher/arcfour cipher/aes-128-cbc
                       cipher/aes-256-cbc))
(define %kx      (list kx/rsa kx/rsa-export kx/dhe-dss kx/dhe-dss))
(define %macs    (list mac/sha1 mac/rmd160 mac/md5))

;; Message sent by the client.
(define %message
  (cons "hello, world!" (iota 4444)))

(define (import-something import-proc file fmt)
  (let* ((path (search-path %load-path file))
         (size (stat:size (stat path)))
         (raw  (make-u8vector size)))
    (uniform-vector-read! raw (open-input-file path))
    (import-proc raw fmt)))

(define (import-key import-proc file)
  (import-something import-proc file openpgp-certificate-format/base64))

(define (import-rsa-params file)
  (import-something pkcs1-import-rsa-parameters file
                    x509-certificate-format/pem))

;; Debugging.
;; (set-log-level! 3)
;; (set-log-procedure! (lambda (level str)
;;                       (format #t "[~a|~a] ~a" (getpid) level str)))

(dynamic-wind
    (lambda ()
      #t)

    (lambda ()
      (let ((socket-pair (socketpair PF_UNIX SOCK_STREAM 0))
            (pub         (import-key import-openpgp-certificate
                                     "openpgp-pub.asc"))
            (sec         (import-key import-openpgp-private-key
                                     "openpgp-sec.asc")))
        (let ((pid (primitive-fork)))
          (if (= 0 pid)

              (let ((client (make-session connection-end/client))
                    (cred   (make-certificate-credentials)))
                ;; client-side (child process)
                (set-session-default-priority! client)
                (set-session-certificate-type-priority! client %certs)
                (set-session-kx-priority! client %kx)
                (set-session-protocol-priority! client %protos)
                (set-session-cipher-priority! client %ciphers)
                (set-session-mac-priority! client %macs)

                (set-certificate-credentials-openpgp-keys! cred pub sec)
                (set-session-credentials! client cred)
                (set-session-dh-prime-bits! client 1024)

                (set-session-transport-fd! client (fileno (car socket-pair)))

                (handshake client)
                (write %message (session-record-port client))
                (bye client close-request/rdwr)

                (exit))

              (let ((server (make-session connection-end/server))
                    (rsa    (import-rsa-params "rsa-parameters.pem"))
                    (dh     (make-dh-parameters 1024)))
                ;; server-side
                (set-session-default-priority! server)
                (set-session-certificate-type-priority! server %certs)
                (set-session-kx-priority! server %kx)
                (set-session-protocol-priority! server %protos)
                (set-session-cipher-priority! server %ciphers)
                (set-session-mac-priority! server %macs)
                (set-server-session-certificate-request! server
                         certificate-request/require)

                (set-session-transport-fd! server (fileno (cdr socket-pair)))
                (let ((cred (make-certificate-credentials)))
                  (set-certificate-credentials-dh-parameters! cred dh)
                  (set-certificate-credentials-rsa-export-parameters! cred rsa)
                  (set-certificate-credentials-openpgp-keys! cred pub sec)
                  (set-session-credentials! server cred))
                (set-session-dh-prime-bits! server 1024)

                (handshake server)
                (let ((msg (read (session-record-port server)))
                      (auth-type (session-authentication-type server)))
                  (bye server close-request/rdwr)
                  (exit (and (eq? auth-type credentials/certificate)
                             (equal? msg %message)))))))))

    (lambda ()
      ;; failure
      (exit 1)))

;;; arch-tag: 1a973ed5-f45d-45a4-8160-900b6a8c27ff
