from DataTarget import DataTarget
from utils.datatypes import *

import gtk



#
# Abstract class for DisplayTargets.
#
class DisplayTarget(gtk.HBox, DataTarget):

    # observer actions
    OBS_MOVE = 0    


    # user actions
    ACTION_CLICK = 0
    ACTION_DOUBLECLICK = 1
    ACTION_PRESS = 2
    ACTION_RELEASE = 3
    ACTION_MENU = 4
    ACTION_SCROLL = 5
    ACTION_ENTER = 6
    ACTION_LEAVE = 7
    ACTION_MOTION = 8

    # placement anchors
    ANCHOR_NW = "nw"
    ANCHOR_N = "n"
    ANCHOR_NE = "ne"
    ANCHOR_E = "e"
    ANCHOR_SE = "se"
    ANCHOR_S = "s"
    ANCHOR_SW = "sw"
    ANCHOR_W = "w"
    ANCHOR_CENTER = "center"


    def __init__(self, parent, display):

        # coordinates and size of this widget
        self.__geometry = (0, 0, 0, 0)

        # relative settings (is_rel_x, is_rel_y)
        self.__is_relative = (0, 0)

        # relative coordinates of this widget 
        self.__relative_coords = (0, 0)

        # the widget to which this one is positioned relatively
        self.__relative = None

        # the placement anchor
        self.__anchor = self.ANCHOR_NW

        # the requested size or -1
        self.__requested_size = (-1, -1)

        # the size in percents or 0
        self.__percent_size = (0, 0)


        DataTarget.__init__(self, parent, display)
        gtk.HBox.__init__(self)
        self.connect("expose-event", self.__on_expose)

        self._set_property_type("x", TYPE_INT)
        self._set_property_type("y", TYPE_INT)
        self._set_property_type("width", TYPE_SIZE)
        self._set_property_type("height", TYPE_SIZE)
        self._set_property_type("relative-to", TYPE_LIST)

        self._set_property_type("visible", TYPE_BOOL)
        
        self._set_property_type("on-enter", TYPE_LIST)
        self._set_property_type("on-leave", TYPE_LIST)
        self._set_property_type("on-click", TYPE_LIST)
        self._set_property_type("on-press", TYPE_LIST)
        self._set_property_type("on-release", TYPE_LIST)
        self._set_property_type("on-doublelick", TYPE_LIST)
        self._set_property_type("on-motion", TYPE_LIST)
        self._set_property_type("on-menu", TYPE_LIST)

        # disabled for now
        #if (parent):
        #    parent.add_observer(self.__on_observe_parent)
        
        self.show()



    #
    # Reacts on expose events of this widget.
    #
    def __on_expose(self, src, event):

        x, y, ow, oh = self.__geometry
        w, h = src.size_request()

        if ((w, h) != (ow, oh)):
            self.__geometry = (x, y, w, h)
            self.update_observer(self.OBS_MOVE, x, y, w, h)
        #end if





    #
    # Reacts on notifications by the relative of this widget.
    #
    def __on_observe_relative(self, src, cmd, *args):

        if (cmd == src.OBS_MOVE):
            x, y = self.__relative_coords
            self.__relative = src
            self.set_position(x, y)



    #
    # Reacts on notifications by the parent.
    # TODO: percentual sized targets must not block the size of their parents
    #
    def __on_observe_parent(self, src, cmd, *args):

        if (cmd == src.OBS_MOVE):
            x, y, w, h = args
            pw, ph = self.__percent_size
            if (pw == ph == 0): return

            cx, cy, width, height = self.get_geometry()
            rw, rh = self.__requested_size
            new_width, new_height = rw, rh
            
            if (pw > 0):
                new_width = int(w * (pw / 100.0))
            if (ph > 0):
                new_height = int(h * (ph / 100.0))

            # prevent the child from growing over its parent
            if (w > cx and cx + new_width > w): new_width = w - cx
            if (h > cy and cy + new_height > h): new_height = h - cy
            
            if ((width, height) != (new_width, new_height)):
                self.set_size_request(new_width, new_height)



    def set_config(self, key, value):

        if (key == "x"):
            nil, y, w, h = self.get_geometry()
            rx, ry = self.__relative_coords
            self.__relative_coords = (value, ry)
            self.set_position(value, y)

        elif (key == "y"):
            x, nil, w, h = self.get_geometry()
            rx, ry = self.__relative_coords
            self.__relative_coords = (rx, value)
            self.set_position(x, value)

        elif (key == "width"):
            rw, rh = self.__requested_size
            pw, ph = self.__percent_size
            if (value < 0):
                self.__percent_size = (-value, ph)
                self.__requested_size = (-1, rh)
            else:
                self.__percent_size = (0, ph)
                self.__requested_size = (value, rh)
            rw, rh = self.__requested_size
            self.set_size_request(rw, rh)
            #self.update_observer(self.OBS_MOVE, x, y, value, height)

        elif (key == "height"):
            rw, rh = self.__requested_size
            pw, ph = self.__percent_size
            if (value < 0):
                self.__percent_size = (pw, -value)
                self.__requested_size = (rw, -1)
            else:
                self.__percent_size = (pw, 0)
                self.__requested_size = (rw, value)
            rw, rh = self.__requested_size
            self.set_size_request(rw, rh)


        elif (key == "relative-to"):
            name, mode = value
            if (mode == "x"): self.__is_relative = (1, 0)
            elif (mode == "y"): self.__is_relative = (0, 1)
            elif (mode == "xy"): self.__is_relative = (1, 1)

            # no longer watch the old relative
            if (self.__relative):
                self.__relative.remove_observer(self.__on_observe_relative)

            target = self._get_parent().get_child_by_id(name)
            target.add_observer(self.__on_observe_relative)

        elif (key == "anchor"):
            x, y, w, h = self.get_geometry()
            self.__anchor = value
            self.set_position(x, y)
            

        elif (key == "visible"):
            if (value): self.show()
            else: self.hide()


        elif (key == "on-enter"):
            self.set_action_call(self.ACTION_ENTER, value)
        elif (key == "on-leave"):
            self.set_action_call(self.ACTION_LEAVE, value)
        elif (key == "on-click"):
            self.set_action_call(self.ACTION_CLICK, value)
        elif (key == "on-press"):
            self.set_action_call(self.ACTION_PRESS, value)
        elif (key == "on-release"):
            self.set_action_call(self.ACTION_RELEASE, value)
        elif (key == "on-doubleclick"):
            self.set_action_call(self.ACTION_DOUBLECLICK, value)
        elif (key == "on-motion"):
            self.set_action_call(self.ACTION_MOTION, value)
        elif (key == "on-menu"):
            self.set_action_call(self.ACTION_MENU, value)
            
        else:
            DataTarget.set_config(self, key, value)



    #
    # Returns the true coordinates of this target when the given coordinates
    # are the hotspot.
    #
    def get_anchored_coords(self, x, y, w, h):

        anchor = self.__anchor
        if (anchor in [self.ANCHOR_NW, self.ANCHOR_W, self.ANCHOR_SW]):
            ax = x
        elif (anchor in [self.ANCHOR_N, self.ANCHOR_CENTER, self.ANCHOR_S]):
            ax = x - (w / 2)
        else:
            ax = x - w

        if (anchor in [self.ANCHOR_NW, self.ANCHOR_N, self.ANCHOR_NE]):
            ay = y
        elif (anchor in [self.ANCHOR_W, self.ANCHOR_CENTER, self.ANCHOR_E]):
            ay = y - (h / 2)
        else:
            ay = y - h

        return (ax, ay)
        
    


    #
    # Returns the target ID to which this target is placed relatively.
    #
    def get_relative_to(self):

        return self.__relative



    #
    # Sets the position of this target. The coordinates are not anchored.
    #
    def set_position(self, x, y, update = 1):

        # handle relative positioning
        if (self.__relative):
            is_rel_x, is_rel_y = self.__is_relative
            tx, ty, tw, th = self.__relative.get_geometry()
            atx, aty = self.__relative.get_anchored_coords(tx, ty, tw, th)
            x += atx
            y += aty
            if (is_rel_x): x += tw
            if (is_rel_y): y += th
        #end if

        nil, nil, w, h = self.__geometry
        self.__geometry = (x, y, w, h)
        if (update):
            self.update_observer(self.OBS_MOVE, x, y, w, h)



    #
    # Returns the geometry (coordinates and size) of this target.
    #
    def get_geometry(self):

        x, y, nil, nil = self.__geometry
        w, h = self.size_request()
        return (x, y, w, h)




    #
    # Returns the target at the given position and its path.
    #
    def get_target_at(self, px, py):

        x, y, w, h = self.get_geometry()
        if (x <= px <= x + w and y <= py <= y + h):
            return ([self], [])
        else:
            return ([], [])
