from ContainerTarget import ContainerTarget
from utils.Hash2D import Hash2D
from utils.datatypes import *

import gtk
import os



#
# Class for grouping different targets together.
#
class TargetGroup(ContainerTarget):

    def __init__(self, parent, display):

        # the area zones covered by the child targets
        self.__zones = Hash2D()

        # the background pixmap
        self.__pixbuf = None

        # flag for ensuring that only one collapse action takes place
        self.__block = 0
        
        
        ContainerTarget.__init__(self, parent, display)
        self.__layout = gtk.Fixed()
        self.__layout.show()

        self.add(self.__layout)

        self.__image = gtk.Image()
        self.__image.show()
        self.__layout.put(self.__image, 0, 0)

        self._set_property_type("bg-uri", TYPE_STRING)
        self._set_property_type("bg-color", TYPE_STRING)

        self.add_observer(self.__on_observe_size)




    def __on_observe_size(self, src, cmd, *data):

        if (cmd == src.OBS_MOVE):
            x, y, w, h = data
            self.__tile_background(w, h)



    def add_children(self, childrendata):

        ContainerTarget.add_children(self, childrendata)
        for child in self._get_children():
            self.__layout.put(child, 0, 0)
            self.__zones.set(0, 0, 0, 0, child)
            child.add_observer(self.__on_observe_target)
        #end for

        self._init_children()



    def __on_observe_target(self, src, cmd, *args):

        if (cmd == src.OBS_MOVE):
            x, y, w, h = args
            x, y = src.get_anchored_coords(x, y, w, h)
            self.__layout.move(src, x, y)
            self.__zones.set(x, y, x + w, y + h, src)

            # give the group a chance to shrink
            self.__collapse_bg()



    #
    # Makes the background collapse so that the group can shrink.
    #
    def __collapse_bg(self):

        if (self.__block): return
        self.__block = 1
        self.__image.hide()
        gtk.idle_add(self.__image.show)

        def x(self): self.__block = 0
        gtk.timeout_add(500, x, self)



    def __hex2dec(self, n):

        d = {"a": 10, "b": 11, "c": 12, "d": 13, "e": 14, "f": 15}
        value = 0
        factor = 16 * (len(n) - 1)
        for i in xrange(len(n)):
            digit = n[i].lower()
            v = d.get(digit) or int(digit)
            value += v * factor
            factor /= 16
        #end for

        return value


    #
    # Sets the background color.
    #
    def __set_color(self, color):

        if (color[0] == "#" and len(color) == 9):
            alpha = self.__hex2dec(color[-2:])
            color = color[:-2]
        else:
            alpha = 255

        self.__pixbuf = gtk.gdk.Pixbuf(0, 1, 8, 20, 20)
        c = gtk.gdk.color_parse(color)
        fillr = (c.red / 256) << 24
        fillg = (c.green / 256) << 16
        fillb = (c.blue / 256) << 8
        fillcolor = fillr | fillg | fillb | alpha
        self.__pixbuf.fill(fillcolor)
        nil, nil, w, h = self.get_geometry()
        self.__tile_background(w, h)


    #
    # Sets the background image.
    #
    def __set_background(self, uri):

        uri = os.path.join(self.get_path(), uri)
        self.__pixbuf = gtk.gdk.pixbuf_new_from_file(uri)
        nil, nil, w, h = self.get_geometry()
        self.__tile_background(w, h)



    #
    # Tiles the background image over the visible area.
    #
    def __tile_background(self, width = 0, height = 0):

        x, y, w, h = self.get_geometry()
        if (not width): width = w
        if (not height): height = h

        if (not self.__pixbuf): return
        w = self.__pixbuf.get_width()
        h = self.__pixbuf.get_height()
        width = max(width, 1)#w)
        height = max(height, 1)#h)

        bg = gtk.gdk.Pixbuf(0, 1, 8, width, height)
        for x in xrange(0, width, w):
            for y in xrange(0, height, h):
                pw = min(w, width - x)
                ph = min(h, height - y)
                self.__pixbuf.copy_area(0, 0, pw, ph, bg, x, y)
            #end for
        #end for

        self.__image.set_from_pixbuf(bg)
        self.__image.set_size_request(width, height)
        #self.__image.set_size_request(-1, -1)



    def set_config(self, key, value):

        value = self._convert_type(key, value)

        if (key == "bg-uri"):
            self.__set_background(value)
            
        elif (key == "bg-color"):
            self.__set_color(value)

        else:
            ContainerTarget.set_config(self, key, value)



    def get_target_at(self, px, py, is_root = 0):

        x, y, w, h = self.get_geometry()

        if (is_root): x, y = 0, 0
        if (x <= px <= x + w and y <= py <= y + h):
            targets = self.__zones.get(px - x, py - y)
            ctargets, cpath = [], []
            
            children = self._get_children()
            children.reverse()
            for t in children:
                if (t in targets):
                    ctargets, cpath = t.get_target_at(px - x , py - y)
                    if (ctargets): break
            #end for

            return ([self] + ctargets, cpath)

        else:
            return ([], [])
        
        #end if
