/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */
package org.gnu.gtk;

import java.util.Vector;

import org.gnu.glib.EventMap;
import org.gnu.glib.GObject;
import org.gnu.gtk.event.EntryCompletionEvent;
import org.gnu.gtk.event.EntryCompletionListener;
import org.gnu.gtk.event.GtkEventType;

/**
 */
public class EntryCompletion extends GObject {

	public EntryCompletion() {
		if (-1 == handle) {
			handle = gtk_entry_completion_new();
			initializeEventHandlers();
		}
	}
	
	public EntryCompletion(int nativeHandle) {
		handle = nativeHandle;
	}
	
	public Entry getEntry() {
		return new Entry(gtk_entry_completion_get_entry(handle));
	}
	
	public void setModel(TreeModel model) {
		gtk_entry_completion_set_model(handle, model.getHandle());
	}
	
	public TreeModel getModel() {
		return new TreeModel(gtk_entry_completion_get_model(handle));
	}
	
	public void setMinimumKeyLength(int length) {
		gtk_entry_completion_set_minimum_key_length(handle, length);
	}
	
	public int getMinimumKeyLength() {
		return gtk_entry_completion_get_minimum_key_length(handle);
	}
	
	public void complete() {
		gtk_entry_completion_complete(handle);
	}
	
	public void insertActionText(int index, String text) {
		gtk_entry_completion_insert_action_text(handle, index, text);
	}
	
	public void insertActionMarkup(int index, String markup) {
		gtk_entry_completion_insert_action_markup(handle, index, markup);
	}
	
	public void deleteAction(int index) {
		gtk_entry_completion_delete_action(handle, index);
	}
	
	public void setTextColumn(int column) {
		gtk_entry_completion_set_text_column(handle, column);
	}
	
	/* **************************************
	 * EVENT LISTENERS
	 ****************************************/

	/**
	 * Listeners for handling dialog events
	 */
	private Vector ecListeners = null;

	/**
	 * Register an object to handle dialog events.
	 * @see EntryCompletionListener
	 */
	public void addListener(EntryCompletionListener listener) {
		// Don't add the listener a second time if it is in the Vector.
		int i = findListener(ecListeners, listener);
		if (i == -1) {
			if (null == ecListeners) {
				ecListeners = new Vector();
			}
			ecListeners.addElement(listener);
		}
	}
	
	/**
	 * Removes a listener
	 * @see #addListener(EntryCompletionListener)
	 */
	public void removeListener(EntryCompletionListener listener) {
		int i = findListener(ecListeners, listener);
		if (i > -1) {
			ecListeners.remove(i);
		}
		if (0 == ecListeners.size()) {
			ecListeners = null;
		}
	}

	protected void fireEntrySelectionEvent(EntryCompletionEvent event) {
		if (null == ecListeners) {
			return;
		}
		int size = ecListeners.size();
		int i = 0;
		while (i < size) {
			EntryCompletionListener ecl = (EntryCompletionListener)ecListeners.elementAt(i);
			ecl.entryCompletionEvent(event);
			i++;
		}
	}

	private void handleMatchSelected(int model, int iter) {
		EntryCompletionEvent evt = new EntryCompletionEvent(this, EntryCompletionEvent.Type.MATCH_SELECTED);
		evt.setModel(new TreeModel(model));
		evt.setIter(new TreeIter(iter));
		fireEntrySelectionEvent(evt);
	}

	private void handleActionActivated(int index) {
		EntryCompletionEvent evt = new EntryCompletionEvent(this, EntryCompletionEvent.Type.ACTION_ACTIVATED);
		evt.setIndex(index);
		fireEntrySelectionEvent(evt);
	}

	protected void initializeEventHandlers() {
		evtMap.initialize(this);
	}

	public Class getEventListenerClass(String signal) {
		return evtMap.getListenerClass(signal);
	}

	public GtkEventType getEventType(String signal) {
		return evtMap.getEventType(signal);
	}

	private static EventMap evtMap = new EventMap();
	static {
		addEvents(evtMap);
	}

	/**
	 * Implementation method to build an EventMap for this widget class.
	 * Not useful (or supported) for application use.
	 */
	private static void addEvents(EventMap anEvtMap) {
		anEvtMap.addEvent("match_selected", "handleMatchSelected", EntryCompletionEvent.Type.MATCH_SELECTED, EntryCompletionListener.class);
		anEvtMap.addEvent("action_activated", "handleActionActivated", EntryCompletionEvent.Type.ACTION_ACTIVATED, EntryCompletionListener.class);
	}

	/**
	 * Give us a way to locate a specific listener in a Vector.
	 * @param list The Vector of listeners to search.
	 * @param listener The object that is to be located in the Vector.
	 * @return Returns the index of the listener in the Vector, or -1 if
	 *                 the listener is not contained in the Vector.
	 */
	protected int findListener(Vector list, Object listener) {
		if (null == list || null == listener)
			return -1;
		return list.indexOf(listener);
	}

	/****************************************
	 * BEGINNING OF JNI CODE
	 ****************************************/
	native static final protected int gtk_entry_completion_get_type ();
	native static final protected int gtk_entry_completion_new();
	native static final protected int gtk_entry_completion_get_entry(int completion);
	native static final protected void gtk_entry_completion_set_model(int completion, int model);
	native static final protected int gtk_entry_completion_get_model(int completion);
	native static final protected void gtk_entry_completion_set_minimum_key_length(int completion, int length);
	native static final protected int gtk_entry_completion_get_minimum_key_length(int completion);
	native static final protected void gtk_entry_completion_complete(int completion);
	native static final protected void gtk_entry_completion_insert_action_text(int completion, int index, String text);
	native static final protected void gtk_entry_completion_insert_action_markup(int completion, int index, String markup);
	native static final protected void gtk_entry_completion_delete_action(int completion, int index);
	native static final protected void gtk_entry_completion_set_text_column(int completion,  int column);
	/****************************************
	 * END OF JNI CODE
	 ****************************************/
}
