/*
 * Copyright 2001 Sun Microsystems Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include "htmlboxaccessible.h"
#include "a11y/htmlviewaccessible.h"
#include <libgtkhtml/layout/htmlbox.h>

static void         html_box_accessible_class_init               (HtmlBoxAccessibleClass  *klass);
static void         html_box_accessible_initialize               (AtkObject         *obj,
                                                                  gpointer          data);
static gint         html_box_accessible_get_index_in_parent      (AtkObject         *obj);
static AtkStateSet* html_box_accessible_ref_state_set	           (AtkObject         *obj);

static void         html_box_accessible_component_interface_init (AtkComponentIface *iface);
static guint        html_box_accessible_add_focus_handler        (AtkComponent      *component,
                                                                  AtkFocusHandler   handler);
static void         html_box_accessible_get_extents              (AtkComponent      *component,
                                                                  gint              *x,
                                                                  gint              *y,
                                                                  gint              *width,
                                                                  gint              *height,
                                                                  AtkCoordType      coord_type);
static gboolean     html_box_accessible_grab_focus               (AtkComponent      *component);
static void         html_box_accessible_remove_focus_handler     (AtkComponent      *component,
                                                                  guint             handler_id);

static AtkGObjectAccessibleClass *parent_class = NULL;

GType
html_box_accessible_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo tinfo = {
			sizeof (HtmlBoxAccessibleClass),
			(GBaseInitFunc) NULL, /* base init */
			(GBaseFinalizeFunc) NULL, /* base finalize */
			(GClassInitFunc) html_box_accessible_class_init,
			(GClassFinalizeFunc) NULL, /* class finalize */
			NULL, /* class data */
			sizeof (HtmlBoxAccessible),
			0, /* nb preallocs */
			(GInstanceInitFunc) NULL, /* instance init */
			NULL /* value table */
		};

		static const GInterfaceInfo atk_component_info = {
			(GInterfaceInitFunc) html_box_accessible_component_interface_init,
			(GInterfaceFinalizeFunc) NULL,
			NULL
		};

		type = g_type_register_static (ATK_TYPE_GOBJECT_ACCESSIBLE, "HtmlBoxAccessible", &tinfo, 0);
		g_type_add_interface_static (type, ATK_TYPE_COMPONENT, &atk_component_info);
	}

	return type;
}

static void
html_box_accessible_class_init (HtmlBoxAccessibleClass *klass)
{
	AtkObjectClass *class = ATK_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	class->get_index_in_parent = html_box_accessible_get_index_in_parent;
	class->ref_state_set = html_box_accessible_ref_state_set;
	class->initialize = html_box_accessible_initialize;
}

AtkObject*
html_box_accessible_new (GObject *obj)
{
	GObject *object;
	AtkObject *atk_object;

	g_return_val_if_fail (HTML_IS_BOX (obj), NULL);
	object = g_object_new (HTML_TYPE_BOX_ACCESSIBLE, NULL);
	atk_object = ATK_OBJECT (object);
	atk_object_initialize (atk_object, obj);
	atk_object->role = ATK_ROLE_UNKNOWN;
	return atk_object;
}

static void
html_box_accessible_initialize (AtkObject *obj, gpointer data)
{
	HtmlBox *box;
	ATK_OBJECT_CLASS (parent_class)->initialize (obj, data);

	box = HTML_BOX (data);

	/*
	 * We do not set the parent here for the root node of a HtmlView
	 */
	if (box->parent) { 
		atk_object_set_parent (obj,
                        atk_gobject_accessible_for_object (G_OBJECT (box->parent)));
	}
}

static gint
html_box_accessible_get_index_in_parent (AtkObject *obj)
{
	AtkObject *parent;
	AtkGObjectAccessible *atk_gobj;
	HtmlBox *box;
	HtmlBox *parent_box;
	gint n_children = 0;
	GObject *g_obj;

	g_return_val_if_fail (HTML_IS_BOX_ACCESSIBLE (obj), -1);

	atk_gobj = ATK_GOBJECT_ACCESSIBLE (obj);
	g_obj = atk_gobject_accessible_get_object (atk_gobj);
	if (g_obj == NULL)
		return -1;

	g_return_val_if_fail (HTML_IS_BOX (g_obj), -1);
	box = HTML_BOX (g_obj);
	parent = atk_object_get_parent (obj);
	if (HTML_IS_VIEW_ACCESSIBLE (parent)) {
		return 0;
	}
	else if (ATK_IS_GOBJECT_ACCESSIBLE (parent)) {
		parent_box = HTML_BOX (atk_gobject_accessible_get_object (ATK_GOBJECT_ACCESSIBLE (parent)));
	}
	else {
		g_assert_not_reached ();
		return -1;
	}

	if (parent_box) {
		HtmlBox *child;

		child = parent_box->children;

		while (child) {
			if (child == box)
				return n_children;

			n_children++;
			child = child->next;
		}
	}
	return -1;
}

static AtkStateSet*
html_box_accessible_ref_state_set (AtkObject *obj)
{
	AtkGObjectAccessible *atk_gobj;
	GObject *g_obj;
	AtkStateSet *state_set;

	g_return_val_if_fail (HTML_IS_BOX_ACCESSIBLE (obj), NULL);
	atk_gobj = ATK_GOBJECT_ACCESSIBLE (obj);
	state_set = ATK_OBJECT_CLASS (parent_class)->ref_state_set (obj);

	g_obj = atk_gobject_accessible_get_object (atk_gobj);
	if (g_obj == NULL) {
		/* Object is defunct */
		atk_state_set_add_state (state_set, ATK_STATE_DEFUNCT);
	}
	else {
		HtmlBox *box;
  
		box = HTML_BOX (g_obj);

		/* FIXME
		 * We need to check whether the box is on the screen
                 */
		if (HTML_BOX_GET_STYLE (box)->display != HTML_DISPLAY_NONE) {
			atk_state_set_add_state (state_set, ATK_STATE_VISIBLE);
			atk_state_set_add_state (state_set, ATK_STATE_SHOWING);
		}
	}
	return state_set;
} 

static void
html_box_accessible_component_interface_init (AtkComponentIface *iface)
{
  g_return_if_fail (iface != NULL);

  iface->add_focus_handler = html_box_accessible_add_focus_handler;
  iface->get_extents = html_box_accessible_get_extents;
  iface->grab_focus = html_box_accessible_grab_focus;
  iface->remove_focus_handler = html_box_accessible_remove_focus_handler;
}

static guint
html_box_accessible_add_focus_handler (AtkComponent *component, AtkFocusHandler handler)
{
	return g_signal_connect_closure (component, 
					"focus-event",
					g_cclosure_new (G_CALLBACK (handler), NULL,
							(GClosureNotify) NULL),
					FALSE);
}

static void
html_box_accessible_get_extents (AtkComponent *component, gint *x, gint *y, gint *width, gint *height, AtkCoordType coord_type)
{
	AtkGObjectAccessible *atk_gobj;
	HtmlBox *box;
	GObject *g_obj;

	g_return_if_fail (HTML_IS_BOX_ACCESSIBLE (component));

	atk_gobj = ATK_GOBJECT_ACCESSIBLE (component);
	g_obj = atk_gobject_accessible_get_object (atk_gobj);
	if (g_obj == NULL)
		return;

	g_return_if_fail (HTML_IS_BOX (g_obj));
	box = HTML_BOX (g_obj);

	*x = html_box_get_absolute_x (box);
	*y = html_box_get_absolute_y (box);
	*width = box->width;
	*height = box->height;

	g_print ("%d %d %d %d\n",
           html_box_get_absolute_x (box),
           html_box_get_absolute_y (box),
           html_box_get_containing_block_width (box),
           html_box_get_containing_block_height (box));
}

static gboolean
html_box_accessible_grab_focus (AtkComponent *component)
{
  return TRUE;
}

static void
html_box_accessible_remove_focus_handler (AtkComponent *component, guint handler_id)
{
	g_signal_handler_disconnect (ATK_OBJECT (component), handler_id);
}
