/****************************************************************/
/*                                                              */
/*      fcompl.cpp                                              */
/*                                                              */
/*      This file is a part of kpl - a program for graphical    */
/*      presentation of data sets and functions.                */
/*                                                              */
/*      Copyright (C) 2000 by Werner Stille                     */
/*      <stille@uni-freiburg.de>                                */
/*                                                              */
/*      Released under the GPL; see file LICENSE for details.   */
/*                                                              */
/****************************************************************/

/* Example functions for kpl */

#include <math.h>
#include <complex>

/****************************************************************/
/*                                                              */
/*    epsilon_c(a, x)                                           */
/*      calculates the complex frequency dependent dielectric  */
/*      constant for one Debye relaxation process               */
/*                                                              */
/*    a:       relaxation strength                              */
/*    x:       relative frequency f / f0                        */
/*                                                              */
/*    Returns: a / (1 + i * x)                                  */
/*                                                              */
/****************************************************************/

complex<double> epsilon_c(double a, const double x)
{
  return a / (1.0 + complex<double>(0.0, x));
}

/****************************************************************/
/*                                                              */
/*    epsilon_1(log_f, p)                                       */
/*      calculates the real part of the complex frequency       */
/*      dependent dielectric constant for Debye relaxation      */
/*      processes                                               */
/*                                                              */
/*    log_f:   log10(f / Hz)                                    */
/*                                                              */
/*    Returns: p[0] + Re(epsilon_c(p[1], f / p[2]) +            */
/*                    Re(epsilon_c(p[3], f / p[4]) + ...        */
/*                                                              */
/****************************************************************/

double epsilon_1(double log_f, const double *p)
{
  double f = pow(10.0, log_f);
  double e1 = p[0];
  for (int i = 1; i < 18; i += 2)
    if (p[i])
      e1 += real(epsilon_c(p[i], f / p[i + 1]));
  return e1;
}

/****************************************************************/
/*                                                              */
/*    epsilon_2(log_f, p)                                       */
/*      calculates the (negative) imaginary part of the complex */
/*      frequency dependent dielectric constant for Debye       */
/*      relaxation processes                                    */
/*                                                              */
/*    log_f:    log10(f / Hz)                                   */
/*                                                              */
/*    Returns:  -Im(epsilon_c(p[1], f / p[2]) -                 */
/*               Im(epsilon_c(p[3], f / p[4]) - ...             */
/*                                                              */
/****************************************************************/

double epsilon_2(double log_f, const double *p)
{
  double f = pow(10.0, log_f);
  double e2 = 0.0;
  for (int i = 1; i < 18; i += 2)
    if (p[i])
      e2 -= imag(epsilon_c(p[i], f / p[i + 1]));
  return e2;
}
