/***************************************************************************
                          lmfit.h  -  description
                             -------------------
    begin                : Sat May 22 1999
    copyright            : (C) 2002 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef LMDIF_H
#define LMDIF_H

#include <qlist.h>
#include <qobject.h>

class ArrayItem;
class FunItem;

/**
  * Parameter fit class. Provides methods for nonlinear Levenberg-Marquardt
  * parameter fits and general linear least square parameter fits.
  *
  * The functions lmdif, lmpar, fdjac2, qrsolv, qrfac, and enorm
  * use code from public domain Fortran version of Argonne National Laboratories MINPACK,
  * C translation by Steve Moshier. Modifications by Werner Stille.
  *
  * The functions minv, igamc, igam, simq, and mtransp
  * are from the Cephes Math Library Release 2.7, Copyright by Stephen L. Moshier.
  *
  * The function lgam (original name: gsl_sf_lngamma_impl, Author: G. Jungman) is
  * from the GNU Scientific Library. Modifications by Werner Stille.
  */
class LMFit : public QObject
{
  Q_OBJECT
public:
  /**
   * Constructor.
   * @param parent pointer to parent widget.
   */
  LMFit(QObject* parent);
  /** Destructor. */
  ~LMFit();
  /** Performs general linear least square parameter fit. */
  bool linFit(int nd, int np, double px[], double fvec[], double a[],
              int ipvt[], double qtf[], double wa1[], double wa2[],
              double wa3[], double wa4[]);
  /**
   * Performs nonlinear Levenberg-Marquardt parameter fit.
   */
  void lmdif(int nd, int np, double px[], double fvec[], double ftol,
             double ptol, double gtol, int maxfev, double epsfcn,
             double diag[], int mode, double factor, int nprint,
             int* info, int* nfev, double fjac[], int ldfjac,
             int ipvt[], double qtf[], double wa1[], double wa2[],
             double wa3[], double wa4[]);
  /**
   * Calculates the euclidean norm of x.
   * @param n length of x.
   * @param x input array.
   * @return euclidean norm of x.
   */
  static double enorm(int n, const double x[]);
  /**
   * Finds the inverse of a square matrix.
   * @param A input matrix.
   * @param X output matrix.
   * @param n numbers of rows and columns of A.
   * @param B scratch pad array of length n.
   * @param IPS scratch pad array of length n.
   * @return 0 for success.
   */
  static int minv(double A[], double X[], int n, double B[], int IPS[]);
  /**
   * Calculates complemented incomplete gamma integral.
   * @param a argument of gamma function.
   * @param x upper boundary of integral.
   * @return complemented incomplete gamma integral.
   */
  static double igamc(double a, double x);
  /**
   * Calculates incomplete gamma integral.
   * @param a argument of gamma function.
   * @param x upper boundary of integral.
   * @return incomplete gamma integral.
   */
  static double igam(double a, double x);
  /**
   * Calculates logarithm of gamma function using the Lanczos method.
   * @param x argument of gamma function.
   * @return logarithm of gamma function.
   */
  static double lgam(double x);
  /**
   * Solves simultaneous linear equations AX = B by Gaussian elimination
   * with partial pivoting.
   * @param A input matrix stored as a n * n vector (destroyed on output).
   * @param B input right hand vector.
   * @param X solution vector.
   * @param n numbers of rows and columns of A.
   * @param flag = 0 to solve, -1 to do a new back substitution for different B vector
   * @param IPS scratch pad array of length n.
   * @return 0 for success.
   */
  static int simq(double A[], double B[], double X[], int n, int flag, int IPS[]);
  /**
   * Transposes a square matrix.
   * @param n numbers of rows and columns of A.
   * @param A input matrix stored as a n * n vector.
   * @param T output transposed matrix stored as a n * n vector
   * (may occupy the same storage as A).
   */
  static void mtransp(int n, double* A, double* T);

protected:
  void fcn(int nd, int np, const double p[], double fvec[], int* iflag);
  void lmpar(int n, double r[], int ldr, const int ipvt[],
             const double diag[], const double qtb[], double delta,
             double* par, double x[],
             double sdiag[], double wa1[], double wa2[]);
  void fdjac2(int m, int n, double x[], const double fvec[], double fjac[],
              int ldfjac, int *iflag, double epsfcn, double wa[]);
  void qrsolv(int n, double r[], int ldr, const int ipvt[],
              const double diag[], const double qtb[], double x[],
              double sdiag[], double wa[]);
  void qrfac(int m, int n, double a[], int lda, int pivot, int ipvt[],
             int lipvt, double rdiag[], double acnorm[], double wa[]);
  bool* bFit;
  bool userBreak;
  double* p;
  QArray<double> sig;
  QList<ArrayItem>* ad;
  QList<FunItem>* fd;

  friend class FitDlg;

signals:
  /**
    * Emitted when a message should be displayed.
    * @param msg message string
    */
  void updateMessage(const QString& msg);
};

#endif
