/***************************************************************************
                          arrayitem.h  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sat Aug 28 1999
    copyright            : (C) 2004 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef ARRAYITEM_H
#define ARRAYITEM_H

#include <kurl.h>
#include "kplitem.h"

class QTextStream;
class KSimpleConfig;

/**
  * Array item class. Class for array items showing data arrays by lines
  * or markers.
  * @author Werner Stille
  */
class ArrayItem : public ScaledItem
{
public:
  /** Constructor */
  ArrayItem();
  /** Copy constructor */
  ArrayItem(const ArrayItem& a);
  /**
    * Constructor. Initializes corresponding to autoplot settings.
    * @param aut pointer to structure containing autoplot settings.
    */
  ArrayItem(Kpl::AutoStruct* aut);
  /**
    * Constructor. Initializes corresponding to plot file.
    * @param plo pointer to KSimpleConfig object containing item data.
    * @param aut pointer to structure containing autoplot settings.
    * @param url URL of plot file.
    */
  ArrayItem(KSimpleConfig* plo, Kpl::AutoStruct* aut, const KURL& url);
  /**
    * Constructor. Initializes corresponding to arguments.
    * @param active true for visible item.
    * @param fillStyle > 0 for filled polygon representation.
    * @param symb 0 for circles, < 0 for different markers,
    * >0 for different line types.
    * @param color color for function plot.
    * @param fx normalization factor for x values.
    * @param fy normalization factor for y values.
    * @param ix x column index.
    * @param iy y column index.
    * @param ie y error column index.
    * @param istart start row index.
    * @param n number of data points.
    * @param errbars true to plot error bars.
    * @param url URL of data file.
    * @param internal true for internal stored array data.
    * @param relSize relative size of lines or symbols.
    * @param idec 0 for decimal point, 1 for comma.
    */
  ArrayItem(bool active, int fillStyle, int symb, const QString& color,
            double fx, double fy, int ix, int iy, int ie, int istart,
            int n, int errbars, const KURL& url,
            bool internal = false, double relSize = 1.0, int idec = 0);
  /** Destructor */
  virtual ~ArrayItem();
  /** Assignment operator */
  ArrayItem& operator=(const ArrayItem& f);
  /**
    * Copies array storage.
    * @param xv source array.
    * @param nc number of columns.
    * @param nr number of rows.
    * @return pointer to array copy.
    */
  static double** copyX(double **xv, int nc, int nr);
  /**
    * Deallocates array storage.
    * @param xv pointer to array.
    */
  static void freeX(double ***xv);
  /**
    * Determine number of columns of an array to be read from a text stream.
    * @param t text stream to read from.
    * @param idec 0 for decimal point, 1 for comma.
    * @return number of columns, 0 for error.
    */
  static int nColumns(QTextStream& t, int idec = 0);
  /**
    * Reads double values from a text stream into an array.
    * @param t text stream to read from.
    * @param nc number of columns.
    * @param xv pointer to array.
    * @param idec 0 for decimal point, 1 for comma.
    * @return number of lines, 0 for error.
    */
  static int readLines(QTextStream& t, int nc, double ***xv, int idec = 0);
  /**
    * Reads double values from a file into an array.
    * @param url URL of data file.
    * @param nc number of columns.
    * @param xv pointer to array.
    * @param idec 0 for decimal point, 1 for comma.
    * @return number of lines, 0 for error.
    */
  static int readFile(const KURL& url, int *nc, double ***xv, int idec = 0);
  /**
    * Reads double values from a string into an array.
    * @param text string to read from.
    * @param nc number of columns.
    * @param xv pointer to array.
    * @return number of lines, 0 for error.
    */
  static int readFile(QString* text, int *nc, double ***xv);
  virtual ItemTypes iType() const;
  virtual void draw(KplGraph* g);
  virtual void writePlo(KSimpleConfig* plo, const KURL& url, bool abs,
                        KplDoc* m) const;
  virtual void setText(KplCheckListItem* it, bool* arrays, bool* funcs) const;
  virtual int editItem(QWidget* parent, KplDoc* m, int i);
  virtual KplItem* copy() const;
  virtual void expoItem(int* iex, int* iey, double* fx, double* fy) const;
  virtual void minMax(double* xmi, double* xma, double* ymi, double* yma) const;

protected:
  bool internal;
  int ix, iy, ie, istart, n, errbars, nrows, ncols, idec;
  double** x;
  KURL url;

  friend class ArrayDlg;
  friend class FitDlg;
  friend class SplFitDlg;
  friend class LMFit;
  friend class KplDoc;
  friend class KplApp;
};

#endif
