/*
 * CPU counter utility for GNU/Linux
 * Special feature for detecting Hyperthreading enabled processor
 *
 * Copyright 2003 Lee Jung-Hoon, HPC Labs, Univ. of Seoul
 *
 * All rights reserved.
 *
 * 
 * This program is a Linux version of "CPU Counting" Utility from Intel Corp.
 * made by High Performance Computing Laboratory, University of Seoul.
 * May be used under the terms of the GNU General Public License (GPL)
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sched.h>
#include <linux/unistd.h>
#ifdef GETOPTLONG
#include <getopt.h>
#endif /* GETOPTLONG */
#include <errno.h>
#include <string.h>

int HTSupported (void);

/* for more informations, http://www.delorie.com/gnu/docs/gcc/gcc_86.html */
/* and http://www.delorie.com/gnu/docs/gcc/gcc_167.html */
#define cpuid(in,a,b,c,d)\
  asm("cpuid": "=a" (a), "=b" (b), "=c" (c), "=d" (d) : "a" (in));

#define PROCLOC          "/proc/"
#define CPU_MAX_COUNT	128

/*
 * provide the proper syscall information if our libc
 * is not yet updated.
 */
#ifndef __NR_sched_setaffinity
#define __NR_sched_setaffinity  241
#define __NR_sched_getaffinity  242
_syscall3 (int, sched_setaffinity, pid_t, pid, unsigned int, len, unsigned long *, user_mask_ptr)
_syscall3 (int, sched_getaffinity, pid_t, pid, unsigned int, len, unsigned long *, user_mask_ptr)
#endif

/* global variables */
const char *version = "0.9.6";
/* command line arguments */
int verbose_mode = 0;
int mds_mode = 0;

/* display command line usage */
void usage ()
{
   fprintf(stderr,
"Cpucounter for linux version %s by Lee Jung-Hoon (jhlee94@sidae.uos.ac.kr)\n"
"Usage:\n"
"  cpucount4linux -h                            -- display command usage and exit\n"
"  cpucount4linux -V                            -- display version information and exit\n"
"  cpucount4linux [-v]                          -- detecting support for hyper-threading technology enabled processors\n"
"Options:\n"
"  -v                                           -- enable verbose output\n"
"  -m                                           -- enable mds support\n"
#ifdef GETOPTLONG
"Long options:\n"
"  -h --help    -v --verbose    -m --mds    -V --Version\n"
#endif
"High Performance Laboratory, University of Seoul.\n" , version);
   exit(1);
}


/* Just count all of processors in the system being viewd by operating
 * system
 */
static int count_number_processor ()
{
   FILE *filep;
   char filename [4096];
   char tmpstring [4096];

   int c=-1;

   sprintf (filename, "%scpuinfo", PROCLOC);
   filep=fopen(filename, "r");

   if (filep==NULL) return (0);

   do {
	   fgets (tmpstring, 4096, filep);
	   if (feof(filep)!=0) break;
	   if (strlen(tmpstring) > 128) 
		   tmpstring[128]=0;
	   if (strncmp(tmpstring, "processor", 9)==0) {
		   c++;
		   if (c==CPU_MAX_COUNT) break;
	   } /* end of if (strncmp(tmpstring, "processor", 9)==0) */
   } while (0==0);

   return (c);
} /* end of function static int count_number_processor () */



/* display output by MDS format */
static void with_mds ()
{
  verbose_mode = 0;
  mds_mode = 1;
//  if (!HTSupported()) printf("0\n");
//  else printf("1\n");

//  exit (0);
}

/* display version information */
static void display_version ()
{
   fprintf(stderr,
"Cpucounter for linux version %s by Lee Jung-Hoon (jhlee94@sidae.uos.ac.kr)\n"
"Copyright 2003 Lee Jung-Hoon, HPC Labs, Univ. of Seoul\n\n"
"This program is distributed in the hope that it will be useful,\n"
"but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
"GNU General Public License for more details.\n\n"
"High Performance Computing Laboratory, University of Seoul.\n",
version);
   exit(1);
}

/* determine options or parameters in the command line arguments */
void do_option(int argc, char **argv)
{
  const char *flags = "hvmV";
#ifdef GETOPTLONG
  static struct option long_options[] =
  {
     {"help",           no_argument,    NULL,   'h'},
     {"verbose",        no_argument,    NULL,   'v'},
     {"mds",        	no_argument,    NULL,   'm'},
     {"Version",        no_argument,    NULL,   'V'},
     {0, 0, 0, 0}
   };
   int option_index;
#endif /* GETOPTLONG */
   int c;

#ifdef GETOPTLONG
   while ((c = getopt_long(argc, argv, flags, long_options, &option_index)) != EOF) {
#else
   while ((c = getopt(argc, argv, flags)) != EOF) {
#endif /* GETOPTLONG */

   switch (c) {
   case 'h':
     usage();
     exit(0);
     break;
   case 'v':
     verbose_mode = 1;
     break;
   case 'm':
	 with_mds();
	 break;
   case 'V':
     display_version();
     exit(0);
     break;
   case '?':
     exit(1);
     break;
   }
   }
}

/* Detect if Hyper-Threading Technology is supported */
int HTSupported (void) {
  int i, brandname;
  unsigned long eax,ebx,ecx,edx,unused;

      /* dumping processor's brand signature from the registers */
	  if (verbose_mode || mds_mode) {
		  /* You have to know that verbose_mode and mds_mode is confliction
		   */
		  if (verbose_mode ) printf("CPU vendor : ");
		  else printf("Mds-Cpu-Smt-vendor: ");

		  cpuid(0,unused,ebx,ecx,edx);

 	      for(i=0;i<4;i++)
      		putchar(ebx >> (8*i));
   	 		 /* becuase of 32-bit registers*/
  	 	  for(i=0;i<4;i++)
   	  		putchar(edx >> (8*i));
 	 	  for(i=0;i<4;i++)
   	  		putchar(ecx >> (8*i));
  	 	  printf("\n");
	  } /* end of if (verbose_mode) */

  /* detect cpu brandname and hyperthreading features */
  cpuid(1,eax,ebx,unused,edx);
 
  /* first, look cpu brandname, ref. page 10, 23 */
  brandname = ebx & 0xff;

  if (brandname > 0) {
	if (verbose_mode || mds_mode) {

    if (verbose_mode ) printf ("Processor brandname : ");
	else printf ("Mds-Cpu-Smt-brand: ");

    switch (brandname) {
    case 1:
        printf ("Intel Celeron processor\n");
        break;
    case 2:
        printf ("Intel Pentium 3 processor\n");
        break;
    case 3:
        printf ("Intel Pentium 3 Xeon processor or Intel Celeron processor\n");
        break;
    case 4:
        printf ("Intel Pentium 3 processor\n");
        break;
    case 6:
        printf ("Mobile Intel Pentium 3 processor-M\n");
        break;
    case 7:
        printf ("Mobile Intel Celeron processor\n");
        break;
    case 8:
        printf ("Intel Pentium 4 processor or Intel Genuine processor\n");
        break;
    case 9:
        printf ("Intel Pentium 4 processor\n");
        break;
    case 10:
        printf ("Intel Celeron Processor\n");
        break;
    case 11:
        printf ("Intel Xeon processor or Intel Xeon processor MP\n");
        break;
    case 12:
        printf ("Intel Xeon processor MP\n");
        break;
    case 14:
        printf ("Mobile Intel Pentium 4 processor-M or Intel Xeon processor\n");
        break;
    case 15:
        printf ("Mobile Intel Celeron processor\n");
        break;
     } /* end of switch (brandname) */
	} /* end of if (verbose_mode || mds_mode) */
   } /* end of if (brandname > 0) */

   else {
		if (verbose_mode)
		{
	      printf ("Processor brandname : Poor\n"); 
		  printf ("Your processor does not support cpuid instruction or hyper-threading technology\n");
		} /* end of if (verbose_mode) */
		if (mds_mode)
			printf ("Mds-Cpu-Smt-brand: Poor\n");
		 
		exit (-1);
   } /* end of else */

   if (edx & (1<<28)) return 1;
   else return 0;      
}


/* returnms the number of logical processors per physical processors */
int LogicalProcessorsPerPackage (void) {
   unsigned long eax,ebx,edx,unused;

   /* call cpuid with eax = 1 */
   cpuid(1,eax,ebx,unused,edx);
    
   /* EBX[23:16] indicate number of logical processors per package */
   return ((ebx & 0x00FF0000) >> 16);
}

/* returns the 8-bit unique Initial APIC ID for the processor this
 * code is actually running on. The default value returned is 0xFF if
 * Hyper-Threading Technology is not supported
 */

int GetAPIC_ID (void) {
   unsigned long eax,ebx,edx,unused;
 
   /* call cpuid with eax = 1 */
   cpuid(1,eax,ebx,unused,edx);
  
   return ((ebx >> 24) & 0xff);
}


int main (int argc, char **argv) {
  int i, PHY_ID_MASK, PHY_ID_SHIFT;
  int HT_Enabled, Logical_Per_Package;
  int Number_of_Physical;
  unsigned long new_mask; unsigned int len = sizeof(new_mask);
  unsigned long cur_mask;
  unsigned long dwAffinityMask;
  pid_t current_pid;
  int ret;

  int APIC_ID, LOG_ID, PHY_ID;

  /* determine options or parameters in the command line arguments */
  do_option(argc, argv);

  /* At first, just count whole bunch of processors include logical
   * processors
   */
  Number_of_Physical = count_number_processor() + 1;

  /* Check to see if Hyper-Threading Technology is available */
  if (HTSupported()) {
	HT_Enabled = 0;

	if (!mds_mode) printf ("Hyper-Threading capabable processor : Capable\n");
	else printf ("Mds-Cpu-Smt-capable: Capable\n");

	Logical_Per_Package = LogicalProcessorsPerPackage();

     
     /* Just because logical processors is >1 
      * does not mean that Hyper-Threading Technology is enabled in the BIOS
      */
	if (Logical_Per_Package > 1) {

        /* Calculate the approprate shifts and mask based on the
         * number of logical processors
         */
		i = 1; PHY_ID_MASK = 0xFF; PHY_ID_SHIFT = 0;
        
		/* Physical processor ID and Logical processor IDs are derived
		 * from the APIC ID. We'll calculate approprate shift and
		 * mask values knowing the number of logical processors per
		 * physical processor package
		 */
		while (i < Logical_Per_Package) {
			i *= 2;
			PHY_ID_MASK <<= 1;
			PHY_ID_SHIFT++;
		} /* end of while (i < Logical_Per_Package) */

		current_pid = getpid();
		ret = sched_getaffinity(current_pid, len, &cur_mask);

		dwAffinityMask = 1;
		while (dwAffinityMask != 0 && dwAffinityMask <= cur_mask) {
		/* Check to make sure we can utilize this processor first */
			if (dwAffinityMask & cur_mask) {
				if ( sched_setaffinity(current_pid, len, &dwAffinityMask) >= 0 ) {
					sleep(0); /* we may not be running on the cpu that we just set the affinity to. */
	   		 				  /* sleep gives the OS a chance to switch us to the desired CPU */
					APIC_ID = GetAPIC_ID ();
					LOG_ID = APIC_ID & ~PHY_ID_MASK;
					PHY_ID = APIC_ID >> PHY_ID_SHIFT;

					/* print out table of processor IDs */
					if (verbose_mode)
					  printf ("OS Affinity ID: %08lx, APIC ID: %d, PHY ID: %d, LOG ID: %d\n", dwAffinityMask, APIC_ID, PHY_ID, LOG_ID);

					if (LOG_ID != 0) HT_Enabled = 1;
				} /* end of if ( sched_setaffinity(current_pid, len, &dwAffinityMask) >= 0) */

				else {
					/* This shouldn't happen since we check to make sure we
					 * can utilize this processor before trying to set
					 * affinity mask.
					 * and This is the information about error handling
					 * http://www.itworld.com/nl/lnx_tip/03092001/
					 */
					if (!mds_mode)
					  printf("\nerror: %s \n", strerror(errno)); 

					if (verbose_mode) {
					  printf("Please, make sure your linux kernel has the system calls which respect CPU affinity.\n");
					  printf("See http://www.kernel.org/pub/linux/kernel/people/rml/cpu-affinity/ \n\n");
					} /* end of if (verbose_mode) */
					break;
				} 

			} /* end of if (dwAffinityMask & cur_mask) */

			dwAffinityMask = dwAffinityMask << 1;

		} /*  end of while (dwAffinityMask != 0 && dwAffinityMask <= cur_mask) */


		if (HT_Enabled) 
		{
//			printf ("Processors with Hyper-Threading Technology enabled was detected.\n");
			if (!mds_mode) {
		    	printf ("Number of physical processors : %d\n", (Number_of_Physical / Logical_Per_Package));
	        	printf ("Logical processors per physical : %d\n", Logical_Per_Package);
				printf ("Hyper-Threading technology : Enabled\n");
			} /* end of if (!mds_mode) */
			else {
				printf ("Mds-Cpu-Smt-Physical-count: %d\n", (Number_of_Physical / Logical_Per_Package));
	        	printf ("Mds-Cpu-Smt-LP-count: %d\n", Logical_Per_Package);
				printf ("Mds-Cpu-Smt-enabled: Enabled\n");
			} /* end of else */
		} /* end of if (HT_Enabled) */
		else 
		{
//			printf ("Processors with Hyper-Threading Technology enabled was not detected.\n");
			if (!mds_mode) {
		    	printf ("Number of physical processors : %d\n", Number_of_Physical);
	        	printf ("Logical processors per physical : %d\n", Logical_Per_Package);
				printf ("Hyper-Threading technology : Not Detected\n");
			} /* end of if (!mds_mode) */
			else {
				printf ("Mds-Cpu-Smt-Physical-count: %d\n", Number_of_Physical);
	        	printf ("Mds-Cpu-Smt-LP-count: %d\n", Logical_Per_Package);
				printf ("Mds-Cpu-Smt-enabled: Not Detected\n");
			} /* end of else */
		} /* end of else */
	
	} /* end of if (Logical_Per_Package > 1) */     
	else 
	{ // printf ("Processors with Hyper-Threading Technology is not enabled \n");
		if (!mds_mode) {
	    	printf ("Number of physical processors : %d\n", Number_of_Physical);
	    	printf ("Logical processors per physical : %d\n", Logical_Per_Package);
			printf ("Hyper-Threading technology : Not Enabled\n");
		} /* end of if (!mds_mode) */
		else {
			printf ("Mds-Cpu-Smt-Physical-count: %d\n", Number_of_Physical);
	       	printf ("Mds-Cpu-Smt-LP-count: %d\n", Logical_Per_Package);
			printf ("Mds-Cpu-Smt-enabled: Not Enabled\n");
		} /* end of else */
	} /* end of else */
 
  } /* end of if (HTSupported()) */
  else 
  { 
	  if (!mds_mode) {
	     printf ("Hyper-Threading capable processor : Not Capable\n"); 
	     printf ("Number of physical processors : %d\n", Number_of_Physical);
		 printf ("Logical processors per physical : 0\n");
	     printf ("Hyper-Threading technology : Disabled\n"); 
	  } /* if (!mds_mode) */
	  else {
	     printf ("Mds-Cpu-Smt-capable: Not Capable\n"); 
		 printf ("Mds-Cpu-Smt-Physical-count: %d\n", Number_of_Physical);
	     printf ("Mds-Cpu-Smt-LP-count: 0\n");
		 printf ("Mds-Cpu-Smt-enabled: Disabled\n");
	  } /* end of else */
  } /* end of else */

  exit (0);
}

