/*
 * Written by Mark Goodwin (markgw@sgi.com), December 2000.
 *
 * Copyright (c) 2000-2001 Silicon Graphics, Inc.  All Rights Reserved.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it would be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 * Further, this software is distributed without any warranty that it is
 * free of the rightful claim of any third person regarding infringement
 * or the like.  Any license provided herein, whether implied or
 * otherwise, applies only to this software file.  Patent licenses, if
 * any, provided herein do not apply to combinations of this program with
 * other software, or any other product whatsoever.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write the Free Software Foundation, Inc., 59
 * Temple Place - Suite 330, Boston MA 02111-1307, USA.
 * 
 * Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
 * Mountain View, CA  94043, or:
 * 
 * http://www.sgi.com 
 * 
 * For further information regarding this notice, see: 
 * 
 * http://oss.sgi.com/projects/GenInfo/SGIGPLNoticeExplan/
 */

#include <stdlib.h>
#include <stdio.h>
#include <getopt.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>
#include <curses.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <netdb.h>

#include <pcp/pmapi.h>

static struct {
	char		*metricname;
	pmDesc		desc;
	double		*values;
} metrics[] = {
#define LS_WAIT 0
    { "lockstat.detail.time.wait", {0}, NULL },
#define LS_HOLD 1
    { "lockstat.detail.time.hold", {0}, NULL },
#define LS_MAXWAIT 2
    { "lockstat.detail.time.max_wait", {0}, NULL },
#define LS_MAXHOLD 3
    { "lockstat.detail.time.max_hold", {0}, NULL },
#define LS_SPIN 4
    { "lockstat.detail.count.spin", {0}, NULL },
#define LS_NOWAIT 5
    { "lockstat.detail.count.nowait", {0}, NULL },
};

#define NMETRICS (sizeof(metrics)/sizeof(metrics[0]))

/* lockstat.detail instance domain */
static int n_instlist;
static int *instlist;
static char **namelist;
static int *sorted;
static void lockstat_fetch(void);
static char lasterror[1024];
static char fetch_ctime[64];
static double fetch_delta;
static int lines, cols;
static int sortmode = 's';
static pmResult	*curr_result = NULL;
static pmResult	*prev_result = NULL;

static char *host = NULL;
static char *archive = NULL;
static pmLogLabel label;
struct timeval archiveEnd;

int opt_debug;

static double
tvsub(struct timeval *ap, struct timeval *bp)
{
    return ap->tv_sec - bp->tv_sec + (double)(ap->tv_usec - bp->tv_usec)/1000000.0;
}

void
dohelp()
{
    int line = 1;

    clear();
    mvprintw(line++, 0, "The following keys are available:");
    mvprintw(line++, 0, "   q  quit");
    mvprintw(line++, 0, "   t  sort on total wait (1st column)");
    mvprintw(line++, 0, "   w  sort on avg wait time (2nd column)");
    mvprintw(line++, 0, "   h  sort on avg hold time (3rd column)");
    mvprintw(line++, 0, "   s  sort on spin call rate (4th column)");
    mvprintw(line++, 0, "   n  sort on nospin call rate (5th column)");

    if (archive) {
	line++;
	mvprintw(line++, 0, "   == Archive replay controls ==");
	mvprintw(line++, 0, "   f forward archive one step");
	mvprintw(line++, 0, "   B rewind to start of archive");
    }

    line++;
    mvprintw(line++, 0, "   == press any key to exit help ==");
    refresh();

    while (getch() == ERR)
    	usleep(1000);
    clear();
    refresh();
}

static void
usage()
{
    fprintf(stderr, "Usage: locktop [-h host] [-a archive] [-t interval]\n");
    exit(1);
}

int
main(int argc, char *argv[])
{
    int pcpcon;
    int i;
    char *p;
    int c;
    int j;
    int key = '#';
    int iter;
    int line;
    int err;
    int cpu;
    double update_interval = 2.0;
    double total_calls;
    char name[1024];

    while ((c = getopt(argc, argv, "a:h:t:")) != EOF) {
	switch (c) {
	case 't':
	    update_interval = atof(optarg);
	    break;
	case 'a':
	    archive = optarg;
	    break;
	case 'h':
	    host = optarg;
	    break;
	default:
	    usage();
	}
    }

    if (archive && host) {
	fprintf(stderr, "Error: -a and -h are mutually exclusive\n");
    	usage();
    }

    if (archive == NULL && host == NULL) {
    	gethostname(name, sizeof(name));
	host = strdup(name);
    }

    if (archive) {
	if ((pcpcon = pmNewContext(PM_CONTEXT_ARCHIVE, archive)) < 0) {
	    fprintf(stderr, "Error opening archive \"%s\": %s\n", archive, pmErrStr(pcpcon));
	    exit(1);
	}
	if ((err = pmGetArchiveLabel(&label)) < 0) {
	    fprintf(stderr, "Error reading archive \"%s\": %s\n", archive, pmErrStr(err));
	    exit(1);
	}
	pmGetArchiveEnd(&archiveEnd);

	if ((err = pmSetMode(PM_MODE_INTERP, &label.ll_start, (int)(1000.0 * update_interval))) < 0) {
	    fprintf(stderr, "Error pmSetMode for archive \"%s\": %s\n", archive, pmErrStr(err));
	    exit(1);
	}
    }
    else {
	if ((pcpcon = pmNewContext(PM_CONTEXT_HOST, host)) < 0) {
	    fprintf(stderr, "Error connecting to host \"%s\": %s\n", host, pmErrStr(pcpcon));
	    exit(1);
	}
    }

    for (i=0; i < NMETRICS; i++) {
	pmID id;
	if ((err = pmLookupName(1, &metrics[i].metricname, &id)) < 0) {
	    fprintf(stderr, "Error looking up \"%s\" on host \"%s\" : %s\n",
	    	metrics[i].metricname, host, pmErrStr(err));
	    exit(1);
	}

	if ((err = pmLookupDesc(id, &metrics[i].desc)) < 0) {
	    fprintf(stderr, "Error looking up descriptor for \"%s\" on host \"%s\" : %s\n",
	    	metrics[i].metricname, host, pmErrStr(err));
	    exit(1);
	}
    }

    initscr();
    noecho();
    cbreak();
    nodelay(stdscr, TRUE);
    clear();
    getmaxyx(stdscr, lines, cols);
    if (archive) {
	attron(A_BOLD);
	mvprintw(lines/2, cols/2-20, "Please wait .. scanning archive for lock names");
	attroff(A_BOLD);
    }
    refresh();

    /* same instance domain is shared by all lockstat metrics */
    if (archive) {
	if ((err = pmGetInDomArchive(metrics[0].desc.indom, &instlist, &namelist)) < 0) {
	    fprintf(stderr, "Error looking up descriptor for \"%s\" in archive \"%s\" : %s\n",
		metrics[0].metricname, archive, pmErrStr(err));
	    exit(1);
	}
	else
	    n_instlist = err;
    }
    else {
	if ((err = pmGetInDom(metrics[0].desc.indom, &instlist, &namelist)) < 0) {
	    fprintf(stderr, "Error looking up descriptor for \"%s\" on host \"%s\" : %s\n",
		metrics[0].metricname, host, pmErrStr(err));
	    exit(1);
	}
	else
	    n_instlist = err;
    }
    
    sorted = (int *)malloc(n_instlist * sizeof(int));
    for (i=0; i < n_instlist; i++) {
    	sorted[i] = i;
    }

    for (i=0; i < NMETRICS; i++) {
	metrics[i].values = (double *)malloc(n_instlist * sizeof(double));
	memset(metrics[i].values, 0, n_instlist * sizeof(double));
    }


    lockstat_fetch();

    for(iter=0; ; iter++) {
	if (iter == 0) {
	    clear();
	    refresh();
	}

	getmaxyx(stdscr, lines, cols);
	attron(A_BOLD);
	line=0;
	mvprintw(line++, 0, "Host %s.  Sort mode '%c'  %s",
	    pmGetContextHostName(pcpcon), sortmode, fetch_ctime);
	clrtoeol();
	mvprintw(line++, 0, lasterror);
	clrtoeol();

	mvprintw(line++, 0,
	"%9s %9s %9s %9s %9s %3s %s", " Tot Wait", " Avg Wait", " Avg Hold", "     Spin", "  No Spin", "", "");
	mvprintw(line++, 0,
	"%9s %9s %9s %9s %9s %3s %s", "     usec", "usec/call", "usec/call", "calls/sec", "calls/sec", "CPU", "Lock:Caller");
	attroff(A_BOLD);

	if (!archive)
	    lockstat_fetch();

	if (curr_result == NULL || prev_result == NULL) {
	    move(4,0);
	    clrtobot();
	    refresh();
	}
	else
	for (i=0; i < n_instlist && line < lines; i++) {
	    j = sorted[i];
	    if (j < 0 || j >= n_instlist) {
		move(0,0); clrtobot();
	    	printw("FATAL: sorted[%d] = %d\n", i, j);
		exit(1);
	    }

	    strncpy(name, namelist[j], sizeof(name));
	    if ((p = strrchr(name, ':')) != NULL) {
		*p = '\0';
	    	cpu = atoi(p+4);
	    }
	    else
	    	cpu = 0;
	    	
	    total_calls = metrics[LS_SPIN].values[j] +
	    	metrics[LS_NOWAIT].values[j];
	    if (total_calls == 0) {
		/* instance j is not very interesting */
	    	continue;
	    }

	    mvprintw(line++, 0, "%9.1lf %9.3lf %9.3lf %9.1lf %9.1lf %3d %s",

		/* total wait time during interval (microseconds) */
	    	1000 * metrics[LS_WAIT].values[j],

		/* average wait time during interval (microseconds/call) */
	    	1000 * metrics[LS_WAIT].values[j] / total_calls,

		/* average hold time during interval (microseconds/call) */
	    	1000 * metrics[LS_HOLD].values[j] / total_calls,

		/* spin rate: calls/second */
	    	metrics[LS_SPIN].values[j] / fetch_delta,

		/* nospin rate: calls/second */
	    	metrics[LS_NOWAIT].values[j] / fetch_delta,

		/* cpu number */
		cpu,

		/* lock:caller */
		name);

	    clrtoeol();
	}

	refresh();

	if (!archive) {
	    if ((key = getch()) != ERR) {
		switch(key) {
		case 'q':
		case 'Q':
		    move(lines, 0);
		    clrtoeol();
		    refresh();
		    endwin();
		    exit(0);
		case 't':
		case 'w':
		case 'h':
		case 's':
		case 'n':
		    sortmode = key;
		    break;
		default:
		    dohelp();
		    break;
		}
	    }
	    usleep(1000000.0 * update_interval);
	}
	else
	for (;;) {
	    if (key == '#') {
		while ((key = getch()) == ERR) {
		    if (curr_result == NULL) {
			mvprintw(1, 0, "Position: outside log bounds.");
		    }
		    else {
			mvprintw(1, 0, "Position: START+%.2lf secs. Press 'f' for forward or 'B' rewind to start",
			    tvsub(&curr_result->timestamp, &label.ll_start));
		    }
		    clrtoeol();
		    refresh();
		    usleep(1000);
		}
	    }
	    switch(key) {
	    case 'q':
	    case 'Q':
		move(lines, 0);
		clrtoeol();
		refresh();
		endwin();
		exit(0);

	    case 't':
	    case 'w':
	    case 'h':
	    case 's':
	    case 'n':
		sortmode = key;
		break;
	    case 'f':
		lockstat_fetch();
		break;
	    case 'B':
		pmSetMode(PM_MODE_INTERP, &label.ll_start,
		    (int)(1000.0 * update_interval));
		prev_result = curr_result = NULL;
		lockstat_fetch();
		break;
	    default:
		dohelp();
		key = '#';
		break;
	    }

	    if (key != '#')
	    	break;
	}
	key = '#';
    }
}


static pmValue *
find_result_value(pmResult *r, pmID id, int inst)
{
    int v, k;

    if (r != NULL) {
	/* find the metric */
	for (v=0; v < r->numpmid; v++) {
	    if (r->vset[v]->pmid == id) {
		for (k=0; k < r->vset[v]->numval; k++) {
		    if (r->vset[v]->vlist[k].inst == inst) {
			/* success */
		    	return &r->vset[v]->vlist[k];
		    }
		}
		break;
	    }
	}
    }

    /* fail */
    return (pmValue *)NULL;
}

int
cmp_sort(const void *b, const void *a)
{
    double da, db;
    double tca, tcb;

    tca = metrics[LS_NOWAIT].values[*(int *)a] +
    	metrics[LS_SPIN].values[*(int *)a];
    if (tca == 0)
    	tca = 1;
    tcb = metrics[LS_NOWAIT].values[*(int *)b] +
    	metrics[LS_SPIN].values[*(int *)b];
    if (tcb == 0)
    	tcb = 1;

    switch (sortmode) {
    case 't':
	da = metrics[LS_WAIT].values[*(int *)a];
	db = metrics[LS_WAIT].values[*(int *)b];
	break;
    case 'h':
	da = metrics[LS_HOLD].values[*(int *)a] / tca;
	db = metrics[LS_HOLD].values[*(int *)b] / tcb;
	break;
    case 'w':
	da = metrics[LS_WAIT].values[*(int *)a] / tca;
	db = metrics[LS_WAIT].values[*(int *)b] / tcb;
	break;
    case 'n':
	da = metrics[LS_NOWAIT].values[*(int *)a];
	db = metrics[LS_NOWAIT].values[*(int *)b];
	break;
    case 's':
	da = metrics[LS_SPIN].values[*(int *)a];
	db = metrics[LS_SPIN].values[*(int *)b];
	break;
    }

    if (da == db)
	return 0;
    else
    if (da > db)
    	return 1;
    return -1;
}

static void
lockstat_fetch()
{
    pmID		pmidlist[NMETRICS];
    int			err;
    int			m;
    int			i;
    char		*p;
    pmValue		*curr_value;
    pmValue		*prev_value;

    lasterror[0] = '\0';

    if (prev_result != NULL)
    	pmFreeResult(prev_result);
    prev_result = curr_result;
    curr_result = NULL;

    for (i=0; i < NMETRICS; i++) {
    	pmidlist[i] = metrics[i].desc.pmid;
    }

    if (archive) {
	if ((err = pmFetch(NMETRICS, pmidlist, &curr_result)) < 0) {
	    sprintf(lasterror, "Error: pmFetchArchive: %s", pmErrStr(err));
	    return;
	}
    }
    else {
	if ((err = pmFetch(NMETRICS, pmidlist, &curr_result)) < 0) {
	    sprintf(lasterror, "Error: pmFetch: %s", pmErrStr(err));
	    return;
	}
    }

    if (curr_result == NULL) {
	sprintf(lasterror, "Error: pmFetchArchive: %s", pmErrStr(err));
	return;
    }

    if (curr_result->vset[0]->numval < 0) {
	sprintf(lasterror, "Error: pmFetch: numval is negative: %s",
	    pmErrStr(curr_result->vset[0]->numval));
    	return;
    }

    strcpy(fetch_ctime, ctime(&curr_result->timestamp.tv_sec));
    if ((p = strrchr(fetch_ctime, '\n')) != NULL)
    	*p = '\0';

    if (curr_result && prev_result) {
	/* delta is in seconds */
	fetch_delta = tvsub(&curr_result->timestamp, &prev_result->timestamp);

	/*
	 * Extract values for all instances of all metrics
	 * and do rate conversion where necessary.
	 */
	for (m=0; m < NMETRICS; m++) {
	    for (i=0; i < n_instlist; i++) {
		curr_value = find_result_value(curr_result, metrics[m].desc.pmid, instlist[i]);
		if (curr_value == NULL)
		    continue;
		if (metrics[m].desc.sem == PM_SEM_COUNTER) {
		    /* counter: do rate conversion */
		    prev_value = find_result_value(prev_result, metrics[m].desc.pmid, instlist[i]);
		    if (curr_value == NULL || prev_value == NULL)
			metrics[m].values[i] = 0.0; /* TODO : error handling */
		    else {
			/*
			 * Save counter deltas. Rate conv is done later in report.
			 * This assumes all values are unsigned int (32bit)
			 */
			metrics[m].values[i] = (double)(
			    curr_value->value.lval - prev_value->value.lval);
		    }
		}
		else {
		    /* instant or discrete: just use the value */
		    metrics[m].values[i] = (double)curr_value->value.lval;
		}
	    }
	}

	/*
	 * sort by current sort mode
	 */
	qsort(sorted, n_instlist, sizeof(int), cmp_sort);
    }
}
