/**
 * OpenAL cross platform audio library
 * Copyright (C) 1999-2000 by authors.
 * This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the
 *  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 *  Boston, MA  02111-1307, USA.
 * Or go to http://www.gnu.org/copyleft/lgpl.html
 */

// includes
#include <Windows.h>
#include <ToolUtils.h>
#include <Controls.h>
#include <Stdio.h>
#include "openal.h"

// EAX-test
#define OPENAL
#include "eax.h"

// forward declarations
ALvoid LoadWAVFile(ALbyte *file,ALenum *format,ALvoid **data,ALsizei *size,ALsizei *freq);
ALvoid UnloadWAV(ALvoid *data);

// defines
#define MAXLONG			0x7FFFFFFF

#define rNewWindow		128 // window ID

#define rMenubar		128 // menu stuff
#define mApple			128
#define iAbout			1
#define mFile			129
#define iQuit			11
#define mEdit			130
#define mSelect			131

#define cButtonPlay		128 // control IDs
#define cButtonStart  	129
#define cButtonPause    130
#define cButtonStop		131
#define cButtonPlusX    132
#define cButtonMinusX   133
#define cButtonPlusY    134
#define cButtonMinusY   135
#define cButtonPlusZ    136
#define cButtonMinusZ   137
#define cSelect  		138

#define rTextStringList 128 // string resources


// typedefs
typedef struct
{
	ControlHandle hBtnPlay;
	ControlHandle hBtnStart;
	ControlHandle hBtnPause;
	ControlHandle hBtnStop;
	ControlHandle hBtnPlusX;
	ControlHandle hBtnMinusX;
	ControlHandle hBtnPlusY;
	ControlHandle hBtnMinusY;
	ControlHandle hBtnPlusZ;
	ControlHandle hBtnMinusZ;
	ControlHandle hSelect;
} DocRec; // holds control handles

typedef DocRec **DocRecHandle;

// prototypes
void Initmanagers(void);
void CreateMainMenu(void);
void CreateMainWindow(void);
void ProcessEvent(EventRecord *);
void ProcessMouseDown(EventRecord *);
void ProcessMenuChoice(SInt32);
void ProcessEvent (EventRecord *);
void Activate(EventRecord *);
void ActivateWindow(WindowPtr, Boolean);
void OSEvent(EventRecord *);
void ProcessControl(EventRecord *, WindowPtr);
void UpdateLoc();
void Update(EventRecord *);
void GetControls(WindowPtr);

// globals
Boolean gDone;  // indicates when main loop should be exited
Boolean gbInBackground;  // indicates when program is in background
unsigned int gSourceID[2]; // two source IDs
unsigned int gSampleSet[4]; // two sample set ID numbers
float gLoc0[3]; // location of source #0
PropSetFunction pfPropSet;
PropGetFunction pfPropGet;
Boolean bEAXDetected;

// main
int main(void)
{
	EventRecord	EventRec;
	DocRecHandle hDocRec; 
	char *pBuffer1, *pBuffer2;
	long lBuffer1Len, lBuffer2Len;
	ALenum formatBuffer1;
	ALsizei freqBuffer1;
	FSSpec sfFile;
	Str255 sFileName;
	short fRefNum;
	SInt16 iNumSources, iNumSampleSets;
	// EAX test
	unsigned long ulEAXVal;
	long lGlobalReverb;
	
	// init globals
	gDone = false; // we haven't started yet...
	gLoc0[0] = 0;
	gLoc0[1] = 0;
	gLoc0[2] = 0;
	
	// initialize OpenAL
	alInit(NULL, 0);
	// EAX test -- set EAX environment if EAX is available
	if (alIsExtensionPresent((ALubyte *)"EAX") == AL_TRUE)
	{
		pfPropSet = alGetProcAddress((ALubyte *)"EAXSet");
		if (pfPropSet != NULL)
		{
		    bEAXDetected = true;
		    
		    lGlobalReverb = 0; // covers attenuation range of -10000 (-100dB) to 0 (0dB)
			pfPropSet(PROPSETID_EAX_ListenerProperties, DSPROPERTY_EAXLISTENER_ROOM, 0, &lGlobalReverb, sizeof(unsigned long));
			ulEAXVal = EAX_ENVIRONMENT_GENERIC;
			pfPropSet(PROPSETID_EAX_ListenerProperties, DSPROPERTY_EAXLISTENER_ENVIRONMENT, 0, &ulEAXVal, sizeof(unsigned long));
		} else
		{
		    bEAXDetected = false;
		}
	}

	// load up some audio data...
	LoadWAVFile((char *)"\pboom.wav",&formatBuffer1, (void **) &pBuffer1,(unsigned int *)&lBuffer1Len,&freqBuffer1);
	
	lBuffer2Len = 1200000; // max amount of data to read
	pBuffer2 = NewPtrClear(lBuffer2Len); // create a long buffer 
	if (pBuffer2 == NULL)
	{
		gDone = true;  // don't let program run if can't get buffer space
	} else
	{
		if (FSMakeFSSpec(0,0,"\pfunk_s.pcm",&sfFile) == 0)
		{
			FSpOpenDF(&sfFile, fsRdPerm, &fRefNum); // load some data into the buffer
			FSRead(fRefNum, &lBuffer2Len, pBuffer2);			
 			FSClose(fRefNum);
		}
	}

	// generate two OpenAL sample sets and two sources
	iNumSources = 2;
	iNumSampleSets = 4;
	alGenSources(iNumSources, &gSourceID[0]);
	if (iNumSources == 2)  // FIXME: really should have error-checking here
	{
		alGenBuffers(iNumSampleSets, &gSampleSet[0]);
		if (iNumSampleSets == 4)  // FIXME: need error-checking here
		{
		    // now have two buffers and two sources, so need to fill buffers and attach to sources
			alBufferData(gSampleSet[0], formatBuffer1, pBuffer1, lBuffer1Len, freqBuffer1);
			alSourcei(gSourceID[0], AL_BUFFER, gSampleSet[0]);
			
			alBufferData(gSampleSet[1], AL_FORMAT_STEREO16, pBuffer2, lBuffer2Len/3, 22050);
			alBufferData(gSampleSet[2], AL_FORMAT_STEREO16, (pBuffer2 + (lBuffer2Len/3)), lBuffer2Len/3, 22050);
			alBufferData(gSampleSet[3], AL_FORMAT_STEREO16, (pBuffer2 + 2 * (lBuffer2Len/3)), lBuffer2Len/3, 22050);
			alQueuei(gSourceID[1], AL_BUFFER, gSampleSet[1]);
			alQueuei(gSourceID[1], AL_BUFFER, gSampleSet[2]);
			alQueuei(gSourceID[1], AL_BUFFER, gSampleSet[3]);
		}
	}
	 
	// get rid of un-needed buffer pointers
	UnloadWAV(pBuffer1);
	if (pBuffer2 != NULL)
	{
		DisposePtr(pBuffer2);
	}
	
	Initmanagers();
	CreateMainMenu();
	CreateMainWindow();
	
	// main event loop
	while (!gDone)  
	{
	 	if (WaitNextEvent(everyEvent,&EventRec,MAXLONG,NULL)) 
	 		ProcessEvent(&EventRec);
	}
	
	
	// terminate OpenAL
	alExit();

	return 0; 	
}

// initialize OS managers
void Initmanagers(void)
{
	MaxApplZone();
	MoreMasters();
	
	InitGraf(&qd.thePort);
	InitFonts();
	InitWindows();
	InitMenus();
	TEInit();
	InitDialogs(nil);
	InitCursor();
	
	FlushEvents(everyEvent, 0);
}

// create the main menu
void CreateMainMenu (void)
{
	Handle hMenuBar;
	MenuHandle hMenu;
	
	hMenuBar = GetNewMBar(rMenubar);
	if (hMenuBar == NULL)
	{
	 	ExitToShell ();
	}
	SetMenuBar(hMenuBar);
	DrawMenuBar();
	
	hMenu = GetMenuHandle(mApple);
	if (hMenu == NULL)
	{
	 	ExitToShell();
	} else
	{
		AppendResMenu(hMenu, 'DRVR');
	}
}
	 

// create the main window
void CreateMainWindow (void)
{
	WindowPtr	pWindow;
	DocRecHandle hDocRec;
	
	/*     // can create main window without a resource...
	Rect	windRect;
	windRect = qd.screenBits.bounds;
	InsetRect(&windRect, 50, 50);
	pWindow = NewCWindow(nil, &windRect, "\pMinApp", true, documentProc, 
						(WindowPtr) -1, false, 0);
	*/
	
	pWindow = GetNewWindow(rNewWindow, NULL, (pWindow) - 1);
		
	SetPort(pWindow);						// set window to current graf port
	
	if (!(hDocRec = (DocRecHandle) NewHandle(sizeof(DocRec))))
		ExitToShell();
		
	SetWRefCon(pWindow, (SInt32) hDocRec);  // attach control handle structure to the window
	GetControls(pWindow);  // get controls for window from the resource fork
	ShowWindow(pWindow);
}

// process an event
void ProcessEvent(EventRecord *pEventRec)
{
	SInt8 charCode;
	
	switch(pEventRec->what)
	{
		case mouseDown:
			ProcessMouseDown(pEventRec);
			break;
			
		case keyDown:
		case autoKey:
			charCode = pEventRec->message & charCodeMask;
			if((pEventRec->modifiers & cmdKey) != 0)
			{
				ProcessMenuChoice(MenuKey(charCode));
			}
			break;
			
		case activateEvt:
			Activate(pEventRec);
			break;
		
		case osEvt:
			OSEvent(pEventRec);
			HiliteMenu(0);
			break;
			
		case updateEvt:
			Update(pEventRec);
			break;
	}
}

// process mouseDown event
void ProcessMouseDown(EventRecord *pEventRec)
{
	WindowPtr	pWindow;
	SInt16		PartCode;

	PartCode = FindWindow(pEventRec->where, &pWindow);
	
	switch(PartCode)
	{
		case inMenuBar:
			ProcessMenuChoice(MenuSelect(pEventRec->where));
			break;
			
		case inSysWindow:
			SystemClick(pEventRec, pWindow);
			break;
			
		case inContent:
			if (pWindow != FrontWindow())
				SelectWindow(pWindow);
			else
				ProcessControl(pEventRec, pWindow);
			break;
			
		case inDrag:
			DragWindow(pWindow, pEventRec->where, &qd.screenBits.bounds);
			break;
			
		case inGoAway:
			if (TrackGoAway(pWindow, pEventRec->where) == true)
				gDone = true;
			break;
	}
}

// process menu selection
void ProcessMenuChoice(SInt32 MenuChoice)
{
	SInt16	MenuID, MenuItem;
	Str255	ItemName;
	SInt16	daDriverRefNum;

	
	MenuID = HiWord(MenuChoice);
	MenuItem = LoWord(MenuChoice);
	
	if (MenuID == 0) return;
	
	switch(MenuID)
	{
		case mApple:
		  	if (MenuItem == iAbout)
		  		SysBeep(10);
		  	else
		  	{
		  		GetMenuItemText(GetMenuHandle(mApple),MenuItem,ItemName);
		  		daDriverRefNum = OpenDeskAcc(ItemName);
		  	}
		  	break;
		  	
		 case mFile:
		 	if (MenuItem == iQuit)
		 		gDone = true;
		 	break;
	}
	
	HiliteMenu (0);
}

// Activate
void Activate (EventRecord *pEventRec)
{
	WindowPtr	pWindow;
	Boolean		bBecomingActive;
	
	pWindow = (WindowPtr) pEventRec->message;
	
	bBecomingActive = ((pEventRec->modifiers & activeFlag) == activeFlag);
	
	ActivateWindow(pWindow, bBecomingActive);
}

// ActivateWindow
void ActivateWindow (WindowPtr pWindow, Boolean bBecomingActive)
{
	DocRecHandle	hDocRec;
	SInt16	HiliteState;
	
	hDocRec = (DocRecHandle) (GetWRefCon(pWindow));
	
	if (bBecomingActive)
		HiliteState = 0;
	else
		HiliteState = 255;
		
	HiliteControl((*hDocRec)->hBtnPlay, HiliteState);
	HiliteControl((*hDocRec)->hBtnStart, HiliteState);
	HiliteControl((*hDocRec)->hBtnPause, HiliteState);
	HiliteControl((*hDocRec)->hBtnStop, HiliteState);
}

// OSEvent
void OSEvent(EventRecord *pEventRec)
{
	switch ((pEventRec->message >> 24) & 0x000000FF)
	{
		case suspendResumeMessage:
			// DrawGrowIcon(FrontWindow());
			gbInBackground = ((pEventRec->message & resumeFlag) == 0);
			ActivateWindow(FrontWindow(), !gbInBackground);
			break;
			
		case mouseMovedMessage:
			break;
	}
}

// ProcessControl -- figure out which control has been activated and react
void ProcessControl(EventRecord *pEventRec, WindowPtr pWindow)
{
	Point mouseXY;
	ControlPartCode partCode;
	ControlHandle hControl;
	DocRecHandle hDocRec;
	SInt16 controlValue;
	unsigned long ulEAXVal;
	ALint tempInt;
	
	mouseXY = pEventRec->where;
	GlobalToLocal(&mouseXY);
	
	if (partCode = FindControl(mouseXY, pWindow, &hControl))
	{
		hDocRec = (DocRecHandle) (GetWRefCon(pWindow));
		if (hControl == (*hDocRec)->hBtnPlay)
		{
			// play short sound if not already playing
			alGetSourcei(gSourceID[0], AL_SOURCE_STATE, &tempInt);
			if (tempInt != AL_PLAYING)
			{
				alSourcePlay(gSourceID[0]);
			}
		}
		if (hControl == (*hDocRec)->hBtnStart)
		{
			// play long stream
			alSourcei(gSourceID[1], AL_LOOPING, AL_TRUE);
			alSourcePlay(gSourceID[1]);
		}
		if (hControl == (*hDocRec)->hBtnPause)
		{
			alSourcePause(gSourceID[1]);
		}
		if (hControl == (*hDocRec)->hBtnStop)
		{
			alSourceStop(gSourceID[1]);
		}
		if (hControl == (*hDocRec)->hBtnPlusX)
		{
			gLoc0[0] += 1;
			alSourcefv(gSourceID[0], AL_POSITION, gLoc0);
			UpdateLoc();
		}
		if (hControl == (*hDocRec)->hBtnMinusX)
		{
			gLoc0[0] -= 1;
			alSourcefv(gSourceID[0], AL_POSITION, gLoc0);
			UpdateLoc();
		}
		if (hControl == (*hDocRec)->hBtnPlusY)
		{
			gLoc0[1] += 1;
			alSourcefv(gSourceID[0], AL_POSITION, gLoc0);
			UpdateLoc();
		}
		if (hControl == (*hDocRec)->hBtnMinusY)
		{
			gLoc0[1] -= 1;
			alSourcefv(gSourceID[0], AL_POSITION, gLoc0);
			UpdateLoc();
		}
		if (hControl == (*hDocRec)->hBtnPlusZ)
		{
			gLoc0[2] += 1;
			alSourcefv(gSourceID[0], AL_POSITION, gLoc0);
			UpdateLoc();
		}
		if (hControl == (*hDocRec)->hBtnMinusZ)
		{
			gLoc0[2] -= 1;
			alSourcefv(gSourceID[0], AL_POSITION, gLoc0);
			UpdateLoc();
		}
		if (hControl == (*hDocRec)->hSelect)
		{
			TrackControl(hControl,mouseXY,(ControlActionUPP) -1);
			controlValue = GetControlValue(hControl);
			if (alIsExtensionPresent((ALubyte *)"EAX") == AL_TRUE)
			{
				if (pfPropSet != NULL)
				{
					ulEAXVal = (unsigned long) controlValue - 1;
					pfPropSet(PROPSETID_EAX_ListenerProperties, DSPROPERTY_EAXLISTENER_ENVIRONMENT, 0, &ulEAXVal, sizeof(unsigned long));
				}
			}	
		}	
	}
}

// UpdateLoc -- update displayed location
void UpdateLoc ()
{
	Str255	textString;
	Rect    clearRect;
	
	SetRect(&clearRect, 80, 80, 190, 120);
	FillRect(&clearRect, &qd.white);

	sprintf((char *) textString, "((%d, %d, %d)", (int) gLoc0[0], (int) gLoc0[1], (int) gLoc0[2]);
	textString[0] = strlen(textString);
	MoveTo(95, 100);
	DrawString(textString);
}

// Update -- update window display
void Update (EventRecord *pEventRec)
{
	WindowPtr pWindow;
	Str255	textString;
	
	pWindow = (WindowPtr) pEventRec->message;
	
	BeginUpdate(pWindow); // NOTE:  BeginUpdate/EndUpdate are critical to process of putting a program in the background/foreground
	
	if (!EmptyRgn(pWindow->visRgn))
	{
		SetPort(pWindow);
		UpdateControls(pWindow, pWindow->visRgn);
		
		GetIndString(textString, rTextStringList, 1);
		MoveTo(45,45);
		DrawString(textString);
		
		GetIndString(textString, rTextStringList, 2);
		MoveTo(45,155);
		DrawString(textString);
		
		if (bEAXDetected == true)
		{
			GetIndString(textString, rTextStringList, 3);
			MoveTo(45,305);
			DrawString(textString);
		}
		
		UpdateLoc();
	}
	
	EndUpdate(pWindow);
}

// GetControls -- attach resources to hDocRec for window
void GetControls(WindowPtr pWindow)
{
	DocRecHandle hDocRec;
	
	hDocRec = (DocRecHandle) (GetWRefCon(pWindow));
	
	// put handles to the controls into hDocRec
	if (!((*hDocRec)->hBtnPlay = GetNewControl(cButtonPlay, pWindow)))
		ExitToShell();
		
	if (!((*hDocRec)->hBtnStart = GetNewControl(cButtonStart, pWindow)))
		ExitToShell();
		
	if (!((*hDocRec)->hBtnPause = GetNewControl(cButtonPause, pWindow)))
		ExitToShell();
		
	if (!((*hDocRec)->hBtnStop = GetNewControl(cButtonStop, pWindow)))
		ExitToShell();
		
	if (!((*hDocRec)->hBtnPlusX = GetNewControl(cButtonPlusX, pWindow)))
		ExitToShell();
		
	if (!((*hDocRec)->hBtnMinusX = GetNewControl(cButtonMinusX, pWindow)))
		ExitToShell();
		
	if (!((*hDocRec)->hBtnPlusY = GetNewControl(cButtonPlusY, pWindow)))
		ExitToShell();
		
	if (!((*hDocRec)->hBtnMinusY = GetNewControl(cButtonMinusY, pWindow)))
		ExitToShell();
	
	if (!((*hDocRec)->hBtnPlusZ = GetNewControl(cButtonPlusZ, pWindow)))
		ExitToShell();
		
	if (!((*hDocRec)->hBtnMinusZ = GetNewControl(cButtonMinusZ, pWindow)))
		ExitToShell();
	
	if (bEAXDetected == true)
	{	
		if (!((*hDocRec)->hSelect = GetNewControl(cSelect, pWindow)))
			ExitToShell();
    }
}

typedef struct                                  /* WAV File-header */
{
  ALubyte  Id[4];
  ALsizei  Size;
  ALubyte  Type[4];
} WAVFileHdr_Struct;

typedef struct                                  /* WAV Fmt-header */
{
  ALushort Format;                              
  ALushort Channels;
  ALuint   SamplesPerSec;
  ALuint   BytesPerSec;
  ALushort BlockAlign;
  ALushort BitsPerSample;
} WAVFmtHdr_Struct;

typedef struct									/* WAV FmtEx-header */
{
  ALushort Size;
  ALushort SamplesPerBlock;
} WAVFmtExHdr_Struct;

typedef struct                                  /* WAV Smpl-header */
{
  ALuint   Manufacturer;
  ALuint   Product;
  ALuint   SamplePeriod;                          
  ALuint   Note;                                  
  ALuint   FineTune;                              
  ALuint   SMPTEFormat;
  ALuint   SMPTEOffest;
  ALuint   Loops;
  ALuint   SamplerData;
  struct
  {
    ALuint Identifier;
    ALuint Type;
    ALuint Start;
    ALuint End;
    ALuint Fraction;
    ALuint Count;
  }      Loop[1];
} WAVSmplHdr_Struct;

typedef struct                                  /* WAV Chunk-header */
{
  ALubyte  Id[4];
  ALuint   Size;
} WAVChunkHdr_Struct;

void SwapWords(unsigned int *puint)
{
    unsigned int tempint;
	char *pChar1, *pChar2;
	
	tempint = *puint;
	pChar2 = (char *)&tempint;
	pChar1 = (char *)puint;
	
	pChar1[0]=pChar2[3];
	pChar1[1]=pChar2[2];
	pChar1[2]=pChar2[1];
	pChar1[3]=pChar2[0];
}

void SwapBytes(unsigned short *pshort)
{
    unsigned short tempshort;
    char *pChar1, *pChar2;
    
    tempshort = *pshort;
    pChar2 = (char *)&tempshort;
    pChar1 = (char *)pshort;
    
    pChar1[0]=pChar2[1];
    pChar1[1]=pChar2[0];
}
 
ALvoid LoadWAVFile(ALbyte *file,ALenum *format,ALvoid **data,ALsizei *size,ALsizei *freq)
{
	WAVChunkHdr_Struct ChunkHdr;
	WAVFmtExHdr_Struct FmtExHdr;
	WAVFileHdr_Struct FileHdr;
	WAVSmplHdr_Struct SmplHdr;
	WAVFmtHdr_Struct FmtHdr;
	FSSpec sfFile;
	short fRefNum;
	long numBytes;
	int i;
	
	*format=AL_FORMAT_MONO16;
	*data=NULL;
	*size=0;
	*freq=22050;
	if (file)
	{		
		if (FSMakeFSSpec(0,0,(const unsigned char *)file,&sfFile) == 0)
		{
		    FSpOpenDF(&sfFile, fsRdPerm, &fRefNum);
		    
		    numBytes = sizeof(WAVFileHdr_Struct);
		    FSRead(fRefNum, &numBytes, &FileHdr);
			SwapWords(&FileHdr.Size);
			FileHdr.Size=((FileHdr.Size+1)&~1)-4;
			while ((FileHdr.Size!=0))
			{
			    numBytes = sizeof(WAVChunkHdr_Struct);
			    FSRead(fRefNum, &numBytes, &ChunkHdr);
			    if (numBytes != sizeof(WAVChunkHdr_Struct)) break;
			    SwapWords(&ChunkHdr.Size);
			    
				if (!memcmp(ChunkHdr.Id,"fmt ",4))
				{
				    numBytes = sizeof(WAVFmtHdr_Struct);
				    FSRead(fRefNum, &numBytes, &FmtHdr);
				    SwapBytes(&FmtHdr.Format);
					if (FmtHdr.Format==0x0001)
					{
					    SwapBytes(&FmtHdr.Channels);
					    SwapBytes(&FmtHdr.BitsPerSample);
					    SwapWords(&FmtHdr.SamplesPerSec);
					    SwapBytes(&FmtHdr.BlockAlign);
					    
						*format=(FmtHdr.Channels==1?
								(FmtHdr.BitsPerSample==8?AL_FORMAT_MONO8:AL_FORMAT_MONO16):
								(FmtHdr.BitsPerSample==8?AL_FORMAT_STEREO8:AL_FORMAT_STEREO16));
						*freq=FmtHdr.SamplesPerSec;
						SetFPos(fRefNum, fsFromMark, sizeof(WAVFmtHdr_Struct)-ChunkHdr.Size);
					} 
					else
					{
					    numBytes = sizeof(WAVFmtExHdr_Struct);
					    FSRead(fRefNum, &numBytes, &FmtExHdr);
					    SetFPos(fRefNum, fsFromMark, ChunkHdr.Size-sizeof(WAVFmtHdr_Struct)-sizeof(WAVFmtExHdr_Struct));
					}
				}
				else if (!memcmp(ChunkHdr.Id,"data",4))
				{
					if (FmtHdr.Format==0x0001)
					{
						*size=ChunkHdr.Size;
						*data=NewPtrClear(ChunkHdr.Size);
						if (*data) 
						{
						    numBytes = ChunkHdr.Size;
						    FSRead(fRefNum, &numBytes, *data);
						    if (FmtHdr.BitsPerSample == 16) 
						    {
						        for (i = 0; i < (numBytes / 2); i++)
						        {
						        	SwapBytes(&(*(unsigned short **)data)[i]);
						        }
						    }
						}
					}
					else if (FmtHdr.Format==0x0011)
					{
						//IMA ADPCM
					}
					else if (FmtHdr.Format==0x0055)
					{
						//MP3 WAVE
					}
				}
				else if (!memcmp(ChunkHdr.Id,"smpl",4))
				{
				    numBytes = sizeof(WAVSmplHdr_Struct);
				    FSRead(fRefNum, &numBytes, &SmplHdr);
				    SetFPos(fRefNum, fsFromMark, ChunkHdr.Size-sizeof(WAVSmplHdr_Struct));
				}
				else SetFPos(fRefNum, fsFromMark,ChunkHdr.Size);
				SetFPos(fRefNum, fsFromMark, ChunkHdr.Size&1);
				FileHdr.Size-=(((ChunkHdr.Size+1)&~1)+8);
			}
		}
		FSClose(fRefNum);
	}
}

ALvoid UnloadWAV(ALvoid *data)
{
	if (data)
		DisposePtr(data);
}

