﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/drs/model/ProductCodeMode.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>

using namespace Aws::Utils;


namespace Aws
{
  namespace drs
  {
    namespace Model
    {
      namespace ProductCodeModeMapper
      {

        static const int ENABLED_HASH = HashingUtils::HashString("ENABLED");
        static const int DISABLED_HASH = HashingUtils::HashString("DISABLED");


        ProductCodeMode GetProductCodeModeForName(const Aws::String& name)
        {
          int hashCode = HashingUtils::HashString(name.c_str());
          if (hashCode == ENABLED_HASH)
          {
            return ProductCodeMode::ENABLED;
          }
          else if (hashCode == DISABLED_HASH)
          {
            return ProductCodeMode::DISABLED;
          }
          EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
          if(overflowContainer)
          {
            overflowContainer->StoreOverflow(hashCode, name);
            return static_cast<ProductCodeMode>(hashCode);
          }

          return ProductCodeMode::NOT_SET;
        }

        Aws::String GetNameForProductCodeMode(ProductCodeMode enumValue)
        {
          switch(enumValue)
          {
          case ProductCodeMode::NOT_SET:
            return {};
          case ProductCodeMode::ENABLED:
            return "ENABLED";
          case ProductCodeMode::DISABLED:
            return "DISABLED";
          default:
            EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
            if(overflowContainer)
            {
              return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
            }

            return {};
          }
        }

      } // namespace ProductCodeModeMapper
    } // namespace Model
  } // namespace drs
} // namespace Aws
