﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/dynamodbstreams/model/OperationType.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>

using namespace Aws::Utils;


namespace Aws
{
  namespace DynamoDBStreams
  {
    namespace Model
    {
      namespace OperationTypeMapper
      {

        static const int INSERT_HASH = HashingUtils::HashString("INSERT");
        static const int MODIFY_HASH = HashingUtils::HashString("MODIFY");
        static const int REMOVE_HASH = HashingUtils::HashString("REMOVE");


        OperationType GetOperationTypeForName(const Aws::String& name)
        {
          int hashCode = HashingUtils::HashString(name.c_str());
          if (hashCode == INSERT_HASH)
          {
            return OperationType::INSERT;
          }
          else if (hashCode == MODIFY_HASH)
          {
            return OperationType::MODIFY;
          }
          else if (hashCode == REMOVE_HASH)
          {
            return OperationType::REMOVE;
          }
          EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
          if(overflowContainer)
          {
            overflowContainer->StoreOverflow(hashCode, name);
            return static_cast<OperationType>(hashCode);
          }

          return OperationType::NOT_SET;
        }

        Aws::String GetNameForOperationType(OperationType enumValue)
        {
          switch(enumValue)
          {
          case OperationType::NOT_SET:
            return {};
          case OperationType::INSERT:
            return "INSERT";
          case OperationType::MODIFY:
            return "MODIFY";
          case OperationType::REMOVE:
            return "REMOVE";
          default:
            EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
            if(overflowContainer)
            {
              return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
            }

            return {};
          }
        }

      } // namespace OperationTypeMapper
    } // namespace Model
  } // namespace DynamoDBStreams
} // namespace Aws
