#include "stdafx.h"
#include "resource.h"
#include <cstddef>
#include <cassert>
#include <algorithm>
#include "BarcodeRendering.h"
#include "PictureWnd.h"


int CPictureWnd::OnCreate(LPCREATESTRUCT lpCreateStruct)
{
	CClientDC dc(*this);
	m_dc.CreateCompatibleDC(dc);

	return 0;
}

void CPictureWnd::OnDestroy()
{
	DestroyBitmap();
}

void CPictureWnd::DoPaint(CDCHandle dc)
{
	// Вывод с выравниванием изображения по центру окна
	CSize bmpSize;
	GetBitmapSize(&bmpSize);

	CRect clientRect;
	GetClientRect(clientRect);

	const int x0 = ((GetStyle() & WS_HSCROLL) == 0) ?
		(clientRect.Width() - bmpSize.cx) / 2: 0;

	const int y0 = ((GetStyle() & WS_VSCROLL) == 0) ?
		(clientRect.Height() - bmpSize.cy) / 2: 0;

	dc.BitBlt(x0, y0, bmpSize.cx, bmpSize.cy, m_dc, 0, 0, SRCCOPY);
}

void CPictureWnd::CreateBitmap(int width, int height)
{
	assert(m_bmp.IsNull());
	assert(width >= 0);
	assert(height >= 0);

	if (width == 0 || height == 0)
		return;

	CClientDC dc(*this);
	m_bmp.CreateCompatibleBitmap(dc, width, height);
	m_bmpOld = m_dc.SelectBitmap(m_bmp);
}

void CPictureWnd::RecreateBitmap(int width, int height)
{
	CSize bmpSize;
	GetBitmapSize(&bmpSize);

	if (bmpSize == CSize(width, height))
		return;

	DestroyBitmap();
	CreateBitmap(width, height);
}

void CPictureWnd::DestroyBitmap()
{
	if (m_bmp.IsNull())
		return;

	m_dc.SelectBitmap(m_bmpOld);
	m_bmp.DeleteObject();
}

void CPictureWnd::GetBitmapSize(SIZE *size) const
{
	assert(size != NULL);

	size->cx = 0;
	size->cy = 0;
	if (!m_bmp.IsNull())
		m_bmp.GetSize(*size);
}

void CPictureWnd::UpdateScrolls()
{
	CSize bmpSize;
	GetBitmapSize(&bmpSize);
	if (bmpSize.cx == 0 || bmpSize.cy == 0)
		bmpSize = CSize(1, 1);

	CRect clientRect;
	GetClientRect(clientRect);

	SetScrollSize(bmpSize, FALSE);
	SetScrollOffset(
		(bmpSize.cx - clientRect.Width()) / 2,
		(bmpSize.cy - clientRect.Height()) / 2);
}

void CPictureWnd::SetBarcode(const ag_matrix *pBarcode,
	UINT pointSizeInPixels, double pointSizeInMillimetres)
{
	assert(pBarcode != NULL);
	assert(pointSizeInPixels > 0);
	assert(pointSizeInMillimetres > 0);

	enum {borderCoeff = 3};
	enum {separatorCoeff = 3};
	const COLORREF rectColor = RGB(0xFF, 0x00, 0x00);

	const CSize virtualBarcodeSize(
		pointSizeInPixels * pBarcode->width,
		pointSizeInPixels * pBarcode->height);

	CSize realBarcodeSize(
		int(pointSizeInMillimetres * 100 * pBarcode->width),
		int(pointSizeInMillimetres * 100 * pBarcode->height));
	AtlHiMetricToPixel(&realBarcodeSize, &realBarcodeSize);

	const int maxBarcodePointSize = std::max(
		int(pointSizeInPixels),
		int((pBarcode->width > 0) ? realBarcodeSize.cx / pBarcode->width: 0));
	const int border = borderCoeff * maxBarcodePointSize;
	const int separator = separatorCoeff * maxBarcodePointSize;

	const CSize bmpSize(
		std::max(virtualBarcodeSize.cx, realBarcodeSize.cx) + 2 * border,
		virtualBarcodeSize.cy + separator + realBarcodeSize.cy + 2 * border);

	// Пересоздание битмапа
	RecreateBitmap(bmpSize.cx, bmpSize.cy);
	m_dc.FillRect(CRect(0, 0, bmpSize.cx, bmpSize.cy),
		GetWndClassInfo().m_wc.hbrBackground);

	// Отрисовка штрих-кода
	CPoint oldWindowOrg;

	m_dc.SetWindowOrg(
		-CPoint(
			(bmpSize.cx - virtualBarcodeSize.cx) / 2,
			border),
		&oldWindowOrg);
	DrawBarcode(HDC(m_dc), pBarcode,
		pointSizeInPixels, pointSizeInPixels);

	m_dc.SetWindowOrg(
		-CPoint(
			(bmpSize.cx - realBarcodeSize.cx) / 2,
			border + virtualBarcodeSize.cy + separator));
	m_dc.FillSolidRect(
		CRect(0, 0, realBarcodeSize.cx, realBarcodeSize.cy),
		rectColor);

	m_dc.SetWindowOrg(oldWindowOrg);

	// Обновление состояния полос прокрутки
	UpdateScrolls();
}

void CPictureWnd::ResetContent()
{
	DestroyBitmap();
	UpdateScrolls();
}
