/* SPDX-FileCopyrightText: 2008 Blender Authors
 *
 * SPDX-License-Identifier: GPL-2.0-or-later */

/** \file
 * \ingroup spgraph
 */

#include <cstdlib>

#include "DNA_scene_types.h"

#include "BLI_math_base.h"
#include "BLI_utildefines.h"

#include "BKE_context.hh"
#include "BKE_global.hh"

#include "UI_view2d.hh"

#include "ED_anim_api.hh"
#include "ED_screen.hh"

#include "graph_intern.hh"

#include "RNA_access.hh"
#include "RNA_define.hh"

#include "DEG_depsgraph.hh"

#include "WM_api.hh"
#include "WM_types.hh"

/* ************************** view-based operators **********************************/
/* XXX should these really be here? */

/* -------------------------------------------------------------------- */
/** \name Set Cursor
 *
 * The 'cursor' in the Graph Editor consists of two parts:
 * 1) Current Frame Indicator (as per ANIM_OT_change_frame)
 * 2) Value Indicator (stored per Graph Editor instance)
 * \{ */

static bool graphview_cursor_poll(bContext *C)
{
  /* prevent changes during render */
  if (G.is_rendering) {
    return false;
  }

  return ED_operator_graphedit_active(C);
}

/* Set the new frame number */
static void graphview_cursor_apply(bContext *C, wmOperator *op)
{
  Scene *scene = CTX_data_scene(C);
  SpaceGraph *sipo = CTX_wm_space_graph(C);
  /* this isn't technically "frame", but it'll do... */
  float frame = RNA_float_get(op->ptr, "frame");

  /* adjust the frame or the cursor x-value */
  if (sipo->mode == SIPO_MODE_DRIVERS) {
    /* adjust cursor x-value */
    sipo->cursorTime = frame;
  }
  else {
    /* adjust the frame
     * NOTE: sync this part of the code with ANIM_OT_change_frame
     */
    /* 1) frame is rounded to the nearest int, since frames are ints */
    scene->r.cfra = round_fl_to_int(frame);

    if (scene->r.flag & SCER_LOCK_FRAME_SELECTION) {
      /* Clip to preview range
       * NOTE: Preview range won't go into negative values,
       *       so only clamping once should be fine.
       */
      CLAMP(scene->r.cfra, PSFRA, PEFRA);
    }
    else {
      /* Prevent negative frames */
      FRAMENUMBER_MIN_CLAMP(scene->r.cfra);
    }

    scene->r.subframe = 0.0f;
    DEG_id_tag_update(&scene->id, ID_RECALC_FRAME_CHANGE);
  }

  /* set the cursor value */
  sipo->cursorVal = RNA_float_get(op->ptr, "value");

  /* send notifiers - notifiers for frame should force an update for both vars ok... */
  WM_event_add_notifier(C, NC_SCENE | ND_FRAME, scene);
}

/* ... */

/* Non-modal callback for running operator without user input */
static int graphview_cursor_exec(bContext *C, wmOperator *op)
{
  graphview_cursor_apply(C, op);
  return OPERATOR_FINISHED;
}

/* ... */

/* set the operator properties from the initial event */
static void graphview_cursor_setprops(bContext *C, wmOperator *op, const wmEvent *event)
{
  ARegion *region = CTX_wm_region(C);
  float viewx, viewy;

  /* abort if not active region (should not really be possible) */
  if (region == nullptr) {
    return;
  }

  /* convert from region coordinates to View2D 'tot' space */
  UI_view2d_region_to_view(&region->v2d, event->mval[0], event->mval[1], &viewx, &viewy);

  /* store the values in the operator properties */
  /* NOTE: we don't clamp frame here, as it might be used for the drivers cursor */
  RNA_float_set(op->ptr, "frame", viewx);
  RNA_float_set(op->ptr, "value", viewy);
}

/* Modal Operator init */
static int graphview_cursor_invoke(bContext *C, wmOperator *op, const wmEvent *event)
{
  bScreen *screen = CTX_wm_screen(C);

  /* Change to frame that mouse is over before adding modal handler,
   * as user could click on a single frame (jump to frame) as well as
   * click-dragging over a range (modal scrubbing). Apply this change.
   */
  graphview_cursor_setprops(C, op, event);
  graphview_cursor_apply(C, op);

  /* Signal that a scrubbing operating is starting */
  if (screen) {
    screen->scrubbing = true;
  }

  /* add temp handler */
  WM_event_add_modal_handler(C, op);
  return OPERATOR_RUNNING_MODAL;
}

/* Modal event handling of cursor changing */
static int graphview_cursor_modal(bContext *C, wmOperator *op, const wmEvent *event)
{
  bScreen *screen = CTX_wm_screen(C);
  Scene *scene = CTX_data_scene(C);

  /* execute the events */
  switch (event->type) {
    case EVT_ESCKEY:
      if (screen) {
        screen->scrubbing = false;
      }

      WM_event_add_notifier(C, NC_SCENE | ND_FRAME, scene);
      return OPERATOR_FINISHED;

    case MOUSEMOVE:
      /* set the new values */
      graphview_cursor_setprops(C, op, event);
      graphview_cursor_apply(C, op);
      break;

    case LEFTMOUSE:
    case RIGHTMOUSE:
    case MIDDLEMOUSE:
      /* We check for either mouse-button to end, to work with all user keymaps. */
      if (event->val == KM_RELEASE) {
        if (screen) {
          screen->scrubbing = false;
        }

        WM_event_add_notifier(C, NC_SCENE | ND_FRAME, scene);
        return OPERATOR_FINISHED;
      }
      break;
  }

  return OPERATOR_RUNNING_MODAL;
}

static void GRAPH_OT_cursor_set(wmOperatorType *ot)
{
  /* identifiers */
  ot->name = "Set Cursor";
  ot->idname = "GRAPH_OT_cursor_set";
  ot->description = "Interactively set the current frame and value cursor";

  /* api callbacks */
  ot->exec = graphview_cursor_exec;
  ot->invoke = graphview_cursor_invoke;
  ot->modal = graphview_cursor_modal;
  ot->poll = graphview_cursor_poll;

  /* flags */
  ot->flag = OPTYPE_BLOCKING | OPTYPE_GRAB_CURSOR_X | OPTYPE_UNDO;

  /* rna */
  RNA_def_float(ot->srna, "frame", 0, MINAFRAMEF, MAXFRAMEF, "Frame", "", MINAFRAMEF, MAXFRAMEF);
  RNA_def_float(ot->srna, "value", 0, -FLT_MAX, FLT_MAX, "Value", "", -100.0f, 100.0f);
}

/** \} */

/* -------------------------------------------------------------------- */
/** \name Hide/Reveal
 * \{ */

static int graphview_curves_hide_exec(bContext *C, wmOperator *op)
{
  bAnimContext ac;
  ListBase anim_data = {nullptr, nullptr};
  ListBase all_data = {nullptr, nullptr};
  int filter;
  const bool unselected = RNA_boolean_get(op->ptr, "unselected");

  /* get editor data */
  if (ANIM_animdata_get_context(C, &ac) == 0) {
    return OPERATOR_CANCELLED;
  }

  /* get list of all channels that selection may need to be flushed to
   * - hierarchy must not affect what we have access to here...
   */
  filter = (ANIMFILTER_DATA_VISIBLE | ANIMFILTER_FCURVESONLY | ANIMFILTER_LIST_CHANNELS |
            ANIMFILTER_NODUPLIS);
  ANIM_animdata_filter(
      &ac, &all_data, eAnimFilter_Flags(filter), ac.data, eAnimCont_Types(ac.datatype));

  /* filter data
   * - of the remaining visible curves, we want to hide the ones that are
   *   selected/unselected (depending on "unselected" prop)
   */
  filter = (ANIMFILTER_DATA_VISIBLE | ANIMFILTER_FCURVESONLY | ANIMFILTER_CURVE_VISIBLE |
            ANIMFILTER_NODUPLIS);
  if (unselected) {
    filter |= ANIMFILTER_UNSEL;
  }
  else {
    filter |= ANIMFILTER_SEL;
  }

  ANIM_animdata_filter(
      &ac, &anim_data, eAnimFilter_Flags(filter), ac.data, eAnimCont_Types(ac.datatype));

  LISTBASE_FOREACH (bAnimListElem *, ale, &anim_data) {
    /* hack: skip object channels for now, since flushing those will always flush everything,
     * but they are always included */
    /* TODO: find out why this is the case, and fix that */
    if (ale->type == ANIMTYPE_OBJECT) {
      continue;
    }

    /* change the hide setting, and unselect it... */
    ANIM_channel_setting_set(&ac, ale, ACHANNEL_SETTING_VISIBLE, ACHANNEL_SETFLAG_CLEAR);
    ANIM_channel_setting_set(&ac, ale, ACHANNEL_SETTING_SELECT, ACHANNEL_SETFLAG_CLEAR);

    /* now, also flush selection status up/down as appropriate */
    ANIM_flush_setting_anim_channels(
        &ac, &all_data, ale, ACHANNEL_SETTING_VISIBLE, ACHANNEL_SETFLAG_CLEAR);
  }

  /* cleanup */
  ANIM_animdata_freelist(&anim_data);
  BLI_freelistN(&all_data);

  /* unhide selected */
  if (unselected) {
    /* turn off requirement for visible */
    filter = ANIMFILTER_SEL | ANIMFILTER_NODUPLIS | ANIMFILTER_LIST_CHANNELS |
             ANIMFILTER_FCURVESONLY;

    /* flushing has been done */
    ANIM_animdata_filter(
        &ac, &anim_data, eAnimFilter_Flags(filter), ac.data, eAnimCont_Types(ac.datatype));

    LISTBASE_FOREACH (bAnimListElem *, ale, &anim_data) {
      /* hack: skip object channels for now, since flushing those
       * will always flush everything, but they are always included */

      /* TODO: find out why this is the case, and fix that */
      if (ale->type == ANIMTYPE_OBJECT) {
        continue;
      }

      /* change the hide setting, and unselect it... */
      ANIM_channel_setting_set(&ac, ale, ACHANNEL_SETTING_VISIBLE, ACHANNEL_SETFLAG_ADD);
      ANIM_channel_setting_set(&ac, ale, ACHANNEL_SETTING_SELECT, ACHANNEL_SETFLAG_ADD);

      /* now, also flush selection status up/down as appropriate */
      ANIM_flush_setting_anim_channels(
          &ac, &anim_data, ale, ACHANNEL_SETTING_VISIBLE, ACHANNEL_SETFLAG_ADD);
    }
    ANIM_animdata_freelist(&anim_data);
  }

  /* send notifier that things have changed */
  WM_event_add_notifier(C, NC_ANIMATION | ND_ANIMCHAN | NA_EDITED, nullptr);

  return OPERATOR_FINISHED;
}

static void GRAPH_OT_hide(wmOperatorType *ot)
{
  /* identifiers */
  ot->name = "Hide Curves";
  ot->idname = "GRAPH_OT_hide";
  ot->description = "Hide selected curves from Graph Editor view";

  /* api callbacks */
  ot->exec = graphview_curves_hide_exec;
  ot->poll = ED_operator_graphedit_active;

  /* flags */
  ot->flag = OPTYPE_REGISTER | OPTYPE_UNDO;

  /* props */
  RNA_def_boolean(
      ot->srna, "unselected", false, "Unselected", "Hide unselected rather than selected curves");
}

/* ........ */

static int graphview_curves_reveal_exec(bContext *C, wmOperator *op)
{
  bAnimContext ac;
  ListBase anim_data = {nullptr, nullptr};
  ListBase all_data = {nullptr, nullptr};
  int filter;
  const bool select = RNA_boolean_get(op->ptr, "select");

  /* get editor data */
  if (ANIM_animdata_get_context(C, &ac) == 0) {
    return OPERATOR_CANCELLED;
  }

  /* get list of all channels that selection may need to be flushed to
   * - hierarchy must not affect what we have access to here...
   */
  filter = (ANIMFILTER_DATA_VISIBLE | ANIMFILTER_LIST_CHANNELS | ANIMFILTER_NODUPLIS |
            ANIMFILTER_FCURVESONLY);
  ANIM_animdata_filter(
      &ac, &all_data, eAnimFilter_Flags(filter), ac.data, eAnimCont_Types(ac.datatype));

  /* filter data
   * - just go through all visible channels, ensuring that everything is set to be curve-visible
   */
  filter = (ANIMFILTER_DATA_VISIBLE | ANIMFILTER_LIST_VISIBLE | ANIMFILTER_NODUPLIS |
            ANIMFILTER_FCURVESONLY);
  ANIM_animdata_filter(
      &ac, &anim_data, eAnimFilter_Flags(filter), ac.data, eAnimCont_Types(ac.datatype));

  LISTBASE_FOREACH (bAnimListElem *, ale, &anim_data) {
    /* hack: skip object channels for now, since flushing those will always flush everything,
     * but they are always included. */
    /* TODO: find out why this is the case, and fix that */
    if (ale->type == ANIMTYPE_OBJECT) {
      continue;
    }

    /* select if it is not visible */
    if (ANIM_channel_setting_get(&ac, ale, ACHANNEL_SETTING_VISIBLE) == 0) {
      ANIM_channel_setting_set(&ac,
                               ale,
                               ACHANNEL_SETTING_SELECT,
                               select ? ACHANNEL_SETFLAG_ADD : ACHANNEL_SETFLAG_CLEAR);
    }

    /* change the visibility setting */
    ANIM_channel_setting_set(&ac, ale, ACHANNEL_SETTING_VISIBLE, ACHANNEL_SETFLAG_ADD);

    /* now, also flush selection status up/down as appropriate */
    ANIM_flush_setting_anim_channels(
        &ac, &all_data, ale, ACHANNEL_SETTING_VISIBLE, eAnimChannels_SetFlag(true));
  }

  /* cleanup */
  ANIM_animdata_freelist(&anim_data);
  BLI_freelistN(&all_data);

  /* send notifier that things have changed */
  WM_event_add_notifier(C, NC_ANIMATION | ND_ANIMCHAN | NA_EDITED, nullptr);

  return OPERATOR_FINISHED;
}

static void GRAPH_OT_reveal(wmOperatorType *ot)
{
  /* identifiers */
  ot->name = "Reveal Curves";
  ot->idname = "GRAPH_OT_reveal";
  ot->description = "Make previously hidden curves visible again in Graph Editor view";

  /* api callbacks */
  ot->exec = graphview_curves_reveal_exec;
  ot->poll = ED_operator_graphedit_active;

  /* flags */
  ot->flag = OPTYPE_REGISTER | OPTYPE_UNDO;

  RNA_def_boolean(ot->srna, "select", true, "Select", "");
}

/** \} */

/* -------------------------------------------------------------------- */
/** \name Registration: operator types
 * \{ */

void graphedit_operatortypes()
{
  /* view */
  WM_operatortype_append(GRAPH_OT_cursor_set);

  WM_operatortype_append(GRAPH_OT_previewrange_set);
  WM_operatortype_append(GRAPH_OT_view_all);
  WM_operatortype_append(GRAPH_OT_view_selected);
  WM_operatortype_append(GRAPH_OT_view_frame);

  WM_operatortype_append(GRAPH_OT_ghost_curves_create);
  WM_operatortype_append(GRAPH_OT_ghost_curves_clear);

  WM_operatortype_append(GRAPH_OT_hide);
  WM_operatortype_append(GRAPH_OT_reveal);

  /* keyframes */
  /* selection */
  WM_operatortype_append(GRAPH_OT_clickselect);
  WM_operatortype_append(GRAPH_OT_select_all);
  WM_operatortype_append(GRAPH_OT_select_box);
  WM_operatortype_append(GRAPH_OT_select_lasso);
  WM_operatortype_append(GRAPH_OT_select_circle);
  WM_operatortype_append(GRAPH_OT_select_column);
  WM_operatortype_append(GRAPH_OT_select_linked);
  WM_operatortype_append(GRAPH_OT_select_more);
  WM_operatortype_append(GRAPH_OT_select_less);
  WM_operatortype_append(GRAPH_OT_select_leftright);
  WM_operatortype_append(GRAPH_OT_select_key_handles);

  /* editing */
  WM_operatortype_append(GRAPH_OT_snap);
  WM_operatortype_append(GRAPH_OT_equalize_handles);
  WM_operatortype_append(GRAPH_OT_mirror);
  WM_operatortype_append(GRAPH_OT_frame_jump);
  WM_operatortype_append(GRAPH_OT_keyframe_jump);
  WM_operatortype_append(GRAPH_OT_snap_cursor_value);
  WM_operatortype_append(GRAPH_OT_handle_type);
  WM_operatortype_append(GRAPH_OT_interpolation_type);
  WM_operatortype_append(GRAPH_OT_extrapolation_type);
  WM_operatortype_append(GRAPH_OT_easing_type);
  WM_operatortype_append(GRAPH_OT_bake_keys);
  WM_operatortype_append(GRAPH_OT_keys_to_samples);
  WM_operatortype_append(GRAPH_OT_samples_to_keys);
  WM_operatortype_append(GRAPH_OT_sound_to_samples);
  WM_operatortype_append(GRAPH_OT_smooth);
  WM_operatortype_append(GRAPH_OT_clean);
  WM_operatortype_append(GRAPH_OT_decimate);
  WM_operatortype_append(GRAPH_OT_blend_to_neighbor);
  WM_operatortype_append(GRAPH_OT_breakdown);
  WM_operatortype_append(GRAPH_OT_ease);
  WM_operatortype_append(GRAPH_OT_shear);
  WM_operatortype_append(GRAPH_OT_scale_average);
  WM_operatortype_append(GRAPH_OT_scale_from_neighbor);
  WM_operatortype_append(GRAPH_OT_blend_offset);
  WM_operatortype_append(GRAPH_OT_blend_to_ease);
  WM_operatortype_append(GRAPH_OT_match_slope);
  WM_operatortype_append(GRAPH_OT_time_offset);
  WM_operatortype_append(GRAPH_OT_blend_to_default);
  WM_operatortype_append(GRAPH_OT_push_pull);
  WM_operatortype_append(GRAPH_OT_gaussian_smooth);
  WM_operatortype_append(GRAPH_OT_butterworth_smooth);
  WM_operatortype_append(GRAPH_OT_euler_filter);
  WM_operatortype_append(GRAPH_OT_delete);
  WM_operatortype_append(GRAPH_OT_duplicate);

  WM_operatortype_append(GRAPH_OT_copy);
  WM_operatortype_append(GRAPH_OT_paste);

  WM_operatortype_append(GRAPH_OT_keyframe_insert);
  WM_operatortype_append(GRAPH_OT_click_insert);

  /* F-Curve Modifiers */
  WM_operatortype_append(GRAPH_OT_fmodifier_add);
  WM_operatortype_append(GRAPH_OT_fmodifier_copy);
  WM_operatortype_append(GRAPH_OT_fmodifier_paste);

  /* Drivers */
  WM_operatortype_append(GRAPH_OT_driver_variables_copy);
  WM_operatortype_append(GRAPH_OT_driver_variables_paste);
  WM_operatortype_append(GRAPH_OT_driver_delete_invalid);
}

void ED_operatormacros_graph()
{
  wmOperatorType *ot;
  wmOperatorTypeMacro *otmacro;

  ot = WM_operatortype_append_macro("GRAPH_OT_duplicate_move",
                                    "Duplicate",
                                    "Make a copy of all selected keyframes and move them",
                                    OPTYPE_UNDO | OPTYPE_REGISTER);
  WM_operatortype_macro_define(ot, "GRAPH_OT_duplicate");
  otmacro = WM_operatortype_macro_define(ot, "TRANSFORM_OT_translate");
  RNA_boolean_set(otmacro->ptr, "use_duplicated_keyframes", true);
  RNA_boolean_set(otmacro->ptr, "use_proportional_edit", false);
}

/** \} */

/* -------------------------------------------------------------------- */
/** \name Registration: Key-Maps
 * \{ */

void graphedit_keymap(wmKeyConfig *keyconf)
{
  /* keymap for all regions */
  WM_keymap_ensure(keyconf, "Graph Editor Generic", SPACE_GRAPH, RGN_TYPE_WINDOW);

  /* channels */
  /* Channels are not directly handled by the Graph Editor module,
   * but are inherited from the Animation module.
   * All the relevant operations, keymaps, drawing, etc.
   * can therefore all be found in that module instead,
   * as these are all used for the Graph Editor too.
   */

  /* keyframes */
  WM_keymap_ensure(keyconf, "Graph Editor", SPACE_GRAPH, RGN_TYPE_WINDOW);
}

/** \} */
