! This file is part of s-dftd3.
! SPDX-Identifier: LGPL-3.0-or-later
!
! s-dftd3 is free software: you can redistribute it and/or modify it under
! the terms of the GNU Lesser General Public License as published by
! the Free Software Foundation, either version 3 of the License, or
! (at your option) any later version.
!
! s-dftd3 is distributed in the hope that it will be useful,
! but WITHOUT ANY WARRANTY; without even the implied warranty of
! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
! GNU Lesser General Public License for more details.
!
! You should have received a copy of the GNU Lesser General Public License
! along with s-dftd3.  If not, see <https://www.gnu.org/licenses/>.

module dftd3_reference
   use mctc_env, only : wp
   implicit none
   private

   public :: number_of_references, reference_cn
   public :: get_c6
   public :: copy_c6, init_reference_c6, print_c6


   integer, parameter :: max_elem = 94
   integer, parameter :: max_ref = 5

   integer, parameter :: number_of_references(max_elem) = [ &
      &  2, 1, & ! H,He
      &  2, 3, 5, 5, 4, 3, 2, 1, & ! Li-Ne
      &  2, 3, 4, 5, 4, 3, 2, 1, & ! Na-Ar
      &  2, 3, & ! K,Ca
      &        3, 3, 3, 3, 3, 3, 4, 4, 2, 2, & ! Sc-Zn
      &        4, 5, 4, 3, 2, 1, & ! Ga-Kr
      &  2, 3, & ! Rb,Sr
      &        3, 3, 3, 3, 3, 3, 3, 3, 2, 2, & ! Y-Cd
      &        4, 5, 4, 3, 2, 1, & ! In-Xe
      &  2, 3, & ! Cs,Ba
      &     3, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, & ! La-Eu
      &        2, 3, 3, 3, 3, 3, 3, 3, 2, 2, & ! Lu-Hg
      &        4, 5, 4, 3, 2, 1, & ! Tl-Rn
      &  2, 3, & ! Fr,Ra
      &     2, 2, 2, 2, 2, 2] ! Ac-Pu

   real(wp), parameter :: reference_cn(max_ref, max_elem) = reshape([ &
      & 0.9118_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! H
      & 0.0000_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! He
      & 0.0000_wp,  0.9865_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Li
      & 0.0000_wp,  0.9808_wp,  1.9697_wp,  0.0000_wp,  0.0000_wp, & ! Be
      & 0.0000_wp,  0.9706_wp,  1.9441_wp,  2.9128_wp,  4.5856_wp, & ! B
      & 0.0000_wp,  0.9868_wp,  1.9985_wp,  2.9987_wp,  3.9844_wp, & ! C
      & 0.0000_wp,  0.9944_wp,  2.0143_wp,  2.9903_wp,  0.0000_wp, & ! N
      & 0.0000_wp,  0.9925_wp,  1.9887_wp,  0.0000_wp,  0.0000_wp, & ! O
      & 0.0000_wp,  0.9982_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! F
      & 0.0000_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Ne
      & 0.0000_wp,  0.9684_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Na
      & 0.0000_wp,  0.9628_wp,  1.9496_wp,  0.0000_wp,  0.0000_wp, & ! Mg
      & 0.0000_wp,  0.9648_wp,  1.9311_wp,  2.9146_wp,  0.0000_wp, & ! Al
      & 0.0000_wp,  0.9507_wp,  1.9435_wp,  2.9407_wp,  3.8677_wp, & ! Si
      & 0.0000_wp,  0.9947_wp,  2.0102_wp,  2.9859_wp,  0.0000_wp, & ! P
      & 0.0000_wp,  0.9948_wp,  1.9903_wp,  0.0000_wp,  0.0000_wp, & ! S
      & 0.0000_wp,  0.9972_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Cl
      & 0.0000_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Ar
      & 0.0000_wp,  0.9767_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! K
      & 0.0000_wp,  0.9831_wp,  1.9349_wp,  0.0000_wp,  0.0000_wp, & ! Ca
      & 0.0000_wp,  1.8627_wp,  2.8999_wp,  0.0000_wp,  0.0000_wp, & ! Sc
      & 0.0000_wp,  1.8299_wp,  3.8675_wp,  0.0000_wp,  0.0000_wp, & ! Ti
      & 0.0000_wp,  1.9138_wp,  2.9110_wp,  0.0000_wp,  0.0000_wp, & ! V
      & 0.0000_wp,  1.8269_wp, 10.6191_wp,  0.0000_wp,  0.0000_wp, & ! Cr
      & 0.0000_wp,  1.6406_wp,  9.8849_wp,  0.0000_wp,  0.0000_wp, & ! Mn
      & 0.0000_wp,  1.6483_wp,  9.1376_wp,  0.0000_wp,  0.0000_wp, & ! Fe
      & 0.0000_wp,  1.7149_wp,  2.9263_wp,  7.7785_wp,  0.0000_wp, & ! Co
      & 0.0000_wp,  1.7937_wp,  6.5458_wp,  6.2918_wp,  0.0000_wp, & ! Ni
      & 0.0000_wp,  0.9576_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Cu
      & 0.0000_wp,  1.9419_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Zn
      & 0.0000_wp,  0.9601_wp,  1.9315_wp,  2.9233_wp,  0.0000_wp, & ! Ga
      & 0.0000_wp,  0.9434_wp,  1.9447_wp,  2.9186_wp,  3.8972_wp, & ! Ge
      & 0.0000_wp,  0.9889_wp,  1.9793_wp,  2.9709_wp,  0.0000_wp, & ! As
      & 0.0000_wp,  0.9901_wp,  1.9812_wp,  0.0000_wp,  0.0000_wp, & ! Se
      & 0.0000_wp,  0.9974_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Br
      & 0.0000_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Kr
      & 0.0000_wp,  0.9738_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Rb
      & 0.0000_wp,  0.9801_wp,  1.9143_wp,  0.0000_wp,  0.0000_wp, & ! Sr
      & 0.0000_wp,  1.9153_wp,  2.8903_wp,  0.0000_wp,  0.0000_wp, & ! Y
      & 0.0000_wp,  1.9355_wp,  3.9106_wp,  0.0000_wp,  0.0000_wp, & ! Zr
      & 0.0000_wp,  1.9545_wp,  2.9225_wp,  0.0000_wp,  0.0000_wp, & ! Nb
      & 0.0000_wp,  1.9420_wp, 11.0556_wp,  0.0000_wp,  0.0000_wp, & ! Mo
      & 0.0000_wp,  1.6682_wp,  9.5402_wp,  0.0000_wp,  0.0000_wp, & ! Tc
      & 0.0000_wp,  1.8584_wp,  8.8895_wp,  0.0000_wp,  0.0000_wp, & ! Ru
      & 0.0000_wp,  1.9003_wp,  2.9696_wp,  0.0000_wp,  0.0000_wp, & ! Rh
      & 0.0000_wp,  1.8630_wp,  5.7095_wp,  0.0000_wp,  0.0000_wp, & ! Pd
      & 0.0000_wp,  0.9679_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Ag
      & 0.0000_wp,  1.9539_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Cd
      & 0.0000_wp,  0.9633_wp,  1.9378_wp,  2.9353_wp,  0.0000_wp, & ! In
      & 0.0000_wp,  0.9514_wp,  1.9505_wp,  2.9259_wp,  3.9123_wp, & ! Sn
      & 0.0000_wp,  0.9749_wp,  1.9523_wp,  2.9315_wp,  0.0000_wp, & ! Sb
      & 0.0000_wp,  0.9811_wp,  1.9639_wp,  0.0000_wp,  0.0000_wp, & ! Te
      & 0.0000_wp,  0.9968_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! I
      & 0.0000_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Xe
      & 0.0000_wp,  0.9909_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Cs
      & 0.0000_wp,  0.9797_wp,  1.8467_wp,  0.0000_wp,  0.0000_wp, & ! Ba
      & 0.0000_wp,  1.9373_wp,  2.9175_wp,  0.0000_wp,  0.0000_wp, & ! La
      & 2.7991_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Ce
      & 0.0000_wp,  2.9425_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Pr
      & 0.0000_wp,  2.9455_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Nd
      & 0.0000_wp,  2.9413_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Pm
      & 0.0000_wp,  2.9300_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Sm
      & 0.0000_wp,  1.8286_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Eu
      & 0.0000_wp,  2.8732_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Gd
      & 0.0000_wp,  2.9086_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Tb
      & 0.0000_wp,  2.8965_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Dy
      & 0.0000_wp,  2.9242_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Ho
      & 0.0000_wp,  2.9282_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Er
      & 0.0000_wp,  2.9246_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Tm
      & 0.0000_wp,  2.8482_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Yb
      & 0.0000_wp,  2.9219_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Lu
      & 0.0000_wp,  1.9254_wp,  3.8840_wp,  0.0000_wp,  0.0000_wp, & ! Hf
      & 0.0000_wp,  1.9459_wp,  2.8988_wp,  0.0000_wp,  0.0000_wp, & ! Ta
      & 0.0000_wp,  1.9292_wp, 10.9153_wp,  0.0000_wp,  0.0000_wp, & ! W
      & 0.0000_wp,  1.8104_wp,  9.8054_wp,  0.0000_wp,  0.0000_wp, & ! Re
      & 0.0000_wp,  1.8858_wp,  9.1527_wp,  0.0000_wp,  0.0000_wp, & ! Os
      & 0.0000_wp,  1.8648_wp,  2.9424_wp,  0.0000_wp,  0.0000_wp, & ! Ir
      & 0.0000_wp,  1.9188_wp,  6.6669_wp,  0.0000_wp,  0.0000_wp, & ! Pt
      & 0.0000_wp,  0.9846_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Au
      & 0.0000_wp,  1.9896_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Hg
      & 0.0000_wp,  0.9267_wp,  1.9302_wp,  2.9420_wp,  0.0000_wp, & ! Tl
      & 0.0000_wp,  0.9383_wp,  1.9356_wp,  2.9081_wp,  3.9098_wp, & ! Pb
      & 0.0000_wp,  0.9820_wp,  1.9655_wp,  2.9500_wp,  0.0000_wp, & ! Bi
      & 0.0000_wp,  0.9815_wp,  1.9639_wp,  0.0000_wp,  0.0000_wp, & ! Po
      & 0.0000_wp,  0.9954_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! At
      & 0.0000_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Rn
      & 0.0000_wp,  0.9705_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Fr
      & 0.0000_wp,  0.9662_wp,  1.8075_wp,  0.0000_wp,  0.0000_wp, & ! Ra
      & 0.0000_wp,  2.9070_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Ac
      & 0.0000_wp,  2.8844_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Th
      & 0.0000_wp,  2.8738_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Pa
      & 0.0000_wp,  2.8878_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! U
      & 0.0000_wp,  2.9095_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp, & ! Np
      & 0.0000_wp,  1.9209_wp,  0.0000_wp,  0.0000_wp,  0.0000_wp],& ! Pu
      & shape(reference_cn))


   real(wp), allocatable :: reference_c6(:, :, :)


contains


elemental function get_c6(iref, jref, ati, atj) result(c6)
   integer, intent(in) :: iref, jref, ati, atj
   real(wp) :: c6
   integer :: ic
   if (ati > atj) then
      ic = atj + ati*(ati-1)/2
      c6 = reference_c6(iref, jref, ic)
   else
      ic = ati + atj*(atj-1)/2
      c6 = reference_c6(jref, iref, ic)
   endif
end function get_c6


subroutine print_c6(iunit)
   integer, intent(in) :: iunit
   integer :: ipair, ioffset, icn

   if (.not.allocated(reference_c6)) call copy_c6(reference_c6)

   do ipair = 1, max_elem*(1+max_elem)/2, 95
      write(iunit, '(3x,"c6ab(:, :, ",i0,":",i0,") = reshape([ &")') &
         &  ipair, ipair+94
      do ioffset = 0, 94
         do icn = 1, 5
            if (icn.eq.5 .and. ioffset.eq.94) then
               write(iunit, '(6x,"&",4(f10.4,"_wp,"),f10.4,"_wp],&")') &
                  &  reference_c6(:, icn, ipair+ioffset)
               write(iunit, '(6x,"&",4x,"[max_ref, max_ref, 95])",/)')
            else
               write(iunit, '(6x,"&",5(f10.4,"_wp,"),1x,"&")') &
                  &  reference_c6(:, icn, ipair+ioffset)
            end if
         end do
      end do
   end do

end subroutine print_c6


subroutine init_reference_c6
   !$omp critical(d3_reference)
   if (.not.allocated(reference_c6)) call copy_c6(reference_c6)
   !$omp end critical(d3_reference)
end subroutine init_reference_c6


pure subroutine copy_c6(c6ab)
   real(wp), allocatable, target, intent(out) :: c6ab(:, :, :)
   real(wp), pointer :: c6ab_view(:)

   allocate(c6ab(max_ref, max_ref, (max_elem+1)*max_elem/2), source=0.0_wp)
   c6ab_view(1 : max_ref * max_ref * (max_elem+1)*max_elem/2) => c6ab

   c6ab_view(0 * max_ref * max_ref + 1 : 19 * max_ref * max_ref) = [ &
      &    3.0267_wp,    4.7379_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    4.7379_wp,    7.5916_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    2.0835_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    3.1287_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    1.5583_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   38.9448_wp,   14.3165_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   68.9391_wp,   24.0573_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   22.1508_wp,    8.7773_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1163.4454_wp,  282.2106_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  282.2106_wp,   85.3197_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   24.4415_wp,   18.4656_wp,   12.5931_wp,    0.0000_wp,    0.0000_wp, &
      &   41.3078_wp,   30.7866_wp,   20.4039_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   14.8246_wp,   11.4655_wp,    8.1628_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  494.6190_wp,  352.2160_wp,  194.7036_wp,    0.0000_wp,    0.0000_wp, &
      &  148.1323_wp,  107.5083_wp,   66.2009_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  257.4863_wp,  186.3453_wp,  113.9677_wp,    0.0000_wp,    0.0000_wp, &
      &  186.3453_wp,  135.8445_wp,   84.4000_wp,    0.0000_wp,    0.0000_wp, &
      &  113.9677_wp,   84.4000_wp,   55.1364_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   17.3143_wp,   14.7237_wp,   12.4952_wp,    9.7472_wp,    9.2092_wp, &
      &   28.2767_wp,   23.8412_wp,   20.0597_wp,   15.3531_wp,   14.4541_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   11.0975_wp,    9.5836_wp,    8.2586_wp,    6.6434_wp,    6.3181_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  283.7308_wp,  231.4209_wp,  188.7611_wp,  129.9875_wp,  120.6628_wp, &
      &   93.6729_wp,   77.6633_wp,   64.2553_wp,   47.0338_wp,   43.9746_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  161.5971_wp,  133.7265_wp,  110.4185_wp,   80.4276_wp,   75.1311_wp, &
      &  119.0815_wp,   98.9938_wp,   82.1430_wp,   60.5234_wp,   56.6527_wp, &
      &   76.7230_wp,   64.4301_wp,   53.9919_wp,   40.9358_wp,   38.4745_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  107.1777_wp,   89.7429_wp,   74.9865_wp,   56.4116_wp,   52.9606_wp, &
      &   89.7429_wp,   75.3686_wp,   63.1685_wp,   47.8580_wp,   44.9888_wp, &
      &   74.9865_wp,   63.1685_wp,   53.1128_wp,   40.5192_wp,   38.1398_wp, &
      &   56.4116_wp,   47.8580_wp,   40.5192_wp,   31.4436_wp,   29.6804_wp, &
      &   52.9606_wp,   44.9888_wp,   38.1398_wp,   29.6804_wp,   28.0315_wp, &
      &   12.1402_wp,   11.3932_wp,    9.4203_wp,    8.8210_wp,    7.3662_wp, &
      &   19.2653_wp,   18.0575_wp,   14.7623_wp,   13.7992_wp,   11.3299_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    8.1841_wp,    7.7065_wp,    6.5027_wp,    6.1196_wp,    5.2567_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  169.9030_wp,  160.4590_wp,  123.7707_wp,  115.9498_wp,   87.8183_wp, &
      &   60.0860_wp,   56.3468_wp,   44.9320_wp,   41.9713_wp,   33.2158_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  102.9560_wp,   96.5431_wp,   76.7552_wp,   71.6818_wp,   56.4662_wp, &
      &   77.0951_wp,   72.3162_wp,   57.8805_wp,   54.0770_wp,   43.0620_wp, &
      &   51.5628_wp,   48.3072_wp,   39.2814_wp,   36.6977_wp,   29.8819_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(19 * max_ref * max_ref + 1 : 38 * max_ref * max_ref) = [ &
      &   71.2794_wp,   66.7986_wp,   54.0854_wp,   50.5303_wp,   40.8962_wp, &
      &   60.2996_wp,   56.5295_wp,   45.9632_wp,   42.9646_wp,   34.9990_wp, &
      &   50.9090_wp,   47.7462_wp,   38.9834_wp,   36.4604_wp,   29.8916_wp, &
      &   39.2449_wp,   36.8166_wp,   30.3536_wp,   28.4121_wp,   23.6210_wp, &
      &   37.0042_wp,   34.7226_wp,   28.6763_wp,   26.8506_wp,   22.3797_wp, &
      &   49.1130_wp,   46.0681_wp,   37.8419_wp,   35.4129_wp,   29.2830_wp, &
      &   46.0681_wp,   43.2452_wp,   35.5219_wp,   33.2540_wp,   27.5206_wp, &
      &   37.8419_wp,   35.5219_wp,   29.3602_wp,   27.5063_wp,   22.9517_wp, &
      &   35.4129_wp,   33.2540_wp,   27.5063_wp,   25.7809_wp,   21.5377_wp, &
      &   29.2830_wp,   27.5206_wp,   22.9517_wp,   21.5377_wp,   18.2067_wp, &
      &    8.7171_wp,    8.1417_wp,    7.6610_wp,    6.7746_wp,    0.0000_wp, &
      &   13.5164_wp,   12.5980_wp,   11.8214_wp,   10.3987_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    6.1380_wp,    5.7601_wp,    5.4559_wp,    4.8836_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  108.4854_wp,  101.2701_wp,   94.0060_wp,   81.9770_wp,    0.0000_wp, &
      &   40.2944_wp,   37.4939_wp,   35.0194_wp,   30.6022_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   68.6458_wp,   63.8580_wp,   59.5983_wp,   52.0237_wp,    0.0000_wp, &
      &   52.0852_wp,   48.4885_wp,   45.3164_wp,   39.6478_wp,    0.0000_wp, &
      &   35.7902_wp,   33.3274_wp,   31.2407_wp,   27.4225_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   49.1132_wp,   45.7319_wp,   42.8348_wp,   37.5685_wp,    0.0000_wp, &
      &   41.9015_wp,   39.0433_wp,   36.6063_wp,   32.1644_wp,    0.0000_wp, &
      &   35.6778_wp,   33.2677_wp,   31.2221_wp,   27.4842_wp,    0.0000_wp, &
      &   28.0129_wp,   26.1481_wp,   24.5919_wp,   21.7205_wp,    0.0000_wp, &
      &   26.5083_wp,   24.7525_wp,   23.2900_wp,   20.5886_wp,    0.0000_wp, &
      &   34.8146_wp,   32.4848_wp,   30.5305_wp,   26.9351_wp,    0.0000_wp, &
      &   32.7009_wp,   30.5410_wp,   28.6938_wp,   25.3318_wp,    0.0000_wp, &
      &   27.1704_wp,   25.3827_wp,   23.8965_wp,   21.1488_wp,    0.0000_wp, &
      &   25.4799_wp,   23.8136_wp,   22.4279_wp,   19.8669_wp,    0.0000_wp, &
      &   21.4199_wp,   20.0468_wp,   18.9172_wp,   16.8169_wp,    0.0000_wp, &
      &   25.2685_wp,   23.6295_wp,   22.2794_wp,   19.7707_wp,    0.0000_wp, &
      &   23.6295_wp,   22.1241_wp,   20.8501_wp,   18.5180_wp,    0.0000_wp, &
      &   22.2794_wp,   20.8501_wp,   19.6768_wp,   17.4928_wp,    0.0000_wp, &
      &   19.7707_wp,   18.5180_wp,   17.4928_wp,   15.5817_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    6.7180_wp,    6.0575_wp,    5.3717_wp,    0.0000_wp,    0.0000_wp, &
      &   10.2371_wp,    9.1812_wp,    8.0848_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    4.8949_wp,    4.4593_wp,    4.0179_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   76.9613_wp,   67.9312_wp,   58.6703_wp,    0.0000_wp,    0.0000_wp, &
      &   29.5575_wp,   26.2708_wp,   22.8930_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   50.1252_wp,   44.4968_wp,   38.7027_wp,    0.0000_wp,    0.0000_wp, &
      &   38.4132_wp,   34.1983_wp,   29.8523_wp,    0.0000_wp,    0.0000_wp, &
      &   26.8954_wp,   24.0592_wp,   21.1282_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   36.7247_wp,   32.8136_wp,   28.7713_wp,    0.0000_wp,    0.0000_wp, &
      &   31.5354_wp,   28.2327_wp,   24.8185_wp,    0.0000_wp,    0.0000_wp, &
      &   27.0247_wp,   24.2416_wp,   21.3648_wp,    0.0000_wp,    0.0000_wp, &
      &   21.4989_wp,   19.3571_wp,   17.1424_wp,    0.0000_wp,    0.0000_wp, &
      &   20.3998_wp,   18.3831_wp,   16.2987_wp,    0.0000_wp,    0.0000_wp, &
      &   26.5929_wp,   23.9120_wp,   21.1428_wp,    0.0000_wp,    0.0000_wp, &
      &   25.0097_wp,   22.5178_wp,   19.9090_wp,    0.0000_wp,    0.0000_wp, &
      &   20.9597_wp,   18.9034_wp,   16.7855_wp,    0.0000_wp,    0.0000_wp, &
      &   19.6943_wp,   17.7750_wp,   15.8009_wp,    0.0000_wp,    0.0000_wp, &
      &   16.7544_wp,   15.1751_wp,   13.5525_wp,    0.0000_wp,    0.0000_wp, &
      &   19.6546_wp,   17.7698_wp,   15.8364_wp,    0.0000_wp,    0.0000_wp, &
      &   18.4128_wp,   16.6775_wp,   14.8600_wp,    0.0000_wp,    0.0000_wp, &
      &   17.4093_wp,   15.7631_wp,   14.0807_wp,    0.0000_wp,    0.0000_wp, &
      &   15.5249_wp,   14.0793_wp,   12.6077_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   15.5059_wp,   14.0764_wp,   12.6277_wp,    0.0000_wp,    0.0000_wp, &
      &   14.0764_wp,   12.8161_wp,   11.5009_wp,    0.0000_wp,    0.0000_wp, &
      &   12.6277_wp,   11.5009_wp,   10.3708_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    5.1616_wp,    4.2672_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    7.7441_wp,    6.3000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    3.8825_wp,    3.3077_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(38 * max_ref * max_ref + 1 : 57 * max_ref * max_ref) = [ &
      &   55.0933_wp,   42.0627_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   21.7605_wp,   17.1733_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   36.7453_wp,   28.8615_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   28.4098_wp,   22.5413_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   20.2083_wp,   16.3185_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   27.4821_wp,   22.0931_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   23.7369_wp,   19.2029_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   20.4592_wp,   16.6531_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   16.4607_wp,   13.5581_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   15.6578_wp,   12.9291_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   20.2827_wp,   16.6373_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   19.0974_wp,   15.6857_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   16.1312_wp,   13.3529_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   15.1883_wp,   12.5970_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   13.0546_wp,   10.9391_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   15.2418_wp,   12.7107_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   14.3015_wp,   11.9470_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   13.5613_wp,   11.3571_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   12.1511_wp,   10.2214_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   12.1834_wp,   10.2867_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   11.0994_wp,    9.4048_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   10.0222_wp,    8.5397_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    9.6916_wp,    8.2739_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    8.2739_wp,    7.1341_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    4.0112_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    5.9403_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    3.1025_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   40.4731_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   16.3388_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   27.4867_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   21.4067_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   15.4176_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   20.9022_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   18.1424_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   15.7127_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   12.7577_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   12.1608_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   15.6740_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   14.7716_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   12.5629_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   11.8513_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   10.2716_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   11.9479_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   11.2243_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   10.6740_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    9.6050_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    9.6606_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    8.8252_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    8.0175_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    7.7691_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    6.6950_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    6.2896_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   46.8232_wp,   20.7567_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   82.5641_wp,   34.9952_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   26.8628_wp,   12.7287_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(57 * max_ref * max_ref + 1 : 76 * max_ref * max_ref) = [ &
      & 1367.3272_wp,  425.7161_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  335.3450_wp,  125.8245_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  587.4563_wp,  218.6572_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  418.6488_wp,  158.3268_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  232.8060_wp,   96.5125_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  338.7212_wp,  136.9536_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  276.5671_wp,  113.4016_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  225.7843_wp,   93.7159_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  156.1138_wp,   68.2755_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  145.0020_wp,   63.8123_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  203.7631_wp,   87.3956_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  192.3729_wp,   82.0144_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  148.7618_wp,   65.2739_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  139.3753_wp,   61.0174_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  105.9248_wp,   48.1622_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  130.6563_wp,   58.4853_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  121.9601_wp,   54.4454_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  113.3031_wp,   50.8672_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   98.8806_wp,   44.4924_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   93.0263_wp,   42.8911_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   82.1820_wp,   38.1227_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   71.0999_wp,   33.2636_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   66.8423_wp,   31.6060_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   51.2498_wp,   24.9560_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   49.2799_wp,   23.7745_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1608.0286_wp,  505.4363_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  505.4363_wp,  186.1052_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   38.3531_wp,   31.9913_wp,   21.8370_wp,    0.0000_wp,    0.0000_wp, &
      &   65.3703_wp,   54.1021_wp,   35.9190_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   23.0320_wp,   19.4648_wp,   13.8496_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  830.8156_wp,  670.0330_wp,  370.8190_wp,    0.0000_wp,    0.0000_wp, &
      &  240.1627_wp,  195.7057_wp,  120.7808_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  418.2164_wp,  340.2602_wp,  208.6769_wp,    0.0000_wp,    0.0000_wp, &
      &  301.3777_wp,  246.2227_wp,  153.1575_wp,    0.0000_wp,    0.0000_wp, &
      &  181.1653_wp,  149.3832_wp,   97.8101_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  258.1303_wp,  212.2180_wp,  136.9932_wp,    0.0000_wp,    0.0000_wp, &
      &  212.9979_wp,  175.5646_wp,  114.4190_wp,    0.0000_wp,    0.0000_wp, &
      &  175.3914_wp,  144.9783_wp,   95.3575_wp,    0.0000_wp,    0.0000_wp, &
      &  126.5425_wp,  105.3054_wp,   71.3023_wp,    0.0000_wp,    0.0000_wp, &
      &  118.0748_wp,   98.3724_wp,   66.8696_wp,    0.0000_wp,    0.0000_wp, &
      &  162.6082_wp,  134.9227_wp,   90.3305_wp,    0.0000_wp,    0.0000_wp, &
      &  152.6130_wp,  126.6657_wp,   84.6178_wp,    0.0000_wp,    0.0000_wp, &
      &  120.7386_wp,  100.5817_wp,   68.2984_wp,    0.0000_wp,    0.0000_wp, &
      &  112.8210_wp,   93.9978_wp,   63.7944_wp,    0.0000_wp,    0.0000_wp, &
      &   88.2620_wp,   74.0028_wp,   51.3525_wp,    0.0000_wp,    0.0000_wp, &
      &  107.6150_wp,   89.9646_wp,   61.8372_wp,    0.0000_wp,    0.0000_wp, &
      &  100.1443_wp,   83.7703_wp,   57.5398_wp,    0.0000_wp,    0.0000_wp, &
      &   93.4130_wp,   78.1760_wp,   53.8803_wp,    0.0000_wp,    0.0000_wp, &
      &   81.5352_wp,   68.3141_wp,   47.2059_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   78.2250_wp,   65.7609_wp,   46.0347_wp,    0.0000_wp,    0.0000_wp, &
      &   69.3675_wp,   58.4303_wp,   41.0605_wp,    0.0000_wp,    0.0000_wp, &
      &   60.2983_wp,   50.8669_wp,   35.9665_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   57.1605_wp,   48.2865_wp,   34.3314_wp,    0.0000_wp,    0.0000_wp, &
      &   44.7141_wp,   37.9884_wp,   27.4841_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   42.6771_wp,   36.1911_wp,   26.0533_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(76 * max_ref * max_ref + 1 : 95 * max_ref * max_ref) = [ &
      &  985.1697_wp,  794.7597_wp,  442.4019_wp,    0.0000_wp,    0.0000_wp, &
      &  355.9180_wp,  289.6082_wp,  176.8683_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  683.3758_wp,  554.5967_wp,  334.2493_wp,    0.0000_wp,    0.0000_wp, &
      &  554.5967_wp,  451.2708_wp,  274.1590_wp,    0.0000_wp,    0.0000_wp, &
      &  334.2493_wp,  274.1590_wp,  175.5616_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   36.2909_wp,   33.1048_wp,   26.3524_wp,   21.0584_wp,    0.0000_wp, &
      &   60.8558_wp,   55.2636_wp,   43.3661_wp,   34.0700_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   22.3224_wp,   20.5158_wp,   16.7031_wp,   13.6982_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  705.8254_wp,  627.5309_wp,  456.3613_wp,  322.7925_wp,    0.0000_wp, &
      &  214.7265_wp,  193.0707_wp,  146.4426_wp,  110.2136_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  372.6302_wp,  334.7335_wp,  253.0649_wp,  189.6605_wp,    0.0000_wp, &
      &  270.7461_wp,  243.7612_wp,  185.7310_wp,  140.5632_wp,    0.0000_wp, &
      &  167.2929_wp,  151.5931_wp,  118.1233_wp,   92.0126_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  236.4780_wp,  213.8848_wp,  165.5938_wp,  127.9645_wp,    0.0000_wp, &
      &  196.2014_wp,  177.7254_wp,  138.2838_wp,  107.5158_wp,    0.0000_wp, &
      &  162.4473_wp,  147.3766_wp,  115.2480_wp,   90.1440_wp,    0.0000_wp, &
      &  119.1545_wp,  108.5517_wp,   86.0532_wp,   68.4330_wp,    0.0000_wp, &
      &  111.4377_wp,  101.5889_wp,   80.7005_wp,   64.3332_wp,    0.0000_wp, &
      &  152.0934_wp,  138.3254_wp,  109.0541_wp,   86.1605_wp,    0.0000_wp, &
      &  142.6094_wp,  129.6922_wp,  102.2095_wp,   80.7111_wp,    0.0000_wp, &
      &  113.8560_wp,  103.7907_wp,   82.4303_wp,   65.6968_wp,    0.0000_wp, &
      &  106.3508_wp,   96.9560_wp,   77.0087_wp,   61.3829_wp,    0.0000_wp, &
      &   84.3073_wp,   77.1362_wp,   61.9610_wp,   50.0397_wp,    0.0000_wp, &
      &  102.2000_wp,   93.3542_wp,   74.6117_wp,   59.9074_wp,    0.0000_wp, &
      &   95.0916_wp,   86.8728_wp,   69.4501_wp,   55.7793_wp,    0.0000_wp, &
      &   88.8407_wp,   81.2026_wp,   65.0164_wp,   52.3136_wp,    0.0000_wp, &
      &   77.6520_wp,   71.0231_wp,   56.9713_wp,   45.9399_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   75.0755_wp,   68.7983_wp,   55.5250_wp,   45.0876_wp,    0.0000_wp, &
      &   66.7451_wp,   61.2175_wp,   49.5319_wp,   40.3374_wp,    0.0000_wp, &
      &   58.1951_wp,   53.4365_wp,   43.3788_wp,   35.4602_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   55.3412_wp,   50.8577_wp,   41.3914_wp,   33.9334_wp,    0.0000_wp, &
      &   43.7364_wp,   40.3220_wp,   33.1265_wp,   27.4436_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   41.5966_wp,   38.3154_wp,   31.3944_wp,   25.9335_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  838.9648_wp,  746.3419_wp,  543.9378_wp,  386.0980_wp,    0.0000_wp, &
      &  316.4588_wp,  284.2251_wp,  214.6796_wp,  160.6594_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  603.4689_wp,  541.0285_wp,  406.0687_wp,  301.3307_wp,    0.0000_wp, &
      &  492.0829_wp,  441.7322_wp,  333.0505_wp,  248.5692_wp,    0.0000_wp, &
      &  304.7337_wp,  275.2609_wp,  212.1971_wp,  163.1100_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  540.5406_wp,  486.2365_wp,  369.3829_wp,  278.5639_wp,    0.0000_wp, &
      &  486.2365_wp,  437.7575_wp,  333.5341_wp,  252.4953_wp,    0.0000_wp, &
      &  369.3829_wp,  333.5341_wp,  256.7497_wp,  196.9411_wp,    0.0000_wp, &
      &  278.5639_wp,  252.4953_wp,  196.9411_wp,  153.5945_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   29.5947_wp,   27.4989_wp,   25.7021_wp,   24.9833_wp,   21.0064_wp, &
      &   48.5182_wp,   44.9030_wp,   41.8222_wp,   40.6972_wp,   33.7126_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   18.8500_wp,   17.6356_wp,   16.5806_wp,   16.1104_wp,   13.8547_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  495.3449_wp,  450.9583_wp,  412.4323_wp,  408.0802_wp,  308.7953_wp, &
      &  162.0508_wp,  148.7478_wp,  137.4458_wp,  134.4490_wp,  107.2721_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  279.7863_wp,  256.5881_wp,  236.8991_wp,  231.8251_wp,  184.2566_wp, &
      &  205.7544_wp,  189.1090_wp,  174.9315_wp,  171.0398_wp,  137.1783_wp, &
      &  131.8920_wp,  121.8333_wp,  113.2820_wp,  110.3134_wp,   90.6956_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(95 * max_ref * max_ref + 1 : 114 * max_ref * max_ref) = [ &
      &  184.5111_wp,  170.1891_wp,  158.0165_wp,  154.0388_wp,  125.7764_wp, &
      &  154.2901_wp,  142.5123_wp,  132.4831_wp,  129.0898_wp,  105.9826_wp, &
      &  128.7459_wp,  119.0889_wp,  110.8447_wp,  107.9647_wp,   89.1190_wp, &
      &   96.5313_wp,   89.5948_wp,   83.6586_wp,   81.3478_wp,   68.1106_wp, &
      &   90.5740_wp,   84.1158_wp,   78.5835_wp,   76.4038_wp,   64.1083_wp, &
      &  122.1387_wp,  113.2046_wp,  105.5713_wp,  102.7202_wp,   85.5223_wp, &
      &  114.4405_wp,  106.0806_wp,   98.9278_wp,   96.2872_wp,   80.1427_wp, &
      &   92.4931_wp,   85.9068_wp,   80.2619_wp,   78.0574_wp,   65.4879_wp, &
      &   86.3999_wp,   80.2618_wp,   74.9967_wp,   72.9530_wp,   61.2169_wp, &
      &   69.7224_wp,   64.9672_wp,   60.8687_wp,   59.1583_wp,   50.2053_wp, &
      &   83.8498_wp,   78.0190_wp,   73.0073_wp,   70.9761_wp,   59.9319_wp, &
      &   78.0466_wp,   72.6373_wp,   67.9820_wp,   66.1052_wp,   55.8394_wp, &
      &   73.0858_wp,   68.0504_wp,   63.7151_wp,   61.9530_wp,   52.4136_wp, &
      &   64.0576_wp,   59.6887_wp,   55.9197_wp,   54.3857_wp,   46.1031_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   62.5349_wp,   58.3557_wp,   54.7443_wp,   53.2061_wp,   45.3688_wp, &
      &   55.8175_wp,   52.1311_wp,   48.9391_wp,   47.5664_wp,   40.6635_wp, &
      &   48.9066_wp,   45.7273_wp,   42.9679_wp,   41.7687_wp,   35.8247_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   46.6936_wp,   43.6855_wp,   41.0728_wp,   39.9172_wp,   34.3187_wp, &
      &   37.4421_wp,   35.1290_wp,   33.1075_wp,   32.1730_wp,   27.9086_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   35.4550_wp,   33.2416_wp,   31.3107_wp,   30.4338_wp,   26.3358_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  591.0458_wp,  538.3412_wp,  492.6562_wp,  487.1761_wp,  369.7717_wp, &
      &  237.1326_wp,  217.5049_wp,  200.8038_wp,  196.6526_wp,  156.1545_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  447.6423_wp,  409.9174_wp,  377.8360_wp,  370.3778_wp,  291.8716_wp, &
      &  367.5811_wp,  337.0250_wp,  310.9897_wp,  304.6797_wp,  241.3903_wp, &
      &  236.1388_wp,  217.5471_wp,  201.7758_wp,  196.7841_wp,  159.9115_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  408.9606_wp,  375.5233_wp,  347.1154_wp,  339.4502_wp,  271.3202_wp, &
      &  369.6232_wp,  339.6516_wp,  314.1809_wp,  307.1015_wp,  246.3013_wp, &
      &  285.4825_wp,  262.9770_wp,  243.8407_wp,  237.9434_wp,  193.0534_wp, &
      &  219.9479_wp,  203.2226_wp,  189.0034_wp,  184.0291_wp,  151.4603_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  317.8574_wp,  292.9994_wp,  271.8844_wp,  265.1189_wp,  215.8998_wp, &
      &  292.9994_wp,  270.2650_wp,  250.9376_wp,  244.6288_wp,  199.7530_wp, &
      &  271.8844_wp,  250.9376_wp,  233.1254_wp,  227.1882_wp,  185.9996_wp, &
      &  265.1189_wp,  244.6288_wp,  227.1882_wp,  221.5046_wp,  181.0237_wp, &
      &  215.8998_wp,  199.7530_wp,  185.9996_wp,  181.0237_wp,  149.7734_wp, &
      &   23.7604_wp,   23.0908_wp,   22.7429_wp,   21.2507_wp,    0.0000_wp, &
      &   38.1397_wp,   37.0236_wp,   36.4936_wp,   33.9193_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   15.6689_wp,   15.2619_wp,   15.0245_wp,   14.1577_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  350.8030_wp,  339.6693_wp,  337.4662_wp,  304.7707_wp,    0.0000_wp, &
      &  121.5067_wp,  117.7472_wp,  116.3941_wp,  106.8466_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  208.7331_wp,  202.2282_wp,  199.9576_wp,  183.3056_wp,    0.0000_wp, &
      &  155.3644_wp,  150.6068_wp,  148.8194_wp,  136.8674_wp,    0.0000_wp, &
      &  102.6176_wp,   99.5656_wp,   98.1859_wp,   91.0206_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  142.3481_wp,  138.0774_wp,  136.2402_wp,  126.0194_wp,    0.0000_wp, &
      &  119.9353_wp,  116.3818_wp,  114.7968_wp,  106.3923_wp,    0.0000_wp, &
      &  100.8432_wp,   97.8947_wp,   96.5324_wp,   89.6391_wp,    0.0000_wp, &
      &   77.0429_wp,   74.8497_wp,   73.7397_wp,   68.8007_wp,    0.0000_wp, &
      &   72.5145_wp,   70.4629_wp,   69.4115_wp,   64.8127_wp,    0.0000_wp, &
      &   96.7503_wp,   93.9663_wp,   92.6080_wp,   86.2427_wp,    0.0000_wp, &
      &   90.6767_wp,   88.0769_wp,   86.8128_wp,   80.8496_wp,    0.0000_wp, &
      &   74.0784_wp,   71.9914_wp,   70.9269_wp,   66.2343_wp,    0.0000_wp, &
      &   69.2515_wp,   67.3088_wp,   66.3194_wp,   61.9433_wp,    0.0000_wp, &
      &   56.7852_wp,   55.2375_wp,   54.3941_wp,   51.0048_wp,    0.0000_wp, &
      &   67.7873_wp,   65.9127_wp,   64.9209_wp,   60.7663_wp,    0.0000_wp, &
      &   63.1669_wp,   61.4286_wp,   60.5085_wp,   56.6508_wp,    0.0000_wp, &
      &   59.2859_wp,   57.6632_wp,   56.7981_wp,   53.2076_wp,    0.0000_wp, &
      &   52.1509_wp,   50.7398_wp,   49.9818_wp,   46.8635_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   51.3098_wp,   49.9374_wp,   49.1729_wp,   46.1927_wp,    0.0000_wp, &
      &   45.9937_wp,   44.7770_wp,   44.0904_wp,   41.4600_wp,    0.0000_wp, &
      &   40.5167_wp,   39.4617_wp,   38.8578_wp,   36.5875_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   38.8084_wp,   37.8035_wp,   37.2205_wp,   35.0733_wp,    0.0000_wp, &
      &   31.5577_wp,   30.7688_wp,   30.2894_wp,   28.6362_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(114 * max_ref * max_ref + 1 : 133 * max_ref * max_ref) = [ &
      &   29.7767_wp,   29.0280_wp,   28.5805_wp,   26.9990_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  420.0064_wp,  406.7150_wp,  403.9802_wp,  365.2110_wp,    0.0000_wp, &
      &  176.9205_wp,  171.4469_wp,  169.5698_wp,  155.4495_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  330.7801_wp,  320.4176_wp,  317.0824_wp,  289.9090_wp,    0.0000_wp, &
      &  273.5268_wp,  265.0378_wp,  262.1696_wp,  240.1582_wp,    0.0000_wp, &
      &  180.9843_wp,  175.5006_wp,  173.2227_wp,  159.9562_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  307.2965_wp,  297.8076_wp,  294.3403_wp,  270.3535_wp,    0.0000_wp, &
      &  278.9272_wp,  270.3580_wp,  267.1404_wp,  245.6529_wp,    0.0000_wp, &
      &  218.5302_wp,  211.9205_wp,  209.2061_wp,  193.1132_wp,    0.0000_wp, &
      &  171.3557_wp,  166.2698_wp,  163.9565_wp,  152.0452_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  244.3546_wp,  236.9864_wp,  233.8752_wp,  216.1303_wp,    0.0000_wp, &
      &  226.0657_wp,  219.2863_wp,  216.3693_wp,  200.1422_wp,    0.0000_wp, &
      &  210.4840_wp,  204.1998_wp,  201.4465_wp,  186.5031_wp,    0.0000_wp, &
      &  204.8770_wp,  198.7616_wp,  196.1196_wp,  181.4840_wp,    0.0000_wp, &
      &  169.4356_wp,  164.4669_wp,  162.1289_wp,  150.6308_wp,    0.0000_wp, &
      &  191.6887_wp,  186.0684_wp,  183.4322_wp,  170.4045_wp,    0.0000_wp, &
      &  186.0684_wp,  180.6238_wp,  178.0622_wp,  165.4518_wp,    0.0000_wp, &
      &  183.4322_wp,  178.0622_wp,  175.5534_wp,  163.0764_wp,    0.0000_wp, &
      &  170.4045_wp,  165.4518_wp,  163.0764_wp,  151.6886_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   20.0948_wp,   19.8667_wp,   19.4729_wp,    0.0000_wp,    0.0000_wp, &
      &   31.7713_wp,   31.4127_wp,   30.7747_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   13.6108_wp,   13.4598_wp,   13.2090_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  273.7867_wp,  271.3011_wp,  265.5464_wp,    0.0000_wp,    0.0000_wp, &
      &   98.1249_wp,   97.0849_wp,   95.0531_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  167.9513_wp,  166.1807_wp,  162.6875_wp,    0.0000_wp,    0.0000_wp, &
      &  126.0993_wp,  124.7512_wp,  122.1534_wp,    0.0000_wp,    0.0000_wp, &
      &   84.8691_wp,   83.9181_wp,   82.1973_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  117.1121_wp,  115.8168_wp,  113.4324_wp,    0.0000_wp,    0.0000_wp, &
      &   99.2159_wp,   98.1139_wp,   96.1100_wp,    0.0000_wp,    0.0000_wp, &
      &   83.8847_wp,   82.9497_wp,   81.2693_wp,    0.0000_wp,    0.0000_wp, &
      &   64.8918_wp,   64.1573_wp,   62.8783_wp,    0.0000_wp,    0.0000_wp, &
      &   61.2184_wp,   60.5255_wp,   59.3239_wp,    0.0000_wp,    0.0000_wp, &
      &   81.0919_wp,   80.1804_wp,   78.5729_wp,    0.0000_wp,    0.0000_wp, &
      &   76.0552_wp,   75.2039_wp,   73.7006_wp,    0.0000_wp,    0.0000_wp, &
      &   62.5953_wp,   61.8916_wp,   60.6690_wp,    0.0000_wp,    0.0000_wp, &
      &   58.5776_wp,   57.9221_wp,   56.7826_wp,    0.0000_wp,    0.0000_wp, &
      &   48.5681_wp,   48.0221_wp,   47.0943_wp,    0.0000_wp,    0.0000_wp, &
      &   57.6734_wp,   57.0255_wp,   55.9136_wp,    0.0000_wp,    0.0000_wp, &
      &   53.8107_wp,   53.2086_wp,   52.1753_wp,    0.0000_wp,    0.0000_wp, &
      &   50.5946_wp,   50.0298_wp,   49.0629_wp,    0.0000_wp,    0.0000_wp, &
      &   44.6529_wp,   44.1581_wp,   43.3130_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   44.1470_wp,   43.6548_wp,   42.8245_wp,    0.0000_wp,    0.0000_wp, &
      &   39.7091_wp,   39.2683_wp,   38.5276_wp,    0.0000_wp,    0.0000_wp, &
      &   35.1377_wp,   34.7510_wp,   34.1042_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   33.7264_wp,   33.3550_wp,   32.7365_wp,    0.0000_wp,    0.0000_wp, &
      &   27.7137_wp,   27.4117_wp,   26.9163_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   26.0940_wp,   25.8107_wp,   25.3435_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  328.5990_wp,  325.5969_wp,  318.7120_wp,    0.0000_wp,    0.0000_wp, &
      &  142.5430_wp,  141.0599_wp,  138.1166_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  264.6665_wp,  261.9437_wp,  256.4315_wp,    0.0000_wp,    0.0000_wp, &
      &  219.9489_wp,  217.6629_wp,  213.1036_wp,    0.0000_wp,    0.0000_wp, &
      &  148.2130_wp,  146.5849_wp,  143.5523_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  248.5008_wp,  245.8568_wp,  240.7165_wp,    0.0000_wp,    0.0000_wp, &
      &  226.2156_wp,  223.7942_wp,  219.1285_wp,    0.0000_wp,    0.0000_wp, &
      &  178.8926_wp,  176.9365_wp,  173.2779_wp,    0.0000_wp,    0.0000_wp, &
      &  141.8517_wp,  140.2618_wp,  137.3907_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(133 * max_ref * max_ref + 1 : 152 * max_ref * max_ref) = [ &
      &  200.5374_wp,  198.3266_wp,  194.2314_wp,    0.0000_wp,    0.0000_wp, &
      &  186.0051_wp,  183.9483_wp,  180.1622_wp,    0.0000_wp,    0.0000_wp, &
      &  173.5798_wp,  171.6537_wp,  168.1302_wp,    0.0000_wp,    0.0000_wp, &
      &  168.8225_wp,  166.9597_wp,  163.5352_wp,    0.0000_wp,    0.0000_wp, &
      &  140.9955_wp,  139.4086_wp,  136.5758_wp,    0.0000_wp,    0.0000_wp, &
      &  159.4898_wp,  157.6971_wp,  154.4935_wp,    0.0000_wp,    0.0000_wp, &
      &  154.9229_wp,  153.1823_wp,  150.0748_wp,    0.0000_wp,    0.0000_wp, &
      &  152.6470_wp,  150.9364_wp,  147.8750_wp,    0.0000_wp,    0.0000_wp, &
      &  142.2982_wp,  140.6958_wp,  137.8540_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  134.0066_wp,  132.4922_wp,  129.8413_wp,    0.0000_wp,    0.0000_wp, &
      &  132.4922_wp,  130.9965_wp,  128.3764_wp,    0.0000_wp,    0.0000_wp, &
      &  129.8413_wp,  128.3764_wp,  125.8109_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   16.7052_wp,   16.5273_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   26.0170_wp,   25.7414_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   11.6302_wp,   11.5092_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  210.6626_wp,  208.7743_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   77.9865_wp,   77.2023_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  132.9808_wp,  131.6498_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  100.7212_wp,   99.6995_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   69.0069_wp,   68.2806_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   94.7612_wp,   93.7743_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   80.7272_wp,   79.8833_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   68.6331_wp,   67.9132_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   53.7343_wp,   53.1638_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   50.8102_wp,   50.2708_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   66.8407_wp,   66.1354_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   62.7465_wp,   62.0864_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   52.0208_wp,   51.4721_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   48.7448_wp,   48.2328_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   40.8516_wp,   40.4209_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   48.2624_wp,   47.7540_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   45.0952_wp,   44.6217_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   42.4816_wp,   42.0367_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   37.6295_wp,   37.2381_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   37.3688_wp,   36.9783_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   33.7313_wp,   33.3801_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   29.9918_wp,   29.6822_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   28.8445_wp,   28.5468_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   23.9505_wp,   23.7059_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   22.5121_wp,   22.2832_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  253.5136_wp,  251.2326_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  113.1025_wp,  111.9829_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  208.4978_wp,  206.4521_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  174.1399_wp,  172.4148_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  119.3911_wp,  118.1567_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  197.7594_wp,  195.7647_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  180.5430_wp,  178.7127_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  144.0669_wp,  142.5806_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  115.4552_wp,  114.2405_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  161.8686_wp,  160.1879_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  150.5231_wp,  148.9559_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  140.7824_wp,  139.3125_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  136.8460_wp,  135.4234_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  115.3542_wp,  114.1358_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  130.4725_wp,  129.0956_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  126.8312_wp,  125.4930_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  124.9162_wp,  123.6011_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  116.8355_wp,  115.6006_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  110.7006_wp,  109.5270_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  109.4486_wp,  108.2893_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  107.2964_wp,  106.1606_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(152 * max_ref * max_ref + 1 : 171 * max_ref * max_ref) = [ &
      &   92.3460_wp,   91.3667_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   91.3667_wp,   90.3985_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   13.8700_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   21.3055_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    9.9130_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  163.5497_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   62.2205_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  105.7229_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   80.7174_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   56.1517_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   76.7949_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   65.7580_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   56.1926_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   44.4613_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   42.1324_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   55.0898_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   51.7684_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   43.2064_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   40.5441_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   34.3050_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   40.3435_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   37.7522_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   35.6334_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   31.6804_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   31.5783_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   28.6000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   25.5514_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   24.6178_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   20.6394_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   19.3774_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  197.3440_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   90.1735_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  165.1006_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  138.5190_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   96.3813_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  157.9595_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  144.5798_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  116.2817_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   94.0428_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  130.8927_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  122.0012_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  114.3348_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  111.1044_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   94.4027_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  106.7698_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  103.8653_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  102.2701_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   95.9336_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   91.4014_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   90.3716_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   88.6268_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   76.9383_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   76.1256_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   64.6462_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(171 * max_ref * max_ref + 1 : 190 * max_ref * max_ref) = [ &
      &   76.2376_wp,   29.3000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  134.4441_wp,   48.4499_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   44.0411_wp,   18.6655_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2387.1574_wp,  560.2954_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  555.1516_wp,  168.7412_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  972.3197_wp,  292.2224_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  694.8596_wp,  213.4444_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  379.4225_wp,  132.4751_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  554.1980_wp,  186.9566_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  452.5193_wp,  155.7781_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  370.1535_wp,  129.5963_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  254.2237_wp,   95.8497_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  236.1802_wp,   89.8569_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  332.2694_wp,  121.9292_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  314.5800_wp,  114.5723_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  242.6366_wp,   92.0240_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  227.6600_wp,   86.1481_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  172.9685_wp,   69.0434_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  213.2001_wp,   83.2153_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  199.4405_wp,   77.6317_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  185.2181_wp,   72.6818_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  162.0160_wp,   63.8686_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  152.1420_wp,   61.9948_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  134.6508_wp,   55.4032_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  116.7069_wp,   48.6380_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  109.6439_wp,   46.3520_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   84.3688_wp,   37.1982_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   81.0861_wp,   35.2969_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2798.6124_wp,  666.0357_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  840.8299_wp,  249.2200_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1642.0587_wp,  473.7661_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1327.3176_wp,  387.3349_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  723.6346_wp,  240.1725_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1387.9361_wp,  424.8926_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1233.2237_wp,  382.6422_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  894.6228_wp,  291.6028_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  628.9537_wp,  220.7264_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  967.6283_wp,  322.8233_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  881.2624_wp,  296.9409_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  805.4924_wp,  274.8056_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  798.8390_wp,  268.9778_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  602.2116_wp,  215.8797_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  684.4968_wp,  244.5729_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  663.1184_wp,  237.2191_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  659.2394_wp,  234.4921_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  595.1033_wp,  215.8467_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  535.2384_wp,  199.4672_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  530.5192_wp,  197.3806_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  519.3802_wp,  193.3412_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  413.1512_wp,  160.3304_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  409.5105_wp,  158.7330_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  322.1155_wp,  129.4283_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4983.5009_wp, 1112.2277_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1112.2277_wp,  338.0207_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(190 * max_ref * max_ref + 1 : 209 * max_ref * max_ref) = [ &
      &   65.8180_wp,   52.9871_wp,   30.6783_wp,    0.0000_wp,    0.0000_wp, &
      &  113.4857_wp,   90.7927_wp,   50.2368_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   39.0701_wp,   31.8796_wp,   19.7247_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1614.4719_wp, 1310.0314_wp,  525.3196_wp,    0.0000_wp,    0.0000_wp, &
      &  434.1576_wp,  345.6120_wp,  168.9009_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  757.9070_wp,  602.5661_wp,  291.7021_wp,    0.0000_wp,    0.0000_wp, &
      &  543.5842_wp,  434.0284_wp,  214.3171_wp,    0.0000_wp,    0.0000_wp, &
      &  316.5102_wp,  252.5778_wp,  136.5998_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  454.8439_wp,  362.6862_wp,  191.4135_wp,    0.0000_wp,    0.0000_wp, &
      &  373.8815_wp,  298.7264_wp,  160.0734_wp,    0.0000_wp,    0.0000_wp, &
      &  306.9078_wp,  245.9498_wp,  133.5995_wp,    0.0000_wp,    0.0000_wp, &
      &  217.9457_wp,  175.1434_wp,  100.0610_wp,    0.0000_wp,    0.0000_wp, &
      &  203.0679_wp,  163.3534_wp,   93.9194_wp,    0.0000_wp,    0.0000_wp, &
      &  281.7035_wp,  225.9629_wp,  126.6836_wp,    0.0000_wp,    0.0000_wp, &
      &  265.1206_wp,  212.8957_wp,  118.8255_wp,    0.0000_wp,    0.0000_wp, &
      &  208.0223_wp,  167.4145_wp,   96.0553_wp,    0.0000_wp,    0.0000_wp, &
      &  194.6719_wp,  156.7702_wp,   89.8285_wp,    0.0000_wp,    0.0000_wp, &
      &  150.8203_wp,  122.0707_wp,   72.5524_wp,    0.0000_wp,    0.0000_wp, &
      &  184.5498_wp,  148.9793_wp,   87.1906_wp,    0.0000_wp,    0.0000_wp, &
      &  172.0455_wp,  139.0219_wp,   81.2525_wp,    0.0000_wp,    0.0000_wp, &
      &  160.2750_wp,  129.5841_wp,   76.1365_wp,    0.0000_wp,    0.0000_wp, &
      &  140.0273_wp,  113.4161_wp,   66.8706_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  133.3847_wp,  108.2718_wp,   65.2525_wp,    0.0000_wp,    0.0000_wp, &
      &  118.2411_wp,   96.1584_wp,   58.3423_wp,    0.0000_wp,    0.0000_wp, &
      &  102.7126_wp,   83.7279_wp,   51.2425_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   97.0824_wp,   79.2218_wp,   48.9331_wp,    0.0000_wp,    0.0000_wp, &
      &   75.5850_wp,   62.0580_wp,   39.4048_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   72.3072_wp,   59.2779_wp,   37.3258_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1907.7081_wp, 1545.7064_wp,  626.5526_wp,    0.0000_wp,    0.0000_wp, &
      &  648.3105_wp,  516.6917_wp,  247.8223_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1252.5932_wp,  997.2486_wp,  468.1527_wp,    0.0000_wp,    0.0000_wp, &
      & 1013.7695_wp,  809.2182_wp,  384.1308_wp,    0.0000_wp,    0.0000_wp, &
      &  590.8737_wp,  470.3059_wp,  245.0983_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1088.8420_wp,  866.9517_wp,  426.0568_wp,    0.0000_wp,    0.0000_wp, &
      &  973.0493_wp,  775.0832_wp,  384.8555_wp,    0.0000_wp,    0.0000_wp, &
      &  721.4089_wp,  575.4721_wp,  296.5572_wp,    0.0000_wp,    0.0000_wp, &
      &  525.8367_wp,  419.7129_wp,  227.8108_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  790.6347_wp,  630.0456_wp,  329.8171_wp,    0.0000_wp,    0.0000_wp, &
      &  722.5111_wp,  576.2962_wp,  303.9725_wp,    0.0000_wp,    0.0000_wp, &
      &  664.0977_wp,  529.8946_wp,  281.9814_wp,    0.0000_wp,    0.0000_wp, &
      &  653.4174_wp,  521.7535_wp,  275.2242_wp,    0.0000_wp,    0.0000_wp, &
      &  507.2703_wp,  405.6765_wp,  223.6162_wp,    0.0000_wp,    0.0000_wp, &
      &  575.4627_wp,  460.2635_wp,  253.1519_wp,    0.0000_wp,    0.0000_wp, &
      &  557.4255_wp,  446.0170_wp,  245.5654_wp,    0.0000_wp,    0.0000_wp, &
      &  552.4889_wp,  442.0702_wp,  242.4454_wp,    0.0000_wp,    0.0000_wp, &
      &  502.9654_wp,  402.8286_wp,  223.9566_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  456.8443_wp,  366.7711_wp,  207.8531_wp,    0.0000_wp,    0.0000_wp, &
      &  452.3745_wp,  363.1951_wp,  205.6006_wp,    0.0000_wp,    0.0000_wp, &
      &  442.8735_wp,  355.6161_wp,  201.3895_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  357.4737_wp,  288.2182_wp,  168.0065_wp,    0.0000_wp,    0.0000_wp, &
      &  354.0935_wp,  285.4905_wp,  166.2878_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  281.6823_wp,  228.0892_wp,  136.1638_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3240.4393_wp, 2661.0830_wp, 1030.5394_wp,    0.0000_wp,    0.0000_wp, &
      &  860.1771_wp,  688.9911_wp,  337.7795_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(209 * max_ref * max_ref + 1 : 228 * max_ref * max_ref) = [ &
      & 2352.6862_wp, 1886.4221_wp,  832.2379_wp,    0.0000_wp,    0.0000_wp, &
      & 1886.4221_wp, 1525.1891_wp,  664.0620_wp,    0.0000_wp,    0.0000_wp, &
      &  832.2379_wp,  664.0620_wp,  343.3343_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   54.9669_wp,   29.9953_wp,   29.8615_wp,    0.0000_wp,    0.0000_wp, &
      &   94.2083_wp,   48.9753_wp,   48.4929_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   32.8995_wp,   19.3841_wp,   19.4304_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1278.1183_wp,  504.8268_wp,  481.9586_wp,    0.0000_wp,    0.0000_wp, &
      &  354.0317_wp,  163.6446_wp,  159.5715_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  617.3358_wp,  282.4420_wp,  275.0061_wp,    0.0000_wp,    0.0000_wp, &
      &  443.7190_wp,  207.8107_wp,  203.0678_wp,    0.0000_wp,    0.0000_wp, &
      &  261.9443_wp,  132.9868_wp,  131.2980_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  375.0418_wp,  186.1419_wp,  183.2413_wp,    0.0000_wp,    0.0000_wp, &
      &  308.8721_wp,  155.8228_wp,  153.7050_wp,    0.0000_wp,    0.0000_wp, &
      &  253.9534_wp,  130.1725_wp,  128.6716_wp,    0.0000_wp,    0.0000_wp, &
      &  181.6834_wp,   97.7550_wp,   97.1632_wp,    0.0000_wp,    0.0000_wp, &
      &  169.4186_wp,   91.7951_wp,   91.3072_wp,    0.0000_wp,    0.0000_wp, &
      &  234.1950_wp,  123.6379_wp,  122.6115_wp,    0.0000_wp,    0.0000_wp, &
      &  220.1854_wp,  115.9726_wp,  114.9710_wp,    0.0000_wp,    0.0000_wp, &
      &  173.4630_wp,   93.8911_wp,   93.3603_wp,    0.0000_wp,    0.0000_wp, &
      &  162.2609_wp,   87.8153_wp,   87.3048_wp,    0.0000_wp,    0.0000_wp, &
      &  126.3491_wp,   71.0854_wp,   70.9670_wp,    0.0000_wp,    0.0000_wp, &
      &  154.2947_wp,   85.3404_wp,   85.0464_wp,    0.0000_wp,    0.0000_wp, &
      &  143.7658_wp,   79.5383_wp,   79.2586_wp,    0.0000_wp,    0.0000_wp, &
      &  134.0314_wp,   74.5587_wp,   74.3297_wp,    0.0000_wp,    0.0000_wp, &
      &  117.1112_wp,   65.5203_wp,   65.3415_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  111.9359_wp,   64.0072_wp,   63.9839_wp,    0.0000_wp,    0.0000_wp, &
      &   99.2939_wp,   57.2611_wp,   57.2805_wp,    0.0000_wp,    0.0000_wp, &
      &   86.3365_wp,   50.3373_wp,   50.3920_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   81.7242_wp,   48.0936_wp,   48.1892_wp,    0.0000_wp,    0.0000_wp, &
      &   63.8564_wp,   38.8097_wp,   38.9899_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   61.0154_wp,   36.7468_wp,   36.8800_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1512.5338_wp,  602.4929_wp,  575.7400_wp,    0.0000_wp,    0.0000_wp, &
      &  527.0917_wp,  239.9714_wp,  233.4674_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1015.5454_wp,  452.7600_wp,  439.2030_wp,    0.0000_wp,    0.0000_wp, &
      &  822.8957_wp,  371.7768_wp,  361.4502_wp,    0.0000_wp,    0.0000_wp, &
      &  486.4653_wp,  238.1540_wp,  234.0068_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  888.6025_wp,  412.9058_wp,  402.9357_wp,    0.0000_wp,    0.0000_wp, &
      &  795.2206_wp,  373.1845_wp,  364.6668_wp,    0.0000_wp,    0.0000_wp, &
      &  592.7050_wp,  288.0834_wp,  282.8628_wp,    0.0000_wp,    0.0000_wp, &
      &  435.4205_wp,  221.8339_wp,  219.1014_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  651.2040_wp,  320.5896_wp,  315.2655_wp,    0.0000_wp,    0.0000_wp, &
      &  595.6725_wp,  295.6101_wp,  291.0037_wp,    0.0000_wp,    0.0000_wp, &
      &  548.2115_wp,  274.3493_wp,  270.3548_wp,    0.0000_wp,    0.0000_wp, &
      &  538.5754_wp,  267.7120_wp,  263.5853_wp,    0.0000_wp,    0.0000_wp, &
      &  420.8895_wp,  217.9628_wp,  215.7027_wp,    0.0000_wp,    0.0000_wp, &
      &  477.2797_wp,  246.7358_wp,  244.1234_wp,    0.0000_wp,    0.0000_wp, &
      &  462.3511_wp,  239.3701_wp,  236.8740_wp,    0.0000_wp,    0.0000_wp, &
      &  457.9608_wp,  236.2965_wp,  233.7282_wp,    0.0000_wp,    0.0000_wp, &
      &  417.7400_wp,  218.4319_wp,  216.4038_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  380.4107_wp,  202.9672_wp,  201.5466_wp,    0.0000_wp,    0.0000_wp, &
      &  376.6142_wp,  200.7622_wp,  199.3322_wp,    0.0000_wp,    0.0000_wp, &
      &  368.7131_wp,  196.6601_wp,  195.2671_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  298.7686_wp,  164.3655_wp,  163.7482_wp,    0.0000_wp,    0.0000_wp, &
      &  295.9028_wp,  162.6806_wp,  162.0542_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  236.1417_wp,  133.4404_wp,  133.2865_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(228 * max_ref * max_ref + 1 : 247 * max_ref * max_ref) = [ &
      & 2549.9412_wp,  988.5881_wp,  943.3962_wp,    0.0000_wp,    0.0000_wp, &
      &  700.7281_wp,  327.7092_wp,  320.0043_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1888.7902_wp,  803.2380_wp,  774.1368_wp,    0.0000_wp,    0.0000_wp, &
      & 1510.1427_wp,  640.7076_wp,  618.1676_wp,    0.0000_wp,    0.0000_wp, &
      &  683.8786_wp,  333.6780_wp,  327.6458_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1522.4676_wp,  660.6780_wp,  638.4869_wp,    0.0000_wp,    0.0000_wp, &
      &  660.6780_wp,  324.5068_wp,  318.8004_wp,    0.0000_wp,    0.0000_wp, &
      &  638.4869_wp,  318.8004_wp,  313.9846_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   53.6875_wp,   29.0009_wp,   30.2593_wp,    0.0000_wp,    0.0000_wp, &
      &   91.3334_wp,   47.2755_wp,   49.0983_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   32.5316_wp,   18.8084_wp,   19.7205_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1192.9128_wp,  488.2931_wp,  486.2098_wp,    0.0000_wp,    0.0000_wp, &
      &  337.3896_wp,  157.6667_wp,  161.3013_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  587.4898_wp,  272.0201_wp,  277.9371_wp,    0.0000_wp,    0.0000_wp, &
      &  423.6217_wp,  200.3669_wp,  205.3134_wp,    0.0000_wp,    0.0000_wp, &
      &  253.0543_wp,  128.2862_wp,  132.8881_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  361.0865_wp,  179.5077_wp,  185.4074_wp,    0.0000_wp,    0.0000_wp, &
      &  298.0871_wp,  150.3520_wp,  155.5661_wp,    0.0000_wp,    0.0000_wp, &
      &  245.6647_wp,  125.7017_wp,  130.2662_wp,    0.0000_wp,    0.0000_wp, &
      &  177.0660_wp,   94.4758_wp,   98.4363_wp,    0.0000_wp,    0.0000_wp, &
      &  165.2912_wp,   88.7389_wp,   92.5154_wp,    0.0000_wp,    0.0000_wp, &
      &  227.5723_wp,  119.4376_wp,  124.1849_wp,    0.0000_wp,    0.0000_wp, &
      &  213.8682_wp,  112.0523_wp,  116.4471_wp,    0.0000_wp,    0.0000_wp, &
      &  169.2064_wp,   90.7838_wp,   94.6013_wp,    0.0000_wp,    0.0000_wp, &
      &  158.2705_wp,   84.9253_wp,   88.4703_wp,    0.0000_wp,    0.0000_wp, &
      &  124.0013_wp,   68.8297_wp,   71.9595_wp,    0.0000_wp,    0.0000_wp, &
      &  151.0199_wp,   82.5834_wp,   86.2109_wp,    0.0000_wp,    0.0000_wp, &
      &  140.7115_wp,   76.9834_wp,   80.3464_wp,    0.0000_wp,    0.0000_wp, &
      &  131.3015_wp,   72.1828_wp,   75.3609_wp,    0.0000_wp,    0.0000_wp, &
      &  114.8256_wp,   63.4645_wp,   66.2607_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  110.1452_wp,   62.0309_wp,   64.9037_wp,    0.0000_wp,    0.0000_wp, &
      &   97.8345_wp,   55.5160_wp,   58.1127_wp,    0.0000_wp,    0.0000_wp, &
      &   85.2219_wp,   48.8396_wp,   51.1410_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   80.7975_wp,   46.6765_wp,   48.9131_wp,    0.0000_wp,    0.0000_wp, &
      &   63.4748_wp,   37.7222_wp,   39.6015_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   60.5594_wp,   35.7117_wp,   37.4554_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1413.1570_wp,  582.5270_wp,  580.9067_wp,    0.0000_wp,    0.0000_wp, &
      &  501.2393_wp,  231.2612_wp,  235.9718_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  962.9831_wp,  436.1091_wp,  443.7557_wp,    0.0000_wp,    0.0000_wp, &
      &  781.6993_wp,  358.3602_wp,  365.2736_wp,    0.0000_wp,    0.0000_wp, &
      &  467.4506_wp,  229.5754_wp,  236.7246_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  847.4615_wp,  397.8618_wp,  407.3331_wp,    0.0000_wp,    0.0000_wp, &
      &  759.4532_wp,  359.6497_wp,  368.7014_wp,    0.0000_wp,    0.0000_wp, &
      &  568.9077_wp,  277.8074_wp,  286.1317_wp,    0.0000_wp,    0.0000_wp, &
      &  420.8711_wp,  214.0193_wp,  221.7708_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  626.2353_wp,  309.1081_wp,  318.9550_wp,    0.0000_wp,    0.0000_wp, &
      &  573.4988_wp,  285.0957_wp,  294.4479_wp,    0.0000_wp,    0.0000_wp, &
      &  528.4593_wp,  264.6302_wp,  273.5887_wp,    0.0000_wp,    0.0000_wp, &
      &  518.6333_wp,  258.2554_wp,  266.7243_wp,    0.0000_wp,    0.0000_wp, &
      &  407.8160_wp,  210.3904_wp,  218.3896_wp,    0.0000_wp,    0.0000_wp, &
      &  462.3247_wp,  238.1644_wp,  247.1603_wp,    0.0000_wp,    0.0000_wp, &
      &  447.9590_wp,  231.0773_wp,  239.8294_wp,    0.0000_wp,    0.0000_wp, &
      &  443.4723_wp,  228.1087_wp,  236.6374_wp,    0.0000_wp,    0.0000_wp, &
      &  405.3462_wp,  210.9228_wp,  219.1391_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  370.2631_wp,  196.1128_wp,  204.1620_wp,    0.0000_wp,    0.0000_wp, &
      &  366.5151_wp,  193.9845_wp,  201.9181_wp,    0.0000_wp,    0.0000_wp, &
      &  358.8554_wp,  190.0290_wp,  197.8037_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(247 * max_ref * max_ref + 1 : 266 * max_ref * max_ref) = [ &
      &  292.1853_wp,  158.9892_wp,  165.9632_wp,    0.0000_wp,    0.0000_wp, &
      &  289.3512_wp,  157.3604_wp,  164.2459_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  231.9208_wp,  129.2180_wp,  135.1586_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2374.6690_wp,  960.3376_wp,  951.6268_wp,    0.0000_wp,    0.0000_wp, &
      &  668.7546_wp,  316.2518_wp,  323.6190_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1779.5162_wp,  774.6863_wp,  781.7679_wp,    0.0000_wp,    0.0000_wp, &
      & 1422.3244_wp,  619.4301_wp,  624.3036_wp,    0.0000_wp,    0.0000_wp, &
      &  656.7474_wp,  321.8441_wp,  331.4851_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1438.2841_wp,  636.8706_wp,  644.9388_wp,    0.0000_wp,    0.0000_wp, &
      &  635.0457_wp,  312.9202_wp,  322.5705_wp,    0.0000_wp,    0.0000_wp, &
      &  615.2494_wp,  307.5913_wp,  317.7589_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1361.9185_wp,  612.2225_wp,  621.6228_wp,    0.0000_wp,    0.0000_wp, &
      &  612.2225_wp,  301.9988_wp,  311.2408_wp,    0.0000_wp,    0.0000_wp, &
      &  621.6228_wp,  311.2408_wp,  321.5890_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   49.4819_wp,   28.4067_wp,   27.6880_wp,    0.0000_wp,    0.0000_wp, &
      &   83.8313_wp,   46.3089_wp,   44.8842_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   30.1851_wp,   18.4359_wp,   18.1127_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1069.0426_wp,  479.1402_wp,  448.2923_wp,    0.0000_wp,    0.0000_wp, &
      &  306.5482_wp,  154.5762_wp,  147.6410_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  533.3499_wp,  266.7050_wp,  254.3767_wp,    0.0000_wp,    0.0000_wp, &
      &  385.2754_wp,  196.4027_wp,  187.9531_wp,    0.0000_wp,    0.0000_wp, &
      &  231.8065_wp,  125.6778_wp,  121.4601_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  330.1023_wp,  175.8909_wp,  169.5221_wp,    0.0000_wp,    0.0000_wp, &
      &  272.8747_wp,  147.3153_wp,  142.2728_wp,    0.0000_wp,    0.0000_wp, &
      &  225.1783_wp,  123.1529_wp,  119.1771_wp,    0.0000_wp,    0.0000_wp, &
      &  162.9990_wp,   92.5451_wp,   90.0574_wp,    0.0000_wp,    0.0000_wp, &
      &  152.2500_wp,   86.9264_wp,   84.6570_wp,    0.0000_wp,    0.0000_wp, &
      &  209.1450_wp,  117.0098_wp,  113.6129_wp,    0.0000_wp,    0.0000_wp, &
      &  196.4885_wp,  109.7797_wp,  106.5711_wp,    0.0000_wp,    0.0000_wp, &
      &  155.8387_wp,   88.9429_wp,   86.6060_wp,    0.0000_wp,    0.0000_wp, &
      &  145.7580_wp,   83.2111_wp,   81.0229_wp,    0.0000_wp,    0.0000_wp, &
      &  114.5870_wp,   67.4383_wp,   65.9473_wp,    0.0000_wp,    0.0000_wp, &
      &  139.3512_wp,   80.9144_wp,   78.9761_wp,    0.0000_wp,    0.0000_wp, &
      &  129.8297_wp,   75.4303_wp,   73.6275_wp,    0.0000_wp,    0.0000_wp, &
      &  121.2119_wp,   70.7347_wp,   69.0809_wp,    0.0000_wp,    0.0000_wp, &
      &  106.0489_wp,   62.2020_wp,   60.7834_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  101.9325_wp,   60.7933_wp,   59.5445_wp,    0.0000_wp,    0.0000_wp, &
      &   90.6007_wp,   54.4109_wp,   53.3400_wp,    0.0000_wp,    0.0000_wp, &
      &   79.0003_wp,   47.8815_wp,   46.9875_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   74.9671_wp,   45.7630_wp,   44.9493_wp,    0.0000_wp,    0.0000_wp, &
      &   59.0665_wp,   36.9962_wp,   36.4513_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   56.3108_wp,   35.0290_wp,   34.4789_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1267.3139_wp,  571.6455_wp,  535.4483_wp,    0.0000_wp,    0.0000_wp, &
      &  454.8030_wp,  226.7886_wp,  216.1867_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  872.2579_wp,  427.7156_wp,  406.5642_wp,    0.0000_wp,    0.0000_wp, &
      &  708.7684_wp,  351.3928_wp,  334.6783_wp,    0.0000_wp,    0.0000_wp, &
      &  426.8815_wp,  224.9822_wp,  216.4395_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  770.3432_wp,  390.0432_wp,  372.8019_wp,    0.0000_wp,    0.0000_wp, &
      &  690.9156_wp,  352.5575_wp,  337.4128_wp,    0.0000_wp,    0.0000_wp, &
      &  519.1328_wp,  272.2496_wp,  261.7283_wp,    0.0000_wp,    0.0000_wp, &
      &  385.6611_wp,  209.6742_wp,  202.7168_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(266 * max_ref * max_ref + 1 : 285 * max_ref * max_ref) = [ &
      &  572.1129_wp,  302.8985_wp,  291.6318_wp,    0.0000_wp,    0.0000_wp, &
      &  524.2808_wp,  279.3563_wp,  269.2413_wp,    0.0000_wp,    0.0000_wp, &
      &  483.4599_wp,  259.2925_wp,  250.1591_wp,    0.0000_wp,    0.0000_wp, &
      &  474.1585_wp,  253.0661_wp,  243.9654_wp,    0.0000_wp,    0.0000_wp, &
      &  374.2069_wp,  206.1062_wp,  199.6661_wp,    0.0000_wp,    0.0000_wp, &
      &  424.1470_wp,  233.3176_wp,  225.9849_wp,    0.0000_wp,    0.0000_wp, &
      &  411.0121_wp,  226.3767_wp,  219.3033_wp,    0.0000_wp,    0.0000_wp, &
      &  406.7672_wp,  223.4784_wp,  216.4101_wp,    0.0000_wp,    0.0000_wp, &
      &  372.2326_wp,  206.6274_wp,  200.4053_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  340.6020_wp,  192.1113_wp,  186.7665_wp,    0.0000_wp,    0.0000_wp, &
      &  337.1250_wp,  190.0299_wp,  184.7241_wp,    0.0000_wp,    0.0000_wp, &
      &  330.0938_wp,  186.1576_wp,  180.9710_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  269.4863_wp,  155.7507_wp,  151.9416_wp,    0.0000_wp,    0.0000_wp, &
      &  266.8557_wp,  154.1576_wp,  150.3757_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  214.4002_wp,  126.6027_wp,  123.8635_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2124.1985_wp,  941.9837_wp,  879.7445_wp,    0.0000_wp,    0.0000_wp, &
      &  607.9951_wp,  310.1108_wp,  296.7431_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1604.9178_wp,  760.0718_wp,  718.1630_wp,    0.0000_wp,    0.0000_wp, &
      & 1282.1610_wp,  607.5571_wp,  574.2351_wp,    0.0000_wp,    0.0000_wp, &
      &  599.4521_wp,  315.4686_wp,  303.3970_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1299.5197_wp,  624.7801_wp,  591.8695_wp,    0.0000_wp,    0.0000_wp, &
      &  579.9584_wp,  306.7389_wp,  295.2374_wp,    0.0000_wp,    0.0000_wp, &
      &  562.7180_wp,  301.4420_wp,  290.7309_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1232.3235_wp,  600.5221_wp,  570.2282_wp,    0.0000_wp,    0.0000_wp, &
      &  559.1157_wp,  296.0029_wp,  284.9497_wp,    0.0000_wp,    0.0000_wp, &
      &  568.6316_wp,  305.0201_wp,  294.2417_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1116.0984_wp,  548.3908_wp,  521.4602_wp,    0.0000_wp,    0.0000_wp, &
      &  548.3908_wp,  290.1436_wp,  279.2755_wp,    0.0000_wp,    0.0000_wp, &
      &  521.4602_wp,  279.2755_wp,  269.3349_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   39.1221_wp,   28.0562_wp,   21.3134_wp,    0.0000_wp,    0.0000_wp, &
      &   65.9837_wp,   45.8173_wp,   33.9427_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   24.1463_wp,   18.1754_wp,   14.4051_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  844.0936_wp,  480.5944_wp,  314.2807_wp,    0.0000_wp,    0.0000_wp, &
      &  240.5033_wp,  153.7539_wp,  107.7527_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  418.1635_wp,  265.4088_wp,  184.9079_wp,    0.0000_wp,    0.0000_wp, &
      &  302.6061_wp,  195.2317_wp,  137.8919_wp,    0.0000_wp,    0.0000_wp, &
      &  182.1558_wp,  124.4645_wp,   91.0985_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  259.3021_wp,  174.3719_wp,  126.3736_wp,    0.0000_wp,    0.0000_wp, &
      &  214.6360_wp,  145.9477_wp,  106.7244_wp,    0.0000_wp,    0.0000_wp, &
      &  177.4000_wp,  121.9340_wp,   89.9610_wp,    0.0000_wp,    0.0000_wp, &
      &  128.7233_wp,   91.4528_wp,   68.9958_wp,    0.0000_wp,    0.0000_wp, &
      &  120.3278_wp,   85.8806_wp,   65.0349_wp,    0.0000_wp,    0.0000_wp, &
      &  164.9882_wp,  115.7182_wp,   86.5464_wp,    0.0000_wp,    0.0000_wp, &
      &  155.1171_wp,  108.5820_wp,   81.2335_wp,    0.0000_wp,    0.0000_wp, &
      &  123.2591_wp,   87.8889_wp,   66.6131_wp,    0.0000_wp,    0.0000_wp, &
      &  115.3697_wp,   82.2331_wp,   62.3963_wp,    0.0000_wp,    0.0000_wp, &
      &   91.0233_wp,   66.5568_wp,   51.4610_wp,    0.0000_wp,    0.0000_wp, &
      &  110.4861_wp,   79.9024_wp,   61.2471_wp,    0.0000_wp,    0.0000_wp, &
      &  103.0189_wp,   74.4890_wp,   57.1766_wp,    0.0000_wp,    0.0000_wp, &
      &   96.2547_wp,   69.8471_wp,   53.7726_wp,    0.0000_wp,    0.0000_wp, &
      &   84.3631_wp,   61.4202_wp,   47.5033_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   81.2000_wp,   59.9829_wp,   46.8138_wp,    0.0000_wp,    0.0000_wp, &
      &   72.2872_wp,   53.6733_wp,   42.1046_wp,    0.0000_wp,    0.0000_wp, &
      &   63.1867_wp,   47.2297_wp,   37.3024_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   60.0099_wp,   45.1290_wp,   35.7811_wp,    0.0000_wp,    0.0000_wp, &
      &   47.5318_wp,   36.4594_wp,   29.3923_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(285 * max_ref * max_ref + 1 : 304 * max_ref * max_ref) = [ &
      &   45.2820_wp,   34.5350_wp,   27.7372_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1000.4652_wp,  573.1436_wp,  376.4906_wp,    0.0000_wp,    0.0000_wp, &
      &  357.1508_wp,  225.7837_wp,  157.3830_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  684.4015_wp,  426.2276_wp,  293.6777_wp,    0.0000_wp,    0.0000_wp, &
      &  556.6127_wp,  349.9328_wp,  243.0049_wp,    0.0000_wp,    0.0000_wp, &
      &  335.0118_wp,  223.1855_wp,  160.5554_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  604.4071_wp,  387.8695_wp,  272.5015_wp,    0.0000_wp,    0.0000_wp, &
      &  542.2459_wp,  350.4296_wp,  247.4372_wp,    0.0000_wp,    0.0000_wp, &
      &  407.7288_wp,  270.1623_wp,  193.9815_wp,    0.0000_wp,    0.0000_wp, &
      &  303.1266_wp,  207.6299_wp,  152.2308_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  449.2216_wp,  300.3928_wp,  216.8001_wp,    0.0000_wp,    0.0000_wp, &
      &  411.8879_wp,  276.9512_wp,  200.7344_wp,    0.0000_wp,    0.0000_wp, &
      &  379.9385_wp,  256.9682_wp,  187.0036_wp,    0.0000_wp,    0.0000_wp, &
      &  372.7838_wp,  250.8827_wp,  182.1854_wp,    0.0000_wp,    0.0000_wp, &
      &  294.5037_wp,  203.9633_wp,  150.8344_wp,    0.0000_wp,    0.0000_wp, &
      &  333.8380_wp,  230.9090_wp,  170.6767_wp,    0.0000_wp,    0.0000_wp, &
      &  323.5914_wp,  224.0303_wp,  165.7628_wp,    0.0000_wp,    0.0000_wp, &
      &  320.2734_wp,  221.1986_wp,  163.4733_wp,    0.0000_wp,    0.0000_wp, &
      &  293.2680_wp,  204.4081_wp,  152.0010_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  268.8029_wp,  189.9052_wp,  142.6795_wp,    0.0000_wp,    0.0000_wp, &
      &  266.0762_wp,  187.8570_wp,  141.1092_wp,    0.0000_wp,    0.0000_wp, &
      &  260.5647_wp,  184.0280_wp,  138.2944_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  213.3668_wp,  153.8057_wp,  117.4258_wp,    0.0000_wp,    0.0000_wp, &
      &  211.2928_wp,  152.2383_wp,  116.2100_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  170.3465_wp,  124.9265_wp,   96.7593_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1683.6701_wp,  945.8700_wp,  617.0566_wp,    0.0000_wp,    0.0000_wp, &
      &  478.9744_wp,  308.3901_wp,  218.9044_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1263.0809_wp,  759.3684_wp,  513.7768_wp,    0.0000_wp,    0.0000_wp, &
      & 1011.5955_wp,  607.0700_wp,  412.0610_wp,    0.0000_wp,    0.0000_wp, &
      &  471.3832_wp,  313.0538_wp,  225.6881_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1021.6871_wp,  623.5332_wp,  425.4745_wp,    0.0000_wp,    0.0000_wp, &
      &  456.2023_wp,  304.3094_wp,  220.1013_wp,    0.0000_wp,    0.0000_wp, &
      &  442.6736_wp,  298.8190_wp,  217.6494_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  968.8565_wp,  598.7981_wp,  412.1387_wp,    0.0000_wp,    0.0000_wp, &
      &  440.0762_wp,  293.6598_wp,  212.6450_wp,    0.0000_wp,    0.0000_wp, &
      &  447.3763_wp,  302.3474_wp,  220.4170_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  877.3855_wp,  546.5244_wp,  378.0660_wp,    0.0000_wp,    0.0000_wp, &
      &  431.6683_wp,  287.8657_wp,  208.4094_wp,    0.0000_wp,    0.0000_wp, &
      &  410.5661_wp,  276.8819_wp,  201.8641_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  690.7425_wp,  430.2353_wp,  298.4894_wp,    0.0000_wp,    0.0000_wp, &
      &  430.2353_wp,  285.6905_wp,  206.3426_wp,    0.0000_wp,    0.0000_wp, &
      &  298.4894_wp,  206.3426_wp,  153.3435_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   43.0028_wp,   26.1810_wp,   18.4694_wp,    0.0000_wp,    0.0000_wp, &
      &   72.4111_wp,   42.3927_wp,   29.1208_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   26.4978_wp,   17.1526_wp,   12.6985_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  891.9756_wp,  416.0836_wp,  256.5689_wp,    0.0000_wp,    0.0000_wp, &
      &  260.9019_wp,  138.7491_wp,   90.4231_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  453.3686_wp,  238.9606_wp,  154.7687_wp,    0.0000_wp,    0.0000_wp, &
      &  328.4012_wp,  176.6817_wp,  116.1032_wp,    0.0000_wp,    0.0000_wp, &
      &  199.6426_wp,  114.6465_wp,   77.7823_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(304 * max_ref * max_ref + 1 : 323 * max_ref * max_ref) = [ &
      &  283.4744_wp,  159.8492_wp,  107.4866_wp,    0.0000_wp,    0.0000_wp, &
      &  234.8000_wp,  134.2136_wp,   91.1091_wp,    0.0000_wp,    0.0000_wp, &
      &  194.1378_wp,  112.4648_wp,   77.0812_wp,    0.0000_wp,    0.0000_wp, &
      &  141.4071_wp,   85.1312_wp,   59.6313_wp,    0.0000_wp,    0.0000_wp, &
      &  132.1984_wp,   80.0375_wp,   56.2921_wp,    0.0000_wp,    0.0000_wp, &
      &  181.0035_wp,  107.3382_wp,   74.5511_wp,    0.0000_wp,    0.0000_wp, &
      &  169.9796_wp,  100.6463_wp,   69.9883_wp,    0.0000_wp,    0.0000_wp, &
      &  135.2994_wp,   81.8731_wp,   57.6893_wp,    0.0000_wp,    0.0000_wp, &
      &  126.5416_wp,   76.5846_wp,   54.0676_wp,    0.0000_wp,    0.0000_wp, &
      &   99.9768_wp,   62.3931_wp,   44.9179_wp,    0.0000_wp,    0.0000_wp, &
      &  121.3233_wp,   74.6998_wp,   53.2828_wp,    0.0000_wp,    0.0000_wp, &
      &  113.0248_wp,   69.6215_wp,   49.7698_wp,    0.0000_wp,    0.0000_wp, &
      &  105.6080_wp,   65.3415_wp,   46.8681_wp,    0.0000_wp,    0.0000_wp, &
      &   92.4636_wp,   57.4915_wp,   41.4862_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   89.1346_wp,   56.3622_wp,   41.0237_wp,    0.0000_wp,    0.0000_wp, &
      &   79.3047_wp,   50.4858_wp,   36.9692_wp,    0.0000_wp,    0.0000_wp, &
      &   69.2598_wp,   44.4888_wp,   32.8495_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   65.8112_wp,   42.5764_wp,   31.5582_wp,    0.0000_wp,    0.0000_wp, &
      &   52.0780_wp,   34.5481_wp,   26.0926_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   49.5959_wp,   32.6806_wp,   24.5957_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1058.5103_wp,  497.3582_wp,  307.9557_wp,    0.0000_wp,    0.0000_wp, &
      &  386.3426_wp,  202.9676_wp,  131.8257_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  739.0411_wp,  381.2968_wp,  244.7517_wp,    0.0000_wp,    0.0000_wp, &
      &  601.4499_wp,  314.0119_wp,  203.2173_wp,    0.0000_wp,    0.0000_wp, &
      &  366.0024_wp,  204.0446_wp,  136.1619_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  656.0738_wp,  350.4522_wp,  228.9125_wp,    0.0000_wp,    0.0000_wp, &
      &  589.1475_wp,  317.3287_wp,  208.2902_wp,    0.0000_wp,    0.0000_wp, &
      &  444.6283_wp,  246.5588_wp,  164.4045_wp,    0.0000_wp,    0.0000_wp, &
      &  332.3156_wp,  191.3897_wp,  130.0809_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  490.8156_wp,  274.9257_wp,  184.0945_wp,    0.0000_wp,    0.0000_wp, &
      &  450.2234_wp,  253.8800_wp,  170.7515_wp,    0.0000_wp,    0.0000_wp, &
      &  415.6112_wp,  235.9676_wp,  159.3286_wp,    0.0000_wp,    0.0000_wp, &
      &  407.2425_wp,  230.0182_wp,  155.1095_wp,    0.0000_wp,    0.0000_wp, &
      &  323.0948_wp,  188.5834_wp,  129.3324_wp,    0.0000_wp,    0.0000_wp, &
      &  366.1204_wp,  213.4101_wp,  146.3175_wp,    0.0000_wp,    0.0000_wp, &
      &  354.8432_wp,  207.0995_wp,  142.1674_wp,    0.0000_wp,    0.0000_wp, &
      &  351.0224_wp,  204.3319_wp,  140.1443_wp,    0.0000_wp,    0.0000_wp, &
      &  321.7679_wp,  189.3102_wp,  130.6248_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  295.1737_wp,  176.5165_wp,  123.1170_wp,    0.0000_wp,    0.0000_wp, &
      &  292.1257_wp,  174.5775_wp,  121.7541_wp,    0.0000_wp,    0.0000_wp, &
      &  286.0527_wp,  171.0306_wp,  119.3486_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  234.4533_wp,  143.7001_wp,  101.9746_wp,    0.0000_wp,    0.0000_wp, &
      &  232.1447_wp,  142.2151_wp,  100.9143_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  187.1728_wp,  117.2057_wp,   84.5107_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1768.0322_wp,  814.2468_wp,  503.8621_wp,    0.0000_wp,    0.0000_wp, &
      &  518.0583_wp,  278.7259_wp,  184.8170_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1351.4785_wp,  670.8337_wp,  425.2382_wp,    0.0000_wp,    0.0000_wp, &
      & 1079.1532_wp,  535.8149_wp,  341.7674_wp,    0.0000_wp,    0.0000_wp, &
      &  513.6516_wp,  285.8068_wp,  191.6322_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1097.1539_wp,  553.7931_wp,  353.3283_wp,    0.0000_wp,    0.0000_wp, &
      &  497.3474_wp,  278.1965_wp,  187.1364_wp,    0.0000_wp,    0.0000_wp, &
      &  483.5993_wp,  274.1703_wp,  185.5544_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1042.6625_wp,  534.1266_wp,  343.4916_wp,    0.0000_wp,    0.0000_wp, &
      &  479.4969_wp,  268.4657_wp,  180.9105_wp,    0.0000_wp,    0.0000_wp, &
      &  488.7888_wp,  277.5005_wp,  187.9838_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  945.6072_wp,  488.7947_wp,  315.7492_wp,    0.0000_wp,    0.0000_wp, &
      &  470.2529_wp,  263.1127_wp,  177.2993_wp,    0.0000_wp,    0.0000_wp, &
      &  448.0698_wp,  253.9370_wp,  172.1980_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(323 * max_ref * max_ref + 1 : 342 * max_ref * max_ref) = [ &
      &  743.3179_wp,  384.6931_wp,  249.6819_wp,    0.0000_wp,    0.0000_wp, &
      &  468.2961_wp,  260.7773_wp,  175.3867_wp,    0.0000_wp,    0.0000_wp, &
      &  326.3558_wp,  190.5528_wp,  131.8288_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  802.7484_wp,  420.4207_wp,  273.3937_wp,    0.0000_wp,    0.0000_wp, &
      &  420.4207_wp,  239.5554_wp,  162.6844_wp,    0.0000_wp,    0.0000_wp, &
      &  273.3937_wp,  162.6844_wp,  113.8463_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   33.9110_wp,   24.3452_wp,   18.0977_wp,    0.0000_wp,    0.0000_wp, &
      &   56.7993_wp,   39.2742_wp,   28.5665_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   21.1750_wp,   16.0560_wp,   12.4277_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  698.4539_wp,  379.2117_wp,  253.2432_wp,    0.0000_wp,    0.0000_wp, &
      &  203.6149_wp,  127.5492_wp,   88.9538_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  353.5358_wp,  219.4775_wp,  152.2986_wp,    0.0000_wp,    0.0000_wp, &
      &  256.6043_wp,  162.6108_wp,  114.1613_wp,    0.0000_wp,    0.0000_wp, &
      &  156.2845_wp,  106.0286_wp,   76.3494_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  221.7547_wp,  147.6298_wp,  105.5581_wp,    0.0000_wp,    0.0000_wp, &
      &  183.9737_wp,  124.1170_wp,   89.4354_wp,    0.0000_wp,    0.0000_wp, &
      &  152.3887_wp,  104.1437_wp,   75.6327_wp,    0.0000_wp,    0.0000_wp, &
      &  111.3569_wp,   79.0832_wp,   58.4515_wp,    0.0000_wp,    0.0000_wp, &
      &  104.1990_wp,   74.3930_wp,   55.1698_wp,    0.0000_wp,    0.0000_wp, &
      &  142.3467_wp,   99.5880_wp,   73.1079_wp,    0.0000_wp,    0.0000_wp, &
      &  133.7700_wp,   93.3848_wp,   68.6293_wp,    0.0000_wp,    0.0000_wp, &
      &  106.7326_wp,   76.1132_wp,   56.5429_wp,    0.0000_wp,    0.0000_wp, &
      &   99.9017_wp,   71.2110_wp,   52.9928_wp,    0.0000_wp,    0.0000_wp, &
      &   79.2630_wp,   58.1776_wp,   43.9902_wp,    0.0000_wp,    0.0000_wp, &
      &   95.9782_wp,   69.5638_wp,   52.2028_wp,    0.0000_wp,    0.0000_wp, &
      &   89.4868_wp,   64.8468_wp,   48.7558_wp,    0.0000_wp,    0.0000_wp, &
      &   83.6916_wp,   60.8920_wp,   45.9140_wp,    0.0000_wp,    0.0000_wp, &
      &   73.4185_wp,   53.6171_wp,   40.6376_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   70.8942_wp,   52.6351_wp,   40.1709_wp,    0.0000_wp,    0.0000_wp, &
      &   63.1865_wp,   47.1816_wp,   36.1919_wp,    0.0000_wp,    0.0000_wp, &
      &   55.3356_wp,   41.6270_wp,   32.1584_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   52.6310_wp,   39.8628_wp,   30.8925_wp,    0.0000_wp,    0.0000_wp, &
      &   41.8942_wp,   32.4321_wp,   25.5314_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   39.8659_wp,   30.6648_wp,   24.0741_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  828.8936_wp,  453.5444_wp,  303.9124_wp,    0.0000_wp,    0.0000_wp, &
      &  301.7401_wp,  186.4669_wp,  129.7332_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  576.5121_wp,  349.7116_wp,  240.9962_wp,    0.0000_wp,    0.0000_wp, &
      &  469.6437_wp,  288.3371_wp,  200.0041_wp,    0.0000_wp,    0.0000_wp, &
      &  285.9969_wp,  188.2525_wp,  133.7853_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  512.0823_wp,  322.2781_wp,  225.1663_wp,    0.0000_wp,    0.0000_wp, &
      &  460.0496_wp,  292.0247_wp,  204.8282_wp,    0.0000_wp,    0.0000_wp, &
      &  347.6394_wp,  227.4327_wp,  161.5357_wp,    0.0000_wp,    0.0000_wp, &
      &  260.2229_wp,  177.0542_wp,  127.6833_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  383.7445_wp,  253.7638_wp,  180.8346_wp,    0.0000_wp,    0.0000_wp, &
      &  352.2408_wp,  234.4843_wp,  167.6917_wp,    0.0000_wp,    0.0000_wp, &
      &  325.3117_wp,  218.0646_wp,  156.4437_wp,    0.0000_wp,    0.0000_wp, &
      &  318.8589_wp,  212.5126_wp,  152.3164_wp,    0.0000_wp,    0.0000_wp, &
      &  253.3965_wp,  174.6758_wp,  126.8932_wp,    0.0000_wp,    0.0000_wp, &
      &  287.1606_wp,  197.6569_wp,  143.5585_wp,    0.0000_wp,    0.0000_wp, &
      &  278.4020_wp,  191.8429_wp,  139.4802_wp,    0.0000_wp,    0.0000_wp, &
      &  275.4138_wp,  189.2497_wp,  137.5043_wp,    0.0000_wp,    0.0000_wp, &
      &  252.6740_wp,  175.4919_wp,  128.1271_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  232.2625_wp,  163.8794_wp,  120.7060_wp,    0.0000_wp,    0.0000_wp, &
      &  229.8776_wp,  162.0750_wp,  119.3719_wp,    0.0000_wp,    0.0000_wp, &
      &  225.1357_wp,  158.7933_wp,  117.0121_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  185.1823_wp,  133.7324_wp,   99.9119_wp,    0.0000_wp,    0.0000_wp, &
      &  183.3661_wp,  132.3480_wp,   98.8747_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(342 * max_ref * max_ref + 1 : 361 * max_ref * max_ref) = [ &
      &  148.4319_wp,  109.3155_wp,   82.7591_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1388.9522_wp,  742.2520_wp,  497.3105_wp,    0.0000_wp,    0.0000_wp, &
      &  406.0876_wp,  256.7579_wp,  181.7040_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1056.5193_wp,  613.9233_wp,  419.0635_wp,    0.0000_wp,    0.0000_wp, &
      &  845.4686_wp,  490.7481_wp,  336.7038_wp,    0.0000_wp,    0.0000_wp, &
      &  402.1872_wp,  263.7930_wp,  188.2696_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  857.1730_wp,  507.3401_wp,  348.0642_wp,    0.0000_wp,    0.0000_wp, &
      &  389.5974_wp,  256.8889_wp,  183.8257_wp,    0.0000_wp,    0.0000_wp, &
      &  378.9366_wp,  253.4168_wp,  182.2062_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  814.8402_wp,  489.8996_wp,  338.2283_wp,    0.0000_wp,    0.0000_wp, &
      &  375.8242_wp,  247.9636_wp,  177.7007_wp,    0.0000_wp,    0.0000_wp, &
      &  383.0606_wp,  256.5291_wp,  184.5856_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  739.0542_wp,  448.6262_wp,  310.8375_wp,    0.0000_wp,    0.0000_wp, &
      &  368.6136_wp,  243.0148_wp,  174.1594_wp,    0.0000_wp,    0.0000_wp, &
      &  351.3939_wp,  234.7665_wp,  169.0900_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  581.8114_wp,  353.2709_wp,  245.7629_wp,    0.0000_wp,    0.0000_wp, &
      &  367.0785_wp,  240.7834_wp,  172.3036_wp,    0.0000_wp,    0.0000_wp, &
      &  256.8278_wp,  176.6607_wp,  129.3444_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  627.5403_wp,  386.2602_wp,  269.0494_wp,    0.0000_wp,    0.0000_wp, &
      &  329.6273_wp,  221.5340_wp,  159.7402_wp,    0.0000_wp,    0.0000_wp, &
      &  215.5738_wp,  151.0768_wp,  111.6491_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  491.3349_wp,  303.0480_wp,  212.1109_wp,    0.0000_wp,    0.0000_wp, &
      &  303.0480_wp,  204.9925_wp,  148.3171_wp,    0.0000_wp,    0.0000_wp, &
      &  212.1109_wp,  148.3171_wp,  109.5041_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   36.3234_wp,   23.2310_wp,   20.7741_wp,   18.7500_wp,    0.0000_wp, &
      &   60.5746_wp,   37.4496_wp,   33.3307_wp,   29.7680_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   22.7233_wp,   15.3545_wp,   13.8804_wp,   12.7523_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  701.5324_wp,  362.3746_wp,  319.8448_wp,  270.7598_wp,    0.0000_wp, &
      &  212.8978_wp,  121.6037_wp,  107.5582_wp,   93.8356_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  369.1928_wp,  209.2227_wp,  184.8938_wp,  160.8870_wp,    0.0000_wp, &
      &  268.6284_wp,  155.0554_wp,  137.2998_wp,  120.1880_wp,    0.0000_wp, &
      &  166.2166_wp,  101.0801_wp,   89.7891_wp,   79.7845_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  234.8693_wp,  140.7428_wp,  124.9087_wp,  110.5365_wp,    0.0000_wp, &
      &  195.1731_wp,  118.3530_wp,  105.2019_wp,   93.4541_wp,    0.0000_wp, &
      &  161.8720_wp,   99.3343_wp,   88.4405_wp,   78.8624_wp,    0.0000_wp, &
      &  119.1091_wp,   75.4525_wp,   67.3932_wp,   60.6537_wp,    0.0000_wp, &
      &  111.5063_wp,   70.9871_wp,   63.4543_wp,   57.1987_wp,    0.0000_wp, &
      &  151.8703_wp,   95.0051_wp,   84.7552_wp,   76.0072_wp,    0.0000_wp, &
      &  142.5140_wp,   89.0992_wp,   79.5273_wp,   71.3318_wp,    0.0000_wp, &
      &  114.0913_wp,   72.6443_wp,   64.9856_wp,   58.6057_wp,    0.0000_wp, &
      &  106.6917_wp,   67.9774_wp,   60.8513_wp,   54.9066_wp,    0.0000_wp, &
      &   84.9547_wp,   55.5654_wp,   49.9148_wp,   45.3891_wp,    0.0000_wp, &
      &  102.7503_wp,   66.4218_wp,   59.5628_wp,   53.9695_wp,    0.0000_wp, &
      &   95.7051_wp,   61.9256_wp,   55.5685_wp,   50.3814_wp,    0.0000_wp, &
      &   89.5345_wp,   58.1611_wp,   52.2322_wp,   47.4168_wp,    0.0000_wp, &
      &   78.4696_wp,   51.2318_wp,   46.0841_wp,   41.9193_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   75.9912_wp,   50.3016_wp,   45.3066_wp,   41.3623_wp,    0.0000_wp, &
      &   67.7117_wp,   45.1002_wp,   40.6799_wp,   37.2178_wp,    0.0000_wp, &
      &   59.2688_wp,   39.8131_wp,   35.9892_wp,   33.0215_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   56.4307_wp,   38.1313_wp,   34.4935_wp,   31.6985_wp,    0.0000_wp, &
      &   44.9407_wp,   31.0516_wp,   28.2122_wp,   26.1031_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   42.7270_wp,   29.3598_wp,   26.6627_wp,   24.6343_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(361 * max_ref * max_ref + 1 : 380 * max_ref * max_ref) = [ &
      &  834.1826_wp,  433.3736_wp,  382.6821_wp,  324.6241_wp,    0.0000_wp, &
      &  314.1718_wp,  177.8355_wp,  157.3578_wp,  136.9841_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  598.3302_wp,  333.4767_wp,  294.6091_wp,  255.1561_wp,    0.0000_wp, &
      &  488.1715_wp,  274.9840_wp,  243.1809_wp,  211.3388_wp,    0.0000_wp, &
      &  302.4518_wp,  179.4553_wp,  159.0980_wp,  140.3518_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  535.9706_wp,  307.2540_wp,  271.8226_wp,  237.3961_wp,    0.0000_wp, &
      &  482.2990_wp,  278.4173_wp,  246.4539_wp,  215.7066_wp,    0.0000_wp, &
      &  366.7236_wp,  216.8393_wp,  192.2755_wp,  169.4903_wp,    0.0000_wp, &
      &  276.8948_wp,  168.8027_wp,  149.9951_wp,  133.3820_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  405.9907_wp,  241.9159_wp,  214.5726_wp,  189.5488_wp,    0.0000_wp, &
      &  373.0125_wp,  223.5548_wp,  198.4174_wp,  175.5963_wp,    0.0000_wp, &
      &  344.9479_wp,  207.9084_wp,  184.6285_wp,  163.6721_wp,    0.0000_wp, &
      &  337.4624_wp,  202.6388_wp,  179.9690_wp,  159.4063_wp,    0.0000_wp, &
      &  270.0881_wp,  166.5674_wp,  148.2259_wp,  132.2884_wp,    0.0000_wp, &
      &  305.9256_wp,  188.4852_wp,  167.7356_wp,  149.6709_wp,    0.0000_wp, &
      &  296.5789_wp,  182.9511_wp,  162.8541_wp,  145.3801_wp,    0.0000_wp, &
      &  293.1661_wp,  180.4846_wp,  160.6578_wp,  143.3518_wp,    0.0000_wp, &
      &  269.4807_wp,  167.3765_wp,  149.1158_wp,  133.3956_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  248.2116_wp,  156.3423_wp,  139.5430_wp,  125.3725_wp,    0.0000_wp, &
      &  245.5988_wp,  154.6243_wp,  138.0159_wp,  123.9905_wp,    0.0000_wp, &
      &  240.5175_wp,  151.4983_wp,  135.2456_wp,  121.5256_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  198.3570_wp,  127.6513_wp,  114.3091_wp,  103.3931_wp,    0.0000_wp, &
      &  196.3755_wp,  126.3319_wp,  113.1318_wp,  102.3223_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  159.1957_wp,  104.4091_wp,   93.8084_wp,   85.3616_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1382.9771_wp,  710.1818_wp,  628.1238_wp,  531.3044_wp,    0.0000_wp, &
      &  423.3809_wp,  245.0188_wp,  217.5967_wp,  190.9521_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1081.7690_wp,  585.9578_wp,  517.8769_wp,  445.0681_wp,    0.0000_wp, &
      &  862.5036_wp,  468.6899_wp,  414.8338_wp,  357.0721_wp,    0.0000_wp, &
      &  423.9523_wp,  251.5841_wp,  223.4273_wp,  197.3201_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  882.4503_wp,  484.0723_wp,  427.8866_wp,  369.0835_wp,    0.0000_wp, &
      &  411.0572_wp,  245.0035_wp,  217.6909_wp,  192.5221_wp,    0.0000_wp, &
      &  401.1295_wp,  241.6833_wp,  214.8399_wp,  190.5518_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  841.8335_wp,  467.3984_wp,  413.4479_wp,  357.9783_wp,    0.0000_wp, &
      &  396.2748_wp,  236.5325_wp,  210.2464_wp,  186.0369_wp,    0.0000_wp, &
      &  405.5809_wp,  244.6571_wp,  217.5178_wp,  193.0012_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  765.3178_wp,  427.9936_wp,  378.7327_wp,  328.6407_wp,    0.0000_wp, &
      &  388.5729_wp,  231.8180_wp,  206.0757_wp,  182.3353_wp,    0.0000_wp, &
      &  371.5149_wp,  223.9437_wp,  199.2038_wp,  176.7692_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  601.4178_wp,  337.1436_wp,  298.7169_wp,  259.5722_wp,    0.0000_wp, &
      &  386.4486_wp,  229.7022_wp,  204.1750_wp,  180.4774_wp,    0.0000_wp, &
      &  272.6301_wp,  168.5969_wp,  150.4863_wp,  134.6287_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  651.9688_wp,  368.4719_wp,  326.2603_wp,  284.0167_wp,    0.0000_wp, &
      &  349.1941_wp,  211.3025_wp,  187.9564_wp,  166.9417_wp,    0.0000_wp, &
      &  229.5077_wp,  144.2159_wp,  128.9628_wp,  115.9223_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  509.7764_wp,  289.1937_wp,  256.4238_wp,  223.6354_wp,    0.0000_wp, &
      &  321.3448_wp,  195.5430_wp,  174.0536_wp,  154.8608_wp,    0.0000_wp, &
      &  225.7346_wp,  141.5815_wp,  126.5885_wp,  113.7287_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  532.7794_wp,  306.4968_wp,  271.6288_wp,  237.6994_wp,    0.0000_wp, &
      &  306.4968_wp,  186.5462_wp,  166.0911_wp,  147.8057_wp,    0.0000_wp, &
      &  271.6288_wp,  166.0911_wp,  148.0657_wp,  132.0085_wp,    0.0000_wp, &
      &  237.6994_wp,  147.8057_wp,  132.0085_wp,  118.2863_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   37.1596_wp,   22.6822_wp,   19.4648_wp,   20.1853_wp,    0.0000_wp, &
      &   62.2532_wp,   36.5901_wp,   31.0981_wp,   32.5149_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   23.0792_wp,   14.9833_wp,   13.0979_wp,   13.4305_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(380 * max_ref * max_ref + 1 : 399 * max_ref * max_ref) = [ &
      &  738.8580_wp,  355.9606_wp,  291.0488_wp,  320.6585_wp,    0.0000_wp, &
      &  221.1739_wp,  119.0725_wp,   99.3454_wp,  106.1545_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  383.9140_wp,  204.9086_wp,  170.5936_wp,  182.6955_wp,    0.0000_wp, &
      &  278.7116_wp,  151.7798_wp,  126.9812_wp,  135.2702_wp,    0.0000_wp, &
      &  171.2043_wp,   98.8004_wp,   83.6030_wp,   87.7852_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  242.4174_wp,  137.6254_wp,  116.0923_wp,  122.3915_wp,    0.0000_wp, &
      &  201.1351_wp,  115.7000_wp,   97.9268_wp,  102.9220_wp,    0.0000_wp, &
      &  166.5614_wp,   97.0816_wp,   82.4474_wp,   86.3895_wp,    0.0000_wp, &
      &  122.0144_wp,   73.6865_wp,   63.0740_wp,   65.5618_wp,    0.0000_wp, &
      &  114.1506_wp,   69.3194_wp,   59.4247_wp,   61.6974_wp,    0.0000_wp, &
      &  155.8507_wp,   92.8107_wp,   79.2056_wp,   82.5930_wp,    0.0000_wp, &
      &  146.2711_wp,   87.0432_wp,   74.3149_wp,   77.5124_wp,    0.0000_wp, &
      &  116.8072_wp,   70.9444_wp,   60.8678_wp,   63.2092_wp,    0.0000_wp, &
      &  109.2305_wp,   66.3897_wp,   57.0046_wp,   59.1980_wp,    0.0000_wp, &
      &   86.6601_wp,   54.2397_wp,   46.9071_wp,   48.4149_wp,    0.0000_wp, &
      &  104.9834_wp,   64.8519_wp,   55.8954_wp,   57.8469_wp,    0.0000_wp, &
      &   97.7761_wp,   60.4608_wp,   52.1532_wp,   53.9690_wp,    0.0000_wp, &
      &   91.4295_wp,   56.7868_wp,   49.0511_wp,   50.7198_wp,    0.0000_wp, &
      &   80.0853_wp,   50.0218_wp,   43.3096_wp,   44.7442_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   77.4010_wp,   49.0996_wp,   42.6462_wp,   43.9161_wp,    0.0000_wp, &
      &   68.9086_wp,   44.0173_wp,   38.3194_wp,   39.4125_wp,    0.0000_wp, &
      &   60.2575_wp,   38.8597_wp,   33.9427_wp,   34.8596_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   57.3248_wp,   37.2158_wp,   32.5553_wp,   33.3924_wp,    0.0000_wp, &
      &   45.5155_wp,   30.3004_wp,   26.7000_wp,   27.2705_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   43.3129_wp,   28.6554_wp,   25.2210_wp,   25.7940_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  877.9354_wp,  425.6455_wp,  348.5798_wp,  383.3773_wp,    0.0000_wp, &
      &  326.8271_wp,  174.2001_wp,  145.1854_wp,  155.5786_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  623.6305_wp,  326.7864_wp,  271.2223_wp,  291.9263_wp,    0.0000_wp, &
      &  508.1585_wp,  269.3779_wp,  224.1812_wp,  240.5302_wp,    0.0000_wp, &
      &  312.5862_wp,  175.5343_wp,  147.6912_wp,  156.1327_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  556.5389_wp,  300.8271_wp,  251.1877_wp,  268.1207_wp,    0.0000_wp, &
      &  500.3525_wp,  272.5409_wp,  227.9565_wp,  242.8419_wp,    0.0000_wp, &
      &  379.2280_wp,  212.1194_wp,  178.3984_wp,  188.7699_wp,    0.0000_wp, &
      &  285.1174_wp,  164.9924_wp,  139.7108_wp,  146.6121_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  419.3593_wp,  236.5952_wp,  199.2855_wp,  210.4188_wp,    0.0000_wp, &
      &  385.0048_wp,  218.6066_wp,  184.4175_wp,  194.4184_wp,    0.0000_wp, &
      &  355.7641_wp,  203.2777_wp,  171.7269_wp,  180.7673_wp,    0.0000_wp, &
      &  348.2477_wp,  198.1506_wp,  167.3173_wp,  176.3143_wp,    0.0000_wp, &
      &  277.6794_wp,  162.7632_wp,  138.2635_wp,  144.6644_wp,    0.0000_wp, &
      &  314.5698_wp,  184.1847_wp,  156.4432_wp,  163.7299_wp,    0.0000_wp, &
      &  304.9145_wp,  178.7735_wp,  151.9152_wp,  158.9453_wp,    0.0000_wp, &
      &  301.4974_wp,  176.3750_wp,  149.8328_wp,  156.8549_wp,    0.0000_wp, &
      &  276.7985_wp,  163.5301_wp,  139.2201_wp,  145.4133_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  254.4683_wp,  152.7036_wp,  130.5102_wp,  135.8470_wp,    0.0000_wp, &
      &  251.8097_wp,  151.0286_wp,  129.0763_wp,  134.3737_wp,    0.0000_wp, &
      &  246.5864_wp,  147.9753_wp,  126.4947_wp,  131.6741_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  202.7705_wp,  124.6317_wp,  107.1976_wp,  111.0248_wp,    0.0000_wp, &
      &  200.7571_wp,  123.3457_wp,  106.0905_wp,  109.8900_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  162.3264_wp,  101.9116_wp,   88.1829_wp,   90.9553_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1458.4501_wp,  697.7650_wp,  570.8659_wp,  630.1412_wp,    0.0000_wp, &
      &  439.2845_wp,  239.8849_wp,  201.3799_wp,  214.4961_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1132.1074_wp,  574.7522_wp,  474.8225_wp,  515.5415_wp,    0.0000_wp, &
      &  902.6014_wp,  459.6812_wp,  380.4201_wp,  412.6142_wp,    0.0000_wp, &
      &  438.2482_wp,  246.1156_wp,  207.4440_wp,  219.3679_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  921.8990_wp,  474.6254_wp,  393.0487_wp,  425.1277_wp,    0.0000_wp, &
      &  424.6693_wp,  239.6563_wp,  202.2392_wp,  213.6129_wp,    0.0000_wp, &
      &  413.7841_wp,  236.3296_wp,  199.8502_wp,  210.4559_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(399 * max_ref * max_ref + 1 : 418 * max_ref * max_ref) = [ &
      &  878.0939_wp,  458.1136_wp,  380.4377_wp,  410.0129_wp,    0.0000_wp, &
      &  409.3555_wp,  231.3631_wp,  195.3516_wp,  206.2651_wp,    0.0000_wp, &
      &  418.3113_wp,  239.2323_wp,  202.3746_wp,  213.0501_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  797.5195_wp,  419.4037_wp,  348.8510_wp,  375.1745_wp,    0.0000_wp, &
      &  401.4329_wp,  226.7601_wp,  191.4715_wp,  202.2050_wp,    0.0000_wp, &
      &  383.2557_wp,  218.9943_wp,  185.3265_wp,  195.1629_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  626.6472_wp,  330.3808_wp,  275.2551_wp,  295.8599_wp,    0.0000_wp, &
      &  399.4530_wp,  224.7176_wp,  189.6203_wp,  200.4598_wp,    0.0000_wp, &
      &  280.2977_wp,  164.7847_wp,  140.4748_wp,  146.9885_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  678.4531_wp,  360.9692_wp,  300.9640_wp,  322.6845_wp,    0.0000_wp, &
      &  360.0172_wp,  206.6126_wp,  174.9529_wp,  184.0478_wp,    0.0000_wp, &
      &  235.4499_wp,  140.9080_wp,  120.6249_wp,  125.7205_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  530.3221_wp,  283.3011_wp,  236.6842_wp,  253.5365_wp,    0.0000_wp, &
      &  331.0614_wp,  191.1798_wp,  162.1293_wp,  170.3117_wp,    0.0000_wp, &
      &  231.6422_wp,  138.3424_wp,  118.3800_wp,  123.4417_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  553.0834_wp,  300.1055_wp,  251.1837_wp,  267.9482_wp,    0.0000_wp, &
      &  315.7713_wp,  182.3876_wp,  154.7198_wp,  162.5286_wp,    0.0000_wp, &
      &  279.7052_wp,  162.3831_wp,  138.0223_wp,  144.8424_wp,    0.0000_wp, &
      &  244.2053_wp,  144.4518_wp,  123.3171_wp,  128.8641_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  574.7436_wp,  309.2543_wp,  258.3888_wp,  276.2327_wp,    0.0000_wp, &
      &  309.2543_wp,  178.3311_wp,  151.2409_wp,  158.9405_wp,    0.0000_wp, &
      &  258.3888_wp,  151.2409_wp,  128.7820_wp,  134.8933_wp,    0.0000_wp, &
      &  276.2327_wp,  158.9405_wp,  134.8933_wp,  141.8763_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   28.5940_wp,   22.1550_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   47.6499_wp,   35.9871_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   18.0216_wp,   14.5147_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  569.4526_wp,  367.4308_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  168.8075_wp,  119.4397_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  292.7955_wp,  205.9313_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  212.9797_wp,  151.8426_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  130.8012_wp,   97.5340_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  185.1741_wp,  136.3705_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  153.8824_wp,  114.3466_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  127.6686_wp,   95.7007_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   93.7660_wp,   72.1216_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   87.8052_wp,   67.7835_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  119.6350_wp,   91.1025_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  112.3914_wp,   85.4731_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   89.9515_wp,   69.4058_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   84.2028_wp,   64.9641_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   67.0823_wp,   52.7984_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   81.0855_wp,   63.2709_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   75.5987_wp,   58.9903_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   70.7639_wp,   55.3789_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   62.1299_wp,   48.7630_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   60.1335_wp,   47.7239_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   53.6420_wp,   42.7458_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   47.0585_wp,   37.7083_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   44.8098_wp,   36.0749_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   35.8100_wp,   29.2808_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   34.0544_wp,   27.7282_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  676.4773_wp,  438.7198_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  249.8197_wp,  175.2531_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  476.2426_wp,  330.0018_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  388.4209_wp,  271.2755_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  238.5370_wp,  174.3619_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(418 * max_ref * max_ref + 1 : 437 * max_ref * max_ref) = [ &
      &  424.7723_wp,  301.4650_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  381.9923_wp,  272.6384_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  289.6824_wp,  210.9019_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  217.8958_wp,  162.8021_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  320.1913_wp,  234.7548_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  294.1406_wp,  216.6154_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  271.8911_wp,  201.1573_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  266.3120_wp,  196.2996_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  212.5281_wp,  160.1946_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  240.7974_wp,  181.3267_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  233.4898_wp,  175.9628_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  230.9091_wp,  173.7006_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  212.1340_wp,  160.7225_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  195.4075_wp,  149.6421_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  193.3865_wp,  148.0244_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  189.4122_wp,  145.0250_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  156.3161_wp,  121.6329_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  154.7756_wp,  120.3927_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  125.6776_wp,   99.1398_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1129.9518_wp,  722.0002_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  337.1117_wp,  240.1791_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  868.3674_wp,  585.3704_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  694.5058_wp,  467.9156_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  335.3592_wp,  244.6716_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  706.0720_wp,  481.6472_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  325.0850_wp,  238.0128_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  316.7102_wp,  234.0374_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  672.4128_wp,  463.3924_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  313.6026_wp,  229.7226_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  320.2198_wp,  236.8532_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  610.5700_wp,  423.4193_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  307.5763_wp,  225.2020_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  293.6816_wp,  216.9203_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  480.6938_wp,  333.5014_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  306.1181_wp,  223.4009_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  215.5023_wp,  162.3545_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  519.3078_wp,  363.4171_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  275.7206_wp,  204.4480_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  181.3482_wp,  138.3572_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  406.7268_wp,  285.1018_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  253.7038_wp,  188.9445_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  178.3924_wp,  135.9019_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  423.0877_wp,  300.7356_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  242.1030_wp,  180.2719_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  214.8093_wp,  160.4015_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  187.8440_wp,  142.1634_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  439.6365_wp,  310.5133_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  237.1200_wp,  176.3358_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  198.5202_wp,  149.1479_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  212.1434_wp,  157.3396_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  337.1808_wp,  238.1207_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  238.1207_wp,  174.9969_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   29.8689_wp,   21.2139_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   49.3469_wp,   34.0312_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   18.9695_wp,   14.1185_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(437 * max_ref * max_ref + 1 : 456 * max_ref * max_ref) = [ &
      &  540.9160_wp,  317.8826_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  169.4949_wp,  109.0127_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  293.3106_wp,  187.3012_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  214.4256_wp,  139.2420_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  134.7933_wp,   91.6227_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  189.6293_wp,  127.2488_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  158.0848_wp,  107.2026_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  131.5252_wp,   90.1351_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   97.6845_wp,   68.8077_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   91.5728_wp,   64.7798_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  124.1032_wp,   86.4709_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  116.3987_wp,   81.0686_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   93.6870_wp,   66.2838_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   87.6124_wp,   62.0228_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   70.2820_wp,   50.8856_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   84.7311_wp,   60.7331_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   78.9183_wp,   56.6140_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   73.9210_wp,   53.2070_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   64.8633_wp,   46.8911_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   63.0774_wp,   46.1355_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   56.2884_wp,   41.3862_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   49.3897_wp,   36.5764_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   47.1122_wp,   35.0629_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   37.7559_wp,   28.6291_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   35.8427_wp,   27.0524_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  644.3408_wp,  380.6568_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  249.3925_wp,  159.1228_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  472.9143_wp,  297.5440_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  386.8884_wp,  245.8054_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  243.5663_wp,  162.0032_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  427.1610_wp,  275.6299_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  385.1449_wp,  250.0730_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  294.9017_wp,  195.5973_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  224.7197_wp,  153.0776_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  327.2810_wp,  218.5282_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  301.1709_wp,  202.1306_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  278.9696_wp,  188.1613_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  272.5606_wp,  183.2556_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  219.8870_wp,  151.3141_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  248.9743_wp,  171.1889_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  241.4380_wp,  166.1887_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  238.5063_wp,  163.8884_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  219.8016_wp,  152.2000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  203.2427_wp,  142.4591_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  201.0704_wp,  140.8804_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  196.9331_wp,  138.0396_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  163.3848_wp,  116.6637_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  161.7337_wp,  115.4501_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  131.8119_wp,   95.6586_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1062.5928_wp,  621.4841_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  337.8986_wp,  219.9852_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  846.8527_wp,  519.5182_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  675.0299_wp,  415.5638_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  341.1923_wp,  227.0118_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  693.6787_wp,  430.3675_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  331.2310_wp,  221.2467_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  324.2891_wp,  218.6603_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(456 * max_ref * max_ref + 1 : 475 * max_ref * max_ref) = [ &
      &  664.1009_wp,  416.5328_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  319.3834_wp,  213.6191_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  328.0001_wp,  221.3943_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  605.0623_wp,  381.9629_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  313.1286_wp,  209.3428_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  300.3130_wp,  202.5980_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  475.5594_wp,  300.9323_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  311.0603_wp,  207.2926_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  221.9656_wp,  153.1295_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  517.0910_wp,  329.5196_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  282.6667_wp,  191.3125_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  187.7214_wp,  131.3100_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  404.5558_wp,  258.7290_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  260.5341_wp,  177.2034_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  184.5403_wp,  128.8760_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  424.8790_wp,  275.0381_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  248.4855_wp,  169.0461_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  220.4525_wp,  150.5946_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  193.8570_wp,  134.3706_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  440.0900_wp,  282.9656_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  243.1960_wp,  165.2361_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  204.3119_wp,  140.4552_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  216.9447_wp,  147.1597_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  336.6493_wp,  216.9570_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  242.6893_wp,  162.9236_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  340.5213_wp,  223.6622_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  223.6622_wp,  153.4528_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   35.1697_wp,   33.2515_wp,   26.9562_wp,   22.5390_wp,    0.0000_wp, &
      &   58.4207_wp,   54.9919_wp,   43.8881_wp,   36.1447_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   22.0458_wp,   20.9874_wp,   17.4392_wp,   14.9296_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  656.7441_wp,  602.8104_wp,  441.4888_wp,  332.3464_wp,    0.0000_wp, &
      &  202.6900_wp,  188.7622_wp,  145.1043_wp,  115.1284_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  351.1141_wp,  326.6672_wp,  250.1889_wp,  197.7454_wp,    0.0000_wp, &
      &  256.2049_wp,  238.9068_wp,  184.5744_wp,  147.1923_wp,    0.0000_wp, &
      &  159.9273_wp,  150.2148_wp,  118.9619_wp,   97.2346_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  225.3933_wp,  211.2756_wp,  166.1525_wp,  134.8856_wp,    0.0000_wp, &
      &  187.5797_wp,  176.0976_wp,  139.2546_wp,  113.6748_wp,    0.0000_wp, &
      &  155.8080_wp,  146.4890_wp,  116.4830_wp,   95.6039_wp,    0.0000_wp, &
      &  115.1706_wp,  108.7508_wp,   87.7688_wp,   73.0754_wp,    0.0000_wp, &
      &  107.8705_wp,  101.9223_wp,   82.4447_wp,   68.7935_wp,    0.0000_wp, &
      &  146.5544_wp,  138.1497_wp,  110.8384_wp,   91.7661_wp,    0.0000_wp, &
      &  137.4723_wp,  129.5670_wp,  103.9180_wp,   86.0172_wp,    0.0000_wp, &
      &  110.2770_wp,  104.1881_wp,   84.2691_wp,   70.3184_wp,    0.0000_wp, &
      &  103.0738_wp,   97.3838_wp,   78.7812_wp,   65.7584_wp,    0.0000_wp, &
      &   82.3248_wp,   78.0505_wp,   63.9117_wp,   53.9617_wp,    0.0000_wp, &
      &   99.4407_wp,   94.1329_wp,   76.6641_wp,   64.3977_wp,    0.0000_wp, &
      &   92.6025_wp,   87.6645_wp,   71.4234_wp,   60.0229_wp,    0.0000_wp, &
      &   86.6194_wp,   82.0402_wp,   66.9550_wp,   56.3612_wp,    0.0000_wp, &
      &   75.8800_wp,   71.9079_wp,   58.8126_wp,   49.6159_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   73.6253_wp,   69.9039_wp,   57.5425_wp,   48.8310_wp,    0.0000_wp, &
      &   65.6214_wp,   62.3522_wp,   51.4705_wp,   43.7962_wp,    0.0000_wp, &
      &   57.4016_wp,   54.5964_wp,   45.2336_wp,   38.6263_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   54.6782_wp,   52.0476_wp,   43.2385_wp,   37.0129_wp,    0.0000_wp, &
      &   43.5685_wp,   41.5909_wp,   34.9008_wp,   30.1557_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   41.3681_wp,   39.4587_wp,   33.0228_wp,   28.4654_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(475 * max_ref * max_ref + 1 : 494 * max_ref * max_ref) = [ &
      &  781.3524_wp,  717.7531_wp,  527.0671_wp,  398.0180_wp,    0.0000_wp, &
      &  298.4007_wp,  277.5143_wp,  212.3618_wp,  167.7278_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  567.1638_wp,  526.4236_wp,  399.9263_wp,  313.4658_wp,    0.0000_wp, &
      &  463.5691_wp,  430.8297_wp,  328.9697_wp,  259.2061_wp,    0.0000_wp, &
      &  289.7425_wp,  271.2420_wp,  212.3044_wp,  171.5385_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  510.5854_wp,  475.7318_wp,  366.3371_wp,  291.1704_wp,    0.0000_wp, &
      &  459.9544_wp,  428.9521_wp,  331.4055_wp,  264.2993_wp,    0.0000_wp, &
      &  351.1192_wp,  328.5100_wp,  256.7121_wp,  207.0871_wp,    0.0000_wp, &
      &  266.4534_wp,  250.3494_wp,  198.4813_wp,  162.4065_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  389.2656_wp,  364.6110_wp,  285.9761_wp,  231.5432_wp,    0.0000_wp, &
      &  357.9411_wp,  335.5206_wp,  263.8740_wp,  214.2270_wp,    0.0000_wp, &
      &  331.2849_wp,  310.7704_wp,  245.0585_wp,  199.4760_wp,    0.0000_wp, &
      &  323.8179_wp,  303.5907_wp,  238.9624_wp,  194.1778_wp,    0.0000_wp, &
      &  260.3091_wp,  244.9446_wp,  195.2461_wp,  160.6091_wp,    0.0000_wp, &
      &  294.7925_wp,  277.3523_wp,  220.9763_wp,  181.6981_wp,    0.0000_wp, &
      &  285.8072_wp,  268.9370_wp,  214.3903_wp,  176.3816_wp,    0.0000_wp, &
      &  282.3871_wp,  265.6407_wp,  211.5571_wp,  173.8931_wp,    0.0000_wp, &
      &  259.9184_wp,  244.7964_wp,  195.7667_wp,  161.5591_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  239.8183_wp,  226.2796_wp,  182.1380_wp,  151.2611_wp,    0.0000_wp, &
      &  237.2563_wp,  223.8453_wp,  180.1361_wp,  149.5674_wp,    0.0000_wp, &
      &  232.3409_wp,  219.2194_wp,  176.4499_wp,  146.5382_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  192.0578_wp,  181.7178_wp,  147.7169_wp,  123.8442_wp,    0.0000_wp, &
      &  190.1139_wp,  179.8683_wp,  146.1873_wp,  122.5429_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  154.3423_wp,  146.3855_wp,  120.0234_wp,  101.4572_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1292.8121_wp, 1184.8327_wp,  865.1066_wp,  648.6879_wp,    0.0000_wp, &
      &  403.0745_wp,  375.8495_wp,  290.5739_wp,  231.9989_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1019.8835_wp,  942.5828_wp,  706.2041_wp,  545.4548_wp,    0.0000_wp, &
      &  813.4943_wp,  751.8144_wp,  564.1147_wp,  436.2844_wp,    0.0000_wp, &
      &  405.6488_wp,  379.6650_wp,  297.1084_wp,  240.0800_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  833.7424_wp,  771.9763_wp,  581.9150_wp,  452.4148_wp,    0.0000_wp, &
      &  393.5300_wp,  368.5417_wp,  288.9924_wp,  234.0165_wp,    0.0000_wp, &
      &  384.8172_wp,  360.9196_wp,  284.4870_wp,  231.5184_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  796.8617_wp,  739.0281_wp,  560.2945_wp,  438.2812_wp,    0.0000_wp, &
      &  379.4109_wp,  355.3498_wp,  278.8293_wp,  225.9136_wp,    0.0000_wp, &
      &  389.1325_wp,  365.0239_wp,  287.8797_wp,  234.4093_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  725.2514_wp,  673.2823_wp,  512.2075_wp,  402.1202_wp,    0.0000_wp, &
      &  371.9427_wp,  348.3293_wp,  273.2440_wp,  221.3382_wp,    0.0000_wp, &
      &  356.2469_wp,  334.1078_wp,  263.3800_wp,  214.3808_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  569.7645_wp,  529.0006_wp,  402.8309_wp,  316.6134_wp,    0.0000_wp, &
      &  369.6302_wp,  345.9796_wp,  270.9168_wp,  219.0704_wp,    0.0000_wp, &
      &  262.1530_wp,  246.6796_wp,  196.7704_wp,  162.0547_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  618.8387_wp,  575.3229_wp,  439.8829_wp,  347.1545_wp,    0.0000_wp, &
      &  335.0615_wp,  314.4244_wp,  248.3153_wp,  202.4740_wp,    0.0000_wp, &
      &  221.1217_wp,  208.5110_wp,  167.5629_wp,  138.9928_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  483.7815_wp,  449.9040_wp,  344.5352_wp,  272.3998_wp,    0.0000_wp, &
      &  308.5529_wp,  289.7588_wp,  229.4318_wp,  187.5580_wp,    0.0000_wp, &
      &  217.4040_wp,  204.9523_wp,  164.5553_wp,  136.3818_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  507.1316_wp,  472.6401_wp,  364.4340_wp,  290.1268_wp,    0.0000_wp, &
      &  294.2408_wp,  276.3130_wp,  218.7964_wp,  178.8774_wp,    0.0000_wp, &
      &  260.7548_wp,  244.9914_wp,  194.3868_wp,  159.2584_wp,    0.0000_wp, &
      &  228.6785_wp,  215.3374_wp,  172.2004_wp,  142.1634_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  525.8162_wp,  489.5499_wp,  376.1112_wp,  298.3141_wp,    0.0000_wp, &
      &  288.0073_wp,  270.4023_wp,  213.9621_wp,  174.8058_wp,    0.0000_wp, &
      &  241.3636_wp,  227.0001_wp,  180.7289_wp,  148.5642_wp,    0.0000_wp, &
      &  256.7960_wp,  241.0016_wp,  190.4799_wp,  155.4813_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  401.9613_wp,  374.2537_wp,  287.7668_wp,  228.4811_wp,    0.0000_wp, &
      &  287.8520_wp,  269.7344_wp,  212.0275_wp,  172.1215_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(494 * max_ref * max_ref + 1 : 513 * max_ref * max_ref) = [ &
      &  405.4565_wp,  378.7329_wp,  294.3129_wp,  236.1661_wp,    0.0000_wp, &
      &  264.4724_wp,  248.7003_wp,  197.8459_wp,  162.4735_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  483.7516_wp,  451.4003_wp,  349.5389_wp,  279.4346_wp,    0.0000_wp, &
      &  451.4003_wp,  421.6497_wp,  327.6772_wp,  262.9134_wp,    0.0000_wp, &
      &  349.5389_wp,  327.6772_wp,  257.8727_wp,  209.5130_wp,    0.0000_wp, &
      &  279.4346_wp,  262.9134_wp,  209.5130_wp,  172.3153_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   31.8170_wp,   29.9527_wp,   28.6359_wp,   25.9255_wp,   23.1784_wp, &
      &   51.9905_wp,   48.7247_wp,   46.4409_wp,   41.7323_wp,   36.9664_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   20.4318_wp,   19.3761_wp,   18.6186_wp,   17.0614_wp,   15.4801_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  527.9268_wp,  484.0165_wp,  454.1125_wp,  392.4109_wp,  330.5666_wp, &
      &  172.9425_wp,  160.4502_wp,  151.8759_wp,  134.1408_wp,  116.2577_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  298.4220_wp,  276.5717_wp,  261.6034_wp,  230.6287_wp,  199.4034_wp, &
      &  219.7375_wp,  204.1690_wp,  193.4457_wp,  171.2891_wp,  148.9411_wp, &
      &  141.1489_wp,  131.9934_wp,  125.6209_wp,  112.4708_wp,   99.1700_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  197.3448_wp,  184.2055_wp,  175.0938_wp,  156.2783_wp,  137.2617_wp, &
      &  165.1930_wp,  154.4409_wp,  146.9606_wp,  131.5234_wp,  115.9145_wp, &
      &  137.9956_wp,  129.2227_wp,  123.0969_wp,  110.4653_wp,   97.6880_wp, &
      &  103.6947_wp,   97.4953_wp,   93.1301_wp,   84.1398_wp,   75.0316_wp, &
      &   97.3490_wp,   91.5895_wp,   87.5282_wp,   79.1656_wp,   70.6917_wp, &
      &  131.0954_wp,  123.0574_wp,  117.4198_wp,  105.8004_wp,   94.0357_wp, &
      &  122.8805_wp,  115.3488_wp,  110.0651_wp,   99.1763_wp,   88.1532_wp, &
      &   99.4698_wp,   93.5893_wp,   89.4448_wp,   80.9084_wp,   72.2595_wp, &
      &   92.9621_wp,   87.4781_wp,   83.6131_wp,   75.6514_wp,   67.5854_wp, &
      &   75.2076_wp,   71.0143_wp,   68.0343_wp,   61.9038_wp,   55.6857_wp, &
      &   90.3328_wp,   85.1621_wp,   81.5019_wp,   73.9675_wp,   66.3293_wp, &
      &   84.1276_wp,   79.3278_wp,   75.9287_wp,   68.9323_wp,   61.8399_wp, &
      &   78.8235_wp,   74.3627_wp,   71.2017_wp,   64.6946_wp,   58.0974_wp, &
      &   69.1710_wp,   65.3041_wp,   62.5609_wp,   56.9134_wp,   51.1874_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   67.5932_wp,   63.9245_wp,   61.3099_wp,   55.9314_wp,   50.4741_wp, &
      &   60.4041_wp,   57.1752_wp,   54.8693_wp,   50.1269_wp,   45.3141_wp, &
      &   53.0110_wp,   50.2343_wp,   48.2476_wp,   44.1611_wp,   40.0132_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   50.6412_wp,   48.0233_wp,   46.1468_wp,   42.2879_wp,   38.3697_wp, &
      &   40.7500_wp,   38.7583_wp,   37.3202_wp,   34.3642_wp,   31.3603_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   38.5746_wp,   36.6615_wp,   35.2842_wp,   32.4515_wp,   29.5740_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  630.1041_wp,  578.0882_wp,  542.6657_wp,  469.5496_wp,  396.2431_wp, &
      &  253.1433_wp,  234.6048_wp,  221.9031_wp,  195.6231_wp,  169.1405_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  477.3744_wp,  441.5239_wp,  417.0307_wp,  366.3319_wp,  315.2720_wp, &
      &  392.2533_wp,  363.3320_wp,  343.5106_wp,  302.5213_wp,  261.2272_wp, &
      &  252.3964_wp,  235.2670_wp,  223.4249_wp,  198.9542_wp,  174.2312_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  436.5060_wp,  405.1611_wp,  383.6305_wp,  339.1034_wp,  294.1970_wp, &
      &  394.6650_wp,  366.6561_wp,  347.3879_wp,  307.5504_wp,  267.3616_wp, &
      &  305.1643_wp,  284.3739_wp,  269.9943_wp,  240.2935_wp,  210.2972_wp, &
      &  235.4382_wp,  220.2317_wp,  209.6430_wp,  187.7933_wp,  165.6908_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  339.8328_wp,  316.9700_wp,  301.1380_wp,  268.4370_wp,  235.3944_wp, &
      &  313.3911_wp,  292.5335_wp,  278.0671_wp,  248.1965_wp,  218.0074_wp, &
      &  290.9092_wp,  271.7438_wp,  258.4338_wp,  230.9563_wp,  203.1783_wp, &
      &  283.6909_wp,  264.8945_wp,  251.8482_wp,  224.9146_wp,  197.6946_wp, &
      &  231.3302_wp,  216.7246_wp,  206.5206_wp,  185.4774_wp,  164.1811_wp, &
      &  261.8235_wp,  245.2683_wp,  233.7044_wp,  209.8561_wp,  185.7226_wp, &
      &  253.9744_wp,  237.9580_wp,  226.7660_wp,  203.6865_wp,  180.3305_wp, &
      &  250.6401_wp,  234.7785_wp,  223.7008_wp,  200.8543_wp,  177.7373_wp, &
      &  231.7564_wp,  217.3350_wp,  207.2390_wp,  186.4258_wp,  165.3571_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  215.3101_wp,  202.2863_wp,  193.1315_wp,  174.2716_wp,  155.1691_wp, &
      &  212.9437_wp,  200.0526_wp,  190.9927_wp,  172.3272_wp,  153.4225_wp, &
      &  208.5685_wp,  195.9558_wp,  187.0906_wp,  168.8263_wp,  150.3278_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  174.1969_wp,  164.1266_wp,  157.0031_wp,  142.3413_wp,  127.4785_wp, &
      &  172.3933_wp,  162.4206_wp,  155.3672_wp,  140.8492_wp,  126.1326_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(513 * max_ref * max_ref + 1 : 532 * max_ref * max_ref) = [ &
      &  141.2063_wp,  133.3794_wp,  127.8113_wp,  116.3589_wp,  104.7412_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1032.6651_wp,  946.5756_wp,  887.7075_wp,  766.5714_wp,  645.2973_wp, &
      &  345.4434_wp,  321.1574_wp,  304.4222_wp,  269.8403_wp,  234.9742_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  843.3644_wp,  777.5361_wp,  732.6433_wp,  639.8000_wp,  546.4784_wp, &
      &  672.6917_wp,  620.6735_wp,  585.0515_wp,  511.5374_wp,  437.6779_wp, &
      &  352.9336_wp,  329.0554_wp,  312.5424_wp,  278.4205_wp,  243.9590_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  694.7024_wp,  641.4064_wp,  605.0343_wp,  529.7799_wp,  454.0771_wp, &
      &  343.1679_wp,  320.1342_wp,  304.1902_wp,  271.2475_wp,  237.9706_wp, &
      &  337.5761_wp,  315.3367_wp,  299.9014_wp,  268.0269_wp,  235.8103_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  668.3786_wp,  618.0401_wp,  583.6237_wp,  512.4274_wp,  440.7662_wp, &
      &  330.9710_wp,  308.8304_wp,  293.4899_wp,  261.8074_wp,  229.8053_wp, &
      &  341.5627_wp,  319.1099_wp,  303.5228_wp,  271.3354_wp,  238.8007_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  610.7639_wp,  565.2639_wp,  534.1256_wp,  469.7122_wp,  404.8543_wp, &
      &  324.3435_wp,  302.6272_wp,  287.5851_wp,  256.5149_wp,  225.1326_wp, &
      &  312.4130_wp,  291.8713_wp,  277.6090_wp,  248.1606_wp,  218.4003_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  479.9639_wp,  444.4196_wp,  420.0642_wp,  369.7049_wp,  319.0075_wp, &
      &  321.6404_wp,  299.9706_wp,  284.9729_wp,  253.9905_wp,  222.7038_wp, &
      &  232.7992_wp,  218.2207_wp,  208.0381_wp,  187.0285_wp,  165.7746_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  524.1848_wp,  485.7648_wp,  459.4314_wp,  404.9622_wp,  350.0882_wp, &
      &  294.5184_wp,  275.2607_wp,  261.8866_wp,  234.2680_wp,  206.3483_wp, &
      &  197.9394_wp,  185.9246_wp,  177.4973_wp,  160.1207_wp,  142.5295_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  410.2106_wp,  380.3853_wp,  359.9172_wp,  317.5943_wp,  274.9614_wp, &
      &  271.9711_wp,  254.3735_wp,  242.1349_wp,  216.8675_wp,  191.3189_wp, &
      &  194.4157_wp,  182.5691_wp,  174.2654_wp,  157.1410_wp,  139.8068_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  433.8508_wp,  402.9146_wp,  381.6588_wp,  337.6931_wp,  293.3577_wp, &
      &  259.3224_wp,  242.5572_wp,  230.8956_wp,  206.8210_wp,  182.4798_wp, &
      &  230.2133_wp,  215.4752_wp,  205.2122_wp,  184.0272_wp,  162.6068_wp, &
      &  203.6278_wp,  191.0014_wp,  182.1731_wp,  163.9594_wp,  145.5291_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  447.9868_wp,  415.6439_wp,  393.4530_wp,  347.5445_wp,  301.2669_wp, &
      &  253.6134_wp,  237.1737_wp,  225.7435_wp,  202.1442_wp,  178.2858_wp, &
      &  213.9134_wp,  200.4003_wp,  190.9759_wp,  171.5246_wp,  151.8497_wp, &
      &  225.7080_wp,  211.0440_wp,  200.8574_wp,  179.8178_wp,  158.5536_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  342.4261_wp,  317.8624_wp,  300.9874_wp,  266.0920_wp,  230.9258_wp, &
      &  251.5386_wp,  234.8327_wp,  223.2574_wp,  199.3430_wp,  175.1844_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  349.9885_wp,  325.6928_wp,  308.9517_wp,  274.3339_wp,  239.3959_wp, &
      &  234.3407_wp,  219.4508_wp,  209.0707_wp,  187.6481_wp,  165.9764_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  415.9530_wp,  386.7129_wp,  366.5742_wp,  324.9456_wp,  282.9469_wp, &
      &  389.7370_wp,  362.6832_wp,  344.0232_wp,  305.4579_wp,  266.5350_wp, &
      &  306.0934_wp,  285.8049_wp,  271.7272_wp,  242.6615_wp,  213.2887_wp, &
      &  248.1866_wp,  232.5085_wp,  221.5612_wp,  198.9801_wp,  176.1294_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  363.5474_wp,  339.2404_wp,  322.3974_wp,  287.6103_wp,  252.4591_wp, &
      &  339.2404_wp,  316.8521_wp,  301.3102_wp,  269.2215_wp,  236.7867_wp, &
      &  322.3974_wp,  301.3102_wp,  286.6545_wp,  256.4009_wp,  225.8139_wp, &
      &  287.6103_wp,  269.2215_wp,  256.4009_wp,  229.9498_wp,  203.1922_wp, &
      &  252.4591_wp,  236.7867_wp,  225.8139_wp,  203.1922_wp,  180.2912_wp, &
      &   27.7884_wp,   27.3574_wp,   26.7042_wp,   25.8144_wp,    0.0000_wp, &
      &   44.6512_wp,   43.9067_wp,   42.7789_wp,   41.2434_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   18.3284_wp,   18.0816_wp,   17.7051_wp,   17.1900_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  415.1532_wp,  406.3427_wp,  392.4365_wp,  373.0901_wp,    0.0000_wp, &
      &  142.8471_wp,  140.1353_wp,  135.9832_wp,  130.3034_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  245.4831_wp,  240.7559_wp,  233.5151_wp,  223.6099_wp,    0.0000_wp, &
      &  182.5338_wp,  179.1390_wp,  173.9415_wp,  166.8295_wp,    0.0000_wp, &
      &  120.2224_wp,  118.1520_wp,  115.0137_wp,  110.7405_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(532 * max_ref * max_ref + 1 : 551 * max_ref * max_ref) = [ &
      &  166.9060_wp,  163.9655_wp,  159.4969_wp,  153.4055_wp,    0.0000_wp, &
      &  140.5625_wp,  138.1446_wp,  134.4704_wp,  129.4615_wp,    0.0000_wp, &
      &  118.1354_wp,  116.1535_wp,  113.1412_wp,  109.0331_wp,    0.0000_wp, &
      &   90.1373_wp,   88.7108_wp,   86.5486_wp,   83.6036_wp,    0.0000_wp, &
      &   84.8298_wp,   83.5023_wp,   81.4896_wp,   78.7478_wp,    0.0000_wp, &
      &  113.2616_wp,  111.4257_wp,  108.6407_wp,  104.8462_wp,    0.0000_wp, &
      &  106.1686_wp,  104.4532_wp,  101.8469_wp,   98.2930_wp,    0.0000_wp, &
      &   86.6895_wp,   85.3379_wp,   83.2865_wp,   80.4902_wp,    0.0000_wp, &
      &   81.0563_wp,   79.7986_wp,   77.8874_wp,   75.2808_wp,    0.0000_wp, &
      &   66.4147_wp,   65.4414_wp,   63.9621_wp,   61.9438_wp,    0.0000_wp, &
      &   79.3086_wp,   78.1138_wp,   76.2992_wp,   73.8246_wp,    0.0000_wp, &
      &   73.9142_wp,   72.8073_wp,   71.1238_wp,   68.8264_wp,    0.0000_wp, &
      &   69.3770_wp,   68.3477_wp,   66.7818_wp,   64.6446_wp,    0.0000_wp, &
      &   61.0425_wp,   60.1519_wp,   58.7945_wp,   56.9398_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   60.0289_wp,   59.1764_wp,   57.8784_wp,   56.1057_wp,    0.0000_wp, &
      &   53.8153_wp,   53.0645_wp,   51.9199_wp,   50.3555_wp,    0.0000_wp, &
      &   47.4202_wp,   46.7753_wp,   45.7899_wp,   44.4417_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   45.4180_wp,   44.8079_wp,   43.8762_wp,   42.6013_wp,    0.0000_wp, &
      &   36.9389_wp,   36.4728_wp,   35.7582_wp,   34.7785_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   34.8686_wp,   34.4226_wp,   33.7391_wp,   32.8021_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  496.9338_wp,  486.4638_wp,  469.9596_wp,  447.0179_wp,    0.0000_wp, &
      &  208.1802_wp,  204.1937_wp,  198.0692_wp,  189.6770_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  389.4927_wp,  381.8396_wp,  370.0638_wp,  353.9178_wp,    0.0000_wp, &
      &  321.8732_wp,  315.6688_wp,  306.1275_wp,  293.0440_wp,    0.0000_wp, &
      &  212.3392_wp,  208.5236_wp,  202.7244_wp,  194.8198_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  361.1913_wp,  354.3706_wp,  343.9343_wp,  329.6624_wp,    0.0000_wp, &
      &  327.7229_wp,  321.6043_wp,  312.2501_wp,  299.4628_wp,    0.0000_wp, &
      &  256.4231_wp,  251.8127_wp,  244.7896_wp,  235.2041_wp,    0.0000_wp, &
      &  200.7514_wp,  197.3088_wp,  192.0916_wp,  184.9886_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  286.5909_wp,  281.4893_wp,  273.7334_wp,  263.1595_wp,    0.0000_wp, &
      &  265.0711_wp,  260.4044_wp,  253.3112_wp,  243.6410_wp,    0.0000_wp, &
      &  246.7379_wp,  242.4359_wp,  235.9012_wp,  226.9949_wp,    0.0000_wp, &
      &  240.2303_wp,  236.0287_wp,  229.6362_wp,  220.9165_wp,    0.0000_wp, &
      &  198.4079_wp,  195.0844_wp,  190.0483_wp,  183.1915_wp,    0.0000_wp, &
      &  224.4774_wp,  220.7146_wp,  215.0103_wp,  207.2419_wp,    0.0000_wp, &
      &  217.8920_wp,  214.2514_wp,  208.7309_wp,  201.2119_wp,    0.0000_wp, &
      &  214.8363_wp,  211.2380_wp,  205.7784_wp,  198.3402_wp,    0.0000_wp, &
      &  199.5009_wp,  196.2122_wp,  191.2265_wp,  184.4364_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  186.6348_wp,  183.6462_wp,  179.1160_wp,  172.9456_wp,    0.0000_wp, &
      &  184.5467_wp,  181.5905_wp,  177.1085_wp,  171.0030_wp,    0.0000_wp, &
      &  180.8007_wp,  177.9088_wp,  173.5234_wp,  167.5491_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  152.6035_wp,  150.2742_wp,  146.7405_wp,  141.9245_wp,    0.0000_wp, &
      &  150.9999_wp,  148.6943_wp,  145.1961_wp,  140.4282_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  124.8561_wp,  123.0358_wp,  120.2698_wp,  116.4964_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  810.6554_wp,  793.6306_wp,  766.5311_wp,  728.6195_wp,    0.0000_wp, &
      &  287.5535_wp,  282.3070_wp,  274.2325_wp,  263.1547_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  679.0044_wp,  665.3221_wp,  644.0351_wp,  614.6705_wp,    0.0000_wp, &
      &  543.0123_wp,  532.2461_wp,  515.4323_wp,  492.1704_wp,    0.0000_wp, &
      &  297.1426_wp,  291.8601_wp,  283.8013_wp,  272.7959_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  562.6947_wp,  551.4999_wp,  534.1575_wp,  510.2912_wp,    0.0000_wp, &
      &  289.5582_wp,  284.4516_wp,  276.6634_wp,  266.0294_wp,    0.0000_wp, &
      &  286.3045_wp,  281.3331_wp,  273.7684_wp,  263.4496_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  544.6797_wp,  534.0266_wp,  517.5611_wp,  494.9276_wp,    0.0000_wp, &
      &  279.4990_wp,  274.5929_wp,  267.1055_wp,  256.8763_wp,    0.0000_wp, &
      &  289.8560_wp,  284.8345_wp,  277.1939_wp,  266.7717_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  499.4970_wp,  489.8210_wp,  474.8898_wp,  454.3822_wp,    0.0000_wp, &
      &  273.8352_wp,  269.0270_wp,  261.6872_wp,  251.6587_wp,    0.0000_wp, &
      &  265.0778_wp,  260.4970_wp,  253.5169_wp,  243.9879_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(551 * max_ref * max_ref + 1 : 570 * max_ref * max_ref) = [ &
      &  393.1922_wp,  385.6577_wp,  374.0038_wp,  357.9757_wp,    0.0000_wp, &
      &  271.0770_wp,  266.2939_wp,  258.9856_wp,  248.9955_wp,    0.0000_wp, &
      &  200.0486_wp,  196.7631_wp,  191.7598_wp,  184.9307_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  430.9146_wp,  422.6874_wp,  410.0199_wp,  392.6408_wp,    0.0000_wp, &
      &  250.2831_wp,  245.9713_wp,  239.4120_wp,  230.4659_wp,    0.0000_wp, &
      &  171.4077_wp,  168.6780_wp,  164.5248_wp,  158.8578_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  338.0096_wp,  331.6389_wp,  321.8092_wp,  308.3075_wp,    0.0000_wp, &
      &  231.7618_wp,  227.8113_wp,  221.8034_wp,  213.6098_wp,    0.0000_wp, &
      &  168.1967_wp,  165.5084_wp,  161.4178_wp,  155.8359_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  359.7150_wp,  353.0074_wp,  342.7215_wp,  328.6403_wp,    0.0000_wp, &
      &  221.0233_wp,  217.2635_wp,  211.5422_wp,  203.7369_wp,    0.0000_wp, &
      &  196.7046_wp,  193.4021_wp,  188.3702_wp,  181.5005_wp,    0.0000_wp, &
      &  175.4121_wp,  172.5578_wp,  168.2143_wp,  162.2878_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  370.0359_wp,  363.0566_wp,  352.3403_wp,  337.6607_wp,    0.0000_wp, &
      &  216.0043_wp,  212.3221_wp,  206.7172_wp,  199.0695_wp,    0.0000_wp, &
      &  183.4110_wp,  180.3695_wp,  175.7399_wp,  169.4227_wp,    0.0000_wp, &
      &  192.1103_wp,  188.8450_wp,  183.8623_wp,  177.0557_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  283.3323_wp,  278.0578_wp,  269.9330_wp,  258.7830_wp,    0.0000_wp, &
      &  212.8346_wp,  209.1322_wp,  203.4807_wp,  195.7595_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  292.5030_wp,  287.1778_wp,  279.0375_wp,  267.9108_wp,    0.0000_wp, &
      &  200.6408_wp,  197.2771_wp,  192.1686_wp,  185.2060_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  346.3508_wp,  339.9625_wp,  330.1881_wp,  316.8201_wp,    0.0000_wp, &
      &  325.7268_wp,  319.7871_wp,  310.7110_wp,  298.3060_wp,    0.0000_wp, &
      &  259.1650_wp,  254.6359_wp,  247.7421_wp,  238.3364_wp,    0.0000_wp, &
      &  212.8299_wp,  209.2695_wp,  203.8708_wp,  196.5179_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  307.1010_wp,  301.6800_wp,  293.4322_wp,  282.1828_wp,    0.0000_wp, &
      &  287.5847_wp,  282.5717_wp,  274.9497_wp,  264.5566_wp,    0.0000_wp, &
      &  273.9655_wp,  269.2308_wp,  262.0354_wp,  252.2261_wp,    0.0000_wp, &
      &  245.8719_wp,  241.7135_wp,  235.4019_wp,  226.8019_wp,    0.0000_wp, &
      &  217.4476_wp,  213.8707_wp,  208.4500_wp,  201.0683_wp,    0.0000_wp, &
      &  262.9498_wp,  258.5263_wp,  251.8155_wp,  242.6736_wp,    0.0000_wp, &
      &  258.5263_wp,  254.1924_wp,  247.6174_wp,  238.6601_wp,    0.0000_wp, &
      &  251.8155_wp,  247.6174_wp,  241.2494_wp,  232.5744_wp,    0.0000_wp, &
      &  242.6736_wp,  238.6601_wp,  232.5744_wp,  224.2851_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   25.3098_wp,   25.3149_wp,   25.1268_wp,    0.0000_wp,    0.0000_wp, &
      &   40.2010_wp,   40.2117_wp,   39.8967_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   17.0243_wp,   17.0280_wp,   16.9157_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  354.5429_wp,  355.0450_wp,  351.8104_wp,    0.0000_wp,    0.0000_wp, &
      &  125.4768_wp,  125.5564_wp,  124.4868_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  215.0238_wp,  215.1671_wp,  213.3149_wp,    0.0000_wp,    0.0000_wp, &
      &  160.9687_wp,  161.0637_wp,  159.7099_wp,    0.0000_wp,    0.0000_wp, &
      &  107.6371_wp,  107.6713_wp,  106.8086_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  148.8008_wp,  148.8590_wp,  147.6502_wp,    0.0000_wp,    0.0000_wp, &
      &  125.8437_wp,  125.8888_wp,  124.8845_wp,    0.0000_wp,    0.0000_wp, &
      &  106.2140_wp,  106.2491_wp,  105.4167_wp,    0.0000_wp,    0.0000_wp, &
      &   81.8381_wp,   81.8564_wp,   81.2400_wp,    0.0000_wp,    0.0000_wp, &
      &   77.1533_wp,   77.1700_wp,   76.5939_wp,    0.0000_wp,    0.0000_wp, &
      &  102.4346_wp,  102.4620_wp,  101.6786_wp,    0.0000_wp,    0.0000_wp, &
      &   96.0599_wp,   96.0872_wp,   95.3553_wp,    0.0000_wp,    0.0000_wp, &
      &   78.8838_wp,   78.9029_wp,   78.3177_wp,    0.0000_wp,    0.0000_wp, &
      &   73.8064_wp,   73.8257_wp,   73.2814_wp,    0.0000_wp,    0.0000_wp, &
      &   60.9913_wp,   61.0041_wp,   60.5723_wp,    0.0000_wp,    0.0000_wp, &
      &   72.5403_wp,   72.5567_wp,   72.0328_wp,    0.0000_wp,    0.0000_wp, &
      &   67.6636_wp,   67.6796_wp,   67.1936_wp,    0.0000_wp,    0.0000_wp, &
      &   63.5919_wp,   63.6074_wp,   63.1549_wp,    0.0000_wp,    0.0000_wp, &
      &   56.0812_wp,   56.0962_wp,   55.7037_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   55.3627_wp,   55.3752_wp,   54.9943_wp,    0.0000_wp,    0.0000_wp, &
      &   49.7552_wp,   49.7665_wp,   49.4291_wp,    0.0000_wp,    0.0000_wp, &
      &   43.9819_wp,   43.9931_wp,   43.7022_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   42.1923_wp,   42.2027_wp,   41.9260_wp,    0.0000_wp,    0.0000_wp, &
      &   34.5792_wp,   34.5882_wp,   34.3729_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(570 * max_ref * max_ref + 1 : 589 * max_ref * max_ref) = [ &
      &   32.5843_wp,   32.5935_wp,   32.3892_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  425.1864_wp,  425.7733_wp,  421.9195_wp,    0.0000_wp,    0.0000_wp, &
      &  182.4744_wp,  182.6062_wp,  181.0487_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  339.5742_wp,  339.8427_wp,  336.8869_wp,    0.0000_wp,    0.0000_wp, &
      &  281.7173_wp,  281.9249_wp,  279.5035_wp,    0.0000_wp,    0.0000_wp, &
      &  188.6214_wp,  188.7025_wp,  187.1504_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  317.6182_wp,  317.8133_wp,  315.1147_wp,    0.0000_wp,    0.0000_wp, &
      &  288.8485_wp,  289.0163_wp,  286.5809_wp,    0.0000_wp,    0.0000_wp, &
      &  227.6936_wp,  227.7982_wp,  225.9245_wp,    0.0000_wp,    0.0000_wp, &
      &  179.8635_wp,  179.9198_wp,  178.4835_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  255.0099_wp,  255.1153_wp,  253.0289_wp,    0.0000_wp,    0.0000_wp, &
      &  236.3323_wp,  236.4253_wp,  234.5066_wp,    0.0000_wp,    0.0000_wp, &
      &  220.3806_wp,  220.4624_wp,  218.6853_wp,    0.0000_wp,    0.0000_wp, &
      &  214.4121_wp,  214.4981_wp,  212.7679_wp,    0.0000_wp,    0.0000_wp, &
      &  178.4799_wp,  178.5302_wp,  177.1287_wp,    0.0000_wp,    0.0000_wp, &
      &  201.9019_wp,  201.9603_wp,  200.3747_wp,    0.0000_wp,    0.0000_wp, &
      &  196.0806_wp,  196.1374_wp,  194.6017_wp,    0.0000_wp,    0.0000_wp, &
      &  193.2410_wp,  193.2996_wp,  191.7848_wp,    0.0000_wp,    0.0000_wp, &
      &  179.9374_wp,  179.9862_wp,  178.5907_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  169.1308_wp,  169.1712_wp,  167.8868_wp,    0.0000_wp,    0.0000_wp, &
      &  167.2264_wp,  167.2672_wp,  165.9975_wp,    0.0000_wp,    0.0000_wp, &
      &  163.8681_wp,  163.9083_wp,  162.6659_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  139.3163_wp,  139.3453_wp,  138.3253_wp,    0.0000_wp,    0.0000_wp, &
      &  137.8443_wp,  137.8735_wp,  136.8644_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  114.7449_wp,  114.7677_wp,  113.9577_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  692.9116_wp,  693.9818_wp,  687.7370_wp,    0.0000_wp,    0.0000_wp, &
      &  254.3670_wp,  254.5376_wp,  252.4498_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  587.9647_wp,  588.5763_wp,  583.4009_wp,    0.0000_wp,    0.0000_wp, &
      &  471.4822_wp,  471.9821_wp,  467.8832_wp,    0.0000_wp,    0.0000_wp, &
      &  264.3790_wp,  264.5072_wp,  262.3613_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  488.8715_wp,  489.3305_wp,  485.0594_wp,    0.0000_wp,    0.0000_wp, &
      &  258.0084_wp,  258.1295_wp,  256.0474_wp,    0.0000_wp,    0.0000_wp, &
      &  255.8702_wp,  255.9746_wp,  253.9272_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  475.0348_wp,  475.4453_wp,  471.3428_wp,    0.0000_wp,    0.0000_wp, &
      &  249.2258_wp,  249.3437_wp,  247.3402_wp,    0.0000_wp,    0.0000_wp, &
      &  259.1489_wp,  259.2539_wp,  257.1841_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  436.5718_wp,  436.9293_wp,  433.1830_wp,    0.0000_wp,    0.0000_wp, &
      &  244.1554_wp,  244.2727_wp,  242.3111_wp,    0.0000_wp,    0.0000_wp, &
      &  237.0577_wp,  237.1592_wp,  235.2729_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  344.2956_wp,  344.5858_wp,  341.6637_wp,    0.0000_wp,    0.0000_wp, &
      &  241.4585_wp,  241.5803_wp,  239.6357_wp,    0.0000_wp,    0.0000_wp, &
      &  180.4705_wp,  180.5365_wp,  179.1559_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  377.8276_wp,  378.1133_wp,  374.9026_wp,    0.0000_wp,    0.0000_wp, &
      &  223.9857_wp,  224.0757_wp,  222.2961_wp,    0.0000_wp,    0.0000_wp, &
      &  155.4154_wp,  155.4652_wp,  154.3023_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  297.0396_wp,  297.2697_wp,  294.7778_wp,    0.0000_wp,    0.0000_wp, &
      &  207.7948_wp,  207.8748_wp,  206.2367_wp,    0.0000_wp,    0.0000_wp, &
      &  152.4128_wp,  152.4629_wp,  151.3202_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  317.0174_wp,  317.2233_wp,  314.5706_wp,    0.0000_wp,    0.0000_wp, &
      &  198.2213_wp,  198.2993_wp,  196.7403_wp,    0.0000_wp,    0.0000_wp, &
      &  176.7844_wp,  176.8558_wp,  175.4826_wp,    0.0000_wp,    0.0000_wp, &
      &  158.4907_wp,  158.5463_wp,  157.3426_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  325.3419_wp,  325.5667_wp,  322.8238_wp,    0.0000_wp,    0.0000_wp, &
      &  193.6432_wp,  193.7213_wp,  192.1970_wp,    0.0000_wp,    0.0000_wp, &
      &  165.1961_wp,  165.2584_wp,  163.9864_wp,    0.0000_wp,    0.0000_wp, &
      &  172.2718_wp,  172.3493_wp,  171.0031_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(589 * max_ref * max_ref + 1 : 608 * max_ref * max_ref) = [ &
      &  249.6385_wp,  249.8210_wp,  247.7460_wp,    0.0000_wp,    0.0000_wp, &
      &  190.0771_wp,  190.1685_wp,  188.6560_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  259.0459_wp,  259.1916_wp,  257.0583_wp,    0.0000_wp,    0.0000_wp, &
      &  180.4189_wp,  180.4812_wp,  179.0747_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  305.9501_wp,  306.1275_wp,  303.5764_wp,    0.0000_wp,    0.0000_wp, &
      &  288.3934_wp,  288.5493_wp,  286.1624_wp,    0.0000_wp,    0.0000_wp, &
      &  231.3353_wp,  231.4312_wp,  229.5690_wp,    0.0000_wp,    0.0000_wp, &
      &  191.4849_wp,  191.5424_wp,  190.0442_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  273.6530_wp,  273.7680_wp,  271.5474_wp,    0.0000_wp,    0.0000_wp, &
      &  256.8517_wp,  256.9523_wp,  254.8857_wp,    0.0000_wp,    0.0000_wp, &
      &  245.0690_wp,  245.1604_wp,  243.2002_wp,    0.0000_wp,    0.0000_wp, &
      &  220.7846_wp,  220.8565_wp,  219.1161_wp,    0.0000_wp,    0.0000_wp, &
      &  196.1962_wp,  196.2489_wp,  194.7306_wp,    0.0000_wp,    0.0000_wp, &
      &  236.3473_wp,  236.4206_wp,  234.5631_wp,    0.0000_wp,    0.0000_wp, &
      &  232.5072_wp,  232.5784_wp,  230.7558_wp,    0.0000_wp,    0.0000_wp, &
      &  226.6845_wp,  226.7519_wp,  224.9818_wp,    0.0000_wp,    0.0000_wp, &
      &  218.7531_wp,  218.8149_wp,  217.1161_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  213.6738_wp,  213.7298_wp,  212.0915_wp,    0.0000_wp,    0.0000_wp, &
      &  213.7298_wp,  213.7862_wp,  212.1475_wp,    0.0000_wp,    0.0000_wp, &
      &  212.0915_wp,  212.1475_wp,  210.5231_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   22.4834_wp,   22.6230_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   35.2740_wp,   35.5123_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   15.4553_wp,   15.5387_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  294.7897_wp,  297.6835_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  107.3380_wp,  108.2048_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  183.3742_wp,  184.8832_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  138.2747_wp,  139.3617_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   93.8911_wp,   94.5515_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  129.2495_wp,  130.1881_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  109.8060_wp,  110.5802_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   93.1003_wp,   93.7374_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   72.4613_wp,   72.9221_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   68.4416_wp,   68.8714_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   90.3424_wp,   90.9343_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   84.7723_wp,   85.3265_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   70.0352_wp,   70.4744_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   65.5868_wp,   65.9967_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   54.6824_wp,   55.0029_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   64.7632_wp,   65.1543_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   60.4728_wp,   60.8359_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   56.9187_wp,   57.2578_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   50.3344_wp,   50.6298_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   49.8795_wp,   50.1632_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   44.9510_wp,   45.2019_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   39.8821_wp,   40.1002_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   38.3225_wp,   38.5294_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   31.6699_wp,   31.8313_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   29.7958_wp,   29.9501_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  354.2869_wp,  357.7246_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  155.8202_wp,  157.1005_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  288.2736_wp,  290.7276_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  240.1565_wp,  242.1476_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  163.2221_wp,  164.4383_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  272.0098_wp,  274.1894_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  247.9692_wp,  249.9248_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  196.9743_wp,  198.4473_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  157.0191_wp,  158.1182_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(608 * max_ref * max_ref + 1 : 627 * max_ref * max_ref) = [ &
      &  221.0537_wp,  222.6812_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  205.2976_wp,  206.7880_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  191.7992_wp,  193.1738_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  186.4927_wp,  187.8378_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  156.4781_wp,  157.5423_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  176.9930_wp,  178.1984_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  171.9910_wp,  173.1582_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  169.4313_wp,  170.5858_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  158.2104_wp,  159.2666_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  149.4563_wp,  150.4203_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  147.7679_wp,  148.7219_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  144.8387_wp,  145.7724_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  124.0909_wp,  124.8495_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  122.7763_wp,  123.5274_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  102.9471_wp,  103.5467_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  577.2318_wp,  582.8914_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  219.4647_wp,  221.1642_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  495.9754_wp,  500.4152_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  399.0226_wp,  402.5380_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  229.3147_wp,  231.0107_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  413.7468_wp,  417.3648_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  224.1375_wp,  225.7784_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  222.9256_wp,  224.5209_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  403.6576_wp,  407.0970_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  216.6921_wp,  218.2711_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  225.8818_wp,  227.4937_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  371.8071_wp,  374.9283_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  212.2792_wp,  213.8276_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  206.7277_wp,  208.2022_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  293.9185_wp,  296.3607_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  209.7393_wp,  211.2809_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  158.8704_wp,  159.9362_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  322.8433_wp,  325.4945_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  195.4536_wp,  196.8395_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  137.5406_wp,  138.4296_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  254.5306_wp,  256.5939_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  181.6836_wp,  182.9550_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  134.8051_wp,  135.6806_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  272.3053_wp,  274.4620_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  173.3793_wp,  174.5910_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  155.0169_wp,  156.0859_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  139.7541_wp,  140.6812_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  278.7705_wp,  281.0154_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  169.3133_wp,  170.5004_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  145.1828_wp,  146.1681_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  150.7426_wp,  151.8006_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  214.5059_wp,  216.2140_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  165.5860_wp,  166.7816_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  223.6348_wp,  225.3457_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  158.2213_wp,  159.3049_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  263.3458_wp,  265.3939_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  248.8211_wp,  250.7247_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  201.2706_wp,  202.7225_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  167.9500_wp,  169.0925_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  237.6295_wp,  239.3636_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  223.5744_wp,  225.1792_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  213.6651_wp,  215.1816_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  193.2591_wp,  194.5928_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  172.5823_wp,  173.7316_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(627 * max_ref * max_ref + 1 : 646 * max_ref * max_ref) = [ &
      &  207.0772_wp,  208.4957_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  203.8400_wp,  205.2305_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  198.9308_wp,  200.2786_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  192.2421_wp,  193.5315_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  188.3605_wp,  189.5974_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  188.4031_wp,  189.6408_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  186.9994_wp,  188.2263_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  167.1297_wp,  168.1791_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  168.1791_wp,  169.2375_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   19.8182_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   30.7261_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   13.9211_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  244.5554_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   91.3333_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  155.5616_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  118.1240_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   81.3306_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  111.5371_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   95.1763_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   81.0518_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   63.6768_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   60.2544_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   79.1080_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   74.2865_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   61.7254_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   57.8665_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   48.6500_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   57.3895_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   53.6496_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   50.5746_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   44.8546_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   44.6003_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   40.3043_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   35.8961_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   34.5452_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   28.7800_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   27.0430_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  294.5574_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  132.4348_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  243.5965_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  203.7473_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  140.3605_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  231.6904_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  211.6949_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  169.3547_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  136.1238_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  190.3934_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  177.1811_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  165.8230_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  161.1703_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  136.2102_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  154.0575_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  149.7933_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  147.5193_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  138.1076_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  131.0928_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  129.6123_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  127.0791_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(646 * max_ref * max_ref + 1 : 665 * max_ref * max_ref) = [ &
      &  109.6839_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  108.5226_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   91.6430_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  480.2472_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  188.4819_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  416.9777_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  336.6483_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  197.7645_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  348.8474_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  193.5868_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  193.0224_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  341.6273_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  187.3231_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  195.6666_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  315.3251_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  183.5171_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  179.1965_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  249.9413_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  181.1819_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  138.9788_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  274.6419_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  169.5097_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  120.9219_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  217.2071_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  157.8661_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  118.4578_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  232.7594_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  150.7190_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  135.1147_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  122.4536_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  237.7447_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  147.1417_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  126.8094_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  131.1600_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  183.5347_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  143.4525_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  192.0484_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  137.8590_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  225.4537_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  213.4821_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  174.0230_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  146.2910_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  205.0598_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  193.3634_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  185.0751_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  168.0194_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  150.7260_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  180.1818_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  177.4716_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  173.3582_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  167.7505_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  164.8494_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  164.8833_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  163.6906_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  147.1830_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  148.0689_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(665 * max_ref * max_ref + 1 : 684 * max_ref * max_ref) = [ &
      &  130.4017_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   85.9499_wp,   35.1963_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  150.9340_wp,   57.9039_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   50.0491_wp,   22.6231_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2647.3331_wp,  657.4076_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  618.5681_wp,  199.6771_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1082.7052_wp,  345.4378_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  774.9656_wp,  252.9574_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  425.1704_wp,  157.9479_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  620.0878_wp,  222.5081_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  506.9421_wp,  185.7206_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  415.2201_wp,  154.7827_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  286.2486_wp,  114.9721_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  266.1018_wp,  107.8668_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  373.5384_wp,  145.9974_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  353.6302_wp,  137.2125_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  273.3757_wp,  110.4856_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  256.5198_wp,  103.4559_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  195.5996_wp,   83.2418_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  240.6940_wp,  100.1418_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  225.1919_wp,   93.4584_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  209.2378_wp,   87.5447_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  183.1486_wp,   77.0040_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  172.3398_wp,   74.8868_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  152.6757_wp,   67.0045_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  132.4879_wp,   58.9015_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  124.5879_wp,   56.1779_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   96.2235_wp,   45.2489_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   92.3827_wp,   42.8957_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3104.1016_wp,  781.8527_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  936.3006_wp,  294.6795_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1826.4648_wp,  559.1337_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1477.6556_wp,  457.7831_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  808.8537_wp,  285.4219_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1546.9229_wp,  502.9933_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1375.2558_wp,  453.3636_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  999.7400_wp,  346.4959_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  704.9609_wp,  263.2439_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1082.0563_wp,  383.9084_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  986.0438_wp,  353.4142_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  901.7739_wp,  327.3080_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  894.0072_wp,  320.2665_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  675.8635_wp,  257.9020_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  768.1335_wp,  292.1604_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  744.2475_wp,  283.4368_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  739.7269_wp,  280.1183_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  668.4401_wp,  258.1520_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  602.2221_wp,  239.0572_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  596.8763_wp,  236.5467_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  584.3784_wp,  231.7265_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  466.1565_wp,  192.7877_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  462.0252_wp,  190.8602_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  364.4067_wp,  156.1010_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(684 * max_ref * max_ref + 1 : 703 * max_ref * max_ref) = [ &
      & 5530.2806_wp, 1305.8840_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1240.8631_wp,  401.0371_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3599.4349_wp, 1013.0706_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2958.0410_wp,  812.3489_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1152.0197_wp,  401.6572_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2834.0920_wp,  826.1041_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1105.4608_wp,  389.9057_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1056.1444_wp,  381.2151_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2641.2435_wp,  789.4117_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1074.1973_wp,  376.3919_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1065.4841_wp,  385.5830_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2363.7175_wp,  718.2125_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1053.5834_wp,  369.0649_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  984.9504_wp,  353.5929_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1874.0326_wp,  566.1856_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1057.5718_wp,  366.8686_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  692.7359_wp,  261.7869_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1968.7631_wp,  612.6444_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  911.9433_wp,  332.2215_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  566.7169_wp,  221.5102_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1547.1982_wp,  480.6236_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  831.7960_wp,  306.2720_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  559.2177_wp,  217.7195_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1541.8997_wp,  501.5940_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  795.8859_wp,  292.2992_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  704.2730_wp,  259.8039_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  596.8306_wp,  228.5106_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1625.1034_wp,  519.9826_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  781.8441_wp,  286.1241_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  640.5782_wp,  240.6629_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  705.9328_wp,  255.8512_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1259.4427_wp,  399.3534_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  807.8786_wp,  286.0131_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1186.2631_wp,  401.0548_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  697.2012_wp,  262.7332_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1442.3807_wp,  477.9737_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1322.7322_wp,  446.0798_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  968.1682_wp,  345.9921_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  727.9978_wp,  277.1701_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1155.1526_wp,  411.0389_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1059.6044_wp,  382.5012_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  994.2046_wp,  362.8020_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  859.6583_wp,  322.1096_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  724.9394_wp,  281.0734_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  909.4257_wp,  343.3978_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  890.4931_wp,  337.2160_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  860.3672_wp,  327.7032_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  818.2247_wp,  314.6518_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  778.9204_wp,  304.5312_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  780.0986_wp,  304.7279_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  773.1255_wp,  302.2530_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  650.3606_wp,  263.4592_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  656.6574_wp,  265.4647_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  542.3160_wp,  226.8590_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 6138.7755_wp, 1457.7817_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1457.7817_wp,  476.2591_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(703 * max_ref * max_ref + 1 : 722 * max_ref * max_ref) = [ &
      &   78.3901_wp,   64.4065_wp,   36.8606_wp,    0.0000_wp,    0.0000_wp, &
      &  135.0664_wp,  110.3275_wp,   60.1892_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   46.6658_wp,   38.8333_wp,   23.8401_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1944.2260_wp, 1618.2465_wp,  628.2399_wp,    0.0000_wp,    0.0000_wp, &
      &  517.8393_wp,  421.4610_wp,  201.6900_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  903.9265_wp,  734.7926_wp,  348.1690_wp,    0.0000_wp,    0.0000_wp, &
      &  648.6422_wp,  529.5494_wp,  256.1295_wp,    0.0000_wp,    0.0000_wp, &
      &  376.6468_wp,  306.9433_wp,  163.4621_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  541.5729_wp,  441.1261_wp,  228.9512_wp,    0.0000_wp,    0.0000_wp, &
      &  445.2608_wp,  363.3607_wp,  191.6429_wp,    0.0000_wp,    0.0000_wp, &
      &  365.6571_wp,  299.2914_wp,  160.1098_wp,    0.0000_wp,    0.0000_wp, &
      &  259.5279_wp,  212.8759_wp,  120.1300_wp,    0.0000_wp,    0.0000_wp, &
      &  241.8504_wp,  198.5660_wp,  112.8093_wp,    0.0000_wp,    0.0000_wp, &
      &  335.4597_wp,  274.7031_wp,  151.9745_wp,    0.0000_wp,    0.0000_wp, &
      &  315.8923_wp,  258.9885_wp,  142.6113_wp,    0.0000_wp,    0.0000_wp, &
      &  247.8285_wp,  203.5706_wp,  115.4107_wp,    0.0000_wp,    0.0000_wp, &
      &  232.0032_wp,  190.6976_wp,  107.9688_wp,    0.0000_wp,    0.0000_wp, &
      &  179.8395_wp,  148.5141_wp,   87.3891_wp,    0.0000_wp,    0.0000_wp, &
      &  219.9608_wp,  181.1973_wp,  104.9004_wp,    0.0000_wp,    0.0000_wp, &
      &  205.1574_wp,  169.1772_wp,   97.8085_wp,    0.0000_wp,    0.0000_wp, &
      &  191.1268_wp,  157.6836_wp,   91.6725_wp,    0.0000_wp,    0.0000_wp, &
      &  167.0865_wp,  138.0927_wp,   80.5868_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  159.1460_wp,  131.7846_wp,   78.6933_wp,    0.0000_wp,    0.0000_wp, &
      &  141.1592_wp,  117.1018_wp,   70.4285_wp,    0.0000_wp,    0.0000_wp, &
      &  122.6928_wp,  102.0116_wp,   61.9159_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  115.9647_wp,   96.5080_wp,   59.1415_wp,    0.0000_wp,    0.0000_wp, &
      &   90.4081_wp,   75.6758_wp,   47.7411_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   86.4627_wp,   72.2703_wp,   45.1941_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2296.1906_wp, 1907.9592_wp,  749.3005_wp,    0.0000_wp,    0.0000_wp, &
      &  773.9854_wp,  630.8304_wp,  296.0005_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1495.8141_wp, 1218.1274_wp,  558.7819_wp,    0.0000_wp,    0.0000_wp, &
      & 1210.9593_wp,  988.7857_wp,  458.8001_wp,    0.0000_wp,    0.0000_wp, &
      &  703.3906_wp,  571.9448_wp,  292.9174_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1298.5356_wp, 1057.0218_wp,  508.8061_wp,    0.0000_wp,    0.0000_wp, &
      & 1160.2539_wp,  944.7535_wp,  459.7370_wp,    0.0000_wp,    0.0000_wp, &
      &  859.5678_wp,  700.6746_wp,  354.5526_wp,    0.0000_wp,    0.0000_wp, &
      &  625.6996_wp,  509.9927_wp,  272.6252_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  941.4121_wp,  766.3847_wp,  394.3455_wp,    0.0000_wp,    0.0000_wp, &
      &  860.3421_wp,  701.0009_wp,  363.5872_wp,    0.0000_wp,    0.0000_wp, &
      &  790.6765_wp,  644.4017_wp,  337.3757_wp,    0.0000_wp,    0.0000_wp, &
      &  778.3539_wp,  634.9125_wp,  329.3386_wp,    0.0000_wp,    0.0000_wp, &
      &  603.7134_wp,  492.9574_wp,  267.8525_wp,    0.0000_wp,    0.0000_wp, &
      &  684.9632_wp,  559.3821_wp,  303.2521_wp,    0.0000_wp,    0.0000_wp, &
      &  663.5656_wp,  542.1262_wp,  294.2134_wp,    0.0000_wp,    0.0000_wp, &
      &  657.7877_wp,  537.4352_wp,  290.4761_wp,    0.0000_wp,    0.0000_wp, &
      &  598.7498_wp,  489.5943_wp,  268.4526_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  543.9627_wp,  445.7860_wp,  249.4267_wp,    0.0000_wp,    0.0000_wp, &
      &  538.6724_wp,  441.4719_wp,  246.7293_wp,    0.0000_wp,    0.0000_wp, &
      &  527.3838_wp,  432.2783_wp,  241.6949_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  425.8997_wp,  350.4273_wp,  201.9997_wp,    0.0000_wp,    0.0000_wp, &
      &  421.8867_wp,  347.1255_wp,  199.9353_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  335.8690_wp,  277.4759_wp,  164.0275_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3915.3529_wp, 3302.2504_wp, 1234.9320_wp,    0.0000_wp,    0.0000_wp, &
      & 1027.9671_wp,  841.9983_wp,  404.3537_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(722 * max_ref * max_ref + 1 : 741 * max_ref * max_ref) = [ &
      & 2819.4935_wp, 2314.8420_wp,  994.4676_wp,    0.0000_wp,    0.0000_wp, &
      & 2265.4155_wp, 1876.9417_wp,  794.5479_wp,    0.0000_wp,    0.0000_wp, &
      &  991.9143_wp,  808.6909_wp,  410.7857_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2260.4217_wp, 1849.7319_wp,  816.9275_wp,    0.0000_wp,    0.0000_wp, &
      &  957.2271_wp,  779.9488_wp,  399.3277_wp,    0.0000_wp,    0.0000_wp, &
      &  922.1430_wp,  752.1641_wp,  392.1967_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2128.3951_wp, 1740.7410_wp,  784.6886_wp,    0.0000_wp,    0.0000_wp, &
      &  923.6557_wp,  754.6753_wp,  385.2756_wp,    0.0000_wp,    0.0000_wp, &
      &  931.2172_wp,  759.5978_wp,  396.8216_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1918.6941_wp, 1568.2263_wp,  716.2778_wp,    0.0000_wp,    0.0000_wp, &
      &  906.2364_wp,  740.1763_wp,  377.6508_wp,    0.0000_wp,    0.0000_wp, &
      &  855.9005_wp,  699.1302_wp,  363.3125_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1511.3043_wp, 1238.5036_wp,  563.7114_wp,    0.0000_wp,    0.0000_wp, &
      &  905.6187_wp,  739.8315_wp,  374.7437_wp,    0.0000_wp,    0.0000_wp, &
      &  612.3549_wp,  501.5114_wp,  270.7543_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1614.7190_wp, 1318.8096_wp,  613.8418_wp,    0.0000_wp,    0.0000_wp, &
      &  799.0125_wp,  651.8405_wp,  342.1821_wp,    0.0000_wp,    0.0000_wp, &
      &  506.8036_wp,  415.8340_wp,  230.1231_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1263.2327_wp, 1034.0602_wp,  481.0447_wp,    0.0000_wp,    0.0000_wp, &
      &  731.2330_wp,  596.9767_wp,  315.9355_wp,    0.0000_wp,    0.0000_wp, &
      &  499.4375_wp,  409.6773_wp,  226.0537_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1290.8718_wp, 1052.2336_wp,  506.7272_wp,    0.0000_wp,    0.0000_wp, &
      &  698.0762_wp,  570.3024_wp,  301.3614_wp,    0.0000_wp,    0.0000_wp, &
      &  617.2493_wp,  504.9905_wp,  267.8234_wp,    0.0000_wp,    0.0000_wp, &
      &  530.3589_wp,  434.4503_wp,  236.7608_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1351.3702_wp, 1101.6666_wp,  523.7002_wp,    0.0000_wp,    0.0000_wp, &
      &  684.7715_wp,  559.3883_wp,  294.8004_wp,    0.0000_wp,    0.0000_wp, &
      &  565.7790_wp,  462.8902_wp,  248.7379_wp,    0.0000_wp,    0.0000_wp, &
      &  614.6139_wp,  502.4634_wp,  262.8982_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1037.7284_wp,  848.7952_wp,  401.1714_wp,    0.0000_wp,    0.0000_wp, &
      &  697.8623_wp,  569.9151_wp,  292.9783_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1009.6809_wp,  822.5167_wp,  407.9499_wp,    0.0000_wp,    0.0000_wp, &
      &  618.5928_wp,  505.2608_wp,  271.9818_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1216.5138_wp,  991.9123_wp,  484.8813_wp,    0.0000_wp,    0.0000_wp, &
      & 1123.9167_wp,  916.2312_wp,  453.9182_wp,    0.0000_wp,    0.0000_wp, &
      &  841.3610_wp,  686.5985_wp,  355.5621_wp,    0.0000_wp,    0.0000_wp, &
      &  649.2640_wp,  530.2448_wp,  287.6167_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1004.4824_wp,  818.4887_wp,  422.1813_wp,    0.0000_wp,    0.0000_wp, &
      &  926.0025_wp,  755.0584_wp,  393.7660_wp,    0.0000_wp,    0.0000_wp, &
      &  872.4465_wp,  711.5862_wp,  374.0975_wp,    0.0000_wp,    0.0000_wp, &
      &  761.7364_wp,  621.9263_wp,  333.4648_wp,    0.0000_wp,    0.0000_wp, &
      &  650.4849_wp,  531.8762_wp,  292.4289_wp,    0.0000_wp,    0.0000_wp, &
      &  808.3370_wp,  660.0928_wp,  355.9394_wp,    0.0000_wp,    0.0000_wp, &
      &  792.0816_wp,  647.0220_wp,  349.6598_wp,    0.0000_wp,    0.0000_wp, &
      &  766.7421_wp,  626.5605_wp,  340.0688_wp,    0.0000_wp,    0.0000_wp, &
      &  731.7462_wp,  598.2102_wp,  326.9624_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  700.0532_wp,  573.0850_wp,  317.0938_wp,    0.0000_wp,    0.0000_wp, &
      &  700.8015_wp,  573.7135_wp,  317.2498_wp,    0.0000_wp,    0.0000_wp, &
      &  694.6556_wp,  568.7416_wp,  314.6933_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  590.7326_wp,  485.0650_wp,  275.4476_wp,    0.0000_wp,    0.0000_wp, &
      &  596.0214_wp,  489.3461_wp,  277.4689_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  496.9120_wp,  409.3761_wp,  237.9219_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4349.2577_wp, 3670.9261_wp, 1380.8411_wp,    0.0000_wp,    0.0000_wp, &
      & 1210.8108_wp,  992.7970_wp,  481.0605_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3381.3672_wp, 2782.4577_wp, 1185.8018_wp,    0.0000_wp,    0.0000_wp, &
      & 2782.4577_wp, 2312.6602_wp,  968.0787_wp,    0.0000_wp,    0.0000_wp, &
      & 1185.8018_wp,  968.0787_wp,  491.7448_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(741 * max_ref * max_ref + 1 : 760 * max_ref * max_ref) = [ &
      &   70.0570_wp,   51.2356_wp,   36.0543_wp,    0.0000_wp,    0.0000_wp, &
      &  119.2961_wp,   85.6815_wp,   58.2755_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   42.4195_wp,   31.9889_wp,   23.6462_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1585.3950_wp, 1074.5875_wp,  567.3202_wp,    0.0000_wp,    0.0000_wp, &
      &  442.9448_wp,  306.8862_wp,  189.8894_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  771.4844_wp,  532.5110_wp,  326.9186_wp,    0.0000_wp,    0.0000_wp, &
      &  556.1503_wp,  387.6564_wp,  242.0080_wp,    0.0000_wp,    0.0000_wp, &
      &  330.7190_wp,  235.5232_wp,  157.4277_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  472.4364_wp,  334.1351_wp,  219.3347_wp,    0.0000_wp,    0.0000_wp, &
      &  389.8823_wp,  277.4050_wp,  184.2847_wp,    0.0000_wp,    0.0000_wp, &
      &  321.2652_wp,  230.1384_wp,  154.5280_wp,    0.0000_wp,    0.0000_wp, &
      &  231.1173_wp,  168.1354_wp,  117.1570_wp,    0.0000_wp,    0.0000_wp, &
      &  215.7263_wp,  157.3598_wp,  110.1737_wp,    0.0000_wp,    0.0000_wp, &
      &  297.2171_wp,  214.7760_wp,  147.6049_wp,    0.0000_wp,    0.0000_wp, &
      &  279.4727_wp,  201.9744_wp,  138.4353_wp,    0.0000_wp,    0.0000_wp, &
      &  220.8830_wp,  161.1115_wp,  112.6680_wp,    0.0000_wp,    0.0000_wp, &
      &  206.6589_wp,  150.7894_wp,  105.3858_wp,    0.0000_wp,    0.0000_wp, &
      &  161.7738_wp,  119.7654_wp,   85.9668_wp,    0.0000_wp,    0.0000_wp, &
      &  197.0611_wp,  144.9207_wp,  102.8479_wp,    0.0000_wp,    0.0000_wp, &
      &  183.6851_wp,  135.1839_wp,   95.8897_wp,    0.0000_wp,    0.0000_wp, &
      &  171.3574_wp,  126.3470_wp,   89.9626_wp,    0.0000_wp,    0.0000_wp, &
      &  149.8906_wp,  110.8238_wp,   79.1550_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  143.6602_wp,  107.0438_wp,   77.6335_wp,    0.0000_wp,    0.0000_wp, &
      &  127.6282_wp,   95.4458_wp,   69.5806_wp,    0.0000_wp,    0.0000_wp, &
      &  111.1606_wp,   83.5108_wp,   61.2796_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  105.3447_wp,   79.4076_wp,   58.6359_wp,    0.0000_wp,    0.0000_wp, &
      &   82.7337_wp,   63.1889_wp,   47.5914_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   78.9316_wp,   60.0628_wp,   44.9753_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1876.8634_wp, 1272.7028_wp,  678.1726_wp,    0.0000_wp,    0.0000_wp, &
      &  658.7963_wp,  454.9928_wp,  277.5701_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1266.7602_wp,  869.3739_wp,  521.1430_wp,    0.0000_wp,    0.0000_wp, &
      & 1028.1365_wp,  709.6961_wp,  429.5059_wp,    0.0000_wp,    0.0000_wp, &
      &  611.6961_wp,  430.3940_wp,  279.6834_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1112.2980_wp,  771.8606_wp,  479.6884_wp,    0.0000_wp,    0.0000_wp, &
      &  996.3757_wp,  693.4732_wp,  434.5102_wp,    0.0000_wp,    0.0000_wp, &
      &  745.1801_wp,  524.2267_wp,  338.0153_wp,    0.0000_wp,    0.0000_wp, &
      &  549.9242_wp,  392.0660_wp,  262.7634_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  819.5418_wp,  578.0550_wp,  377.0639_wp,    0.0000_wp,    0.0000_wp, &
      &  750.3817_wp,  530.8180_wp,  348.3162_wp,    0.0000_wp,    0.0000_wp, &
      &  691.2029_wp,  490.2154_wp,  323.8305_wp,    0.0000_wp,    0.0000_wp, &
      &  678.7577_wp,  480.7496_wp,  315.6210_wp,    0.0000_wp,    0.0000_wp, &
      &  532.6876_wp,  382.0537_wp,  259.1062_wp,    0.0000_wp,    0.0000_wp, &
      &  603.9916_wp,  433.0171_wp,  293.2324_wp,    0.0000_wp,    0.0000_wp, &
      &  585.2467_wp,  419.8718_wp,  284.5819_wp,    0.0000_wp,    0.0000_wp, &
      &  579.5105_wp,  415.3578_wp,  280.7471_wp,    0.0000_wp,    0.0000_wp, &
      &  529.4213_wp,  381.1414_wp,  260.2254_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  483.3797_wp,  350.5775_wp,  242.8256_wp,    0.0000_wp,    0.0000_wp, &
      &  478.5204_wp,  346.9710_wp,  240.1493_wp,    0.0000_wp,    0.0000_wp, &
      &  468.5269_wp,  339.8082_wp,  235.2717_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  381.2462_wp,  279.7355_wp,  197.8796_wp,    0.0000_wp,    0.0000_wp, &
      &  377.5642_wp,  276.9773_wp,  195.8267_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  302.5143_wp,  224.3126_wp,  161.5056_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3166.0625_wp, 2163.7031_wp, 1110.5426_wp,    0.0000_wp,    0.0000_wp, &
      &  879.0346_wp,  613.8463_wp,  381.7778_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(760 * max_ref * max_ref + 1 : 779 * max_ref * max_ref) = [ &
      & 2350.8018_wp, 1603.6362_wp,  916.0855_wp,    0.0000_wp,    0.0000_wp, &
      & 1882.1338_wp, 1293.5159_wp,  732.1243_wp,    0.0000_wp,    0.0000_wp, &
      &  860.3229_wp,  605.8453_wp,  391.8360_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1896.7809_wp, 1297.2964_wp,  756.5443_wp,    0.0000_wp,    0.0000_wp, &
      &  831.6679_wp,  586.3135_wp,  381.4719_wp,    0.0000_wp,    0.0000_wp, &
      &  805.0465_wp,  570.8618_wp,  376.1623_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1794.3180_wp, 1232.3433_wp,  730.0584_wp,    0.0000_wp,    0.0000_wp, &
      &  802.0324_wp,  566.8670_wp,  368.1481_wp,    0.0000_wp,    0.0000_wp, &
      &  813.3302_wp,  577.0286_wp,  380.7419_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1622.4689_wp, 1116.9531_wp,  668.2729_wp,    0.0000_wp,    0.0000_wp, &
      &  786.7255_wp,  555.7215_wp,  360.7708_wp,    0.0000_wp,    0.0000_wp, &
      &  746.4335_wp,  529.6824_wp,  348.3728_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1276.4352_wp,  880.6069_wp,  526.0324_wp,    0.0000_wp,    0.0000_wp, &
      &  784.7489_wp,  553.4764_wp,  357.4837_wp,    0.0000_wp,    0.0000_wp, &
      &  538.9029_wp,  387.0982_wp,  261.7042_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1371.4492_wp,  947.5834_wp,  575.0060_wp,    0.0000_wp,    0.0000_wp, &
      &  698.6760_wp,  496.3331_wp,  328.6223_wp,    0.0000_wp,    0.0000_wp, &
      &  448.7768_wp,  324.9751_wp,  223.5637_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1072.3265_wp,  742.6641_wp,  450.9120_wp,    0.0000_wp,    0.0000_wp, &
      &  640.6641_wp,  456.3878_wp,  303.9644_wp,    0.0000_wp,    0.0000_wp, &
      &  441.9258_wp,  319.6860_wp,  219.4714_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1105.3016_wp,  768.2107_wp,  477.8977_wp,    0.0000_wp,    0.0000_wp, &
      &  611.3403_wp,  435.6701_wp,  289.9137_wp,    0.0000_wp,    0.0000_wp, &
      &  540.8706_wp,  386.3776_wp,  257.9190_wp,    0.0000_wp,    0.0000_wp, &
      &  467.8612_wp,  336.9173_wp,  229.2527_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1153.5737_wp,  799.3710_wp,  492.5198_wp,    0.0000_wp,    0.0000_wp, &
      &  599.2652_wp,  426.7405_wp,  283.4424_wp,    0.0000_wp,    0.0000_wp, &
      &  497.3996_wp,  356.4737_wp,  240.1326_wp,    0.0000_wp,    0.0000_wp, &
      &  536.6745_wp,  381.7903_wp,  252.4245_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  884.1648_wp,  614.0983_wp,  377.2375_wp,    0.0000_wp,    0.0000_wp, &
      &  606.8585_wp,  429.3743_wp,  280.2569_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  870.6935_wp,  609.0499_wp,  387.0821_wp,    0.0000_wp,    0.0000_wp, &
      &  544.3140_wp,  389.5811_wp,  262.5735_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1045.5505_wp,  729.5813_wp,  458.9033_wp,    0.0000_wp,    0.0000_wp, &
      &  969.0777_wp,  678.2590_wp,  430.7925_wp,    0.0000_wp,    0.0000_wp, &
      &  733.3555_wp,  519.4437_wp,  340.6535_wp,    0.0000_wp,    0.0000_wp, &
      &  572.5452_wp,  410.5273_wp,  278.1125_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  874.7836_wp,  617.9985_wp,  403.9646_wp,    0.0000_wp,    0.0000_wp, &
      &  808.5991_wp,  573.2139_wp,  377.6930_wp,    0.0000_wp,    0.0000_wp, &
      &  763.3410_wp,  542.3655_wp,  359.4274_wp,    0.0000_wp,    0.0000_wp, &
      &  669.7473_wp,  478.7132_wp,  321.7190_wp,    0.0000_wp,    0.0000_wp, &
      &  575.5731_wp,  414.6203_wp,  283.5949_wp,    0.0000_wp,    0.0000_wp, &
      &  711.7506_wp,  509.5503_wp,  343.7975_wp,    0.0000_wp,    0.0000_wp, &
      &  697.8037_wp,  500.0071_wp,  337.9057_wp,    0.0000_wp,    0.0000_wp, &
      &  676.2036_wp,  485.2461_wp,  328.9431_wp,    0.0000_wp,    0.0000_wp, &
      &  646.4774_wp,  464.9254_wp,  316.7189_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  620.3334_wp,  448.1381_wp,  307.9733_wp,    0.0000_wp,    0.0000_wp, &
      &  620.8918_wp,  448.4872_wp,  308.0920_wp,    0.0000_wp,    0.0000_wp, &
      &  615.5317_wp,  444.7322_wp,  305.6508_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  526.8299_wp,  384.2901_wp,  268.9880_wp,    0.0000_wp,    0.0000_wp, &
      &  531.3426_wp,  387.3876_wp,  270.8810_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  445.7013_wp,  328.1316_wp,  233.4581_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3520.9466_wp, 2411.4577_wp, 1244.1578_wp,    0.0000_wp,    0.0000_wp, &
      & 1037.4434_wp,  726.8613_wp,  455.2460_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2813.7007_wp, 1920.9295_wp, 1091.2408_wp,    0.0000_wp,    0.0000_wp, &
      & 2305.6086_wp, 1586.5402_wp,  890.7440_wp,    0.0000_wp,    0.0000_wp, &
      & 1028.2536_wp,  725.1809_wp,  469.2639_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(779 * max_ref * max_ref + 1 : 798 * max_ref * max_ref) = [ &
      & 2365.8925_wp, 1624.3636_wp,  955.0153_wp,    0.0000_wp,    0.0000_wp, &
      & 1624.3636_wp, 1132.2983_wp,  679.4591_wp,    0.0000_wp,    0.0000_wp, &
      &  955.0153_wp,  679.4591_wp,  451.0845_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   63.7801_wp,   39.6669_wp,   36.5405_wp,    0.0000_wp,    0.0000_wp, &
      &  107.7306_wp,   64.7068_wp,   58.8625_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   39.1088_wp,   25.7037_wp,   24.0883_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1363.2842_wp,  678.4810_wp,  562.1583_wp,    0.0000_wp,    0.0000_wp, &
      &  391.9179_wp,  216.7170_wp,  190.2283_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  681.5283_wp,  374.0135_wp,  327.2192_wp,    0.0000_wp,    0.0000_wp, &
      &  493.0100_wp,  275.3644_wp,  242.7264_wp,    0.0000_wp,    0.0000_wp, &
      &  297.4750_wp,  175.6564_wp,  158.7442_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  423.2302_wp,  246.0147_wp,  220.8361_wp,    0.0000_wp,    0.0000_wp, &
      &  350.1970_wp,  206.0014_wp,  185.7762_wp,    0.0000_wp,    0.0000_wp, &
      &  289.2869_wp,  172.1810_wp,  155.9706_wp,    0.0000_wp,    0.0000_wp, &
      &  209.9066_wp,  129.2409_wp,  118.6243_wp,    0.0000_wp,    0.0000_wp, &
      &  196.1543_wp,  121.3842_wp,  111.6080_wp,    0.0000_wp,    0.0000_wp, &
      &  269.0490_wp,  163.4548_wp,  149.2658_wp,    0.0000_wp,    0.0000_wp, &
      &  252.8213_wp,  153.4229_wp,  139.9877_wp,    0.0000_wp,    0.0000_wp, &
      &  200.7817_wp,  124.1920_wp,  114.1335_wp,    0.0000_wp,    0.0000_wp, &
      &  187.8203_wp,  116.1973_wp,  106.7629_wp,    0.0000_wp,    0.0000_wp, &
      &  148.0069_wp,   94.1173_wp,   87.3116_wp,    0.0000_wp,    0.0000_wp, &
      &  179.7777_wp,  112.9339_wp,  104.3380_wp,    0.0000_wp,    0.0000_wp, &
      &  167.5501_wp,  105.3151_wp,   97.2881_wp,    0.0000_wp,    0.0000_wp, &
      &  156.4540_wp,   98.7165_wp,   91.3072_wp,    0.0000_wp,    0.0000_wp, &
      &  136.9578_wp,   86.8035_wp,   80.3758_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  131.7852_wp,   84.7876_wp,   78.9339_wp,    0.0000_wp,    0.0000_wp, &
      &  117.2328_wp,   75.8994_wp,   70.7865_wp,    0.0000_wp,    0.0000_wp, &
      &  102.2836_wp,   66.7498_wp,   62.3893_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   97.0977_wp,   63.7680_wp,   59.7299_wp,    0.0000_wp,    0.0000_wp, &
      &   76.6712_wp,   51.5158_wp,   48.5769_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   73.0315_wp,   48.7600_wp,   45.8824_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1616.2863_wp,  808.9973_wp,  672.4303_wp,    0.0000_wp,    0.0000_wp, &
      &  581.2480_wp,  318.1678_wp,  277.7936_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1113.8339_wp,  600.5511_wp,  520.6868_wp,    0.0000_wp,    0.0000_wp, &
      &  905.7750_wp,  493.3057_wp,  429.6462_wp,    0.0000_wp,    0.0000_wp, &
      &  546.7935_wp,  314.6402_wp,  281.2950_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  985.0494_wp,  546.7176_wp,  480.7313_wp,    0.0000_wp,    0.0000_wp, &
      &  883.8541_wp,  494.0245_wp,  435.7814_wp,    0.0000_wp,    0.0000_wp, &
      &  665.0386_wp,  381.0309_wp,  339.8622_wp,    0.0000_wp,    0.0000_wp, &
      &  494.9507_wp,  292.9765_wp,  265.0243_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  733.1807_wp,  423.6992_wp,  379.4212_wp,    0.0000_wp,    0.0000_wp, &
      &  672.1834_wp,  390.7192_wp,  350.7076_wp,    0.0000_wp,    0.0000_wp, &
      &  620.0792_wp,  362.5672_wp,  326.2431_wp,    0.0000_wp,    0.0000_wp, &
      &  608.0848_wp,  354.0018_wp,  317.8567_wp,    0.0000_wp,    0.0000_wp, &
      &  480.7318_wp,  287.9444_wp,  261.6469_wp,    0.0000_wp,    0.0000_wp, &
      &  544.8882_wp,  326.0087_wp,  296.0806_wp,    0.0000_wp,    0.0000_wp, &
      &  528.0841_wp,  316.3169_wp,  287.3830_wp,    0.0000_wp,    0.0000_wp, &
      &  522.5721_wp,  312.3050_wp,  283.4537_wp,    0.0000_wp,    0.0000_wp, &
      &  478.5196_wp,  288.6637_wp,  262.9675_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  438.3890_wp,  268.3077_wp,  245.7286_wp,    0.0000_wp,    0.0000_wp, &
      &  433.9036_wp,  265.4089_wp,  243.0089_wp,    0.0000_wp,    0.0000_wp, &
      &  424.8748_wp,  259.9991_wp,  238.0843_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  347.5334_wp,  217.4322_wp,  200.6665_wp,    0.0000_wp,    0.0000_wp, &
      &  344.1318_wp,  215.2101_wp,  198.5775_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  276.9908_wp,  176.6690_wp,  164.0761_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(798 * max_ref * max_ref + 1 : 817 * max_ref * max_ref) = [ &
      & 2711.1542_wp, 1335.9899_wp, 1099.8393_wp,    0.0000_wp,    0.0000_wp, &
      &  778.5238_wp,  435.0649_wp,  383.0224_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2048.5251_wp, 1070.9104_wp,  912.5205_wp,    0.0000_wp,    0.0000_wp, &
      & 1637.9667_wp,  856.7131_wp,  729.6211_wp,    0.0000_wp,    0.0000_wp, &
      &  768.2000_wp,  441.4951_wp,  394.1110_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1659.1156_wp,  879.0165_wp,  754.6182_wp,    0.0000_wp,    0.0000_wp, &
      &  743.4318_wp,  429.3019_wp,  383.8645_wp,    0.0000_wp,    0.0000_wp, &
      &  721.7837_wp,  421.5388_wp,  378.9395_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1574.1401_wp,  844.1565_wp,  729.1622_wp,    0.0000_wp,    0.0000_wp, &
      &  716.8609_wp,  414.2030_wp,  370.5158_wp,    0.0000_wp,    0.0000_wp, &
      &  729.4243_wp,  426.5157_wp,  383.5999_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1426.0687_wp,  770.4185_wp,  667.9695_wp,    0.0000_wp,    0.0000_wp, &
      &  703.0784_wp,  406.0223_wp,  363.0711_wp,    0.0000_wp,    0.0000_wp, &
      &  668.9667_wp,  390.6056_wp,  350.9659_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1121.5489_wp,  606.7040_wp,  525.9535_wp,    0.0000_wp,    0.0000_wp, &
      &  700.5432_wp,  402.9200_wp,  359.6267_wp,    0.0000_wp,    0.0000_wp, &
      &  485.9447_wp,  291.2297_wp,  264.3347_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1208.7451_wp,  660.1005_wp,  575.3920_wp,    0.0000_wp,    0.0000_wp, &
      &  627.0790_wp,  367.7937_wp,  331.1873_wp,    0.0000_wp,    0.0000_wp, &
      &  406.3267_wp,  247.5981_wp,  226.1684_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  945.1713_wp,  517.5812_wp,  451.4130_wp,    0.0000_wp,    0.0000_wp, &
      &  575.7752_wp,  339.6278_wp,  306.5124_wp,    0.0000_wp,    0.0000_wp, &
      &  399.9295_wp,  243.2086_wp,  221.9858_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  978.9767_wp,  544.6643_wp,  479.1089_wp,    0.0000_wp,    0.0000_wp, &
      &  549.3294_wp,  323.9921_wp,  292.3514_wp,    0.0000_wp,    0.0000_wp, &
      &  486.3325_wp,  288.0243_wp,  260.2108_wp,    0.0000_wp,    0.0000_wp, &
      &  422.5208_wp,  254.6698_wp,  231.6783_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1019.7589_wp,  562.9344_wp,  493.3692_wp,    0.0000_wp,    0.0000_wp, &
      &  538.2480_wp,  316.9417_wp,  285.7823_wp,    0.0000_wp,    0.0000_wp, &
      &  448.1600_wp,  267.4973_wp,  242.4422_wp,    0.0000_wp,    0.0000_wp, &
      &  481.5000_wp,  282.7021_wp,  254.4567_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  781.1133_wp,  431.5544_wp,  377.9964_wp,    0.0000_wp,    0.0000_wp, &
      &  542.9499_wp,  314.9908_wp,  282.1724_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  774.5871_wp,  438.4080_wp,  388.7295_wp,    0.0000_wp,    0.0000_wp, &
      &  490.5097_wp,  292.4005_wp,  265.0402_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  928.1982_wp,  521.1802_wp,  460.4959_wp,    0.0000_wp,    0.0000_wp, &
      &  862.0363_wp,  487.8759_wp,  432.6309_wp,    0.0000_wp,    0.0000_wp, &
      &  656.8592_wp,  382.1952_wp,  343.0520_wp,    0.0000_wp,    0.0000_wp, &
      &  516.5656_wp,  309.1964_wp,  280.8241_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  782.9442_wp,  453.6905_wp,  406.6196_wp,    0.0000_wp,    0.0000_wp, &
      &  724.9805_wp,  423.2076_wp,  380.4575_wp,    0.0000_wp,    0.0000_wp, &
      &  685.2716_wp,  402.0877_wp,  362.2407_wp,    0.0000_wp,    0.0000_wp, &
      &  603.1519_wp,  358.4772_wp,  324.6431_wp,    0.0000_wp,    0.0000_wp, &
      &  520.4592_wp,  314.4398_wp,  286.6200_wp,    0.0000_wp,    0.0000_wp, &
      &  641.5564_wp,  382.6458_wp,  347.0371_wp,    0.0000_wp,    0.0000_wp, &
      &  629.2190_wp,  375.9179_wp,  341.1482_wp,    0.0000_wp,    0.0000_wp, &
      &  610.1761_wp,  365.6312_wp,  332.1968_wp,    0.0000_wp,    0.0000_wp, &
      &  584.0160_wp,  351.5641_wp,  319.9916_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  561.5474_wp,  341.0338_wp,  311.4245_wp,    0.0000_wp,    0.0000_wp, &
      &  562.0001_wp,  341.2040_wp,  311.5368_wp,    0.0000_wp,    0.0000_wp, &
      &  557.2071_wp,  338.4610_wp,  309.0838_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  479.0040_wp,  296.3944_wp,  272.4910_wp,    0.0000_wp,    0.0000_wp, &
      &  482.9873_wp,  298.5634_wp,  274.3837_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  406.8727_wp,  256.1501_wp,  236.8855_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3017.6856_wp, 1493.7993_wp, 1232.9252_wp,    0.0000_wp,    0.0000_wp, &
      &  920.1358_wp,  517.6961_wp,  457.0717_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(817 * max_ref * max_ref + 1 : 836 * max_ref * max_ref) = [ &
      & 2449.4723_wp, 1277.5813_wp, 1086.8394_wp,    0.0000_wp,    0.0000_wp, &
      & 2003.7829_wp, 1044.3401_wp,  887.4810_wp,    0.0000_wp,    0.0000_wp, &
      &  918.2378_wp,  528.6574_wp,  472.1081_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2072.6170_wp, 1106.6630_wp,  953.4821_wp,    0.0000_wp,    0.0000_wp, &
      & 1429.3576_wp,  780.8212_wp,  680.2683_wp,    0.0000_wp,    0.0000_wp, &
      &  857.6527_wp,  504.4325_wp,  454.7356_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1822.7181_wp,  987.8257_wp,  857.6011_wp,    0.0000_wp,    0.0000_wp, &
      &  987.8257_wp,  568.6789_wp,  507.5216_wp,    0.0000_wp,    0.0000_wp, &
      &  857.6011_wp,  507.5216_wp,  458.6896_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   58.6768_wp,   40.0304_wp,   35.3152_wp,    0.0000_wp,    0.0000_wp, &
      &   98.5183_wp,   65.3129_wp,   56.8466_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   36.3418_wp,   25.9534_wp,   23.3592_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1209.0304_wp,  684.1040_wp,  549.6114_wp,    0.0000_wp,    0.0000_wp, &
      &  353.5008_wp,  218.9706_wp,  184.1222_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  613.9850_wp,  377.9693_wp,  316.7162_wp,    0.0000_wp,    0.0000_wp, &
      &  445.3736_wp,  278.0881_wp,  234.9568_wp,    0.0000_wp,    0.0000_wp, &
      &  271.2606_wp,  177.3357_wp,  153.2948_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  384.9072_wp,  248.4266_wp,  213.3735_wp,    0.0000_wp,    0.0000_wp, &
      &  319.1151_wp,  207.9943_wp,  179.5291_wp,    0.0000_wp,    0.0000_wp, &
      &  264.1285_wp,  173.7946_wp,  150.7698_wp,    0.0000_wp,    0.0000_wp, &
      &  192.7765_wp,  130.4379_wp,  114.6328_wp,    0.0000_wp,    0.0000_wp, &
      &  180.3042_wp,  122.5075_wp,  107.8711_wp,    0.0000_wp,    0.0000_wp, &
      &  246.5241_wp,  164.9980_wp,  144.2558_wp,    0.0000_wp,    0.0000_wp, &
      &  231.5976_wp,  154.8764_wp,  135.3493_wp,    0.0000_wp,    0.0000_wp, &
      &  184.5477_wp,  125.3625_wp,  110.3639_wp,    0.0000_wp,    0.0000_wp, &
      &  172.6395_wp,  117.3051_wp,  103.2775_wp,    0.0000_wp,    0.0000_wp, &
      &  136.7050_wp,   95.0009_wp,   84.5041_wp,    0.0000_wp,    0.0000_wp, &
      &  165.6943_wp,  113.9989_wp,  100.9460_wp,    0.0000_wp,    0.0000_wp, &
      &  154.4356_wp,  106.3141_wp,   94.1631_wp,    0.0000_wp,    0.0000_wp, &
      &  144.3122_wp,   99.6625_wp,   88.3944_wp,    0.0000_wp,    0.0000_wp, &
      &  126.4307_wp,   87.6504_wp,   77.8681_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  121.9868_wp,   85.6024_wp,   76.4658_wp,    0.0000_wp,    0.0000_wp, &
      &  108.6378_wp,   76.6347_wp,   68.6087_wp,    0.0000_wp,    0.0000_wp, &
      &   94.9281_wp,   67.4128_wp,   60.5217_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   90.2234_wp,   64.4010_wp,   57.9467_wp,    0.0000_wp,    0.0000_wp, &
      &   71.5493_wp,   52.0412_wp,   47.1941_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   68.0769_wp,   49.2635_wp,   44.5776_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1434.6641_wp,  816.0317_wp,  657.1151_wp,    0.0000_wp,    0.0000_wp, &
      &  523.4058_wp,  321.5378_wp,  269.1778_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1000.5663_wp,  607.0073_wp,  504.6753_wp,    0.0000_wp,    0.0000_wp, &
      &  814.9289_wp,  498.3486_wp,  416.4183_wp,    0.0000_wp,    0.0000_wp, &
      &  496.4807_wp,  317.7923_wp,  271.7876_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  889.0302_wp,  552.3550_wp,  465.2259_wp,    0.0000_wp,    0.0000_wp, &
      &  798.6094_wp,  499.0761_wp,  421.6461_wp,    0.0000_wp,    0.0000_wp, &
      &  603.3575_wp,  384.7678_wp,  328.5727_wp,    0.0000_wp,    0.0000_wp, &
      &  451.5308_wp,  295.7786_wp,  255.9320_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  666.1507_wp,  427.8837_wp,  366.6169_wp,    0.0000_wp,    0.0000_wp, &
      &  611.3147_wp,  394.5403_wp,  338.8785_wp,    0.0000_wp,    0.0000_wp, &
      &  564.4901_wp,  366.0984_wp,  315.2073_wp,    0.0000_wp,    0.0000_wp, &
      &  553.1447_wp,  357.4612_wp,  307.2381_wp,    0.0000_wp,    0.0000_wp, &
      &  439.4281_wp,  290.6641_wp,  252.7061_wp,    0.0000_wp,    0.0000_wp, &
      &  497.9703_wp,  329.0947_wp,  285.9926_wp,    0.0000_wp,    0.0000_wp, &
      &  482.7036_wp,  319.3099_wp,  277.6188_wp,    0.0000_wp,    0.0000_wp, &
      &  477.4772_wp,  315.2763_wp,  273.8649_wp,    0.0000_wp,    0.0000_wp, &
      &  437.9280_wp,  291.3813_wp,  254.0460_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  402.1974_wp,  270.8059_wp,  237.4448_wp,    0.0000_wp,    0.0000_wp, &
      &  398.0419_wp,  267.8858_wp,  234.8321_wp,    0.0000_wp,    0.0000_wp, &
      &  389.7889_wp,  262.4295_wp,  230.0871_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(836 * max_ref * max_ref + 1 : 855 * max_ref * max_ref) = [ &
      &  320.0676_wp,  219.4462_wp,  194.0301_wp,    0.0000_wp,    0.0000_wp, &
      &  316.9110_wp,  217.2078_wp,  192.0192_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  255.9769_wp,  178.3183_wp,  158.7915_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2400.6918_wp, 1343.3714_wp, 1078.8336_wp,    0.0000_wp,    0.0000_wp, &
      &  703.3060_wp,  439.5228_wp,  371.4433_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1830.8523_wp, 1081.9930_wp,  887.6167_wp,    0.0000_wp,    0.0000_wp, &
      & 1463.8890_wp,  864.1411_wp,  710.7573_wp,    0.0000_wp,    0.0000_wp, &
      &  697.3148_wp,  445.9895_wp,  381.2592_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1486.0620_wp,  888.3356_wp,  733.0114_wp,    0.0000_wp,    0.0000_wp, &
      &  675.3314_wp,  433.7865_wp,  371.3244_wp,    0.0000_wp,    0.0000_wp, &
      &  656.9546_wp,  425.7299_wp,  366.3716_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1412.6818_wp,  853.0464_wp,  707.8111_wp,    0.0000_wp,    0.0000_wp, &
      &  651.2847_wp,  418.3068_wp,  358.5232_wp,    0.0000_wp,    0.0000_wp, &
      &  664.0452_wp,  430.7613_wp,  370.8800_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1281.3349_wp,  778.5266_wp,  648.1120_wp,    0.0000_wp,    0.0000_wp, &
      &  638.7014_wp,  410.1033_wp,  351.3471_wp,    0.0000_wp,    0.0000_wp, &
      &  608.8442_wp,  394.4854_wp,  339.4910_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1007.8571_wp,  613.0224_wp,  510.7385_wp,    0.0000_wp,    0.0000_wp, &
      &  635.9639_wp,  406.9894_wp,  348.1021_wp,    0.0000_wp,    0.0000_wp, &
      &  444.2328_wp,  294.1186_wp,  255.7658_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1087.9837_wp,  667.0305_wp,  557.9514_wp,    0.0000_wp,    0.0000_wp, &
      &  571.1862_wp,  371.4691_wp,  320.2256_wp,    0.0000_wp,    0.0000_wp, &
      &  372.5215_wp,  250.0266_wp,  218.8458_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  851.0634_wp,  523.0020_wp,  438.0603_wp,    0.0000_wp,    0.0000_wp, &
      &  524.9588_wp,  343.0012_wp,  296.3749_wp,    0.0000_wp,    0.0000_wp, &
      &  366.5329_wp,  245.6037_wp,  214.8053_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  883.8890_wp,  550.4092_wp,  464.0668_wp,    0.0000_wp,    0.0000_wp, &
      &  500.8355_wp,  327.1995_wp,  282.7396_wp,    0.0000_wp,    0.0000_wp, &
      &  443.7007_wp,  290.8988_wp,  251.7856_wp,    0.0000_wp,    0.0000_wp, &
      &  386.6491_wp,  257.1961_wp,  224.1590_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  919.5354_wp,  568.9088_wp,  478.0372_wp,    0.0000_wp,    0.0000_wp, &
      &  490.5973_wp,  320.0960_wp,  276.4113_wp,    0.0000_wp,    0.0000_wp, &
      &  409.4287_wp,  270.1688_wp,  234.5558_wp,    0.0000_wp,    0.0000_wp, &
      &  438.6502_wp,  285.5945_wp,  246.3077_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  704.3789_wp,  436.1042_wp,  366.6648_wp,    0.0000_wp,    0.0000_wp, &
      &  493.6363_wp,  318.2136_wp,  273.1065_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  701.3549_wp,  442.9886_wp,  376.2399_wp,    0.0000_wp,    0.0000_wp, &
      &  448.0328_wp,  295.2762_wp,  256.2171_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  839.2828_wp,  526.4964_wp,  445.7279_wp,    0.0000_wp,    0.0000_wp, &
      &  780.4805_wp,  492.8316_wp,  418.6209_wp,    0.0000_wp,    0.0000_wp, &
      &  597.4745_wp,  385.9385_wp,  331.6718_wp,    0.0000_wp,    0.0000_wp, &
      &  472.1406_wp,  312.1568_wp,  271.3137_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  711.6823_wp,  458.1920_wp,  393.0366_wp,    0.0000_wp,    0.0000_wp, &
      &  659.8039_wp,  427.3622_wp,  367.7137_wp,    0.0000_wp,    0.0000_wp, &
      &  624.2061_wp,  406.0194_wp,  350.0801_wp,    0.0000_wp,    0.0000_wp, &
      &  550.6011_wp,  361.9295_wp,  313.6909_wp,    0.0000_wp,    0.0000_wp, &
      &  476.4489_wp,  317.4077_wp,  276.8993_wp,    0.0000_wp,    0.0000_wp, &
      &  586.0084_wp,  386.3091_wp,  335.2921_wp,    0.0000_wp,    0.0000_wp, &
      &  574.9009_wp,  379.5097_wp,  329.6139_wp,    0.0000_wp,    0.0000_wp, &
      &  557.7850_wp,  369.1125_wp,  320.9667_wp,    0.0000_wp,    0.0000_wp, &
      &  534.2915_wp,  354.8970_wp,  309.1643_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  514.5077_wp,  344.2425_wp,  300.9258_wp,    0.0000_wp,    0.0000_wp, &
      &  514.8945_wp,  344.4169_wp,  301.0426_wp,    0.0000_wp,    0.0000_wp, &
      &  510.5464_wp,  341.6492_wp,  298.6808_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  440.2999_wp,  299.1535_wp,  263.4034_wp,    0.0000_wp,    0.0000_wp, &
      &  443.8848_wp,  301.3463_wp,  265.2340_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(855 * max_ref * max_ref + 1 : 874 * max_ref * max_ref) = [ &
      &  375.1339_wp,  258.5304_wp,  229.1191_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2673.9344_wp, 1501.7789_wp, 1209.2081_wp,    0.0000_wp,    0.0000_wp, &
      &  832.1368_wp,  522.9449_wp,  443.2682_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2188.2864_wp, 1290.4826_wp, 1057.8748_wp,    0.0000_wp,    0.0000_wp, &
      & 1789.7638_wp, 1052.8629_wp,  865.2462_wp,    0.0000_wp,    0.0000_wp, &
      &  833.7135_wp,  534.0203_wp,  456.8822_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1858.6377_wp, 1118.1402_wp,  926.1439_wp,    0.0000_wp,    0.0000_wp, &
      & 1286.4770_wp,  787.4950_wp,  660.7377_wp,    0.0000_wp,    0.0000_wp, &
      &  781.5377_wp,  509.4181_wp,  439.6548_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1638.4938_wp,  998.1224_wp,  832.1743_wp,    0.0000_wp,    0.0000_wp, &
      &  896.7952_wp,  574.4591_wp,  491.3081_wp,    0.0000_wp,    0.0000_wp, &
      &  782.3088_wp,  512.4995_wp,  443.4182_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1475.2500_wp,  906.0801_wp,  758.7789_wp,    0.0000_wp,    0.0000_wp, &
      &  906.0801_wp,  580.5824_wp,  496.1127_wp,    0.0000_wp,    0.0000_wp, &
      &  758.7789_wp,  496.1127_wp,  428.8962_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   46.0406_wp,   37.5478_wp,   31.6681_wp,    0.0000_wp,    0.0000_wp, &
      &   76.3661_wp,   60.9773_wp,   50.4173_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   29.1617_wp,   24.5530_wp,   21.3516_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  890.5126_wp,  628.1817_wp,  461.5751_wp,    0.0000_wp,    0.0000_wp, &
      &  267.3761_wp,  202.5173_wp,  159.4404_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  463.3065_wp,  349.1897_wp,  273.5446_wp,    0.0000_wp,    0.0000_wp, &
      &  337.9035_wp,  257.6088_wp,  204.1519_wp,    0.0000_wp,    0.0000_wp, &
      &  209.0959_wp,  165.2066_wp,  135.2428_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  295.3592_wp,  231.0449_wp,  187.4666_wp,    0.0000_wp,    0.0000_wp, &
      &  245.8527_wp,  193.7557_wp,  158.3563_wp,    0.0000_wp,    0.0000_wp, &
      &  204.3092_wp,  162.1983_wp,  133.5090_wp,    0.0000_wp,    0.0000_wp, &
      &  150.7447_wp,  122.1941_wp,  102.4883_wp,    0.0000_wp,    0.0000_wp, &
      &  141.2522_wp,  114.8462_wp,   96.6030_wp,    0.0000_wp,    0.0000_wp, &
      &  191.9528_wp,  154.3288_wp,  128.4978_wp,    0.0000_wp,    0.0000_wp, &
      &  180.3390_wp,  144.8738_wp,  120.5946_wp,    0.0000_wp,    0.0000_wp, &
      &  144.6355_wp,  117.5524_wp,   98.8903_wp,    0.0000_wp,    0.0000_wp, &
      &  135.3716_wp,  110.0255_wp,   92.5983_wp,    0.0000_wp,    0.0000_wp, &
      &  108.2356_wp,   89.4229_wp,   76.3869_wp,    0.0000_wp,    0.0000_wp, &
      &  130.6045_wp,  107.1304_wp,   90.9043_wp,    0.0000_wp,    0.0000_wp, &
      &  121.8066_wp,   99.9381_wp,   84.8582_wp,    0.0000_wp,    0.0000_wp, &
      &  114.0004_wp,   93.7434_wp,   79.7672_wp,    0.0000_wp,    0.0000_wp, &
      &  100.1080_wp,   82.5251_wp,   70.4244_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   97.0675_wp,   80.7360_wp,   69.4181_wp,    0.0000_wp,    0.0000_wp, &
      &   86.6803_wp,   72.3520_wp,   62.4324_wp,    0.0000_wp,    0.0000_wp, &
      &   76.0239_wp,   63.7402_wp,   55.2484_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   72.4186_wp,   60.9414_wp,   52.9845_wp,    0.0000_wp,    0.0000_wp, &
      &   57.9784_wp,   49.4166_wp,   43.4760_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   55.0487_wp,   46.7498_wp,   41.0035_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1058.3899_wp,  749.4974_wp,  553.0357_wp,    0.0000_wp,    0.0000_wp, &
      &  395.0058_wp,  297.1713_wp,  232.5932_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  751.6628_wp,  559.8856_wp,  433.8278_wp,    0.0000_wp,    0.0000_wp, &
      &  614.0265_wp,  460.4234_wp,  359.1888_wp,    0.0000_wp,    0.0000_wp, &
      &  379.7626_wp,  295.1816_wp,  238.0169_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  673.1193_wp,  511.0810_wp,  403.2914_wp,    0.0000_wp,    0.0000_wp, &
      &  605.9135_wp,  462.1678_wp,  366.3171_wp,    0.0000_wp,    0.0000_wp, &
      &  461.0586_wp,  357.3576_wp,  287.5225_wp,    0.0000_wp,    0.0000_wp, &
      &  348.3397_wp,  275.6435_wp,  225.9636_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(874 * max_ref * max_ref + 1 : 893 * max_ref * max_ref) = [ &
      &  510.2352_wp,  397.6602_wp,  321.5163_wp,    0.0000_wp,    0.0000_wp, &
      &  469.1092_wp,  366.9549_wp,  297.7436_wp,    0.0000_wp,    0.0000_wp, &
      &  433.9678_wp,  340.7358_wp,  277.4312_wp,    0.0000_wp,    0.0000_wp, &
      &  424.8072_wp,  332.5980_wp,  270.1899_wp,    0.0000_wp,    0.0000_wp, &
      &  340.3478_wp,  271.2960_wp,  223.9586_wp,    0.0000_wp,    0.0000_wp, &
      &  385.5895_wp,  307.1384_wp,  253.4123_wp,    0.0000_wp,    0.0000_wp, &
      &  373.9427_wp,  298.0661_wp,  246.1086_wp,    0.0000_wp,    0.0000_wp, &
      &  369.6794_wp,  294.2436_wp,  242.6676_wp,    0.0000_wp,    0.0000_wp, &
      &  340.0592_wp,  272.2421_wp,  225.7030_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  313.8657_wp,  253.4985_wp,  211.9069_wp,    0.0000_wp,    0.0000_wp, &
      &  310.5861_wp,  250.7573_wp,  209.5599_wp,    0.0000_wp,    0.0000_wp, &
      &  304.2109_wp,  245.6719_wp,  205.3695_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  251.7723_wp,  206.0459_wp,  174.3920_wp,    0.0000_wp,    0.0000_wp, &
      &  249.2671_wp,  203.9397_wp,  172.5758_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  202.8572_wp,  167.9014_wp,  143.6389_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1765.8946_wp, 1237.9643_wp,  905.0848_wp,    0.0000_wp,    0.0000_wp, &
      &  534.6303_wp,  407.5906_wp,  323.6381_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1365.4837_wp,  996.0157_wp,  757.0858_wp,    0.0000_wp,    0.0000_wp, &
      & 1092.6733_wp,  797.4773_wp,  607.0919_wp,    0.0000_wp,    0.0000_wp, &
      &  533.7978_wp,  414.4811_wp,  334.3206_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1112.0098_wp,  818.5161_wp,  627.5560_wp,    0.0000_wp,    0.0000_wp, &
      &  517.7340_wp,  403.1540_wp,  326.0787_wp,    0.0000_wp,    0.0000_wp, &
      &  505.2346_wp,  396.3470_wp,  322.6655_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1060.6255_wp,  787.0565_wp,  608.3000_wp,    0.0000_wp,    0.0000_wp, &
      &  499.5011_wp,  389.1787_wp,  314.9983_wp,    0.0000_wp,    0.0000_wp, &
      &  510.8967_wp,  401.0919_wp,  326.7670_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  963.9434_wp,  718.8469_wp,  558.2324_wp,    0.0000_wp,    0.0000_wp, &
      &  489.8160_wp,  381.4804_wp,  308.6801_wp,    0.0000_wp,    0.0000_wp, &
      &  468.4059_wp,  367.3750_wp,  299.1495_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  758.9955_wp,  566.4300_wp,  440.5516_wp,    0.0000_wp,    0.0000_wp, &
      &  487.1911_wp,  378.4474_wp,  305.5273_wp,    0.0000_wp,    0.0000_wp, &
      &  344.8105_wp,  274.8422_wp,  227.2947_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  820.9313_wp,  616.5993_wp,  482.1362_wp,    0.0000_wp,    0.0000_wp, &
      &  439.9326_wp,  346.0423_wp,  282.4334_wp,    0.0000_wp,    0.0000_wp, &
      &  290.7494_wp,  234.1315_wp,  195.4456_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  643.1247_wp,  483.8537_wp,  379.2688_wp,    0.0000_wp,    0.0000_wp, &
      &  405.0817_wp,  319.7637_wp,  261.8630_wp,    0.0000_wp,    0.0000_wp, &
      &  285.8977_wp,  229.9399_wp,  191.7333_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  670.3490_wp,  509.6999_wp,  403.1484_wp,    0.0000_wp,    0.0000_wp, &
      &  386.5245_wp,  305.0887_wp,  249.8695_wp,    0.0000_wp,    0.0000_wp, &
      &  343.0710_wp,  271.4559_wp,  222.9546_wp,    0.0000_wp,    0.0000_wp, &
      &  300.6753_wp,  240.5139_wp,  199.5325_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  695.8348_wp,  526.3795_wp,  414.3156_wp,    0.0000_wp,    0.0000_wp, &
      &  378.4594_wp,  298.4129_wp,  244.1847_wp,    0.0000_wp,    0.0000_wp, &
      &  317.3472_wp,  252.3281_wp,  208.1570_wp,    0.0000_wp,    0.0000_wp, &
      &  338.3535_wp,  266.2568_wp,  217.6457_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  533.6535_wp,  403.8400_wp,  318.3239_wp,    0.0000_wp,    0.0000_wp, &
      &  379.2299_wp,  296.1989_wp,  240.3673_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  534.5207_wp,  410.9866_wp,  328.4707_wp,    0.0000_wp,    0.0000_wp, &
      &  346.8387_wp,  275.6055_wp,  227.0405_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  637.9233_wp,  487.9858_wp,  388.0446_wp,    0.0000_wp,    0.0000_wp, &
      &  594.5765_wp,  457.1641_wp,  365.2802_wp,    0.0000_wp,    0.0000_wp, &
      &  458.8906_wp,  359.1580_wp,  291.7241_wp,    0.0000_wp,    0.0000_wp, &
      &  365.7390_wp,  291.3944_wp,  240.5109_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  545.7845_wp,  426.0703_wp,  345.1578_wp,    0.0000_wp,    0.0000_wp, &
      &  507.1503_wp,  397.7592_wp,  323.6293_wp,    0.0000_wp,    0.0000_wp, &
      &  480.5552_wp,  378.1231_wp,  308.5781_wp,    0.0000_wp,    0.0000_wp, &
      &  425.5849_wp,  337.5775_wp,  277.5321_wp,    0.0000_wp,    0.0000_wp, &
      &  370.1677_wp,  296.6250_wp,  246.1182_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(893 * max_ref * max_ref + 1 : 912 * max_ref * max_ref) = [ &
      &  453.4186_wp,  360.4496_wp,  296.9165_wp,    0.0000_wp,    0.0000_wp, &
      &  445.0799_wp,  354.1868_wp,  292.0475_wp,    0.0000_wp,    0.0000_wp, &
      &  432.2496_wp,  344.6133_wp,  284.6411_wp,    0.0000_wp,    0.0000_wp, &
      &  414.6509_wp,  331.5230_wp,  274.5376_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  400.4998_wp,  321.9415_wp,  267.9623_wp,    0.0000_wp,    0.0000_wp, &
      &  400.7722_wp,  322.0978_wp,  268.0526_wp,    0.0000_wp,    0.0000_wp, &
      &  397.4644_wp,  319.5340_wp,  265.9986_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  344.9826_wp,  280.4702_wp,  235.9264_wp,    0.0000_wp,    0.0000_wp, &
      &  347.6800_wp,  282.4933_wp,  237.5019_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  295.7994_wp,  242.9684_wp,  206.3637_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1969.4796_wp, 1385.1228_wp, 1016.3718_wp,    0.0000_wp,    0.0000_wp, &
      &  633.9471_wp,  485.4129_wp,  387.1236_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1631.5424_wp, 1188.1982_wp,  902.1097_wp,    0.0000_wp,    0.0000_wp, &
      & 1335.0700_wp,  972.1233_wp,  738.6401_wp,    0.0000_wp,    0.0000_wp, &
      &  638.7768_wp,  496.5128_wp,  401.0541_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1394.0976_wp, 1031.4745_wp,  795.1764_wp,    0.0000_wp,    0.0000_wp, &
      &  971.5959_wp,  730.1446_wp,  571.8530_wp,    0.0000_wp,    0.0000_wp, &
      &  602.4253_wp,  474.6807_wp,  388.0718_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1233.9095_wp,  922.0660_wp,  717.6632_wp,    0.0000_wp,    0.0000_wp, &
      &  687.1848_wp,  534.0108_wp,  431.3512_wp,    0.0000_wp,    0.0000_wp, &
      &  604.1475_wp,  477.8870_wp,  392.0655_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1114.0924_wp,  837.9849_wp,  656.3774_wp,    0.0000_wp,    0.0000_wp, &
      &  694.2983_wp,  539.3532_wp,  435.5692_wp,    0.0000_wp,    0.0000_wp, &
      &  585.8126_wp,  462.6546_wp,  379.2075_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  845.8972_wp,  643.4721_wp,  509.7424_wp,    0.0000_wp,    0.0000_wp, &
      &  643.4721_wp,  502.0073_wp,  407.0659_wp,    0.0000_wp,    0.0000_wp, &
      &  509.7424_wp,  407.0659_wp,  337.1075_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   51.0527_wp,   41.3385_wp,   28.2627_wp,    0.0000_wp,    0.0000_wp, &
      &   85.0800_wp,   67.8924_wp,   44.5963_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   32.0588_wp,   26.6217_wp,   19.3556_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1011.6099_wp,  763.7814_wp,  391.4255_wp,    0.0000_wp,    0.0000_wp, &
      &  300.6533_wp,  232.9539_wp,  138.3613_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  521.4249_wp,  402.7727_wp,  236.8457_wp,    0.0000_wp,    0.0000_wp, &
      &  379.5254_wp,  295.3816_wp,  177.6837_wp,    0.0000_wp,    0.0000_wp, &
      &  233.4606_wp,  185.0244_wp,  119.1206_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  330.3257_wp,  260.3796_wp,  164.5777_wp,    0.0000_wp,    0.0000_wp, &
      &  274.5334_wp,  217.4762_wp,  139.4763_wp,    0.0000_wp,    0.0000_wp, &
      &  227.7959_wp,  181.3885_wp,  117.9758_wp,    0.0000_wp,    0.0000_wp, &
      &  167.3783_wp,  134.9705_wp,   91.2518_wp,    0.0000_wp,    0.0000_wp, &
      &  156.7256_wp,  126.6571_wp,   86.1276_wp,    0.0000_wp,    0.0000_wp, &
      &  213.4799_wp,  171.2581_wp,  114.0774_wp,    0.0000_wp,    0.0000_wp, &
      &  200.5464_wp,  160.9223_wp,  107.0906_wp,    0.0000_wp,    0.0000_wp, &
      &  160.4520_wp,  129.7079_wp,   88.2158_wp,    0.0000_wp,    0.0000_wp, &
      &  150.1398_wp,  121.4381_wp,   82.6496_wp,    0.0000_wp,    0.0000_wp, &
      &  119.5995_wp,   97.8417_wp,   68.6237_wp,    0.0000_wp,    0.0000_wp, &
      &  144.5697_wp,  117.6479_wp,   81.4217_wp,    0.0000_wp,    0.0000_wp, &
      &  134.7887_wp,  109.7755_wp,   76.0532_wp,    0.0000_wp,    0.0000_wp, &
      &  126.0801_wp,  102.8506_wp,   71.5723_wp,    0.0000_wp,    0.0000_wp, &
      &  110.6119_wp,   90.4660_wp,   63.3081_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  107.0588_wp,   88.0609_wp,   62.5886_wp,    0.0000_wp,    0.0000_wp, &
      &   95.4966_wp,   78.7920_wp,   56.3962_wp,    0.0000_wp,    0.0000_wp, &
      &   83.6409_wp,   69.2838_wp,   50.0406_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   79.6121_wp,   66.1093_wp,   48.0540_wp,    0.0000_wp,    0.0000_wp, &
      &   63.5079_wp,   53.2880_wp,   39.6668_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(912 * max_ref * max_ref + 1 : 931 * max_ref * max_ref) = [ &
      &   60.3521_wp,   50.5087_wp,   37.3723_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1201.5234_wp,  908.4025_wp,  469.7696_wp,    0.0000_wp,    0.0000_wp, &
      &  444.5269_wp,  343.5592_wp,  201.5403_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  847.3497_wp,  651.2564_wp,  374.2691_wp,    0.0000_wp,    0.0000_wp, &
      &  691.4497_wp,  533.7085_wp,  310.8083_wp,    0.0000_wp,    0.0000_wp, &
      &  425.2656_wp,  333.8166_wp,  208.4131_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  756.6195_wp,  586.9862_wp,  350.2804_wp,    0.0000_wp,    0.0000_wp, &
      &  680.5427_wp,  529.3043_wp,  318.7388_wp,    0.0000_wp,    0.0000_wp, &
      &  516.4766_wp,  405.2011_wp,  251.6421_wp,    0.0000_wp,    0.0000_wp, &
      &  388.8226_wp,  308.4359_wp,  199.1588_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  571.0501_wp,  449.1191_wp,  281.8480_wp,    0.0000_wp,    0.0000_wp, &
      &  524.6479_wp,  413.5991_wp,  261.4105_wp,    0.0000_wp,    0.0000_wp, &
      &  485.0098_wp,  383.1734_wp,  243.9188_wp,    0.0000_wp,    0.0000_wp, &
      &  474.9418_wp,  374.8284_wp,  237.4146_wp,    0.0000_wp,    0.0000_wp, &
      &  379.3148_wp,  302.3644_wp,  197.9963_wp,    0.0000_wp,    0.0000_wp, &
      &  429.7677_wp,  342.4883_wp,  224.0034_wp,    0.0000_wp,    0.0000_wp, &
      &  416.7080_wp,  332.2783_wp,  217.6347_wp,    0.0000_wp,    0.0000_wp, &
      &  412.0413_wp,  328.3317_wp,  214.5176_wp,    0.0000_wp,    0.0000_wp, &
      &  378.6058_wp,  302.7493_wp,  199.9418_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  348.7744_wp,  280.5613_wp,  188.4044_wp,    0.0000_wp,    0.0000_wp, &
      &  345.1434_wp,  277.6002_wp,  186.3093_wp,    0.0000_wp,    0.0000_wp, &
      &  338.0296_wp,  271.9434_wp,  182.6170_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  278.9123_wp,  226.4833_wp,  155.9405_wp,    0.0000_wp,    0.0000_wp, &
      &  276.1455_wp,  224.2095_wp,  154.3118_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  224.0794_wp,  183.5205_wp,  129.1105_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2007.6137_wp, 1518.8755_wp,  767.9339_wp,    0.0000_wp,    0.0000_wp, &
      &  599.9302_wp,  468.0977_wp,  282.4384_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1543.4258_wp, 1178.3302_wp,  649.5000_wp,    0.0000_wp,    0.0000_wp, &
      & 1234.9316_wp,  945.9943_wp,  521.8458_wp,    0.0000_wp,    0.0000_wp, &
      &  597.4915_wp,  469.5752_wp,  293.1198_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1255.3446_wp,  961.3657_wp,  539.8662_wp,    0.0000_wp,    0.0000_wp, &
      &  579.1623_wp,  455.9109_wp,  286.2241_wp,    0.0000_wp,    0.0000_wp, &
      &  564.5560_wp,  446.1492_wp,  283.8784_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1195.8434_wp,  919.3099_wp,  524.9173_wp,    0.0000_wp,    0.0000_wp, &
      &  558.7211_wp,  440.2561_wp,  276.6567_wp,    0.0000_wp,    0.0000_wp, &
      &  570.7922_wp,  451.2866_wp,  287.5807_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1086.0231_wp,  836.7341_wp,  482.5622_wp,    0.0000_wp,    0.0000_wp, &
      &  547.8906_wp,  431.6391_wp,  271.1001_wp,    0.0000_wp,    0.0000_wp, &
      &  523.3223_wp,  413.8230_wp,  263.3409_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  854.7474_wp,  659.7201_wp,  381.4006_wp,    0.0000_wp,    0.0000_wp, &
      &  545.1724_wp,  428.9662_wp,  268.1325_wp,    0.0000_wp,    0.0000_wp, &
      &  383.9186_wp,  306.7585_wp,  201.4707_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  923.8743_wp,  714.1830_wp,  417.8614_wp,    0.0000_wp,    0.0000_wp, &
      &  491.3210_wp,  388.9066_wp,  248.7895_wp,    0.0000_wp,    0.0000_wp, &
      &  323.0552_wp,  259.8051_wp,  173.9292_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  723.3202_wp,  560.3648_wp,  329.3116_wp,    0.0000_wp,    0.0000_wp, &
      &  452.0831_wp,  358.6562_wp,  231.0081_wp,    0.0000_wp,    0.0000_wp, &
      &  317.7432_wp,  255.3330_wp,  170.5557_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  752.9678_wp,  585.2935_wp,  350.8564_wp,    0.0000_wp,    0.0000_wp, &
      &  431.3478_wp,  342.2979_wp,  220.4802_wp,    0.0000_wp,    0.0000_wp, &
      &  382.5646_wp,  304.2749_wp,  197.0680_wp,    0.0000_wp,    0.0000_wp, &
      &  334.5583_wp,  267.8666_wp,  177.1019_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  782.2483_wp,  606.4740_wp,  359.9017_wp,    0.0000_wp,    0.0000_wp, &
      &  422.4144_wp,  335.0287_wp,  215.4030_wp,    0.0000_wp,    0.0000_wp, &
      &  353.5546_wp,  281.9707_wp,  184.3086_wp,    0.0000_wp,    0.0000_wp, &
      &  377.6302_wp,  299.4461_wp,  192.0591_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(931 * max_ref * max_ref + 1 : 950 * max_ref * max_ref) = [ &
      &  599.6192_wp,  465.8147_wp,  276.9970_wp,    0.0000_wp,    0.0000_wp, &
      &  423.9154_wp,  334.5615_wp,  211.4238_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  599.3141_wp,  468.4529_wp,  286.9927_wp,    0.0000_wp,    0.0000_wp, &
      &  386.6256_wp,  307.8815_wp,  200.7562_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  715.9811_wp,  558.1522_wp,  338.2627_wp,    0.0000_wp,    0.0000_wp, &
      &  666.7601_wp,  521.1441_wp,  318.9968_wp,    0.0000_wp,    0.0000_wp, &
      &  513.0410_wp,  404.9175_wp,  256.3933_wp,    0.0000_wp,    0.0000_wp, &
      &  407.5902_wp,  324.9095_wp,  212.6937_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  610.5327_wp,  480.8926_wp,  302.9389_wp,    0.0000_wp,    0.0000_wp, &
      &  566.8294_wp,  447.7131_wp,  284.5505_wp,    0.0000_wp,    0.0000_wp, &
      &  536.7795_wp,  424.7791_wp,  271.6474_wp,    0.0000_wp,    0.0000_wp, &
      &  474.6638_wp,  377.4276_wp,  245.0471_wp,    0.0000_wp,    0.0000_wp, &
      &  412.0594_wp,  329.6696_wp,  218.1156_wp,    0.0000_wp,    0.0000_wp, &
      &  505.5091_wp,  402.4540_wp,  262.3480_wp,    0.0000_wp,    0.0000_wp, &
      &  496.1023_wp,  395.2472_wp,  258.1617_wp,    0.0000_wp,    0.0000_wp, &
      &  481.6229_wp,  384.1644_wp,  251.7973_wp,    0.0000_wp,    0.0000_wp, &
      &  461.7577_wp,  368.9607_wp,  243.1165_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  445.4778_wp,  357.2511_wp,  237.8235_wp,    0.0000_wp,    0.0000_wp, &
      &  445.7916_wp,  357.4737_wp,  237.8953_wp,    0.0000_wp,    0.0000_wp, &
      &  442.0786_wp,  354.5768_wp,  236.1084_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  382.7591_wp,  309.3530_wp,  210.3812_wp,    0.0000_wp,    0.0000_wp, &
      &  385.7986_wp,  311.6901_wp,  211.7410_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  327.3827_wp,  266.5777_wp,  184.8477_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2237.9665_wp, 1696.1214_wp,  863.7655_wp,    0.0000_wp,    0.0000_wp, &
      &  710.7779_wp,  556.1255_wp,  338.5014_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1844.3774_wp, 1408.3895_wp,  773.9463_wp,    0.0000_wp,    0.0000_wp, &
      & 1509.2981_wp, 1156.2604_wp,  634.8168_wp,    0.0000_wp,    0.0000_wp, &
      &  714.7188_wp,  562.4313_wp,  351.9564_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1572.3759_wp, 1207.9868_wp,  685.7489_wp,    0.0000_wp,    0.0000_wp, &
      & 1093.2424_wp,  848.3727_wp,  496.6399_wp,    0.0000_wp,    0.0000_wp, &
      &  672.5553_wp,  532.9719_wp,  342.0346_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1389.5979_wp, 1072.1935_wp,  621.0438_wp,    0.0000_wp,    0.0000_wp, &
      &  768.8428_wp,  605.3389_wp,  378.6511_wp,    0.0000_wp,    0.0000_wp, &
      &  674.0068_wp,  535.3153_wp,  346.0266_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1253.3465_wp,  970.2070_wp,  569.4159_wp,    0.0000_wp,    0.0000_wp, &
      &  776.7368_wp,  611.2764_wp,  382.3225_wp,    0.0000_wp,    0.0000_wp, &
      &  653.6028_wp,  519.0845_wp,  334.7174_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  949.6490_wp,  739.7915_wp,  444.3471_wp,    0.0000_wp,    0.0000_wp, &
      &  719.3659_wp,  567.7670_wp,  357.9634_wp,    0.0000_wp,    0.0000_wp, &
      &  567.4441_wp,  453.7197_wp,  298.8508_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1067.0169_wp,  829.2408_wp,  493.7430_wp,    0.0000_wp,    0.0000_wp, &
      &  829.2408_wp,  649.7143_wp,  397.0367_wp,    0.0000_wp,    0.0000_wp, &
      &  493.7430_wp,  397.0367_wp,  265.8654_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   39.5745_wp,   33.1875_wp,   26.8142_wp,    0.0000_wp,    0.0000_wp, &
      &   65.0863_wp,   53.3648_wp,   42.1627_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   25.4622_wp,   22.0616_wp,   18.4864_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  730.5533_wp,  517.3330_wp,  364.4102_wp,    0.0000_wp,    0.0000_wp, &
      &  223.8871_wp,  172.9708_wp,  129.8907_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  387.2681_wp,  297.5186_wp,  222.1489_wp,    0.0000_wp,    0.0000_wp, &
      &  283.5695_wp,  220.6746_wp,  166.9872_wp,    0.0000_wp,    0.0000_wp, &
      &  177.5174_wp,  143.8851_wp,  112.4391_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(950 * max_ref * max_ref + 1 : 969 * max_ref * max_ref) = [ &
      &  249.9382_wp,  200.3299_wp,  155.1628_wp,    0.0000_wp,    0.0000_wp, &
      &  208.6362_wp,  168.5944_wp,  131.6650_wp,    0.0000_wp,    0.0000_wp, &
      &  173.8767_wp,  141.6201_wp,  111.5108_wp,    0.0000_wp,    0.0000_wp, &
      &  129.2726_wp,  107.7128_wp,   86.4986_wp,    0.0000_wp,    0.0000_wp, &
      &  121.2876_wp,  101.3825_wp,   81.6853_wp,    0.0000_wp,    0.0000_wp, &
      &  164.1279_wp,  135.5517_wp,  108.0200_wp,    0.0000_wp,    0.0000_wp, &
      &  154.1938_wp,  127.2125_wp,  101.4184_wp,    0.0000_wp,    0.0000_wp, &
      &  124.2389_wp,  103.7956_wp,   83.6946_wp,    0.0000_wp,    0.0000_wp, &
      &  116.3247_wp,   97.1744_wp,   78.4368_wp,    0.0000_wp,    0.0000_wp, &
      &   93.6269_wp,   79.5750_wp,   65.2905_wp,    0.0000_wp,    0.0000_wp, &
      &  112.6378_wp,   95.0190_wp,   77.3772_wp,    0.0000_wp,    0.0000_wp, &
      &  105.0905_wp,   88.6613_wp,   72.2950_wp,    0.0000_wp,    0.0000_wp, &
      &   98.4728_wp,   83.2744_wp,   68.0720_wp,    0.0000_wp,    0.0000_wp, &
      &   86.6163_wp,   73.4259_wp,   60.2641_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   84.2725_wp,   72.1173_wp,   59.6460_wp,    0.0000_wp,    0.0000_wp, &
      &   75.3910_wp,   64.7448_wp,   53.7863_wp,    0.0000_wp,    0.0000_wp, &
      &   66.3025_wp,   57.1928_wp,   47.7847_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   63.2599_wp,   54.7781_wp,   45.9133_wp,    0.0000_wp,    0.0000_wp, &
      &   50.9770_wp,   44.7087_wp,   37.9961_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   48.3429_wp,   42.2432_wp,   35.7884_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  869.3774_wp,  618.6669_wp,  437.6491_wp,    0.0000_wp,    0.0000_wp, &
      &  330.2273_wp,  253.0900_wp,  189.1300_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  626.2104_wp,  474.4180_wp,  350.6141_wp,    0.0000_wp,    0.0000_wp, &
      &  512.6759_wp,  391.3606_wp,  291.4916_wp,    0.0000_wp,    0.0000_wp, &
      &  320.6678_wp,  255.2289_wp,  196.3102_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  564.0716_wp,  437.0209_wp,  328.9434_wp,    0.0000_wp,    0.0000_wp, &
      &  508.5289_wp,  396.0634_wp,  299.5256_wp,    0.0000_wp,    0.0000_wp, &
      &  388.9956_wp,  308.5459_wp,  236.9854_wp,    0.0000_wp,    0.0000_wp, &
      &  295.9256_wp,  240.2687_wp,  188.0446_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  431.2132_wp,  344.2113_wp,  265.5808_wp,    0.0000_wp,    0.0000_wp, &
      &  396.9862_wp,  318.1795_wp,  246.4680_wp,    0.0000_wp,    0.0000_wp, &
      &  367.7306_wp,  295.9631_wp,  230.0988_wp,    0.0000_wp,    0.0000_wp, &
      &  359.6929_wp,  288.5406_wp,  223.9239_wp,    0.0000_wp,    0.0000_wp, &
      &  289.9361_wp,  237.2798_wp,  187.1659_wp,    0.0000_wp,    0.0000_wp, &
      &  328.4072_wp,  268.5416_wp,  211.7390_wp,    0.0000_wp,    0.0000_wp, &
      &  318.5912_wp,  260.7012_wp,  205.7533_wp,    0.0000_wp,    0.0000_wp, &
      &  314.8297_wp,  257.2059_wp,  202.7848_wp,    0.0000_wp,    0.0000_wp, &
      &  290.2077_wp,  238.6014_wp,  189.1560_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  268.7824_wp,  223.0883_wp,  178.4935_wp,    0.0000_wp,    0.0000_wp, &
      &  265.9528_wp,  220.6477_wp,  176.5078_wp,    0.0000_wp,    0.0000_wp, &
      &  260.5335_wp,  216.2066_wp,  173.0241_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  216.8070_wp,  182.4721_wp,  148.0763_wp,    0.0000_wp,    0.0000_wp, &
      &  214.6380_wp,  180.5916_wp,  146.5298_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  175.5876_wp,  149.5222_wp,  122.8625_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1447.3048_wp, 1015.7997_wp,  715.2376_wp,    0.0000_wp,    0.0000_wp, &
      &  449.3030_wp,  349.4753_wp,  265.8033_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1131.1769_wp,  835.2692_wp,  607.2921_wp,    0.0000_wp,    0.0000_wp, &
      &  905.7515_wp,  668.7646_wp,  488.3352_wp,    0.0000_wp,    0.0000_wp, &
      &  450.9597_wp,  358.3732_wp,  276.2842_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  923.5811_wp,  689.6088_wp,  505.2854_wp,    0.0000_wp,    0.0000_wp, &
      &  437.8432_wp,  349.0702_wp,  269.9059_wp,    0.0000_wp,    0.0000_wp, &
      &  428.2629_wp,  344.2948_wp,  267.9088_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  883.1492_wp,  665.8143_wp,  491.8643_wp,    0.0000_wp,    0.0000_wp, &
      &  422.5708_wp,  337.0553_wp,  260.9491_wp,    0.0000_wp,    0.0000_wp, &
      &  433.1914_wp,  348.5490_wp,  271.4387_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  803.8807_wp,  609.6159_wp,  452.4755_wp,    0.0000_wp,    0.0000_wp, &
      &  414.3575_wp,  330.3567_wp,  255.7125_wp,    0.0000_wp,    0.0000_wp, &
      &  397.1558_wp,  319.1584_wp,  248.6021_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(969 * max_ref * max_ref + 1 : 988 * max_ref * max_ref) = [ &
      &  633.4394_wp,  480.6382_wp,  357.8667_wp,    0.0000_wp,    0.0000_wp, &
      &  411.8187_wp,  327.3553_wp,  252.8515_wp,    0.0000_wp,    0.0000_wp, &
      &  294.2258_wp,  240.6605_wp,  190.7170_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  686.1772_wp,  524.7748_wp,  392.1922_wp,    0.0000_wp,    0.0000_wp, &
      &  373.3442_wp,  301.0397_wp,  234.9229_wp,    0.0000_wp,    0.0000_wp, &
      &  249.0737_wp,  206.0020_wp,  164.9015_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  538.1435_wp,  412.2468_wp,  309.3380_wp,    0.0000_wp,    0.0000_wp, &
      &  344.2322_wp,  278.6526_wp,  218.2588_wp,    0.0000_wp,    0.0000_wp, &
      &  244.8184_wp,  202.2130_wp,  161.6818_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  562.4800_wp,  436.4054_wp,  329.8121_wp,    0.0000_wp,    0.0000_wp, &
      &  328.5015_wp,  265.8752_wp,  208.3381_wp,    0.0000_wp,    0.0000_wp, &
      &  291.9635_wp,  236.9046_wp,  186.3559_wp,    0.0000_wp,    0.0000_wp, &
      &  256.9295_wp,  210.9797_wp,  167.7448_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  582.9014_wp,  449.5806_wp,  338.0805_wp,    0.0000_wp,    0.0000_wp, &
      &  321.5506_wp,  259.9472_wp,  203.5228_wp,    0.0000_wp,    0.0000_wp, &
      &  270.5502_wp,  220.7203_wp,  174.4070_wp,    0.0000_wp,    0.0000_wp, &
      &  287.4614_wp,  231.8285_wp,  181.5196_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  447.4259_wp,  345.1043_wp,  260.4199_wp,    0.0000_wp,    0.0000_wp, &
      &  321.2502_wp,  256.9480_wp,  199.5641_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  450.1543_wp,  353.7880_wp,  270.1794_wp,    0.0000_wp,    0.0000_wp, &
      &  295.4307_wp,  240.9124_wp,  189.8458_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  536.1093_wp,  418.8573_wp,  318.1224_wp,    0.0000_wp,    0.0000_wp, &
      &  500.5162_wp,  393.3571_wp,  300.2053_wp,    0.0000_wp,    0.0000_wp, &
      &  388.6043_wp,  311.5991_wp,  241.8646_wp,    0.0000_wp,    0.0000_wp, &
      &  311.6254_wp,  254.8812_wp,  201.1040_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  461.6795_wp,  369.1734_wp,  285.6102_wp,    0.0000_wp,    0.0000_wp, &
      &  429.6996_wp,  345.3872_wp,  268.4538_wp,    0.0000_wp,    0.0000_wp, &
      &  407.6339_wp,  328.8341_wp,  256.3994_wp,    0.0000_wp,    0.0000_wp, &
      &  362.0366_wp,  294.6634_wp,  231.5527_wp,    0.0000_wp,    0.0000_wp, &
      &  316.0442_wp,  260.1176_wp,  206.3914_wp,    0.0000_wp,    0.0000_wp, &
      &  385.9883_wp,  314.9253_wp,  247.9607_wp,    0.0000_wp,    0.0000_wp, &
      &  379.0427_wp,  309.6035_wp,  244.0467_wp,    0.0000_wp,    0.0000_wp, &
      &  368.3695_wp,  301.4924_wp,  238.0966_wp,    0.0000_wp,    0.0000_wp, &
      &  353.7383_wp,  290.4181_wp,  229.9806_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  342.3814_wp,  282.7302_wp,  225.1670_wp,    0.0000_wp,    0.0000_wp, &
      &  342.5970_wp,  282.8460_wp,  225.2330_wp,    0.0000_wp,    0.0000_wp, &
      &  339.8157_wp,  280.6374_wp,  223.5555_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  296.2668_wp,  247.6203_wp,  199.5549_wp,    0.0000_wp,    0.0000_wp, &
      &  298.5174_wp,  249.3404_wp,  200.8289_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  255.1516_wp,  215.5683_wp,  175.6524_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1615.8387_wp, 1138.5265_wp,  805.0065_wp,    0.0000_wp,    0.0000_wp, &
      &  533.6135_wp,  417.0831_wp,  318.8041_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1351.1988_wp,  995.6116_wp,  723.6911_wp,    0.0000_wp,    0.0000_wp, &
      & 1106.1259_wp,  814.1975_wp,  594.0387_wp,    0.0000_wp,    0.0000_wp, &
      &  539.9639_wp,  429.5383_wp,  331.8687_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1159.9071_wp,  871.2600_wp,  642.4279_wp,    0.0000_wp,    0.0000_wp, &
      &  812.6286_wp,  621.3058_wp,  466.5106_wp,    0.0000_wp,    0.0000_wp, &
      &  511.4722_wp,  413.1903_wp,  323.0124_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1029.7665_wp,  782.7166_wp,  582.5544_wp,    0.0000_wp,    0.0000_wp, &
      &  580.9007_wp,  461.9141_wp,  357.0819_wp,    0.0000_wp,    0.0000_wp, &
      &  513.6197_wp,  416.7152_wp,  326.9477_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  931.7445_wp,  713.6477_wp,  534.6266_wp,    0.0000_wp,    0.0000_wp, &
      &  586.8945_wp,  466.5358_wp,  360.5430_wp,    0.0000_wp,    0.0000_wp, &
      &  497.9286_wp,  403.2032_wp,  316.2990_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  710.2837_wp,  551.1240_wp,  417.9889_wp,    0.0000_wp,    0.0000_wp, &
      &  544.8075_wp,  435.1062_wp,  337.8186_wp,    0.0000_wp,    0.0000_wp, &
      &  435.0888_wp,  356.6338_wp,  282.8832_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(988 * max_ref * max_ref + 1 : 1007 * max_ref * max_ref) = [ &
      &  796.1951_wp,  614.8185_wp,  464.1239_wp,    0.0000_wp,    0.0000_wp, &
      &  623.1386_wp,  488.3264_wp,  374.0345_wp,    0.0000_wp,    0.0000_wp, &
      &  380.5730_wp,  314.9171_wp,  252.0075_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  598.1988_wp,  468.5695_wp,  358.4840_wp,    0.0000_wp,    0.0000_wp, &
      &  468.5695_wp,  379.2655_wp,  297.6676_wp,    0.0000_wp,    0.0000_wp, &
      &  358.4840_wp,  297.6676_wp,  239.0071_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   43.1661_wp,   31.1024_wp,   26.9803_wp,    0.0000_wp,    0.0000_wp, &
      &   71.1985_wp,   49.9090_wp,   42.7714_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   27.5745_wp,   20.7776_wp,   18.4026_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  794.2006_wp,  482.4342_wp,  390.2766_wp,    0.0000_wp,    0.0000_wp, &
      &  245.2561_wp,  161.3499_wp,  134.6931_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  424.4120_wp,  277.4174_wp,  230.8996_wp,    0.0000_wp,    0.0000_wp, &
      &  310.3781_wp,  205.9530_wp,  172.6021_wp,    0.0000_wp,    0.0000_wp, &
      &  194.3984_wp,  134.4576_wp,  114.5587_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  273.7195_wp,  187.1327_wp,  158.7124_wp,    0.0000_wp,    0.0000_wp, &
      &  228.2822_wp,  157.5949_wp,  134.2505_wp,    0.0000_wp,    0.0000_wp, &
      &  190.0445_wp,  132.4776_wp,  113.3508_wp,    0.0000_wp,    0.0000_wp, &
      &  141.1109_wp,  100.8958_wp,   87.2376_wp,    0.0000_wp,    0.0000_wp, &
      &  132.3281_wp,   94.9987_wp,   82.2887_wp,    0.0000_wp,    0.0000_wp, &
      &  179.2708_wp,  126.9085_wp,  109.2841_wp,    0.0000_wp,    0.0000_wp, &
      &  168.3115_wp,  119.1265_wp,  102.6155_wp,    0.0000_wp,    0.0000_wp, &
      &  135.4760_wp,   97.2961_wp,   84.3298_wp,    0.0000_wp,    0.0000_wp, &
      &  126.7778_wp,   91.1171_wp,   79.0279_wp,    0.0000_wp,    0.0000_wp, &
      &  101.8194_wp,   74.7287_wp,   65.3949_wp,    0.0000_wp,    0.0000_wp, &
      &  122.6426_wp,   89.1659_wp,   77.7074_wp,    0.0000_wp,    0.0000_wp, &
      &  114.3496_wp,   83.2227_wp,   72.5863_wp,    0.0000_wp,    0.0000_wp, &
      &  107.1084_wp,   78.1976_wp,   68.3029_wp,    0.0000_wp,    0.0000_wp, &
      &   94.1007_wp,   69.0005_wp,   60.4149_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   91.4959_wp,   67.8114_wp,   59.6194_wp,    0.0000_wp,    0.0000_wp, &
      &   81.7640_wp,   60.9159_wp,   53.6934_wp,    0.0000_wp,    0.0000_wp, &
      &   71.8071_wp,   53.8670_wp,   47.6422_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   68.4888_wp,   51.6117_wp,   45.7289_wp,    0.0000_wp,    0.0000_wp, &
      &   55.0240_wp,   42.2084_wp,   37.6967_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   52.2072_wp,   39.8762_wp,   35.5563_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  945.4221_wp,  577.0181_wp,  467.8086_wp,    0.0000_wp,    0.0000_wp, &
      &  361.4287_wp,  236.1444_wp,  196.6926_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  685.6628_wp,  442.3381_wp,  366.3245_wp,    0.0000_wp,    0.0000_wp, &
      &  561.0010_wp,  365.0705_wp,  303.5312_wp,    0.0000_wp,    0.0000_wp, &
      &  351.4302_wp,  238.3187_wp,  201.3988_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  617.9244_wp,  407.6999_wp,  340.7455_wp,    0.0000_wp,    0.0000_wp, &
      &  557.0043_wp,  369.5814_wp,  309.6367_wp,    0.0000_wp,    0.0000_wp, &
      &  425.9837_wp,  288.1291_wp,  243.3210_wp,    0.0000_wp,    0.0000_wp, &
      &  324.0408_wp,  224.5657_wp,  191.4938_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  472.3738_wp,  321.4594_wp,  272.0913_wp,    0.0000_wp,    0.0000_wp, &
      &  434.7251_wp,  297.2339_wp,  252.1101_wp,    0.0000_wp,    0.0000_wp, &
      &  402.6239_wp,  276.5427_wp,  235.0108_wp,    0.0000_wp,    0.0000_wp, &
      &  393.6613_wp,  269.6241_wp,  228.9329_wp,    0.0000_wp,    0.0000_wp, &
      &  317.2064_wp,  221.9100_wp,  190.0213_wp,    0.0000_wp,    0.0000_wp, &
      &  359.2547_wp,  251.1477_wp,  215.0145_wp,    0.0000_wp,    0.0000_wp, &
      &  348.4449_wp,  243.8439_wp,  208.8723_wp,    0.0000_wp,    0.0000_wp, &
      &  344.2987_wp,  240.5751_wp,  205.9678_wp,    0.0000_wp,    0.0000_wp, &
      &  317.2629_wp,  223.2549_wp,  191.6962_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  293.5214_wp,  208.9067_wp,  180.2700_wp,    0.0000_wp,    0.0000_wp, &
      &  290.4146_wp,  206.6258_wp,  178.2877_wp,    0.0000_wp,    0.0000_wp, &
      &  284.4682_wp,  202.4803_wp,  174.7517_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  236.2750_wp,  171.1187_wp,  148.8148_wp,    0.0000_wp,    0.0000_wp, &
      &  233.9031_wp,  169.3584_wp,  147.2755_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1007 * max_ref * max_ref + 1 : 1026 * max_ref * max_ref) = [ &
      &  190.9352_wp,  140.4273_wp,  122.9822_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1567.4954_wp,  948.5140_wp,  767.4205_wp,    0.0000_wp,    0.0000_wp, &
      &  490.5807_wp,  326.5947_wp,  274.5928_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1234.6013_wp,  779.0547_wp,  640.1431_wp,    0.0000_wp,    0.0000_wp, &
      &  986.3727_wp,  624.2850_wp,  514.1385_wp,    0.0000_wp,    0.0000_wp, &
      &  493.3926_wp,  334.8827_wp,  283.4399_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1009.1933_wp,  643.1893_wp,  530.5111_wp,    0.0000_wp,    0.0000_wp, &
      &  478.9598_wp,  326.2503_wp,  276.5669_wp,    0.0000_wp,    0.0000_wp, &
      &  468.5358_wp,  321.8583_wp,  273.7179_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  965.2308_wp,  621.1807_wp,  514.4744_wp,    0.0000_wp,    0.0000_wp, &
      &  462.0314_wp,  315.0982_wp,  267.2985_wp,    0.0000_wp,    0.0000_wp, &
      &  473.8973_wp,  325.8582_wp,  277.2406_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  878.8027_wp,  568.8348_wp,  472.2401_wp,    0.0000_wp,    0.0000_wp, &
      &  453.0262_wp,  308.8482_wp,  261.9807_wp,    0.0000_wp,    0.0000_wp, &
      &  434.2176_wp,  298.4577_wp,  253.9853_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  691.6314_wp,  448.7527_wp,  373.2232_wp,    0.0000_wp,    0.0000_wp, &
      &  450.1943_wp,  306.0340_wp,  259.3191_wp,    0.0000_wp,    0.0000_wp, &
      &  321.1384_wp,  225.3888_wp,  193.5970_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  750.3251_wp,  489.7933_wp,  408.0411_wp,    0.0000_wp,    0.0000_wp, &
      &  408.3677_wp,  281.5134_wp,  239.7905_wp,    0.0000_wp,    0.0000_wp, &
      &  271.6188_wp,  193.0894_wp,  166.7533_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  587.7544_wp,  385.0153_wp,  321.4850_wp,    0.0000_wp,    0.0000_wp, &
      &  376.4045_wp,  260.6567_wp,  222.4640_wp,    0.0000_wp,    0.0000_wp, &
      &  267.0048_wp,  189.5285_wp,  163.5813_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  615.4364_wp,  407.4574_wp,  341.3760_wp,    0.0000_wp,    0.0000_wp, &
      &  359.1057_wp,  248.7388_wp,  212.3500_wp,    0.0000_wp,    0.0000_wp, &
      &  318.8732_wp,  221.7615_wp,  189.7380_wp,    0.0000_wp,    0.0000_wp, &
      &  280.3953_wp,  197.6518_wp,  170.0773_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  637.7634_wp,  419.6709_wp,  350.7171_wp,    0.0000_wp,    0.0000_wp, &
      &  351.4986_wp,  243.1909_wp,  207.5282_wp,    0.0000_wp,    0.0000_wp, &
      &  295.4451_wp,  206.6729_wp,  177.2508_wp,    0.0000_wp,    0.0000_wp, &
      &  313.9577_wp,  216.9806_wp,  185.2154_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  488.7501_wp,  322.3988_wp,  269.9886_wp,    0.0000_wp,    0.0000_wp, &
      &  351.1222_wp,  240.3260_wp,  204.2511_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  492.6319_wp,  330.4713_wp,  278.3620_wp,    0.0000_wp,    0.0000_wp, &
      &  322.9831_wp,  225.4398_wp,  193.0221_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  586.7732_wp,  391.0508_wp,  328.3978_wp,    0.0000_wp,    0.0000_wp, &
      &  547.8346_wp,  367.3197_wp,  309.2352_wp,    0.0000_wp,    0.0000_wp, &
      &  425.2069_wp,  291.2261_wp,  247.3262_wp,    0.0000_wp,    0.0000_wp, &
      &  340.8328_wp,  238.4311_wp,  204.2277_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  505.4452_wp,  344.9100_wp,  292.3920_wp,    0.0000_wp,    0.0000_wp, &
      &  470.3078_wp,  322.7816_wp,  274.2990_wp,    0.0000_wp,    0.0000_wp, &
      &  446.0876_wp,  307.3740_wp,  261.6413_wp,    0.0000_wp,    0.0000_wp, &
      &  396.0183_wp,  275.5707_wp,  235.5323_wp,    0.0000_wp,    0.0000_wp, &
      &  345.5051_wp,  243.4182_wp,  209.1152_wp,    0.0000_wp,    0.0000_wp, &
      &  422.1971_wp,  294.5421_wp,  252.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  414.5411_wp,  289.5934_wp,  247.9167_wp,    0.0000_wp,    0.0000_wp, &
      &  402.8017_wp,  282.0464_wp,  241.6956_wp,    0.0000_wp,    0.0000_wp, &
      &  386.7290_wp,  271.7387_wp,  233.2016_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  374.0632_wp,  264.6717_wp,  227.8311_wp,    0.0000_wp,    0.0000_wp, &
      &  374.2900_wp,  264.7815_wp,  227.9132_wp,    0.0000_wp,    0.0000_wp, &
      &  371.2277_wp,  262.7255_wp,  226.1893_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  323.1941_wp,  232.0580_wp,  201.0472_wp,    0.0000_wp,    0.0000_wp, &
      &  325.6650_wp,  233.6608_wp,  202.3758_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  277.8680_wp,  202.2567_wp,  176.3002_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1026 * max_ref * max_ref + 1 : 1045 * max_ref * max_ref) = [ &
      & 1749.7615_wp, 1063.3477_wp,  862.1222_wp,    0.0000_wp,    0.0000_wp, &
      &  582.3742_wp,  389.9183_wp,  328.6889_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1473.5487_wp,  928.8255_wp,  763.1756_wp,    0.0000_wp,    0.0000_wp, &
      & 1203.4301_wp,  760.2437_wp,  625.8949_wp,    0.0000_wp,    0.0000_wp, &
      &  590.3744_wp,  401.5048_wp,  340.2392_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1266.8436_wp,  812.9338_wp,  672.6836_wp,    0.0000_wp,    0.0000_wp, &
      &  886.2750_wp,  580.4145_wp,  484.7257_wp,    0.0000_wp,    0.0000_wp, &
      &  559.3453_wp,  386.3915_wp,  329.4057_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1125.3403_wp,  730.4970_wp,  607.3080_wp,    0.0000_wp,    0.0000_wp, &
      &  634.8024_wp,  431.8288_wp,  366.0539_wp,    0.0000_wp,    0.0000_wp, &
      &  561.6169_wp,  389.7673_wp,  332.9103_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1018.2915_wp,  666.2259_wp,  555.7204_wp,    0.0000_wp,    0.0000_wp, &
      &  641.4164_wp,  436.1440_wp,  369.6744_wp,    0.0000_wp,    0.0000_wp, &
      &  544.0821_wp,  377.2249_wp,  322.2101_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  775.8353_wp,  514.9128_wp,  432.2782_wp,    0.0000_wp,    0.0000_wp, &
      &  595.1503_wp,  406.9338_wp,  345.7600_wp,    0.0000_wp,    0.0000_wp, &
      &  475.0273_wp,  333.9387_wp,  286.9340_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  869.8962_wp,  574.2456_wp,  480.9081_wp,    0.0000_wp,    0.0000_wp, &
      &  680.0524_wp,  456.5618_wp,  385.2595_wp,    0.0000_wp,    0.0000_wp, &
      &  415.1463_wp,  295.1012_wp,  254.7749_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  653.1775_wp,  438.0426_wp,  369.4328_wp,    0.0000_wp,    0.0000_wp, &
      &  511.8315_wp,  354.9282_wp,  303.2934_wp,    0.0000_wp,    0.0000_wp, &
      &  390.8937_wp,  279.0333_wp,  241.3497_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  713.9427_wp,  478.2848_wp,  402.9498_wp,    0.0000_wp,    0.0000_wp, &
      &  478.2848_wp,  332.2425_wp,  284.1821_wp,    0.0000_wp,    0.0000_wp, &
      &  402.9498_wp,  284.1821_wp,  244.6586_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   40.2372_wp,   28.9661_wp,   28.0631_wp,    0.0000_wp,    0.0000_wp, &
      &   66.1211_wp,   46.3489_wp,   44.6498_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   25.8716_wp,   19.4689_wp,   19.0208_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  722.4294_wp,  443.9106_wp,  412.2171_wp,    0.0000_wp,    0.0000_wp, &
      &  225.8042_wp,  149.1209_wp,  141.5639_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  390.4310_wp,  256.2327_wp,  242.8825_wp,    0.0000_wp,    0.0000_wp, &
      &  286.0395_wp,  190.4826_wp,  181.1811_wp,    0.0000_wp,    0.0000_wp, &
      &  180.2178_wp,  124.7151_wp,  119.7960_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  253.3425_wp,  173.4360_wp,  166.1533_wp,    0.0000_wp,    0.0000_wp, &
      &  211.5559_wp,  146.2020_wp,  140.3599_wp,    0.0000_wp,    0.0000_wp, &
      &  176.3381_wp,  123.0227_wp,  118.3487_wp,    0.0000_wp,    0.0000_wp, &
      &  131.4019_wp,   93.8997_wp,   90.8284_wp,    0.0000_wp,    0.0000_wp, &
      &  123.2913_wp,   88.4515_wp,   85.6281_wp,    0.0000_wp,    0.0000_wp, &
      &  166.7111_wp,  118.0137_wp,  113.9126_wp,    0.0000_wp,    0.0000_wp, &
      &  156.5006_wp,  110.7946_wp,  106.9320_wp,    0.0000_wp,    0.0000_wp, &
      &  126.2362_wp,   90.6249_wp,   87.7320_wp,    0.0000_wp,    0.0000_wp, &
      &  118.1427_wp,   84.8954_wp,   82.1930_wp,    0.0000_wp,    0.0000_wp, &
      &   95.1612_wp,   69.7730_wp,   67.8363_wp,    0.0000_wp,    0.0000_wp, &
      &  114.4763_wp,   83.1710_wp,   80.7118_wp,    0.0000_wp,    0.0000_wp, &
      &  106.7438_wp,   77.6479_wp,   75.3636_wp,    0.0000_wp,    0.0000_wp, &
      &  100.0370_wp,   72.9965_wp,   70.8909_wp,    0.0000_wp,    0.0000_wp, &
      &   87.9430_wp,   64.4651_wp,   62.6532_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   85.6427_wp,   63.4125_wp,   61.7619_wp,    0.0000_wp,    0.0000_wp, &
      &   76.5871_wp,   57.0049_wp,   55.5744_wp,    0.0000_wp,    0.0000_wp, &
      &   67.3354_wp,   50.4709_wp,   49.2629_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   64.2707_wp,   48.3824_wp,   47.2645_wp,    0.0000_wp,    0.0000_wp, &
      &   51.7740_wp,   39.6636_wp,   38.8697_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   49.0990_wp,   37.4647_wp,   36.6837_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1045 * max_ref * max_ref + 1 : 1064 * max_ref * max_ref) = [ &
      &  860.6209_wp,  531.1676_wp,  493.9241_wp,    0.0000_wp,    0.0000_wp, &
      &  332.4392_wp,  218.2333_wp,  206.7940_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  629.5912_wp,  408.3022_wp,  385.7118_wp,    0.0000_wp,    0.0000_wp, &
      &  515.6457_wp,  337.2256_wp,  319.2176_wp,    0.0000_wp,    0.0000_wp, &
      &  324.9551_wp,  220.7375_wp,  211.0831_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  569.1410_wp,  376.8776_wp,  358.0287_wp,    0.0000_wp,    0.0000_wp, &
      &  513.4163_wp,  341.7946_wp,  325.1370_wp,    0.0000_wp,    0.0000_wp, &
      &  393.6776_wp,  266.8484_wp,  254.9948_wp,    0.0000_wp,    0.0000_wp, &
      &  300.4977_wp,  208.3470_wp,  200.2114_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  436.9463_wp,  297.8174_wp,  285.0138_wp,    0.0000_wp,    0.0000_wp, &
      &  402.3662_wp,  275.4925_wp,  263.9234_wp,    0.0000_wp,    0.0000_wp, &
      &  372.8898_wp,  256.4133_wp,  245.8989_wp,    0.0000_wp,    0.0000_wp, &
      &  364.4241_wp,  249.9808_wp,  239.5605_wp,    0.0000_wp,    0.0000_wp, &
      &  294.5220_wp,  206.0675_wp,  198.4222_wp,    0.0000_wp,    0.0000_wp, &
      &  333.5219_wp,  233.2106_wp,  224.5211_wp,    0.0000_wp,    0.0000_wp, &
      &  323.5276_wp,  226.4605_wp,  218.0675_wp,    0.0000_wp,    0.0000_wp, &
      &  319.6077_wp,  223.4125_wp,  215.0564_wp,    0.0000_wp,    0.0000_wp, &
      &  294.7986_wp,  207.4505_wp,  199.9979_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  273.1568_wp,  194.3373_wp,  187.8007_wp,    0.0000_wp,    0.0000_wp, &
      &  270.2524_wp,  192.2172_wp,  185.7368_wp,    0.0000_wp,    0.0000_wp, &
      &  264.7345_wp,  188.3752_wp,  182.0388_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  220.4099_wp,  159.4914_wp,  154.6693_wp,    0.0000_wp,    0.0000_wp, &
      &  218.1903_wp,  157.8524_wp,  153.0708_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  178.5021_wp,  131.1338_wp,  127.5483_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1424.0271_wp,  873.2984_wp,  809.4255_wp,    0.0000_wp,    0.0000_wp, &
      &  452.2464_wp,  302.4635_wp,  287.8260_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1129.6670_wp,  718.4983_wp,  674.5740_wp,    0.0000_wp,    0.0000_wp, &
      &  902.4405_wp,  576.1374_wp,  541.0773_wp,    0.0000_wp,    0.0000_wp, &
      &  456.2063_wp,  310.3909_wp,  296.8146_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  924.8556_wp,  593.4994_wp,  558.7593_wp,    0.0000_wp,    0.0000_wp, &
      &  443.0816_wp,  302.4898_wp,  289.4965_wp,    0.0000_wp,    0.0000_wp, &
      &  433.9484_wp,  298.5692_wp,  286.3025_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  885.7657_wp,  573.5972_wp,  541.3741_wp,    0.0000_wp,    0.0000_wp, &
      &  427.4577_wp,  292.2135_wp,  279.7105_wp,    0.0000_wp,    0.0000_wp, &
      &  438.9752_wp,  302.3112_wp,  289.9530_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  807.1258_wp,  525.4732_wp,  496.6905_wp,    0.0000_wp,    0.0000_wp, &
      &  419.1134_wp,  286.4278_wp,  274.1490_wp,    0.0000_wp,    0.0000_wp, &
      &  402.1750_wp,  276.9472_wp,  265.5767_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  635.3257_wp,  414.7918_wp,  392.2297_wp,    0.0000_wp,    0.0000_wp, &
      &  416.3230_wp,  283.7785_wp,  271.4250_wp,    0.0000_wp,    0.0000_wp, &
      &  298.3014_wp,  209.5976_wp,  201.9007_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  689.9647_wp,  452.7353_wp,  428.8509_wp,    0.0000_wp,    0.0000_wp, &
      &  378.4295_wp,  261.2635_wp,  250.7187_wp,    0.0000_wp,    0.0000_wp, &
      &  252.7595_wp,  179.7824_wp,  173.6495_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  540.6599_wp,  356.1344_wp,  337.5770_wp,    0.0000_wp,    0.0000_wp, &
      &  349.0258_wp,  242.0147_wp,  232.4750_wp,    0.0000_wp,    0.0000_wp, &
      &  248.4203_wp,  176.4513_wp,  170.3774_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  567.1044_wp,  376.9928_wp,  358.3743_wp,    0.0000_wp,    0.0000_wp, &
      &  332.9896_wp,  230.9788_wp,  221.8767_wp,    0.0000_wp,    0.0000_wp, &
      &  295.8366_wp,  206.0649_wp,  198.1008_wp,    0.0000_wp,    0.0000_wp, &
      &  260.6337_wp,  183.8900_wp,  177.3010_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  587.1819_wp,  388.1200_wp,  368.4032_wp,    0.0000_wp,    0.0000_wp, &
      &  325.8858_wp,  225.8165_wp,  216.8602_wp,    0.0000_wp,    0.0000_wp, &
      &  274.3366_wp,  192.1416_wp,  184.9554_wp,    0.0000_wp,    0.0000_wp, &
      &  291.0309_wp,  201.5536_wp,  193.4900_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  450.0559_wp,  298.3894_wp,  283.3292_wp,    0.0000_wp,    0.0000_wp, &
      &  325.0466_wp,  223.0154_wp,  213.6379_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1064 * max_ref * max_ref + 1 : 1083 * max_ref * max_ref) = [ &
      &  454.8031_wp,  306.0633_wp,  291.8725_wp,    0.0000_wp,    0.0000_wp, &
      &  299.8317_wp,  209.4546_wp,  201.5508_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  541.1501_wp,  361.8780_wp,  344.5686_wp,    0.0000_wp,    0.0000_wp, &
      &  505.6698_wp,  340.0675_wp,  324.2726_wp,    0.0000_wp,    0.0000_wp, &
      &  393.6370_wp,  270.0591_wp,  258.7879_wp,    0.0000_wp,    0.0000_wp, &
      &  316.4721_wp,  221.4649_wp,  213.2345_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  467.7042_wp,  319.6946_wp,  306.1192_wp,    0.0000_wp,    0.0000_wp, &
      &  435.5268_wp,  299.3280_wp,  286.9870_wp,    0.0000_wp,    0.0000_wp, &
      &  413.3244_wp,  285.1360_wp,  273.6234_wp,    0.0000_wp,    0.0000_wp, &
      &  367.4287_wp,  255.8441_wp,  246.0490_wp,    0.0000_wp,    0.0000_wp, &
      &  321.1117_wp,  226.2274_wp,  218.1530_wp,    0.0000_wp,    0.0000_wp, &
      &  391.8528_wp,  273.5010_wp,  263.1821_wp,    0.0000_wp,    0.0000_wp, &
      &  384.8152_wp,  268.9427_wp,  258.8699_wp,    0.0000_wp,    0.0000_wp, &
      &  374.0353_wp,  261.9897_wp,  252.3030_wp,    0.0000_wp,    0.0000_wp, &
      &  359.2842_wp,  252.4918_wp,  243.3399_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  347.8393_wp,  246.0921_wp,  237.5192_wp,    0.0000_wp,    0.0000_wp, &
      &  348.0402_wp,  246.1937_wp,  237.6065_wp,    0.0000_wp,    0.0000_wp, &
      &  345.2124_wp,  244.2954_wp,  235.7943_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  301.1390_wp,  216.0945_wp,  209.1971_wp,    0.0000_wp,    0.0000_wp, &
      &  303.4107_wp,  217.5738_wp,  210.5970_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  259.3964_wp,  188.6339_wp,  183.1095_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1590.4415_wp,  979.4032_wp,  908.8152_wp,    0.0000_wp,    0.0000_wp, &
      &  537.2565_wp,  361.3047_wp,  344.2670_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1347.9007_wp,  856.7169_wp,  804.0036_wp,    0.0000_wp,    0.0000_wp, &
      & 1100.5370_wp,  701.6550_wp,  658.5196_wp,    0.0000_wp,    0.0000_wp, &
      &  545.9768_wp,  372.2612_wp,  356.1148_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1161.9289_wp,  750.5933_wp,  707.8343_wp,    0.0000_wp,    0.0000_wp, &
      &  814.8402_wp,  536.8566_wp,  508.6839_wp,    0.0000_wp,    0.0000_wp, &
      &  518.4374_wp,  358.6133_wp,  344.3053_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1033.8735_wp,  674.9929_wp,  638.4463_wp,    0.0000_wp,    0.0000_wp, &
      &  587.0202_wp,  400.4209_wp,  383.0425_wp,    0.0000_wp,    0.0000_wp, &
      &  520.8781_wp,  361.8771_wp,  347.7992_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  936.5528_wp,  615.9759_wp,  583.7653_wp,    0.0000_wp,    0.0000_wp, &
      &  593.1414_wp,  404.4300_wp,  386.8574_wp,    0.0000_wp,    0.0000_wp, &
      &  504.5037_wp,  350.2910_wp,  336.5535_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  714.9495_wp,  476.7090_wp,  453.3284_wp,    0.0000_wp,    0.0000_wp, &
      &  550.7557_wp,  377.5535_wp,  361.5781_wp,    0.0000_wp,    0.0000_wp, &
      &  441.3319_wp,  310.5005_wp,  299.2251_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  801.0442_wp,  531.3700_wp,  504.6607_wp,    0.0000_wp,    0.0000_wp, &
      &  627.5810_wp,  423.1279_wp,  403.4188_wp,    0.0000_wp,    0.0000_wp, &
      &  386.2986_wp,  274.6918_wp,  265.3352_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  602.7956_wp,  405.9398_wp,  386.9728_wp,    0.0000_wp,    0.0000_wp, &
      &  474.6369_wp,  329.6826_wp,  316.7499_wp,    0.0000_wp,    0.0000_wp, &
      &  363.9513_wp,  259.8584_wp,  251.2231_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  658.8753_wp,  443.0659_wp,  422.3203_wp,    0.0000_wp,    0.0000_wp, &
      &  443.6280_wp,  308.7059_wp,  296.6934_wp,    0.0000_wp,    0.0000_wp, &
      &  374.5394_wp,  264.4321_wp,  254.9713_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  608.5041_wp,  411.1361_wp,  392.3548_wp,    0.0000_wp,    0.0000_wp, &
      &  411.1361_wp,  286.9519_wp,  275.9575_wp,    0.0000_wp,    0.0000_wp, &
      &  392.3548_wp,  275.9575_wp,  265.8746_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   33.5413_wp,   27.9518_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   54.9658_wp,   44.7835_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   21.7896_wp,   18.7731_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1083 * max_ref * max_ref + 1 : 1102 * max_ref * max_ref) = [ &
      &  614.1477_wp,  432.6078_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  188.3374_wp,  144.6563_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  325.5775_wp,  248.6573_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  238.6962_wp,  184.6530_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  149.7198_wp,  120.5975_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  210.6811_wp,  167.8350_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  176.0603_wp,  141.4060_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  146.9017_wp,  118.9236_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  109.4741_wp,   90.6508_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  102.7753_wp,   85.3773_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  138.8758_wp,  113.9993_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  130.5144_wp,  107.0286_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  105.3572_wp,   87.4951_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   98.7032_wp,   81.9715_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   79.6657_wp,   67.3070_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   95.7137_wp,   80.2663_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   89.3419_wp,   74.9354_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   83.7875_wp,   70.4509_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   73.8057_wp,   62.2209_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   71.8926_wp,   61.1748_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   64.3880_wp,   54.9856_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   56.7511_wp,   48.6912_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   54.1903_wp,   46.6726_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   43.8480_wp,   38.2539_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   41.5780_wp,   36.1486_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  731.0832_wp,  517.5680_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  277.9428_wp,  211.8419_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  526.4776_wp,  396.6125_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  431.2807_wp,  327.3509_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  270.1633_wp,  213.7477_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  474.5713_wp,  365.5269_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  427.9962_wp,  331.3832_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  327.7472_wp,  258.4065_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  249.6759_wp,  201.4661_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  363.3585_wp,  288.2863_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  334.6673_wp,  266.6008_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  310.1186_wp,  248.0750_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  303.3752_wp,  241.8989_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  244.8685_wp,  199.1553_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  277.3577_wp,  225.3945_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  269.1220_wp,  218.8617_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  265.9503_wp,  215.9419_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  245.2996_wp,  200.4348_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  227.5024_wp,  187.6597_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  225.1181_wp,  185.6197_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  220.5585_wp,  181.9100_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  183.9876_wp,  153.9032_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  182.1547_wp,  152.3270_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  149.4269_wp,  126.4825_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1218.4699_wp,  851.0088_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  379.0836_wp,  293.2911_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  951.4935_wp,  698.8618_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  762.6174_wp,  560.1298_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  380.4283_wp,  300.6094_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  776.8914_wp,  576.9636_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  369.4933_wp,  292.9063_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  361.5057_wp,  288.9408_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1102 * max_ref * max_ref + 1 : 1121 * max_ref * max_ref) = [ &
      &  743.2041_wp,  557.2880_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  356.7245_wp,  282.9332_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  365.7124_wp,  292.5514_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  676.6553_wp,  510.3644_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  349.8288_wp,  277.3525_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  335.4436_wp,  268.0353_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  533.6878_wp,  402.8383_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  347.6786_wp,  274.8452_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  249.1613_wp,  202.6684_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  577.8127_wp,  439.5099_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  315.3482_wp,  252.8298_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  211.2390_wp,  173.7381_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  453.6327_wp,  345.6995_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  290.9083_wp,  234.1511_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  207.6181_wp,  170.5403_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  473.9244_wp,  365.6925_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  277.6823_wp,  223.4805_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  247.0487_wp,  199.3629_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  217.7112_wp,  177.7934_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  490.9783_wp,  376.6281_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  271.8088_wp,  218.5075_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  229.0537_wp,  185.8430_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  243.2009_wp,  195.0917_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  377.3531_wp,  289.5580_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  271.4634_wp,  215.9581_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  379.5655_wp,  296.6728_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  249.8348_wp,  202.5685_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  451.5887_wp,  350.8010_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  421.7420_wp,  329.5387_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  327.8852_wp,  261.3654_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  263.3258_wp,  214.0794_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  389.3075_wp,  309.4543_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  362.5076_wp,  289.6411_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  344.0065_wp,  275.8484_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  305.7755_wp,  247.3728_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  267.2135_wp,  218.5854_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  326.0371_wp,  264.3957_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  320.2231_wp,  259.9706_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  311.2799_wp,  253.2180_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  299.0137_wp,  243.9926_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  289.6469_wp,  237.7217_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  289.8328_wp,  237.8238_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  287.5025_wp,  235.9871_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  251.1189_wp,  208.6044_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  253.0095_wp,  210.0411_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  216.7336_wp,  182.0016_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1360.7517_wp,  954.1320_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  450.4714_wp,  350.2295_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1136.9099_wp,  833.3168_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  931.6022_wp,  682.1817_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  455.7267_wp,  360.4891_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  976.2025_wp,  729.3316_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  684.9796_wp,  520.8724_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  431.9705_wp,  346.9360_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  867.0056_wp,  655.4329_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  490.3817_wp,  387.7336_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  433.9278_wp,  349.9975_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1121 * max_ref * max_ref + 1 : 1140 * max_ref * max_ref) = [ &
      &  784.8188_wp,  597.8528_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  495.4800_wp,  391.6637_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  420.8552_wp,  338.8335_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  599.0530_wp,  462.3383_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  460.2367_wp,  365.5261_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  368.2898_wp,  300.1493_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  671.1717_wp,  515.4953_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  526.0977_wp,  410.0667_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  322.5821_wp,  265.4033_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  505.0269_wp,  393.5088_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  396.2742_wp,  318.9665_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  304.0550_wp,  251.0367_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  551.0802_wp,  429.5143_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  370.6378_wp,  298.6678_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  313.1143_wp,  255.6479_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  508.7742_wp,  398.4662_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  343.6707_wp,  277.6023_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  327.8015_wp,  266.8459_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  426.7450_wp,  333.1583_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  333.1583_wp,  268.6130_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   35.6879_wp,   27.9281_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   58.3568_wp,   44.5006_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   23.1563_wp,   18.8607_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  621.0536_wp,  408.4429_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  197.0940_wp,  141.0901_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  340.4122_wp,  242.0950_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  249.9915_wp,  180.5207_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  158.6937_wp,  119.4623_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  222.6273_wp,  165.6587_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  186.2199_wp,  139.8793_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  155.4778_wp,  117.8843_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  116.3922_wp,   90.4277_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  109.2887_wp,   85.2274_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  147.4138_wp,  113.4369_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  138.3672_wp,  106.4332_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  111.9252_wp,   87.2915_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  104.7692_wp,   81.7500_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   84.7139_wp,   67.4028_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  101.7377_wp,   80.2492_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   94.8770_wp,   74.8925_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   88.9858_wp,   70.4420_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   78.3016_wp,   62.2131_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   76.4092_wp,   61.3234_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   68.3943_wp,   55.1392_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   60.2334_wp,   48.8497_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   57.5500_wp,   46.8692_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   46.5340_wp,   38.4906_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   44.1062_wp,   36.3401_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  740.5787_wp,  489.5129_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  289.8474_wp,  205.9647_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  547.6721_wp,  384.1728_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  449.1567_wp,  317.9002_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  285.2161_wp,  210.5498_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1140 * max_ref * max_ref + 1 : 1159 * max_ref * max_ref) = [ &
      &  497.0337_wp,  356.8654_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  448.8071_wp,  324.0830_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  345.2997_wp,  254.2236_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  264.7291_wp,  199.6670_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  383.6790_wp,  284.2139_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  353.5976_wp,  263.1438_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  327.9622_wp,  245.1614_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  320.3429_wp,  238.7691_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  259.8815_wp,  197.7904_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  294.2465_wp,  223.7783_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  285.4802_wp,  217.3199_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  281.9469_wp,  214.2984_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  260.3892_wp,  199.2714_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  241.7614_wp,  187.0163_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  239.1784_wp,  184.9522_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  234.3154_wp,  181.2582_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  195.7044_wp,  153.8592_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  193.7273_wp,  152.2646_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  158.9650_wp,  126.7356_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1222.5553_wp,  800.2593_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  395.5135_wp,  286.2735_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  978.4251_wp,  670.2633_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  781.6600_wp,  537.0889_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  400.4535_wp,  295.6941_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  802.6020_wp,  555.6679_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  389.1859_wp,  288.3875_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  381.7298_wp,  285.2814_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  770.0303_wp,  538.5502_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  375.5261_wp,  278.5869_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  386.2241_wp,  288.9111_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  702.4213_wp,  494.2237_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  368.1871_wp,  273.0331_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  353.8227_wp,  264.5314_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  553.0919_wp,  389.9953_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  365.5531_wp,  270.2890_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  263.4511_wp,  200.9381_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  601.4053_wp,  426.8579_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  333.1518_wp,  249.8201_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  223.7644_wp,  172.7698_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  471.5329_wp,  335.7642_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  307.5209_wp,  231.6170_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  219.8750_wp,  169.5246_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  495.6313_wp,  356.9172_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  293.4074_wp,  221.0248_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  260.8701_wp,  197.2313_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  230.4010_wp,  176.4697_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  512.6220_wp,  366.8668_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  287.0958_wp,  216.0211_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  242.1824_wp,  184.1433_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  256.3643_wp,  192.6099_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  393.0510_wp,  281.8675_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  285.8180_wp,  212.7468_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  398.4572_wp,  290.7895_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  264.5716_wp,  200.8224_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  473.4203_wp,  343.2943_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  442.8639_wp,  323.1080_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  346.0568_wp,  257.9007_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  279.2886_wp,  212.5148_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1159 * max_ref * max_ref + 1 : 1178 * max_ref * max_ref) = [ &
      &  410.9058_wp,  305.1468_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  383.0204_wp,  286.0551_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  363.7536_wp,  272.7228_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  323.9292_wp,  245.2119_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  283.7237_wp,  217.3760_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  345.6085_wp,  262.2840_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  339.4814_wp,  257.9689_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  330.1075_wp,  251.4082_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  317.2880_wp,  242.4605_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  307.5539_wp,  236.5779_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  307.7214_wp,  236.6604_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  305.2454_wp,  234.8470_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  266.9700_wp,  208.2113_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  268.9491_wp,  209.6083_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  230.5493_wp,  182.0867_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1366.3938_wp,  898.5342_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  470.3119_wp,  342.3410_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1167.0471_wp,  798.4528_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  952.7533_wp,  653.2782_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  479.3797_wp,  354.6096_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1009.4418_wp,  703.7992_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  710.2172_wp,  505.6747_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  456.4875_wp,  342.9987_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  900.1154_wp,  635.1370_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  515.3861_wp,  381.3405_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  459.0164_wp,  346.4792_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  816.5495_wp,  580.8412_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  520.7629_wp,  385.1206_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  444.4932_wp,  335.1312_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  624.9506_wp,  450.9776_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  484.0286_wp,  359.9059_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  389.8300_wp,  297.8478_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  699.5382_wp,  502.1168_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  549.6338_wp,  401.2382_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  341.9237_wp,  264.0201_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  527.9387_wp,  384.9436_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  418.2579_wp,  315.3365_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  322.4084_wp,  249.9360_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  577.0107_wp,  420.3746_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  391.0675_wp,  295.3036_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  331.0815_wp,  253.6663_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  533.4134_wp,  390.5770_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  362.6432_wp,  274.6154_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  346.6055_wp,  264.5941_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  445.8701_wp,  325.9662_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  351.3712_wp,  265.5450_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  468.1900_wp,  345.0599_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  345.0599_wp,  263.4508_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   44.0417_wp,   42.2827_wp,   34.8931_wp,   30.3193_wp,    0.0000_wp, &
      &   72.9076_wp,   69.6754_wp,   56.5189_wp,   48.4270_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   27.9586_wp,   27.0236_wp,   22.9076_wp,   20.3402_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1178 * max_ref * max_ref + 1 : 1197 * max_ref * max_ref) = [ &
      &  828.9673_wp,  769.2514_wp,  565.2654_wp,  444.6978_wp,    0.0000_wp, &
      &  253.1370_wp,  239.0419_wp,  185.9170_wp,  153.8264_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  438.3396_wp,  413.4965_wp,  320.2807_wp,  264.0439_wp,    0.0000_wp, &
      &  320.0912_wp,  302.6678_wp,  236.6771_wp,  196.7454_wp,    0.0000_wp, &
      &  199.4111_wp,  190.1313_wp,  152.9239_wp,  130.1139_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  281.1737_wp,  267.4750_wp,  213.4484_wp,  180.4568_wp,    0.0000_wp, &
      &  234.2171_wp,  223.1629_wp,  179.1755_wp,  152.2615_wp,    0.0000_wp, &
      &  194.7649_wp,  185.8576_wp,  150.1267_wp,  128.2191_wp,    0.0000_wp, &
      &  144.1259_wp,  138.1855_wp,  113.4826_wp,   98.2235_wp,    0.0000_wp, &
      &  135.0814_wp,  129.5979_wp,  106.6946_wp,   92.5355_wp,    0.0000_wp, &
      &  183.3368_wp,  175.4585_wp,  143.1602_wp,  123.2705_wp,    0.0000_wp, &
      &  172.1229_wp,  164.6812_wp,  134.3014_wp,  115.6130_wp,    0.0000_wp, &
      &  138.2831_wp,  132.6505_wp,  109.2000_wp,   94.7153_wp,    0.0000_wp, &
      &  129.3853_wp,  124.1085_wp,  102.1877_wp,   88.6572_wp,    0.0000_wp, &
      &  103.6198_wp,   99.7522_wp,   83.2270_wp,   72.9703_wp,    0.0000_wp, &
      &  124.9798_wp,  120.1279_wp,   99.6435_wp,   86.9569_wp,    0.0000_wp, &
      &  116.4959_wp,  111.9707_wp,   92.9086_wp,   81.1110_wp,    0.0000_wp, &
      &  109.0825_wp,  104.8972_wp,   87.2076_wp,   76.2530_wp,    0.0000_wp, &
      &   95.7727_wp,   92.1403_wp,   76.7798_wp,   67.2704_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   92.9968_wp,   89.6520_wp,   75.2351_wp,   66.2741_wp,    0.0000_wp, &
      &   83.0278_wp,   80.0986_wp,   67.4193_wp,   59.5344_wp,    0.0000_wp, &
      &   72.8571_wp,   70.3511_wp,   59.4511_wp,   52.6690_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   69.4558_wp,   67.1232_wp,   56.8922_wp,   50.5156_wp,    0.0000_wp, &
      &   55.6589_wp,   53.9364_wp,   46.2082_wp,   41.3746_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   52.8517_wp,   51.1747_wp,   43.7215_wp,   39.0674_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  986.0999_wp,  915.9413_wp,  675.2268_wp,  532.8538_wp,    0.0000_wp, &
      &  373.3794_wp,  352.0074_wp,  272.3735_wp,  224.4097_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  709.3565_wp,  667.2844_wp,  512.0902_wp,  418.8567_wp,    0.0000_wp, &
      &  579.9234_wp,  546.2762_wp,  421.5573_wp,  346.4940_wp,    0.0000_wp, &
      &  361.3835_wp,  343.3098_wp,  272.5677_wp,  229.4425_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  637.5832_wp,  602.4143_wp,  469.4540_wp,  389.0698_wp,    0.0000_wp, &
      &  574.3420_wp,  543.2188_wp,  424.8967_wp,  353.2654_wp,    0.0000_wp, &
      &  438.2474_wp,  416.0033_wp,  329.5920_wp,  276.9940_wp,    0.0000_wp, &
      &  332.3410_wp,  316.9852_wp,  255.2803_wp,  217.4318_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  485.5392_wp,  461.5093_wp,  367.2028_wp,  309.6779_wp,    0.0000_wp, &
      &  446.5937_wp,  424.8293_wp,  339.0315_wp,  286.6412_wp,    0.0000_wp, &
      &  413.3754_wp,  393.5617_wp,  315.0177_wp,  266.9974_wp,    0.0000_wp, &
      &  404.3294_wp,  384.6752_wp,  307.2515_wp,  259.9858_wp,    0.0000_wp, &
      &  324.9345_wp,  310.4122_wp,  251.4716_wp,  215.2424_wp,    0.0000_wp, &
      &  368.0275_wp,  351.5157_wp,  284.6146_wp,  243.5106_wp,    0.0000_wp, &
      &  356.9069_wp,  340.9396_wp,  276.2154_wp,  236.4461_wp,    0.0000_wp, &
      &  352.7247_wp,  336.8285_wp,  272.5864_wp,  233.1422_wp,    0.0000_wp, &
      &  324.7346_wp,  310.5048_wp,  252.4456_wp,  216.7223_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  299.9853_wp,  287.3922_wp,  235.3301_wp,  203.2097_wp,    0.0000_wp, &
      &  296.8214_wp,  284.3350_wp,  232.7649_wp,  200.9558_wp,    0.0000_wp, &
      &  290.7255_wp,  278.5087_wp,  228.0473_wp,  196.9225_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  240.9104_wp,  231.4554_wp,  191.5816_wp,  166.8867_wp,    0.0000_wp, &
      &  238.4988_wp,  229.1229_wp,  189.6139_wp,  165.1495_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  194.2646_wp,  187.0916_wp,  156.3179_wp,  137.2010_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1638.8153_wp, 1517.3687_wp, 1110.1907_wp,  869.9598_wp,    0.0000_wp, &
      &  505.6838_wp,  478.0141_wp,  374.0585_wp,  311.3187_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1281.2612_wp, 1198.9287_wp,  905.1186_wp,  729.9594_wp,    0.0000_wp, &
      & 1024.1614_wp,  957.9897_wp,  723.9814_wp,  584.4834_wp,    0.0000_wp, &
      &  507.2605_wp,  481.6729_wp,  382.2720_wp,  321.8065_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1045.8401_wp,  980.8464_wp,  745.8282_wp,  605.3047_wp,    0.0000_wp, &
      &  492.1904_wp,  467.6700_wp,  372.0117_wp,  313.7920_wp,    0.0000_wp, &
      &  481.0066_wp,  457.8184_wp,  366.3017_wp,  310.4193_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1197 * max_ref * max_ref + 1 : 1216 * max_ref * max_ref) = [ &
      &  999.0971_wp,  938.7641_wp,  718.5537_wp,  586.5752_wp,    0.0000_wp, &
      &  474.8112_wp,  451.1658_wp,  359.1057_wp,  303.0566_wp,    0.0000_wp, &
      &  486.4497_wp,  463.0777_wp,  370.7390_wp,  314.3438_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  908.9753_wp,  855.0744_wp,  657.1002_wp,  538.2672_wp,    0.0000_wp, &
      &  465.5647_wp,  442.3420_wp,  351.9807_wp,  296.9854_wp,    0.0000_wp, &
      &  445.7766_wp,  424.2272_wp,  339.4399_wp,  287.7026_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  715.2799_wp,  672.8526_wp,  517.5497_wp,  424.4016_wp,    0.0000_wp, &
      &  462.8304_wp,  439.4728_wp,  348.9930_wp,  293.9840_wp,    0.0000_wp, &
      &  328.7836_wp,  314.0026_wp,  254.5781_wp,  218.1566_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  775.2736_wp,  730.5221_wp,  564.6362_wp,  464.8411_wp,    0.0000_wp, &
      &  419.0707_wp,  399.1079_wp,  320.0431_wp,  271.7273_wp,    0.0000_wp, &
      &  277.6350_wp,  265.7575_wp,  217.2467_wp,  187.4120_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  607.0964_wp,  572.1729_wp,  442.9855_wp,  365.3129_wp,    0.0000_wp, &
      &  386.0628_wp,  367.9592_wp,  295.9209_wp,  251.8512_wp,    0.0000_wp, &
      &  272.9769_wp,  261.2268_wp,  213.3373_wp,  183.8976_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  634.7263_wp,  599.8255_wp,  468.1580_wp,  388.6270_wp,    0.0000_wp, &
      &  368.3224_wp,  351.0300_wp,  282.3130_wp,  240.2860_wp,    0.0000_wp, &
      &  326.8852_wp,  311.6813_wp,  251.2206_wp,  214.2453_wp,    0.0000_wp, &
      &  286.9299_wp,  274.2537_wp,  222.9925_wp,  191.5317_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  658.2526_wp,  621.3305_wp,  482.9577_wp,  399.5106_wp,    0.0000_wp, &
      &  360.5731_wp,  343.5620_wp,  276.0871_wp,  234.8390_wp,    0.0000_wp, &
      &  302.6341_wp,  288.8824_wp,  233.7477_wp,  199.9712_wp,    0.0000_wp, &
      &  322.0771_wp,  306.7120_wp,  246.1570_wp,  209.2104_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  504.3802_wp,  476.0165_wp,  370.2813_wp,  306.5977_wp,    0.0000_wp, &
      &  360.6626_wp,  342.8931_wp,  273.5167_wp,  231.2700_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  507.2452_wp,  480.5972_wp,  378.4634_wp,  316.5382_wp,    0.0000_wp, &
      &  330.9585_wp,  315.9166_wp,  255.4101_wp,  218.3027_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  604.7183_wp,  572.2658_wp,  448.7448_wp,  373.9479_wp,    0.0000_wp, &
      &  564.1564_wp,  534.5127_wp,  420.8557_wp,  351.9154_wp,    0.0000_wp, &
      &  436.7407_wp,  415.4641_wp,  331.7913_wp,  280.7261_wp,    0.0000_wp, &
      &  349.1330_wp,  333.4815_wp,  270.0988_wp,  231.1618_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  519.3501_wp,  493.7984_wp,  393.4823_wp,  332.2932_wp,    0.0000_wp, &
      &  482.9115_wp,  459.6285_wp,  367.6282_wp,  311.4260_wp,    0.0000_wp, &
      &  457.8119_wp,  436.0573_wp,  349.6800_wp,  296.8562_wp,    0.0000_wp, &
      &  405.9303_wp,  387.3404_wp,  312.6183_wp,  266.7944_wp,    0.0000_wp, &
      &  353.6035_wp,  338.1831_wp,  275.1658_wp,  236.3761_wp,    0.0000_wp, &
      &  432.6167_wp,  413.0126_wp,  333.9114_wp,  285.3599_wp,    0.0000_wp, &
      &  424.7008_wp,  405.5453_wp,  328.1518_wp,  280.6350_wp,    0.0000_wp, &
      &  412.5541_wp,  394.1073_wp,  319.3719_wp,  273.4592_wp,    0.0000_wp, &
      &  395.9157_wp,  378.4539_wp,  307.3832_wp,  263.6781_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  382.5968_wp,  366.1519_wp,  298.6933_wp,  257.1422_wp,    0.0000_wp, &
      &  382.8381_wp,  366.3652_wp,  298.8244_wp,  257.2275_wp,    0.0000_wp, &
      &  379.6844_wp,  363.3697_wp,  296.4552_wp,  255.2418_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  329.9250_wp,  316.5231_wp,  260.5925_wp,  226.0224_wp,    0.0000_wp, &
      &  332.4791_wp,  318.9305_wp,  262.4494_wp,  227.5472_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  283.1269_wp,  272.2288_wp,  226.0215_wp,  197.3762_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1828.5696_wp, 1694.2091_wp, 1243.0036_wp,  976.6194_wp,    0.0000_wp, &
      &  599.8987_wp,  567.6002_wp,  445.7605_wp,  372.1527_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1529.6862_wp, 1430.6640_wp, 1078.8447_wp,  869.2745_wp,    0.0000_wp, &
      & 1250.1173_wp, 1168.5036_wp,  881.5739_wp,  710.6790_wp,    0.0000_wp, &
      &  606.8145_wp,  576.2896_wp,  457.8102_wp,  385.7594_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1311.8922_wp, 1231.6979_wp,  940.6363_wp,  766.4059_wp,    0.0000_wp, &
      &  916.1592_wp,  862.8833_wp,  667.6471_wp,  550.2164_wp,    0.0000_wp, &
      &  573.8021_wp,  546.6690_wp,  438.9483_wp,  373.0875_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1163.5963_wp, 1095.0419_wp,  842.9594_wp,  691.5997_wp,    0.0000_wp, &
      &  652.5654_wp,  619.6537_wp,  492.2291_wp,  414.7814_wp,    0.0000_wp, &
      &  575.8020_wp,  549.0572_wp,  442.2260_wp,  376.8173_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1216 * max_ref * max_ref + 1 : 1235 * max_ref * max_ref) = [ &
      & 1051.8917_wp,  991.4013_wp,  767.2243_wp,  632.3580_wp,    0.0000_wp, &
      &  659.2865_wp,  626.0254_wp,  497.1382_wp,  418.8482_wp,    0.0000_wp, &
      &  557.9340_wp,  531.7720_wp,  427.8632_wp,  364.3226_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  799.9994_wp,  755.8918_wp,  590.4050_wp,  490.5720_wp,    0.0000_wp, &
      &  611.3911_wp,  581.0549_wp,  463.0670_wp,  391.2722_wp,    0.0000_wp, &
      &  486.2116_wp,  464.5679_wp,  377.1375_wp,  323.4882_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  897.6451_wp,  847.3557_wp,  659.5747_wp,  546.3930_wp,    0.0000_wp, &
      &  700.4229_wp,  663.0380_wp,  521.6978_wp,  436.2161_wp,    0.0000_wp, &
      &  424.3145_wp,  406.2117_wp,  332.1088_wp,  286.5085_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  672.6516_wp,  636.7480_wp,  500.7277_wp,  418.4851_wp,    0.0000_wp, &
      &  524.8236_wp,  500.1628_wp,  402.3825_wp,  342.6308_wp,    0.0000_wp, &
      &  399.3109_wp,  382.5457_wp,  313.5888_wp,  271.1133_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  735.2525_wp,  696.1321_wp,  547.3300_wp,  457.2780_wp,    0.0000_wp, &
      &  490.3325_wp,  467.3771_wp,  376.3625_wp,  320.7454_wp,    0.0000_wp, &
      &  412.2889_wp,  394.0370_wp,  320.4011_wp,  275.2351_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  678.0996_wp,  642.6456_wp,  506.9866_wp,  424.7783_wp,    0.0000_wp, &
      &  454.0576_wp,  432.9973_wp,  349.3149_wp,  298.1562_wp,    0.0000_wp, &
      &  432.3183_wp,  412.9408_wp,  334.9874_wp,  287.2026_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  567.3303_wp,  537.2058_wp,  423.1021_wp,  354.1221_wp,    0.0000_wp, &
      &  440.2733_wp,  419.6868_wp,  338.1165_wp,  288.2865_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  593.3574_wp,  563.0312_wp,  446.1184_wp,  375.1486_wp,    0.0000_wp, &
      &  430.3607_wp,  411.1576_wp,  333.6408_wp,  286.0771_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  757.7397_wp,  716.7957_wp,  561.9006_wp,  468.2521_wp,    0.0000_wp, &
      &  716.7957_wp,  678.9920_wp,  534.7248_wp,  447.3269_wp,    0.0000_wp, &
      &  561.9006_wp,  534.7248_wp,  427.9140_wp,  362.7514_wp,    0.0000_wp, &
      &  468.2521_wp,  447.3269_wp,  362.7514_wp,  310.8315_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   41.5334_wp,   39.0500_wp,   37.9122_wp,   34.5503_wp,   31.6020_wp, &
      &   67.9430_wp,   63.5189_wp,   61.4868_wp,   55.5643_wp,   50.3940_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   26.7862_wp,   25.4105_wp,   24.7833_wp,   22.8875_wp,   21.2154_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  708.4597_wp,  642.4190_wp,  611.5612_wp,  527.1851_wp,  455.9199_wp, &
      &  228.0814_wp,  210.3815_wp,  202.2053_wp,  179.0706_wp,  159.1376_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  393.7991_wp,  362.7434_wp,  348.3949_wp,  307.8814_wp,  273.0046_wp, &
      &  289.4713_wp,  267.5036_wp,  257.3551_wp,  228.5584_wp,  203.7227_wp, &
      &  184.6849_wp,  172.1767_wp,  166.4248_wp,  149.7680_wp,  135.2605_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  258.7146_wp,  240.5934_wp,  232.2513_wp,  208.2379_wp,  187.3808_wp, &
      &  216.4276_wp,  201.6758_wp,  194.8895_wp,  175.2745_wp,  158.2137_wp, &
      &  180.6996_wp,  168.7248_wp,  163.2184_wp,  147.2385_wp,  133.3214_wp, &
      &  135.4439_wp,  127.1409_wp,  123.3340_wp,  112.1331_wp,  102.3247_wp, &
      &  127.1521_wp,  119.4559_wp,  115.9286_wp,  105.5295_wp,   96.4179_wp, &
      &  171.4367_wp,  160.5876_wp,  155.6083_wp,  141.0407_wp,  128.3113_wp, &
      &  160.7910_wp,  150.6040_wp,  145.9274_wp,  132.2562_wp,  120.3190_wp, &
      &  130.0766_wp,  122.2030_wp,  118.5942_wp,  107.9636_wp,   98.6540_wp, &
      &  121.6598_wp,  114.3072_wp,  110.9372_wp,  101.0131_wp,   92.3260_wp, &
      &   98.3431_wp,   92.8021_wp,   90.2683_wp,   82.7170_wp,   76.0810_wp, &
      &  118.1509_wp,  111.2768_wp,  108.1299_wp,   98.8025_wp,   90.6184_wp, &
      &  110.0951_wp,  103.7070_wp,  100.7826_wp,   92.1154_wp,   84.5138_wp, &
      &  103.2113_wp,   97.2826_wp,   94.5696_wp,   86.5174_wp,   79.4522_wp, &
      &   90.6876_wp,   85.5479_wp,   83.1968_wp,   76.2100_wp,   70.0800_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   88.5408_wp,   83.7120_wp,   81.5058_wp,   74.9029_wp,   69.0939_wp, &
      &   79.1704_wp,   74.9289_wp,   72.9923_wp,   67.1811_wp,   62.0665_wp, &
      &   69.6046_wp,   65.9635_wp,   64.3022_wp,   59.3029_wp,   54.9005_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   66.4996_wp,   63.0806_wp,   61.5214_wp,   56.8151_wp,   52.6661_wp, &
      &   53.6243_wp,   51.0492_wp,   49.8778_wp,   46.3030_wp,   43.1432_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   50.8110_wp,   48.3258_wp,   47.1942_wp,   43.7556_wp,   40.7195_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1235 * max_ref * max_ref + 1 : 1254 * max_ref * max_ref) = [ &
      &  845.1348_wp,  767.0955_wp,  730.6587_wp,  630.8687_wp,  546.5096_wp, &
      &  334.7023_wp,  308.2470_wp,  296.0155_wp,  261.5400_wp,  231.9019_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  631.9434_wp,  580.4500_wp,  556.6241_wp,  489.7532_wp,  432.3765_wp, &
      &  518.6518_wp,  477.2848_wp,  458.1425_wp,  404.2763_wp,  358.0089_wp, &
      &  331.2828_wp,  307.5475_wp,  296.6147_wp,  265.2575_wp,  238.0549_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  575.4018_wp,  531.0794_wp,  510.6087_wp,  452.6192_wp,  402.6252_wp, &
      &  519.8424_wp,  480.3735_wp,  462.1521_wp,  410.4176_wp,  365.7711_wp, &
      &  400.7894_wp,  371.8779_wp,  358.5500_wp,  320.3924_wp,  287.3337_wp, &
      &  308.0840_wp,  287.3337_wp,  277.7927_wp,  250.1421_wp,  226.0504_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  445.7623_wp,  414.1403_wp,  399.5774_wp,  357.7428_wp,  321.4363_wp, &
      &  410.8960_wp,  382.1255_wp,  368.8796_wp,  330.7531_wp,  297.6403_wp, &
      &  381.2230_wp,  354.8687_wp,  342.7407_wp,  307.7570_wp,  277.3459_wp, &
      &  372.0785_wp,  346.1431_wp,  334.2022_wp,  299.8218_wp,  269.9679_wp, &
      &  302.4879_wp,  282.6684_wp,  273.5628_wp,  247.0572_wp,  223.9283_wp, &
      &  342.4169_wp,  319.9307_wp,  309.5992_wp,  279.5392_wp,  253.3166_wp, &
      &  332.1722_wp,  310.4238_wp,  300.4318_wp,  271.3488_wp,  245.9773_wp, &
      &  327.9460_wp,  306.3712_wp,  296.4572_wp,  267.6299_wp,  242.4940_wp, &
      &  303.0033_wp,  283.4898_wp,  274.5293_wp,  248.3782_wp,  225.5418_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  281.3255_wp,  263.8277_wp,  255.8013_wp,  232.2434_wp,  211.6318_wp, &
      &  278.2787_wp,  260.9497_wp,  253.0003_wp,  229.6756_wp,  209.2713_wp, &
      &  272.5881_wp,  255.6336_wp,  247.8563_wp,  225.0338_wp,  205.0687_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  227.5635_wp,  214.1723_wp,  208.0402_wp,  189.8824_wp,  173.9517_wp, &
      &  225.2381_wp,  211.9709_wp,  205.8954_wp,  187.9096_wp,  172.1318_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  184.5845_wp,  174.2639_wp,  169.5457_wp,  155.4627_wp,  143.0794_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1390.2912_wp, 1259.5964_wp, 1198.2993_wp, 1031.5350_wp,  891.2091_wp, &
      &  456.4613_wp,  422.0055_wp,  406.0845_wp,  360.9675_wp,  322.1273_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1123.0616_wp, 1026.6097_wp,  981.8206_wp,  857.4537_wp,  751.4576_wp, &
      &  896.4155_wp,  819.9926_wp,  784.4219_wp,  685.8003_wp,  601.8691_wp, &
      &  464.1193_wp,  430.8853_wp,  415.5727_wp,  371.7123_wp,  333.7206_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  923.0182_wp,  845.5447_wp,  809.6247_wp,  709.4418_wp,  623.8277_wp, &
      &  451.1340_wp,  419.1432_wp,  404.4096_wp,  362.1394_wp,  325.5023_wp, &
      &  443.0605_wp,  412.3939_wp,  398.2795_wp,  357.6129_wp,  322.2908_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  886.6318_wp,  813.8964_wp,  780.2017_wp,  685.8851_wp,  605.1321_wp, &
      &  435.1948_wp,  404.4378_wp,  390.2655_wp,  349.6106_wp,  314.3804_wp, &
      &  448.2770_wp,  417.3325_wp,  403.0919_wp,  362.0440_wp,  326.3851_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  809.4134_wp,  743.9253_wp,  713.6060_wp,  628.5419_wp,  555.6200_wp, &
      &  426.5933_wp,  396.4079_wp,  382.4997_wp,  342.6120_wp,  308.0520_wp, &
      &  410.3518_wp,  381.9818_wp,  368.9209_wp,  331.3093_wp,  298.6591_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  636.6561_wp,  585.4069_wp,  561.6694_wp,  495.0936_wp,  438.0628_wp, &
      &  423.3208_wp,  393.1215_wp,  379.2021_wp,  339.3449_wp,  304.8385_wp, &
      &  305.5108_wp,  285.6107_wp,  276.4674_wp,  249.8939_wp,  226.7483_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  693.7442_wp,  638.7587_wp,  613.3230_wp,  541.7184_wp,  480.2245_wp, &
      &  386.6184_wp,  360.1114_wp,  347.9151_wp,  312.7295_wp,  282.1503_wp, &
      &  259.5584_wp,  243.2984_wp,  235.8366_wp,  214.0118_wp,  194.9556_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  543.3629_wp,  500.6304_wp,  480.8588_wp,  425.1904_wp,  377.4064_wp, &
      &  356.9159_wp,  332.7595_wp,  321.6484_wp,  289.5301_wp,  261.5954_wp, &
      &  255.0100_wp,  238.9587_wp,  231.5914_wp,  210.0623_wp,  191.2703_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  572.8169_wp,  528.9965_wp,  508.7598_wp,  451.4360_wp,  402.0434_wp, &
      &  340.4265_wp,  317.3978_wp,  306.8041_wp,  276.1884_wp,  249.5667_wp, &
      &  302.4131_wp,  282.1750_wp,  272.8668_wp,  245.9388_wp,  222.5235_wp, &
      &  267.2082_wp,  250.0264_wp,  242.1351_wp,  219.1541_wp,  199.1182_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  592.0464_wp,  546.0473_wp,  524.7928_wp,  464.7381_wp,  413.0565_wp, &
      &  333.0301_wp,  310.4237_wp,  300.0232_wp,  269.9857_wp,  243.8751_wp, &
      &  280.8455_wp,  262.3696_wp,  253.8779_wp,  229.2387_wp,  207.7855_wp, &
      &  296.8872_wp,  276.6393_wp,  267.3225_wp,  240.4605_wp,  217.1392_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  453.2089_wp,  418.1730_wp,  401.9752_wp,  356.2413_wp,  316.9274_wp, &
      &  331.0063_wp,  307.8194_wp,  297.1406_wp,  266.4758_wp,  239.8928_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1254 * max_ref * max_ref + 1 : 1273 * max_ref * max_ref) = [ &
      &  461.1971_wp,  427.1025_wp,  411.3778_wp,  366.5785_wp,  327.8656_wp, &
      &  307.2635_wp,  286.9464_wp,  277.6083_wp,  250.5077_wp,  226.8951_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  548.0549_wp,  506.8838_wp,  487.8786_wp,  433.8635_wp,  387.2413_wp, &
      &  513.0315_wp,  475.1014_wp,  457.6031_wp,  407.7353_wp,  364.6366_wp, &
      &  401.7251_wp,  373.7047_wp,  360.8017_wp,  323.6722_wp,  291.4396_wp, &
      &  324.8229_wp,  303.5179_wp,  293.7290_wp,  265.2550_wp,  240.4178_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  477.1034_wp,  443.4892_wp,  428.0103_wp,  383.5189_wp,  344.9065_wp, &
      &  444.9048_wp,  414.0613_wp,  399.8646_wp,  358.9520_wp,  323.4073_wp, &
      &  422.6322_wp,  393.6595_wp,  380.3291_wp,  341.8407_wp,  308.3757_wp, &
      &  376.6108_wp,  351.5201_wp,  339.9859_wp,  306.5241_wp,  277.3717_wp, &
      &  330.1301_wp,  308.9411_wp,  299.2117_wp,  270.8052_wp,  245.9922_wp, &
      &  401.9415_wp,  375.3694_wp,  363.1576_wp,  327.6790_wp,  296.7505_wp, &
      &  394.8165_wp,  368.8196_wp,  356.8734_wp,  322.1450_wp,  291.8644_wp, &
      &  383.9466_wp,  358.8398_wp,  347.3049_wp,  313.7342_wp,  284.4506_wp, &
      &  369.1013_wp,  345.2176_wp,  334.2483_wp,  302.2666_wp,  274.3493_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  357.7871_wp,  335.1182_wp,  324.7135_wp,  294.2742_wp,  267.6724_wp, &
      &  357.9609_wp,  335.2667_wp,  324.8502_wp,  294.3804_wp,  267.7542_wp, &
      &  355.0652_wp,  332.5825_wp,  322.2635_wp,  292.0737_wp,  265.6909_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  310.5093_wp,  291.7179_wp,  283.1051_wp,  257.7196_wp,  235.4797_wp, &
      &  312.7994_wp,  293.8239_wp,  285.1261_wp,  259.5011_wp,  237.0549_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  267.9532_wp,  252.4408_wp,  245.3409_wp,  224.2655_wp,  205.7618_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1554.3698_wp, 1409.5386_wp, 1341.6152_wp, 1156.6447_wp, 1000.9319_wp, &
      &  542.8777_wp,  502.4986_wp,  483.8461_wp,  430.8811_wp,  385.2516_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1338.5665_wp, 1223.3073_wp, 1169.7495_wp, 1021.1992_wp,  894.7056_wp, &
      & 1091.6636_wp,  998.1919_wp,  954.6446_wp,  834.0979_wp,  731.6257_wp, &
      &  555.3233_wp,  515.7606_wp,  497.5316_wp,  445.2988_wp,  400.0650_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1161.3674_wp, 1065.4342_wp, 1020.9627_wp,  896.6822_wp,  790.3938_wp, &
      &  818.4130_wp,  754.1146_wp,  724.2934_wp,  640.4740_wp,  568.6159_wp, &
      &  529.9186_wp,  493.8129_wp,  477.2036_wp,  429.2279_wp,  387.5181_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1037.2812_wp,  953.9322_wp,  915.3455_wp,  806.9968_wp,  714.0953_wp, &
      &  596.8919_wp,  554.4107_wp,  534.8274_wp,  478.7355_wp,  430.1818_wp, &
      &  533.0732_wp,  497.2424_wp,  480.7668_wp,  433.0658_wp,  391.5532_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  941.7792_wp,  867.5475_wp,  833.2050_wp,  736.4802_wp,  653.4187_wp, &
      &  602.9692_wp,  559.9850_wp,  540.1792_wp,  483.4478_wp,  434.3422_wp, &
      &  515.7374_wp,  480.9564_wp,  464.9556_wp,  418.6918_wp,  378.4693_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  721.2906_wp,  666.4197_wp,  641.0643_wp,  569.2766_wp,  507.4919_wp, &
      &  560.5104_wp,  521.1563_wp,  503.0236_wp,  450.9840_wp,  405.9028_wp, &
      &  452.4616_wp,  423.1570_wp,  409.6963_wp,  370.5143_wp,  336.3592_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  807.2358_wp,  744.9993_wp,  716.2239_wp,  634.9209_wp,  565.0031_wp, &
      &  634.7831_wp,  587.9355_wp,  566.2902_wp,  504.7650_wp,  451.7317_wp, &
      &  396.8920_wp,  372.0410_wp,  360.6378_wp,  327.2637_wp,  298.1122_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  609.6152_wp,  564.4589_wp,  543.6104_wp,  484.3486_wp,  433.2544_wp, &
      &  484.8855_wp,  452.1682_wp,  437.1184_wp,  393.6270_wp,  355.8286_wp, &
      &  374.1870_wp,  351.0591_wp,  340.4508_wp,  309.3410_wp,  282.1480_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  666.8954_wp,  617.3791_wp,  594.5277_wp,  529.5371_wp,  473.4591_wp, &
      &  453.1794_wp,  422.7438_wp,  408.7434_wp,  368.2710_wp,  333.0973_wp, &
      &  383.7851_wp,  359.1380_wp,  347.8166_wp,  314.8522_wp,  286.1251_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  616.7522_wp,  571.5602_wp,  550.7159_wp,  491.3041_wp,  439.9855_wp, &
      &  420.1268_wp,  392.1471_wp,  379.2791_wp,  342.0404_wp,  309.6665_wp, &
      &  401.8433_wp,  375.7345_wp,  363.7391_wp,  328.8698_wp,  298.4950_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  514.4477_wp,  476.5966_wp,  459.1219_wp,  409.4318_wp,  366.5943_wp, &
      &  406.9029_wp,  379.6335_wp,  367.0907_wp,  330.8354_wp,  299.3318_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  541.5490_wp,  502.5513_wp,  484.5759_wp,  433.1992_wp,  388.7621_wp, &
      &  400.3474_wp,  374.3430_wp,  362.3966_wp,  327.6519_wp,  297.3658_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1273 * max_ref * max_ref + 1 : 1292 * max_ref * max_ref) = [ &
      &  685.6138_wp,  634.1195_wp,  610.3382_wp,  542.8413_wp,  484.6532_wp, &
      &  651.1403_wp,  603.0965_wp,  580.9260_wp,  517.8029_wp,  463.2993_wp, &
      &  517.1479_wp,  481.4185_wp,  464.9677_wp,  417.6093_wp,  376.5051_wp, &
      &  435.6252_wp,  407.2275_wp,  394.1807_wp,  356.2359_wp,  323.1509_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  627.5677_wp,  582.7693_wp,  562.1276_wp,  502.9891_wp,  451.7578_wp, &
      &  582.7693_wp,  542.0439_wp,  523.2910_wp,  469.3751_wp,  422.5950_wp, &
      &  562.1276_wp,  523.2910_wp,  505.4148_wp,  453.9195_wp,  409.2008_wp, &
      &  502.9891_wp,  469.3751_wp,  453.9195_wp,  409.1453_wp,  370.1674_wp, &
      &  451.7578_wp,  422.5950_wp,  409.2008_wp,  370.1674_wp,  336.1020_wp, &
      &   37.7681_wp,   37.1186_wp,   36.2277_wp,   35.0939_wp,    0.0000_wp, &
      &   60.9470_wp,   59.7796_wp,   58.1962_wp,   56.1946_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   24.8583_wp,   24.5068_wp,   24.0141_wp,   23.3792_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  586.6888_wp,  569.5020_wp,  546.8547_wp,  518.6427_wp,    0.0000_wp, &
      &  197.7698_wp,  193.0801_wp,  186.8353_wp,  179.0274_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  340.3030_wp,  332.0665_wp,  321.1170_wp,  307.4421_wp,    0.0000_wp, &
      &  252.1649_wp,  246.3580_wp,  238.6037_wp,  228.8894_wp,    0.0000_wp, &
      &  164.5348_wp,  161.2234_wp,  156.7541_wp,  151.1215_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  229.0421_wp,  224.2438_wp,  217.7905_wp,  209.6745_wp,    0.0000_wp, &
      &  192.5542_wp,  188.6577_wp,  183.4000_wp,  176.7743_wp,    0.0000_wp, &
      &  161.5570_wp,  158.4037_wp,  154.1335_wp,  148.7395_wp,    0.0000_wp, &
      &  122.6873_wp,  120.5099_wp,  117.5338_wp,  113.7542_wp,    0.0000_wp, &
      &  115.4018_wp,  113.3867_wp,  110.6276_wp,  107.1200_wp,    0.0000_wp, &
      &  154.4851_wp,  151.6334_wp,  147.7519_wp,  142.8348_wp,    0.0000_wp, &
      &  144.8462_wp,  142.1737_wp,  138.5349_wp,  133.9240_wp,    0.0000_wp, &
      &  118.0337_wp,  115.9742_wp,  113.1548_wp,  109.5710_wp,    0.0000_wp, &
      &  110.4089_wp,  108.4883_wp,  105.8584_wp,  102.5148_wp,    0.0000_wp, &
      &   90.1797_wp,   88.7434_wp,   86.7574_wp,   84.2177_wp,    0.0000_wp, &
      &  107.8436_wp,  106.0538_wp,  103.5908_wp,  100.4500_wp,    0.0000_wp, &
      &  100.5182_wp,   98.8583_wp,   96.5724_wp,   93.6562_wp,    0.0000_wp, &
      &   94.3654_wp,   92.8262_wp,   90.7043_wp,   87.9958_wp,    0.0000_wp, &
      &   83.0577_wp,   81.7279_wp,   79.8912_wp,   77.5440_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   81.5358_wp,   80.2899_wp,   78.5600_wp,   76.3424_wp,    0.0000_wp, &
      &   73.0718_wp,   71.9816_wp,   70.4634_wp,   68.5141_wp,    0.0000_wp, &
      &   64.4291_wp,   63.4972_wp,   62.1954_wp,   60.5209_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   61.6912_wp,   60.8174_wp,   59.5937_wp,   58.0176_wp,    0.0000_wp, &
      &   50.1473_wp,   49.4970_wp,   48.5766_wp,   47.3835_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   47.4073_wp,   46.7775_wp,   45.8894_wp,   44.7408_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  701.6903_wp,  681.3650_wp,  654.5716_wp,  621.1943_wp,    0.0000_wp, &
      &  288.9888_wp,  281.9923_wp,  272.6893_wp,  261.0668_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  541.9826_wp,  528.3545_wp,  510.2824_wp,  487.7418_wp,    0.0000_wp, &
      &  446.9230_wp,  435.9923_wp,  421.4626_wp,  403.3092_wp,    0.0000_wp, &
      &  292.0246_wp,  285.7251_wp,  277.2781_wp,  266.6745_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  499.7292_wp,  487.9847_wp,  472.3352_wp,  452.7605_wp,    0.0000_wp, &
      &  452.8494_wp,  442.3946_wp,  428.4437_wp,  410.9788_wp,    0.0000_wp, &
      &  352.7849_wp,  345.1303_wp,  334.8650_wp,  321.9748_wp,    0.0000_wp, &
      &  274.7333_wp,  269.2400_wp,  261.8224_wp,  252.4713_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  393.6899_wp,  385.3102_wp,  374.0566_wp,  359.9159_wp,    0.0000_wp, &
      &  363.7846_wp,  356.1681_wp,  345.9236_wp,  333.0384_wp,    0.0000_wp, &
      &  338.3189_wp,  331.3449_wp,  321.9519_wp,  310.1284_wp,    0.0000_wp, &
      &  329.6625_wp,  322.8072_wp,  313.5794_wp,  301.9669_wp,    0.0000_wp, &
      &  271.0387_wp,  265.8053_wp,  258.7140_wp,  249.7552_wp,    0.0000_wp, &
      &  306.6912_wp,  300.7558_wp,  292.7145_wp,  282.5564_wp,    0.0000_wp, &
      &  297.6598_wp,  291.9225_wp,  284.1462_wp,  274.3202_wp,    0.0000_wp, &
      &  293.6168_wp,  287.9266_wp,  280.2178_wp,  270.4800_wp,    0.0000_wp, &
      &  272.2813_wp,  267.1401_wp,  260.1578_wp,  251.3243_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  254.2418_wp,  249.6469_wp,  243.3785_wp,  235.4268_wp,    0.0000_wp, &
      &  251.4344_wp,  246.8844_wp,  240.6781_wp,  232.8057_wp,    0.0000_wp, &
      &  246.3350_wp,  241.8844_wp,  235.8128_wp,  228.1108_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  207.4043_wp,  203.9103_wp,  199.1082_wp,  192.9893_wp,    0.0000_wp, &
      &  205.2512_wp,  201.7898_wp,  197.0330_wp,  190.9724_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1292 * max_ref * max_ref + 1 : 1311 * max_ref * max_ref) = [ &
      &  169.4533_wp,  166.7796_wp,  163.0783_wp,  158.3420_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1147.6700_wp, 1113.9369_wp, 1069.4211_wp, 1013.8323_wp,    0.0000_wp, &
      &  397.8416_wp,  388.7844_wp,  376.6755_wp,  361.4928_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  950.6543_wp,  925.3266_wp,  891.8232_wp,  850.0618_wp,    0.0000_wp, &
      &  759.7988_wp,  739.8380_wp,  713.3777_wp,  680.3184_wp,    0.0000_wp, &
      &  409.0173_wp,  400.2337_wp,  388.4468_wp,  373.6423_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  785.8323_wp,  765.4303_wp,  738.4121_wp,  704.7245_wp,    0.0000_wp, &
      &  398.3152_wp,  389.8645_wp,  378.5124_wp,  364.2456_wp,    0.0000_wp, &
      &  393.0115_wp,  384.9081_wp,  373.9963_wp,  360.2627_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  758.9406_wp,  739.7751_wp,  714.3497_wp,  682.6170_wp,    0.0000_wp, &
      &  384.4487_wp,  376.3336_wp,  365.4257_wp,  351.7089_wp,    0.0000_wp, &
      &  397.8333_wp,  389.6578_wp,  378.6459_wp,  364.7838_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  695.0683_wp,  677.8014_wp,  654.8718_wp,  626.2392_wp,    0.0000_wp, &
      &  376.7544_wp,  368.7908_wp,  358.0883_wp,  344.6316_wp,    0.0000_wp, &
      &  364.0189_wp,  356.5349_wp,  346.4532_wp,  333.7597_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  547.1981_wp,  533.7253_wp,  515.8152_wp,  493.4310_wp,    0.0000_wp, &
      &  373.2591_wp,  365.2940_wp,  354.5977_wp,  341.1548_wp,    0.0000_wp, &
      &  273.8552_wp,  268.6291_wp,  261.5413_wp,  252.5817_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  598.5168_wp,  584.0079_wp,  564.7110_wp,  540.5946_wp,    0.0000_wp, &
      &  343.5205_wp,  336.5178_wp,  327.0807_wp,  315.1977_wp,    0.0000_wp, &
      &  234.1750_wp,  229.9158_wp,  224.1126_wp,  216.7567_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  469.4569_wp,  458.2135_wp,  443.2412_wp,  424.5124_wp,    0.0000_wp, &
      &  317.8624_wp,  311.4859_wp,  302.8803_wp,  292.0345_wp,    0.0000_wp, &
      &  229.8877_wp,  225.6812_wp,  219.9538_wp,  212.6969_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  498.0472_wp,  486.4645_wp,  471.0186_wp,  451.6893_wp,    0.0000_wp, &
      &  303.1856_wp,  297.1110_wp,  288.9115_wp,  278.5760_wp,    0.0000_wp, &
      &  269.8010_wp,  264.4748_wp,  257.2754_wp,  248.1924_wp,    0.0000_wp, &
      &  240.0396_wp,  235.5276_wp,  229.4010_wp,  221.6512_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  513.0585_wp,  500.9011_wp,  484.7111_wp,  464.4665_wp,    0.0000_wp, &
      &  296.4064_wp,  290.4433_wp,  282.3973_wp,  272.2578_wp,    0.0000_wp, &
      &  251.3164_wp,  246.4555_wp,  239.8735_wp,  231.5614_wp,    0.0000_wp, &
      &  263.9496_wp,  258.6194_wp,  251.4301_wp,  242.3721_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  393.0249_wp,  383.8010_wp,  371.5028_wp,  356.1095_wp,    0.0000_wp, &
      &  292.8589_wp,  286.7436_wp,  278.5192_wp,  268.1748_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  403.8621_wp,  394.8434_wp,  382.7819_wp,  367.6637_wp,    0.0000_wp, &
      &  274.7816_wp,  269.4208_wp,  262.1684_wp,  253.0149_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  478.3993_wp,  467.5119_wp,  452.9662_wp,  434.7417_wp,    0.0000_wp, &
      &  449.3006_wp,  439.2692_wp,  425.8475_wp,  409.0175_wp,    0.0000_wp, &
      &  355.8474_wp,  348.4425_wp,  338.4772_wp,  325.9378_wp,    0.0000_wp, &
      &  290.9600_wp,  285.3363_wp,  277.7186_wp,  268.0968_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  421.8616_wp,  412.9661_wp,  401.0098_wp,  385.9785_wp,    0.0000_wp, &
      &  394.5822_wp,  386.4260_wp,  375.4438_wp,  361.6218_wp,    0.0000_wp, &
      &  375.6047_wp,  367.9463_wp,  357.6221_wp,  344.6189_wp,    0.0000_wp, &
      &  336.4289_wp,  329.8047_wp,  320.8460_wp,  309.5410_wp,    0.0000_wp, &
      &  296.8145_wp,  291.2302_wp,  283.6451_wp,  274.0484_wp,    0.0000_wp, &
      &  359.5564_wp,  352.5427_wp,  343.0487_wp,  331.0618_wp,    0.0000_wp, &
      &  353.4245_wp,  346.5656_wp,  337.2763_wp,  325.5439_wp,    0.0000_wp, &
      &  344.1036_wp,  337.4828_wp,  328.5083_wp,  317.1679_wp,    0.0000_wp, &
      &  331.3946_wp,  325.0999_wp,  316.5569_wp,  305.7538_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  322.3608_wp,  316.3984_wp,  308.2846_wp,  298.0072_wp,    0.0000_wp, &
      &  322.4812_wp,  316.5125_wp,  308.3906_wp,  298.1033_wp,    0.0000_wp, &
      &  319.9354_wp,  314.0234_wp,  305.9773_wp,  295.7851_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  281.8050_wp,  276.8856_wp,  270.1506_wp,  261.5889_wp,    0.0000_wp, &
      &  283.7761_wp,  278.8076_wp,  272.0076_wp,  263.3649_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  244.7902_wp,  240.7515_wp,  235.1882_wp,  228.0904_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1311 * max_ref * max_ref + 1 : 1330 * max_ref * max_ref) = [ &
      & 1286.1750_wp, 1248.8109_wp, 1199.4646_wp, 1137.8099_wp,    0.0000_wp, &
      &  474.5641_wp,  463.9632_wp,  449.7669_wp,  431.9480_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1132.1713_wp, 1101.9644_wp, 1061.9947_wp, 1012.1518_wp,    0.0000_wp, &
      &  924.1369_wp,  899.7796_wp,  867.4828_wp,  827.1106_wp,    0.0000_wp, &
      &  489.8228_wp,  479.3863_wp,  465.3683_wp,  447.7500_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  992.4001_wp,  967.1672_wp,  933.6956_wp,  891.9140_wp,    0.0000_wp, &
      &  707.0933_wp,  690.2435_wp,  667.7703_wp,  639.6032_wp,    0.0000_wp, &
      &  471.4064_wp,  461.8766_wp,  449.0205_wp,  432.8219_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  892.0877_wp,  870.1334_wp,  840.9533_wp,  804.4937_wp,    0.0000_wp, &
      &  526.5426_wp,  515.3541_wp,  500.3168_wp,  481.4071_wp,    0.0000_wp, &
      &  475.3821_wp,  465.9289_wp,  453.1570_wp,  437.0500_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  813.4345_wp,  793.8771_wp,  767.8415_wp,  735.2812_wp,    0.0000_wp, &
      &  531.7424_wp,  520.4182_wp,  505.2035_wp,  486.0786_wp,    0.0000_wp, &
      &  459.5750_wp,  450.4159_wp,  438.0410_wp,  422.4320_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  627.6849_wp,  613.2470_wp,  593.9624_wp,  569.7978_wp,    0.0000_wp, &
      &  495.6948_wp,  485.3374_wp,  471.3990_wp,  453.8578_wp,    0.0000_wp, &
      &  406.0312_wp,  398.3318_wp,  387.8818_wp,  374.6663_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  700.5163_wp,  684.1292_wp,  662.2699_wp,  634.8991_wp,    0.0000_wp, &
      &  555.7794_wp,  543.4822_wp,  527.0000_wp,  506.2976_wp,    0.0000_wp, &
      &  358.1519_wp,  351.6390_wp,  342.7635_wp,  331.5119_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  533.3804_wp,  521.5071_wp,  505.6089_wp,  485.6592_wp,    0.0000_wp, &
      &  431.9865_wp,  423.3751_wp,  411.7430_wp,  397.0742_wp,    0.0000_wp, &
      &  338.3541_wp,  332.2995_wp,  324.0357_wp,  313.5500_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  583.3346_wp,  570.2790_wp,  552.8128_wp,  530.9113_wp,    0.0000_wp, &
      &  404.0384_wp,  396.0359_wp,  385.2196_wp,  371.5740_wp,    0.0000_wp, &
      &  344.8082_wp,  338.3472_wp,  329.5690_wp,  318.4602_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  540.9125_wp,  528.9954_wp,  513.0345_wp,  493.0084_wp,    0.0000_wp, &
      &  375.0995_wp,  367.7493_wp,  357.8055_wp,  345.2533_wp,    0.0000_wp, &
      &  360.3420_wp,  353.4857_wp,  344.1857_wp,  332.4291_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  450.6427_wp,  440.7041_wp,  427.3871_wp,  410.6685_wp,    0.0000_wp, &
      &  362.8740_wp,  355.7080_wp,  346.0216_wp,  333.8012_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  476.5783_wp,  466.2941_wp,  452.4998_wp,  435.1777_wp,    0.0000_wp, &
      &  359.0555_wp,  352.2132_wp,  342.9362_wp,  331.2119_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  598.2891_wp,  584.7110_wp,  566.5642_wp,  543.8198_wp,    0.0000_wp, &
      &  570.3015_wp,  557.6272_wp,  540.6624_wp,  519.3820_wp,    0.0000_wp, &
      &  458.7676_wp,  449.3448_wp,  436.6520_wp,  420.6714_wp,    0.0000_wp, &
      &  390.5107_wp,  383.0280_wp,  372.8876_wp,  360.0757_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  553.3592_wp,  541.5209_wp,  525.6310_wp,  505.6697_wp,    0.0000_wp, &
      &  515.9467_wp,  505.1900_wp,  490.7203_wp,  472.5189_wp,    0.0000_wp, &
      &  498.7395_wp,  488.4839_wp,  474.6720_wp,  457.2860_wp,    0.0000_wp, &
      &  448.9769_wp,  440.1085_wp,  428.1222_wp,  413.0024_wp,    0.0000_wp, &
      &  405.6849_wp,  398.0011_wp,  387.5752_wp,  374.3929_wp,    0.0000_wp, &
      &  492.9379_wp,  483.0596_wp,  469.7255_wp,  452.9187_wp,    0.0000_wp, &
      &  483.0596_wp,  473.4703_wp,  460.5150_wp,  444.1770_wp,    0.0000_wp, &
      &  469.7255_wp,  460.5150_wp,  448.0573_wp,  432.3357_wp,    0.0000_wp, &
      &  452.9187_wp,  444.1770_wp,  432.3357_wp,  417.3790_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   35.4841_wp,   35.4861_wp,   35.2461_wp,    0.0000_wp,    0.0000_wp, &
      &   56.7427_wp,   56.7156_wp,   56.2769_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   23.6911_wp,   23.7100_wp,   23.5851_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  521.2511_wp,  519.4518_wp,  512.7164_wp,    0.0000_wp,    0.0000_wp, &
      &  180.3056_wp,  179.9812_wp,  178.1745_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  309.5439_wp,  308.9426_wp,  305.7629_wp,    0.0000_wp,    0.0000_wp, &
      &  230.6309_wp,  230.2641_wp,  228.0340_wp,    0.0000_wp,    0.0000_wp, &
      &  152.4956_wp,  152.3806_wp,  151.1275_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1330 * max_ref * max_ref + 1 : 1349 * max_ref * max_ref) = [ &
      &  211.4898_wp,  211.2799_wp,  209.4559_wp,    0.0000_wp,    0.0000_wp, &
      &  178.3919_wp,  178.2509_wp,  176.7765_wp,    0.0000_wp,    0.0000_wp, &
      &  150.1754_wp,  150.0870_wp,  148.8998_wp,    0.0000_wp,    0.0000_wp, &
      &  114.9752_wp,  114.9640_wp,  114.1553_wp,    0.0000_wp,    0.0000_wp, &
      &  108.2922_wp,  108.2898_wp,  107.5434_wp,    0.0000_wp,    0.0000_wp, &
      &  144.3032_wp,  144.2600_wp,  143.1947_wp,    0.0000_wp,    0.0000_wp, &
      &  135.3122_wp,  135.2720_wp,  134.2742_wp,    0.0000_wp,    0.0000_wp, &
      &  110.7747_wp,  110.7711_wp,  110.0087_wp,    0.0000_wp,    0.0000_wp, &
      &  103.6503_wp,  103.6477_wp,  102.9373_wp,    0.0000_wp,    0.0000_wp, &
      &   85.2343_wp,   85.2657_wp,   84.7425_wp,    0.0000_wp,    0.0000_wp, &
      &  101.6124_wp,  101.6315_wp,  100.9745_wp,    0.0000_wp,    0.0000_wp, &
      &   94.7508_wp,   94.7705_wp,   94.1622_wp,    0.0000_wp,    0.0000_wp, &
      &   89.0359_wp,   89.0582_wp,   88.4953_wp,    0.0000_wp,    0.0000_wp, &
      &   78.4819_wp,   78.5067_wp,   78.0224_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   77.2970_wp,   77.3365_wp,   76.8863_wp,    0.0000_wp,    0.0000_wp, &
      &   69.3901_wp,   69.4317_wp,   69.0399_wp,    0.0000_wp,    0.0000_wp, &
      &   61.3159_wp,   61.3585_wp,   61.0258_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   58.7881_wp,   58.8332_wp,   58.5223_wp,    0.0000_wp,    0.0000_wp, &
      &   48.0535_wp,   48.1037_wp,   47.8771_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   45.3613_wp,   45.4044_wp,   45.1836_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  624.4076_wp,  622.3099_wp,  614.3523_wp,    0.0000_wp,    0.0000_wp, &
      &  262.8846_wp,  262.3697_wp,  259.6698_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  490.8760_wp,  489.7820_wp,  484.5038_wp,    0.0000_wp,    0.0000_wp, &
      &  406.0728_wp,  405.2529_wp,  401.0304_wp,    0.0000_wp,    0.0000_wp, &
      &  268.8595_wp,  268.5418_wp,  266.1377_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  456.0400_wp,  455.2477_wp,  450.7267_wp,    0.0000_wp,    0.0000_wp, &
      &  414.0544_wp,  413.3856_wp,  409.3685_wp,    0.0000_wp,    0.0000_wp, &
      &  324.6278_wp,  324.2349_wp,  321.3129_wp,    0.0000_wp,    0.0000_wp, &
      &  254.7805_wp,  254.5963_wp,  252.5189_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  362.9518_wp,  362.5561_wp,  359.3640_wp,    0.0000_wp,    0.0000_wp, &
      &  335.9240_wp,  335.5914_wp,  332.6960_wp,    0.0000_wp,    0.0000_wp, &
      &  312.8750_wp,  312.5939_wp,  309.9478_wp,    0.0000_wp,    0.0000_wp, &
      &  304.6285_wp,  304.3388_wp,  301.7357_wp,    0.0000_wp,    0.0000_wp, &
      &  252.1604_wp,  252.0272_wp,  250.0579_wp,    0.0000_wp,    0.0000_wp, &
      &  285.2779_wp,  285.1239_wp,  282.8902_wp,    0.0000_wp,    0.0000_wp, &
      &  276.9809_wp,  276.8372_wp,  274.6795_wp,    0.0000_wp,    0.0000_wp, &
      &  273.0920_wp,  272.9417_wp,  270.8005_wp,    0.0000_wp,    0.0000_wp, &
      &  253.8269_wp,  253.7228_wp,  251.7949_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  237.9018_wp,  237.8571_wp,  236.1456_wp,    0.0000_wp,    0.0000_wp, &
      &  235.2519_wp,  235.2059_wp,  233.5109_wp,    0.0000_wp,    0.0000_wp, &
      &  230.5137_wp,  230.4702_wp,  228.8128_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  195.1861_wp,  195.2131_wp,  193.9269_wp,    0.0000_wp,    0.0000_wp, &
      &  193.1449_wp,  193.1704_wp,  191.8960_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  160.2674_wp,  160.3331_wp,  159.3608_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1019.3121_wp, 1015.7738_wp, 1002.5559_wp,    0.0000_wp,    0.0000_wp, &
      &  364.4090_wp,  363.8482_wp,  360.3799_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  855.1882_wp,  852.9036_wp,  843.0540_wp,    0.0000_wp,    0.0000_wp, &
      &  684.7073_wp,  682.9621_wp,  675.2096_wp,    0.0000_wp,    0.0000_wp, &
      &  376.8093_wp,  376.3706_wp,  373.0245_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  709.1303_wp,  707.3769_wp,  699.4586_wp,    0.0000_wp,    0.0000_wp, &
      &  367.3908_wp,  366.9900_wp,  363.7755_wp,    0.0000_wp,    0.0000_wp, &
      &  363.4777_wp,  363.1446_wp,  360.0716_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  687.1235_wp,  685.5687_wp,  678.1485_wp,    0.0000_wp,    0.0000_wp, &
      &  354.7793_wp,  354.4029_wp,  351.3179_wp,    0.0000_wp,    0.0000_wp, &
      &  368.0557_wp,  367.7252_wp,  364.6262_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  630.4901_wp,  629.1390_wp,  622.4635_wp,    0.0000_wp,    0.0000_wp, &
      &  347.6374_wp,  347.2648_wp,  344.2372_wp,    0.0000_wp,    0.0000_wp, &
      &  336.7719_wp,  336.4671_wp,  333.6308_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1349 * max_ref * max_ref + 1 : 1368 * max_ref * max_ref) = [ &
      &  496.9143_wp,  495.8801_wp,  490.6786_wp,    0.0000_wp,    0.0000_wp, &
      &  344.0982_wp,  343.7087_wp,  340.6774_wp,    0.0000_wp,    0.0000_wp, &
      &  255.1077_wp,  254.9795_wp,  253.0195_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  544.4156_wp,  543.3438_wp,  537.7470_wp,    0.0000_wp,    0.0000_wp, &
      &  318.0485_wp,  317.7750_wp,  315.1229_wp,    0.0000_wp,    0.0000_wp, &
      &  219.0427_wp,  218.9858_wp,  217.3993_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  427.6439_wp,  426.8361_wp,  422.5068_wp,    0.0000_wp,    0.0000_wp, &
      &  294.7348_wp,  294.5079_wp,  292.0979_wp,    0.0000_wp,    0.0000_wp, &
      &  214.9236_wp,  214.8607_wp,  213.2923_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  455.0836_wp,  454.3170_wp,  449.8664_wp,    0.0000_wp,    0.0000_wp, &
      &  281.1633_wp,  280.9482_wp,  278.6529_wp,    0.0000_wp,    0.0000_wp, &
      &  250.5626_wp,  250.3896_wp,  248.3821_wp,    0.0000_wp,    0.0000_wp, &
      &  223.8964_wp,  223.7999_wp,  222.1108_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  467.8493_wp,  467.0009_wp,  462.3208_wp,    0.0000_wp,    0.0000_wp, &
      &  274.7751_wp,  274.5580_wp,  272.3038_wp,    0.0000_wp,    0.0000_wp, &
      &  233.8236_wp,  233.6876_wp,  231.8603_wp,    0.0000_wp,    0.0000_wp, &
      &  244.6310_wp,  244.4292_wp,  242.4149_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  358.8167_wp,  358.1870_wp,  354.6423_wp,    0.0000_wp,    0.0000_wp, &
      &  270.5526_wp,  270.2783_wp,  267.9569_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  370.5936_wp,  370.0674_wp,  366.6162_wp,    0.0000_wp,    0.0000_wp, &
      &  255.4289_wp,  255.2713_wp,  253.2520_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  438.1102_wp,  437.4417_wp,  433.2665_wp,    0.0000_wp,    0.0000_wp, &
      &  412.2793_wp,  411.7018_wp,  407.8626_wp,    0.0000_wp,    0.0000_wp, &
      &  328.8076_wp,  328.4905_wp,  325.6786_wp,    0.0000_wp,    0.0000_wp, &
      &  270.6774_wp,  270.5297_wp,  268.4136_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  389.2997_wp,  388.8946_wp,  385.5109_wp,    0.0000_wp,    0.0000_wp, &
      &  364.8243_wp,  364.4882_wp,  361.3931_wp,    0.0000_wp,    0.0000_wp, &
      &  347.7286_wp,  347.4358_wp,  344.5346_wp,    0.0000_wp,    0.0000_wp, &
      &  312.4631_wp,  312.2619_wp,  309.7636_wp,    0.0000_wp,    0.0000_wp, &
      &  276.7786_wp,  276.6685_wp,  274.5751_wp,    0.0000_wp,    0.0000_wp, &
      &  334.2234_wp,  334.0263_wp,  331.3844_wp,    0.0000_wp,    0.0000_wp, &
      &  328.6756_wp,  328.4909_wp,  325.9092_wp,    0.0000_wp,    0.0000_wp, &
      &  320.2529_wp,  320.0878_wp,  317.5986_wp,    0.0000_wp,    0.0000_wp, &
      &  308.7739_wp,  308.6359_wp,  306.2735_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  301.0545_wp,  300.9617_wp,  298.7331_wp,    0.0000_wp,    0.0000_wp, &
      &  301.1506_wp,  301.0567_wp,  298.8255_wp,    0.0000_wp,    0.0000_wp, &
      &  298.8156_wp,  298.7247_wp,  296.5153_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  264.4516_wp,  264.4446_wp,  262.6226_wp,    0.0000_wp,    0.0000_wp, &
      &  266.2386_wp,  266.2278_wp,  264.3868_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  230.7426_wp,  230.7947_wp,  229.3135_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1144.1913_wp, 1140.3397_wp, 1125.7134_wp,    0.0000_wp,    0.0000_wp, &
      &  435.5555_wp,  434.9391_wp,  430.8890_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1018.3413_wp, 1015.6122_wp, 1003.8665_wp,    0.0000_wp,    0.0000_wp, &
      &  832.5060_wp,  830.3661_wp,  820.9055_wp,    0.0000_wp,    0.0000_wp, &
      &  451.6295_wp,  451.1258_wp,  447.1555_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  897.8043_wp,  895.7304_wp,  885.9576_wp,    0.0000_wp,    0.0000_wp, &
      &  644.4494_wp,  643.2697_wp,  636.7834_wp,    0.0000_wp,    0.0000_wp, &
      &  436.8022_wp,  436.4520_wp,  432.8478_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  810.1028_wp,  808.4252_wp,  799.9471_wp,    0.0000_wp,    0.0000_wp, &
      &  485.6176_wp,  485.0856_wp,  480.8308_wp,    0.0000_wp,    0.0000_wp, &
      &  441.1513_wp,  440.8392_wp,  437.2712_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  740.6211_wp,  739.2097_wp,  731.6741_wp,    0.0000_wp,    0.0000_wp, &
      &  490.3206_wp,  489.7750_wp,  485.4685_wp,    0.0000_wp,    0.0000_wp, &
      &  426.4202_wp,  426.1118_wp,  422.6552_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  574.2932_wp,  573.3700_wp,  567.8341_wp,    0.0000_wp,    0.0000_wp, &
      &  457.9302_wp,  457.4717_wp,  453.5417_wp,    0.0000_wp,    0.0000_wp, &
      &  378.4312_wp,  378.2579_wp,  375.3725_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1368 * max_ref * max_ref + 1 : 1387 * max_ref * max_ref) = [ &
      &  639.7465_wp,  638.6462_wp,  632.3502_wp,    0.0000_wp,    0.0000_wp, &
      &  510.5682_wp,  509.8781_wp,  505.1830_wp,    0.0000_wp,    0.0000_wp, &
      &  335.0059_wp,  334.9230_wp,  332.4971_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  489.6956_wp,  489.0100_wp,  484.4740_wp,    0.0000_wp,    0.0000_wp, &
      &  400.8319_wp,  400.5365_wp,  397.2874_wp,    0.0000_wp,    0.0000_wp, &
      &  316.9123_wp,  316.8578_wp,  314.6082_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  535.2189_wp,  534.4511_wp,  529.4579_wp,    0.0000_wp,    0.0000_wp, &
      &  375.1303_wp,  374.8656_wp,  371.8493_wp,    0.0000_wp,    0.0000_wp, &
      &  321.7225_wp,  321.5899_wp,  319.1735_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  497.0959_wp,  496.4313_wp,  491.8812_wp,    0.0000_wp,    0.0000_wp, &
      &  348.6074_wp,  348.3802_wp,  345.6140_wp,    0.0000_wp,    0.0000_wp, &
      &  335.7605_wp,  335.5949_wp,  333.0245_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  414.1512_wp,  413.5913_wp,  409.7995_wp,    0.0000_wp,    0.0000_wp, &
      &  337.0131_wp,  336.7771_wp,  334.0775_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  438.8861_wp,  438.3534_wp,  434.4358_wp,    0.0000_wp,    0.0000_wp, &
      &  334.4917_wp,  334.3241_wp,  331.7567_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  548.1267_wp,  547.2913_wp,  542.0891_wp,    0.0000_wp,    0.0000_wp, &
      &  523.6131_wp,  522.8863_wp,  518.0408_wp,    0.0000_wp,    0.0000_wp, &
      &  424.4718_wp,  424.0862_wp,  420.5155_wp,    0.0000_wp,    0.0000_wp, &
      &  363.5974_wp,  363.4077_wp,  360.5967_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  509.9698_wp,  509.3849_wp,  504.8763_wp,    0.0000_wp,    0.0000_wp, &
      &  476.6811_wp,  476.2087_wp,  472.1243_wp,    0.0000_wp,    0.0000_wp, &
      &  461.3868_wp,  460.9671_wp,  457.0793_wp,    0.0000_wp,    0.0000_wp, &
      &  416.8945_wp,  416.6107_wp,  413.2653_wp,    0.0000_wp,    0.0000_wp, &
      &  378.0963_wp,  377.9250_wp,  375.0423_wp,    0.0000_wp,    0.0000_wp, &
      &  457.1057_wp,  456.7589_wp,  453.0254_wp,    0.0000_wp,    0.0000_wp, &
      &  448.3333_wp,  448.0173_wp,  444.3975_wp,    0.0000_wp,    0.0000_wp, &
      &  436.4430_wp,  436.1655_wp,  432.6941_wp,    0.0000_wp,    0.0000_wp, &
      &  421.4187_wp,  421.1871_wp,  417.8991_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  425.5355_wp,  425.3152_wp,  422.0187_wp,    0.0000_wp,    0.0000_wp, &
      &  425.3152_wp,  425.1015_wp,  421.8178_wp,    0.0000_wp,    0.0000_wp, &
      &  422.0187_wp,  421.8178_wp,  418.5789_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   32.5171_wp,   32.8151_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   51.4656_wp,   51.9496_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   22.0813_wp,   22.2717_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  449.8512_wp,  454.3954_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  159.8670_wp,  161.4303_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  273.7398_wp,  276.4309_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  205.2237_wp,  207.2185_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  137.6192_wp,  138.9273_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  190.1094_wp,  191.9273_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  160.9679_wp,  162.4939_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  136.0239_wp,  137.3020_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  105.0660_wp,  106.0350_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   99.1109_wp,  100.0210_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  131.4051_wp,  132.6251_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  123.2538_wp,  124.3958_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  101.4216_wp,  102.3489_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   94.9489_wp,   95.8139_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   78.6686_wp,   79.3716_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   93.4562_wp,   94.2991_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   87.2028_wp,   87.9869_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   82.0374_wp,   82.7713_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   72.4531_wp,   73.0954_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   71.6059_wp,   72.2358_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   64.4141_wp,   64.9764_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   57.0738_wp,   57.5655_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   54.7990_wp,   55.2689_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   45.0848_wp,   45.4618_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1387 * max_ref * max_ref + 1 : 1406 * max_ref * max_ref) = [ &
      &   42.4954_wp,   42.8512_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  539.8536_wp,  545.2855_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  232.6053_wp,  234.8776_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  432.1792_wp,  436.4452_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  358.8052_wp,  362.3258_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  240.8943_wp,  243.2097_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  404.7631_wp,  408.7149_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  368.2792_wp,  371.8615_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  290.7378_wp,  293.5337_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  230.0798_wp,  232.2617_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  325.6912_wp,  328.8157_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  301.9852_wp,  304.8715_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  281.7256_wp,  284.4096_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  274.1045_wp,  276.7161_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  228.5370_wp,  230.6872_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  258.5121_wp,  260.9446_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  251.1081_wp,  253.4677_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  247.4732_wp,  249.7992_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  230.5860_wp,  232.7426_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  217.0337_wp,  219.0430_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  214.6002_wp,  216.5866_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  210.3176_wp,  212.2628_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  179.2306_wp,  180.8604_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  177.3458_wp,  178.9581_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  148.0216_wp,  149.3430_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  880.1741_wp,  889.0026_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  325.1068_wp,  328.2207_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  747.9306_wp,  755.3478_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  600.2698_wp,  606.1859_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  338.0494_wp,  341.2732_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  622.1841_wp,  628.3317_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  330.0418_wp,  333.1801_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  327.4340_wp,  330.5365_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  605.0508_wp,  610.9949_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  318.9171_wp,  321.9433_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  331.6800_wp,  334.8197_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  556.3166_wp,  561.7643_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  312.4714_wp,  315.4349_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  303.5459_wp,  306.4123_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  439.1660_wp,  443.4401_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  309.0063_wp,  311.9395_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  231.7484_wp,  233.8940_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  481.8077_wp,  486.5025_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  286.8720_wp,  289.5784_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  199.8924_wp,  201.7229_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  379.2212_wp,  382.8900_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  266.2910_wp,  268.7934_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  196.0288_wp,  197.8251_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  404.6929_wp,  408.6064_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  254.0877_wp,  256.4721_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  226.8510_wp,  228.9654_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  203.6855_wp,  205.5633_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  415.1303_wp,  419.1592_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  248.2237_wp,  250.5536_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  212.1168_wp,  214.0853_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  221.0250_wp,  223.0896_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1406 * max_ref * max_ref + 1 : 1425 * max_ref * max_ref) = [ &
      &  318.9642_wp,  322.0346_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  243.5475_wp,  245.8435_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  331.0651_wp,  334.2414_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  231.4006_wp,  233.5632_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  390.4542_wp,  394.2289_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  368.2183_wp,  371.7655_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  295.8817_wp,  298.6948_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  245.3469_wp,  247.6485_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  349.7760_wp,  353.1156_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  328.4750_wp,  331.5987_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  313.5280_wp,  316.5011_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  282.7195_wp,  285.3820_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  251.5229_wp,  253.8712_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  302.6743_wp,  305.5214_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  297.8064_wp,  300.6043_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  290.4211_wp,  293.1445_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  280.3577_wp,  282.9800_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  274.0630_wp,  276.6109_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  274.1377_wp,  276.6862_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  272.0583_wp,  274.5862_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  242.0556_wp,  244.2752_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  243.6296_wp,  245.8647_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  212.2820_wp,  214.2004_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  989.8921_wp,  999.7871_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  389.4684_wp,  393.1809_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  890.6723_wp,  899.4930_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  729.7176_wp,  736.9021_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  405.6158_wp,  409.4712_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  790.0257_wp,  797.7901_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  571.8432_wp,  577.3805_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  394.3185_wp,  398.0382_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  715.7318_wp,  722.7240_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  436.3046_wp,  440.4474_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  398.8895_wp,  402.6413_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  656.2276_wp,  662.6068_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  440.4599_wp,  444.6402_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  385.5923_wp,  389.2112_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  511.6772_wp,  516.5932_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  412.2275_wp,  416.1207_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  343.9096_wp,  347.1000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  568.7909_wp,  574.2805_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  456.9707_wp,  461.3223_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  305.6667_wp,  308.4743_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  437.9992_wp,  442.1710_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  362.5374_wp,  365.9282_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  289.6001_wp,  292.2483_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  478.2481_wp,  482.8217_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  339.5620_wp,  342.7279_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  292.8227_wp,  295.5179_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  444.9662_wp,  449.2061_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  315.9288_wp,  318.8629_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  305.1216_wp,  307.9410_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  370.9344_wp,  374.4467_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  305.2122_wp,  308.0475_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1425 * max_ref * max_ref + 1 : 1444 * max_ref * max_ref) = [ &
      &  393.7655_wp,  397.4979_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  303.8410_wp,  306.6549_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  488.9709_wp,  493.6625_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  468.1680_wp,  472.6441_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  382.6809_wp,  386.2858_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  330.0106_wp,  333.0801_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  457.7156_wp,  462.0716_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  428.9963_wp,  433.0589_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  415.8173_wp,  419.7450_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  377.2255_wp,  380.7615_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  343.4912_wp,  346.6856_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  412.9852_wp,  416.8715_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  405.4400_wp,  409.2485_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  395.1626_wp,  398.8660_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  382.1376_wp,  385.7084_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  386.0997_wp,  389.7029_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  385.9988_wp,  389.5996_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  383.1837_wp,  386.7550_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  351.9667_wp,  355.2175_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  355.2175_wp,  358.4995_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   29.6055_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   46.3794_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   20.4594_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  386.9969_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  140.9818_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  240.7689_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  181.6260_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  123.4070_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  169.8578_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  144.3697_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  122.4634_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   95.3973_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   90.1296_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  118.9178_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  111.5923_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   92.2876_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   86.4581_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   72.1617_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   85.4293_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   79.7773_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   75.1427_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   66.5081_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   65.9403_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   59.4468_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   52.8269_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   50.7900_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   42.0634_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   39.5951_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  465.2817_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  204.7986_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  378.6134_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  315.4581_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  214.5479_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  357.2960_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  325.7625_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  258.8687_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  206.4581_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1444 * max_ref * max_ref + 1 : 1463 * max_ref * max_ref) = [ &
      &  290.4985_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  269.8372_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  252.1337_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  245.1798_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  205.8095_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  232.7786_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  226.2211_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  222.8660_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  208.1507_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  196.7374_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  194.5237_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  190.6822_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  163.5368_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  161.8119_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  135.8592_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  758.3253_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  288.7275_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  651.5392_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  524.3547_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  301.6273_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  543.5631_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  294.8706_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  293.2741_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  530.4312_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  285.1325_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  297.1871_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  488.6522_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  279.3612_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  272.0795_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  386.5067_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  276.0406_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  209.3622_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  424.4125_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  257.2803_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  181.3837_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  334.8259_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  239.2190_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  177.7905_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  358.0954_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  228.3256_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  204.2663_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  184.2665_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  366.5597_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  222.9856_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  191.3621_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  198.6584_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  282.2888_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  218.0929_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  294.2138_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  208.4055_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  346.1345_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  327.0838_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  264.7126_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  221.0120_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  312.4265_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  293.9966_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  281.0046_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  254.2458_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  227.1324_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1463 * max_ref * max_ref + 1 : 1482 * max_ref * max_ref) = [ &
      &  272.4110_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  268.1700_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  261.7361_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  252.9678_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  247.9275_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  247.9867_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  246.1495_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  220.1581_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  221.5367_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  194.0699_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  854.5223_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  346.6791_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  776.0859_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  637.4660_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  362.3556_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  692.2386_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  505.2262_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  353.9178_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  629.5230_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  389.9385_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  358.5675_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  578.7876_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  393.6121_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  346.7063_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  453.7998_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  369.1531_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  310.7442_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  503.3811_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  407.0970_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  277.2771_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  389.9532_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  326.1157_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  263.1051_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  425.2684_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  305.7201_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  265.0611_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  396.3468_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  284.7964_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  275.7577_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  330.7646_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  274.9744_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  351.5285_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  274.4407_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  434.0932_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  416.5053_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  343.1281_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  297.7796_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  408.5952_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  383.9457_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  372.6480_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  339.3505_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  310.1806_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  370.9572_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  364.5064_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  355.6742_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  344.4458_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1482 * max_ref * max_ref + 1 : 1501 * max_ref * max_ref) = [ &
      &  348.2239_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  348.2164_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  345.8283_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  318.8905_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  321.8029_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  290.2223_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  105.0096_wp,   42.3058_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  184.0223_wp,   68.9537_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   61.3184_wp,   27.4695_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3242.2404_wp,  732.9120_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  752.3418_wp,  231.4165_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1316.2189_wp,  399.4020_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  943.8547_wp,  294.1459_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  517.8726_wp,  187.1318_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  754.9243_wp,  262.2387_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  617.6058_wp,  219.6387_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  506.5054_wp,  183.6568_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  349.4860_wp,  137.8005_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  324.9872_wp,  129.4461_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  455.6960_wp,  174.2718_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  431.5172_wp,  163.6715_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  333.8130_wp,  132.4662_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  313.2347_wp,  123.9803_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  239.2442_wp,  100.4742_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  294.1483_wp,  120.4982_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  275.2567_wp,  112.4278_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  255.7499_wp,  105.3726_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  223.9076_wp,   92.7050_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  210.8672_wp,   90.5303_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  186.8791_wp,   81.0845_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  162.2001_wp,   71.3269_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  152.5672_wp,   68.1239_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  117.9718_wp,   55.0786_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  113.1764_wp,   52.1041_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3798.9439_wp,  873.2665_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1138.9256_wp,  340.0624_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2220.9652_wp,  642.1846_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1799.0180_wp,  527.5630_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  983.9584_wp,  335.2430_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1881.7164_wp,  583.6850_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1673.1922_wp,  527.3282_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1217.3141_wp,  406.3724_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  858.6799_wp,  312.0540_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1317.0056_wp,  451.6171_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1200.5803_wp,  416.4861_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1098.1239_wp,  386.4327_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1088.8566_wp,  377.4920_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  823.8201_wp,  306.7839_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  936.3069_wp,  347.4030_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  907.2993_wp,  337.1135_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  901.7293_wp,  332.8876_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  815.1518_wp,  307.6574_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  735.0061_wp,  286.0418_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  728.4653_wp,  282.9669_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  713.2229_wp,  277.2121_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1501 * max_ref * max_ref + 1 : 1520 * max_ref * max_ref) = [ &
      &  569.6701_wp,  231.9483_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  564.6006_wp,  229.5837_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  445.8171_wp,  188.5950_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 6808.3900_wp, 1449.5860_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1511.8898_wp,  465.5182_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4387.1669_wp, 1149.8295_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3619.2660_wp,  921.7934_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1402.1290_wp,  471.0509_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3450.8369_wp,  942.2745_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1344.5274_wp,  457.9012_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1285.9957_wp,  449.4774_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3215.8544_wp,  904.2225_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1308.6418_wp,  442.0547_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1297.3700_wp,  454.7799_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2877.5279_wp,  824.7780_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1283.1193_wp,  433.3139_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1199.7581_wp,  416.6755_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2283.0674_wp,  650.0627_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1287.9616_wp,  430.1040_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  844.4635_wp,  310.7016_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2396.3445_wp,  706.1412_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1110.3588_wp,  392.0831_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  691.3086_wp,  264.1180_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1884.2824_wp,  554.0925_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1013.0459_wp,  362.0455_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  682.0579_wp,  259.4193_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1875.7209_wp,  581.8458_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  969.5462_wp,  345.4470_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  858.1614_wp,  307.2297_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  727.5711_wp,  271.6004_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1976.8609_wp,  601.5808_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  952.3401_wp,  337.9454_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  780.5520_wp,  285.2553_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  859.7252_wp,  301.6400_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1533.4063_wp,  461.7019_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  983.5916_wp,  336.0821_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1443.0004_wp,  467.8731_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  849.3446_wp,  311.5800_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1755.5105_wp,  556.5016_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1609.8951_wp,  520.7490_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1179.2492_wp,  407.5724_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  887.2092_wp,  329.4410_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1406.2206_wp,  483.7117_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1290.3249_wp,  451.1717_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1210.8497_wp,  428.6170_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1047.5077_wp,  382.0562_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  883.9799_wp,  335.0524_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1108.3145_wp,  407.7958_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1085.3677_wp,  400.6439_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1048.8144_wp,  389.6837_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  997.6271_wp,  374.6769_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  950.1975_wp,  363.5286_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  951.6305_wp,  363.7202_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  943.1487_wp,  360.8053_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  794.2184_wp,  316.0700_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  801.8614_wp,  318.3827_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1520 * max_ref * max_ref + 1 : 1539 * max_ref * max_ref) = [ &
      &  662.9556_wp,  273.2999_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 7560.4361_wp, 1620.7604_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1776.8880_wp,  553.9780_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 5305.1627_wp, 1372.8515_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4496.9846_wp, 1124.9345_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1681.3249_wp,  564.3454_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4289.6097_wp, 1186.3885_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2950.2475_wp,  837.6135_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1515.3755_wp,  537.9093_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3674.8556_wp, 1057.7660_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1819.0406_wp,  607.1951_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1502.0119_wp,  541.1259_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3256.3633_wp,  959.7908_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1826.4461_wp,  613.2010_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1473.7970_wp,  524.1562_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2398.7520_wp,  735.1498_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1687.7490_wp,  570.2621_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1239.1934_wp,  460.1263_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2726.0486_wp,  822.5829_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2068.8557_wp,  647.6686_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1053.7818_wp,  403.9077_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1968.3066_wp,  621.1968_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1387.1242_wp,  492.8301_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  982.3016_wp,  380.9010_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2129.9275_wp,  678.2767_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1295.8507_wp,  460.8194_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1051.3610_wp,  390.6048_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1935.8392_wp,  627.0057_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1193.7040_wp,  427.3247_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1107.7988_wp,  408.6188_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1657.7829_wp,  524.5075_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1162.5520_wp,  413.7796_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1663.0447_wp,  550.2565_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1094.9943_wp,  406.5459_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2226.4867_wp,  697.3745_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2062.6273_wp,  661.8423_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1514.1166_wp,  525.0776_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1190.0927_wp,  442.0537_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1891.9207_wp,  636.7612_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1716.1526_wp,  591.2764_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1633.6348_wp,  570.2993_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1409.0375_wp,  510.2857_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1220.0459_wp,  458.3940_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1566.5042_wp,  561.2533_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1521.2386_wp,  549.3044_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1461.4083_wp,  533.2393_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1386.5806_wp,  513.0291_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1394.5709_wp,  517.5807_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1389.9562_wp,  517.0109_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1372.2422_wp,  512.4692_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1207.5672_wp,  465.0703_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1219.6272_wp,  469.4818_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1043.3955_wp,  415.7690_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1539 * max_ref * max_ref + 1 : 1558 * max_ref * max_ref) = [ &
      & 9330.7294_wp, 1975.9738_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1975.9738_wp,  649.0924_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   99.4579_wp,   83.0141_wp,   44.8321_wp,    0.0000_wp,    0.0000_wp, &
      &  171.7033_wp,  142.5680_wp,   72.9013_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   59.1141_wp,   49.9232_wp,   29.1864_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2551.5260_wp, 2180.4015_wp,  750.3227_wp,    0.0000_wp,    0.0000_wp, &
      &  664.5719_wp,  551.0792_wp,  242.2872_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1160.4419_wp,  961.1049_wp,  417.8652_wp,    0.0000_wp,    0.0000_wp, &
      &  832.6251_wp,  692.8415_wp,  308.1410_wp,    0.0000_wp,    0.0000_wp, &
      &  479.3661_wp,  397.2194_wp,  197.5839_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  690.6852_wp,  572.3002_wp,  276.3430_wp,    0.0000_wp,    0.0000_wp, &
      &  567.5136_wp,  471.0503_wp,  231.6543_wp,    0.0000_wp,    0.0000_wp, &
      &  465.9766_wp,  388.0268_wp,  193.8381_wp,    0.0000_wp,    0.0000_wp, &
      &  329.4784_wp,  274.5914_wp,  145.9330_wp,    0.0000_wp,    0.0000_wp, &
      &  306.9686_wp,  256.0547_wp,  137.1250_wp,    0.0000_wp,    0.0000_wp, &
      &  426.3667_wp,  354.8466_wp,  184.3462_wp,    0.0000_wp,    0.0000_wp, &
      &  401.8774_wp,  334.9207_wp,  173.0249_wp,    0.0000_wp,    0.0000_wp, &
      &  314.6837_wp,  262.6181_wp,  140.2904_wp,    0.0000_wp,    0.0000_wp, &
      &  294.7168_wp,  246.1256_wp,  131.2676_wp,    0.0000_wp,    0.0000_wp, &
      &  228.0556_wp,  191.2494_wp,  106.5800_wp,    0.0000_wp,    0.0000_wp, &
      &  279.0742_wp,  233.5006_wp,  127.7419_wp,    0.0000_wp,    0.0000_wp, &
      &  260.4481_wp,  218.1596_wp,  119.1434_wp,    0.0000_wp,    0.0000_wp, &
      &  242.5528_wp,  203.2396_wp,  111.7057_wp,    0.0000_wp,    0.0000_wp, &
      &  212.1297_wp,  178.0536_wp,   98.2670_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  201.7411_wp,  169.6030_wp,   96.0950_wp,    0.0000_wp,    0.0000_wp, &
      &  178.9549_wp,  150.7056_wp,   86.0773_wp,    0.0000_wp,    0.0000_wp, &
      &  155.5437_wp,  131.2633_wp,   75.7417_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  146.9174_wp,  124.0802_wp,   72.3852_wp,    0.0000_wp,    0.0000_wp, &
      &  114.4676_wp,   97.1926_wp,   58.5836_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  109.5000_wp,   92.8494_wp,   55.4092_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3009.7233_wp, 2565.9398_wp,  895.1626_wp,    0.0000_wp,    0.0000_wp, &
      &  995.3660_wp,  826.9565_wp,  355.3439_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1926.3504_wp, 1599.6001_wp,  669.7642_wp,    0.0000_wp,    0.0000_wp, &
      & 1559.5050_wp, 1298.9425_wp,  550.6761_wp,    0.0000_wp,    0.0000_wp, &
      &  897.4029_wp,  742.3673_wp,  353.0531_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1665.7662_wp, 1381.4389_wp,  611.4261_wp,    0.0000_wp,    0.0000_wp, &
      & 1487.2964_wp, 1233.5817_wp,  552.8634_wp,    0.0000_wp,    0.0000_wp, &
      & 1098.7485_wp,  911.8321_wp,  427.3949_wp,    0.0000_wp,    0.0000_wp, &
      &  796.1066_wp,  659.7476_wp,  329.5915_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1201.2021_wp,  994.8926_wp,  475.6506_wp,    0.0000_wp,    0.0000_wp, &
      & 1097.3980_wp,  909.6567_wp,  438.8597_wp,    0.0000_wp,    0.0000_wp, &
      & 1007.8503_wp,  835.4801_wp,  407.4620_wp,    0.0000_wp,    0.0000_wp, &
      &  993.2763_wp,  824.3552_wp,  397.6803_wp,    0.0000_wp,    0.0000_wp, &
      &  767.6095_wp,  637.1516_wp,  324.2955_wp,    0.0000_wp,    0.0000_wp, &
      &  871.1647_wp,  723.2504_wp,  367.1449_wp,    0.0000_wp,    0.0000_wp, &
      &  844.0120_wp,  700.9983_wp,  356.2697_wp,    0.0000_wp,    0.0000_wp, &
      &  836.9970_wp,  695.2639_wp,  351.6844_wp,    0.0000_wp,    0.0000_wp, &
      &  761.1433_wp,  632.6159_wp,  325.3328_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  690.8651_wp,  575.3317_wp,  302.7908_wp,    0.0000_wp,    0.0000_wp, &
      &  684.2353_wp,  569.8509_wp,  299.5059_wp,    0.0000_wp,    0.0000_wp, &
      &  669.9106_wp,  557.9918_wp,  293.4143_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  540.3160_wp,  451.5825_wp,  245.8635_wp,    0.0000_wp,    0.0000_wp, &
      &  535.2678_wp,  447.3680_wp,  243.3421_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  425.7909_wp,  357.1935_wp,  200.1094_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1558 * max_ref * max_ref + 1 : 1577 * max_ref * max_ref) = [ &
      & 5171.2978_wp, 4495.8369_wp, 1476.8694_wp,    0.0000_wp,    0.0000_wp, &
      & 1322.2619_wp, 1104.1068_wp,  486.9346_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3659.1655_wp, 3070.4624_wp, 1190.5306_wp,    0.0000_wp,    0.0000_wp, &
      & 2951.3362_wp, 2505.6537_wp,  952.5771_wp,    0.0000_wp,    0.0000_wp, &
      & 1267.8490_wp, 1051.9461_wp,  495.4172_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2924.4395_wp, 2443.4277_wp,  978.5884_wp,    0.0000_wp,    0.0000_wp, &
      & 1222.7426_wp, 1013.0298_wp,  481.8238_wp,    0.0000_wp,    0.0000_wp, &
      & 1176.2602_wp,  975.9665_wp,  473.6984_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2748.9797_wp, 2294.6062_wp,  940.9324_wp,    0.0000_wp,    0.0000_wp, &
      & 1180.9424_wp,  982.3638_wp,  465.0211_wp,    0.0000_wp,    0.0000_wp, &
      & 1187.6789_wp,  985.4293_wp,  479.3449_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2475.1427_wp, 2064.0309_wp,  859.3730_wp,    0.0000_wp,    0.0000_wp, &
      & 1158.6662_wp,  963.2725_wp,  455.7896_wp,    0.0000_wp,    0.0000_wp, &
      & 1092.6689_wp,  908.1340_wp,  438.9190_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1952.1126_wp, 1632.7022_wp,  676.8113_wp,    0.0000_wp,    0.0000_wp, &
      & 1158.6672_wp,  963.6534_wp,  452.1352_wp,    0.0000_wp,    0.0000_wp, &
      &  780.0838_wp,  649.5624_wp,  328.0436_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2079.4950_wp, 1732.0375_wp,  737.0898_wp,    0.0000_wp,    0.0000_wp, &
      & 1018.6688_wp,  845.2119_wp,  413.4328_wp,    0.0000_wp,    0.0000_wp, &
      &  644.6600_wp,  537.5804_wp,  279.2953_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1628.4040_wp, 1359.6156_wp,  578.0899_wp,    0.0000_wp,    0.0000_wp, &
      &  931.8610_wp,  773.6698_wp,  381.9591_wp,    0.0000_wp,    0.0000_wp, &
      &  635.3811_wp,  529.7155_wp,  274.2873_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1656.9384_wp, 1375.9674_wp,  609.2865_wp,    0.0000_wp,    0.0000_wp, &
      &  889.9346_wp,  739.5090_wp,  364.3760_wp,    0.0000_wp,    0.0000_wp, &
      &  787.1393_wp,  654.9942_wp,  324.0498_wp,    0.0000_wp,    0.0000_wp, &
      &  675.0933_wp,  562.1481_wp,  286.9784_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1736.3987_wp, 1442.5133_wp,  629.2493_wp,    0.0000_wp,    0.0000_wp, &
      &  873.1622_wp,  725.5203_wp,  356.3892_wp,    0.0000_wp,    0.0000_wp, &
      &  720.7044_wp,  599.5116_wp,  301.1629_wp,    0.0000_wp,    0.0000_wp, &
      &  784.5252_wp,  652.3996_wp,  317.8255_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1335.7319_wp, 1113.8541_wp,  482.4149_wp,    0.0000_wp,    0.0000_wp, &
      &  891.6885_wp,  740.9872_wp,  353.7227_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1292.6630_wp, 1072.0643_wp,  491.2073_wp,    0.0000_wp,    0.0000_wp, &
      &  787.2771_wp,  653.7336_wp,  329.1343_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1559.7012_wp, 1295.4563_wp,  583.5203_wp,    0.0000_wp,    0.0000_wp, &
      & 1439.3610_wp, 1194.8773_wp,  546.6402_wp,    0.0000_wp,    0.0000_wp, &
      & 1073.9458_wp,  891.8106_wp,  429.3221_wp,    0.0000_wp,    0.0000_wp, &
      &  825.7096_wp,  685.5002_wp,  348.1978_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1281.9152_wp, 1062.7315_wp,  509.4520_wp,    0.0000_wp,    0.0000_wp, &
      & 1180.9729_wp,  979.5742_wp,  475.5382_wp,    0.0000_wp,    0.0000_wp, &
      & 1112.0376_wp,  922.5046_wp,  452.0196_wp,    0.0000_wp,    0.0000_wp, &
      &  969.6559_wp,  804.9599_wp,  403.4560_wp,    0.0000_wp,    0.0000_wp, &
      &  826.6644_wp,  686.9957_wp,  354.4023_wp,    0.0000_wp,    0.0000_wp, &
      & 1028.5424_wp,  853.9157_wp,  430.8021_wp,    0.0000_wp,    0.0000_wp, &
      & 1007.7913_wp,  836.9364_wp,  423.2918_wp,    0.0000_wp,    0.0000_wp, &
      &  975.3242_wp,  810.2357_wp,  411.8184_wp,    0.0000_wp,    0.0000_wp, &
      &  930.3838_wp,  773.1293_wp,  396.1357_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  889.6147_wp,  740.1496_wp,  384.5873_wp,    0.0000_wp,    0.0000_wp, &
      &  890.6264_wp,  741.0220_wp,  384.7697_wp,    0.0000_wp,    0.0000_wp, &
      &  882.8069_wp,  734.5865_wp,  381.6942_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  749.8475_wp,  625.5504_wp,  334.8115_wp,    0.0000_wp,    0.0000_wp, &
      &  756.6271_wp,  631.1397_wp,  337.2327_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  630.2436_wp,  527.3521_wp,  289.7981_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 5743.5673_wp, 4997.9230_wp, 1652.2832_wp,    0.0000_wp,    0.0000_wp, &
      & 1556.9958_wp, 1301.3999_wp,  579.7980_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1577 * max_ref * max_ref + 1 : 1596 * max_ref * max_ref) = [ &
      & 4394.3607_wp, 3697.6569_wp, 1419.9417_wp,    0.0000_wp,    0.0000_wp, &
      & 3631.4540_wp, 3095.7594_wp, 1160.9040_wp,    0.0000_wp,    0.0000_wp, &
      & 1516.5369_wp, 1260.1619_wp,  593.3572_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3639.4735_wp, 3045.1620_wp, 1233.0006_wp,    0.0000_wp,    0.0000_wp, &
      & 2485.6310_wp, 2101.0372_wp,  872.2882_wp,    0.0000_wp,    0.0000_wp, &
      & 1391.1764_wp, 1154.9423_wp,  567.2445_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3160.1776_wp, 2637.6782_wp, 1102.2638_wp,    0.0000_wp,    0.0000_wp, &
      & 1635.3622_wp, 1360.4000_wp,  638.0792_wp,    0.0000_wp,    0.0000_wp, &
      & 1384.6011_wp, 1149.7163_wp,  571.0295_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2819.5147_wp, 2352.1278_wp, 1001.6660_wp,    0.0000_wp,    0.0000_wp, &
      & 1650.9334_wp, 1369.3825_wp,  644.4515_wp,    0.0000_wp,    0.0000_wp, &
      & 1349.3970_wp, 1122.7672_wp,  552.6718_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2098.4813_wp, 1750.3829_wp,  768.8382_wp,    0.0000_wp,    0.0000_wp, &
      & 1520.1317_wp, 1266.0803_wp,  599.6614_wp,    0.0000_wp,    0.0000_wp, &
      & 1148.5106_wp,  956.0229_wp,  486.0298_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2373.9131_wp, 1980.9508_wp,  859.6392_wp,    0.0000_wp,    0.0000_wp, &
      & 1811.0626_wp, 1516.2134_wp,  678.1232_wp,    0.0000_wp,    0.0000_wp, &
      &  984.1861_wp,  820.4174_wp,  427.1818_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1735.4827_wp, 1447.6221_wp,  650.7128_wp,    0.0000_wp,    0.0000_wp, &
      & 1270.1002_wp, 1056.4422_wp,  519.5738_wp,    0.0000_wp,    0.0000_wp, &
      &  919.9110_wp,  767.3168_wp,  403.0280_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1890.1371_wp, 1572.3023_wp,  711.0892_wp,    0.0000_wp,    0.0000_wp, &
      & 1185.1747_wp,  986.7643_wp,  485.8072_wp,    0.0000_wp,    0.0000_wp, &
      &  972.1452_wp,  810.5895_wp,  412.5321_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1727.2738_wp, 1436.0400_wp,  657.9647_wp,    0.0000_wp,    0.0000_wp, &
      & 1093.0044_wp,  910.5268_wp,  450.6082_wp,    0.0000_wp,    0.0000_wp, &
      & 1024.0901_wp,  852.7680_wp,  431.4907_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1460.5858_wp, 1219.4904_wp,  549.4299_wp,    0.0000_wp,    0.0000_wp, &
      & 1063.3544_wp,  885.4039_wp,  436.2139_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1493.7369_wp, 1241.2980_wp,  578.1096_wp,    0.0000_wp,    0.0000_wp, &
      & 1016.1802_wp,  845.1580_wp,  429.5551_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1964.0913_wp, 1635.6158_wp,  730.5114_wp,    0.0000_wp,    0.0000_wp, &
      & 1834.2595_wp, 1525.9278_wp,  694.2560_wp,    0.0000_wp,    0.0000_wp, &
      & 1377.5185_wp, 1145.3931_wp,  553.0827_wp,    0.0000_wp,    0.0000_wp, &
      & 1105.9339_wp,  919.0928_wp,  467.1608_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1710.8832_wp, 1420.0022_wp,  669.7960_wp,    0.0000_wp,    0.0000_wp, &
      & 1561.8289_wp, 1296.6408_wp,  622.6826_wp,    0.0000_wp,    0.0000_wp, &
      & 1492.4625_wp, 1239.0351_wp,  600.9779_wp,    0.0000_wp,    0.0000_wp, &
      & 1300.6349_wp, 1080.2049_wp,  538.6735_wp,    0.0000_wp,    0.0000_wp, &
      & 1137.6146_wp,  945.5274_wp,  484.6892_wp,    0.0000_wp,    0.0000_wp, &
      & 1442.5135_wp, 1197.3790_wp,  592.2077_wp,    0.0000_wp,    0.0000_wp, &
      & 1403.5743_wp, 1165.3603_wp,  579.8000_wp,    0.0000_wp,    0.0000_wp, &
      & 1352.0527_wp, 1122.9068_wp,  563.1051_wp,    0.0000_wp,    0.0000_wp, &
      & 1287.7738_wp, 1069.7955_wp,  542.0984_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1295.6354_wp, 1076.7503_wp,  546.9393_wp,    0.0000_wp,    0.0000_wp, &
      & 1292.0505_wp, 1073.8698_wp,  546.3872_wp,    0.0000_wp,    0.0000_wp, &
      & 1276.8973_wp, 1061.4185_wp,  541.6776_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1131.7221_wp,  942.2524_wp,  492.1377_wp,    0.0000_wp,    0.0000_wp, &
      & 1142.9378_wp,  951.5410_wp,  496.7990_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  985.1266_wp,  822.0568_wp,  440.4668_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 7016.4917_wp, 6141.9784_wp, 2013.0282_wp,    0.0000_wp,    0.0000_wp, &
      & 1760.0602_wp, 1469.0705_wp,  681.3467_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 5726.9887_wp, 4845.1776_wp, 1816.0675_wp,    0.0000_wp,    0.0000_wp, &
      & 4845.1776_wp, 4171.7842_wp, 1511.3430_wp,    0.0000_wp,    0.0000_wp, &
      & 1816.0675_wp, 1511.3430_wp,  716.5475_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1596 * max_ref * max_ref + 1 : 1615 * max_ref * max_ref) = [ &
      &   89.1018_wp,   64.6906_wp,   46.8202_wp,    0.0000_wp,    0.0000_wp, &
      &  152.1106_wp,  108.3059_wp,   75.8394_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   53.7982_wp,   40.3329_wp,   30.5886_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2085.0661_wp, 1384.1283_wp,  748.9052_wp,    0.0000_wp,    0.0000_wp, &
      &  570.4266_wp,  389.9070_wp,  248.4867_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  994.0316_wp,  676.7374_wp,  428.0498_wp,    0.0000_wp,    0.0000_wp, &
      &  716.0758_wp,  492.5824_wp,  316.4815_wp,    0.0000_wp,    0.0000_wp, &
      &  422.2754_wp,  297.9018_wp,  205.0850_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  604.4915_wp,  423.0815_wp,  286.0290_wp,    0.0000_wp,    0.0000_wp, &
      &  498.4534_wp,  351.1166_wp,  240.1361_wp,    0.0000_wp,    0.0000_wp, &
      &  410.5123_wp,  291.2744_wp,  201.2083_wp,    0.0000_wp,    0.0000_wp, &
      &  294.1692_wp,  212.3569_wp,  152.2250_wp,    0.0000_wp,    0.0000_wp, &
      &  274.4932_wp,  198.7197_wp,  143.1068_wp,    0.0000_wp,    0.0000_wp, &
      &  378.7928_wp,  271.4248_wp,  191.9355_wp,    0.0000_wp,    0.0000_wp, &
      &  356.4610_wp,  255.3534_wp,  180.0326_wp,    0.0000_wp,    0.0000_wp, &
      &  281.1504_wp,  203.4829_wp,  146.3330_wp,    0.0000_wp,    0.0000_wp, &
      &  263.1381_wp,  190.4785_wp,  136.8686_wp,    0.0000_wp,    0.0000_wp, &
      &  205.5407_wp,  151.1411_wp,  111.4633_wp,    0.0000_wp,    0.0000_wp, &
      &  250.5577_wp,  182.9391_wp,  133.4410_wp,    0.0000_wp,    0.0000_wp, &
      &  233.6619_wp,  170.6938_wp,  124.4124_wp,    0.0000_wp,    0.0000_wp, &
      &  217.8948_wp,  159.4968_wp,  116.6803_wp,    0.0000_wp,    0.0000_wp, &
      &  190.6370_wp,  139.9128_wp,  102.6251_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  182.4075_wp,  135.0322_wp,  100.5575_wp,    0.0000_wp,    0.0000_wp, &
      &  162.0371_wp,  120.4004_wp,   90.0932_wp,    0.0000_wp,    0.0000_wp, &
      &  141.0933_wp,  105.3236_wp,   79.2877_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  133.6123_wp,  100.1090_wp,   75.8298_wp,    0.0000_wp,    0.0000_wp, &
      &  104.8095_wp,   79.6151_wp,   61.4499_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  100.0277_wp,   75.6791_wp,   58.0795_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2465.5965_wp, 1637.7878_wp,  894.7448_wp,    0.0000_wp,    0.0000_wp, &
      &  850.0768_wp,  578.7007_wp,  363.4612_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1637.1586_wp, 1106.7221_wp,  683.2371_wp,    0.0000_wp,    0.0000_wp, &
      & 1328.2510_wp,  903.4563_wp,  562.6808_wp,    0.0000_wp,    0.0000_wp, &
      &  783.1302_wp,  545.0752_wp,  364.9162_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1431.7496_wp,  980.4376_wp,  627.5348_wp,    0.0000_wp,    0.0000_wp, &
      & 1281.5229_wp,  880.4900_wp,  568.1457_wp,    0.0000_wp,    0.0000_wp, &
      &  955.5488_wp,  664.5904_wp,  441.2004_wp,    0.0000_wp,    0.0000_wp, &
      &  701.9398_wp,  495.8077_wp,  342.2242_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1049.1934_wp,  732.1211_wp,  491.8929_wp,    0.0000_wp,    0.0000_wp, &
      &  960.2332_wp,  672.1591_wp,  454.2176_wp,    0.0000_wp,    0.0000_wp, &
      &  883.8815_wp,  620.5160_wp,  422.1202_wp,    0.0000_wp,    0.0000_wp, &
      &  868.8607_wp,  608.8756_wp,  411.5436_wp,    0.0000_wp,    0.0000_wp, &
      &  679.3502_wp,  482.9601_wp,  337.2225_wp,    0.0000_wp,    0.0000_wp, &
      &  770.4924_wp,  547.4653_wp,  381.6757_wp,    0.0000_wp,    0.0000_wp, &
      &  746.5993_wp,  530.8573_wp,  370.3899_wp,    0.0000_wp,    0.0000_wp, &
      &  739.5674_wp,  525.2515_wp,  365.4498_wp,    0.0000_wp,    0.0000_wp, &
      &  674.9489_wp,  481.7395_wp,  338.5362_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  615.5559_wp,  442.8799_wp,  315.6193_wp,    0.0000_wp,    0.0000_wp, &
      &  609.4420_wp,  438.3496_wp,  312.1500_wp,    0.0000_wp,    0.0000_wp, &
      &  596.7186_wp,  429.3015_wp,  305.7996_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  484.7691_wp,  353.1467_wp,  256.8455_wp,    0.0000_wp,    0.0000_wp, &
      &  480.1244_wp,  349.6764_wp,  254.1851_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  384.2306_wp,  283.0351_wp,  209.3707_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4185.8968_wp, 2800.9569_wp, 1467.2405_wp,    0.0000_wp,    0.0000_wp, &
      & 1133.7749_wp,  780.6806_wp,  499.2067_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1615 * max_ref * max_ref + 1 : 1634 * max_ref * max_ref) = [ &
      & 3059.8941_wp, 2050.4253_wp, 1204.1091_wp,    0.0000_wp,    0.0000_wp, &
      & 2456.7601_wp, 1658.3459_wp,  962.2662_wp,    0.0000_wp,    0.0000_wp, &
      & 1103.1556_wp,  767.9412_wp,  511.3063_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2461.7619_wp, 1655.7011_wp,  993.2805_wp,    0.0000_wp,    0.0000_wp, &
      & 1065.8140_wp,  742.6398_wp,  497.6257_wp,    0.0000_wp,    0.0000_wp, &
      & 1030.0975_wp,  722.8068_wp,  490.3181_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2324.7454_wp, 1571.2363_wp,  957.5700_wp,    0.0000_wp,    0.0000_wp, &
      & 1028.4169_wp,  718.6782_wp,  480.2028_wp,    0.0000_wp,    0.0000_wp, &
      & 1040.5570_wp,  730.5584_wp,  496.2446_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2099.5853_wp, 1423.1173_wp,  876.0050_wp,    0.0000_wp,    0.0000_wp, &
      & 1008.8336_wp,  704.4731_wp,  470.5903_wp,    0.0000_wp,    0.0000_wp, &
      &  955.7169_wp,  670.9208_wp,  454.0832_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1653.4435_wp, 1122.6576_wp,  689.4794_wp,    0.0000_wp,    0.0000_wp, &
      & 1006.9636_wp,  701.8768_wp,  466.4241_wp,    0.0000_wp,    0.0000_wp, &
      &  688.3433_wp,  489.6953_wp,  340.5132_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1771.7653_wp, 1206.1393_wp,  753.0932_wp,    0.0000_wp,    0.0000_wp, &
      &  893.5056_wp,  628.2340_wp,  428.1805_wp,    0.0000_wp,    0.0000_wp, &
      &  572.2667_wp,  410.7620_wp,  290.5610_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1386.3132_wp,  945.6808_wp,  590.4390_wp,    0.0000_wp,    0.0000_wp, &
      &  818.8994_wp,  577.5263_wp,  395.8945_wp,    0.0000_wp,    0.0000_wp, &
      &  563.6245_wp,  404.1026_wp,  285.2677_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1423.4023_wp,  975.9836_wp,  625.0103_wp,    0.0000_wp,    0.0000_wp, &
      &  781.6350_wp,  551.4226_wp,  377.5895_wp,    0.0000_wp,    0.0000_wp, &
      &  691.6368_wp,  489.0580_wp,  335.8153_wp,    0.0000_wp,    0.0000_wp, &
      &  597.1154_wp,  426.0198_wp,  298.1443_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1487.1454_wp, 1016.1745_wp,  644.5028_wp,    0.0000_wp,    0.0000_wp, &
      &  766.3677_wp,  540.1679_wp,  369.1952_wp,    0.0000_wp,    0.0000_wp, &
      &  635.3561_wp,  450.9348_wp,  312.4878_wp,    0.0000_wp,    0.0000_wp, &
      &  686.9902_wp,  483.4812_wp,  328.8388_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1141.4078_wp,  781.2881_wp,  493.6053_wp,    0.0000_wp,    0.0000_wp, &
      &  777.7070_wp,  544.0661_wp,  365.3990_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1118.3973_wp,  772.6650_wp,  505.5837_wp,    0.0000_wp,    0.0000_wp, &
      &  694.7959_wp,  492.6342_wp,  341.7316_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1344.7968_wp,  926.3723_wp,  599.8301_wp,    0.0000_wp,    0.0000_wp, &
      & 1245.0382_wp,  860.6632_wp,  562.7686_wp,    0.0000_wp,    0.0000_wp, &
      &  938.9286_wp,  657.9651_wp,  444.1573_wp,    0.0000_wp,    0.0000_wp, &
      &  730.3259_wp,  518.9832_wp,  361.9337_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1120.0021_wp,  782.7401_wp,  526.8660_wp,    0.0000_wp,    0.0000_wp, &
      & 1034.4900_wp,  725.7474_wp,  492.3604_wp,    0.0000_wp,    0.0000_wp, &
      &  976.0066_wp,  686.4772_wp,  468.3886_wp,    0.0000_wp,    0.0000_wp, &
      &  855.1298_wp,  605.4842_wp,  418.8957_wp,    0.0000_wp,    0.0000_wp, &
      &  733.5714_wp,  523.9575_wp,  368.8682_wp,    0.0000_wp,    0.0000_wp, &
      &  908.3670_wp,  644.3560_wp,  447.5546_wp,    0.0000_wp,    0.0000_wp, &
      &  890.4764_wp,  632.2602_wp,  439.8384_wp,    0.0000_wp,    0.0000_wp, &
      &  862.6780_wp,  613.5158_wp,  428.0906_wp,    0.0000_wp,    0.0000_wp, &
      &  824.3488_wp,  587.6805_wp,  412.0610_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  790.4855_wp,  566.2920_wp,  400.4704_wp,    0.0000_wp,    0.0000_wp, &
      &  791.2471_wp,  566.7513_wp,  400.6320_wp,    0.0000_wp,    0.0000_wp, &
      &  784.4000_wp,  562.0006_wp,  397.4445_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  670.3843_wp,  485.2955_wp,  349.3710_wp,    0.0000_wp,    0.0000_wp, &
      &  676.1930_wp,  489.2291_wp,  351.8502_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  566.5101_wp,  414.1664_wp,  302.8960_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4653.7454_wp, 3121.5559_wp, 1643.0879_wp,    0.0000_wp,    0.0000_wp, &
      & 1337.5010_wp,  924.2203_wp,  594.9926_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3666.6556_wp, 2458.0845_wp, 1434.6583_wp,    0.0000_wp,    0.0000_wp, &
      & 3014.0781_wp, 2036.5362_wp, 1171.1205_wp,    0.0000_wp,    0.0000_wp, &
      & 1319.0312_wp,  919.4490_wp,  612.2994_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1634 * max_ref * max_ref + 1 : 1653 * max_ref * max_ref) = [ &
      & 3069.3037_wp, 2072.7978_wp, 1253.1317_wp,    0.0000_wp,    0.0000_wp, &
      & 2105.7491_wp, 1446.2394_wp,  889.9501_wp,    0.0000_wp,    0.0000_wp, &
      & 1221.2152_wp,  860.0465_wp,  587.7222_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2682.0879_wp, 1821.2145_wp, 1124.0457_wp,    0.0000_wp,    0.0000_wp, &
      & 1420.6192_wp,  990.1059_wp,  658.2098_wp,    0.0000_wp,    0.0000_wp, &
      & 1218.3604_wp,  860.7503_wp,  592.2396_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2401.9326_wp, 1637.9243_wp, 1023.5000_wp,    0.0000_wp,    0.0000_wp, &
      & 1434.8956_wp,  997.9222_wp,  664.7346_wp,    0.0000_wp,    0.0000_wp, &
      & 1184.6610_wp,  836.5504_wp,  572.6815_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1798.1970_wp, 1235.6226_wp,  787.9045_wp,    0.0000_wp,    0.0000_wp, &
      & 1323.2583_wp,  925.8359_wp,  619.0974_wp,    0.0000_wp,    0.0000_wp, &
      & 1015.1154_wp,  723.2437_wp,  504.8956_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2029.6163_wp, 1390.9989_wp,  880.0314_wp,    0.0000_wp,    0.0000_wp, &
      & 1557.0674_wp, 1078.7908_wp,  696.3752_wp,    0.0000_wp,    0.0000_wp, &
      &  874.2449_wp,  627.7003_wp,  444.5757_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1493.8895_wp, 1032.5864_wp,  668.2859_wp,    0.0000_wp,    0.0000_wp, &
      & 1114.5962_wp,  786.5886_wp,  538.1720_wp,    0.0000_wp,    0.0000_wp, &
      &  818.6281_wp,  589.4768_wp,  419.6916_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1629.9126_wp, 1125.4492_wp,  730.8135_wp,    0.0000_wp,    0.0000_wp, &
      & 1040.1066_wp,  734.8918_wp,  503.1910_wp,    0.0000_wp,    0.0000_wp, &
      &  859.0221_wp,  613.1365_wp,  428.4259_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1493.4855_wp, 1034.1628_wp,  677.0271_wp,    0.0000_wp,    0.0000_wp, &
      &  960.1346_wp,  679.6572_wp,  466.8855_wp,    0.0000_wp,    0.0000_wp, &
      &  903.9928_wp,  643.4443_wp,  447.9267_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1257.4297_wp,  870.4102_wp,  564.2490_wp,    0.0000_wp,    0.0000_wp, &
      &  933.1680_wp,  659.4710_wp,  451.7417_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1295.9364_wp,  900.7673_wp,  595.7322_wp,    0.0000_wp,    0.0000_wp, &
      &  898.2092_wp,  639.3832_wp,  446.1765_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1689.4085_wp, 1164.0480_wp,  749.9911_wp,    0.0000_wp,    0.0000_wp, &
      & 1583.8792_wp, 1095.4590_wp,  714.0617_wp,    0.0000_wp,    0.0000_wp, &
      & 1204.5072_wp,  845.7173_wp,  572.0823_wp,    0.0000_wp,    0.0000_wp, &
      &  977.8582_wp,  695.6184_wp,  485.3739_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1489.0440_wp, 1037.3050_wp,  691.3637_wp,    0.0000_wp,    0.0000_wp, &
      & 1364.4290_wp,  955.2258_wp,  643.8177_wp,    0.0000_wp,    0.0000_wp, &
      & 1306.5898_wp,  917.1231_wp,  621.9390_wp,    0.0000_wp,    0.0000_wp, &
      & 1145.4236_wp,  810.2212_wp,  558.8413_wp,    0.0000_wp,    0.0000_wp, &
      & 1007.8507_wp,  718.6770_wp,  504.0290_wp,    0.0000_wp,    0.0000_wp, &
      & 1268.2622_wp,  894.7026_wp,  613.9793_wp,    0.0000_wp,    0.0000_wp, &
      & 1235.5723_wp,  873.2287_wp,  601.4334_wp,    0.0000_wp,    0.0000_wp, &
      & 1192.2239_wp,  844.5925_wp,  584.5218_wp,    0.0000_wp,    0.0000_wp, &
      & 1138.1053_wp,  808.6931_wp,  563.2228_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1145.5595_wp,  814.8114_wp,  568.3523_wp,    0.0000_wp,    0.0000_wp, &
      & 1142.8009_wp,  813.2828_wp,  567.8637_wp,    0.0000_wp,    0.0000_wp, &
      & 1130.1276_wp,  805.0118_wp,  563.1134_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1006.5104_wp,  722.4932_wp,  512.5389_wp,    0.0000_wp,    0.0000_wp, &
      & 1016.4155_wp,  729.4949_wp,  517.3851_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  880.8202_wp,  637.9590_wp,  459.5735_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 5675.7434_wp, 3825.0881_wp, 2001.2557_wp,    0.0000_wp,    0.0000_wp, &
      & 1524.9399_wp, 1063.3274_wp,  702.1148_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4754.4826_wp, 3186.2921_wp, 1830.2684_wp,    0.0000_wp,    0.0000_wp, &
      & 3994.2273_wp, 2705.5816_wp, 1519.7933_wp,    0.0000_wp,    0.0000_wp, &
      & 1581.4671_wp, 1105.9643_wp,  739.9160_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3990.6172_wp, 2690.9591_wp, 1603.5452_wp,    0.0000_wp,    0.0000_wp, &
      & 2690.9591_wp, 1850.0003_wp, 1126.9136_wp,    0.0000_wp,    0.0000_wp, &
      & 1603.5452_wp, 1126.9136_wp,  766.0202_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1653 * max_ref * max_ref + 1 : 1672 * max_ref * max_ref) = [ &
      &   44.5104_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   71.8830_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   29.3081_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  704.3345_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  234.5271_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  403.7427_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  298.8782_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  194.1728_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  270.6187_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  227.4189_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  190.7463_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  144.6238_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  136.0287_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  182.2164_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  170.9262_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  139.1811_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  130.2341_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  106.3035_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  127.1355_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  118.5470_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  111.2827_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   97.9882_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   96.1189_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   86.1665_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   75.9910_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   72.7359_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   59.1379_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   55.9048_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  841.8986_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  343.1382_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  644.2220_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  530.8650_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  345.1382_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  592.4047_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  536.5626_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  417.2208_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  324.1533_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  465.2631_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  429.8074_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  399.5853_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  389.5669_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  319.6854_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  361.8015_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  351.1585_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  346.4675_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  321.1378_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  299.7447_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  296.4581_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  290.4555_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  244.4468_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  241.9223_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  199.7138_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1381.0183_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  472.2455_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1672 * max_ref * max_ref + 1 : 1691 * max_ref * max_ref) = [ &
      & 1134.6748_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  907.4012_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  483.9715_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  936.4068_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  471.1854_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  464.4600_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  903.3306_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  454.8200_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  470.1341_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  826.7047_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  445.7570_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  430.3341_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  651.1440_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  441.7809_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  323.4660_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  711.1423_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  405.8737_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  276.3955_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  558.0177_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  375.4568_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  271.3545_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  590.7387_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  358.1677_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  318.8026_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  283.4108_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  608.9242_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  350.2054_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  296.8236_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  312.1102_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  466.8110_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  346.4374_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  478.3350_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  324.3365_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  566.8783_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  532.0683_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  420.5733_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  343.2601_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  498.6284_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  466.1880_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  443.6388_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  397.0824_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  350.0197_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  424.2906_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  417.0350_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  405.9846_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  390.9033_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  380.1606_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  380.3160_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  377.3149_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  332.1884_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  334.5262_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  288.5010_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1547.0780_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  563.1480_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1352.0803_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1104.4325_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  579.7357_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1691 * max_ref * max_ref + 1 : 1710 * max_ref * max_ref) = [ &
      & 1182.0124_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  840.8410_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  556.9697_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1060.9721_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  623.2841_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  561.4468_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  966.6201_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  629.4977_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  543.0506_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  745.1176_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  586.6430_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  479.4247_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  831.8360_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  659.2348_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  422.6628_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  632.6639_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  510.5864_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  399.2351_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  691.5646_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  477.5945_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  407.2558_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  640.9611_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  443.3629_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  425.6143_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  534.5993_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  428.9980_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  564.3910_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  423.8910_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  709.5009_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  675.7911_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  542.3835_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  460.8857_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  654.6723_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  610.0042_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  589.4528_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  530.1369_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  478.5981_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  582.1618_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  570.3846_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  554.4976_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  534.4801_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  539.4167_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  538.9780_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  534.5244_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  487.0333_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  491.6150_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  437.2834_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1884.4554_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  664.8346_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1724.7415_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1433.0469_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  700.8041_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1512.2504_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1064.5933_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  725.6894_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1710 * max_ref * max_ref + 1 : 1729 * max_ref * max_ref) = [ &
      &  688.0353_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   88.6877_wp,   43.2202_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  152.4502_wp,   69.7341_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   53.1523_wp,   28.5060_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2210.3168_wp,  680.2479_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  583.9797_wp,  227.0579_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1018.9540_wp,  390.7983_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  732.1642_wp,  289.4434_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  424.7883_wp,  188.2849_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  610.7615_wp,  262.3192_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  502.4967_wp,  220.5170_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  413.1209_wp,  185.0194_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  293.4486_wp,  140.3959_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  273.5743_wp,  132.0706_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  379.1038_wp,  176.8339_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  357.1399_wp,  165.8805_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  280.4649_wp,  135.1380_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  262.6644_wp,  126.4572_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  204.0039_wp,  103.2933_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  249.2633_wp,  123.4955_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  232.5783_wp,  115.1605_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  216.7820_wp,  108.1159_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  189.7084_wp,   95.2178_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  180.8383_wp,   93.4327_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  160.5233_wp,   83.7765_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  139.7472_wp,   73.9032_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  132.1584_wp,   70.7481_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  103.3616_wp,   57.5599_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   98.8220_wp,   54.4061_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2609.0440_wp,  813.2353_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  873.5134_wp,  332.1495_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1687.5419_wp,  623.3223_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1367.2714_wp,  513.7956_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  792.9063_wp,  334.4652_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1464.4446_wp,  573.5902_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1308.6041_wp,  519.6165_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  969.7366_wp,  404.2910_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  705.7873_wp,  314.3449_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1061.5213_wp,  450.9253_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  970.3957_wp,  416.6272_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  891.9042_wp,  387.3888_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  878.3222_wp,  377.6480_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  681.4251_wp,  310.1096_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  773.1721_wp,  350.9578_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  749.1438_wp,  340.6470_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  742.6702_wp,  336.0824_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  676.2031_wp,  311.5824_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  614.8652_wp,  290.9369_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  608.9119_wp,  287.7449_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  596.1986_wp,  281.9238_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  482.2509_wp,  237.4072_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  477.7197_wp,  234.9544_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  381.0503_wp,  194.0698_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1729 * max_ref * max_ref + 1 : 1748 * max_ref * max_ref) = [ &
      & 4473.7784_wp, 1333.7958_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1162.2835_wp,  457.4338_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3190.2086_wp, 1097.1659_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2571.5319_wp,  877.5528_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1119.4842_wp,  469.0498_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2554.7353_wp,  905.7241_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1079.9647_wp,  456.7087_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1040.8154_wp,  450.3076_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2405.0561_wp,  874.0066_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1043.3187_wp,  440.8726_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1051.1001_wp,  455.8234_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2167.6325_wp,  800.0152_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1023.5395_wp,  432.0839_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  966.6619_wp,  417.2395_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1709.1137_wp,  630.1961_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1022.9773_wp,  428.1936_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  692.4667_wp,  313.8469_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1823.7858_wp,  688.3693_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  902.0482_wp,  393.5556_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  573.5714_wp,  268.2902_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1428.1039_wp,  540.2311_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  825.7735_wp,  364.1170_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  565.2043_wp,  263.3851_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1457.0358_wp,  572.0738_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  788.5809_wp,  347.3566_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  697.7563_wp,  309.2299_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  599.8783_wp,  275.0230_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1525.4141_wp,  589.5695_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  773.5549_wp,  339.6232_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  639.6214_wp,  287.9649_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  694.6969_wp,  302.6820_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1172.9549_wp,  452.0342_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  788.3761_wp,  335.8606_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1139.4352_wp,  463.4148_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  698.7243_wp,  314.6193_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1372.8649_wp,  549.0692_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1268.2581_wp,  515.4512_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  949.6978_wp,  407.7143_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  733.0346_wp,  332.9869_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1133.1627_wp,  483.3174_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1044.8624_wp,  451.9571_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  984.5356_wp,  430.1513_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  859.9043_wp,  385.1320_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  734.6883_wp,  339.6203_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  912.5263_wp,  411.5533_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  894.2815_wp,  404.5340_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  865.7939_wp,  393.8448_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  826.4044_wp,  379.2572_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  791.0214_wp,  368.9206_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  791.8817_wp,  369.0697_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  784.9774_wp,  366.1629_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  668.3328_wp,  322.5272_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  674.2876_wp,  324.7894_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  563.0156_wp,  280.2438_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4971.0778_wp, 1494.4074_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1369.5245_wp,  545.5917_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1748 * max_ref * max_ref + 1 : 1767 * max_ref * max_ref) = [ &
      & 3828.9626_wp, 1307.3665_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3162.1198_wp, 1068.0684_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1338.9789_wp,  561.9076_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3181.4405_wp, 1143.5524_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2178.3229_wp,  814.0637_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1231.9890_wp,  540.0992_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2767.9894_wp, 1026.8226_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1442.9680_wp,  604.1250_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1227.1618_wp,  544.5203_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2472.6905_wp,  935.7467_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1456.5113_wp,  610.1366_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1195.2654_wp,  526.6759_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1844.0715_wp,  721.6673_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1342.8116_wp,  568.7148_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1019.8084_wp,  465.1818_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2084.5975_wp,  805.5111_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1593.6331_wp,  638.7376_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  875.5927_wp,  410.2593_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1527.6056_wp,  612.9682_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1125.1127_wp,  495.2056_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  819.0425_wp,  387.5759_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1664.4640_wp,  669.9917_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1050.0763_wp,  463.2412_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  863.5344_wp,  395.2161_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1522.4806_wp,  621.0687_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  968.8758_wp,  430.0863_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  909.2919_wp,  412.9759_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1286.0441_wp,  518.0207_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  942.3547_wp,  416.1281_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1318.2791_wp,  546.9938_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  902.6294_wp,  411.2941_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1728.2162_wp,  687.2667_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1615.9882_wp,  654.7497_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1218.8063_wp,  525.8956_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  982.2388_wp,  447.1551_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1511.0459_wp,  634.5156_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1381.1677_wp,  591.3671_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1320.7683_wp,  571.5166_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1153.3721_wp,  514.1941_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1010.9240_wp,  464.3765_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1278.3406_wp,  564.5753_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1244.3690_wp,  553.2005_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1199.3914_wp,  537.8508_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1143.2636_wp,  518.5062_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1150.4200_wp,  523.3205_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1147.3718_wp,  522.9067_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1134.1681_wp,  518.6075_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1007.0198_wp,  472.7004_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1016.9588_wp,  477.1430_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  878.3798_wp,  424.5915_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 6073.8935_wp, 1820.4201_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1552.0845_wp,  644.3858_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4983.3126_wp, 1667.4685_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4213.9225_wp, 1385.6402_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1604.0906_wp,  679.3561_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1767 * max_ref * max_ref + 1 : 1786 * max_ref * max_ref) = [ &
      & 4150.3203_wp, 1462.8142_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2790.9891_wp, 1030.6163_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1619.6684_wp,  703.6339_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1527.1461_wp,  667.2018_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4342.2386_wp, 1476.7389_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1476.7389_wp,  647.0250_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   85.4080_wp,   42.3343_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  146.5851_wp,   68.2931_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   51.2952_wp,   27.9324_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2091.5068_wp,  665.5329_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  558.6972_wp,  222.2883_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  974.6167_wp,  382.5760_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  700.4442_wp,  283.3715_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  408.1203_wp,  184.3805_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  586.1896_wp,  256.8646_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  482.5077_wp,  215.9438_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  396.7581_wp,  181.1913_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  282.4641_wp,  137.5122_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  263.3886_wp,  129.3614_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  364.6519_wp,  173.1933_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  343.4153_wp,  162.4665_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  269.9865_wp,  132.3691_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  252.8197_wp,  123.8682_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  196.6165_wp,  101.1917_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  240.1139_wp,  120.9759_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  224.0040_wp,  112.8134_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  208.8357_wp,  105.9150_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  182.7560_wp,   93.2841_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  174.3678_wp,   91.5407_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  154.8067_wp,   82.0849_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  134.8019_wp,   72.4156_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  127.5308_wp,   69.3261_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   99.8331_wp,   56.4117_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   95.4202_wp,   53.3201_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2470.6020_wp,  795.6873_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  834.9263_wp,  325.1636_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1611.7831_wp,  610.1625_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1305.9000_wp,  502.9655_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  760.7352_wp,  327.4991_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1401.3191_wp,  561.5498_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1252.6928_wp,  508.7250_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  929.6042_wp,  395.8577_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  678.1931_wp,  307.8310_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1018.5098_wp,  441.5369_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  931.2917_wp,  407.9629_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  856.2817_wp,  379.3429_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  842.8268_wp,  369.7985_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  655.1209_wp,  303.6993_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  743.2407_wp,  343.7019_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  720.1450_wp,  333.6065_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  713.7913_wp,  329.1343_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  650.2639_wp,  305.1528_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  591.6752_wp,  284.9529_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  585.9134_wp,  281.8265_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  573.6840_wp,  276.1263_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1786 * max_ref * max_ref + 1 : 1805 * max_ref * max_ref) = [ &
      &  464.5062_wp,  232.5513_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  460.1250_wp,  230.1488_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  367.3163_wp,  190.1224_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4216.3998_wp, 1304.7563_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1111.2638_wp,  447.8636_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3036.3662_wp, 1073.8235_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2442.0495_wp,  858.8383_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1073.3952_wp,  459.2919_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2435.0889_wp,  886.5266_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1036.2055_wp,  447.2220_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  998.9971_wp,  440.9674_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2294.2857_wp,  855.5354_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1000.2155_wp,  431.7097_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1008.9483_wp,  446.3717_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2068.9809_wp,  783.1386_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  981.3457_wp,  423.1055_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  927.5254_wp,  408.5880_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1630.6018_wp,  616.9100_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  980.4829_wp,  419.2898_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  665.3571_wp,  307.3826_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1742.1942_wp,  673.8862_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  866.0442_wp,  385.4043_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  551.6130_wp,  262.7854_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1363.8456_wp,  528.8780_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  793.0239_wp,  356.5850_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  543.5117_wp,  257.9795_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1394.0822_wp,  560.0917_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  757.1711_wp,  340.1719_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  669.9835_wp,  302.8447_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  576.6205_wp,  269.3672_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1458.7399_wp,  577.1994_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  742.6757_wp,  332.5971_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  614.5241_wp,  282.0297_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  666.7343_wp,  296.4247_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1120.9931_wp,  442.5579_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  756.1577_wp,  328.8954_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1091.5722_wp,  453.7462_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  671.4732_wp,  308.1251_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1314.2306_wp,  537.5790_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1214.7867_wp,  504.6830_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  911.1851_wp,  399.2448_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  704.6787_wp,  326.1098_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1087.2533_wp,  473.2673_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1002.9138_wp,  442.5728_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  945.3065_wp,  421.2295_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  826.2471_wp,  377.1647_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  706.5967_wp,  332.6173_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  877.0066_wp,  403.0441_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  859.5248_wp,  396.1729_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  832.2670_wp,  385.7096_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  794.6136_wp,  371.4307_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  760.8949_wp,  361.3222_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  761.6998_wp,  361.4679_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  755.0713_wp,  358.6221_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  643.4305_wp,  315.9132_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  649.1282_wp,  318.1278_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1805 * max_ref * max_ref + 1 : 1824 * max_ref * max_ref) = [ &
      &  542.4383_wp,  274.5224_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4685.0846_wp, 1461.9055_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1309.7014_wp,  534.1947_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3642.0492_wp, 1279.5273_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2999.9283_wp, 1045.2541_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1283.6521_wp,  550.2241_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3032.9183_wp, 1119.3527_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2074.8807_wp,  796.8844_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1182.9141_wp,  528.9145_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2642.0526_wp, 1005.1742_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1383.2190_wp,  591.5630_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1178.7222_wp,  533.2574_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2361.6974_wp,  916.0648_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1397.0298_wp,  597.4603_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1147.4757_wp,  515.7797_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1763.1139_wp,  706.5573_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1287.2763_wp,  556.9066_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  980.1462_wp,  455.5995_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1992.1534_wp,  788.6151_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1523.8714_wp,  625.3966_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  842.1651_wp,  401.8371_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1461.6458_wp,  600.1777_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1080.1628_wp,  484.9704_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  787.9846_wp,  379.6309_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1593.3696_wp,  656.0092_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1008.0521_wp,  453.6734_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  829.8526_wp,  387.0905_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1458.1671_wp,  608.1289_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  930.2250_wp,  421.2135_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  873.7547_wp,  404.4764_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1230.4969_wp,  507.2269_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  904.6688_wp,  407.5420_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1263.3482_wp,  535.6228_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  867.5938_wp,  402.8281_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1653.5039_wp,  672.8996_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1547.2492_wp,  641.0897_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1169.3656_wp,  514.9974_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  944.1820_wp,  437.9410_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1448.8711_wp,  621.3216_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1325.1151_wp,  579.0951_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1267.6087_wp,  559.6696_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1107.9903_wp,  503.5686_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  972.0426_wp,  454.8102_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1227.7455_wp,  552.8934_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1195.3394_wp,  541.7614_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1152.4260_wp,  526.7389_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1098.8818_wp,  507.8063_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1105.8143_wp,  512.5245_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1102.9411_wp,  512.1209_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1090.3536_wp,  507.9139_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  968.7947_wp,  462.9825_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  978.3471_wp,  467.3327_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  845.6703_wp,  415.8938_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1824 * max_ref * max_ref + 1 : 1843 * max_ref * max_ref) = [ &
      & 5716.7999_wp, 1780.7379_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1486.4897_wp,  630.9699_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4733.4185_wp, 1631.8559_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3987.3600_wp, 1355.8874_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1537.9273_wp,  665.2422_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3951.9956_wp, 1431.7796_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2655.1171_wp, 1008.8176_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1554.6376_wp,  689.0464_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1465.9695_wp,  653.3875_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4125.9023_wp, 1445.2611_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1417.6855_wp,  633.6333_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3924.4211_wp, 1387.5284_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1387.5284_wp,  620.5206_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   83.3310_wp,   41.0356_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  142.9620_wp,   66.1439_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   50.0748_wp,   27.1150_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2031.2740_wp,  642.1808_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  544.1232_wp,  214.9157_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  949.1255_wp,  369.8142_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  682.1901_wp,  274.0453_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  397.9479_wp,  178.5085_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  571.4113_wp,  248.6071_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  470.4029_wp,  209.0627_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  386.8363_wp,  175.4697_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  275.5610_wp,  133.2643_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  256.9654_wp,  125.3807_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  355.6712_wp,  167.7968_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  334.9245_wp,  157.4078_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  263.3922_wp,  128.3013_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  246.6348_wp,  120.0668_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  191.8733_wp,   98.1467_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  234.2913_wp,  117.3026_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  218.5611_wp,  109.3944_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  203.7729_wp,  102.7150_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  178.3241_wp,   90.4807_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  170.1817_wp,   88.8159_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  151.0970_wp,   79.6566_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  131.5785_wp,   70.2898_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  124.4946_wp,   67.2999_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   97.4789_wp,   54.7947_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   93.1635_wp,   51.7860_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2399.8207_wp,  767.8676_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  812.9350_wp,  314.3316_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1568.9885_wp,  589.6141_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1271.2883_wp,  486.1582_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  741.4881_wp,  316.8969_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1364.8494_wp,  542.9683_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1220.2246_wp,  491.9700_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  905.8860_wp,  383.0238_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  661.3178_wp,  298.0452_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1843 * max_ref * max_ref + 1 : 1862 * max_ref * max_ref) = [ &
      &  992.7507_wp,  427.2858_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  907.7970_wp,  394.8501_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  834.7641_wp,  367.1968_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  821.5324_wp,  357.9364_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  638.9063_wp,  294.1261_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  724.8197_wp,  332.8624_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  702.2959_wp,  323.0966_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  696.0628_wp,  318.7539_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  634.2089_wp,  295.5871_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  577.1696_wp,  276.1129_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  571.5397_wp,  273.0819_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  559.6108_wp,  267.5627_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  453.2313_wp,  225.4562_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  448.9517_wp,  223.1261_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  358.4721_wp,  184.4110_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4092.0575_wp, 1259.0280_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1082.0865_wp,  433.2074_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2952.9794_wp, 1037.1340_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2374.0445_wp,  829.6408_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1046.0388_wp,  444.4629_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2369.1244_wp,  856.4434_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1009.8893_wp,  432.8276_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  973.8228_wp,  426.8683_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2232.6503_wp,  826.7257_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  974.7227_wp,  417.8360_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  983.5427_wp,  432.1120_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2013.7194_wp,  756.8843_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  956.3321_wp,  409.5058_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  904.0753_wp,  395.5412_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1586.8512_wp,  596.2971_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  955.4042_wp,  405.7835_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  648.7727_wp,  297.7522_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1696.0441_wp,  651.4438_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  844.2856_wp,  373.1223_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  537.9977_wp,  254.6469_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1327.6064_wp,  511.3384_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  773.1575_wp,  345.2676_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  530.0826_wp,  249.9799_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1357.7395_wp,  541.6401_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  738.1748_wp,  329.3812_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  653.1705_wp,  293.2805_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  562.3122_wp,  260.9610_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1420.5045_wp,  558.0904_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  724.0207_wp,  322.0375_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  599.1969_wp,  273.1670_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  649.9094_wp,  287.0163_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1091.4214_wp,  427.9630_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  736.9509_wp,  318.3642_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1063.4898_wp,  438.9546_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  654.7776_wp,  298.4106_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1280.1762_wp,  519.9536_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1183.4931_wp,  488.2162_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  888.1417_wp,  386.4453_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  687.2223_wp,  315.8374_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1059.7536_wp,  458.0385_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  977.6523_wp,  428.4016_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  921.5753_wp,  407.7871_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  805.6691_wp,  365.2295_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  689.1790_wp,  322.2035_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1862 * max_ref * max_ref + 1 : 1881 * max_ref * max_ref) = [ &
      &  855.2177_wp,  390.3161_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  838.1837_wp,  383.6775_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  811.6356_wp,  373.5692_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  774.9718_wp,  359.7752_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  742.1649_wp,  350.0554_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  742.9435_wp,  350.1952_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  736.4811_wp,  347.4428_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  627.7342_wp,  306.1959_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  633.2835_wp,  308.3361_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  529.3070_wp,  266.1893_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4547.0720_wp, 1410.8565_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1275.3968_wp,  516.8026_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3541.4930_wp, 1235.8086_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2915.7949_wp, 1009.7020_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1250.8766_wp,  532.4996_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2950.9303_wp, 1081.5963_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2018.9224_wp,  770.4895_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1153.2166_wp,  512.0867_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2571.4964_wp,  971.5644_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1347.7804_wp,  572.5226_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1149.2501_wp,  516.3573_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2299.0481_wp,  885.6282_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1361.3178_wp,  578.2213_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1118.6278_wp,  499.4343_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1716.8038_wp,  683.3681_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1254.4039_wp,  539.0640_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  955.7891_wp,  441.3364_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1939.6215_wp,  762.6134_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1483.9817_wp,  605.0829_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  821.4043_wp,  389.3842_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1423.5531_wp,  580.6563_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1052.9974_wp,  469.6134_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  768.6159_wp,  367.9133_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1552.0522_wp,  634.6293_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  982.6875_wp,  439.3361_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  809.2032_wp,  375.0228_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1420.5469_wp,  588.3919_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  906.8511_wp,  407.9420_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  851.9952_wp,  391.8190_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1198.4194_wp,  490.7833_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  881.9071_wp,  394.6814_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1230.9598_wp,  518.3356_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  846.0616_wp,  390.2136_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1610.4142_wp,  650.8890_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1507.2236_wp,  620.2296_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1139.7842_wp,  498.5669_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  920.7728_wp,  424.1984_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1411.9588_wp,  601.2862_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1291.5707_wp,  560.5406_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1235.6389_wp,  541.7976_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1080.3308_wp,  487.6436_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  948.0215_wp,  440.5691_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1197.0157_wp,  535.3426_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1165.4811_wp,  524.6028_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1123.7195_wp,  510.1049_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1071.6137_wp,  491.8297_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1881 * max_ref * max_ref + 1 : 1900 * max_ref * max_ref) = [ &
      & 1078.3854_wp,  496.4206_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1075.5989_wp,  496.0396_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1063.3516_wp,  491.9827_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  944.9814_wp,  448.6020_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  954.2967_wp,  452.8135_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  825.0473_wp,  403.1233_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 5547.4900_wp, 1718.6565_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1448.1464_wp,  610.6517_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4601.2171_wp, 1575.9396_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3873.7284_wp, 1309.6071_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1498.6941_wp,  643.9008_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3844.0367_wp, 1383.3184_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2582.9722_wp,  975.3429_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1515.4611_wp,  667.0629_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1429.0690_wp,  632.6069_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4011.3635_wp, 1395.9648_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1382.0320_wp,  613.5022_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3816.1118_wp, 1340.2848_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1352.6449_wp,  600.8101_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3710.9375_wp, 1306.6123_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1306.6123_wp,  581.7435_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   81.4123_wp,   41.1186_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  139.6254_wp,   66.4057_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   48.9406_wp,   27.0910_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1976.6955_wp,  653.7125_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  530.7909_wp,  216.8695_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  925.8158_wp,  373.3538_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  665.4868_wp,  276.3828_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  388.5925_wp,  179.3900_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  557.8380_wp,  250.0690_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  459.2753_wp,  210.1448_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  377.7092_wp,  176.2659_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  269.1899_wp,  133.6061_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  251.0346_wp,  125.6675_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  347.3921_wp,  168.3507_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  327.0991_wp,  157.9406_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  257.3029_wp,  128.5963_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  240.9233_wp,  120.3408_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  187.4818_wp,   98.2270_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  228.9065_wp,  117.4740_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  213.5277_wp,  109.5523_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  199.0885_wp,  102.8398_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  174.2214_wp,   90.5682_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  166.3004_wp,   88.8332_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  147.6552_wp,   79.6477_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  128.5846_wp,   70.2511_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  121.6727_wp,   67.2415_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   95.2846_wp,   54.6852_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1900 * max_ref * max_ref + 1 : 1919 * max_ref * max_ref) = [ &
      &   91.0614_wp,   51.6977_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2335.6474_wp,  781.2050_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  792.8336_wp,  317.4096_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1529.9120_wp,  596.0022_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1239.6749_wp,  491.1385_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  723.8209_wp,  318.9485_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1331.4786_wp,  547.7742_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1190.5003_wp,  496.0959_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  884.1322_wp,  385.6361_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  645.7939_wp,  299.4728_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  969.1026_wp,  429.9494_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  886.2193_wp,  397.1752_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  814.9918_wp,  369.2253_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  801.9751_wp,  360.0151_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  623.9763_wp,  295.3345_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  707.8604_wp,  334.2517_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  685.8620_wp,  324.4246_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  679.7433_wp,  320.1057_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  619.4176_wp,  296.6816_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  563.7890_wp,  276.9116_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  558.2816_wp,  273.8804_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  546.6292_wp,  268.3382_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  442.8095_wp,  225.8370_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  438.6243_wp,  223.5079_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  350.2812_wp,  184.5332_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3979.6309_wp, 1283.6315_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1055.3891_wp,  436.8854_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2877.0680_wp, 1050.9848_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2312.2147_wp,  841.0519_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1020.9351_wp,  447.3826_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2308.9962_wp,  866.9228_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  985.7264_wp,  435.5172_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  950.6909_wp,  429.2527_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2176.4184_wp,  836.0994_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  951.3257_wp,  420.4671_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  960.1954_wp,  434.4919_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1963.2723_wp,  765.0512_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  933.3738_wp,  412.0874_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  882.5313_wp,  397.7647_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1546.9195_wp,  602.7113_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  932.3946_wp,  408.4460_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  633.5005_wp,  298.9583_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1653.8773_wp,  657.9635_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  824.2830_wp,  375.1051_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  525.4406_wp,  255.4370_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1294.4951_wp,  516.3770_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  754.8861_wp,  346.9868_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  517.6989_wp,  250.7830_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1324.4770_wp,  546.3363_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  720.7052_wp,  331.0282_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  637.7039_wp,  294.6745_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  549.1278_wp,  261.9299_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1385.5313_wp,  563.2255_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  706.8671_wp,  323.6772_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  585.0856_wp,  274.3346_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  634.4415_wp,  288.5206_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1919 * max_ref * max_ref + 1 : 1938 * max_ref * max_ref) = [ &
      & 1064.3798_wp,  431.9033_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  719.3118_wp,  320.2684_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1037.7490_wp,  442.2545_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  639.4075_wp,  299.7092_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1248.9915_wp,  524.1676_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1154.8172_wp,  491.9157_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  866.9787_wp,  388.7055_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  671.1482_wp,  317.1445_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1034.5024_wp,  460.8179_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  954.4431_wp,  430.8119_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  899.7622_wp,  409.9536_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  786.7340_wp,  366.8929_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  673.1287_wp,  323.3676_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  835.1622_wp,  392.0139_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  818.5381_wp,  385.3099_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  792.6385_wp,  375.0947_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  756.8785_wp,  361.1491_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  724.8987_wp,  351.2182_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  725.6536_wp,  351.3644_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  719.3436_wp,  348.5929_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  613.2405_wp,  306.8959_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  618.6540_wp,  309.0575_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  517.1617_wp,  266.5494_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4422.2696_wp, 1437.9300_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1243.9950_wp,  520.9703_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3449.9906_wp, 1252.6342_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2839.3501_wp, 1024.0101_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1220.8006_wp,  535.9497_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2876.1716_wp, 1094.2695_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1967.8827_wp,  778.5614_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1125.9127_wp,  514.7308_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2507.0805_wp,  981.8553_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1315.2670_wp,  576.2453_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1122.1393_wp,  518.8345_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2241.8085_wp,  894.3917_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1328.5415_wp,  581.9305_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1092.1025_wp,  501.9170_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1674.4360_wp,  689.3091_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1224.2314_wp,  542.3803_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  933.3543_wp,  443.0547_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1891.5883_wp,  769.5990_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1447.4665_wp,  609.8655_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  802.2571_wp,  390.5841_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1388.6668_wp,  585.2065_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1028.0158_wp,  471.9441_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  750.7432_wp,  368.9368_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1514.2016_wp,  639.5705_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  959.3598_wp,  441.4769_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  790.1791_wp,  376.4231_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1386.0622_wp,  592.7151_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  885.3477_wp,  409.8444_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  831.9541_wp,  393.3803_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1169.0360_wp,  494.5443_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  860.9698_wp,  396.5804_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1938 * max_ref * max_ref + 1 : 1957 * max_ref * max_ref) = [ &
      & 1201.2473_wp,  521.8571_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  826.2261_wp,  391.7491_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1570.9643_wp,  656.2319_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1470.5492_wp,  624.9431_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1112.6065_wp,  501.3687_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  899.2107_wp,  425.8977_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1378.0830_wp,  605.1903_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1260.7608_wp,  563.8386_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1206.2616_wp,  544.8078_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1054.8803_wp,  489.9104_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  925.8883_wp,  442.2254_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1168.7524_wp,  537.9899_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1138.0116_wp,  527.0919_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1097.2997_wp,  512.3917_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1046.5046_wp,  493.8687_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1053.1255_wp,  498.4291_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1050.4168_wp,  498.0193_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1038.4792_wp,  493.8976_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  923.0194_wp,  449.9920_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  932.1166_wp,  454.2236_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  806.0002_wp,  404.0174_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 5394.4846_wp, 1752.1410_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1412.9908_wp,  614.7789_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4481.0449_wp, 1598.6396_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3770.6523_wp, 1329.7509_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1462.6834_wp,  647.8742_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3745.6921_wp, 1400.5068_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2517.2239_wp,  986.0439_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1479.4584_wp,  670.6863_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1395.1471_wp,  635.9157_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3907.1923_wp, 1415.2709_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1349.2524_wp,  616.6572_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3717.5407_wp, 1358.1904_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1320.5717_wp,  603.8868_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3615.2067_wp, 1323.9329_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1275.6493_wp,  584.6788_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3522.0508_wp, 1292.4475_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1292.4475_wp,  587.7891_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   79.7138_wp,   39.2564_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  136.6695_wp,   63.7629_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   47.9365_wp,   25.6913_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1928.0810_wp,  653.6846_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  518.9498_wp,  211.7400_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  905.1101_wp,  365.1112_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  650.6557_wp,  269.2305_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  380.3011_wp,  172.7866_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1957 * max_ref * max_ref + 1 : 1976 * max_ref * max_ref) = [ &
      &  545.8015_wp,  241.6343_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  449.4102_wp,  202.6148_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  369.6195_wp,  169.5824_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  263.5484_wp,  127.7725_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  245.7832_wp,  120.0875_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  340.0580_wp,  161.3933_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  320.1657_wp,  151.4713_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  251.9099_wp,  122.9404_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  235.8641_wp,  115.0733_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  183.5940_wp,   93.5193_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  224.1383_wp,  112.0512_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  209.0700_wp,  104.5097_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  194.9397_wp,   98.0628_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  170.5870_wp,   86.3386_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  162.8638_wp,   84.4731_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  144.6075_wp,   75.6908_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  125.9329_wp,   66.7166_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  119.1734_wp,   63.7996_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   93.3410_wp,   51.7557_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   89.1988_wp,   48.9809_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2278.4898_wp,  780.2674_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  774.9719_wp,  310.6864_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1495.1792_wp,  585.2367_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1211.5825_wp,  481.1216_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  708.1495_wp,  308.8219_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1301.8438_wp,  534.3757_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1164.1089_wp,  483.2415_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  864.8322_wp,  373.6925_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  632.0359_wp,  288.3313_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  948.1288_wp,  415.9150_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  867.0844_wp,  383.7764_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  797.4609_wp,  356.3677_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  784.6310_wp,  347.8087_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  610.7486_wp,  283.7443_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  692.8338_wp,  321.2120_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  671.3009_wp,  311.7158_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  665.2819_wp,  307.7168_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  606.3141_wp,  284.7085_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  551.9390_wp,  265.0869_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  546.5395_wp,  262.2219_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  535.1318_wp,  256.9068_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  433.5831_wp,  215.4603_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  429.4810_wp,  213.2614_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  343.0307_wp,  175.5908_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3879.4729_wp, 1285.6965_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1031.6716_wp,  425.9370_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2809.5233_wp, 1039.4662_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2257.1993_wp,  831.2750_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  998.6585_wp,  433.5252_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2255.5157_wp,  854.8216_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  964.2855_wp,  421.6969_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  930.1750_wp,  414.5893_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2126.4190_wp,  822.1810_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  930.5654_wp,  407.0094_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  939.4890_wp,  419.5622_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1918.4254_wp,  751.0893_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  913.0014_wp,  398.9816_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  863.4210_wp,  384.2576_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1976 * max_ref * max_ref + 1 : 1995 * max_ref * max_ref) = [ &
      & 1511.4156_wp,  591.6660_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  911.9735_wp,  395.7987_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  619.9599_wp,  287.5236_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1616.4023_wp,  644.4421_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  806.5432_wp,  362.0493_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  514.3112_wp,  244.9612_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1265.0643_wp,  505.6064_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  738.6836_wp,  334.5615_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  506.7226_wp,  240.5922_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1294.9313_wp,  533.0040_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  705.2129_wp,  319.1993_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  623.9865_wp,  284.0111_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  537.4395_wp,  251.6701_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1354.4595_wp,  550.3952_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  691.6538_wp,  312.2221_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  572.5728_wp,  264.0373_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  620.7184_wp,  278.5957_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1040.3490_wp,  422.1376_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  703.6592_wp,  309.8959_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1014.8959_wp,  429.9304_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  625.7820_wp,  288.4004_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1221.3030_wp,  510.2157_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1129.3623_wp,  478.0396_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  848.2084_wp,  375.6249_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  656.9043_wp,  304.8174_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1012.1053_wp,  445.6375_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  933.8615_wp,  416.0190_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  880.4215_wp,  395.4915_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  769.9512_wp,  353.0881_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  658.9099_wp,  310.2550_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  817.3888_wp,  376.9991_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  801.1286_wp,  370.4418_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  775.8051_wp,  360.4255_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  740.8479_wp,  346.7365_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  709.6039_wp,  336.6910_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  710.3376_wp,  336.8539_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  704.1627_wp,  334.1734_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  600.4064_wp,  293.3036_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  605.6994_wp,  295.4210_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  506.4098_wp,  254.0859_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4311.0961_wp, 1438.5445_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1216.1018_wp,  507.2285_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3368.5646_wp, 1239.5504_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2771.3234_wp, 1012.8078_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1194.1104_wp,  519.2271_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2809.6885_wp, 1077.0858_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1922.5226_wp,  762.0033_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1101.7008_wp,  496.5039_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2449.8188_wp,  963.2715_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1286.4106_wp,  558.2951_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1098.1031_wp,  499.8637_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2190.9387_wp,  875.5931_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1299.4488_wp,  563.9816_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1068.5805_wp,  483.8542_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1636.7956_wp,  672.4914_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1197.4564_wp,  524.9085_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  913.4679_wp,  425.7815_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(1995 * max_ref * max_ref + 1 : 2014 * max_ref * max_ref) = [ &
      & 1848.9097_wp,  751.7567_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1415.0358_wp,  593.2386_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  785.2898_wp,  374.4473_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1357.6810_wp,  569.5060_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1005.8586_wp,  455.1389_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  734.9066_wp,  353.3996_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1480.5878_wp,  622.3214_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  938.6688_wp,  425.6722_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  773.3122_wp,  361.7114_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1355.4425_wp,  575.9965_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  866.2748_wp,  394.9658_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  814.1845_wp,  378.3019_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1142.9352_wp,  481.1524_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  842.3966_wp,  382.4279_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1174.8699_wp,  506.3303_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  808.6413_wp,  376.6190_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1535.9254_wp,  639.2239_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1437.9844_wp,  607.6149_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1088.4965_wp,  484.4102_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  880.0979_wp,  409.4190_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1348.0216_wp,  586.3317_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1233.4280_wp,  545.1890_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1180.2039_wp,  526.2392_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1032.3159_wp,  471.8373_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  906.2740_wp,  424.6928_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1143.6919_wp,  518.5975_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1113.6574_wp,  507.7568_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1073.8792_wp,  493.1741_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1024.2500_wp,  474.8294_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1030.7380_wp,  479.0649_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1028.0990_wp,  478.5801_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1016.4371_wp,  474.4660_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  903.5630_wp,  431.2267_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  912.4671_wp,  435.2963_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  789.1320_wp,  386.1460_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 5258.2068_wp, 1751.8319_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1381.7863_wp,  595.9499_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4374.0826_wp, 1584.5929_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3678.9146_wp, 1317.6067_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1430.7306_wp,  626.9684_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3658.2120_wp, 1380.9239_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2458.7886_wp,  965.8237_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1447.5288_wp,  647.4730_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1365.0615_wp,  613.6537_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3814.4921_wp, 1400.3880_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1320.1809_wp,  594.9070_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3629.8268_wp, 1342.9247_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1292.1268_wp,  582.5641_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3530.0228_wp, 1308.7424_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1248.1896_wp,  563.9015_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(2014 * max_ref * max_ref + 1 : 2033 * max_ref * max_ref) = [ &
      & 3439.1601_wp, 1277.3568_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1264.5213_wp,  567.3171_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3358.3122_wp, 1249.5054_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1249.5054_wp,  549.0281_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   64.1880_wp,   41.3275_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  108.3662_wp,   66.8235_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   39.5786_wp,   27.1595_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1407.7335_wp,  659.0924_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  396.7223_wp,  218.6195_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  689.9287_wp,  376.4610_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  499.1837_wp,  278.5080_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  299.3233_wp,  180.6115_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  426.4897_wp,  251.8424_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  352.9087_wp,  211.5485_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  291.6491_wp,  177.3613_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  211.2565_wp,  134.3264_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  197.4553_wp,  126.3205_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  270.9183_wp,  169.3159_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  254.8122_wp,  158.8301_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  202.2978_wp,  129.2461_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  189.3824_wp,  120.9324_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  149.2918_wp,   98.6213_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  181.2581_wp,  117.9977_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  169.0556_wp,  110.0265_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  157.9221_wp,  103.2648_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  138.4298_wp,   90.9100_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  133.1438_wp,   89.1359_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  118.5388_wp,   79.8949_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  103.6007_wp,   70.4331_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   98.3573_wp,   67.4030_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   77.8742_wp,   54.7600_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   74.1912_wp,   51.7756_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1667.3895_wp,  787.6187_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  589.6821_wp,  319.9572_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1130.8198_wp,  601.0384_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  919.6329_wp,  495.1147_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  551.1193_wp,  321.3077_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  996.7924_wp,  552.1545_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  893.9515_wp,  499.9820_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  671.2900_wp,  388.4558_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  498.0207_wp,  301.4874_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  739.0200_wp,  433.0662_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  677.4806_wp,  399.9816_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  624.7321_wp,  371.7819_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  613.2627_wp,  362.5021_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  483.6888_wp,  297.2065_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  548.3633_wp,  336.3700_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  531.5424_wp,  326.4592_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  526.1827_wp,  322.1177_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  481.6031_wp,  298.4773_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  441.2303_wp,  278.4551_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  436.7785_wp,  275.4054_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  427.7338_wp,  269.8237_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  350.0369_wp,  226.9109_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  346.6461_wp,  224.5698_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(2033 * max_ref * max_ref + 1 : 2052 * max_ref * max_ref) = [ &
      &  279.3515_wp,  185.2621_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2817.6333_wp, 1292.7733_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  790.7854_wp,  439.9765_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2094.5326_wp, 1059.7192_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1680.6338_wp,  847.4486_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  776.0695_wp,  450.5280_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1691.7052_wp,  874.1314_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  750.7848_wp,  438.5550_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  728.1122_wp,  432.1441_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1602.8927_wp,  842.8840_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  724.5886_wp,  423.3069_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  735.7972_wp,  437.4020_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1450.7174_wp,  771.1840_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  710.7237_wp,  414.8741_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  675.5150_wp,  400.3793_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1142.7015_wp,  607.3487_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  708.5769_wp,  411.2230_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  490.5998_wp,  300.6688_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1228.0426_wp,  663.1313_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  632.5743_wp,  377.5747_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  410.0863_wp,  256.7746_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  961.5600_wp,  520.2569_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  580.7773_wp,  349.2095_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  403.6739_wp,  252.1075_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  992.0673_wp,  550.5045_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  554.3462_wp,  333.1240_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  491.2047_wp,  296.4553_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  426.4723_wp,  263.3886_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1034.1932_wp,  567.6026_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  543.2742_wp,  325.7328_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  452.3968_wp,  275.9455_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  486.7180_wp,  290.3039_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  793.8763_wp,  435.0829_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  548.9119_wp,  322.3728_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  783.5114_wp,  445.4982_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  494.4243_wp,  301.5583_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  939.3652_wp,  528.1305_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  871.6966_wp,  495.5668_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  662.7701_wp,  391.3632_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  520.0484_wp,  319.1288_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  789.6369_wp,  464.0671_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  730.9227_wp,  433.7675_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  690.6823_wp,  412.7144_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  607.5073_wp,  369.2466_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  523.7968_wp,  325.3103_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  645.9849_wp,  394.5033_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  633.5809_wp,  387.7336_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  614.3654_wp,  377.4212_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  587.9119_wp,  363.3458_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  565.2991_wp,  353.2521_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  565.7914_wp,  353.3989_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  560.9891_wp,  350.6032_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  482.3172_wp,  308.4763_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  486.3444_wp,  310.6569_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  409.9803_wp,  267.7464_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(2052 * max_ref * max_ref + 1 : 2071 * max_ref * max_ref) = [ &
      & 3135.9242_wp, 1447.9360_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  934.6082_wp,  524.5374_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2507.7589_wp, 1262.8491_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2059.4692_wp, 1031.5808_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  928.2866_wp,  539.6227_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2113.2076_wp, 1103.0742_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1458.1969_wp,  784.0479_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  865.0011_wp,  518.0912_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1854.4837_wp,  989.5940_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  999.3615_wp,  580.1629_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  864.5896_wp,  522.1512_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1665.4372_wp,  901.2747_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1009.4419_wp,  585.9401_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  840.0098_wp,  505.0620_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1253.0730_wp,  694.2958_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  932.8332_wp,  545.9255_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  723.9342_wp,  445.6125_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1411.6759_wp,  775.2973_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1088.9839_wp,  613.9817_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  626.3824_wp,  392.6667_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1045.0617_wp,  589.2413_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  790.5176_wp,  474.8590_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  587.6156_wp,  370.8350_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1140.4525_wp,  644.1313_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  738.1317_wp,  444.1417_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  613.3959_wp,  378.4806_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1047.1279_wp,  596.8654_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  682.2073_wp,  412.2532_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  644.6362_wp,  395.6090_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  880.5313_wp,  497.8410_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  662.6029_wp,  398.9285_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  911.0796_wp,  525.4121_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  640.7393_wp,  394.0161_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1179.9720_wp,  660.9815_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1109.2250_wp,  629.3894_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  851.5816_wp,  504.6237_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  697.1742_wp,  428.4350_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1048.0976_wp,  609.4014_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  963.2234_wp,  567.6368_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  923.8826_wp,  548.4168_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  813.7102_wp,  492.9882_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  719.4208_wp,  444.8426_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  899.5162_wp,  541.4600_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  877.2337_wp,  530.4464_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  847.6165_wp,  515.5966_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  810.5952_wp,  496.8907_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  816.3047_wp,  501.4441_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  814.5718_wp,  501.0205_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  805.9652_wp,  496.8526_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  720.9847_wp,  452.4916_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  728.0053_wp,  456.7532_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  634.2163_wp,  406.0479_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3823.8426_wp, 1763.6917_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1071.5717_wp,  618.8508_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(2071 * max_ref * max_ref + 1 : 2090 * max_ref * max_ref) = [ &
      & 3243.6561_wp, 1611.3765_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2720.7653_wp, 1339.0290_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1114.5114_wp,  652.1810_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2740.5478_wp, 1411.6458_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1860.7370_wp,  992.8293_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1134.1326_wp,  675.1196_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1070.9597_wp,  639.9970_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2838.1357_wp, 1426.3405_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1036.4403_wp,  620.5881_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2705.6826_wp, 1368.9945_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1014.5593_wp,  607.7340_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2632.6657_wp, 1334.4823_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  980.6114_wp,  588.3813_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2566.0607_wp, 1302.7626_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  991.5176_wp,  591.5368_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2506.8352_wp, 1274.6283_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  974.0690_wp,  571.0450_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1891.6719_wp,  999.0617_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  999.0617_wp,  595.3628_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   70.1794_wp,   38.9150_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  120.4056_wp,   62.7971_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   42.3679_wp,   25.6695_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1802.3852_wp,  612.8166_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  463.5442_wp,  204.5514_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  808.4866_wp,  352.0753_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  582.1224_wp,  260.7245_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  335.3665_wp,  169.5717_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  482.7977_wp,  236.2659_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  397.4499_wp,  198.6025_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  327.2333_wp,  166.6202_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  232.1028_wp,  126.4184_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  216.4708_wp,  118.9198_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  299.8469_wp,  159.2422_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  282.8547_wp,  149.3811_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  222.0934_wp,  121.6899_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  208.1604_wp,  113.8754_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  161.9091_wp,   93.0070_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  197.6234_wp,  111.2043_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  184.5904_wp,  103.7034_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  172.0857_wp,   97.3591_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  150.8204_wp,   85.7475_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  143.7754_wp,   84.1363_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  127.7828_wp,   75.4465_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  111.4277_wp,   66.5559_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  105.3970_wp,   63.7152_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   82.7154_wp,   51.8414_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   79.0484_wp,   49.0050_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(2090 * max_ref * max_ref + 1 : 2109 * max_ref * max_ref) = [ &
      & 2124.0831_wp,  732.6457_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  695.0211_wp,  299.2468_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1343.3651_wp,  561.5991_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1089.8214_wp,  462.8827_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  626.4637_wp,  301.2799_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1162.1560_wp,  516.7263_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1038.1047_wp,  468.0868_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  768.1313_wp,  364.1571_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  557.1841_wp,  283.1061_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  839.1776_wp,  406.1522_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  767.3055_wp,  375.2459_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  705.0006_wp,  348.9033_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  695.1617_wp,  340.1321_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  538.2012_wp,  279.2705_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  610.8318_wp,  316.0555_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  592.0135_wp,  306.7678_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  587.0972_wp,  302.6610_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  534.4195_wp,  280.5856_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  486.2319_wp,  261.9772_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  481.5894_wp,  259.1043_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  471.5858_wp,  253.8632_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  381.9584_wp,  213.7617_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  378.3984_wp,  211.5547_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  302.4149_wp,  174.7380_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3687.6305_wp, 1201.3307_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  927.3672_wp,  412.0655_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2564.1443_wp,  988.5095_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2081.6173_wp,  790.5227_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  887.0421_wp,  422.5156_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2045.4927_wp,  816.0391_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  855.1471_wp,  411.3969_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  823.6822_wp,  405.6040_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1922.8302_wp,  787.4378_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  827.7970_wp,  397.1190_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  831.7726_wp,  410.5703_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1730.9890_wp,  720.7679_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  811.9905_wp,  389.2080_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  766.0776_wp,  375.8195_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1368.0049_wp,  567.7536_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  812.0495_wp,  385.7107_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  548.8466_wp,  282.6706_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1454.1614_wp,  620.1711_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  713.7552_wp,  354.4940_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  454.6351_wp,  241.6291_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1140.8713_wp,  486.6971_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  653.4752_wp,  327.9705_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  447.9970_wp,  237.2164_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1157.9011_wp,  515.3839_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  624.4481_wp,  312.8735_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  553.1437_wp,  278.5319_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  475.3065_wp,  247.7080_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1213.2014_wp,  531.1595_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  612.6329_wp,  305.9112_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  506.6985_wp,  259.3743_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  550.9242_wp,  272.6492_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  935.7544_wp,  407.2414_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  625.2396_wp,  302.5447_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(2109 * max_ref * max_ref + 1 : 2128 * max_ref * max_ref) = [ &
      &  903.5991_wp,  417.4763_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  552.5803_wp,  283.3765_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1090.0939_wp,  494.6170_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1006.1273_wp,  464.3202_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  752.0406_wp,  367.2307_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  579.2127_wp,  299.8930_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  896.4651_wp,  435.3339_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  826.5104_wp,  407.0718_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  778.5964_wp,  387.4235_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  679.7752_wp,  346.8564_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  580.5610_wp,  305.8457_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  721.2042_wp,  370.6421_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  706.8707_wp,  364.3169_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  684.3805_wp,  354.6851_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  653.1815_wp,  341.5412_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  625.4480_wp,  332.2173_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  626.1705_wp,  332.3519_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  620.7464_wp,  329.7337_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  528.9381_wp,  290.4179_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  533.6485_wp,  292.4564_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  446.2084_wp,  252.3325_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4098.5678_wp, 1345.9552_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1093.0934_wp,  491.4621_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3083.9899_wp, 1177.8523_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2566.8729_wp,  962.0972_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1062.1534_wp,  506.1454_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2548.5911_wp, 1030.2577_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1752.1144_wp,  733.2622_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  974.9708_wp,  486.4650_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2211.4907_wp,  925.0730_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1146.1276_wp,  544.1619_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  970.8551_wp,  490.4332_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1973.5648_wp,  843.0014_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1155.3947_wp,  549.5950_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  947.3214_wp,  474.3613_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1470.5265_wp,  650.1095_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1066.7077_wp,  512.2607_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  807.7328_wp,  418.9542_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1663.1347_wp,  725.6560_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1272.7235_wp,  575.3515_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  693.6613_wp,  369.4767_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1217.3511_wp,  552.1793_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  891.8764_wp,  446.0410_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  648.9904_wp,  349.0479_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1323.5223_wp,  603.5669_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  832.9670_wp,  417.2520_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  684.9715_wp,  355.9594_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1209.6847_wp,  559.4921_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  768.7904_wp,  387.3909_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  720.7315_wp,  371.9667_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1025.6835_wp,  466.6597_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  747.7247_wp,  374.8322_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1046.5789_wp,  492.7625_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  714.6206_wp,  370.4592_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(2128 * max_ref * max_ref + 1 : 2147 * max_ref * max_ref) = [ &
      & 1375.6067_wp,  619.1366_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1284.5648_wp,  589.8300_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  966.3182_wp,  473.7077_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  777.0365_wp,  402.7499_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1197.5597_wp,  571.5751_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1094.0703_wp,  532.6855_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1045.8286_wp,  514.7943_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  912.5542_wp,  463.1355_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  799.4055_wp,  418.2405_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1011.2975_wp,  508.5171_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  984.3669_wp,  498.2630_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  948.6782_wp,  484.4278_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  904.0756_wp,  466.9936_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  909.9127_wp,  471.3194_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  907.4929_wp,  470.9435_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  897.0258_wp,  467.0673_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  796.6995_wp,  425.6942_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  804.5343_wp,  429.6945_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  695.4350_wp,  382.3429_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 5022.6967_wp, 1639.3995_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1235.5701_wp,  580.3841_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4030.0910_wp, 1502.1922_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3441.9852_wp, 1248.0182_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1273.3916_wp,  611.9003_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3335.7435_wp, 1317.8387_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2250.9654_wp,  928.2705_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1282.3779_wp,  633.7528_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1209.6287_wp,  600.9518_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3507.1716_wp, 1330.3810_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1169.6897_wp,  582.7773_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3325.0797_wp, 1277.2241_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1144.6770_wp,  570.7175_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3231.3241_wp, 1245.1109_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1105.6577_wp,  552.5843_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3146.1740_wp, 1215.5853_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1121.8549_wp,  555.4295_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3070.3853_wp, 1189.3991_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1111.1877_wp,  535.8760_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2276.2532_wp,  933.7228_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1130.0310_wp,  558.9762_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2851.6677_wp, 1053.6587_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1053.6587_wp,  524.9195_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   67.9580_wp,   39.3967_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  116.3178_wp,   63.7816_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   41.1622_wp,   25.8658_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(2147 * max_ref * max_ref + 1 : 2166 * max_ref * max_ref) = [ &
      & 1720.7353_wp,  638.4850_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  445.0101_wp,  209.6685_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  775.7718_wp,  361.1822_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  559.2917_wp,  266.9922_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  323.5941_wp,  172.5188_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  465.2439_wp,  240.7836_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  383.2952_wp,  202.1595_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  315.8875_wp,  169.4228_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  224.5940_wp,  128.1009_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  209.5360_wp,  120.4474_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  289.8455_wp,  161.5691_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  273.3375_wp,  151.5963_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  214.9399_wp,  123.2619_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  201.4294_wp,  115.3491_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  156.9831_wp,   93.9771_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  191.4550_wp,  112.4836_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  178.8074_wp,  104.8991_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  166.7356_wp,   98.4445_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  146.1445_wp,   86.6740_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  139.4937_wp,   84.9288_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  124.0158_wp,   76.1209_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  108.1820_wp,   67.1060_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  102.3811_wp,   64.2056_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   80.4607_wp,   52.1454_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   76.8557_wp,   49.3166_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2027.8568_wp,  762.5439_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  666.6515_wp,  307.1464_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1287.2471_wp,  577.4805_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1045.1961_wp,  475.4827_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  603.2928_wp,  307.3477_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1116.0341_wp,  529.4209_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  997.3837_wp,  479.1923_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  739.4538_wp,  371.7482_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  537.7266_wp,  287.9502_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  808.3322_wp,  414.1717_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  739.3975_wp,  382.4286_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  679.6544_wp,  355.3534_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  669.8712_wp,  346.6120_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  519.8201_wp,  283.7221_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  589.8941_wp,  321.1363_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  571.7481_wp,  311.6696_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  566.8734_wp,  307.5736_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  516.3864_wp,  284.8700_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  470.2977_wp,  265.6124_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  465.7758_wp,  262.7163_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  456.1062_wp,  257.3927_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  369.9833_wp,  216.2940_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  366.5166_wp,  214.0702_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  293.2905_wp,  176.5117_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3526.9705_wp, 1255.3434_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  890.5056_wp,  422.0535_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2451.7734_wp, 1021.3342_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1992.8728_wp,  817.4261_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  853.8199_wp,  431.1749_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1957.4581_wp,  841.3784_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  822.8420_wp,  419.5774_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  793.8541_wp,  413.1795_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(2166 * max_ref * max_ref + 1 : 2185 * max_ref * max_ref) = [ &
      & 1841.5883_wp,  810.5712_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  797.2364_wp,  405.0758_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  801.7112_wp,  418.1829_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1658.7019_wp,  741.1995_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  781.8272_wp,  397.0180_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  738.2489_wp,  382.8866_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1310.7538_wp,  583.8816_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  781.6260_wp,  393.6386_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  529.8069_wp,  287.2154_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1394.4853_wp,  636.8428_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  688.1125_wp,  360.9437_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  439.4052_wp,  245.1092_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1094.0016_wp,  499.7029_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  630.2533_wp,  333.7444_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  432.9245_wp,  240.6788_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1111.8174_wp,  527.9435_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  602.2672_wp,  318.4010_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  533.5343_wp,  283.3520_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  459.0336_wp,  251.5373_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1164.2755_wp,  544.6244_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  590.7694_wp,  311.3689_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  489.0187_wp,  263.6377_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  530.9799_wp,  277.6170_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  897.8373_wp,  417.6349_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  602.1416_wp,  308.4467_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  868.7577_wp,  426.7546_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  533.3938_wp,  288.0409_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1047.5640_wp,  506.1315_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  967.4193_wp,  474.6781_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  724.6816_wp,  374.2592_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  559.3590_wp,  304.6980_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  863.5860_wp,  443.8226_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  796.6299_wp,  414.6873_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  750.7323_wp,  394.4535_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  656.0889_wp,  352.6781_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  561.0491_wp,  310.4631_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  696.2760_wp,  376.7249_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  682.5126_wp,  370.2368_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  660.9421_wp,  360.3420_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  631.0316_wp,  346.8276_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  604.6048_wp,  337.0758_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  605.2828_wp,  337.2239_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  600.0555_wp,  334.5522_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  511.9716_wp,  294.1510_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  516.4905_wp,  296.2434_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  432.3893_wp,  255.1809_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3921.6962_wp, 1405.6043_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1050.0925_wp,  503.0095_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2948.5688_wp, 1217.6072_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2457.8402_wp,  995.6332_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1022.3708_wp,  516.4699_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2440.1762_wp, 1061.3052_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1683.0549_wp,  753.6872_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  940.1178_wp,  495.1913_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2119.4908_wp,  950.9922_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1102.5614_wp,  555.3166_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  936.5992_wp,  498.9051_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(2185 * max_ref * max_ref + 1 : 2204 * max_ref * max_ref) = [ &
      & 1892.7858_wp,  865.5324_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1110.6913_wp,  560.7912_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  913.6348_wp,  482.7380_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1411.7363_wp,  666.0895_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1027.1026_wp,  522.4406_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  779.9682_wp,  425.5550_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1596.2074_wp,  744.0872_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1223.2739_wp,  588.6854_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  670.5114_wp,  374.7726_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1169.6785_wp,  564.8984_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  859.9729_wp,  453.9296_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  627.5659_wp,  353.8700_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1271.8937_wp,  617.3535_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  803.2511_wp,  424.5778_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  661.4373_wp,  361.4953_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1163.0374_wp,  571.8202_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  741.5149_wp,  394.0563_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  695.7803_wp,  377.9042_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  985.5697_wp,  477.2912_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  721.0103_wp,  381.3842_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1006.8382_wp,  503.1205_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  690.0423_wp,  376.3121_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1321.5257_wp,  633.7493_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1234.8582_wp,  603.0805_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  931.2135_wp,  482.6219_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  750.3549_wp,  409.1408_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1152.7975_wp,  583.2130_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1053.9748_wp,  542.9404_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1007.9093_wp,  524.3974_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  880.5098_wp,  471.0117_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  772.2715_wp,  424.6821_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  975.4435_wp,  517.4288_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  949.7160_wp,  506.8159_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  915.6032_wp,  492.5152_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  872.9520_wp,  474.5058_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  878.6753_wp,  478.8244_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  876.4054_wp,  478.3961_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  866.4126_wp,  474.3763_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  770.2908_wp,  431.7667_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  777.8548_wp,  435.8349_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  673.1394_wp,  387.2187_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4812.3323_wp, 1712.8830_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1188.8012_wp,  592.5973_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3852.2949_wp, 1555.2120_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3298.9422_wp, 1294.3501_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1226.1232_wp,  624.0645_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3192.4039_wp, 1359.3802_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2163.4599_wp,  954.9829_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1236.0668_wp,  645.4413_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1166.1633_wp,  611.8361_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3355.8765_wp, 1375.8472_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1127.7999_wp,  593.2406_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(2204 * max_ref * max_ref + 1 : 2223 * max_ref * max_ref) = [ &
      & 3179.9629_wp, 1319.7747_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1103.6788_wp,  580.9422_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3090.4072_wp, 1286.3433_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1066.1553_wp,  562.4103_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3009.0778_wp, 1255.6329_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1081.6001_wp,  565.5841_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2936.7065_wp, 1228.3897_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1069.9921_wp,  546.4196_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2181.0675_wp,  960.8995_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1089.2614_wp,  569.2318_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2729.9739_wp, 1091.3776_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1015.8840_wp,  534.3480_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2617.3310_wp, 1051.8609_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1051.8609_wp,  544.4356_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   72.2203_wp,   37.0295_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  123.5220_wp,   59.6586_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   43.5711_wp,   24.4806_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1704.9679_wp,  575.7499_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  465.4100_wp,  193.4738_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  811.3688_wp,  332.8646_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  583.7067_wp,  246.7464_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  343.2850_wp,  160.9616_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  491.8875_wp,  224.0856_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  405.3285_wp,  188.4765_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  333.5597_wp,  158.2170_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  238.5984_wp,  120.2375_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  222.5870_wp,  113.1311_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  307.5198_wp,  151.3607_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  289.3920_wp,  141.9797_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  228.0857_wp,  115.7612_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  213.5146_wp,  108.3262_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  166.5380_wp,   88.5822_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  203.1554_wp,  105.8582_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  189.4531_wp,   98.7212_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  176.7026_wp,   92.6939_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  154.6287_wp,   81.6520_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  147.8355_wp,   80.1704_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  131.3004_wp,   71.9104_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  114.3820_wp,   63.4527_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  108.3076_wp,   60.7598_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   84.9478_wp,   49.4793_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   81.1416_wp,   46.7595_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2016.3392_wp,  688.5910_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  694.0742_wp,  282.8651_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1337.4783_wp,  530.3870_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1084.2313_wp,  437.4255_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  637.8198_wp,  285.5962_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(2223 * max_ref * max_ref + 1 : 2242 * max_ref * max_ref) = [ &
      & 1167.9358_wp,  488.8472_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1045.0038_wp,  443.0078_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  778.1323_wp,  345.1182_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  570.6508_wp,  268.7620_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  854.0740_wp,  385.0952_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  781.3748_wp,  355.8982_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  719.0323_wp,  331.0143_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  706.9630_wp,  322.6156_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  551.8787_wp,  265.2733_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  625.9394_wp,  300.1974_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  606.4940_wp,  291.3907_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  600.8855_wp,  287.4557_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  548.0877_wp,  266.6113_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  499.4576_wp,  249.0984_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  494.5285_wp,  246.3593_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  484.2101_wp,  241.3800_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  392.9385_wp,  203.4529_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  389.1985_wp,  201.3478_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  311.2487_wp,  166.4466_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3419.5160_wp, 1127.9480_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  924.6197_wp,  389.9518_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2501.4164_wp,  931.7443_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2006.2958_wp,  745.1781_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  898.6515_wp,  400.4754_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2012.0352_wp,  769.8336_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  868.1104_wp,  390.0277_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  838.3710_wp,  384.7744_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1899.2044_wp,  743.4069_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  837.4618_wp,  376.5089_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  846.8568_wp,  389.5084_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1714.8635_wp,  680.7697_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  821.6274_wp,  368.9941_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  777.8991_wp,  356.5084_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1350.2932_wp,  536.2731_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  820.3071_wp,  365.5972_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  559.7016_wp,  268.4867_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1446.6020_wp,  586.1315_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  727.2378_wp,  336.3582_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  464.9730_wp,  229.6866_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1131.7710_wp,  460.0415_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  666.3378_wp,  311.2798_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  458.0409_wp,  225.4697_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1161.4905_wp,  487.6202_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  636.0357_wp,  296.9473_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  562.7864_wp,  264.4013_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  485.4961_wp,  235.3426_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1213.9348_wp,  502.3206_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  623.6996_wp,  290.3139_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  516.8454_wp,  246.3103_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  559.3819_wp,  258.6987_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  931.6793_wp,  385.1327_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  633.5260_wp,  286.8927_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  912.0082_wp,  395.3654_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  565.0858_wp,  269.0965_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1096.4450_wp,  468.2277_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1014.7443_wp,  439.7399_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  764.1475_wp,  348.3079_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  593.5044_wp,  284.8492_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(2242 * max_ref * max_ref + 1 : 2261 * max_ref * max_ref) = [ &
      &  911.7223_wp,  412.8162_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  841.7526_wp,  386.1634_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  793.9592_wp,  367.6206_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  695.1305_wp,  329.3403_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  595.7532_wp,  290.6346_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  738.2129_wp,  351.9875_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  723.6013_wp,  346.0085_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  700.8927_wp,  336.9098_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  669.5829_wp,  324.4969_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  641.7452_wp,  315.7701_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  642.3795_wp,  315.8929_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  636.8114_wp,  313.4113_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  543.7162_wp,  276.2768_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  548.4629_wp,  278.2032_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  459.1205_wp,  240.2275_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3800.8304_wp, 1264.1518_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1090.3500_wp,  465.2606_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2997.0212_wp, 1110.0442_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2460.9470_wp,  906.7106_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1074.3220_wp,  479.7811_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2507.3572_wp,  972.3864_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1717.0123_wp,  693.0607_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  993.5143_wp,  461.6481_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2190.0272_wp,  873.8994_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1156.8729_wp,  515.8086_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  990.8440_wp,  465.5596_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1960.5194_wp,  796.8327_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1168.8393_wp,  520.9380_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  963.5525_wp,  450.2413_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1466.8017_wp,  615.1052_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1077.4205_wp,  485.7214_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  825.0243_wp,  397.9983_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1655.9466_wp,  686.3381_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1268.9174_wp,  544.7906_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  710.0771_wp,  351.2333_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1218.0651_wp,  522.8164_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  906.9214_wp,  423.3418_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  664.7955_wp,  331.8942_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1329.1426_wp,  571.4828_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  846.3103_wp,  396.0475_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  698.3448_wp,  338.1899_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1217.6582_wp,  529.9388_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  781.2040_wp,  367.7646_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  735.1282_wp,  353.3208_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1025.3902_wp,  441.8976_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  759.5084_wp,  355.7893_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1056.3718_wp,  466.9433_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  730.4286_wp,  351.9076_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1377.8901_wp,  586.0395_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1291.3513_wp,  558.5785_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  980.6060_wp,  449.3638_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  795.0694_wp,  382.5686_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1213.1043_wp,  541.7910_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1111.0005_wp,  505.1927_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1063.6161_wp,  488.3600_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  931.6850_wp,  439.6941_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  819.1008_wp,  397.3735_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(2261 * max_ref * max_ref + 1 : 2280 * max_ref * max_ref) = [ &
      & 1031.8071_wp,  482.6582_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1005.0059_wp,  473.0076_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  969.4956_wp,  459.9774_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  925.1907_wp,  443.5507_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  931.1239_wp,  447.6960_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  928.8165_wp,  447.3606_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  918.4190_wp,  443.7162_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  817.3220_wp,  404.6813_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  825.3644_wp,  408.4793_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  714.6477_wp,  363.7360_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4632.8428_wp, 1539.8453_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1241.6267_wp,  550.0548_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3885.5093_wp, 1414.9943_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3259.8698_wp, 1175.4332_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1287.4672_wp,  580.1915_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3259.9132_wp, 1243.1883_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2193.8636_wp,  877.1478_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1304.7191_wp,  601.2846_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1230.5989_wp,  570.2451_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3391.5411_wp, 1253.7691_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1190.2918_wp,  553.0414_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3229.8168_wp, 1203.9727_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1165.0450_wp,  541.6048_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3141.6098_wp, 1173.7842_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1125.5430_wp,  524.4310_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3061.2537_wp, 1146.0194_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1139.6825_wp,  527.0243_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2989.7660_wp, 1121.3978_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1124.6771_wp,  508.1297_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2237.9026_wp,  881.7549_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1148.8304_wp,  530.3616_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2724.3602_wp,  992.6267_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1072.3978_wp,  498.1319_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2606.5136_wp,  957.3153_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1106.4740_wp,  506.8895_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2664.1668_wp, 1011.4677_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1011.4677_wp,  472.7950_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   70.7154_wp,   36.0084_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  120.9302_wp,   57.9760_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   42.6658_wp,   23.8289_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(2280 * max_ref * max_ref + 1 : 2299 * max_ref * max_ref) = [ &
      & 1664.7700_wp,  557.2454_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  455.2827_wp,  187.7050_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  793.6877_wp,  322.8840_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  571.0039_wp,  239.4431_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  336.0507_wp,  156.3709_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  481.4385_wp,  217.6279_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  396.7377_wp,  183.0885_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  326.4963_wp,  153.7302_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  233.6155_wp,  116.9007_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  217.9416_wp,  110.0016_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  301.0692_wp,  147.1244_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  283.3001_wp,  138.0041_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  223.3166_wp,  112.5588_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  209.0416_wp,  105.3302_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  163.0702_wp,   86.1743_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  198.9185_wp,  102.9588_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  185.4931_wp,   96.0193_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  173.0120_wp,   90.1627_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  151.3924_wp,   79.4291_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  144.7587_wp,   78.0079_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  128.5666_wp,   69.9791_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  111.9978_wp,   61.7568_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  106.0552_wp,   59.1423_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   83.1827_wp,   48.1805_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   79.4539_wp,   45.5279_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1968.9928_wp,  666.5534_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  678.8486_wp,  274.3732_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1307.9814_wp,  514.2872_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1060.3386_wp,  424.2493_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  624.2488_wp,  277.3103_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1142.5656_wp,  474.3095_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1022.3674_wp,  429.8982_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  761.4623_wp,  335.0783_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  558.6379_wp,  261.1100_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  835.8969_wp,  373.9540_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  764.7668_wp,  345.6425_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  703.7881_wp,  321.5131_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  691.9095_wp,  313.3296_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  540.2901_wp,  257.7790_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  612.7809_wp,  291.7100_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  593.7403_wp,  283.1587_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  588.2297_wp,  279.3232_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  536.5858_wp,  259.1140_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  489.0107_wp,  242.1588_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  484.1791_wp,  239.4936_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  474.0753_wp,  234.6551_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  384.7496_wp,  197.8642_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  381.0847_wp,  195.8155_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  304.7748_wp,  161.9300_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3337.2227_wp, 1091.5091_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  904.3257_wp,  378.4205_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2444.6758_wp,  902.8445_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1960.2391_wp,  722.1111_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  879.3887_wp,  388.8507_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1966.9050_wp,  746.1810_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  849.5394_wp,  378.7408_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  820.5423_wp,  373.7251_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(2299 * max_ref * max_ref + 1 : 2318 * max_ref * max_ref) = [ &
      & 1856.8743_wp,  720.7667_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  819.4973_wp,  365.6231_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  828.8555_wp,  378.3321_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1676.8129_wp,  660.1463_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  803.9989_wp,  358.3213_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  761.3040_wp,  346.2718_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1320.1912_wp,  520.0492_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  802.6609_wp,  354.9945_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  547.8503_wp,  260.9096_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1414.7048_wp,  568.5111_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  711.8001_wp,  326.7279_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  455.1812_wp,  223.2746_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1106.7246_wp,  446.2432_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  652.2155_wp,  302.4017_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  448.3896_wp,  219.1675_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1136.1897_wp,  473.1486_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  622.5372_wp,  288.4783_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  550.8249_wp,  256.8824_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  475.2458_wp,  228.7263_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1187.3873_wp,  487.3312_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  610.4511_wp,  282.0254_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  505.9042_wp,  239.3413_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  547.4456_wp,  251.2997_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  911.1707_wp,  373.6528_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  619.9597_wp,  278.6211_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  892.3332_wp,  383.7696_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  553.1687_wp,  261.4745_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1072.6768_wp,  454.4189_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  992.8398_wp,  426.8411_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  747.8611_wp,  338.2814_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  581.0286_wp,  276.8003_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  892.3017_wp,  400.8971_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  823.8687_wp,  375.0691_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  777.1265_wp,  357.0949_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  680.4664_wp,  319.9904_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  583.2649_wp,  282.4712_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  722.6654_wp,  342.0170_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  708.3653_wp,  336.2183_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  686.1478_wp,  327.3956_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  655.5207_wp,  315.3605_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  628.2934_wp,  306.9300_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  628.9108_wp,  307.0478_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  623.4596_wp,  304.6384_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  532.3641_wp,  268.6347_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  537.0079_wp,  270.5030_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  449.5592_wp,  233.6551_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3709.4297_wp, 1223.4668_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1066.4469_wp,  451.5684_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2928.7164_wp, 1075.5659_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2404.1111_wp,  878.5837_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1051.2437_wp,  465.8723_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2451.1775_wp,  942.6854_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1678.5781_wp,  672.2671_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  972.4302_wp,  448.4531_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2141.4202_wp,  847.4869_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1131.9421_wp,  500.8568_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  969.8725_wp,  452.3068_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(2318 * max_ref * max_ref + 1 : 2337 * max_ref * max_ref) = [ &
      & 1917.2203_wp,  772.9187_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1143.6857_wp,  505.8297_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  943.0626_wp,  437.4079_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1434.6130_wp,  596.8707_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1054.2484_wp,  471.7011_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  807.5992_wp,  386.7854_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1619.5186_wp,  665.8997_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1241.1151_wp,  528.8006_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  695.1436_wp,  341.4299_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1191.4716_wp,  507.4568_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  887.6202_wp,  411.2731_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  650.8338_wp,  322.6630_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1300.2637_wp,  554.6882_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  828.2826_wp,  384.7714_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  683.5612_wp,  328.6834_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1191.2962_wp,  514.4346_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  764.5705_wp,  357.3192_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  719.5693_wp,  343.3582_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1002.9797_wp,  428.9424_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  743.3221_wp,  345.6658_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1033.6009_wp,  453.3608_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  715.0198_wp,  341.9894_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1347.8497_wp,  568.7508_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1263.3503_wp,  542.1999_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  959.6719_wp,  436.4645_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  778.3235_wp,  371.7768_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1187.1040_wp,  526.0806_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1087.2901_wp,  490.6416_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1040.9736_wp,  474.3436_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  911.9836_wp,  427.2006_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  801.8905_wp,  386.1954_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1009.9583_wp,  468.8968_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  983.7511_wp,  459.5521_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  949.0275_wp,  446.9312_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  905.7052_wp,  431.0179_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  911.5137_wp,  435.0603_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  909.2616_wp,  434.7424_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  899.0951_wp,  431.2148_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  800.1918_wp,  393.3829_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  808.0654_wp,  397.0728_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  699.7231_wp,  353.6841_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4520.9645_wp, 1490.3329_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1214.7181_wp,  534.0818_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3796.0924_wp, 1370.8120_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3183.6011_wp, 1138.7329_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1259.8096_wp,  563.4346_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3186.2327_wp, 1205.0073_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2144.4704_wp,  850.7547_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1276.9567_wp,  584.0468_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1204.4185_wp,  553.9356_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3313.8879_wp, 1214.8410_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1164.9825_wp,  537.2399_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(2337 * max_ref * max_ref + 1 : 2356 * max_ref * max_ref) = [ &
      & 3156.1992_wp, 1166.6903_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1140.2779_wp,  526.1331_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3070.0842_wp, 1137.4645_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1101.6251_wp,  509.4631_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2991.6275_wp, 1110.5819_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1115.3978_wp,  511.9439_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2921.8310_wp, 1086.7437_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1100.5512_wp,  493.4684_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2187.7803_wp,  855.0138_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1124.3657_wp,  515.1733_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2661.1461_wp,  961.7148_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1049.5999_wp,  483.8982_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2546.1249_wp,  927.6008_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1082.8259_wp,  492.3380_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2603.9653_wp,  980.3403_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  990.0070_wp,  459.3155_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2545.1713_wp,  959.5536_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  959.5536_wp,  446.2314_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   69.3791_wp,   35.7480_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  118.6070_wp,   57.5937_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   41.8727_wp,   23.6297_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1627.0951_wp,  555.2598_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  446.0030_wp,  186.7210_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  777.4602_wp,  321.2393_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  559.3898_wp,  238.1420_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  329.5355_wp,  155.3867_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  471.9860_wp,  216.3099_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  388.9890_wp,  181.9383_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  320.1437_wp,  152.7300_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  229.1777_wp,  116.0754_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  213.8101_wp,  109.2144_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  295.3014_wp,  146.1173_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  277.8493_wp,  137.0581_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  219.0722_wp,  111.7498_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  205.0594_wp,  104.5694_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  160.0071_wp,   85.5107_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  195.1630_wp,  102.1888_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  181.9823_wp,   95.2979_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  169.7429_wp,   89.4774_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  148.5273_wp,   78.8150_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  142.0478_wp,   77.3874_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  126.1614_wp,   69.4137_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  109.9031_wp,   61.2454_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  104.0800_wp,   58.6465_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   81.6437_wp,   47.7551_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   77.9788_wp,   45.1298_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(2356 * max_ref * max_ref + 1 : 2375 * max_ref * max_ref) = [ &
      & 1924.6593_wp,  664.0994_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  664.8580_wp,  272.9668_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1280.7911_wp,  511.8009_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1038.3601_wp,  422.1177_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  611.9392_wp,  275.6811_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1119.3391_wp,  471.7945_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1001.6784_wp,  427.5655_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  746.3210_wp,  333.1239_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  547.8252_wp,  259.4546_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  819.4280_wp,  371.7271_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  749.7413_wp,  343.5475_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  690.0184_wp,  319.5322_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  678.2927_wp,  311.4138_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  529.8922_wp,  256.0890_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  600.9701_wp,  289.8009_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  582.2956_wp,  281.2978_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  576.8643_wp,  277.4949_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  526.2842_wp,  257.3786_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  479.6909_wp,  240.4740_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  474.9443_wp,  237.8283_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  465.0326_wp,  233.0206_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  377.4873_wp,  196.4062_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  373.8878_wp,  194.3732_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  299.0622_wp,  160.6761_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3259.9368_wp, 1087.6970_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  885.7538_wp,  376.3047_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2391.9954_wp,  898.8634_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1917.4640_wp,  718.8719_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  861.8975_wp,  386.5382_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1925.1263_wp,  742.7390_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  832.6961_wp,  376.4552_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  804.4238_wp,  371.4100_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1817.7881_wp,  717.2897_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  803.2032_wp,  363.4076_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  812.5860_wp,  375.9796_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1641.7354_wp,  656.8806_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  788.0064_wp,  356.1503_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  746.2943_wp,  344.1155_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1292.4384_wp,  517.4322_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  786.6336_wp,  352.8632_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  537.2011_wp,  259.1578_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1385.3712_wp,  565.5959_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  697.8565_wp,  324.6761_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  446.4219_wp,  221.7117_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1083.6957_wp,  443.9051_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  639.4782_wp,  300.4715_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  439.7506_wp,  217.6401_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1113.0259_wp,  470.5820_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  610.3600_wp,  286.6332_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  540.0415_wp,  255.2100_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  466.0487_wp,  227.1685_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1163.0369_wp,  484.7511_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  598.4935_wp,  280.2274_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  496.0618_wp,  237.7527_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  536.6581_wp,  249.6938_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  892.3516_wp,  371.6376_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  607.6633_wp,  276.9038_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(2375 * max_ref * max_ref + 1 : 2394 * max_ref * max_ref) = [ &
      &  874.3961_wp,  381.5809_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  542.4516_wp,  259.7613_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1050.9705_wp,  451.9002_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  972.8753_wp,  424.4203_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  733.1234_wp,  336.2103_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  569.8286_wp,  274.9818_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  874.7132_wp,  398.4787_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  807.7031_wp,  372.7595_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  761.9321_wp,  354.8650_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  667.2758_wp,  317.9241_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  572.0834_wp,  280.5719_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  708.6950_wp,  339.7906_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  694.6807_wp,  334.0190_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  672.9150_wp,  325.2371_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  642.9169_wp,  313.2573_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  616.2658_wp,  304.8342_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  616.8666_wp,  304.9520_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  611.5213_wp,  302.5558_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  522.2655_wp,  266.7093_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  526.8147_wp,  268.5684_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  441.0932_wp,  231.9047_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3623.6511_wp, 1219.0706_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1044.6045_wp,  448.9840_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2865.2613_wp, 1070.8351_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2351.2803_wp,  874.6707_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1030.2928_wp,  463.0740_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2399.2487_wp,  938.1873_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1643.2139_wp,  668.7656_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  953.4036_wp,  445.6186_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2096.6400_wp,  843.2361_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1109.3035_wp,  497.8395_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  950.9799_wp,  449.4035_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1877.4174_wp,  768.9101_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1120.8436_wp,  502.7808_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  924.5841_wp,  434.6007_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1405.1352_wp,  593.5778_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1033.2364_wp,  468.8041_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  791.9602_wp,  384.1823_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1586.1100_wp,  662.3109_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1215.7298_wp,  525.7481_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  681.7920_wp,  339.0464_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1167.1878_wp,  504.5373_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  870.2063_wp,  408.6237_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  638.3682_wp,  320.3788_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1273.9026_wp,  551.5239_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  812.0214_wp,  382.2739_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  670.2940_wp,  326.4378_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1167.2707_wp,  511.4434_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  749.5779_wp,  354.9728_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  705.5909_wp,  341.0443_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  982.5210_wp,  426.4375_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  728.7203_wp,  343.4086_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1012.8904_wp,  450.6589_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  701.1829_wp,  339.6919_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(2394 * max_ref * max_ref + 1 : 2413 * max_ref * max_ref) = [ &
      & 1320.3887_wp,  565.5647_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1237.8116_wp,  539.0868_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  940.7333_wp,  433.7382_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  763.2869_wp,  369.2991_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1163.4988_wp,  522.9356_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1065.8196_wp,  487.6285_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1020.4995_wp,  471.3898_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  894.2426_wp,  424.4358_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  786.4597_wp,  383.6007_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  990.2588_wp,  465.9059_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  964.6052_wp,  456.5948_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  930.6131_wp,  444.0225_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  888.2036_wp,  428.1726_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  893.9080_wp,  432.1732_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  891.7106_wp,  431.8507_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  881.7600_wp,  428.3345_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  784.8841_wp,  390.6589_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  792.6061_wp,  394.3257_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  686.4452_wp,  351.1344_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 4415.9694_wp, 1484.9661_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1190.2611_wp,  530.8706_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3712.8812_wp, 1364.9240_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3112.5536_wp, 1133.8329_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1234.7344_wp,  559.9910_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3118.0001_wp, 1199.3874_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2098.9702_wp,  846.3794_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1251.8739_wp,  580.3947_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1180.7776_wp,  550.4264_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3241.7266_wp, 1209.4663_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1142.1370_wp,  533.8225_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3087.8489_wp, 1161.4494_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1117.9234_wp,  522.7836_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3003.6922_wp, 1132.3365_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1080.0441_wp,  506.2081_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2927.0118_wp, 1105.5604_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1093.4602_wp,  508.7048_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2858.7977_wp, 1081.8160_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1078.6794_wp,  490.4334_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2141.5264_wp,  850.7546_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1102.2614_wp,  511.9281_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2602.3122_wp,  957.4948_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1029.0199_wp,  480.8218_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2489.9643_wp,  923.4785_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1061.4329_wp,  489.2608_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2548.1802_wp,  975.8085_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  970.6528_wp,  456.3728_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(2413 * max_ref * max_ref + 1 : 2432 * max_ref * max_ref) = [ &
      & 2490.6989_wp,  955.1104_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  940.8140_wp,  443.3639_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2437.4539_wp,  936.4457_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  936.4457_wp,  440.5228_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   68.4949_wp,   37.3986_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  117.1811_wp,   60.4871_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   41.2866_wp,   24.5435_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1612.7581_wp,  592.3651_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  441.3220_wp,  197.5946_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  769.3983_wp,  340.2385_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  553.4266_wp,  251.7223_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  325.6844_wp,  163.4855_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  466.6099_wp,  227.8802_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  384.4701_wp,  191.4109_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  316.3522_wp,  160.4620_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  226.3056_wp,  121.5617_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  211.1076_wp,  114.3083_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  291.6817_wp,  153.2146_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  274.4500_wp,  143.6930_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  216.3044_wp,  116.9313_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  202.4669_wp,  109.3879_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  157.8890_wp,   89.1915_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  192.6324_wp,  106.7340_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  179.6202_wp,   99.5059_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  167.5247_wp,   93.3808_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  146.5709_wp,   82.1806_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  140.1300_wp,   80.5863_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  124.4402_wp,   72.2182_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  108.3826_wp,   63.6396_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  102.6251_wp,   60.9031_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   80.4575_wp,   49.4502_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &   76.8581_wp,   46.7569_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1907.5332_wp,  708.0629_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  657.9922_wp,  289.0123_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1267.8866_wp,  542.7763_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1027.7356_wp,  447.1652_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  605.0843_wp,  290.7410_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1107.5215_wp,  499.1071_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  990.9808_wp,  452.0055_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  738.0233_wp,  351.3735_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  541.3978_wp,  272.9068_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  810.1845_wp,  391.8451_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  741.2030_wp,  361.9140_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  682.0827_wp,  336.4239_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  670.5479_wp,  327.9441_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  523.5513_wp,  269.0182_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  593.7911_wp,  304.4460_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  575.3262_wp,  295.4611_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  569.9851_wp,  291.5055_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  519.9103_wp,  270.1359_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  473.7390_wp,  251.9975_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  469.0567_wp,  249.2287_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  459.2634_wp,  244.1704_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  372.6267_wp,  205.2969_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  369.0769_wp,  203.1740_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp]

   c6ab_view(2432 * max_ref * max_ref + 1 : 2451 * max_ref * max_ref) = [ &
      &  295.0836_wp,  167.5526_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 3231.7117_wp, 1160.0767_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  876.2899_wp,  397.2856_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 2369.0958_wp,  955.0787_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1899.1306_wp,  763.2645_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  852.2547_wp,  407.3976_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1906.2802_wp,  788.4325_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  823.3089_wp,  396.5845_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  795.1856_wp,  390.9216_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1799.6377_wp,  760.5639_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  794.1404_wp,  382.7505_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  803.2347_wp,  395.6779_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1625.1430_wp,  696.0633_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  779.1219_wp,  375.1083_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  737.7269_wp,  362.0987_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1279.3493_wp,  547.9588_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  777.8232_wp,  371.7528_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  530.7581_wp,  271.9114_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1371.1198_wp,  598.7622_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  689.7847_wp,  341.5632_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  440.9182_wp,  232.2310_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1072.4959_wp,  469.5788_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  632.0123_wp,  315.9092_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  434.3472_wp,  228.0072_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1101.2182_wp,  497.4063_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  603.2352_wp,  301.3273_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  533.6923_wp,  268.0922_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  460.4063_wp,  238.2171_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1150.8553_wp,  512.7473_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  591.5255_wp,  294.6262_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  490.1530_wp,  249.5747_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  530.4351_wp,  262.4692_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  882.9919_wp,  392.8083_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  600.7572_wp,  291.4639_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  864.8546_wp,  402.7269_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  536.0077_wp,  272.8496_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      & 1039.6601_wp,  477.3755_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  962.2681_wp,  448.0328_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  724.7596_wp,  354.0246_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  563.0164_wp,  288.8264_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  864.7996_wp,  419.8297_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  798.4380_wp,  392.4501_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  753.1172_wp,  373.4254_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  659.3910_wp,  334.1439_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  565.1383_wp,  294.4333_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  700.2718_wp,  357.0217_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  686.4011_wp,  350.8906_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  664.8552_wp,  341.5611_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  635.1575_wp,  328.8344_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  608.7174_wp,  319.6852_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  609.3146_wp,  319.8129_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  604.0284_wp,  317.2786_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  515.6649_wp,  279.1274_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  520.1674_wp,  281.0990_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp,    0.0000_wp, &
      &  435.3535_wp,  242.2147