///////////////////////////////////////////////////////////////////////////////
//                                                         
// CPlusGenerator.cc
// -----------------
// CPlusGenerator implementation module
//                                               
// Design and Implementation by Bjoern Lemke
//
// (C)opyright 2000-2018 Bjoern Lemke
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; see the file COPYING.  If not, write to
// the Free Software Foundation, 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// IMPLEMENTATION MODULE
//
// Class: CPlusGenerator
// 
// Description: Implementation module for the dragon c++ generator 
//
///////////////////////////////////////////////////////////////////////////////

#include <lfcbase/Exception.h>
#include <lfcbase/Chain.h>
#include <lfcbase/File.h>
#include <lfcbase/Tokenizer.h>
#include <lfcbase/ListT.h>
#include <lfcbase/SetT.h>

#define LOADCOUNT 50

#include "CPlusGenerator.h"
#include "Worm.h"

CPlusGenerator::CPlusGenerator(const Chain& parserName, ParserMode mode, bool dynamicTable) : Dragon(parserName, mode, dynamicTable)
{
}

CPlusGenerator::~CPlusGenerator()
{
}

void CPlusGenerator::generateCode()
{
    generateHeaderCode();
    generateMethodCode();
}

void CPlusGenerator::generateHeaderCode()
{

    Chain headerFileName = _parserName + ".h";
    
    File fout(headerFileName);
    
    fout.open(File::WRITE);
    
    fout << "//\n";
    fout << "// File: "<< _parserName << ".h\n";
    fout << "// This code was produced by the dragon parser generator\n";
    fout << "//\n";
    fout << "#include <lfcbase/Exception.h>\n";
    fout << "#include <lfcbase/Chain.h>\n";
    fout << "#include <lfcbase/ListT.h>\n";
    fout << "#define MAXTOKENVAL 100\n";
    fout << "\n";
    fout << "class " << _parserName << "\n";
    fout << "{\n";
    fout << "\n";
    fout << "public:\n";
    fout << "\n";
    fout << "   " << _parserName << "();\n";
    fout << "   virtual ~" << _parserName << "();\n";
    fout << "\n";
    fout << "   void parse();\n";
    fout << "   ListT<Chain>& getTokenList() { return _tokenList; }\n";
    fout << "\n";
    fout << "protected:\n";
    fout << "\n";
    fout << "   virtual char nextChar() = 0;\n";
    fout << "   virtual void backChar() = 0;\n";

    SetT<Chain> actionList;

    Production *pProd = _productionSet.First();
    while (pProd)
    {
	if ( pProd->getAction() )
	{
	    actionList.Insert(pProd->getAction());
	}
	pProd = _productionSet.Next();
    }   

    Chain *pAction = actionList.First();
    while (pAction)
    {
	fout << "   virtual void " << *pAction << "() = 0;\n";
	pAction = actionList.Next();
    }

    fout << "\n";

    fout << "   enum Token {\n";
    
    Terminal *pTerm = _terminalSet.First();
    while (pTerm)
    {
	fout << "      " << pTerm->getName() << ",\n";
	pTerm = _terminalSet.Next();
    }

    // CHANGE : Dollar changed to ENDTOKEN
    fout << "      ENDTOKEN\n";   
    fout << "   };\n";

    fout << "   void setReserved(Token t);\n";

    fout << "\n";
    fout << "private:\n";
    fout << "\n";
    fout << "   enum Symbol { PROD, TOKEN, NONE };\n";
    fout << "   enum ScannerStateType { START, INTERMEDIATE, FINAL, ANY };\n";
    fout << "   enum Action { SHIFT, REDUCE, ACCEPT, NULLACTION };\n";
    fout << "   enum Production {\n";
    // filter out distinct production symbols
    SetT<Chain> symbolSet;

    pProd = _productionSet.First();
    while (pProd)
    {
	symbolSet.Insert(pProd->getName());
	pProd = _productionSet.Next();
    }   

    Chain *pS = symbolSet.First();
    while (pS)
    {
	fout << "      " << *pS;
	pS = symbolSet.Next();
	if (pS)
	    fout << ",\n";
	else
	    fout << "\n";
	  
    }
    fout << "   };\n";

    _numProd = symbolSet.Size();

    fout << "   void loadScanner();\n";
    
    int mcount = 0;
    fout << "   void loadParser" + Chain(mcount) + "();\n";
    int lcount=0;

    ParseTableEntry *pPTE = _parseTable.First();
    while (pPTE)
    {
	lcount++;
	pPTE = _parseTable.Next();
	if ( pPTE && lcount == LOADCOUNT )
	{
	    mcount++;
	    lcount=0;
	    fout << "   void loadParser" + Chain(mcount) + "();\n";
	}
    }

    mcount++;
    fout << "   void loadParser" + Chain(mcount) + "();\n";

    fout << "\n";
    fout << "   class ScannerStateEntry {\n";
    fout << "\n";
    fout << "   public:\n";
    fout << "\n";
    fout << "      ScannerStateEntry();\n";
    fout << "      ScannerStateEntry(int state, ScannerStateType type = INTERMEDIATE);\n";
    fout << "      ScannerStateEntry& operator = (const ScannerStateEntry& s);\n";
    fout << "      bool operator == (const ScannerStateEntry& s);\n";
    fout << "\n";
    fout << "	   int getState();\n";
    fout << "      ScannerStateType getType();\n";
    fout << "\n";
    fout << "   private:\n";
    fout << "\n";
    fout << "      int _state;\n";
    fout << "      ScannerStateType _type;\n";
    fout << "   };\n";
    fout << "\n";
    fout << "class ScannerTransEntry {\n";
    fout << "\n";
    fout << "   public:\n";
    fout << "\n";
    fout << "      ScannerTransEntry();\n";
    fout << "      ScannerTransEntry(int state, char c);\n";
    fout << "      ScannerTransEntry(int state, char c, int fstate);\n";
    fout << "      ScannerTransEntry& operator = (const ScannerTransEntry& s);\n";
    fout << "      bool operator == (const ScannerTransEntry& s);\n";
    fout << "\n";
    fout << "      int getFState();\n";
    fout << "\n";
    fout << "   private:\n";
    fout << "\n";
    fout << "      int  _state;\n";
    fout << "      char _c;\n";
    fout << "      int _fstate;\n";
    fout << "   };\n";
    fout << "\n";
    fout << "   class Scanner {\n";
    fout << "\n";
    fout << "   public:\n";
    fout << "\n";
    fout << "      Scanner();\n";
    fout << "      Scanner(Token t);\n";
    fout << "\n";
    fout << "      Token getToken();\n";
    fout << "      void addState(ScannerStateEntry s);\n";
    fout << "      void addTransition(ScannerTransEntry t);\n";
    fout << "      bool checkPattern(const Chain& pattern);\n";
    fout << "\n";
    fout << "   private:\n";
    fout << "\n";
    fout << "      Token _token;\n";
    fout << "      ListT<ScannerStateEntry> _stateList;\n";
    fout << "      ListT<ScannerTransEntry> _transList;\n";
    fout << "   };\n";
    fout << "\n";


    fout << "   ListT<Scanner> _scannerList;";
    fout << "\n";
    fout << "   class StackEntry {\n";
    fout << "\n";
    fout << "   public:\n";
    fout << "\n";
    fout << "      StackEntry();\n";
    fout << "      StackEntry(Symbol prod, int num, int state, Chain tval = \"\");\n";
    fout << "      StackEntry& operator = (const StackEntry& a);\n";
    fout << "\n";
    fout << "      int getState();\n";
    fout << "      Symbol getSymbol();\n";
    fout << "      Chain& getTval();\n";

    fout << "\n";
    fout << "   private:\n";
    fout << "\n";
    fout << "      Symbol _symbol;\n";
    fout << "      int _num;\n";
    fout << "      int _state;\n";
    fout << "      Chain _tval;\n";
    fout << "   };\n";
    fout << "\n";
    fout << "   class ActionEntry {\n";
    fout << "\n";
    fout << "   public:\n";
    fout << "\n";
    fout << "      ActionEntry();\n";
    fout << "      ActionEntry(Action action, int num);\n";
    fout << "      bool operator == (const ActionEntry& a);\n";
    fout << "      ActionEntry& operator = (const ActionEntry& a);\n";
    fout << "\n";
    fout << "      Action getAction();\n";
    fout << "      int getNum();\n";
    fout << "\n";
    fout << "   private:\n";
    fout << "\n";
    fout << "      Action _action;\n";
    fout << "      int _num;\n";
    fout << "   };\n";
    fout << "\n";


    if ( _dynamicTable )
    {
	fout << "   ActionEntry** _actionMap;\n";
	fout << "   int** _jumpMap;\n";
    }
    else
    {
	fout << "   ActionEntry _actionMap[" << _numStates + 1 << "][" << _terminalSet.Size()  + 1<< "];\n";
	fout << "   int _jumpMap[" << _numStates + 1 << "][" << _numProd + 1<< "];\n";
    }

    fout << "\n";
    fout << "   class ProdEntry {\n";
    fout << "\n";
    fout << "   public:\n";
    fout << "\n";
    fout << "      ProdEntry();\n";
    fout << "      ProdEntry(int id);\n";
    fout << "      ProdEntry(int id, Production prod, int numSymbol);\n";
    fout << "      bool operator == (const ProdEntry& a);\n";
    fout << "      ProdEntry& operator = (const ProdEntry& a);\n";
    fout << "\n";
    fout << "      int getId();\n";
    fout << "      Production getProd();\n";
    fout << "      int getNumSymbol();\n";
    fout << "\n";
    fout << "   private:\n";
    fout << "\n";
    fout << "      int _id;\n";
    fout << "      Production _prod;\n";
    fout << "      int _numSymbol;\n";
    fout << "\n";
    fout << "   };\n";
    fout << "\n";
    fout << "   ListT<ProdEntry> _prodInfo;\n";
    fout << "\n";
    fout << "   bool isSepIgnore(char c);\n";
    fout << "   bool isSepSign(char c);\n";
    fout << "   bool shiftToken();\n";

    fout << "\n";
    fout << "   bool getAction(int state, Token token, Action& a, int& num);\n";
    fout << "   bool getJump(int state, Production prod, int& fstate);\n";
    fout << "\n";
    fout << "   Token _reservedToken;\n";
    fout << "   bool _isReserved;\n";
    fout << "   Token _token;\n";
    fout << "   char _tokenVal[MAXTOKENVAL];\n";
    fout << "   ListT<Chain> _tokenList;\n";
    fout << "\n";
    fout << "};\n";

    fout.close();
}

void CPlusGenerator::generateMethodCode()
{

    Chain methodFileName = _parserName + ".cc";
    
    File fout(methodFileName);

    fout.open(File::WRITE);

    fout << "//\n";
    fout << "// File: "<< _parserName << ".cc\n";
    fout << "// This code was produced by the dragon parser generator\n";
    fout << "//\n";

    fout << "#include \"" << _parserName << ".h\"\n";
    fout << "#include <lfcbase/Exception.h>\n";
    fout << "#include <lfcbase/StackT.h>\n";

    fout << "\n";
    fout << _parserName << "::ScannerStateEntry::ScannerStateEntry()\n";
    fout << "{\n";
    fout << "}\n";
    fout << "\n";
    fout << _parserName << "::ScannerStateEntry::ScannerStateEntry(int state, ScannerStateType type)\n";
    fout << "{\n";
    fout << "   _state = state;\n";
    fout << "   _type = type;\n";
    fout << "}\n";
    fout << "\n";
    fout << _parserName << "::ScannerStateEntry& " << _parserName << "::ScannerStateEntry::operator = (const " << _parserName << "::ScannerStateEntry& s)\n";
    fout << "{\n";
    fout << "   _state = s._state;\n";
    fout << "   _type = s._type;\n";
    fout << "   return(*this);\n";
    fout << "}\n";
    fout << "\n";
    fout << "bool " << _parserName << "::ScannerStateEntry::operator == (const " << _parserName << "::ScannerStateEntry& s)\n";
    fout << "{\n";
    fout << "   if ( _state == s._state )\n";
    fout << "      return true;\n";
    fout << "   return false;\n";
    fout << "}\n";
    fout << "\n";
    fout << _parserName << "::ScannerStateType " << _parserName << "::ScannerStateEntry::getType()\n";
    fout << "{\n";
    fout << "   return _type;\n";
    fout << "}\n";
    fout << "int " << _parserName << "::ScannerStateEntry::getState()\n";
    fout << "{\n";
    fout << "   return _state;\n";
    fout << "}\n";
    fout << "\n";
    fout << "\n";
    fout << _parserName << "::ScannerTransEntry::ScannerTransEntry()\n";
    fout << "{\n";
    fout << "}\n";
    fout << "\n";
    fout << _parserName << "::ScannerTransEntry::ScannerTransEntry(int state, char c)\n";
    fout << "{\n";
    fout << "   _state = state;\n";
    fout << "   _c = c;\n";
    fout << "}\n";
    fout << "\n";
    fout << _parserName << "::ScannerTransEntry::ScannerTransEntry(int state, char c, int fstate)\n";
    fout << "{\n";
    fout << "   _state = state;\n";
    fout << "   _c = c;\n";
    fout << "   _fstate = fstate;\n";
    fout << "}\n";
    fout << "\n";
    fout << _parserName << "::ScannerTransEntry& " << _parserName << "::ScannerTransEntry::operator = (const " << _parserName << "::ScannerTransEntry& s)\n";
    fout << "{\n";
    fout << "   _state = s._state;\n";
    fout << "   _c = s._c;\n";
    fout << "   _fstate = s._fstate;\n";
    fout << "   return(*this);\n";
    fout << "}\n";
    fout << "\n";
    fout << "bool " << _parserName << "::ScannerTransEntry::operator == (const " << _parserName << "::ScannerTransEntry& s)\n";
    fout << "{\n";
    fout << "   if ( _state == s._state && _c == s._c)\n";
    fout << "      return true;\n";
    fout << "   return false;\n";
    fout << "}\n";
    fout << "\n";
    fout << "int "<< _parserName << "::ScannerTransEntry::getFState()\n";
    fout << "{\n";
    fout << "   return _fstate;\n";
    fout << "}\n";
    fout << "\n";


    fout << "\n";
    fout << _parserName << "::Scanner::Scanner()\n";
    fout << "{\n";
    fout << "}\n";
    fout << "\n";


    fout << "\n";
    fout << _parserName << "::Scanner::Scanner(Token t)\n";
    fout << "{\n";
    fout << "   _token = t;\n";
    fout << "}\n";
    fout << "\n";

    fout << _parserName << "::Token " << _parserName << "::Scanner::getToken()\n";
    fout << "{\n";
    fout << "   return _token;\n";
    fout << "}\n";
    fout << "\n";

    fout << "void " << _parserName << "::Scanner::addState(ScannerStateEntry s)\n";
    fout << "{\n";
    fout << "   _stateList.Insert(s);\n";
    fout << "}\n";
    fout << "\n";

    fout << "void " << _parserName << "::Scanner::addTransition(ScannerTransEntry t)\n";
    fout << "{\n";
    fout << "   _transList.Insert(t);\n";
    fout << "}\n";
    fout << "\n";

    fout << "bool " << _parserName << "::Scanner::checkPattern(const Chain& p)\n";
    fout << "{\n";

    fout << "   unsigned long state = 0;\n";
    fout << "   ScannerStateEntry* pS = _stateList.First();\n";

    fout << "   while (pS)\n";
    fout << "   {\n";
    fout << "      if (pS->getType() == START || pS->getType() == ANY)\n";
    fout << "      {\n";
    fout << "         state = pS->getState();\n";
    fout << "         pS = 0;\n";
    fout << "      }\n";
    fout << "      else\n";
    fout << "      {\n";
    fout << "         pS = _stateList.Next();\n"; 
    fout << "      }\n";
    fout << "   }\n";
    fout << "   unsigned long i = 0;\n";
    fout << "   ScannerTransEntry* pT = \n";
    fout << "   _transList.Find(ScannerTransEntry(state, p[i]));\n";
    fout << "   while (pT)\n";
    fout << "   {\n";
    fout << "      ScannerStateEntry* pS = _stateList.Find(ScannerStateEntry(pT->getFState())); \n";
    fout << "      if (pS)\n";
    fout << "      {\n";
    fout << "         if ( ( i == p.length()-2 ) && ( pS->getType() == FINAL || pS->getType() == ANY))\n";
    fout << "         {\n";
    fout << "            return true;\n";
    fout << "         }\n";
    fout << "         else\n";
    fout << "         {\n";
    fout << "            state = pT->getFState();\n";
    fout << "         }\n";
    fout << "      }\n";
    fout << "      else\n";
    fout << "      {\n";
    fout << "         return false;\n";
    fout << "      }\n";
    fout << "      i++;\n"; 
    fout << "      pT = _transList.Find(ScannerTransEntry(state, p[i]));\n";
    fout << "   }\n";
    fout << "   return false;\n";
    fout << "}\n";


    fout << _parserName << "::StackEntry::StackEntry()\n";
    fout << "{\n";
    fout << "   _symbol = NONE;\n";
    fout << "}\n";
    fout << "\n";
    fout << _parserName << "::StackEntry::StackEntry(Symbol symbol, int num, int state, Chain tval)\n";
    fout << "{\n";
    fout << "   _symbol=symbol;\n";
    fout << "   _num=num;\n";
    fout << "   _state=state;\n";
    fout << "   _tval=tval;\n";
    fout << "}\n";
    fout << "\n";
    fout << _parserName << "::StackEntry& " << _parserName << "::StackEntry::operator = (const " << _parserName << "::StackEntry& se)\n";
    fout << "{\n";
    fout << "   _symbol=se._symbol;\n";
    fout << "   _num=se._num;\n";
    fout << "   _state=se._state;\n";
    fout << "   _tval=se._tval;\n";
    fout << "   return(*this);\n";
    fout << "}\n";
    fout << "\n";
    fout << "int " << _parserName << "::StackEntry::getState()\n";
    fout << "{\n";
    fout << "   return _state;\n";
    fout << "}\n";
    fout << _parserName << "::Symbol " << _parserName << "::StackEntry::getSymbol()\n";
    fout << "{\n";
    fout << "   return _symbol;\n";
    fout << "}\n";
    fout << "Chain& " << _parserName << "::StackEntry::getTval()\n";
    fout << "{\n";
    fout << "   return _tval;\n";
    fout << "}\n";

    fout << "\n";
    fout << _parserName << "::ActionEntry::ActionEntry()\n";
    fout << "{\n";
    fout << "   _action=NULLACTION;\n";
    fout << "}\n";
    fout << "\n";
    fout << _parserName << "::ActionEntry::ActionEntry(Action action , int num)\n";
    fout << "{\n";
    fout << "   _action=action;\n";
    fout << "   _num=num;\n";
    fout << "}\n";
    fout << "\n";
    fout << "bool " << _parserName << "::ActionEntry::operator == (const " << _parserName << "::ActionEntry& ae)\n";
    fout << "{\n";
    fout << "   if (_action == ae._action && _num == ae._num)\n";
    fout << "      return true;\n";
    fout << "   return false;\n";
    fout << "}\n";
    fout << "\n";
    fout << _parserName << "::ActionEntry& " << _parserName << "::ActionEntry::operator = (const " << _parserName << "::ActionEntry& ae)\n";
    fout << "{\n";
    fout << "   _action=ae._action;\n";
    fout << "   _num=ae._num;\n";
    fout << "   return(*this);\n";
    fout << "}\n";
    fout << "\n";
    fout << _parserName << "::Action " << _parserName << "::ActionEntry::getAction()\n";
    fout << "{\n";
    fout << "   return _action;\n";
    fout << "}\n";
    fout << "\n";
    fout << "int " << _parserName << "::ActionEntry::getNum()\n";
    fout << "{\n";
    fout << "   return _num;\n";
    fout << "}\n";
    fout << "\n";
    fout << _parserName << "::ProdEntry::ProdEntry()\n";
    fout << "{\n";
    fout << "}\n";
    fout << "\n";
    fout << _parserName << "::ProdEntry::ProdEntry(int id)\n";
    fout << "{\n";
    fout << "   _id = id;\n";
    fout << "}\n";
    fout << "\n";
    fout << _parserName << "::ProdEntry::ProdEntry(int id, Production prod, int numSymbol)\n";
    fout << "{\n";
    fout << "   _id = id;\n";
    fout << "   _prod = prod;\n";
    fout << "   _numSymbol = numSymbol;\n";
    fout << "}\n";
    fout << "\n";
    fout << "bool " << _parserName << "::ProdEntry::operator == (const " << _parserName << "::ProdEntry& pe)\n";
    fout << "{\n";
    fout << "   if (_id == pe._id)\n";
    fout << "      return true;\n";
    fout << "   return false;\n";
    fout << "}\n";
    fout << "\n";
    fout << _parserName << "::ProdEntry& " << _parserName << "::ProdEntry::operator = (const " << _parserName << "::ProdEntry& pe)\n";
    fout << "{\n";
    fout << "   _id = pe._id;\n";
    fout << "   _prod = pe._prod;\n";
    fout << "   _numSymbol = pe._numSymbol;\n";
    fout << "   return(*this);\n";
    fout << "}\n";
    fout << "\n";
    fout << "\n";
    fout << "int " << _parserName << "::ProdEntry::getId()\n";
    fout << "{\n";
    fout << "   return _id;\n";
    fout << "}\n";
    fout << "\n";
    fout << _parserName << "::Production " << _parserName << "::ProdEntry::getProd()\n";
    fout << "{\n";
    fout << "   return _prod;\n";
    fout << "}\n";
    fout << "\n";
    fout << "int " << _parserName << "::ProdEntry::getNumSymbol()\n";
    fout << "{\n";
    fout << "   return _numSymbol;\n";
    fout << "}\n";


    fout << "void " << _parserName << "::loadScanner()\n";
    fout << "{\n";
    int sid=0;    
    for (int i = 1; i<=_terminalSet.Size(); i++)
    {
	Terminal *pTerm = _terminalSet.First();
	while (pTerm && pTerm->getNum() != i)
	{
	    pTerm = _terminalSet.Next();
	}
	if (pTerm)
	{

	    Worm w(pTerm->getRegExp(), pTerm->getName());
	 
	    w.makeOptimalDEA();
	    
	    Chain scanner = "s" + Chain(sid);
	    
	    fout << "   Scanner " << scanner << "(" << pTerm->getName() << ");\n";
	    
	    FSMState *pS = w.getStateTable().First();	
	    while (pS)
	    {   
		
		fout << "   " << scanner << ".addState(ScannerStateEntry("
		     << pS->Num() << " ,";
		
		switch (pS->Type())
		{
		case START:
		    fout << "START));\n";
		    break;
		case FINAL:
		    fout << "FINAL));\n";	
		    break;
		case ANY:
		    fout << "ANY));\n";	
		    break;
		case NONE:
		    fout << "INTERMEDIATE));\n";
		    break;
		}
		
		pS = w.getStateTable().Next();
	    }
	    
	    FSMTransition *pT = w.getTransitionTable().First();
	    
	    while (pT)
	    {
		
		fout << "   " << scanner << ".addTransition(ScannerTransEntry("
		     << pT->Source() << " ,'" << pT->Sign() << "' ," << pT->Target() << "));\n";
		
		pT = w.getTransitionTable().Next();
	    }
	    
	    
	    fout << "   _scannerList.Insert(" << scanner << ");\n";
	    
	    fout << "\n";	
	    
	    sid++;
	}
	
	
    }

    fout << "}\n";

    int mcount = 0;
    fout << "void " << _parserName << "::loadParser" + Chain(mcount) + "()\n";
    fout << "{\n";
    int lcount=0;

    ParseTableEntry *pPTE = _parseTable.First();
    while (pPTE)
    {
	Chain token = pPTE->getToken();
	if ( token == Chain("$") )
	    token = Chain("ENDTOKEN");

	switch (pPTE->getAction())
	{
	case ParseTableEntry::SHIFT:
	    fout << "   _actionMap["
		 << pPTE->getState() << "][" << token
		 << "] = ActionEntry(SHIFT, " << pPTE->getArg() << ");\n";
	    break;
	case ParseTableEntry::REDUCE:
	    fout << "   _actionMap["
		 << pPTE->getState() << "][" << token
		 << "] = ActionEntry(REDUCE, " << pPTE->getArg() << ");\n";
	    break;
	case ParseTableEntry::JUMP:
	    fout << "   _jumpMap["
		 << pPTE->getState() << "][" << token
		 << "] = " << pPTE->getArg() << ";\n";
	    break;
	case ParseTableEntry::ACCEPT:
	    fout << "   _actionMap["
		 << pPTE->getState() << "][" << token
		 << "] = ActionEntry(ACCEPT, " << pPTE->getArg() << ");\n";
	    break;
	}
	lcount++;
	pPTE = _parseTable.Next();

	if ( pPTE && lcount == LOADCOUNT )
	{
	    mcount++;
	    lcount=0;
	    fout << "}\n";
	    fout << "void " << _parserName << "::loadParser" + Chain(mcount) + "()\n";
	    fout << "{\n";
	}

    }

    fout << "}\n";

    mcount++;
    fout << "void " << _parserName << "::loadParser" + Chain(mcount) + "()\n";
    fout << "{\n";

    Production *pP = _productionSet.First();
    while (pP)
    {

	fout << "   _prodInfo.Insert(ProdEntry("
	     << pP->getId() << ", " << pP->getName() << ", "
	     << pP->getMaxPos() << "));\n";	

	pP = _productionSet.Next();
    }

    fout << "}\n";

    fout << "\n";
    fout << _parserName << "::" << _parserName << "()\n";
    fout << "{\n";

    if ( _dynamicTable )
    {
	fout << "   _actionMap = (ActionEntry**) malloc ( " << _numStates + 1 << " * sizeof(ActionEntry*));\n";
	fout << "   _jumpMap = (int**) malloc( " << _numStates + 1 << " * sizeof (int*));\n";
	fout << "   for ( int c = 0 ; c < " <<  _numStates + 1 << " ; c++ )\n";
	fout << "   {\n";
	fout << "      _actionMap[c] = (ActionEntry*) malloc( " << _terminalSet.Size() +1 << " * sizeof(ActionEntry));\n";
	fout << "      _jumpMap[c] = (int*) malloc( " << _numProd + 1 << " * sizeof (int));\n";
	fout << "   }\n";
    }

    fout << "   loadScanner();\n";

    fout << "   for ( int i=0; i < " << _numStates + 1<< "; i++)\n";
    fout << "   {\n";
    fout << "      int j;\n";
    fout << "      for ( j = 0; j < " << _terminalSet.Size() + 1  <<  "; j++)\n";
    fout << "         _actionMap[i][j] = ActionEntry();\n";
    fout << "      for ( j = 0; j < " << _numProd + 1  <<  "; j++)\n";
    fout << "         _jumpMap[i][j] = -1;\n";
    fout << "   }\n";

    for ( int i=0; i<=mcount; i++)
    {
	fout << "   loadParser" + Chain(i) + "();\n";	
    }    
    fout << "   _isReserved = false;\n";

    fout << "}\n";
    fout << "\n";
    fout << _parserName << "::~" << _parserName << "()\n";
    fout << "{\n";

    if ( _dynamicTable )
    {
	fout << "   for ( int c = 0 ; c < " << _numStates + 1 << " ; c++ )\n";
	fout << "   {\n";
	fout << "      free (_actionMap[c]);\n";
	fout << "      free (_jumpMap[c]);\n";
	fout << "   }\n";
	fout << "   free (_actionMap);\n";
	fout << "   free (_jumpMap);\n";
    }

    fout << "}\n";
    fout << "\n";
    fout << "void " << _parserName << "::parse()\n";
    fout << "{\n";
    fout << "   StackT<StackEntry> s;\n";
    fout << "   StackEntry se(NONE, 0, 0);\n";
    fout << "   s.Push(se);\n";
    fout << "   _isReserved = false;\n";
    fout << "   if ( shiftToken() == false )\n";
    fout << "   {\n";
    fout << "      Chain msg = \"Parse error at token <\" + Chain(_tokenVal) + \">.\";\n";
    fout << "      throw Exception(EXLOC, msg);\n";
    fout << "   }\n";
    fout << "   while ( true )\n";
    fout << "   {\n";
    fout << "      Action a;\n";
    fout << "      int num;\n";
    fout << "      if ( getAction(s.getTop()->getState(), _token, a, num) )\n";
    fout << "      {\n";
    fout << "         switch (a)\n";
    fout << "         {\n";
    fout << "            case SHIFT:\n";
    fout << "            {\n";
    fout << "               StackEntry se(TOKEN, _token, num, _tokenVal);\n";
    fout << "               s.Push(se);\n";
    fout << "               if ( shiftToken() == false )\n";
    fout << "               {\n";
    fout << "                  Chain msg = \"Parse error at token <\" + Chain(_tokenVal) + \">.\";\n";
    fout << "                  throw Exception(EXLOC, msg);\n";
    fout << "               }\n";
    fout << "               break;\n";
    fout << "            }\n";
    fout << "            case REDUCE:\n";
    fout << "            {\n";
    fout << "               _tokenList.Empty();\n";
    fout << "               ProdEntry* pPE = _prodInfo.Find(num);\n";
    fout << "               for (int i=0; i<pPE->getNumSymbol();i++)\n";
    fout << "               {\n";
    fout << "                  StackEntry se;\n";
    fout << "                  s.Pop(se);\n";
    fout << "                  if (se.getSymbol() == TOKEN)\n";
    fout << "                     _tokenList.Insert(se.getTval());\n";
    fout << "               }\n";
    fout << "               int fstate;\n";
    fout << "               if (getJump(s.getTop()->getState(), pPE->getProd(), fstate))\n";
    fout << "               {\n";
    fout << "                  StackEntry se(PROD, pPE->getId(), fstate, _tokenVal);\n";
    fout << "                  s.Push(se);\n";
    fout << "               }\n";
    fout << "               else\n";
    fout << "               {\n";
    fout << "                  Chain msg = \"Cannot reduce after token <\" + Chain(_tokenVal) + \">.\";\n";
    fout << "                  throw Exception(EXLOC, msg);\n";
    fout << "               }\n";
    fout << "               switch (num)\n";
    fout << "               {\n";

    pP = _productionSet.First();
    while (pP)
    {
	if ( pP->getId() != 0 && pP->getAction() )
	{
	    fout << "                  case " << pP->getId() << ":\n" 
		 << "                     " << pP->getAction() << "();\n"
		 << "                     break;\n";	
	}
	    pP = _productionSet.Next();
    }

    fout << "               }\n";
    fout << "               break;\n";
    fout << "            }\n";
    fout << "            case ACCEPT:\n";
    fout << "            {\n";

    pP = _productionSet.First();
    while (pP)
    {
	if ( pP->getId() == 0 && pP->getAction() )
	{ 
	    fout << "               " << pP->getAction() << "();\n";
	}
	pP = _productionSet.Next();
    }


    fout << "               return;\n";
    fout << "            }\n";
    fout << "            default:\n";
    fout << "               break;\n";
    fout << "         }\n";
    fout << "      }\n";
    fout << "      else\n";
    fout << "      {\n";
    fout << "         Chain msg = \"Parse error at token <\" + Chain(_tokenVal) + \">.\";\n";
    fout << "         throw Exception(EXLOC, msg);\n";
    fout << "      }\n";
    fout << "   }\n";
    fout << "}\n";
    fout << "\n";
    fout << "\n";
    fout << "bool " << _parserName << "::getAction(int state, " << _parserName << "::Token token, " <<  _parserName << "::Action& a, int& num)\n";
    fout << "{\n";
    fout << "    if (_actionMap[state][token].getAction() != NULLACTION )\n";
    fout << "    {\n";
    fout << "	a = _actionMap[state][token].getAction();\n";
    fout << "	num = _actionMap[state][token].getNum();\n";
    fout << "	return true;\n";
    fout << "    }\n";
    fout << "    else\n";
    fout << "    {\n";
    fout << "	return false;\n";
    fout << "    }\n";
    fout << "}\n";
    fout << "\n";
    fout << "bool " << _parserName << "::getJump(int state, Production prod, int& fstate)\n";
    fout << "{\n";
    fout << "    if ( _jumpMap[state][prod] > 0 )\n";
    fout << "    {\n";
    fout << "       fstate = _jumpMap[state][prod];\n";
    fout << "	    return true;\n";
    fout << "    }\n";
    fout << "    else\n";
    fout << "    {\n";
    fout << "	    return false;\n";
    fout << "    }\n";
    fout << "}\n";
    fout << "\n";

    fout << "bool " << _parserName << "::isSepSign(char c)\n";
    fout << "{\n";
    
    Chain *pSep = _sepsignList.First();
    
    while (pSep)
    {
	
	fout << "   if (c == " << *pSep << ") return true;\n";
	pSep = _sepsignList.Next();
    }
    fout << "   return false;\n";
    fout << "}\n";

    fout << "bool " << _parserName << "::isSepIgnore(char c)\n";
    fout << "{\n";
    
    pSep = _sepignoreList.First();
    
    while (pSep)
    {
	
	fout << "   if (c == " << *pSep << ") return true;\n";
	pSep = _sepignoreList.Next();
    }
    fout << "   return false;\n";
    fout << "}\n";



    fout << "bool " << _parserName << "::shiftToken()\n";
    fout << "{\n";
    
    
    fout << "   unsigned long i=0;\n";
    fout << "   char c;\n";
    fout << "   while ( isSepIgnore(c = nextChar()) && ! _isReserved );\n";

    fout << "   if ( _isReserved )\n";
    fout << "   {\n";
    fout << "      _token = _reservedToken;\n";
    fout << "      _isReserved = false;\n";
    fout << "      return true;\n";
    fout << "   }\n";

    fout << "   if (c == 0)\n";
    fout << "   {\n";
    fout << "      _token = ENDTOKEN;\n";
    fout << "      return true;\n";
    fout << "   }\n";
    
    fout << "   if (isSepSign(c))\n";
    fout << "   {\n";
    fout << "      _tokenVal[i]=c;\n";
    fout << "       i++;\n";
    fout << "   }\n";
    fout << "   else\n";
    fout << "   {\n";
    fout << "      while (c && ! isSepIgnore(c) && ! ( isSepSign(c) && i>0 ) )\n";
    fout << "      {\n";
    fout << "         _tokenVal[i]=c;\n";
    fout << "         i++;\n";
    fout << "         c = nextChar();\n";
    fout << "      }\n"; 
    fout << "      if (isSepSign(c)) backChar();\n";
    fout << "   }\n";
    fout << "   _tokenVal[i]=0;\n";
    fout << "   Scanner* pS = _scannerList.First();\n";
    fout << "   while (pS)\n";
    fout << "   {\n";
    fout << "      if (pS->checkPattern(_tokenVal))\n";
    fout << "      {\n";
    fout << "         _token = pS->getToken();\n";
    fout << "         return true;\n";
    fout << "      }\n";
    fout << "      pS=_scannerList.Next();\n";
    fout << "   }\n";
    fout << "   return false;\n";
    fout << "}\n";


    fout << "void " << _parserName << "::setReserved(Token t)\n";
    fout << "{\n";
    fout << "   _reservedToken = t;\n";
    fout << "   _isReserved = true;\n";
    fout << "}\n";

    
    fout.close();
    return;
}


    
