/*
 * Copyright (C) 2003 The FFmpeg project
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef AVCODEC_VP3DATA_H
#define AVCODEC_VP3DATA_H

#include <stdint.h>
#include <stdlib.h>

/* these coefficients dequantize intraframe Y plane coefficients
 * (note: same as JPEG) */
static const uint8_t vp31_intra_y_dequant[64] = {
    16, 11, 10, 16,  24,  40,  51,  61,
    12, 12, 14, 19,  26,  58,  60,  55,
    14, 13, 16, 24,  40,  57,  69,  56,
    14, 17, 22, 29,  51,  87,  80,  62,
    18, 22, 37, 58,  68, 109, 103,  77,
    24, 35, 55, 64,  81, 104, 113,  92,
    49, 64, 78, 87, 103, 121, 120, 101,
    72, 92, 95, 98, 112, 100, 103,  99
};

/* these coefficients dequantize intraframe C plane coefficients
 * (note: same as JPEG) */
static const uint8_t vp31_intra_c_dequant[64] = {
    17, 18, 24, 47, 99, 99, 99, 99,
    18, 21, 26, 66, 99, 99, 99, 99,
    24, 26, 56, 99, 99, 99, 99, 99,
    47, 66, 99, 99, 99, 99, 99, 99,
    99, 99, 99, 99, 99, 99, 99, 99,
    99, 99, 99, 99, 99, 99, 99, 99,
    99, 99, 99, 99, 99, 99, 99, 99,
    99, 99, 99, 99, 99, 99, 99, 99
};

/* these coefficients dequantize interframe coefficients (all planes) */
static const uint8_t vp31_inter_dequant[64] = {
    16, 16, 16, 20, 24, 28,  32,  40,
    16, 16, 20, 24, 28, 32,  40,  48,
    16, 20, 24, 28, 32, 40,  48,  64,
    20, 24, 28, 32, 40, 48,  64,  64,
    24, 28, 32, 40, 48, 64,  64,  64,
    28, 32, 40, 48, 64, 64,  64,  96,
    32, 40, 48, 64, 64, 64,  96, 128,
    40, 48, 64, 64, 64, 96, 128, 128
};

static const uint8_t vp31_dc_scale_factor[64] = {
    220, 200, 190, 180, 170, 170, 160, 160,
    150, 150, 140, 140, 130, 130, 120, 120,
    110, 110, 100, 100,  90,  90,  90,  80,
     80,  80,  70,  70,  70,  60,  60,  60,
     60,  50,  50,  50,  50,  40,  40,  40,
     40,  40,  30,  30,  30,  30,  30,  30,
     30,  20,  20,  20,  20,  20,  20,  20,
     20,  10,  10,  10,  10,  10,  10,  10
};

static const uint16_t vp31_ac_scale_factor[64] = {
    500, 450, 400, 370, 340, 310, 285, 265,
    245, 225, 210, 195, 185, 180, 170, 160,
    150, 145, 135, 130, 125, 115, 110, 107,
    100,  96,  93,  89,  85,  82,  75,  74,
     70,  68,  64,  60,  57,  56,  52,  50,
     49,  45,  44,  43,  40,  38,  37,  35,
     33,  32,  30,  29,  28,  25,  24,  22,
     21,  19,  18,  17,  15,  13,  12,  10
};

static const uint8_t vp31_filter_limit_values[64] = {
    30, 25, 20, 20, 15, 15, 14, 14,
    13, 13, 12, 12, 11, 11, 10, 10,
     9,  9,  8,  8,  7,  7,  7,  7,
     6,  6,  6,  6,  5,  5,  5,  5,
     4,  4,  4,  4,  3,  3,  3,  3,
     2,  2,  2,  2,  2,  2,  2,  2,
     0,  0,  0,  0,  0,  0,  0,  0,
     0,  0,  0,  0,  0,  0,  0,  0
};

static const uint8_t superblock_run_length_vlc_lens[34] = {
     1,  3,  3,  4,  4,  6,  6,  6,  6,  8,  8,  8,  8,  8,  8,  8,  8, 10, 10,
    10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
     6, /* This last value is a special case for reading 12 more
         * bits from the stream and adding the value 34. */
};

static const uint8_t fragment_run_length_vlc_len[30] = {
    2,  2,  3,  3,  4,  4,  6,  6,  6,  6,  7,  7,  7,  7,  9,  9,  9,  9,  9,
    9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,
};

static const uint8_t mode_code_vlc_len[8] = {
    1,  2,  3,  4,  5,  6,  7,  7,
};

static const uint8_t motion_vector_vlc_table[63][2] = {
    { 31,  3 }, { 32,  3 }, { 30,  3 }, { 33,  4 }, { 29,  4 }, { 34,  4 },
    { 28,  4 }, { 35,  6 }, { 27,  6 }, { 36,  6 }, { 26,  6 }, { 37,  6 },
    { 25,  6 }, { 38,  6 }, { 24,  6 }, { 39,  7 }, { 23,  7 }, { 40,  7 },
    { 22,  7 }, { 41,  7 }, { 21,  7 }, { 42,  7 }, { 20,  7 }, { 43,  7 },
    { 19,  7 }, { 44,  7 }, { 18,  7 }, { 45,  7 }, { 17,  7 }, { 46,  7 },
    { 16,  7 }, { 47,  8 }, { 15,  8 }, { 48,  8 }, { 14,  8 }, { 49,  8 },
    { 13,  8 }, { 50,  8 }, { 12,  8 }, { 51,  8 }, { 11,  8 }, { 52,  8 },
    { 10,  8 }, { 53,  8 }, {  9,  8 }, { 54,  8 }, {  8,  8 }, { 55,  8 },
    {  7,  8 }, { 56,  8 }, {  6,  8 }, { 57,  8 }, {  5,  8 }, { 58,  8 },
    {  4,  8 }, { 59,  8 }, {  3,  8 }, { 60,  8 }, {  2,  8 }, { 61,  8 },
    {  1,  8 }, { 62,  8 }, {  0,  8 },
};

static const int8_t fixed_motion_vector_table[64] = {
     0,   0,  1,  -1,  2,  -2,  3,  -3,
     4,  -4,  5,  -5,  6,  -6,  7,  -7,
     8,  -8,  9,  -9, 10, -10, 11, -11,
    12, -12, 13, -13, 14, -14, 15, -15,
    16, -16, 17, -17, 18, -18, 19, -19,
    20, -20, 21, -21, 22, -22, 23, -23,
    24, -24, 25, -25, 26, -26, 27, -27,
    28, -28, 29, -29, 30, -30, 31, -31
};

/* only tokens 0..6 indicate eob runs */
static const struct {
    uint8_t base, bits;
} eob_run_table[7] = {
    {1, 0}, {2, 0}, {3, 0}, {4, 2}, {8, 3}, {16, 4}, {0, 12}
};

static const uint8_t zero_run_base[32] = {
    0,  0, 0, 0, 0, 0, 0,   /* 0..6 are never used */
    0,  0,                  /* 7..8 */
    0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* 9..22 */
    1,  2, 3, 4, 5,         /* 23..27 */
    6, 10, 1, 2             /* 28..31 */
};
static const uint8_t zero_run_get_bits[32] = {
    0, 0, 0, 0, 0, 0, 0,    /* 0..6 are never used */
    3, 6,                   /* 7..8 */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* 9..22 */
    0, 0, 0, 0, 0,          /* 23..27 */
    2, 3, 0, 1              /* 28..31 */
};

static const uint8_t coeff_get_bits[32] = {
    0, 0, 0, 0, 0, 0, 0,    /* 0..6 are never used */
    0, 0, 0, 0, 0, 0,       /* 7..12 use constant coeffs */
    1, 1, 1, 1,             /* 13..16 are constants but still need sign bit */
    2, 3, 4, 5, 6,10,       /* 17..22, for reading large coeffs */
    1, 1, 1, 1, 1, 1, 1,    /* 23..29 are constants but still need sign bit */
    2, 2                    /* 30..31 */
};

static const int16_t coeff_table_token_7_8[1] = { 0 };
static const int16_t coeff_table_token_9[1] = { 1 };
static const int16_t coeff_table_token_10[1] = { -1 };
static const int16_t coeff_table_token_11[1] = { 2 };
static const int16_t coeff_table_token_12[1] = { -2 };

static const int16_t coeff_table_token_13[2] = { 3, -3 };
static const int16_t coeff_table_token_14[2] = { 4, -4 };
static const int16_t coeff_table_token_15[2] = { 5, -5 };
static const int16_t coeff_table_token_16[2] = { 6, -6 };

static const int16_t coeff_table_token_23_24_25_26_27_28_29[2] = { 1, -1 };
static const int16_t coeff_table_token_30_31[4] = { 2, 3, -2, -3 };

static const int16_t coeff_table_token_17[4] = {
    7, 8, -7, -8
};

static const int16_t coeff_table_token_18[8] = {
    9, 10, 11, 12, -9, -10, -11, -12
};

static const int16_t coeff_table_token_19[16] = {
    13, 14, 15, 16, 17, 18, 19, 20, -13, -14, -15, -16, -17, -18, -19, -20
};

static const int16_t coeff_table_token_20[32] = {
     21,  22,  23,  24,  25,  26,  27,  28,
     29,  30,  31,  32,  33,  34,  35,  36,
    -21, -22, -23, -24, -25, -26, -27, -28,
    -29, -30, -31, -32, -33, -34, -35, -36
};

static const int16_t coeff_table_token_21[64] = {
     37,  38,  39,  40,  41,  42,  43,  44,
     45,  46,  47,  48,  49,  50,  51,  52,
     53,  54,  55,  56,  57,  58,  59,  60,
     61,  62,  63,  64,  65,  66,  67,  68,
    -37, -38, -39, -40, -41, -42, -43, -44,
    -45, -46, -47, -48, -49, -50, -51, -52,
    -53, -54, -55, -56, -57, -58, -59, -60,
    -61, -62, -63, -64, -65, -66, -67, -68
};

static const int16_t coeff_table_token_22[1024] = {
      69,   70,   71,   72,   73,   74,   75,   76,
      77,   78,   79,   80,   81,   82,   83,   84,
      85,   86,   87,   88,   89,   90,   91,   92,
      93,   94,   95,   96,   97,   98,   99,  100,
     101,  102,  103,  104,  105,  106,  107,  108,
     109,  110,  111,  112,  113,  114,  115,  116,
     117,  118,  119,  120,  121,  122,  123,  124,
     125,  126,  127,  128,  129,  130,  131,  132,
     133,  134,  135,  136,  137,  138,  139,  140,
     141,  142,  143,  144,  145,  146,  147,  148,
     149,  150,  151,  152,  153,  154,  155,  156,
     157,  158,  159,  160,  161,  162,  163,  164,
     165,  166,  167,  168,  169,  170,  171,  172,
     173,  174,  175,  176,  177,  178,  179,  180,
     181,  182,  183,  184,  185,  186,  187,  188,
     189,  190,  191,  192,  193,  194,  195,  196,
     197,  198,  199,  200,  201,  202,  203,  204,
     205,  206,  207,  208,  209,  210,  211,  212,
     213,  214,  215,  216,  217,  218,  219,  220,
     221,  222,  223,  224,  225,  226,  227,  228,
     229,  230,  231,  232,  233,  234,  235,  236,
     237,  238,  239,  240,  241,  242,  243,  244,
     245,  246,  247,  248,  249,  250,  251,  252,
     253,  254,  255,  256,  257,  258,  259,  260,
     261,  262,  263,  264,  265,  266,  267,  268,
     269,  270,  271,  272,  273,  274,  275,  276,
     277,  278,  279,  280,  281,  282,  283,  284,
     285,  286,  287,  288,  289,  290,  291,  292,
     293,  294,  295,  296,  297,  298,  299,  300,
     301,  302,  303,  304,  305,  306,  307,  308,
     309,  310,  311,  312,  313,  314,  315,  316,
     317,  318,  319,  320,  321,  322,  323,  324,
     325,  326,  327,  328,  329,  330,  331,  332,
     333,  334,  335,  336,  337,  338,  339,  340,
     341,  342,  343,  344,  345,  346,  347,  348,
     349,  350,  351,  352,  353,  354,  355,  356,
     357,  358,  359,  360,  361,  362,  363,  364,
     365,  366,  367,  368,  369,  370,  371,  372,
     373,  374,  375,  376,  377,  378,  379,  380,
     381,  382,  383,  384,  385,  386,  387,  388,
     389,  390,  391,  392,  393,  394,  395,  396,
     397,  398,  399,  400,  401,  402,  403,  404,
     405,  406,  407,  408,  409,  410,  411,  412,
     413,  414,  415,  416,  417,  418,  419,  420,
     421,  422,  423,  424,  425,  426,  427,  428,
     429,  430,  431,  432,  433,  434,  435,  436,
     437,  438,  439,  440,  441,  442,  443,  444,
     445,  446,  447,  448,  449,  450,  451,  452,
     453,  454,  455,  456,  457,  458,  459,  460,
     461,  462,  463,  464,  465,  466,  467,  468,
     469,  470,  471,  472,  473,  474,  475,  476,
     477,  478,  479,  480,  481,  482,  483,  484,
     485,  486,  487,  488,  489,  490,  491,  492,
     493,  494,  495,  496,  497,  498,  499,  500,
     501,  502,  503,  504,  505,  506,  507,  508,
     509,  510,  511,  512,  513,  514,  515,  516,
     517,  518,  519,  520,  521,  522,  523,  524,
     525,  526,  527,  528,  529,  530,  531,  532,
     533,  534,  535,  536,  537,  538,  539,  540,
     541,  542,  543,  544,  545,  546,  547,  548,
     549,  550,  551,  552,  553,  554,  555,  556,
     557,  558,  559,  560,  561,  562,  563,  564,
     565,  566,  567,  568,  569,  570,  571,  572,
     573,  574,  575,  576,  577,  578,  579,  580,
     -69,  -70,  -71,  -72,  -73,  -74,  -75,  -76,
     -77,  -78,  -79,  -80,  -81,  -82,  -83,  -84,
     -85,  -86,  -87,  -88,  -89,  -90,  -91,  -92,
     -93,  -94,  -95,  -96,  -97,  -98,  -99, -100,
    -101, -102, -103, -104, -105, -106, -107, -108,
    -109, -110, -111, -112, -113, -114, -115, -116,
    -117, -118, -119, -120, -121, -122, -123, -124,
    -125, -126, -127, -128, -129, -130, -131, -132,
    -133, -134, -135, -136, -137, -138, -139, -140,
    -141, -142, -143, -144, -145, -146, -147, -148,
    -149, -150, -151, -152, -153, -154, -155, -156,
    -157, -158, -159, -160, -161, -162, -163, -164,
    -165, -166, -167, -168, -169, -170, -171, -172,
    -173, -174, -175, -176, -177, -178, -179, -180,
    -181, -182, -183, -184, -185, -186, -187, -188,
    -189, -190, -191, -192, -193, -194, -195, -196,
    -197, -198, -199, -200, -201, -202, -203, -204,
    -205, -206, -207, -208, -209, -210, -211, -212,
    -213, -214, -215, -216, -217, -218, -219, -220,
    -221, -222, -223, -224, -225, -226, -227, -228,
    -229, -230, -231, -232, -233, -234, -235, -236,
    -237, -238, -239, -240, -241, -242, -243, -244,
    -245, -246, -247, -248, -249, -250, -251, -252,
    -253, -254, -255, -256, -257, -258, -259, -260,
    -261, -262, -263, -264, -265, -266, -267, -268,
    -269, -270, -271, -272, -273, -274, -275, -276,
    -277, -278, -279, -280, -281, -282, -283, -284,
    -285, -286, -287, -288, -289, -290, -291, -292,
    -293, -294, -295, -296, -297, -298, -299, -300,
    -301, -302, -303, -304, -305, -306, -307, -308,
    -309, -310, -311, -312, -313, -314, -315, -316,
    -317, -318, -319, -320, -321, -322, -323, -324,
    -325, -326, -327, -328, -329, -330, -331, -332,
    -333, -334, -335, -336, -337, -338, -339, -340,
    -341, -342, -343, -344, -345, -346, -347, -348,
    -349, -350, -351, -352, -353, -354, -355, -356,
    -357, -358, -359, -360, -361, -362, -363, -364,
    -365, -366, -367, -368, -369, -370, -371, -372,
    -373, -374, -375, -376, -377, -378, -379, -380,
    -381, -382, -383, -384, -385, -386, -387, -388,
    -389, -390, -391, -392, -393, -394, -395, -396,
    -397, -398, -399, -400, -401, -402, -403, -404,
    -405, -406, -407, -408, -409, -410, -411, -412,
    -413, -414, -415, -416, -417, -418, -419, -420,
    -421, -422, -423, -424, -425, -426, -427, -428,
    -429, -430, -431, -432, -433, -434, -435, -436,
    -437, -438, -439, -440, -441, -442, -443, -444,
    -445, -446, -447, -448, -449, -450, -451, -452,
    -453, -454, -455, -456, -457, -458, -459, -460,
    -461, -462, -463, -464, -465, -466, -467, -468,
    -469, -470, -471, -472, -473, -474, -475, -476,
    -477, -478, -479, -480, -481, -482, -483, -484,
    -485, -486, -487, -488, -489, -490, -491, -492,
    -493, -494, -495, -496, -497, -498, -499, -500,
    -501, -502, -503, -504, -505, -506, -507, -508,
    -509, -510, -511, -512, -513, -514, -515, -516,
    -517, -518, -519, -520, -521, -522, -523, -524,
    -525, -526, -527, -528, -529, -530, -531, -532,
    -533, -534, -535, -536, -537, -538, -539, -540,
    -541, -542, -543, -544, -545, -546, -547, -548,
    -549, -550, -551, -552, -553, -554, -555, -556,
    -557, -558, -559, -560, -561, -562, -563, -564,
    -565, -566, -567, -568, -569, -570, -571, -572,
    -573, -574, -575, -576, -577, -578, -579, -580
};

static const int16_t *const coeff_tables[32] = {
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    coeff_table_token_7_8,

    coeff_table_token_7_8,
    coeff_table_token_9,
    coeff_table_token_10,
    coeff_table_token_11,
    coeff_table_token_12,
    coeff_table_token_13,
    coeff_table_token_14,
    coeff_table_token_15,

    coeff_table_token_16,
    coeff_table_token_17,
    coeff_table_token_18,
    coeff_table_token_19,
    coeff_table_token_20,
    coeff_table_token_21,
    coeff_table_token_22,
    coeff_table_token_23_24_25_26_27_28_29,

    coeff_table_token_23_24_25_26_27_28_29,
    coeff_table_token_23_24_25_26_27_28_29,
    coeff_table_token_23_24_25_26_27_28_29,
    coeff_table_token_23_24_25_26_27_28_29,
    coeff_table_token_23_24_25_26_27_28_29,
    coeff_table_token_23_24_25_26_27_28_29,
    coeff_table_token_30_31,
    coeff_table_token_30_31
};

static const uint8_t vp3_bias[5 * 16][32][2] = {
    { /* DC bias table 0 */
        { 20,  3 }, { 19,  3 }, {  7,  5 }, { 30,  6 }, {  1,  7 }, {  3,  8 },
        { 28,  9 }, { 26, 12 }, {  8, 12 }, {  6, 11 }, { 29, 10 }, { 14,  4 },
        { 12,  4 }, { 17,  4 }, { 13,  4 }, { 21,  4 }, { 24,  6 }, { 23,  6 },
        { 16,  5 }, { 31,  7 }, { 25,  8 }, {  2,  9 }, {  4, 10 }, {  5, 11 },
        { 27, 11 }, {  0,  6 }, { 22,  5 }, { 18,  4 }, { 15,  5 }, { 11,  5 },
        { 10,  4 }, {  9,  4 },
    },
    { /* DC bias table 1 */
        { 20,  3 }, { 19,  3 }, { 11,  4 }, { 12,  4 }, { 14,  4 }, { 17,  4 },
        {  0,  5 }, {  7,  6 }, {  3,  8 }, { 29, 13 }, {  8, 13 }, { 26, 12 },
        {  6, 11 }, {  5, 11 }, { 28, 11 }, { 25,  9 }, {  1,  7 }, { 13,  4 },
        { 21,  4 }, { 16,  5 }, { 22,  5 }, { 18,  4 }, { 10,  4 }, {  9,  4 },
        { 15,  5 }, { 30,  7 }, { 23,  7 }, { 24,  7 }, { 31,  8 }, {  4, 10 },
        { 27, 10 }, {  2,  9 },
    },
    { /* DC bias table 2 */
        { 20,  3 }, { 19,  3 }, { 11,  4 }, { 12,  4 }, { 14,  4 }, { 17,  4 },
        {  1,  6 }, { 30,  7 }, {  2,  8 }, {  3,  8 }, { 16,  5 }, { 21,  4 },
        { 13,  4 }, {  0,  5 }, { 22,  5 }, { 18,  4 }, { 15,  5 }, { 31,  8 },
        { 25, 10 }, { 27, 11 }, {  6, 11 }, {  5, 11 }, { 26, 12 }, { 29, 14 },
        {  8, 14 }, { 28, 13 }, {  4, 10 }, { 24,  7 }, {  7,  7 }, { 23,  7 },
        { 10,  4 }, {  9,  4 },
    },
    { /* DC bias table 3 */
        {  0,  4 }, { 17,  4 }, { 14,  4 }, { 21,  5 }, {  7,  6 }, { 23,  6 },
        { 10,  3 }, {  9,  3 }, { 11,  4 }, { 12,  4 }, { 20,  4 }, {  3,  7 },
        { 25,  9 }, {  6, 10 }, {  5, 10 }, { 29, 13 }, {  8, 13 }, { 28, 12 },
        { 26, 11 }, { 27, 10 }, {  4,  9 }, {  1,  6 }, { 16,  5 }, { 18,  4 },
        { 13,  4 }, { 19,  4 }, { 22,  7 }, { 30,  7 }, { 24,  7 }, { 31,  8 },
        {  2,  8 }, { 15,  5 },
    },
    { /* DC bias table 4 */
        { 15,  4 }, {  5,  9 }, { 25,  9 }, { 22,  8 }, { 31,  7 }, { 24,  6 },
        {  7,  6 }, { 27,  9 }, {  6, 10 }, { 29, 13 }, {  8, 13 }, { 28, 12 },
        { 26, 11 }, {  4,  8 }, {  2,  7 }, { 17,  4 }, {  0,  4 }, { 14,  4 },
        { 11,  4 }, { 12,  4 }, { 19,  4 }, {  9,  3 }, { 10,  3 }, { 21,  6 },
        { 23,  6 }, { 16,  5 }, { 18,  4 }, { 13,  4 }, {  1,  6 }, {  3,  7 },
        { 30,  7 }, { 20,  5 },
    },
    { /* DC bias table 5 */
        { 15,  4 }, {  1,  5 }, {  7,  6 }, {  3,  6 }, { 17,  4 }, { 19,  4 },
        { 14,  4 }, { 18,  4 }, { 20,  5 }, { 27,  9 }, {  5,  9 }, { 29, 13 },
        {  8, 13 }, { 28, 12 }, { 26, 11 }, { 25, 10 }, {  6, 10 }, { 22, 10 },
        { 21,  7 }, { 23,  6 }, { 12,  4 }, { 11,  4 }, {  0,  4 }, {  9,  3 },
        { 10,  3 }, { 16,  5 }, {  2,  7 }, { 30,  7 }, {  4,  8 }, { 31,  8 },
        { 24,  7 }, { 13,  4 },
    },
    { /* DC bias table 6 */
        { 13,  3 }, { 17,  4 }, { 18,  4 }, { 30,  6 }, { 24,  6 }, {  2,  6 },
        { 27,  9 }, {  6,  9 }, { 21,  8 }, { 31,  7 }, { 14,  4 }, {  1,  5 },
        { 20,  6 }, {  3,  6 }, { 16,  5 }, { 19,  5 }, { 12,  4 }, { 11,  4 },
        {  0,  4 }, { 23,  6 }, {  7,  7 }, {  5,  9 }, { 25, 10 }, {  8, 13 },
        { 29, 14 }, { 22, 14 }, { 28, 12 }, { 26, 11 }, {  4,  8 }, { 15,  5 },
        { 10,  3 }, {  9,  3 },
    },
    { /* DC bias table 7 */
        { 10,  2 }, {  3,  5 }, { 19,  5 }, { 24,  6 }, {  7,  7 }, {  5,  8 },
        { 21,  9 }, {  6,  9 }, { 16,  5 }, { 14,  4 }, { 23,  6 }, {  2,  6 },
        {  1,  5 }, { 11,  4 }, { 12,  4 }, { 20,  7 }, {  4,  7 }, { 25, 10 },
        { 28, 12 }, {  8, 13 }, { 29, 14 }, { 22, 14 }, { 26, 11 }, { 27,  9 },
        { 31,  8 }, { 30,  7 }, { 18,  5 }, { 17,  5 }, { 15,  5 }, { 13,  4 },
        {  0,  4 }, {  9,  3 },
    },
    { /* DC bias table 8 */
        { 29,  5 }, {  7,  5 }, {  0,  4 }, { 13,  4 }, { 26,  6 }, { 19,  6 },
        { 14,  5 }, { 24,  4 }, { 12,  4 }, { 11,  4 }, { 17,  6 }, {  1,  6 },
        { 28,  5 }, { 18,  6 }, {  8,  6 }, { 25,  5 }, { 20,  7 }, { 21,  8 },
        {  6, 11 }, {  5, 11 }, {  4, 10 }, { 22,  9 }, { 15,  6 }, { 31,  5 },
        { 10,  3 }, {  9,  3 }, { 23,  4 }, { 27,  6 }, {  3,  8 }, {  2,  8 },
        { 16,  7 }, { 30,  5 },
    },
    { /* DC bias table 9 */
        { 30,  4 }, {  7,  5 }, {  6, 10 }, {  5, 10 }, {  4,  9 }, { 22,  8 },
        {  3,  7 }, { 16,  6 }, { 13,  4 }, { 24,  4 }, { 19,  6 }, { 26,  6 },
        { 14,  5 }, {  0,  4 }, { 12,  4 }, { 11,  4 }, {  2,  7 }, { 20,  7 },
        { 17,  6 }, { 25,  5 }, { 18,  6 }, { 15,  6 }, { 31,  5 }, { 10,  3 },
        {  9,  3 }, { 23,  4 }, {  1,  6 }, { 21,  8 }, {  8,  8 }, { 29,  7 },
        { 28,  6 }, { 27,  6 },
    },
    { /* DC bias table 10 */
        { 22,  7 }, {  8,  7 }, {  2,  6 }, { 31,  5 }, { 24,  4 }, { 29,  6 },
        {  3,  6 }, { 25,  5 }, { 30,  5 }, {  1,  5 }, { 23,  4 }, { 16,  6 },
        {  7,  6 }, { 19,  6 }, { 26,  6 }, { 13,  4 }, { 12,  4 }, { 11,  4 },
        { 14,  5 }, {  6, 10 }, {  5, 10 }, {  4,  9 }, { 21,  8 }, { 20,  7 },
        { 17,  6 }, {  0,  4 }, { 28,  6 }, { 18,  6 }, { 27,  6 }, { 15,  6 },
        { 10,  3 }, {  9,  3 },
    },
    { /* DC bias table 11 */
        {  0,  3 }, {  4,  7 }, { 21,  7 }, { 30,  6 }, { 15,  5 }, { 25,  5 },
        { 29,  6 }, {  7,  7 }, {  6, 10 }, {  5, 10 }, { 22,  9 }, {  8,  8 },
        { 23,  4 }, { 26,  6 }, { 19,  6 }, { 16,  6 }, {  2,  6 }, { 13,  4 },
        {  1,  5 }, { 14,  5 }, { 12,  4 }, { 11,  4 }, { 20,  7 }, { 31,  7 },
        { 17,  6 }, {  3,  6 }, { 18,  6 }, { 27,  6 }, { 28,  6 }, { 24,  5 },
        { 10,  3 }, {  9,  3 },
    },
    { /* DC bias table 12 */
        {  9,  2 }, {  0,  3 }, { 14,  5 }, {  3,  5 }, { 26,  6 }, { 18,  6 },
        { 17,  6 }, {  8,  8 }, { 21,  8 }, { 30,  7 }, { 12,  4 }, { 11,  4 },
        { 15,  6 }, {  7,  8 }, {  6, 10 }, {  5, 10 }, { 22,  9 }, {  4,  7 },
        { 28,  6 }, { 27,  6 }, { 24,  5 }, { 25,  6 }, {  2,  6 }, {  1,  5 },
        { 23,  5 }, { 29,  7 }, { 19,  7 }, { 16,  7 }, { 31,  8 }, { 20,  8 },
        { 13,  5 }, { 10,  3 },
    },
    { /* DC bias table 13 */
        {  9,  2 }, {  0,  3 }, {  2,  5 }, { 14,  5 }, { 24,  5 }, { 17,  6 },
        { 29,  7 }, { 21,  8 }, {  5,  8 }, { 12,  4 }, { 11,  4 }, { 28,  6 },
        {  4,  6 }, { 15,  6 }, { 27,  6 }, { 23,  5 }, { 25,  6 }, {  6, 10 },
        { 22, 10 }, {  8,  9 }, { 30,  8 }, { 19,  7 }, {  3,  5 }, { 16,  7 },
        { 26,  7 }, {  7,  9 }, { 31,  9 }, { 20,  8 }, { 18,  7 }, { 13,  5 },
        {  1,  5 }, { 10,  3 },
    },
    { /* DC bias table 14 */
        {  9,  2 }, {  0,  3 }, {  3,  4 }, {  1,  4 }, { 12,  4 }, { 11,  4 },
        { 23,  5 }, { 15,  6 }, { 30,  8 }, { 21,  8 }, {  7,  9 }, {  6,  9 },
        { 31,  9 }, { 22, 10 }, {  8, 10 }, {  2,  5 }, {  5,  7 }, { 19,  7 },
        { 16,  7 }, { 26,  7 }, { 13,  5 }, { 18,  7 }, { 29,  8 }, { 20,  8 },
        { 24,  6 }, { 14,  6 }, { 17,  7 }, { 28,  7 }, {  4,  6 }, { 25,  7 },
        { 27,  7 }, { 10,  3 },
    },
    { /* DC bias table 15 */
        { 10,  2 }, {  9,  2 }, { 12,  4 }, { 11,  4 }, { 15,  6 }, {  5,  6 },
        { 30,  9 }, { 29,  9 }, { 28,  8 }, { 22, 12 }, {  8, 12 }, {  7, 11 },
        { 31, 10 }, { 21,  9 }, { 26,  8 }, { 19,  7 }, { 16,  7 }, {  3,  4 },
        {  2,  5 }, {  4,  5 }, { 18,  7 }, { 24,  7 }, { 17,  7 }, {  6,  8 },
        { 25,  8 }, { 13,  5 }, { 14,  6 }, { 27,  8 }, { 20,  8 }, { 23,  7 },
        {  1,  5 }, {  0,  4 },
    },
    { /* AC bias group 1, table 0 */
        { 15,  4 }, { 11,  4 }, { 12,  4 }, { 21,  4 }, {  0,  5 }, { 26,  7 },
        {  1,  7 }, { 24,  6 }, { 22,  5 }, { 30,  5 }, { 14,  4 }, { 10,  4 },
        {  9,  4 }, { 17,  4 }, { 13,  4 }, { 23,  5 }, { 28,  7 }, { 25,  7 },
        { 27,  8 }, {  2,  9 }, { 29, 11 }, {  5, 13 }, {  8, 14 }, {  6, 14 },
        {  4, 12 }, {  3, 10 }, { 31,  7 }, { 20,  4 }, { 18,  4 }, { 16,  5 },
        {  7,  5 }, { 19,  4 },
    },
    { /* AC bias group 1, table 1 */
        { 15,  4 }, {  7,  4 }, { 11,  4 }, { 12,  4 }, {  1,  6 }, { 31,  6 },
        { 26,  7 }, { 27,  8 }, {  2,  8 }, { 22,  6 }, { 17,  4 }, { 14,  4 },
        { 30,  5 }, {  0,  5 }, {  9,  4 }, { 10,  4 }, { 20,  4 }, { 13,  4 },
        { 24,  6 }, { 25,  7 }, {  3,  8 }, {  6, 12 }, {  5, 12 }, { 29, 12 },
        {  8, 12 }, {  4, 10 }, { 28,  9 }, { 21,  5 }, { 18,  4 }, { 16,  5 },
        { 23,  5 }, { 19,  4 },
    },
    { /* AC bias group 1, table 2 */
        { 21,  5 }, { 25,  6 }, {  1,  6 }, { 15,  4 }, { 20,  4 }, {  7,  4 },
        { 11,  4 }, { 12,  4 }, { 17,  4 }, { 14,  4 }, {  3,  7 }, { 22,  7 },
        { 31,  6 }, { 24,  6 }, { 27,  8 }, {  2,  8 }, {  6, 12 }, { 29, 13 },
        {  8, 13 }, {  5, 11 }, {  4, 10 }, { 28,  9 }, { 26,  8 }, { 10,  4 },
        {  9,  4 }, { 19,  4 }, { 18,  4 }, { 30,  5 }, {  0,  5 }, { 13,  4 },
        { 16,  5 }, { 23,  5 },
    },
    { /* AC bias group 1, table 3 */
        { 28,  7 }, { 27,  7 }, { 22,  8 }, {  8, 12 }, {  6, 12 }, { 29, 11 },
        {  5, 10 }, {  4,  9 }, {  2,  7 }, { 21,  6 }, {  1,  6 }, { 15,  4 },
        { 23,  4 }, {  7,  4 }, { 11,  4 }, { 17,  4 }, { 12,  4 }, { 19,  4 },
        { 25,  6 }, { 26,  7 }, {  3,  7 }, { 20,  5 }, { 18,  4 }, { 14,  4 },
        { 31,  6 }, { 24,  6 }, { 30,  5 }, { 10,  4 }, {  9,  4 }, { 13,  4 },
        { 16,  5 }, {  0,  5 },
    },
    { /* AC bias group 1, table 4 */
        { 30,  4 }, { 15,  4 }, { 17,  4 }, {  0,  4 }, {  7,  4 }, { 18,  4 },
        { 23,  4 }, { 21,  7 }, { 27,  7 }, {  2,  7 }, { 26,  7 }, { 25,  6 },
        {  1,  6 }, { 12,  4 }, { 11,  4 }, { 14,  4 }, { 16,  5 }, { 28,  8 },
        {  5, 10 }, { 22, 11 }, {  8, 13 }, {  6, 13 }, { 29, 12 }, {  4,  9 },
        {  3,  7 }, { 20,  6 }, { 13,  4 }, { 19,  5 }, { 31,  6 }, { 24,  6 },
        { 10,  4 }, {  9,  4 },
    },
    { /* AC bias group 1, table 5 */
        {  9,  3 }, { 30,  4 }, {  3,  6 }, { 28,  7 }, { 27,  7 }, { 31,  5 },
        {  7,  4 }, { 24,  5 }, { 19,  5 }, {  0,  4 }, { 12,  4 }, { 11,  4 },
        { 14,  4 }, { 23,  4 }, { 16,  5 }, { 21,  9 }, {  6, 11 }, { 22, 13 },
        {  8, 13 }, { 29, 12 }, {  5, 10 }, {  4,  8 }, {  2,  7 }, { 20,  7 },
        { 26,  7 }, { 13,  4 }, { 18,  5 }, { 25,  6 }, {  1,  6 }, { 17,  5 },
        { 15,  5 }, { 10,  4 },
    },
    { /* AC bias group 1, table 6 */
        { 10,  3 }, {  9,  3 }, { 18,  5 }, { 25,  5 }, { 26,  6 }, { 19,  6 },
        {  1,  5 }, { 31,  5 }, { 17,  5 }, { 14,  4 }, { 24,  5 }, {  3,  6 },
        {  6, 10 }, {  8, 12 }, { 22, 13 }, { 21, 13 }, { 29, 11 }, {  5,  9 },
        { 20,  8 }, { 27,  7 }, { 12,  4 }, { 11,  4 }, { 13,  4 }, {  0,  4 },
        { 23,  4 }, { 15,  5 }, {  7,  5 }, {  4,  8 }, { 28,  8 }, {  2,  7 },
        { 16,  6 }, { 30,  5 },
    },
    { /* AC bias group 1, table 7 */
        {  0,  3 }, { 10,  3 }, {  9,  3 }, {  3,  5 }, { 27,  6 }, { 16,  6 },
        { 13,  4 }, { 31,  5 }, { 17,  6 }, {  4,  7 }, { 28,  7 }, { 11,  4 },
        { 12,  4 }, { 24,  5 }, {  7,  5 }, { 25,  5 }, { 26,  6 }, {  2,  6 },
        {  1,  5 }, { 14,  5 }, { 23,  4 }, { 19,  8 }, { 20, 13 }, {  8, 13 },
        { 22, 13 }, { 21, 13 }, { 29, 11 }, {  6, 10 }, {  5,  9 }, { 18,  7 },
        { 15,  6 }, { 30,  5 },
    },
    { /* AC bias group 1, table 8 */
        {  9,  3 }, { 24,  4 }, {  7,  4 }, { 17,  5 }, { 19,  6 }, { 20,  7 },
        {  2,  8 }, {  3,  9 }, {  4, 11 }, {  6, 13 }, {  5, 13 }, { 22, 12 },
        { 21, 10 }, { 14,  4 }, { 25,  5 }, { 15,  5 }, { 27,  6 }, { 29,  6 },
        { 28,  5 }, { 30,  4 }, { 13,  4 }, { 12,  4 }, { 11,  4 }, {  8,  7 },
        {  1,  7 }, { 16,  6 }, { 31,  5 }, { 23,  4 }, { 18,  6 }, { 26,  6 },
        {  0,  5 }, { 10,  4 },
    },
    { /* AC bias group 1, table 9 */
        { 10,  3 }, {  9,  3 }, { 27,  6 }, { 20,  7 }, {  6, 12 }, {  5, 12 },
        { 22, 11 }, {  4, 10 }, { 21,  9 }, {  8,  8 }, { 25,  5 }, { 14,  4 },
        { 19,  6 }, {  1,  6 }, { 15,  5 }, {  0,  4 }, { 30,  4 }, { 13,  4 },
        { 31,  5 }, { 29,  7 }, {  3,  8 }, {  2,  8 }, { 26,  6 }, { 12,  4 },
        { 11,  4 }, { 28,  6 }, { 16,  6 }, {  7,  5 }, { 23,  4 }, { 18,  6 },
        { 17,  6 }, { 24,  5 },
    },
    { /* AC bias group 1, table 10 */
        {  9,  3 }, {  2,  6 }, {  8,  7 }, { 22, 12 }, {  6, 12 }, {  5, 11 },
        { 21, 10 }, {  4,  9 }, { 20,  8 }, {  1,  5 }, { 15,  5 }, { 26,  5 },
        { 24,  4 }, { 29,  6 }, { 18,  6 }, { 28,  5 }, { 13,  4 }, { 16,  6 },
        { 27,  6 }, { 25,  5 }, { 30,  4 }, { 12,  4 }, { 11,  4 }, { 17,  6 },
        { 19,  7 }, {  3,  7 }, { 31,  5 }, {  0,  4 }, {  7,  5 }, { 14,  5 },
        { 23,  4 }, { 10,  4 },
    },
    { /* AC bias group 1, table 11 */
        { 12,  4 }, { 11,  4 }, { 10,  3 }, {  9,  3 }, { 23,  3 }, {  7,  5 },
        { 14,  5 }, {  3,  6 }, {  6, 11 }, { 22, 12 }, { 21, 12 }, {  5, 10 },
        { 20,  9 }, {  4,  8 }, { 18,  7 }, {  1,  5 }, { 24,  4 }, { 15,  6 },
        { 29,  6 }, { 28,  5 }, { 26,  5 }, {  8,  8 }, { 19,  8 }, { 16,  7 },
        { 27,  6 }, { 13,  5 }, { 30,  5 }, { 25,  5 }, { 17,  7 }, {  2,  7 },
        { 31,  6 }, {  0,  4 },
    },
    { /* AC bias group 1, table 12 */
        { 10,  3 }, {  9,  3 }, { 25,  4 }, {  4,  7 }, { 18,  7 }, { 16,  7 },
        { 17,  7 }, { 28,  5 }, {  0,  3 }, { 23,  3 }, { 24,  4 }, { 29,  6 },
        {  2,  6 }, { 13,  5 }, { 26,  5 }, { 30,  5 }, {  1,  5 }, { 27,  6 },
        {  7,  6 }, {  3,  6 }, {  8,  8 }, { 22, 13 }, { 21, 13 }, {  6, 12 },
        { 20, 11 }, {  5, 10 }, { 19,  9 }, { 15,  7 }, { 14,  6 }, { 31,  6 },
        { 12,  5 }, { 11,  5 },
    },
    { /* AC bias group 1, table 13 */
        { 10,  3 }, {  9,  3 }, {  1,  4 }, { 13,  5 }, { 29,  6 }, {  7,  6 },
        { 23,  3 }, {  0,  3 }, { 24,  4 }, { 30,  5 }, {  3,  5 }, { 28,  5 },
        { 14,  6 }, { 31,  6 }, { 12,  5 }, { 11,  5 }, { 26,  5 }, { 15,  7 },
        {  4,  7 }, { 27,  6 }, { 25,  5 }, { 16,  8 }, { 17,  8 }, { 20, 12 },
        { 22, 13 }, { 21, 13 }, {  6, 11 }, { 19, 10 }, {  5,  9 }, {  8,  9 },
        { 18,  9 }, {  2,  6 },
    },
    { /* AC bias group 1, table 14 */
        { 10,  3 }, {  9,  3 }, { 23,  3 }, { 19, 10 }, { 20, 11 }, { 22, 12 },
        { 21, 12 }, { 18,  9 }, { 17,  8 }, {  5,  7 }, { 14,  6 }, { 30,  5 },
        {  1,  4 }, {  0,  3 }, { 24,  4 }, { 28,  5 }, { 15,  7 }, { 16,  8 },
        {  8,  9 }, {  6,  9 }, { 31,  6 }, {  2,  5 }, { 12,  5 }, { 11,  5 },
        {  4,  6 }, { 27,  6 }, { 26,  5 }, { 13,  6 }, { 29,  7 }, {  7,  7 },
        {  3,  5 }, { 25,  5 },
    },
    { /* AC bias group 1, table 15 */
        {  0,  3 }, { 10,  3 }, {  9,  3 }, { 24,  4 }, { 29,  7 }, { 17, 10 },
        { 22, 13 }, { 21, 13 }, { 18, 13 }, { 20, 14 }, { 19, 14 }, {  8, 11 },
        { 16,  9 }, { 15,  8 }, { 27,  6 }, { 12,  5 }, { 23,  3 }, {  1,  4 },
        { 11,  5 }, { 13,  6 }, {  7,  7 }, { 14,  7 }, {  3,  4 }, {  2,  5 },
        { 26,  5 }, { 28,  6 }, {  5,  6 }, {  4,  5 }, {  6,  7 }, { 31,  7 },
        { 30,  6 }, { 25,  5 },
    },
    { /* AC bias group 2, table 0 */
        { 24,  5 }, { 28,  6 }, { 21,  6 }, { 23,  4 }, {  7,  4 }, { 15,  4 },
        { 17,  4 }, { 25,  6 }, {  2,  8 }, { 22,  9 }, {  8,  9 }, {  1,  7 },
        {  0,  5 }, { 19,  4 }, { 11,  4 }, { 12,  4 }, {  9,  4 }, { 10,  4 },
        { 18,  4 }, { 14,  4 }, { 20,  5 }, { 26,  7 }, { 27,  8 }, {  6, 12 },
        {  5, 12 }, {  4, 11 }, { 29, 10 }, {  3,  9 }, { 31,  6 }, { 30,  5 },
        { 16,  5 }, { 13,  4 },
    },
    { /* AC bias group 2, table 1 */
        { 30,  4 }, {  1,  6 }, { 28,  6 }, { 24,  5 }, { 17,  4 }, { 15,  4 },
        { 18,  4 }, { 23,  4 }, { 31,  5 }, { 27,  7 }, {  3,  8 }, { 21,  8 },
        { 25,  6 }, {  7,  4 }, { 12,  4 }, { 11,  4 }, { 14,  4 }, { 20,  6 },
        { 26,  7 }, {  2,  8 }, {  6, 13 }, {  5, 13 }, { 22, 12 }, {  4, 11 },
        { 29, 10 }, {  8,  9 }, { 16,  5 }, {  9,  4 }, { 10,  4 }, { 19,  5 },
        {  0,  5 }, { 13,  4 },
    },
    { /* AC bias group 2, table 2 */
        { 13,  3 }, { 15,  4 }, {  0,  4 }, { 30,  4 }, { 24,  5 }, { 31,  5 },
        { 23,  4 }, {  7,  4 }, { 20,  7 }, {  8,  8 }, {  4, 10 }, {  5, 12 },
        { 22, 13 }, {  6, 13 }, { 21, 11 }, { 29,  9 }, { 28,  6 }, { 16,  5 },
        { 14,  4 }, {  3,  8 }, {  2,  8 }, { 27,  7 }, { 25,  6 }, { 18,  5 },
        { 11,  4 }, { 12,  4 }, { 10,  4 }, {  9,  4 }, { 17,  5 }, { 19,  6 },
        { 26,  7 }, {  1,  7 },
    },
    { /* AC bias group 2, table 3 */
        { 30,  4 }, { 18,  5 }, { 16,  5 }, {  9,  3 }, { 10,  3 }, { 31,  5 },
        { 26,  6 }, {  1,  6 }, {  0,  4 }, { 14,  4 }, { 17,  5 }, { 24,  5 },
        { 23,  4 }, { 11,  4 }, { 12,  4 }, { 13,  4 }, { 15,  5 }, {  5, 11 },
        {  6, 12 }, { 22, 13 }, { 21, 13 }, {  4, 10 }, { 20,  9 }, {  3,  8 },
        { 19,  7 }, { 25,  6 }, { 28,  6 }, { 27,  7 }, {  2,  8 }, { 29,  9 },
        {  8,  9 }, {  7,  5 },
    },
    { /* AC bias group 2, table 4 */
        { 30,  4 }, { 14,  4 }, {  9,  3 }, { 10,  3 }, { 25,  5 }, { 18,  6 },
        { 20, 10 }, {  5, 11 }, {  6, 12 }, { 22, 13 }, { 21, 13 }, {  4,  9 },
        { 29,  8 }, {  3,  7 }, { 31,  5 }, { 15,  5 }, { 27,  6 }, {  2,  7 },
        {  8,  8 }, { 19,  8 }, { 28,  5 }, { 26,  6 }, { 16,  6 }, { 24,  5 },
        { 13,  4 }, { 12,  4 }, { 11,  4 }, {  0,  4 }, { 23,  4 }, { 17,  6 },
        {  1,  6 }, {  7,  5 },
    },
    { /* AC bias group 2, table 5 */
        {  0,  3 }, { 30,  4 }, { 31,  5 }, { 25,  5 }, {  9,  3 }, { 10,  3 },
        { 13,  4 }, {  1,  5 }, {  7,  5 }, { 27,  6 }, { 29,  8 }, {  8,  8 },
        { 19, 10 }, { 20, 13 }, {  6, 13 }, { 22, 13 }, { 21, 13 }, {  5, 11 },
        {  4,  9 }, { 18,  8 }, { 26,  6 }, { 15,  6 }, { 11,  4 }, { 12,  4 },
        { 14,  5 }, { 28,  5 }, { 24,  5 }, { 17,  7 }, { 16,  7 }, {  2,  7 },
        {  3,  7 }, { 23,  4 },
    },
    { /* AC bias group 2, table 6 */
        { 23,  3 }, {  7,  5 }, { 31,  5 }, { 14,  5 }, { 25,  5 }, {  0,  3 },
        { 10,  3 }, {  9,  3 }, { 18,  9 }, { 22, 12 }, { 21, 12 }, {  6, 12 },
        { 20, 13 }, { 19, 13 }, {  5, 10 }, {  8,  8 }, { 17,  8 }, { 16,  8 },
        { 27,  6 }, { 26,  6 }, {  2,  6 }, { 11,  4 }, { 12,  4 }, {  1,  5 },
        { 30,  5 }, { 28,  5 }, {  3,  6 }, { 29,  8 }, {  4,  8 }, { 15,  7 },
        { 24,  5 }, { 13,  5 },
    },
    { /* AC bias group 2, table 7 */
        { 23,  3 }, {  1,  4 }, { 13,  5 }, { 15,  8 }, {  8,  9 }, { 18, 12 },
        { 20, 13 }, { 19, 13 }, { 22, 13 }, { 21, 13 }, { 17, 12 }, { 16, 11 },
        {  6, 11 }, {  7,  7 }, { 27,  6 }, {  0,  3 }, { 11,  4 }, { 12,  4 },
        {  9,  3 }, { 10,  3 }, {  2,  5 }, { 30,  5 }, { 26,  6 }, {  4,  6 },
        { 25,  5 }, { 31,  6 }, {  5,  8 }, { 29,  8 }, { 14,  7 }, {  3,  5 },
        { 28,  5 }, { 24,  5 },
    },
    { /* AC bias group 2, table 8 */
        { 10,  3 }, {  9,  3 }, { 26,  5 }, { 15,  5 }, { 24,  4 }, {  7,  4 },
        { 16,  6 }, { 17,  6 }, { 25,  5 }, { 30,  4 }, { 13,  4 }, {  1,  7 },
        {  8,  7 }, { 27,  6 }, { 31,  5 }, {  0,  5 }, { 19,  8 }, {  2,  9 },
        { 22, 13 }, { 21, 13 }, {  4, 13 }, {  6, 14 }, {  5, 14 }, { 20, 11 },
        {  3, 10 }, { 18,  7 }, { 29,  6 }, { 12,  4 }, { 11,  4 }, { 14,  5 },
        { 28,  5 }, { 23,  4 },
    },
    { /* AC bias group 2, table 9 */
        {  9,  3 }, { 23,  3 }, { 28,  4 }, { 24,  4 }, { 13,  4 }, { 30,  4 },
        {  2,  7 }, { 18,  7 }, {  1,  6 }, { 14,  5 }, {  0,  4 }, { 25,  5 },
        { 15,  6 }, {  4, 10 }, {  6, 13 }, {  5, 13 }, { 22, 13 }, { 21, 13 },
        { 20, 11 }, { 19,  9 }, {  3,  8 }, { 16,  7 }, { 31,  5 }, { 27,  6 },
        { 17,  7 }, {  8,  7 }, { 12,  4 }, { 11,  4 }, {  7,  5 }, { 29,  6 },
        { 26,  6 }, { 10,  4 },
    },
    { /* AC bias group 2, table 10 */
        {  9,  3 }, { 30,  4 }, { 17,  7 }, {  4,  9 }, { 22, 12 }, { 21, 12 },
        {  5, 12 }, { 20, 13 }, {  6, 13 }, { 19, 10 }, { 18,  8 }, {  8,  6 },
        {  1,  5 }, { 23,  3 }, { 24,  4 }, { 26,  5 }, { 29,  5 }, { 31,  5 },
        { 16,  7 }, {  3,  7 }, {  2,  7 }, { 15,  7 }, { 28,  4 }, { 11,  4 },
        { 12,  4 }, {  7,  5 }, { 25,  5 }, { 13,  5 }, { 14,  6 }, { 27,  6 },
        {  0,  4 }, { 10,  4 },
    },
    { /* AC bias group 2, table 11 */
        { 23,  3 }, { 10,  3 }, {  9,  3 }, {  0,  3 }, { 27,  5 }, { 14,  6 },
        {  2,  6 }, { 24,  4 }, { 13,  5 }, { 26,  5 }, { 30,  5 }, { 29,  5 },
        {  1,  5 }, { 15,  7 }, {  4,  9 }, { 19, 11 }, { 22, 13 }, { 21, 13 },
        {  5, 13 }, { 20, 14 }, {  6, 14 }, { 18, 10 }, { 17,  9 }, { 16,  9 },
        { 31,  6 }, { 28,  4 }, { 25,  5 }, {  7,  6 }, {  8,  7 }, {  3,  7 },
        { 12,  5 }, { 11,  5 },
    },
    { /* AC bias group 2, table 12 */
        { 23,  3 }, { 10,  3 }, {  9,  3 }, {  1,  4 }, { 24,  4 }, {  3,  5 },
        { 26,  5 }, {  4,  7 }, { 15,  8 }, { 16,  9 }, { 17, 10 }, { 18, 11 },
        { 22, 14 }, { 21, 14 }, {  6, 14 }, { 20, 15 }, { 19, 15 }, {  5, 12 },
        { 31,  6 }, { 29,  5 }, {  0,  3 }, { 25,  5 }, {  7,  6 }, { 14,  7 },
        {  8,  7 }, { 13,  6 }, { 30,  6 }, { 11,  5 }, { 28,  4 }, { 12,  5 },
        {  2,  6 }, { 27,  6 },
    },
    { /* AC bias group 2, table 13 */
        { 28,  3 }, {  9,  3 }, { 10,  3 }, { 24,  4 }, {  4,  6 }, { 15,  8 },
        {  5,  9 }, { 16, 10 }, { 22, 13 }, { 21, 13 }, { 18, 14 }, {  6, 14 },
        { 20, 14 }, { 19, 14 }, { 17, 11 }, { 14,  7 }, {  7,  6 }, { 13,  6 },
        {  1,  4 }, {  2,  5 }, { 25,  5 }, {  0,  3 }, { 29,  5 }, { 30,  6 },
        {  8,  7 }, { 31,  7 }, { 12,  5 }, { 11,  5 }, {  3,  5 }, { 27,  6 },
        { 26,  6 }, { 23,  4 },
    },
    { /* AC bias group 2, table 14 */
        { 28,  3 }, { 10,  3 }, {  9,  3 }, { 13,  6 }, { 30,  6 }, {  4,  5 },
        { 25,  5 }, { 29,  5 }, {  0,  3 }, {  1,  4 }, { 12,  5 }, { 11,  5 },
        {  3,  4 }, { 15,  9 }, {  6, 11 }, { 18, 12 }, { 20, 13 }, { 19, 13 },
        { 16, 11 }, { 22, 13 }, { 21, 13 }, { 17, 12 }, { 14,  8 }, { 31,  7 },
        { 26,  6 }, {  2,  5 }, { 27,  6 }, {  7,  7 }, {  5,  8 }, {  8,  8 },
        { 24,  5 }, { 23,  4 },
    },
    { /* AC bias group 2, table 15 */
        {  3,  3 }, { 25,  5 }, { 14,  9 }, {  6,  9 }, { 15, 10 }, { 16, 12 },
        { 18, 13 }, { 17, 13 }, { 20, 13 }, { 19, 13 }, { 22, 13 }, { 21, 13 },
        {  8,  9 }, { 13,  7 }, { 29,  6 }, {  4,  4 }, { 10,  3 }, {  0,  3 },
        {  9,  3 }, { 26,  6 }, { 27,  6 }, { 12,  5 }, { 11,  5 }, {  5,  6 },
        {  7,  8 }, { 31,  8 }, { 30,  7 }, {  1,  4 }, { 24,  5 }, {  2,  5 },
        { 23,  4 }, { 28,  4 },
    },
    { /* AC bias group 3, table 0 */
        {  9,  3 }, { 29,  6 }, {  1,  6 }, { 20,  7 }, {  8,  7 }, { 26,  6 },
        {  0,  4 }, {  7,  4 }, { 16,  5 }, { 24,  5 }, { 31,  5 }, { 18,  5 },
        { 30,  4 }, { 23,  4 }, { 14,  4 }, { 17,  5 }, { 19,  6 }, { 25,  6 },
        { 11,  4 }, { 12,  4 }, {  2,  8 }, {  4, 10 }, {  5, 12 }, { 22, 13 },
        {  6, 13 }, { 21, 11 }, {  3,  9 }, { 27,  7 }, { 28,  6 }, { 15,  5 },
        { 13,  4 }, { 10,  4 },
    },
    { /* AC bias group 3, table 1 */
        { 10,  3 }, {  9,  3 }, { 14,  4 }, { 18,  6 }, { 27,  6 }, { 28,  5 },
        {  3,  7 }, {  6, 12 }, {  5, 12 }, { 22, 12 }, { 21, 12 }, { 20, 10 },
        {  4,  9 }, { 19,  8 }, {  2,  7 }, {  8,  7 }, { 15,  5 }, { 30,  4 },
        { 31,  5 }, { 26,  6 }, { 29,  6 }, { 24,  5 }, { 16,  6 }, { 17,  6 },
        {  0,  4 }, { 23,  4 }, { 13,  4 }, { 12,  4 }, { 11,  4 }, {  1,  6 },
        { 25,  6 }, {  7,  5 },
    },
    { /* AC bias group 3, table 2 */
        { 10,  3 }, {  9,  3 }, {  3,  6 }, {  4,  8 }, {  5, 10 }, { 20, 12 },
        {  6, 12 }, { 22, 12 }, { 21, 12 }, { 19,  9 }, {  8,  7 }, { 15,  5 },
        { 25,  5 }, { 17,  6 }, { 16,  6 }, {  1,  5 }, { 28,  5 }, { 30,  4 },
        { 27,  6 }, { 18,  7 }, {  2,  7 }, { 31,  5 }, { 13,  4 }, { 29,  6 },
        { 26,  6 }, { 24,  5 }, { 23,  4 }, {  7,  5 }, { 14,  5 }, { 12,  4 },
        { 11,  4 }, {  0,  4 },
    },
    { /* AC bias group 3, table 3 */
        {  8,  7 }, { 17,  7 }, { 15,  6 }, { 31,  5 }, {  7,  5 }, { 25,  5 },
        {  0,  3 }, {  9,  3 }, { 10,  3 }, { 13,  4 }, { 14,  5 }, { 28,  5 },
        {  4,  8 }, { 19, 11 }, { 20, 13 }, {  6, 13 }, { 22, 13 }, { 21, 13 },
        {  5, 10 }, { 18,  9 }, { 16,  7 }, {  2,  6 }, {  3,  6 }, { 27,  6 },
        { 12,  4 }, { 11,  4 }, {  1,  5 }, { 29,  6 }, { 26,  6 }, { 24,  5 },
        { 30,  5 }, { 23,  4 },
    },
    { /* AC bias group 3, table 4 */
        { 23,  3 }, { 26,  5 }, { 17,  8 }, { 18, 10 }, {  6, 11 }, { 20, 13 },
        { 19, 13 }, { 22, 13 }, { 21, 13 }, {  5,  9 }, {  8,  8 }, { 16,  8 },
        {  4,  7 }, { 15,  7 }, {  3,  5 }, { 25,  5 }, {  9,  3 }, { 10,  3 },
        {  0,  3 }, { 28,  5 }, { 30,  5 }, { 11,  4 }, { 12,  4 }, {  7,  6 },
        { 27,  6 }, { 29,  6 }, { 14,  6 }, { 13,  5 }, { 24,  5 }, { 31,  6 },
        {  2,  6 }, {  1,  5 },
    },
    { /* AC bias group 3, table 5 */
        { 23,  3 }, {  7,  6 }, {  4,  6 }, { 30,  5 }, { 25,  5 }, {  2,  5 },
        { 10,  3 }, {  9,  3 }, {  0,  3 }, {  1,  4 }, { 29,  6 }, { 31,  6 },
        { 13,  5 }, { 12,  4 }, { 11,  4 }, { 27,  6 }, { 26,  6 }, {  3,  5 },
        { 24,  5 }, {  8,  9 }, { 22, 13 }, { 21, 13 }, { 18, 13 }, { 20, 14 },
        { 19, 14 }, { 17, 11 }, { 16, 11 }, {  6, 11 }, { 15,  9 }, {  5,  9 },
        { 14,  7 }, { 28,  6 },
    },
    { /* AC bias group 3, table 6 */
        { 23,  3 }, { 31,  6 }, {  7,  7 }, { 14,  8 }, { 20, 13 }, { 19, 13 },
        { 22, 13 }, { 21, 13 }, { 16, 13 }, {  8, 13 }, { 18, 13 }, { 17, 13 },
        { 15, 10 }, {  6,  9 }, {  4,  5 }, {  3,  4 }, {  0,  3 }, { 25,  5 },
        { 29,  7 }, {  5,  7 }, { 30,  6 }, {  1,  4 }, { 10,  3 }, {  9,  3 },
        {  2,  5 }, { 28,  6 }, { 13,  6 }, { 11,  4 }, { 12,  4 }, { 27,  6 },
        { 26,  6 }, { 24,  5 },
    },
    { /* AC bias group 3, table 7 */
        {  0,  3 }, {  4,  4 }, { 24,  5 }, {  5,  5 }, {  1,  4 }, { 25,  5 },
        { 26,  7 }, { 31,  7 }, { 27,  6 }, { 23,  3 }, { 10,  3 }, {  9,  3 },
        { 12,  4 }, { 11,  4 }, {  2,  5 }, {  7,  8 }, { 30, 10 }, { 22, 13 },
        { 21, 13 }, {  8, 13 }, { 16, 14 }, { 15, 14 }, { 14, 12 }, { 18, 14 },
        { 17, 14 }, { 20, 14 }, { 19, 14 }, { 29,  9 }, {  6,  7 }, { 28,  7 },
        { 13,  7 }, {  3,  4 },
    },
    { /* AC bias group 3, table 8 */
        { 26,  5 }, { 29,  5 }, { 24,  4 }, {  9,  3 }, { 10,  3 }, { 30,  4 },
        { 13,  4 }, {  8,  6 }, { 22, 12 }, { 21, 12 }, {  5, 12 }, { 20, 13 },
        {  6, 13 }, {  4, 10 }, { 19,  9 }, {  3,  8 }, { 17,  7 }, { 25,  5 },
        {  1,  6 }, { 15,  6 }, { 14,  5 }, {  0,  4 }, { 31,  5 }, { 27,  6 },
        { 16,  7 }, { 18,  8 }, {  2,  8 }, {  7,  5 }, { 28,  5 }, { 12,  4 },
        { 11,  4 }, { 23,  4 },
    },
    { /* AC bias group 3, table 9 */
        {  9,  3 }, { 10,  3 }, {  0,  3 }, { 24,  4 }, { 26,  5 }, {  1,  5 },
        { 28,  4 }, {  7,  5 }, { 25,  5 }, { 11,  4 }, { 12,  4 }, { 15,  7 },
        {  3,  7 }, { 14,  6 }, { 30,  5 }, { 13,  5 }, {  8,  6 }, { 16,  8 },
        {  4, 10 }, {  5, 13 }, { 20, 14 }, {  6, 14 }, { 22, 14 }, { 21, 14 },
        { 19, 13 }, { 18, 11 }, { 17,  9 }, {  2,  7 }, { 23,  4 }, { 29,  5 },
        { 27,  6 }, { 31,  6 },
    },
    { /* AC bias group 3, table 10 */
        {  9,  3 }, { 10,  3 }, { 27,  5 }, { 30,  5 }, { 26,  5 }, { 13,  5 },
        {  0,  3 }, { 29,  4 }, {  3,  6 }, {  2,  6 }, { 25,  5 }, { 12,  4 },
        { 11,  4 }, { 28,  4 }, { 23,  4 }, {  1,  5 }, { 31,  6 }, { 15,  8 },
        {  4,  9 }, { 16, 10 }, { 17, 11 }, { 22, 14 }, { 21, 14 }, {  5, 13 },
        {  6, 14 }, { 20, 15 }, { 19, 15 }, { 18, 13 }, { 14,  7 }, {  7,  6 },
        {  8,  6 }, { 24,  5 },
    },
    { /* AC bias group 3, table 11 */
        { 12,  4 }, { 11,  4 }, {  2,  5 }, { 26,  5 }, {  1,  4 }, {  9,  3 },
        { 10,  3 }, {  3,  5 }, { 30,  6 }, { 14,  7 }, { 15,  9 }, { 18, 13 },
        {  6, 13 }, { 20, 13 }, { 19, 13 }, {  5, 11 }, { 22, 13 }, { 21, 13 },
        { 17, 12 }, { 16, 11 }, {  4,  8 }, { 25,  5 }, { 13,  6 }, {  8,  6 },
        {  0,  3 }, { 28,  4 }, { 23,  4 }, { 29,  4 }, { 31,  7 }, {  7,  7 },
        { 27,  6 }, { 24,  5 },
    },
    { /* AC bias group 3, table 12 */
        {  3,  4 }, { 25,  5 }, {  4,  6 }, { 30,  7 }, {  7,  7 }, { 29,  3 },
        { 10,  3 }, {  9,  3 }, { 23,  4 }, { 28,  4 }, {  0,  3 }, {  1,  4 },
        {  8,  6 }, { 27,  6 }, {  2,  5 }, { 24,  5 }, { 12,  5 }, { 11,  5 },
        { 14,  9 }, {  5, 10 }, { 15, 11 }, { 20, 14 }, { 19, 14 }, { 22, 14 },
        { 21, 14 }, { 16, 14 }, {  6, 14 }, { 18, 14 }, { 17, 14 }, { 31,  8 },
        { 13,  7 }, { 26,  6 },
    },
    { /* AC bias group 3, table 13 */
        { 23,  4 }, {  2,  4 }, { 29,  3 }, { 13,  7 }, { 31,  8 }, { 30,  8 },
        { 27,  6 }, { 24,  5 }, { 28,  4 }, { 12,  5 }, { 11,  5 }, {  5,  9 },
        { 14, 10 }, { 18, 13 }, { 17, 13 }, { 20, 13 }, { 19, 13 }, { 22, 13 },
        { 21, 13 }, {  6, 13 }, { 16, 14 }, { 15, 14 }, {  7,  8 }, {  8,  7 },
        { 25,  6 }, {  4,  6 }, { 26,  6 }, {  0,  3 }, {  3,  4 }, {  1,  4 },
        { 10,  3 }, {  9,  3 },
    },
    { /* AC bias group 3, table 14 */
        {  9,  2 }, {  3,  3 }, { 23,  5 }, { 27,  6 }, { 26,  6 }, {  2,  4 },
        {  0,  3 }, {  4,  5 }, { 24,  6 }, { 12,  6 }, { 11,  6 }, { 25,  7 },
        {  5,  9 }, { 14, 13 }, {  6, 13 }, { 16, 13 }, { 15, 13 }, { 31, 11 },
        { 22, 13 }, { 21, 13 }, { 18, 14 }, { 17, 14 }, { 20, 14 }, { 19, 14 },
        { 30, 11 }, {  8,  9 }, { 13, 10 }, {  7, 10 }, { 28,  5 }, { 29,  4 },
        {  1,  4 }, { 10,  3 },
    },
    { /* AC bias group 3, table 15 */
        { 10,  2 }, {  9,  2 }, {  0,  2 }, {  3,  4 }, { 12,  7 }, { 11,  7 },
        { 24,  9 }, { 23,  9 }, { 27,  8 }, {  5,  9 }, { 25, 10 }, {  6, 13 },
        {  8, 14 }, {  7, 14 }, { 22, 14 }, { 21, 14 }, { 31, 14 }, { 30, 14 },
        { 18, 14 }, { 17, 14 }, { 20, 14 }, { 19, 14 }, { 14, 14 }, { 13, 14 },
        { 16, 14 }, { 15, 14 }, { 26,  8 }, { 29,  5 }, {  2,  5 }, { 28,  6 },
        {  4,  6 }, {  1,  4 },
    },
    { /* AC bias group 4, table 0 */
        {  0,  3 }, { 28,  4 }, { 13,  4 }, {  9,  3 }, { 10,  3 }, {  1,  5 },
        { 14,  5 }, { 25,  5 }, { 31,  5 }, {  7,  5 }, { 16,  7 }, {  4,  9 },
        {  6, 12 }, { 20, 13 }, { 19, 13 }, { 22, 13 }, { 21, 13 }, {  5, 12 },
        { 18, 10 }, { 17,  8 }, { 27,  6 }, { 12,  4 }, { 11,  4 }, { 23,  4 },
        {  8,  7 }, {  3,  7 }, {  2,  7 }, { 15,  7 }, { 30,  5 }, { 24,  5 },
        { 26,  6 }, { 29,  6 },
    },
    { /* AC bias group 4, table 1 */
        { 26,  5 }, { 31,  5 }, {  7,  5 }, {  4,  7 }, { 22, 12 }, { 21, 12 },
        { 18, 13 }, {  6, 13 }, { 20, 13 }, { 19, 13 }, {  5, 10 }, { 17, 10 },
        { 16, 10 }, { 15,  8 }, { 14,  6 }, { 10,  3 }, {  9,  3 }, {  0,  3 },
        { 28,  4 }, { 25,  5 }, { 30,  5 }, {  8,  6 }, {  2,  6 }, { 13,  5 },
        { 23,  4 }, { 12,  4 }, { 24,  5 }, { 29,  5 }, { 11,  4 }, { 27,  6 },
        {  3,  6 }, {  1,  5 },
    },
    { /* AC bias group 4, table 2 */
        {  9,  3 }, { 10,  3 }, {  1,  4 }, { 29,  4 }, { 13,  5 }, { 25,  5 },
        { 28,  4 }, {  0,  3 }, {  3,  5 }, {  8,  6 }, {  4,  7 }, {  5,  9 },
        { 20, 13 }, { 19, 13 }, { 22, 13 }, { 21, 13 }, { 16, 13 }, {  6, 13 },
        { 18, 13 }, { 17, 13 }, { 15, 10 }, { 14,  8 }, { 23,  4 }, { 31,  6 },
        { 27,  6 }, { 24,  5 }, { 12,  4 }, { 11,  4 }, {  7,  6 }, { 30,  6 },
        { 26,  6 }, {  2,  6 },
    },
    { /* AC bias group 4, table 3 */
        {  3,  4 }, { 18, 12 }, { 17, 12 }, { 20, 12 }, { 19, 12 }, { 22, 12 },
        { 21, 12 }, {  6, 12 }, { 16, 13 }, { 15, 13 }, { 14,  9 }, {  5,  8 },
        { 31,  7 }, {  7,  7 }, { 30,  7 }, { 25,  5 }, { 12,  4 }, { 11,  4 },
        {  9,  3 }, { 10,  3 }, {  1,  4 }, { 28,  4 }, {  0,  3 }, { 23,  4 },
        {  2,  5 }, {  4,  6 }, {  8,  7 }, { 13,  7 }, { 29,  4 }, { 24,  5 },
        { 26,  6 }, { 27,  6 },
    },
    { /* AC bias group 4, table 4 */
        { 29,  3 }, { 25,  5 }, {  8,  7 }, { 13,  8 }, { 31, 10 }, {  6, 10 },
        { 18, 13 }, { 17, 13 }, { 20, 13 }, { 19, 13 }, { 22, 13 }, { 21, 13 },
        { 14, 13 }, { 16, 14 }, { 15, 14 }, {  7, 10 }, { 27,  6 }, { 23,  4 },
        {  0,  3 }, { 10,  3 }, {  9,  3 }, { 28,  4 }, { 24,  5 }, { 12,  5 },
        {  3,  4 }, { 11,  5 }, { 26,  6 }, {  5,  7 }, { 30,  7 }, {  1,  4 },
        {  2,  5 }, {  4,  5 },
    },
    { /* AC bias group 4, table 5 */
        { 23,  3 }, {  3,  3 }, {  4,  4 }, {  1,  4 }, {  2,  3 }, {  0,  3 },
        { 24,  6 }, { 26,  9 }, { 17, 13 }, { 16, 13 }, { 19, 13 }, { 18, 13 },
        { 13, 13 }, {  7, 13 }, { 15, 13 }, { 14, 13 }, {  6, 11 }, { 30, 12 },
        { 21, 14 }, { 20, 14 }, { 31, 14 }, { 22, 14 }, { 25,  8 }, {  8,  8 },
        { 27,  8 }, { 29,  5 }, { 12,  6 }, { 11,  6 }, { 28,  6 }, {  5,  6 },
        { 10,  3 }, {  9,  3 },
    },
    { /* AC bias group 4, table 6 */
        { 23,  3 }, {  3,  3 }, {  4,  4 }, {  1,  4 }, {  2,  3 }, {  0,  3 },
        { 24,  6 }, { 26,  9 }, { 17, 13 }, { 16, 13 }, { 19, 13 }, { 18, 13 },
        { 13, 13 }, {  7, 13 }, { 15, 13 }, { 14, 13 }, {  6, 11 }, { 30, 12 },
        { 21, 14 }, { 20, 14 }, { 31, 14 }, { 22, 14 }, { 25,  8 }, {  8,  8 },
        { 27,  8 }, { 29,  5 }, { 12,  6 }, { 11,  6 }, { 28,  6 }, {  5,  6 },
        { 10,  3 }, {  9,  3 },
    },
    { /* AC bias group 4, table 7 */
        { 23,  3 }, {  3,  3 }, {  4,  4 }, {  1,  4 }, {  2,  3 }, {  0,  3 },
        { 24,  6 }, { 26,  9 }, { 17, 13 }, { 16, 13 }, { 19, 13 }, { 18, 13 },
        { 13, 13 }, {  7, 13 }, { 15, 13 }, { 14, 13 }, {  6, 11 }, { 30, 12 },
        { 21, 14 }, { 20, 14 }, { 31, 14 }, { 22, 14 }, { 25,  8 }, {  8,  8 },
        { 27,  8 }, { 29,  5 }, { 12,  6 }, { 11,  6 }, { 28,  6 }, {  5,  6 },
        { 10,  3 }, {  9,  3 },
    },
    { /* AC bias group 4, table 8 */
        { 10,  3 }, {  9,  3 }, { 24,  4 }, {  7,  5 }, { 26,  5 }, {  0,  3 },
        {  2,  6 }, { 15,  7 }, {  6, 12 }, { 20, 13 }, { 19, 13 }, { 22, 13 },
        { 21, 13 }, {  5, 12 }, { 18, 10 }, {  4,  9 }, { 17,  9 }, { 16,  9 },
        {  1,  5 }, { 28,  4 }, { 12,  4 }, { 11,  4 }, { 13,  5 }, { 25,  5 },
        { 30,  5 }, { 29,  5 }, { 14,  6 }, { 27,  6 }, {  3,  7 }, {  8,  7 },
        { 31,  6 }, { 23,  4 },
    },
    { /* AC bias group 4, table 9 */
        { 13,  5 }, {  3,  5 }, {  1,  4 }, { 10,  3 }, {  9,  3 }, { 29,  4 },
        { 25,  5 }, { 31,  6 }, {  8,  6 }, { 12,  4 }, { 11,  4 }, {  0,  3 },
        { 28,  4 }, { 23,  4 }, { 14,  7 }, {  4,  8 }, { 16, 10 }, { 17, 11 },
        { 18, 14 }, {  6, 14 }, { 20, 14 }, { 19, 14 }, { 22, 14 }, { 21, 14 },
        {  5, 13 }, { 15,  9 }, {  7,  6 }, { 24,  5 }, { 27,  6 }, { 30,  6 },
        {  2,  6 }, { 26,  6 },
    },
    { /* AC bias group 4, table 10 */
        { 31,  6 }, {  7,  6 }, { 25,  5 }, { 28,  4 }, {  9,  3 }, { 10,  3 },
        { 12,  4 }, { 11,  4 }, { 30,  6 }, {  8,  6 }, {  2,  5 }, { 29,  4 },
        { 23,  4 }, {  1,  4 }, {  0,  3 }, { 24,  5 }, {  4,  7 }, { 15,  9 },
        {  5, 10 }, { 20, 13 }, { 19, 13 }, { 22, 13 }, { 21, 13 }, {  6, 13 },
        { 18, 14 }, { 17, 14 }, { 16, 12 }, { 14,  8 }, { 27,  6 }, {  3,  5 },
        { 13,  6 }, { 26,  6 },
    },
    { /* AC bias group 4, table 11 */
        { 12,  4 }, { 11,  4 }, { 25,  5 }, { 13,  6 }, { 30,  7 }, {  7,  7 },
        { 28,  4 }, {  3,  4 }, { 24,  5 }, {  4,  6 }, { 27,  6 }, { 23,  4 },
        { 29,  4 }, {  0,  3 }, {  1,  4 }, { 26,  6 }, { 31,  8 }, {  5,  9 },
        { 16, 13 }, {  6, 13 }, { 18, 13 }, { 17, 13 }, { 15, 12 }, { 20, 14 },
        { 19, 14 }, { 22, 14 }, { 21, 14 }, { 14, 10 }, {  8,  7 }, {  2,  5 },
        {  9,  3 }, { 10,  3 },
    },
    { /* AC bias group 4, table 12 */
        {  9,  2 }, { 28,  4 }, {  2,  4 }, { 30,  8 }, {  7,  8 }, {  8,  7 },
        { 27,  6 }, { 24,  6 }, { 25,  6 }, { 29,  4 }, { 11,  5 }, { 12,  5 },
        {  3,  4 }, {  0,  3 }, { 23,  5 }, { 13,  8 }, { 31, 10 }, { 18, 13 },
        { 17, 13 }, { 20, 13 }, { 19, 13 }, {  6, 13 }, { 16, 14 }, { 15, 14 },
        { 22, 14 }, { 21, 14 }, { 14, 13 }, {  5,  9 }, { 26,  7 }, {  4,  6 },
        {  1,  4 }, { 10,  3 },
    },
    { /* AC bias group 4, table 13 */
        {  1,  3 }, {  2,  4 }, { 29,  5 }, { 12,  6 }, { 11,  6 }, {  9,  2 },
        { 10,  2 }, {  0,  3 }, { 23,  6 }, {  4,  6 }, { 28,  6 }, { 30, 12 },
        {  6, 13 }, { 15, 14 }, { 14, 14 }, {  7, 11 }, { 13, 11 }, { 21, 14 },
        { 20, 14 }, { 31, 14 }, { 22, 14 }, { 17, 14 }, { 16, 14 }, { 19, 14 },
        { 18, 14 }, {  5,  9 }, { 25,  8 }, { 27,  8 }, { 24,  9 }, {  8, 10 },
        { 26, 10 }, {  3,  4 },
    },
    { /* AC bias group 4, table 14 */
        {  0,  2 }, {  1,  3 }, {  3,  4 }, {  4,  6 }, {  5, 10 }, { 14, 13 },
        { 13, 13 }, { 16, 13 }, { 15, 13 }, {  6, 13 }, {  8, 14 }, {  7, 14 },
        { 27, 14 }, { 26, 14 }, { 31, 14 }, { 30, 14 }, { 12,  9 }, { 11,  9 },
        { 22, 14 }, { 21, 14 }, { 25, 14 }, { 24, 14 }, { 18, 14 }, { 17, 14 },
        { 20, 14 }, { 19, 14 }, { 23, 11 }, { 29, 10 }, { 28,  7 }, {  2,  5 },
        { 10,  2 }, {  9,  2 },
    },
    { /* AC bias group 4, table 15 */
        { 10,  2 }, {  9,  2 }, {  0,  2 }, {  3,  4 }, { 12,  7 }, { 11,  7 },
        { 24,  9 }, { 23,  9 }, { 27,  8 }, {  5,  9 }, { 25, 10 }, {  6, 13 },
        {  8, 14 }, {  7, 14 }, { 22, 14 }, { 21, 14 }, { 31, 14 }, { 30, 14 },
        { 18, 14 }, { 17, 14 }, { 20, 14 }, { 19, 14 }, { 14, 14 }, { 13, 14 },
        { 16, 14 }, { 15, 14 }, { 26,  8 }, { 29,  5 }, {  2,  5 }, { 28,  6 },
        {  4,  6 }, {  1,  4 },
    },
};

#endif /* AVCODEC_VP3DATA_H */
