/*
Copyright (c) 2019-2023,
Lawrence Livermore National Security, LLC;
See the top-level NOTICE for additional details. All rights reserved.
SPDX-License-Identifier: BSD-3-Clause
*/
#include "units.hpp"

#include <algorithm>
#include <array>
#include <cstring>
#include <tuple>
#include <unordered_map>

namespace units {
namespace precise {
    using unitD = std::tuple<const char*, const char*, precise_unit>;
    static UNITS_CPP14_CONSTEXPR_OBJECT std::array<unitD, 2088> r20_units = {{
        unitD{"05", "lift", {1.0, count, commodities::packaging::lift}},
        unitD{
            "06",
            "small spray",
            {1.0, count, commodities::packaging::small_spray}},
        unitD{"08", "heat lot", {1.0, count, commodities::packaging::heat_lot}},
        unitD{"10", "group", {1.0, count, commodities::packaging::group}},
        unitD{"11", "outfit", {1.0, count, commodities::packaging::outfit}},
        unitD{"13", "ration", {1.0, count, commodities::packaging::ration}},
        unitD{"14", "shot", us::shot},
        unitD{
            "15",
            "stick, military",
            {1.0, count, commodities::packaging::stick}},
        unitD{
            "16",
            "hundred fifteen kg drum",
            {115.0, kg, commodities::packaging::drum}},
        unitD{
            "17",
            "hundred lb drum",
            {100.0, lb, commodities::packaging::drum}},
        unitD{
            "18",
            "fiftyfive gallon (US) drum",
            {55.0, us::gallon, commodities::packaging::drum}},
        unitD{
            "19",
            "tank truck",
            {1.0, count, commodities::packaging::tanktruck}},
        unitD{"1A", "car mile", {1.0, mile, commodities::packaging::car}},
        unitD{"1B", "car count", {1.0, count, commodities::packaging::car}},
        unitD{
            "1C",
            "locomotive count",
            {1.0, count, commodities::packaging::locomotive}},
        unitD{
            "1D",
            "caboose count",
            {1.0, count, commodities::packaging::caboose}},
        unitD{"1E", "empty car", {1.0, count, commodities::packaging::car}},
        unitD{"1F", "train mile", {1.0, mile, commodities::packaging::train}},
        unitD{"1G", "fuel usage gallon (US)", us::gallon},
        unitD{
            "1H",
            "caboose mile",
            {1.0, mile, commodities::packaging::caboose}},
        unitD{
            "1I",
            "fixed rate",
            {1.0, count, commodities::packaging::fixed_rate}},
        unitD{"1J", "ton mile", ton* mile},
        unitD{
            "1K",
            "locomotive mile",
            {1.0, mile, commodities::packaging::locomotive}},
        unitD{
            "1L",
            "total car count",
            {1.0, count, commodities::packaging::car}},
        unitD{"1M", "total car mile", {1.0, mile, commodities::packaging::car}},
        unitD{"1X", "quarter mile", {0.25, mile}},
        unitD{"20", "twenty foot container", {20.0 * 8.0 * 8.5, ft.pow(3)}},
        unitD{"21", "forty foot container", {40.0 * 8.0 * 8.5, ft.pow(3)}},
        unitD{"22", "decilitre per gram", deci* L / g},
        unitD{"23", "gram per cubic centimetre", g / cm.pow(3)},
        unitD{
            "24",
            "theoretical pound",
            {1.0, lb, commodities::packaging::theoretical}},
        unitD{"25", "gram per square centimetre", g / cm.pow(2)},
        unitD{"26", "actual ton", {1.0, ton, commodities::packaging::actual}},
        unitD{
            "27",
            "theoretical ton",
            {1.0, ton, commodities::packaging::theoretical}},
        unitD{"28", "kilogram per square metre", kg / m.pow(2)},
        unitD{
            "29",
            "pound per thousand square foot",
            lb / (thousand * ft.pow(2))},
        unitD{"2A", "radian per second", rad / s},
        unitD{"2B", "radian per second squared", rad / s.pow(2)},
        unitD{"2C", "roentgen", cgs::roentgen},
        unitD{"2G", "volt AC", V},
        unitD{"2H", "volt DC", V},
        unitD{
            "2I",
            "British thermal unit (international table) per hour",
            energy::btu_it / hr},
        unitD{"2J", "cubic centimetre per second", cm.pow(3) / s},
        unitD{"2K", "cubic foot per hour", ft.pow(3) / hr},
        unitD{"2L", "cubic foot per minute", ft.pow(3) / min},
        unitD{"2M", "centimetre per second", cm / s},
        unitD{"2N", "decibel", log::dB},
        unitD{"2P", "kilobyte", data::kB},
        unitD{"2Q", "kilobecquerel", kilo* Bq},
        unitD{"2R", "kilocurie", kilo* cgs::curie},
        unitD{"2U", "megagram", mega* g},
        unitD{"2V", "megagram per hour", mega* g / hr},
        unitD{"2W", "bin", {1.0, count, commodities::packaging::bin}},
        unitD{"2X", "metre per minute", m / min},
        unitD{"2Y", "milliroentgen", milli* cgs::roentgen},
        unitD{"2Z", "millivolt", milli* V},
        unitD{
            "30",
            "horse power day per air dry metric ton",
            hp* day / precise_unit(t, commodities::packaging::dry)},
        unitD{"31", "catch weight", pu* lb},
        unitD{
            "32",
            "kilogram per air dry metric ton",
            kg / precise_unit(t, commodities::packaging::dry)},
        unitD{"33", "kilopascal square metre per gram", kilo* Pa* m.pow(2) / g},
        unitD{"34", "kilopascal per millimetre", kilo* Pa / mm},
        unitD{
            "35",
            "millilitre per square centimetre second",
            mL / (cm.pow(2) * s)},
        unitD{
            "36",
            "cubic foot per minute per square foot",
            ft.pow(3) / min / ft.pow(2)},
        unitD{"37", "ounce per square foot", oz / ft.pow(2)},
        unitD{
            "38",
            "ounce per square foot per 0.01inch",
            oz / ft.pow(2) / precise_unit(0.01, in)},
        unitD{"3B", "megajoule", mega* J},
        unitD{"3C", "manmonth", {1.0, time::mog, commodities::people}},
        unitD{"3E", "pound per pound of product", pu* lb},
        unitD{"3G", "pound per piece of product", lb / count},
        unitD{"3H", "kilogram per kilogram of product", pu* kg},
        unitD{"3I", "kilogram per piece of product", kg / count},
        unitD{"40", "millilitre per second", mL / s},
        unitD{"41", "millilitre per minute", mL / min},
        unitD{
            "43",
            "super bulk bag",
            {1.0, count, commodities::packaging::bulk_bag}},
        unitD{
            "44",
            "fivehundred kg bulk bag",
            {500.0, kg, commodities::packaging::bulk_bag}},
        unitD{
            "45",
            "threehundred kg bulk bag",
            {300.0, kg, commodities::packaging::bulk_bag}},
        unitD{
            "46",
            "fifty lb bulk bag",
            {50.0, lb, commodities::packaging::bulk_bag}},
        unitD{"47", "fifty lb bag", {50.0, lb, commodities::packaging::bag}},
        unitD{
            "48",
            "bulk car load",
            {1.0, count, commodities::packaging::bulk_car}},
        unitD{"4A", "bobbin", {1.0, count, commodities::packaging::bobbin}},
        unitD{"4B", "cap", {1.0, count, commodities::packaging::cap}},
        unitD{"4C", "centistokes", cgs::stokes* centi},
        unitD{"4E", "twenty pack", {20, count}},
        unitD{"4G", "microlitre", micro* L},
        unitD{"4H", "micrometre", micro* m},
        unitD{"4K", "milliampere", milli* A},
        unitD{"4L", "megabyte", data::MB},
        unitD{"4M", "milligram per hour", mg / hr},
        unitD{"4N", "megabecquerel", mega* becquerel},
        unitD{"4O", "microfarad", micro* F},
        unitD{"4P", "newton per metre", N / m},
        unitD{"4Q", "ounce inch", oz* in},
        unitD{"4R", "ounce foot", oz* ft},
        unitD{"4T", "picofarad", pico* F},
        unitD{"4U", "pound per hour", lb / hr},
        unitD{"4W", "ton (US) per hour", ton / hr},
        unitD{"4X", "kilolitre per hour", kilo* L / hr},
        unitD{
            "53",
            "theoretical kilogram",
            {1.0, kg, commodities::packaging::theoretical}},
        unitD{
            "54",
            "theoretical tonne",
            {1.0, mass::tonne, commodities::packaging::theoretical}},
        unitD{"56", "sitas", {100.0, m.pow(2)}},
        unitD{"57", "mesh", clinical::mesh},
        unitD{
            "58",
            "net kilogram",
            {1.0, kg, commodities::packaging::net_quantity}},
        unitD{"59", "part per million", other::ppm},
        unitD{"5A", "barrel (US) per minute", us::barrel / min},
        unitD{"5B", "batch", {1.0, count, commodities::packaging::batch}},
        unitD{"5C", "gallon(US) per thousand", us::gallon / (kilo * count)},
        unitD{"5E", "MMSCF/day", precise::mega* precise::energy::scf / day},
        unitD{"5F", "pound per thousand", lb / (kilo * count)},
        unitD{"5G", "pump", {1.0, count, commodities::packaging::pump}},
        unitD{"5H", "stage", {1.0, count, commodities::packaging::stage}},
        unitD{"5I", "standard cubic foot", ft.pow(3)},
        unitD{"5J", "hydraulic horse power", power::hpS},
        unitD{"5K", "count per minute", count / min},
        unitD{"5P", "seismic level", special::moment_magnitude},
        unitD{"5Q", "seismic line", special::moment_energy},
        unitD{"60", "percent weight", percent* pu* kg},
        unitD{"61", "part per billion", other::ppb},
        unitD{"62", "percent per 1000 hour", percent / (thousand * hr)},
        unitD{
            "63",
            "failure rate in time",
            {1.0, count / s, commodities::failure}},
        unitD{"64", "pound per square inch, gauge", lbf / in.pow(2)},
        unitD{"66", "oersted", cgs::oersted},
        unitD{"69", "test specific scale", generate_custom_unit(333)},
        unitD{"71", "volt ampere per pound", V* A / lb},
        unitD{"72", "watt per pound", W / lb},
        unitD{
            "73",
            "ampere turn per centimetre",
            precise::A* precise::count / cm},
        unitD{"74", "millipascal", milli* Pa},
        unitD{"76", "gauss", cgs::gauss},
        unitD{"77", "milli-inch", milli* inch},
        unitD{"78", "kilogauss", kilo* cgs::gauss},
        unitD{"80", "pound per square inch absolute", lb / in.pow(2)},
        unitD{"81", "henry", henry},
        unitD{"84", "kilopound-force per square inch", kilo* lbf / in.pow(2)},
        unitD{"85", "foot pound-force", ft* lbf},
        unitD{"87", "pound per cubic foot", lb / ft.pow(3)},
        unitD{"89", "poise", cgs::poise},
        unitD{"90", "Saybold universal second", {0.2158, mm.pow(2) / s}},
        unitD{"91", "stokes", cgs::stokes},
        unitD{"92", "calorie per cubic centimetre", cal / cm.pow(3)},
        unitD{"93", "calorie per gram", cal / g},
        unitD{"94", "curl unit", generate_custom_unit(335)},
        unitD{
            "95",
            "twenty thousand gallon (US) tankcar",
            {20000.0, us::gallon, commodities::packaging::tankcar}},
        unitD{
            "96",
            "ten thousand gallon (US) tankcar",
            {10000.0, us::gallon, commodities::packaging::tankcar}},
        unitD{"97", "ten kg drum", {10.0, kg, commodities::packaging::drum}},
        unitD{
            "98",
            "fifteen kg drum",
            {15.0, kg, commodities::packaging::drum}},
        unitD{"A1", "calorie(15 degC)", energy::cal_15},
        unitD{
            "A10",
            "ampere square metre per joule second",
            A* m.pow(2) / (J * s)},
        unitD{"A11", "angstrom", distance::angstrom},
        unitD{"A12", "astronomical unit", distance::au},
        unitD{"A13", "attojoule", atto* J},
        unitD{"A14", "barn", area::barn},
        unitD{"A15", "barn per electronvolt", area::barn / energy::eV},
        unitD{
            "A16",
            "barn per steradian electronvolt",
            area::barn / (rad.pow(2) * energy::eV)},
        unitD{"A17", "barn per steradian", area::barn / rad.pow(2)},
        unitD{"A18", "becquerel per kilogram", Bq / kg},
        unitD{"A19", "becquerel per cubic metre", Bq / m.pow(3)},
        unitD{"A2", "ampere per centimetre", A / cm},
        unitD{
            "A20",
            "British thermal unit (international table) per second square foot degree Rankine",
            energy::btu_it / (s * ft.pow(2) * temperature::degR)},
        unitD{
            "A21",
            "British thermal unit (international table) per pound degree Rankine",
            energy::btu_it / (lb * temperature::degR)},
        unitD{
            "A22",
            "British thermal unit (international table) per second foot degree Rankine",
            energy::btu_it / (s * ft * temperature::degR)},
        unitD{
            "A23",
            "British thermal unit (international table) per hour square foot degree Rankine",
            energy::btu_it / (hr * ft.pow(2) * temperature::degR)},
        unitD{"A24", "candela per square metre", cd / m.pow(2)},
        unitD{"A25", "cheval vapeur", power::hpM},
        unitD{"A26", "coulomb metre", C* m},
        unitD{"A27", "coulomb metre squared per volt", C* m.pow(2) / V},
        unitD{"A28", "coulomb per cubic centimetre", C / cm.pow(3)},
        unitD{"A29", "coulomb per cubic metre", C / m.pow(3)},
        unitD{"A3", "ampere per millimetre", A / mm},
        unitD{"A30", "coulomb per cubic millimetre", C / mm.pow(3)},
        unitD{"A31", "coulomb per kilogram second", C / (kg * s)},
        unitD{"A32", "coulomb per mole", C / mol},
        unitD{"A33", "coulomb per square centimetre", C / cm.pow(2)},
        unitD{"A34", "coulomb per square metre", C / m.pow(2)},
        unitD{"A35", "coulomb per square millimetre", C / mm.pow(2)},
        unitD{"A36", "cubic centimetre per mole", cm.pow(3) / mol},
        unitD{"A37", "cubic decimetre per mole", (deci * m).pow(3) / mol},
        unitD{"A38", "cubic metre per coulomb", m.pow(3) / C},
        unitD{"A39", "cubic metre per kilogram", m.pow(3) / kg},
        unitD{"A4", "ampere per square centimetre", A / cm.pow(2)},
        unitD{"A40", "cubic metre per mole", m.pow(3) / mol},
        unitD{"A41", "ampere per square metre", A / m.pow(2)},
        unitD{"A42", "curie per kilogram", cgs::curie / kg},
        unitD{
            "A43",
            "deadweight tonnage",
            {1.0, mass::tonne, commodities::packaging::deadweight}},
        unitD{"A44", "decalitre", deca* L},
        unitD{"A45", "decametre", deca* m},
        unitD{"A47", "decitex", {0.1, textile::tex}},
        unitD{"A48", "degree Rankine", temperature::degR},
        unitD{"A49", "denier", textile::denier},
        unitD{"A5", "ampere square metre", A* m.pow(2)},
        unitD{
            "A50",
            "dyne second per cubic centimetre",
            cgs::dyn* s / cm.pow(3)},
        unitD{"A51", "dyne second per centimetre", cgs::dyn* s / cm},
        unitD{
            "A52",
            "dyne second per centimetre to the fifth power",
            cgs::dyn* s / cm.pow(5)},
        unitD{"A53", "electronvolt", energy::eV},
        unitD{"A54", "electronvolt per metre", energy::eV / m},
        unitD{"A55", "electronvolt square metre", energy::eV* m.pow(2)},
        unitD{
            "A56",
            "electronvolt square metre per kilogram",
            energy::eV* m.pow(2) / kg},
        unitD{"A57", "erg", cgs::erg},
        unitD{"A58", "erg per centimetre", cgs::erg / cm},
        unitD{
            "A59",
            "8-part cloud cover",
            {constants::pi / 4, sr, commodities::cloud_cover}},
        unitD{
            "A6",
            "ampere per square metre kelvin squared",
            A / (m.pow(2) * K.pow(2))},
        unitD{"A60", "erg per cubic centimetre", cgs::erg / cm.pow(3)},
        unitD{"A61", "erg per gram", cgs::erg / g},
        unitD{"A62", "erg per gram second", cgs::erg / (g * s)},
        unitD{"A63", "erg per second", cgs::erg / s},
        unitD{
            "A64",
            "erg per second square centimetre",
            cgs::erg / (cm.pow(2) * s)},
        unitD{
            "A65",
            "erg per square centimetre second",
            cgs::erg / (cm.pow(2) * s)},
        unitD{"A66", "erg square centimetre", cgs::erg* cm.pow(2)},
        unitD{"A67", "erg square centimetre per gram", cgs::erg* cm.pow(2) / g},
        unitD{"A68", "exajoule", exa* J},
        unitD{"A69", "farad per metre", F / m},
        unitD{"A7", "ampere per square millimetre", A / mm.pow(2)},
        unitD{"A70", "femtojoule", femto* J},
        unitD{"A71", "femtometre", femto* m},
        unitD{"A73", "foot per second squared", ft / s.pow(2)},
        unitD{"A74", "foot pound-force per second", ft* lbf / s},
        unitD{
            "A75",
            "freight ton",
            {1.0, ton, commodities::packaging::freight}},
        unitD{"A76", "Gal", cgs::gal},
        unitD{
            "A77",
            "Gaussian CGS (Centimetre-Gram-Second system) unit of displacement",
            cgs::statC_charge / cm.pow(2)},
        unitD{
            "A78",
            "Gaussian CGS (Centimetre-Gram-Second system) unit of electric current",
            cgs::statC_flux},
        unitD{
            "A79",
            "Gaussian CGS (Centimetre-Gram-Second system) unit of electric charge",
            cgs::statC_charge},
        unitD{"A8", "ampere second", A* s},
        unitD{
            "A80",
            "Gaussian CGS (Centimetre-Gram-Second system) unit of electric field strength",
            cgs::statV / cm},
        unitD{
            "A81",
            "Gaussian CGS (Centimetre-Gram-Second system) unit of electric polarization",
            cgs::statC_charge* cm},
        unitD{
            "A82",
            "Gaussian CGS (Centimetre-Gram-Second system) unit of electric potential",
            cgs::statV},
        unitD{
            "A83",
            "Gaussian CGS (Centimetre-Gram-Second system) unit of magnetization",
            cgs::statT},
        unitD{"A84", "gigacoulomb per cubic metre", giga* C / m.pow(3)},
        unitD{"A85", "gigaelectronvolt", giga* energy::eV},
        unitD{"A86", "gigahertz", giga* Hz},
        unitD{"A87", "gigaohm", giga* ohm},
        unitD{"A88", "gigaohm metre", giga* ohm* m},
        unitD{"A89", "gigapascal", giga* Pa},
        unitD{"A9", "rate", count / s},
        unitD{"A90", "gigawatt", giga* W},
        unitD{"A91", "gon", angle::gon},
        unitD{"A93", "gram per cubic metre", g / m.pow(3)},
        unitD{"A94", "gram per mole", g / mol},
        unitD{"A95", "gray", gray},
        unitD{"A96", "gray per second", gray / s},
        unitD{"A97", "hectopascal", hecto* Pa},
        unitD{"A98", "henry per metre", H / m},
        unitD{"A99", "bit", data::bit},
        unitD{"AA", "ball", {1.0, count, commodities::packaging::ball}},
        unitD{
            "AB",
            "bulk pack",
            {1.0, count, commodities::packaging::bulk_pack}},
        unitD{"ACR", "acre", acre},
        unitD{"ACT", "activity", count / s},
        unitD{"AD", "byte", data::byte},
        unitD{"AE", "ampere per metre", A / m},
        unitD{"AH", "additional minute", min},
        unitD{
            "AI",
            "average minute per call",
            min / precise_unit(one, commodities::call)},
        unitD{"AJ", "cop", {1.0, count, commodities::packaging::cop}},
        unitD{"AK", "fathom", nautical::fathom},
        unitD{"AL", "access line", {1.0, count, commodities::linesofservice}},
        unitD{"AM", "ampoule", {1.0, count, commodities::packaging::ampoule}},
        unitD{"AMH", "ampere hour", A* hr},
        unitD{"AMP", "ampere", A},
        unitD{"ANN", "year", yr},
        unitD{"AP", "aluminium pound only", {1.0, lb, commodities::aluminum}},
        unitD{"APZ", "troy ounce", troy::oz},
        unitD{"AQ", "anti-hemophilic factor (AHF) unit", clinical::AHF},
        unitD{
            "AR",
            "suppository",
            {1.0, count, commodities::packaging::suppository}},
        unitD{"ARE", "are", area::are},
        unitD{
            "AS",
            "assortment",
            {1.0, count, commodities::packaging::assortment}},
        unitD{
            "ASM",
            "alcoholic strength by mass",
            {1.0, pu* kg, commodities::ethanol}},
        unitD{
            "ASU",
            "alcoholic strength by volume",
            {1.0, pu* L, commodities::ethanol}},
        unitD{"ATM", "standard atmosphere", pressure::atm},
        unitD{"ATT", "technical atmosphere", pressure::att},
        unitD{"AV", "capsule", {1.0, count, commodities::packaging::capsule}},
        unitD{
            "AW",
            "powder filled vial",
            {1.0, count, commodities::packaging::powder_vial}},
        unitD{"AY", "assembly", {1.0, count, commodities::packaging::assembly}},
        unitD{
            "AZ",
            "British thermal unit (international table) per pound",
            energy::btu_it / lb},
        unitD{"B0", "Btu per cubic foot", btu / ft.pow(3)},
        unitD{"B1", "barrel (US) per day", us::barrel / day},
        unitD{"B10", "bit per second", data::bit / s},
        unitD{"B11", "joule per kilogram kelvin", J / (kg * K)},
        unitD{"B12", "joule per metre", J / m},
        unitD{"B13", "joule per square metre", J / m.pow(2)},
        unitD{"B14", "joule per metre to the fourth power", J / m.pow(4)},
        unitD{"B15", "joule per mole", J / mol},
        unitD{"B16", "joule per mole kelvin", J / (mol * K)},
        unitD{"B17", "credit", {1.0, count, commodities::credit}},
        unitD{"B18", "joule second", J* s},
        unitD{"B19", "digit", data::digits},
        unitD{"B2", "bunk", {1.0, count, commodities::packaging::bunk}},
        unitD{"B20", "joule square metre per kilogram", J* m.pow(2) / kg},
        unitD{"B21", "kelvin per watt", K / W},
        unitD{"B22", "kiloampere", kilo* A},
        unitD{"B23", "kiloampere per square metre", kilo* A / m.pow(2)},
        unitD{"B24", "kiloampere per metre", kilo* A / m},
        unitD{"B25", "kilobecquerel per kilogram", kilo* Bq / kg},
        unitD{"B26", "kilocoulomb", kilo* C},
        unitD{"B27", "kilocoulomb per cubic metre", kilo* C / m.pow(3)},
        unitD{"B28", "kilocoulomb per square metre", kilo* C / m.pow(2)},
        unitD{"B29", "kiloelectronvolt", kilo* energy::eV},
        unitD{"B3", "batting pound", {1.0, lb, commodities::fiber}},
        unitD{"B30", "gibibit", gibi* data::bit},
        unitD{"B31", "kilogram metre per second", kg* m / s},
        unitD{"B32", "kilogram metre squared", kg* m.pow(2)},
        unitD{"B33", "kilogram metre squared per second", kg* m.pow(2) / s},
        unitD{"B34", "kilogram per cubic decimetre", kg / (deci * m).pow(3)},
        unitD{"B35", "kilogram per litre", kg / L},
        unitD{"B36", "calorie (thermochemical) per gram", energy::cal_th / g},
        unitD{"B37", "kilogram-force", kg* constants::g0.as_unit()},
        unitD{"B38", "kilogram-force metre", kg* constants::g0.as_unit() * m},
        unitD{
            "B39",
            "kilogram-force metre per second",
            kg* constants::g0.as_unit() * m / s},
        unitD{"B4", "barrel, imperial", imp::winebarrel},
        unitD{
            "B40",
            "kilogram-force per square metre",
            kg* constants::g0.as_unit() / m.pow(2)},
        unitD{"B41", "kilojoule per kelvin", kilo* J / K},
        unitD{"B42", "kilojoule per kilogram", kilo* J / kg},
        unitD{"B43", "kilojoule per kilogram kelvin", kilo* J / (kg * K)},
        unitD{"B44", "kilojoule per mole", kilo* J / mol},
        unitD{"B45", "kilomole", kilo* mol},
        unitD{"B46", "kilomole per cubic metre", kilo* mol / m.pow(3)},
        unitD{"B47", "kilonewton", kilo* N},
        unitD{"B48", "kilonewton metre", kilo* N* m},
        unitD{"B49", "kiloohm", kilo* ohm},
        unitD{"B5", "billet", {1.0, count, commodities::packaging::billet}},
        unitD{"B50", "kiloohm metre", kilo* ohm* m},
        unitD{"B51", "kilopond", kilo* gm::pond},
        unitD{"B52", "kilosecond", kilo* s},
        unitD{"B53", "kilosiemens", kilo* S},
        unitD{"B54", "kilosiemens per metre", kilo* S / m},
        unitD{"B55", "kilovolt per metre", kilo* V / m},
        unitD{"B56", "kiloweber per metre", kilo* weber / m},
        unitD{"B57", "light year", distance::ly},
        unitD{"B58", "litre per mole", L / mol},
        unitD{"B59", "lumen hour", lm* hr},
        unitD{"B6", "bun", milli* mol / L},
        unitD{"B60", "lumen per square metre", lm / m.pow(2)},
        unitD{"B61", "lumen per watt", lm / W},
        unitD{"B62", "lumen second", lm* s},
        unitD{"B63", "lux hour", lux* hr},
        unitD{"B64", "lux second", lux* s},
        unitD{"B65", "maxwell", cgs::maxwell},
        unitD{"B66", "megaampere per square metre", mega* A / m.pow(2)},
        unitD{"B67", "megabecquerel per kilogram", mega* Bq / kg},
        unitD{"B68", "gigabit", giga* data::bit},
        unitD{"B69", "megacoulomb per cubic metre", mega* C / m.pow(3)},
        unitD{"B7", "cycle", {constants::tau, rad}},
        unitD{"B70", "megacoulomb per square metre", mega* C / m.pow(2)},
        unitD{"B71", "megaelectronvolt", mega* energy::eV},
        unitD{"B72", "megagram per cubic metre", mega* g / m.pow(3)},
        unitD{"B73", "meganewton", mega* N},
        unitD{"B74", "meganewton metre", mega* N* m},
        unitD{"B75", "megaohm", mega* ohm},
        unitD{"B76", "megaohm metre", mega* ohm* m},
        unitD{"B77", "megasiemens per metre", mega* S / m},
        unitD{"B78", "megavolt", mega* V},
        unitD{"B79", "megavolt per metre", mega* V / m},
        unitD{"B8", "joule per cubic metre", J / m.pow(3)},
        unitD{"B80", "gigabit per second", giga* data::bit / s},
        unitD{
            "B81",
            "reciprocal metre squared reciprocal second",
            m.pow(-2) * s.inv()},
        unitD{
            "B82",
            "inch per linear foot",
            in / precise_unit(1.0, ft, commodities::packaging::linear)},
        unitD{"B83", "metre to the fourth power", m.pow(4)},
        unitD{"B84", "microampere", micro* A},
        unitD{"B85", "microbar", micro* bar},
        unitD{"B86", "microcoulomb", micro* C},
        unitD{"B87", "microcoulomb per cubic metre", micro* C / m.pow(3)},
        unitD{"B88", "microcoulomb per square metre", micro* C / m.pow(2)},
        unitD{"B89", "microfarad per metre", micro* F / m},
        unitD{"B9", "batt", {1.0, count, commodities::packaging::batt}},
        unitD{"B90", "microhenry", micro* H},
        unitD{"B91", "microhenry per metre", micro* H / m},
        unitD{"B92", "micronewton", micro* N},
        unitD{"B93", "micronewton metre", micro* N* m},
        unitD{"B94", "microohm", micro* ohm},
        unitD{"B95", "microohm metre", micro* ohm* m},
        unitD{"B96", "micropascal", micro* Pa},
        unitD{"B97", "microradian", micro* rad},
        unitD{"B98", "microsecond", micro* s},
        unitD{"B99", "microsiemens", micro* S},
        unitD{"BAR", "bar [unit of pressure]", bar},
        unitD{"BB", "base box", {1.0, count, commodities::packaging::base_box}},
        unitD{"BD", "board", {1.0, count, commodities::packaging::board}},
        unitD{"BE", "bundle", {1.0, count, commodities::packaging::bundle}},
        unitD{"BFT", "board foot", i::board_foot},
        unitD{"BG", "bag", {1.0, count, commodities::packaging::bag}},
        unitD{"BH", "brush", {1.0, count, commodities::packaging::brush}},
        unitD{"BHP", "brake horse power", power::hpM},
        unitD{"BIL", "billion (EUR)", tera* count},
        unitD{"BJ", "bucket", {1.0, count, commodities::packaging::bucket}},
        unitD{"BK", "basket", {1.0, count, commodities::packaging::basket}},
        unitD{"BL", "bale", {1.0, count, commodities::packaging::bale}},
        unitD{"BLD", "dry barrel (US)", us::dry::barrel},
        unitD{"BLL", "barrel (US)", us::barrel},
        unitD{"BO", "bottle", {1.0, count, commodities::packaging::bottle}},
        unitD{"BP", "hundred board foot", hundred* i::board_foot},
        unitD{"BQL", "becquerel", Bq},
        unitD{
            "BR",
            "bar [unit of packaging]",
            {1.0, count, commodities::packaging::bar}},
        unitD{"BT", "bolt", {1.0, count, commodities::packaging::bolt}},
        unitD{
            "BTU",
            "British thermal unit (international table)",
            energy::btu_it},
        unitD{"BUA", "bushel (US)", us::dry::bushel},
        unitD{"BUI", "bushel (UK)", imp::bushel},
        unitD{"BW", "base weight", kg},
        unitD{"BX", "box", {1.0, count, commodities::packaging::box}},
        unitD{"BZ", "million BTUs", mega* btu},
        unitD{"C0", "call", {1.0, count, commodities::call}},
        unitD{
            "C1",
            "composite product pound (total weight)",
            {1.0, lb, commodities::packaging::composite}},
        unitD{"C10", "millifarad", milli* F},
        unitD{"C11", "milliGal", milli* cgs::gal},
        unitD{"C12", "milligram per metre", mg / m},
        unitD{"C13", "milligray", milli* gray},
        unitD{"C14", "millihenry", milli* H},
        unitD{"C15", "millijoule", milli* J},
        unitD{"C16", "millimetre per second", mm / s},
        unitD{"C17", "millimetre squared per second", mm.pow(2) / s},
        unitD{"C18", "millimole", milli* mol},
        unitD{"C19", "mole per kilogram", mol / kg},
        unitD{"C2", "carset", {1.0, count, commodities::packaging::carset}},
        unitD{"C20", "millinewton", milli* N},
        unitD{"C21", "kibibit", kibi* data::bit},
        unitD{"C22", "millinewton per metre", milli* N / m},
        unitD{"C23", "milliohm metre", milli* ohm* m},
        unitD{"C24", "millipascal second", milli* Pa* s},
        unitD{"C25", "milliradian", milli* rad},
        unitD{"C26", "millisecond", milli* s},
        unitD{"C27", "millisiemens", milli* S},
        unitD{"C28", "millisievert", milli* sievert},
        unitD{"C29", "millitesla", milli* T},
        unitD{"C3", "microvolt per metre", micro* V / m},
        unitD{"C30", "millivolt per metre", milli* V / m},
        unitD{"C31", "milliwatt", milli* watt},
        unitD{"C32", "milliwatt per square metre", milli* W / m.pow(2)},
        unitD{"C33", "milliweber", milli* weber},
        unitD{"C34", "mole", mol},
        unitD{"C35", "mole per cubic decimetre", mol / (deci * m).pow(3)},
        unitD{"C36", "mole per cubic metre", mol / m.pow(3)},
        unitD{"C37", "kilobit", kilo* data::bit},
        unitD{"C38", "mole per litre", mol / L},
        unitD{"C39", "nanoampere", nano* A},
        unitD{"C4", "carload", {1.0, count, commodities::packaging::car}},
        unitD{"C40", "nanocoulomb", nano* C},
        unitD{"C41", "nanofarad", nano* F},
        unitD{"C42", "nanofarad per metre", nano* F / m},
        unitD{"C43", "nanohenry", nano* H},
        unitD{"C44", "nanohenry per metre", nano* H / m},
        unitD{"C45", "nanometre", nano* m},
        unitD{"C46", "nanoohm metre", nano* ohm* m},
        unitD{"C47", "nanosecond", nano* s},
        unitD{"C48", "nanotesla", nano* T},
        unitD{"C49", "nanowatt", nano* W},
        unitD{"C5", "cost", {1.0, currency, commodities::packaging::actual}},
        unitD{"C50", "neper", log::neper},
        unitD{"C51", "neper per second", log::neper / s},
        unitD{"C52", "picometre", pico* m},
        unitD{"C53", "newton metre second", N* m* s},
        unitD{
            "C54",
            "newton metre squared per kilogram squared",
            N* m.pow(2) / kg.pow(2)},
        unitD{"C55", "newton per square metre", N / m.pow(2)},
        unitD{"C56", "newton per square millimetre", N / mm.pow(2)},
        unitD{"C57", "newton second", N* s},
        unitD{"C58", "newton second per metre", N* s / m},
        unitD{"C59", "octave", log::logbase2},
        unitD{"C6", "cell", {1.0, count, commodities::packaging::cell}},
        unitD{"C60", "ohm centimetre", ohm* cm},
        unitD{"C61", "ohm metre", ohm* m},
        unitD{"C62", "one", count},
        unitD{"C63", "parsec", distance::parsec},
        unitD{"C64", "pascal per kelvin", Pa / K},
        unitD{"C65", "pascal second", Pa* s},
        unitD{"C66", "pascal second per cubic metre", Pa* s / m.pow(3)},
        unitD{"C67", "pascal second per metre", Pa* s / m},
        unitD{"C68", "petajoule", peta* J},
        unitD{"C69", "phon", generate_custom_unit(41)},
        unitD{"C7", "centipoise", centi* cgs::poise},
        unitD{"C70", "picoampere", pico* A},
        unitD{"C71", "picocoulomb", pico* C},
        unitD{"C72", "picofarad per metre", pico* F / m},
        unitD{"C73", "picohenry", pico* H},
        unitD{"C74", "kilobit per second", kilo* data::bit / s},
        unitD{"C75", "picowatt", pico* W},
        unitD{"C76", "picowatt per square metre", pico* W / m.pow(2)},
        unitD{"C77", "pound gage", lb* eflag},
        unitD{"C78", "pound-force", lbf},
        unitD{"C79", "kilovolt ampere hour", kilo* V* A* hr},
        unitD{"C8", "millicoulomb per kilogram", milli* C / kg},
        unitD{"C80", "RAD", cgs::RAD},
        unitD{"C81", "radian", rad},
        unitD{"C82", "radian square metre per mole", rad* m.pow(2) / mol},
        unitD{"C83", "radian square metre per kilogram", rad* m.pow(2) / kg},
        unitD{"C84", "radian per metre", rad / m},
        unitD{"C85", "reciprocal angstrom", distance::angstrom.inv()},
        unitD{"C86", "reciprocal cubic metre", m.pow(-3)},
        unitD{"C87", "reciprocal cubic metre per second", m.pow(-3) / s},
        unitD{
            "C88",
            "reciprocal electron volt per cubic metre",
            energy::eV.inv() / m.pow(3)},
        unitD{"C89", "reciprocal henry", H.inv()},
        unitD{
            "C9",
            "coil group",
            {1.0, count, commodities::packaging::coil_group}},
        unitD{"C90", "reciprocal joule per cubic metre", J.inv() / m.pow(3)},
        unitD{"C91", "reciprocal kelvin", K.inv()},
        unitD{"C92", "reciprocal metre", m.inv()},
        unitD{"C93", "reciprocal square metre", m.pow(-2)},
        unitD{"C94", "reciprocal minute", min.inv()},
        unitD{"C95", "reciprocal mole", mol.inv()},
        unitD{"C96", "reciprocal pascal", Pa.inv()},
        unitD{"C97", "reciprocal second", s.inv()},
        unitD{"C98", "reciprocal second per cubic metre", s.inv() / m.pow(3)},
        unitD{"C99", "reciprocal second per metre squared", s.inv() / m.pow(2)},
        unitD{"CA", "can", {1.0, count, commodities::packaging::can}},
        unitD{
            "CCT",
            "carrying capacity in metric ton",
            {1.0, t, commodities::packaging::capacity}},
        unitD{"CDL", "candela", cd},
        unitD{"CEL", "degree Celsius", degC},
        unitD{"CEN", "hundred", hundred* count},
        unitD{"CG", "card", {1.0, count, commodities::packaging::card}},
        unitD{"CGM", "centigram", centi* g},
        unitD{
            "CH",
            "container",
            {1.0, count, commodities::packaging::container}},
        unitD{"CJ", "cone", {1.0, count, commodities::packaging::cone}},
        unitD{
            "CK",
            "connector",
            {1.0, count, commodities::packaging::connector}},
        unitD{"CKG", "coulomb per kilogram", C / kg},
        unitD{"CL", "coil", {1.0, count, commodities::packaging::coil}},
        unitD{
            "CLF",
            "hundred leave",
            {100.0, count, commodities::packaging::leave}},
        unitD{"CLT", "centilitre", centi* L},
        unitD{"CMK", "square centimetre", cm.pow(2)},
        unitD{"CMQ", "cubic centimetre", cm.pow(3)},
        unitD{"CMT", "centimetre", cm},
        unitD{"CNP", "hundred pack", hundred* count},
        unitD{"CNT", "cental (UK)", imp::hundredweight},
        unitD{"CO", "carboy", {1.0, count, commodities::packaging::carboy}},
        unitD{"COU", "coulomb", C},
        unitD{
            "CQ",
            "cartridge",
            {1.0, count, commodities::packaging::cartridge}},
        unitD{"CR", "crate", {1.0, count, commodities::packaging::crate}},
        unitD{"CS", "case", {1.0, count, commodities::packaging::small_case}},
        unitD{"CT", "carton", {1.0, count, commodities::packaging::carton}},
        unitD{"CTG", "content gram", g},
        unitD{"CTM", "metric carat", metric::carat},
        unitD{"CTN", "content ton (metric)", t},
        unitD{"CU", "cup", us::cup},
        unitD{"CUR", "curie", cgs::curie},
        unitD{"CV", "cover", {1.0, count, commodities::packaging::cover}},
        unitD{
            "CWA",
            "hundred pound (cwt) / hundred weight (US)",
            av::hundredweight},
        unitD{"CWI", "hundred weight (UK)", av::longhundredweight},
        unitD{"CY", "cylinder", {1.0, count, commodities::packaging::cylinder}},
        unitD{"CZ", "combo", {1.0, count, commodities::packaging::combo}},
        unitD{"D03", "kilowatt hour per hour", energy::kWh / hr},
        unitD{
            "D04",
            "lot  [unit of weight]",
            {1.0, count, commodities::packaging::lot}},
        unitD{"D1", "reciprocal second per steradian", s.inv() / rad.pow(2)},
        unitD{"D10", "siemens per metre", S / m},
        unitD{"D11", "mebibit", mebi* data::bit},
        unitD{"D12", "siemens square metre per mole", S* m.pow(2) / mol},
        unitD{"D13", "sievert", sievert},
        unitD{
            "D14",
            "thousand linear yard",
            {1000.0, yd, commodities::packaging::linear}},
        unitD{"D15", "sone", precise::special::sone},
        unitD{"D16", "square centimetre per erg", cm.pow(2) / cgs::erg},
        unitD{
            "D17",
            "square centimetre per steradian erg",
            cm.pow(2) / (rad.pow(2) * cgs::erg)},
        unitD{"D18", "metre kelvin", m* K},
        unitD{"D19", "square metre kelvin per watt", m.pow(2) * K / W},
        unitD{
            "D2",
            "reciprocal second per steradian metre squared",
            s.inv() / (rad.pow(2) * m.pow(2))},
        unitD{"D20", "square metre per joule", m.pow(2) / J},
        unitD{"D21", "square metre per kilogram", m.pow(2) / kg},
        unitD{"D22", "square metre per mole", m.pow(2) / mol},
        unitD{
            "D23",
            "pen gram (protein)",
            precise_unit(g, commodities::protein)},
        unitD{"D24", "square metre per steradian", m.pow(2) / rad.pow(2)},
        unitD{
            "D25",
            "square metre per steradian joule",
            m.pow(2) / (J * rad.pow(2))},
        unitD{"D26", "square metre per volt second", m.pow(2) / (V * s)},
        unitD{"D27", "steradian", rad.pow(2)},
        unitD{"D28", "syphon", {1.0, count, commodities::packaging::syphon}},
        unitD{"D29", "terahertz", tera* Hz},
        unitD{"D30", "terajoule", tera* J},
        unitD{"D31", "terawatt", tera* W},
        unitD{"D32", "terawatt hour", tera* W* hr},
        unitD{"D33", "tesla", tesla},
        unitD{"D34", "tex", textile::tex},
        unitD{"D35", "calorie (thermochemical)", energy::cal_th},
        unitD{"D36", "megabit", mega* data::bit},
        unitD{
            "D37",
            "calorie (thermochemical) per gram kelvin",
            energy::cal_th / (g * K)},
        unitD{
            "D38",
            "calorie (thermochemical) per second centimetre kelvin",
            energy::cal_th / (s * cm * K)},
        unitD{
            "D39",
            "calorie (thermochemical) per second square centimetre kelvin",
            energy::cal_th / (s * cm.pow(2) * K)},
        unitD{"D40", "thousand litre", thousand* L},
        unitD{"D41", "tonne per cubic metre", t / m.pow(3)},
        unitD{"D42", "tropical year", time::at},
        unitD{"D43", "unified atomic mass unit", mass::u},
        unitD{"D44", "var", electrical::VAR},
        unitD{"D45", "volt squared per kelvin squared", V.pow(2) / K.pow(2)},
        unitD{"D46", "volt - ampere", V* A},
        unitD{"D47", "volt per centimetre", V / cm},
        unitD{"D48", "volt per kelvin", V / K},
        unitD{"D49", "millivolt per kelvin", milli* V / K},
        unitD{"D5", "kilogram per square centimetre", kg / cm.pow(2)},
        unitD{"D50", "volt per metre", V / m},
        unitD{"D51", "volt per millimetre", V / mm},
        unitD{"D52", "watt per kelvin", W / K},
        unitD{"D53", "watt per metre kelvin", W / (m * K)},
        unitD{"D54", "watt per square metre", W / m.pow(2)},
        unitD{"D55", "watt per square metre kelvin", W / (m.pow(2) * K)},
        unitD{
            "D56",
            "watt per square metre kelvin to the fourth power",
            W* K.pow(-4) / m.pow(2)},
        unitD{"D57", "watt per steradian", W / rad.pow(2)},
        unitD{
            "D58",
            "watt per steradian square metre",
            W / (rad.pow(2) * m.pow(2))},
        unitD{"D59", "weber per metre", Wb / m},
        unitD{"D6", "roentgen per second", cgs::roentgen / s},
        unitD{"D60", "weber per millimetre", Wb / mm},
        unitD{"D61", "minute [unit of angle]", angle::arcmin},
        unitD{"D62", "second [unit of angle]", angle::arcsec},
        unitD{"D63", "book", {1.0, count, commodities::packaging::book}},
        unitD{"D64", "block", {1.0, count, commodities::packaging::block}},
        unitD{"D65", "round", {1.0, count, commodities::packaging::round}},
        unitD{
            "D66",
            "cassette",
            {1.0, count, commodities::packaging::cassette}},
        unitD{"D67", "dollar per hour", currency / hr},
        unitD{"D68", "number of words", {1.0, count, commodities::word}},
        unitD{"D69", "inch to the fourth power", in.pow(4)},
        unitD{
            "D7",
            "sandwich",
            {1.0,
             count,
             commodities::generateHarmonizedCode(19, 05, 90, 80, false)}},
        unitD{"D70", "calorie (international table)", energy::cal_it},
        unitD{
            "D71",
            "calorie (international table) per second centimetre kelvin",
            energy::cal_it / (s * cm * K)},
        unitD{
            "D72",
            "calorie (international table) per second square centimetre kelvin",
            energy::cal_it / (s * cm.pow(2) * K)},
        unitD{"D73", "joule square metre", J* m.pow(2)},
        unitD{"D74", "kilogram per mole", kg / mol},
        unitD{
            "D75",
            "calorie (international table) per gram",
            energy::cal_it / g},
        unitD{
            "D76",
            "calorie (international table) per gram kelvin",
            energy::cal_it / (g * K)},
        unitD{"D77", "megacoulomb", mega* C},
        unitD{"D78", "megajoule per second", mega* J / s},
        unitD{"D79", "beam", {1.0, count, commodities::packaging::beam}},
        unitD{"D8", "draize score", generate_custom_unit(341)},
        unitD{"D80", "microwatt", micro* W},
        unitD{"D81", "microtesla", micro* T},
        unitD{"D82", "microvolt", micro* V},
        unitD{"D83", "millinewton metre", milli* N* m},
        unitD{"D85", "microwatt per square metre", micro* W / m.pow(2)},
        unitD{"D86", "millicoulomb", milli* C},
        unitD{"D87", "millimole per kilogram", milli* mol / kg},
        unitD{"D88", "millicoulomb per cubic metre", milli* C / m.pow(3)},
        unitD{"D89", "millicoulomb per square metre", milli* C / m.pow(2)},
        unitD{"D9", "dyne per square centimetre", cgs::dyn / cm.pow(2)},
        unitD{"D90", "cubic metre (net)", m.pow(3)},
        unitD{"D91", "rem", cgs::REM},
        unitD{"D92", "band", {1.0, count, commodities::packaging::band}},
        unitD{"D93", "second per cubic metre", s / m.pow(3)},
        unitD{"D94", "second per cubic metre radian", s / (m.pow(3) * rad)},
        unitD{"D95", "joule per gram", J / g},
        unitD{"D96", "pound gross", lb},
        unitD{
            "D97",
            "pallet/unit load",
            {1.0, count, commodities::packaging::pallet}},
        unitD{"D98", "mass pound", lb},
        unitD{"D99", "sleeve", {1.0, count, commodities::packaging::sleeve}},
        unitD{"DAA", "decare", ten* area::are},
        unitD{"DAD", "ten day", ten* day},
        unitD{"DAY", "day", day},
        unitD{"DB", "dry pound", {1.0, lb, commodities::packaging::dry}},
        unitD{"DC", "disk (disc)", {1.0, count, commodities::packaging::disk}},
        unitD{"DD", "degree", deg},
        unitD{"DE", "deal", {1.0, count, commodities::packaging::deal}},
        unitD{"DEC", "decade", ten* yr},
        unitD{"DG", "decigram", deci* g},
        unitD{
            "DI",
            "dispenser",
            {1.0, count, commodities::packaging::dispenser}},
        unitD{"DJ", "decagram", deca* g},
        unitD{"DLT", "decilitre", deci* L},
        unitD{"DMA", "cubic decametre", (deca * m).pow(3)},
        unitD{"DMK", "square decimetre", (deci * m).pow(2)},
        unitD{"DMO", "standard kilolitre", kilo* L},
        unitD{"DMQ", "cubic decimetre", (deci * m).pow(3)},
        unitD{"DMT", "decimetre", deci* m},
        unitD{"DN", "decinewton metre", deci* N* m},
        unitD{
            "DPC",
            "dozen piece",
            {12.0, count, commodities::packaging::piece}},
        unitD{"DPR", "dozen pair", precise_unit(24.0, count)},
        unitD{
            "DPT",
            "displacement tonnage",
            {1.0, ton, commodities::packaging::displacement}},
        unitD{"DQ", "data record", {1.0, count, commodities::record}},
        unitD{"DR", "drum", {1.0, count, commodities::packaging::drum}},
        unitD{"DRA", "dram (US)", apothecaries::drachm},
        unitD{"DRI", "dram mass (UK)", av::dram},
        unitD{"DRL", "dozen roll", {12.0, count, commodities::packaging::roll}},
        unitD{"DRM", "drachm (UK)", imp::drachm},
        unitD{"DS", "display", {1.0, count, commodities::packaging::display}},
        unitD{"DT", "dry ton", {1.0, ton, commodities::packaging::dry}},
        unitD{"DTN", "decitonne", deci* t},
        unitD{"DU", "dyne", cgs::dyn},
        unitD{"DWT", "pennyweight", troy::pennyweight},
        unitD{"DX", "dyne per centimetre", cgs::dyn / cm},
        unitD{
            "DY",
            "directory book",
            {1.0, count, commodities::packaging::book}},
        unitD{"DZN", "dozen", {12, count}},
        unitD{"DZP", "dozen pack", {12.0, count, commodities::packaging::pack}},
        unitD{"E01", "newton per square centimetre", N / cm.pow(2)},
        unitD{"E07", "megawatt hour per hour", electrical::MW* hr / hr},
        unitD{"E08", "megawatt per hertz", electrical::MW / Hz},
        unitD{"E09", "milliampere hour", milli* A* hr},
        unitD{"E10", "degree day", deg* day},
        unitD{"E11", "gigacalorie", giga* energy::cal_it},
        unitD{"E12", "mille", {1000.0, count, commodities::cigarette}},
        unitD{"E14", "kilocalorie (international table)", kilo* energy::cal_it},
        unitD{
            "E15",
            "kilocalorie (thermochemical) per hour",
            kilo* energy::cal_th / hr},
        unitD{"E16", "million Btu(IT) per hour", mega* energy::btu_it / hr},
        unitD{"E17", "cubic foot per second", ft.pow(3) / s},
        unitD{"E18", "tonne per hour", t / hr},
        unitD{"E19", "ping", {3.305, m.pow(2)}},
        unitD{"E2", "belt", {1.0, count, commodities::packaging::belt}},
        unitD{"E20", "megabit per second", mega* data::bit / s},
        unitD{
            "E21",
            "shares",
            {1.0, count, commodities::generateShareCode("     ")}},
        unitD{"E22", "TEU", {20.0 * 8.0 * 8.5, precise::ft.pow(3)}},
        unitD{"E23", "tyre", {1.0, count, commodities::tires}},
        unitD{"E25", "active unit", count* pu},
        unitD{"E27", "dose", {1.0, count, commodities::packaging::dose}},
        unitD{"E28", "air dry ton", {1.0, ton, commodities::packaging::dry}},
        unitD{"E3", "trailer", {1.0, ton, commodities::packaging::trailer}},
        unitD{"E30", "strand", {1.0, ton, commodities::packaging::strand}},
        unitD{"E31", "square metre per litre", m.pow(2) / L},
        unitD{"E32", "litre per hour", L / hr},
        unitD{"E33", "foot per thousand", ft / (thousand * count)},
        unitD{"E34", "gigabyte", giga* data::byte},
        unitD{"E35", "terabyte", tera* data::byte},
        unitD{"E36", "petabyte", peta* data::byte},
        unitD{"E37", "pixel", {1.0, count, commodities::pixel}},
        unitD{"E38", "megapixel", {1.0, mega* count, commodities::pixel}},
        unitD{"E39", "dots per inch", {1.0, count / in, commodities::dots}},
        unitD{
            "E4",
            "gross kilogram",
            {1.0, kg, commodities::packaging::gross_quantity}},
        unitD{"E40", "part per hundred thousand", count / (hundred * thousand)},
        unitD{
            "E41",
            "kilogram-force per square millimetre",
            precise::kilo* precise::gm::pond / mm.pow(2)},
        unitD{
            "E42",
            "kilogram-force per square centimetre",
            precise::kilo* precise::gm::pond / cm.pow(2)},
        unitD{"E43", "joule per square centimetre", J / cm.pow(2)},
        unitD{
            "E44",
            "kilogram-force metre per square centimetre",
            precise::kilo* precise::gm::pond* m / cm.pow(2)},
        unitD{"E45", "milliohm", milli* ohm},
        unitD{"E46", "kilowatt hour per cubic metre", kilo* W* hr / m.pow(3)},
        unitD{"E47", "kilowatt hour per kelvin", kilo* W* hr / K},
        unitD{"E48", "service unit", {1.0, count, commodities::service}},
        unitD{"E49", "working day", {1.0, day, commodities::labor}},
        unitD{"E5", "metric long ton", av::longton},
        unitD{"E50", "accounting unit", {1.0, count, commodities::accounting}},
        unitD{"E51", "job", {1.0, count, commodities::job}},
        unitD{"E52", "run foot", {1.0, ft, commodities::packaging::run}},
        unitD{"E53", "test", {1.0, count, commodities::test}},
        unitD{"E54", "trip", {1.0, count, commodities::trip}},
        unitD{"E55", "use", {1.0, count, commodities::use}},
        unitD{"E56", "well", {1.0, count, commodities::packaging::well}},
        unitD{"E57", "zone", {1.0, count, commodities::packaging::zone}},
        unitD{"E58", "exabit per second", exa* data::bit / s},
        unitD{"E59", "exbibyte", exbi* data::byte},
        unitD{"E60", "pebibyte", pebi* data::byte},
        unitD{"E61", "tebibyte", tebi* data::byte},
        unitD{"E62", "gibibyte", gibi* data::byte},
        unitD{"E63", "mebibyte", mebi* data::byte},
        unitD{"E64", "kibibyte", kibi* data::byte},
        unitD{"E65", "exbibit per metre", exbi* data::bit / m},
        unitD{"E66", "exbibit per square metre", exbi* data::bit / m.pow(2)},
        unitD{"E67", "exbibit per cubic metre", exbi* data::bit / m.pow(3)},
        unitD{"E68", "gigabyte per second", giga* data::byte / s},
        unitD{"E69", "gibibit per metre", gibi* data::bit / m},
        unitD{"E70", "gibibit per square metre", gibi* data::bit / m.pow(2)},
        unitD{"E71", "gibibit per cubic metre", gibi* data::bit / m.pow(3)},
        unitD{"E72", "kibibit per metre", kibi* data::bit / m},
        unitD{"E73", "kibibit per square metre", kibi* data::bit / m.pow(2)},
        unitD{"E74", "kibibit per cubic metre", kibi* data::bit / m.pow(3)},
        unitD{"E75", "mebibit per metre", mebi* data::bit / m},
        unitD{"E76", "mebibit per square metre", mebi* data::bit / m.pow(2)},
        unitD{"E77", "mebibit per cubic metre", mebi* data::bit / m.pow(3)},
        unitD{"E78", "petabit", peta* data::bit},
        unitD{"E79", "petabit per second", peta* data::bit / s},
        unitD{"E80", "pebibit per metre", pebi* data::bit / m},
        unitD{"E81", "pebibit per square metre", pebi* data::bit / m.pow(2)},
        unitD{"E82", "pebibit per cubic metre", pebi* data::bit / m.pow(3)},
        unitD{"E83", "terabit", tera* data::bit},
        unitD{"E84", "terabit per second", tera* data::bit / s},
        unitD{"E85", "tebibit per metre", tebi* data::bit / m},
        unitD{"E86", "tebibit per cubic metre", tebi* data::bit / m.pow(3)},
        unitD{"E87", "tebibit per square metre", tebi* data::bit / m.pow(2)},
        unitD{"E88", "bit per metre", data::bit / m},
        unitD{"E89", "bit per square metre", data::bit / m.pow(2)},
        unitD{"E90", "reciprocal centimetre", cm.inv()},
        unitD{"E91", "reciprocal day", day.inv()},
        unitD{"E92", "cubic decimetre per hour", (deci * m).pow(3) / hr},
        unitD{"E93", "kilogram per hour", kg / hr},
        unitD{"E94", "kilomole per second", kilo* mol / s},
        unitD{"E95", "mole per second", mol / s},
        unitD{"E96", "degree per second", deg / s},
        unitD{"E97", "millimetre per degree Celcius metre", mm / (degC * m)},
        unitD{"E98", "degree Celsius per kelvin", degC / K},
        unitD{"E99", "hectopascal per bar", hecto* Pa / bar},
        unitD{"EA", "each", one},
        unitD{
            "EB",
            "electronic mail box",
            {1.0, count, commodities::electronic_mail_box}},
        unitD{"EC", "each per month", one / time::moj},
        unitD{"EP", "eleven pack", {11, one}},
        unitD{"EQ", "equivalent gallon", gal},
        unitD{"EV", "envelope", {1.0, count, commodities::packaging::envelope}},
        unitD{"F01", "bit per cubic metre", data::bit / m.pow(3)},
        unitD{"F02", "kelvin per kelvin", pu* K},
        unitD{"F03", "kilopascal per bar", kilo* Pa / bar},
        unitD{"F04", "millibar per bar", milli* pu* bar},
        unitD{"F05", "megapascal per bar", mega* Pa / bar},
        unitD{"F06", "poise per bar", cgs::poise / bar},
        unitD{"F07", "pascal per bar", Pa / bar},
        unitD{"F08", "milliampere per inch", milli* A / in},
        unitD{"F1", "thousand cubic foot per day", {1000.0, ft.pow(3) / day}},
        unitD{"F10", "kelvin per hour", K / hr},
        unitD{"F11", "kelvin per minute", K / min},
        unitD{"F12", "kelvin per second", K / s},
        unitD{"F13", "slug", imp::slug},
        unitD{"F14", "gram per kelvin", g / K},
        unitD{"F15", "kilogram per kelvin", kg / K},
        unitD{"F16", "milligram per kelvin", mg / K},
        unitD{"F17", "pound-force per foot", lbf / ft},
        unitD{"F18", "kilogram square centimetre", kg* cm.pow(2)},
        unitD{"F19", "kilogram square millimetre", kg* mm.pow(2)},
        unitD{"F20", "pound inch squared", lb* in.pow(2)},
        unitD{"F21", "pound-force inch", lbf* in},
        unitD{"F22", "pound-force foot per ampere", lbf* ft / A},
        unitD{"F23", "gram per cubic decimetre", g / ((deci * m).pow(3))},
        unitD{"F24", "kilogram per kilomol", g / mol},
        unitD{"F25", "gram per hertz", g / Hz},
        unitD{"F26", "gram per day", g / day},
        unitD{"F27", "gram per hour", g / hr},
        unitD{"F28", "gram per minute", g / min},
        unitD{"F29", "gram per second", g / s},
        unitD{"F30", "kilogram per day", kg / day},
        unitD{"F31", "kilogram per minute", kg / min},
        unitD{"F32", "milligram per day", mg / day},
        unitD{"F33", "milligram per minute", mg / min},
        unitD{"F34", "milligram per second", mg / s},
        unitD{"F35", "gram per day kelvin", g / (day * K)},
        unitD{"F36", "gram per hour kelvin", g / (hr * K)},
        unitD{"F37", "gram per minute kelvin", g / (min * K)},
        unitD{"F38", "gram per second kelvin", g / (s * K)},
        unitD{"F39", "kilogram per day kelvin", kg / (day * K)},
        unitD{"F40", "kilogram per hour kelvin", kg / (hr * K)},
        unitD{"F41", "kilogram per minute kelvin", kg / (min * K)},
        unitD{"F42", "kilogram per second kelvin", kg / (s * K)},
        unitD{"F43", "milligram per day kelvin", mg / (day * K)},
        unitD{"F44", "milligram per hour kelvin", mg / (hr * K)},
        unitD{"F45", "milligram per minute kelvin", mg / (min * K)},
        unitD{"F46", "milligram per second kelvin", mg / (s * K)},
        unitD{"F47", "newton per millimetre", N / mm},
        unitD{"F48", "pound-force per inch", lbf / in},
        unitD{"F49", "rod", us::rod},
        unitD{"F50", "micrometre per kelvin", micro* m / K},
        unitD{"F51", "centimetre per kelvin", cm / K},
        unitD{"F52", "metre per kelvin", m / K},
        unitD{"F53", "millimetre per kelvin", mm / K},
        unitD{"F54", "milliohm per metre", milli* ohm / m},
        unitD{"F55", "ohm per mile (statute)", ohm / mile},
        unitD{"F56", "ohm per kilometre", ohm / km},
        unitD{
            "F57",
            "milliampere per (pound-force per square inch)",
            milli* A / pressure::psi},
        unitD{"F58", "reciprocal bar", bar.inv()},
        unitD{"F59", "milliampere per bar", milli* A / bar},
        unitD{"F60", "degree Celsius per bar", degC / bar},
        unitD{"F61", "kelvin per bar", K / bar},
        unitD{"F62", "gram per day bar", g / (day * bar)},
        unitD{"F63", "gram per hour bar", g / (hr * bar)},
        unitD{"F64", "gram per minute bar", g / (min * bar)},
        unitD{"F65", "gram per second bar", g / (s * bar)},
        unitD{"F66", "kilogram per day bar", kg / (day * bar)},
        unitD{"F67", "kilogram per hour bar", kg / (hr * bar)},
        unitD{"F68", "kilogram per minute bar", kg / (min * bar)},
        unitD{"F69", "kilogram per second bar", kg / (s * bar)},
        unitD{"F70", "milligram per day bar", mg / (day * bar)},
        unitD{"F71", "milligram per hour bar", mg / (hr * bar)},
        unitD{"F72", "milligram per minute bar", mg / (min * bar)},
        unitD{"F73", "milligram per second bar", mg / (s * bar)},
        unitD{"F74", "gram per bar", g / bar},
        unitD{"F75", "milligram per bar", mg / bar},
        unitD{"F76", "milliampere per millimetre", A / m},
        unitD{"F77", "pascal second per kelvin", Pa* s / K},
        unitD{"F78", "inch of water", pressure::inH2O},
        unitD{"F79", "inch of mercury", pressure::inHg},
        unitD{"F80", "water horse power", power::hpS},
        unitD{"F81", "bar per kelvin", bar / K},
        unitD{"F82", "hectopascal per kelvin", hecto* Pa / K},
        unitD{"F83", "kilopascal per kelvin", kilo* Pa / K},
        unitD{"F84", "millibar per kelvin", milli* bar / K},
        unitD{"F85", "megapascal per kelvin", mega* Pa / K},
        unitD{"F86", "poise per kelvin", cgs::poise / K},
        unitD{"F87", "volt per litre minute", V / (L * min)},
        unitD{"F88", "newton centimetre", N* cm},
        unitD{"F89", "newton metre per degree", N* m / deg},
        unitD{
            "F9",
            "fibre per cubic centimetre of air",
            precise_unit(count, commodities::fiber) /
                precise_unit(cm.pow(3), commodities::air)},
        unitD{"F90", "newton metre per ampere", N* m / A},
        unitD{"F91", "bar litre per second", bar* L / s},
        unitD{"F92", "bar cubic metre per second", bar* m.pow(3) / s},
        unitD{"F93", "hectopascal litre per second", hecto* Pa* L / s},
        unitD{
            "F94",
            "hectopascal cubic metre per second",
            hecto* Pa* m.pow(3) / s},
        unitD{"F95", "millibar litre per second", milli* bar* L / s},
        unitD{
            "F96",
            "millibar cubic metre per second",
            milli* bar* m.pow(3) / s},
        unitD{"F97", "megapascal litre per second", mega* Pa* L / s},
        unitD{
            "F98",
            "megapascal cubic metre per second",
            mega* Pa* m.pow(3) / s},
        unitD{"F99", "pascal litre per second", Pa* L / s},
        unitD{"FAH", "degree Fahrenheit", degF},
        unitD{"FAR", "farad", F},
        unitD{"FB", "field", {1.0, count, commodities::packaging::field}},
        unitD{"FBM", "fibre metre", {1.0, meter, commodities::fiber}},
        unitD{"FC", "thousand cubic foot", thousand* ft.pow(3)},
        unitD{
            "FD",
            "million particle per cubic foot",
            precise_unit{
                mega * count / ft.pow(3),
                commodities::packaging::particle}},
        unitD{"FE", "track foot", {1.0, ft, commodities::packaging::track}},
        unitD{"FF", "hundred cubic metre", hundred* m.pow(3)},
        unitD{
            "FG",
            "transdermal patch",
            {1.0, count, commodities::packaging::patch}},
        unitD{"FH", "micromole", micro* mol},
        unitD{
            "FIT",
            "failures in time",
            {1.0, count / s, commodities::failure}},
        unitD{"FL", "flake ton", {1.0, ton, commodities::packaging::flake}},
        unitD{"FM", "million cubic foot", mega* ft.pow(3)},
        unitD{"FOT", "foot", ft},
        unitD{"FP", "pound per square foot", lb / ft.pow(2)},
        unitD{"FR", "foot per minute", ft / min},
        unitD{"FS", "foot per second", ft / s},
        unitD{"FTK", "square foot", ft.pow(2)},
        unitD{"FTQ", "cubic foot", ft.pow(3)},
        unitD{"G01", "pascal cubic metre per second", Pa* m.pow(3) / s},
        unitD{"G04", "centimetre per bar", cm / bar},
        unitD{"G05", "metre per bar", m / bar},
        unitD{"G06", "millimetre per bar", mm / bar},
        unitD{"G08", "square inch per second", in.pow(2) / s},
        unitD{"G09", "square metre per second kelvin", m.pow(2) / (s * K)},
        unitD{"G10", "stokes per kelvin", cgs::stokes / K},
        unitD{"G11", "gram per cubic centimetre bar", g / (cm.pow(3) * bar)},
        unitD{
            "G12",
            "gram per cubic decimetre bar",
            g / ((deci * m).pow(3) * bar)},
        unitD{"G13", "gram per litre bar", g / (L * bar)},
        unitD{"G14", "gram per cubic metre bar", g / (m.pow(3) * bar)},
        unitD{"G15", "gram per millilitre bar", g / (mL * bar)},
        unitD{
            "G16",
            "kilogram per cubic centimetre bar",
            kg / (cm.pow(3) * bar)},
        unitD{"G17", "kilogram per litre bar", kg / (L * bar)},
        unitD{"G18", "kilogram per cubic metre bar", kg / (m.pow(3) * bar)},
        unitD{"G19", "newton metre per kilogram", N* m / kg},
        unitD{"G2", "US gallon per minute", us::gallon / min},
        unitD{"G20", "pound-force foot per pound", lbf* ft / lb},
        unitD{"G21", "cup", us::cup},
        unitD{"G23", "peck", us::dry::peck},
        unitD{"G24", "tablespoon (US)", us::tbsp},
        unitD{"G25", "teaspoon (US)", us::tsp},
        unitD{"G26", "stere", volume::stere},
        unitD{"G27", "cubic centimetre per kelvin", cm.pow(3) / K},
        unitD{"G28", "litre per kelvin", L / K},
        unitD{"G29", "cubic metre per kelvin", m.pow(3) / K},
        unitD{"G3", "Imperial gallon per minute", imp::gallon / min},
        unitD{"G30", "millilitre per kelvin", mL / K},
        unitD{"G31", "kilogram per cubic centimetre", kg / cm.pow(3)},
        unitD{
            "G32",
            "ounce (avoirdupois) per cubic yard",
            av::ounce / yd.pow(3)},
        unitD{"G33", "gram per cubic centimetre kelvin", g / (cm.pow(3) * K)},
        unitD{
            "G34",
            "gram per cubic decimetre kelvin",
            g / ((deci * m).pow(3) * K)},
        unitD{"G35", "gram per litre kelvin", g / (L * K)},
        unitD{"G36", "gram per cubic metre kelvin", g / (m.pow(3) * K)},
        unitD{"G37", "gram per millilitre kelvin", g / (mL * K)},
        unitD{
            "G38",
            "kilogram per cubic centimetre kelvin",
            kg / (cm.pow(3) * K)},
        unitD{"G39", "kilogram per litre kelvin", kg / (L * K)},
        unitD{"G40", "kilogram per cubic metre kelvin", kg / (m.pow(3) * K)},
        unitD{"G41", "square metre per second bar", m.pow(2) / (s * bar)},
        unitD{"G42", "microsiemens per centimetre", micro* siemens / cm},
        unitD{"G43", "microsiemens per metre", micro* siemens / m},
        unitD{"G44", "nanosiemens per centimetre", nano* siemens / cm},
        unitD{"G45", "nanosiemens per metre", nano* siemens / m},
        unitD{"G46", "stokes per bar", cgs::stokes / bar},
        unitD{"G47", "cubic centimetre per day", cm.pow(3) / day},
        unitD{"G48", "cubic centimetre per hour", cm.pow(3) / hr},
        unitD{"G49", "cubic centimetre per minute", cm.pow(3) / min},
        unitD{"G50", "gallon (US) per hour", us::gallon / hr},
        unitD{"G51", "litre per second", L / s},
        unitD{"G52", "cubic metre per day", m.pow(3) / day},
        unitD{"G53", "cubic metre per minute", m.pow(3) / min},
        unitD{"G54", "millilitre per day", mL / day},
        unitD{"G55", "millilitre per hour", mL / hr},
        unitD{"G56", "cubic inch per hour", in.pow(3) / hr},
        unitD{"G57", "cubic inch per minute", in.pow(3) / min},
        unitD{"G58", "cubic inch per second", in.pow(3) / s},
        unitD{"G59", "milliampere per litre minute", milli* A / (L * min)},
        unitD{"G60", "volt per bar", V / bar},
        unitD{"G61", "cubic centimetre per day kelvin", cm.pow(3) / (day * K)},
        unitD{"G62", "cubic centimetre per hour kelvin", cm.pow(3) / (hr * K)},
        unitD{
            "G63",
            "cubic centimetre per minute kelvin",
            cm.pow(3) / (min * K)},
        unitD{"G64", "cubic centimetre per second kelvin", cm.pow(3) / (s * K)},
        unitD{"G65", "litre per day kelvin", L / (day * K)},
        unitD{"G66", "litre per hour kelvin", L / (hr * K)},
        unitD{"G67", "litre per minute kelvin", L / (min * K)},
        unitD{"G68", "litre per second kelvin", L / (s * K)},
        unitD{"G69", "cubic metre per day kelvin", m.pow(3) / (day * K)},
        unitD{
            "G7",
            "microfiche sheet",
            {1.0, count, commodities::packaging::sheet}},
        unitD{"G70", "cubic metre per hour kelvin", m.pow(3) / (hr * K)},
        unitD{"G71", "cubic metre per minute kelvin", m.pow(3) / (min * K)},
        unitD{"G72", "cubic metre per second kelvin", m.pow(3) / (s * K)},
        unitD{"G73", "millilitre per day kelvin", mL / (day * K)},
        unitD{"G74", "millilitre per hour kelvin", mL / (hr * K)},
        unitD{"G75", "millilitre per minute kelvin", mL / (min * K)},
        unitD{"G76", "millilitre per second kelvin", mL / (s * K)},
        unitD{"G77", "millimetre to the fourth power", mm.pow(4)},
        unitD{"G78", "cubic centimetre per day bar", cm.pow(3) / (day * bar)},
        unitD{"G79", "cubic centimetre per hour bar", cm.pow(3) / (hr * bar)},
        unitD{
            "G80",
            "cubic centimetre per minute bar",
            cm.pow(3) / (min * bar)},
        unitD{"G81", "cubic centimetre per second bar", cm.pow(3) / (s * bar)},
        unitD{"G82", "litre per day bar", L / (day * bar)},
        unitD{"G83", "litre per hour bar", L / (hr * bar)},
        unitD{"G84", "litre per minute bar", L / (min * bar)},
        unitD{"G85", "litre per second bar", L / (s * bar)},
        unitD{"G86", "cubic metre per day bar", m.pow(3) / (day * bar)},
        unitD{"G87", "cubic metre per hour bar", m.pow(3) / (hr * bar)},
        unitD{"G88", "cubic metre per minute bar", m.pow(3) / (min * bar)},
        unitD{"G89", "cubic metre per second bar", m.pow(3) / (s * bar)},
        unitD{"G90", "millilitre per day bar", mL / (day * bar)},
        unitD{"G91", "millilitre per hour bar", mL / (hr * bar)},
        unitD{"G92", "millilitre per minute bar", mL / (min * bar)},
        unitD{"G93", "millilitre per second bar", mL / (s * bar)},
        unitD{"G94", "cubic centimetre per bar", cm.pow(3) / bar},
        unitD{"G95", "litre per bar", L / bar},
        unitD{"G96", "cubic metre per bar", m.pow(3) / bar},
        unitD{"G97", "millilitre per bar", mL / bar},
        unitD{"G98", "microhenry per kiloohm", micro* H / (kilo * ohm)},
        unitD{"G99", "microhenry per ohm", micro* H / ohm},
        unitD{"GB", "gallon (US) per day", us::gallon / day},
        unitD{"GBQ", "gigabecquerel", giga* Bq},
        unitD{"GC", "gram per 100 gram", g / (hecto * g)},
        unitD{
            "GD",
            "gross barrel",
            {144.0, count, commodities::packaging::barrel}},
        unitD{"GDW", "gram, dry weight", g},
        unitD{"GE", "pound per gallon (US)", lb / us::gallon},
        unitD{"GF", "gram per metre", g / m},
        unitD{
            "GFI",
            "gram of fissile isotope",
            {1.0, g, commodities::fissile_material}},
        unitD{"GGR", "great gross", {1728.0, count}},
        unitD{"GH", "half gallon (US)", {0.5, us::gallon}},
        unitD{"GIA", "gill (US)", us::gill},
        unitD{"GIC", "gram, including container", g},
        unitD{"GII", "gill (UK)", imp::gill},
        unitD{"GIP", "gram, including inner packaging", g},
        unitD{"GJ", "gram per millilitre", g / mL},
        unitD{"GK", "gram per kilogram", g / kg},
        unitD{"GL", "gram per litre", g / L},
        unitD{"GLD", "dry gallon (US)", us::dry::gallon},
        unitD{"GLI", "gallon (UK)", imp::gallon},
        unitD{"GLL", "gallon (US)", us::gallon},
        unitD{"GM", "gram per square metre", g / m.pow(2)},
        unitD{"GN", "gross gallon", {144.0, gal}},
        unitD{"GO", "milligram per square metre", mg / m.pow(2)},
        unitD{"GP", "milligram per cubic metre", mg / m.pow(3)},
        unitD{"GQ", "microgram per cubic metre", micro* g / m.pow(3)},
        unitD{"GRM", "gram", g},
        unitD{"GRN", "grain", i::grain},
        unitD{"GRO", "gross", {144.0, count}},
        unitD{"GRT", "gross register ton", {100.0, ft.pow(3)}},
        unitD{"GT", "gross ton", av::longton},  // NOT CORRECT
        unitD{"GV", "gigajoule", giga* J},
        unitD{
            "GW",
            "gallon per thousand cubic foot",
            us::gallon / (kilo * ft.pow(3))},
        unitD{"GWH", "gigawatt hour", giga* W* hr},
        unitD{"GY", "gross yard", yd},
        unitD{"GZ", "gage system", eflag},
        unitD{"H03", "henry per kiloohm", H / (kilo * ohm)},
        unitD{"H04", "henry per ohm", H / ohm},
        unitD{"H05", "millihenry per kiloohm", milli* H / (kilo * ohm)},
        unitD{"H06", "millihenry per ohm", milli* H / ohm},
        unitD{"H07", "pascal second per bar", Pa* s / bar},
        unitD{"H08", "microbecquerel", micro* Bq},
        unitD{"H09", "reciprocal year", yr.inv()},
        unitD{
            "H1",
            "half page - electronic",
            {0.5, count, commodities::packaging::page_electronic}},
        unitD{"H10", "reciprocal hour", hr.inv()},
        unitD{"H11", "reciprocal month", time::moj.inv()},
        unitD{"H12", "degree Celsius per hour", degC / hr},
        unitD{"H13", "degree Celsius per minute", degC / min},
        unitD{"H14", "degree Celsius per second", degC / s},
        unitD{"H15", "square centimetre per gram", cm.pow(2) / g},
        unitD{"H16", "square decametre", (ten * m).pow(2)},
        unitD{"H18", "square hectometre", (hecto * m).pow(2)},
        unitD{"H19", "cubic hectometre", (hecto * m).pow(3)},
        unitD{"H2", "half litre", half* L},
        unitD{"H20", "cubic kilometre", km.pow(3)},
        unitD{"H21", "blank", {1.0, count, commodities::blank}},
        unitD{"H22", "volt square inch per pound-force", V* in.pow(2) / lbf},
        unitD{"H23", "volt per inch", V / in},
        unitD{"H24", "volt per microsecond", V / (micro * s)},
        unitD{"H25", "percent per kelvin", percent / K},
        unitD{"H26", "ohm per metre", ohm / m},
        unitD{"H27", "degree per metre", deg / m},
        unitD{"H28", "microfarad per kilometre", micro* F / km},
        unitD{"H29", "microgram per litre", micro* g / L},
        unitD{"H30", "square micrometre", (micro * m).pow(2)},
        unitD{"H31", "ampere per kilogram", A / kg},
        unitD{"H32", "ampere squared second", A.pow(2) * s},
        unitD{"H33", "farad per kilometre", F / km},
        unitD{"H34", "hertz metre", Hz* m},
        unitD{"H35", "kelvin metre per watt", K* m / W},
        unitD{"H36", "megaohm per kilometre", mega* ohm / km},
        unitD{"H37", "megaohm per metre", mega* ohm / m},
        unitD{"H38", "megaampere", mega* A},
        unitD{"H39", "megahertz kilometre", mega* Hz* km},
        unitD{"H40", "newton per ampere", N / A},
        unitD{
            "H41",
            "newton metre watt to the power minus 0,5",
            generate_custom_unit(879)},
        unitD{"H42", "pascal per metre", Pa / m},
        unitD{"H43", "siemens per centimetre", S / cm},
        unitD{"H44", "teraohm", tera* ohm},
        unitD{"H45", "volt second per metre", V* s / m},
        unitD{"H46", "volt per second", V / s},
        unitD{"H47", "watt per cubic metre", W / m.pow(3)},
        unitD{"H48", "attofarad", atto* F},
        unitD{"H49", "centimetre per hour", cm / hr},
        unitD{"H50", "reciprocal cubic centimetre", cm.pow(-3)},
        unitD{"H51", "decibel per kilometre", log::dB / km},
        unitD{"H52", "decibel per metre", log::dB / m},
        unitD{"H53", "kilogram per bar", kg / bar},
        unitD{
            "H54",
            "kilogram per cubic decimetre kelvin",
            kg / ((deci * m).pow(3) * K)},
        unitD{
            "H55",
            "kilogram per cubic decimetre bar",
            kg / ((deci * m).pow(3) * bar)},
        unitD{"H56", "kilogram per square metre second", kg / (m.pow(2) * s)},
        unitD{
            "H57",
            "inch per two pi radiant",
            {1.0 / constants::tau, in / rad}},
        unitD{"H58", "metre per volt second", m / (V * s)},
        unitD{"H59", "square metre per newton", m.pow(2) / N},
        unitD{"H60", "cubic metre per cubic metre", pu* m.pow(3)},
        unitD{"H61", "millisiemens per centimetre", milli* S / cm},
        unitD{"H62", "millivolt per minute", electrical::mV / min},
        unitD{"H63", "milligram per square centimetre", mg / cm.pow(2)},
        unitD{"H64", "milligram per gram", mg / g},
        unitD{"H65", "millilitre per cubic metre", mL / m.pow(3)},
        unitD{"H66", "millimetre per year", mm / yr},
        unitD{"H67", "millimetre per hour", mm / hr},
        unitD{"H68", "millimole per gram", milli* mol / g},
        unitD{"H69", "picopascal per kilometre", pico* Pa / km},
        unitD{"H70", "picosecond", pico* s},
        unitD{"H71", "percent per month", percent / time::moj},
        unitD{"H72", "percent per hectobar", percent / (hecto * bar)},
        unitD{"H73", "percent per decakelvin", percent / (deca * K)},
        unitD{"H74", "watt per metre", W / m},
        unitD{"H75", "decapascal", ten* Pa},
        unitD{"H76", "gram per millimetre", g / mm},
        unitD{"H77", "module width", pu* m},
        unitD{
            "H78",
            "conventional centimetre of water",
            {10.0, precise::pressure::mmH2O}},
        unitD{"H79", "French gauge", precise::clinical::charriere},
        unitD{"H80", "rack unit", precise_unit(1.75, precise::i::inch)},
        unitD{"H81", "millimetre per minute", mm / min},
        unitD{"H82", "big point", typographic::dtp::point},
        unitD{"H83", "litre per kilogram", L / kg},
        unitD{"H84", "gram millimetre", g* mm},
        unitD{"H85", "reciprocal week", time::week.inv()},
        unitD{"H87", "piece", {1.0, one, commodities::packaging::piece}},
        unitD{"H88", "megaohm kilometre", mega* ohm* km},
        unitD{"H89", "percent per ohm", percent / ohm},
        unitD{"H90", "percent per degree", percent / deg},
        unitD{
            "H91",
            "percent per ten thousand",
            percent / (ten * thousand * count)},
        unitD{
            "H92",
            "percent per one hundred thousand",
            percent / (hundred * thousand * count)},
        unitD{"H93", "percent per hundred", percent / (hundred * count)},
        unitD{"H94", "percent per thousand", percent / (thousand * count)},
        unitD{"H95", "percent per volt", percent / V},
        unitD{"H96", "percent per bar", percent / bar},
        unitD{"H98", "percent per inch", percent / in},
        unitD{"H99", "percent per metre", percent / m},
        unitD{"HA", "hank", {1.0, count, commodities::packaging::hank}},
        unitD{"HAR", "hectare", area::hectare},
        unitD{"HBA", "hectobar", hecto* bar},
        unitD{
            "HBX",
            "hundred boxes",
            {100.0, count, commodities::packaging::box}},
        unitD{"HC", "hundred count", hundred* count},
        unitD{"HD", "half dozen", {6.0, count}},
        unitD{"HDW", "hundred kilogram, dry weight", {100.0, kg}},
        unitD{"HE", "hundredth of a carat", centi* metric::carat},
        unitD{"HEA", "head", {1.0, count, commodities::packaging::head}},
        unitD{"HF", "hundred foot", hundred* foot},
        unitD{"HGM", "hectogram", hecto* g},
        unitD{"HH", "hundred cubic foot", hundred* ft.pow(3)},
        unitD{
            "HI",
            "hundred sheet",
            {100.0, count, commodities::packaging::sheet}},
        unitD{
            "HIU",
            "hundred international unit",
            {100.0, precise::laboratory::IU}},
        unitD{"HJ", "metric horse power", precise::power::hpM},
        unitD{"HK", "hundred kilogram", hundred* kg},
        unitD{"HKM", "hundred kilogram, net mass", hundred* kg},
        unitD{
            "HL",
            "hundred foot (linear)",
            {100.0, ft, commodities::packaging::linear}},
        unitD{"HLT", "hectolitre", hecto* L},
        unitD{"HM", "mile per hour", mph},
        unitD{"HMQ", "million cubic metre", mega* m.pow(3)},
        unitD{"HMT", "hectometre", hecto* m},
        unitD{"HN", "conventional millimetre of mercury", pressure::mmHg},
        unitD{"HO", "hundred troy ounce", hundred* troy::oz},
        unitD{"HP", "conventional millimetre of water", pressure::mmH2O},
        unitD{
            "HPA",
            "hectolitre of pure alcohol",
            {1.0, hecto* L, commodities::ethanol}},
        unitD{"HS", "hundred square foot", hundred* ft.pow(2)},
        unitD{"HT", "half hour", half* hr},
        unitD{"HTZ", "hertz", Hz},
        unitD{"HUR", "hour", hr},
        unitD{"HY", "hundred yard", hundred* yd},
        unitD{"IA", "inch pound", in* lb},
        unitD{"IC", "count per inch", count / in},
        unitD{"IE", "person", {1.0, count, commodities::person}},
        unitD{"IF", "inches of water", pressure::inH2O},
        unitD{"II", "column inch", pressure::inH2O* eflag},
        unitD{"IL", "inch per minute", in / min},
        unitD{"IM", "impression", {1.0, count, commodities::impression}},
        unitD{"INH", "inch", in},
        unitD{"INK", "square inch", in.pow(2)},
        unitD{"INQ", "cubic inch", in.pow(3)},
        unitD{"IP", "insurance policy", {1.0, count, commodities::policy}},
        unitD{
            "ISD",
            "international sugar degree",
            {1.0, generate_custom_unit(107), commodities::sugar}},
        unitD{"IT", "count per centimetre", count / cm},
        unitD{"IU", "inch per second", in / s},
        unitD{"IV", "inch per second squared", in / s.pow(2)},
        unitD{"J10", "percent per millimetre", percent / mm},
        unitD{"J12", "per mille per psi", {0.001, pressure::psi.inv()}},
        unitD{"J13", "degree API", special::degreeAPI},
        unitD{"J14", "degree Baume", special::degreeBaumeLight},
        unitD{"J15", "degree Baume (US heavy)", special::degreeBaumeLight},
        unitD{"J16", "degree Baume (US light)", special::degreeBaumeHeavy},
        unitD{
            "J17",
            "degree Balling",
            {1.0, generate_custom_unit(109), commodities::sugar}},
        unitD{
            "J18",
            "degree Brix",
            {1.0, generate_custom_unit(108), commodities::sugar}},
        unitD{
            "J19",
            "degree Fahrenheit hour square foot per British thermal unit (thermochemical)",
            degF* hr* ft.pow(2) / energy::btu_th},
        unitD{"J2", "joule per kilogram", J / kg},
        unitD{"J20", "degree Fahrenheit per kelvin", degF / K},
        unitD{"J21", "degree Fahrenheit per bar", degF / bar},
        unitD{
            "J22",
            "degree Fahrenheit hour square foot per British thermal unit (international table)",
            degF* hr* ft.pow(2) / energy::btu_it},
        unitD{"J23", "degree Fahrenheit per hour", degF / hr},
        unitD{"J24", "degree Fahrenheit per minute", degF / min},
        unitD{"J25", "degree Fahrenheit per second", degF / s},
        unitD{"J26", "reciprocal degree Fahrenheit", degF.inv()},
        unitD{
            "J27",
            "degree Oechsle",
            {1.0, generate_custom_unit(110), commodities::sugar}},
        unitD{"J28", "degree Rankine per hour", temperature::rankine / hr},
        unitD{"J29", "degree Rankine per minute", temperature::rankine / min},
        unitD{"J30", "degree Rankine per second", temperature::rankine / s},
        unitD{"J31", "degree Twaddell", generate_custom_unit(111)},
        unitD{"J32", "micropoise", micro* cgs::poise},
        unitD{"J33", "microgram per kilogram", micro* g / kg},
        unitD{
            "J34",
            "microgram per cubic metre kelvin",
            micro* g / (m.pow(3) * K)},
        unitD{
            "J35",
            "microgram per cubic metre bar",
            micro* g / (m.pow(3) * bar)},
        unitD{"J36", "microlitre per litre", micro* L / L},
        unitD{"J38", "baud", bit / s},
        unitD{"J39", "British thermal unit (mean)", energy::btu_mean},
        unitD{
            "J40",
            "British thermal unit (international table) foot "
            "per hour square foot degree Fahrenheit",
            energy::btu_it* ft / (hr * ft.pow(2) * degF)},
        unitD{
            "J41",
            "British thermal unit (international table) inch "
            "per hour square foot degree Fahrenheit",
            energy::btu_it* in / (hr * ft.pow(2) * degF)},
        unitD{
            "J42",
            "British thermal unit (international table) inch "
            "per second square foot degree Fahrenheit",
            energy::btu_it* in / (s * ft.pow(2) * degF)},
        unitD{
            "J43",
            "British thermal unit (international table) per "
            "pound degree Fahrenheit",
            energy::btu_it / (lb * degF)},
        unitD{
            "J44",
            "British thermal unit (international table) per minute",
            energy::btu_it / min},
        unitD{
            "J45",
            "British thermal unit (international table) per second",
            energy::btu_it / s},
        unitD{
            "J46",
            "British thermal unit (thermochemical) foot per hour square foot degree Fahrenheit",
            energy::btu_th* ft / (hr * ft.pow(2) * degF)},
        unitD{
            "J47",
            "British thermal unit (thermochemical) per hour",
            energy::btu_th / hr},
        unitD{
            "J48",
            "British thermal unit (thermochemical) inch per hour square foot degree Fahrenheit",
            energy::btu_th* in / (hr * ft.pow(2) * degF)},
        unitD{
            "J49",
            "British thermal unit (thermochemical) inch per second square foot degree Fahrenheit",
            energy::btu_th* in / (s * ft.pow(2) * degF)},
        unitD{
            "J50",
            "British thermal unit (thermochemical) per pound degree Fahrenheit",
            energy::btu_th / (lb * degF)},
        unitD{
            "J51",
            "British thermal unit (thermochemical) per minute",
            energy::btu_th / min},
        unitD{
            "J52",
            "British thermal unit (thermochemical) per second",
            energy::btu_th / s},
        unitD{"J53", "coulomb square metre per kilogram", C* m.pow(2) / kg},
        unitD{"J54", "megabaud", mega* bit / s},
        unitD{"J55", "watt second", W* s},
        unitD{"J56", "bar per bar", pu* bar},
        unitD{"J57", "barrel (UK petroleum)", imp::bbl},
        unitD{"J58", "barrel (UK petroleum) per minute", imp::bbl / min},
        unitD{"J59", "barrel (UK petroleum) per day", imp::bbl / day},
        unitD{"J60", "barrel (UK petroleum) per hour", imp::bbl / hr},
        unitD{"J61", "barrel (UK petroleum) per second", imp::bbl / s},
        unitD{"J62", "barrel (US petroleum) per hour", us::barrel / min},
        unitD{"J63", "barrel (US petroleum) per second", us::barrel / s},
        unitD{"J64", "bushel (UK) per day", imp::bushel / day},
        unitD{"J65", "bushel (UK) per hour", imp::bushel / hr},
        unitD{"J66", "bushel (UK) per minute", imp::bushel / min},
        unitD{"J67", "bushel (UK) per second", imp::bushel / s},
        unitD{"J68", "bushel (US dry) per day", us::dry::bushel / day},
        unitD{"J69", "bushel (US dry) per hour", us::dry::bushel / hr},
        unitD{"J70", "bushel (US dry) per minute", us::dry::bushel / min},
        unitD{"J71", "bushel (US dry) per second", us::dry::bushel / s},
        unitD{"J72", "centinewton metre", centi* N* m},
        unitD{"J73", "centipoise per kelvin", centi* cgs::poise / K},
        unitD{"J74", "centipoise per bar", centi* cgs::poise / bar},
        unitD{"J75", "calorie (mean)", energy::cal_mean},
        unitD{
            "J76",
            "calorie (international table) per gram degree Celsius",
            energy::cal_it / (g * degC)},
        unitD{
            "J78",
            "calorie (thermochemical) per centimetre second degree Celsius",
            energy::cal_th / (cm * s * degC)},
        unitD{
            "J79",
            "calorie (thermochemical) per gram degree Celsius",
            energy::cal_th / (g * degC)},
        unitD{
            "J81",
            "calorie (thermochemical) per minute",
            energy::cal_th / min},
        unitD{"J82", "calorie (thermochemical) per second", energy::cal_th / s},
        unitD{"J83", "clo", {0.155, m.pow(2) * K / W}},
        unitD{"J84", "centimetre per second kelvin", cm / (s * K)},
        unitD{"J85", "centimetre per second bar", cm / (s * bar)},
        unitD{"J87", "cubic centimetre per cubic metre", cm.pow(3) / m.pow(3)},
        unitD{"J89", "centimetre of mercury", {10.0, pressure::mmHg}},
        unitD{"J90", "cubic decimetre per day", (deci * m).pow(3) / day},
        unitD{
            "J91",
            "cubic decimetre per cubic metre",
            (deci * m).pow(3) / m.pow(3)},
        unitD{"J92", "cubic decimetre per minute", (deci * m).pow(3) / min},
        unitD{"J93", "cubic decimetre per second", (deci * m).pow(3) / s},
        unitD{"J94", "dyne centimetre", cgs::dyn* cm},
        unitD{"J95", "ounce (UK fluid) per day", imp::floz / day},
        unitD{"J96", "ounce (UK fluid) per hour", imp::floz / hr},
        unitD{"J97", "ounce (UK fluid) per minute", imp::floz / min},
        unitD{"J98", "ounce (UK fluid) per second", imp::floz / s},
        unitD{"J99", "ounce (US fluid) per day", us::floz / day},
        unitD{"JB", "jumbo", {1.0, count, commodities::packaging::jumbo}},
        unitD{"JE", "joule per kelvin", J / K},
        unitD{"JG", "jug", {1.0, count, commodities::packaging::jug}},
        unitD{"JK", "megajoule per kilogram", mega* J / kg},
        unitD{"JM", "megajoule per cubic metre", mega* J / m.pow(3)},
        unitD{
            "JNT",
            "pipeline joint",
            {1.0, count, commodities::packaging::joint}},
        unitD{"JO", "joint", {1.0, count, commodities::packaging::joint}},
        unitD{"JOU", "joule", J},
        unitD{"JPS", "hundred metre", hundred* m},
        unitD{"JR", "jar", {1.0, count, commodities::packaging::jar}},
        unitD{"JWL", "number of jewels", {1.0, count, commodities::jewel}},
        unitD{"K1", "kilowatt demand", electrical::kW},
        unitD{"K10", "ounce (US fluid) per hour", us::floz / hr},
        unitD{"K11", "ounce (US fluid) per minute", us::floz / min},
        unitD{"K12", "ounce (US fluid) per second", us::floz / s},
        unitD{"K13", "foot per degree Fahrenheit", ft / degF},
        unitD{"K14", "foot per hour", ft / hr},
        unitD{"K15", "foot pound-force per hour", ft* lbf / hr},
        unitD{"K16", "foot pound-force per minute", ft* lbf / min},
        unitD{"K17", "foot per psi", ft / pressure::psi},
        unitD{"K18", "foot per second degree Fahrenheit", ft / (s * degF)},
        unitD{"K19", "foot per second psi", ft / (s * pressure::psi)},
        unitD{"K2", "kilovolt ampere reactive demand", electrical::kVAR},
        unitD{"K20", "reciprocal cubic foot", ft.pow(-3)},
        unitD{"K21", "cubic foot per degree Fahrenheit", ft.pow(3) / degF},
        unitD{"K22", "cubic foot per day", ft.pow(3) / day},
        unitD{"K23", "cubic foot per psi", ft.pow(3) / pressure::psi},
        unitD{"K24", "foot of water", {12, pressure::inH2O}},
        unitD{"K25", "foot of mercury", {12, pressure::inHg}},
        unitD{"K26", "gallon (UK) per day", imp::gallon / day},
        unitD{"K27", "gallon (UK) per hour", imp::gallon / hr},
        unitD{"K28", "gallon (UK) per second", imp::gallon / s},
        unitD{"K3", "kilovolt ampere reactive hour", electrical::kVAR* hr},
        unitD{"K30", "gallon (US liquid) per second", us::gallon / s},
        unitD{
            "K31",
            "gram-force per square centimetre",
            g* constants::g0.as_unit() / cm.pow(2)},
        unitD{"K32", "gill (UK) per day", imp::gill / day},
        unitD{"K33", "gill (UK) per hour", imp::gill / hr},
        unitD{"K34", "gill (UK) per minute", imp::gill / min},
        unitD{"K35", "gill (UK) per second", imp::gill / s},
        unitD{"K36", "gill (US) per day", us::gill / day},
        unitD{"K37", "gill (US) per hour", us::gill / hr},
        unitD{"K38", "gill (US) per minute", us::gill / min},
        unitD{"K39", "gill (US) per second", us::gill / s},
        unitD{
            "K40",
            "standard acceleration of free fall",
            constants::g0.as_unit()},
        unitD{"K41", "grain per gallon (US)", i::grain / us::gallon},
        unitD{"K42", "horsepower (boiler)", power::hpS},
        unitD{"K43", "horsepower (electric)", power::hpE},
        unitD{"K45", "inch per degree Fahrenheit", in / degF},
        unitD{"K46", "inch per psi", in / pressure::psi},
        unitD{"K47", "inch per second degree Fahrenheit", in / (s * degF)},
        unitD{"K48", "inch per second psi", in / (s * pressure::psi)},
        unitD{"K49", "reciprocal cubic inch", in.pow(-3)},
        unitD{"K5", "kilovolt ampere (reactive)", kilo* electrical::VAR},
        unitD{"K50", "kilobaud", kilo* bit / s},
        unitD{"K51", "kilocalorie (mean)", kilo* energy::cal_mean},
        unitD{
            "K52",
            "kilocalorie (international table) per hour metre degree Celsius",
            kilo* energy::cal_it / (hr * m * degC)},
        unitD{"K53", "kilocalorie (thermochemical)", kilo* energy::cal_th},
        unitD{
            "K54",
            "kilocalorie (thermochemical) per minute",
            kilo* energy::cal_th / min},
        unitD{
            "K55",
            "kilocalorie (thermochemical) per second",
            kilo* energy::cal_th / s},
        unitD{"K58", "kilomole per hour", kilo* mol / hr},
        unitD{
            "K59",
            "kilomole per cubic metre kelvin",
            kilo* mol / (m.pow(3) * K)},
        unitD{"K6", "kilolitre", kilo* L},
        unitD{
            "K60",
            "kilomole per cubic metre bar",
            kilo* mol / (m.pow(3) * bar)},
        unitD{"K61", "kilomole per minute", kilo* mol / min},
        unitD{"K62", "litre per litre", pu* L},
        unitD{"K63", "reciprocal litre", L.inv()},
        unitD{
            "K64",
            "pound (avoirdupois) per degree Fahrenheit",
            av::pound / degF},
        unitD{"K65", "pound (avoirdupois) square foot", av::pound* ft.pow(2)},
        unitD{"K66", "pound (avoirdupois) per day", av::pound / day},
        unitD{"K67", "pound per foot hour", lb / (ft * hr)},
        unitD{"K68", "pound per foot second", lb / (ft * s)},
        unitD{
            "K69",
            "pound (avoirdupois) per cubic foot degree Fahrenheit",
            av::pound / (ft.pow(3) * degF)},
        unitD{
            "K70",
            "pound (avoirdupois) per cubic foot psi",
            av::pound / (ft.pow(3) * pressure::psi)},
        unitD{
            "K71",
            "pound (avoirdupois) per gallon (UK)",
            av::pound / imp::gallon},
        unitD{
            "K73",
            "pound (avoirdupois) per hour degree Fahrenheit",
            av::pound / (hr * degF)},
        unitD{
            "K74",
            "pound (avoirdupois) per hour psi",
            av::pound / (hr * pressure::psi)},
        unitD{
            "K75",
            "pound (avoirdupois) per cubic inch degree Fahrenheit",
            av::pound / (in.pow(3) * degF)},
        unitD{
            "K76",
            "pound (avoirdupois) per cubic inch psi",
            av::pound / (in.pow(3) * pressure::psi)},
        unitD{"K77", "pound (avoirdupois) per psi", av::pound / pressure::psi},
        unitD{"K78", "pound (avoirdupois) per minute", av::pound / min},
        unitD{
            "K79",
            "pound (avoirdupois) per minute degree Fahrenheit",
            av::pound / (min * degF)},
        unitD{
            "K80",
            "pound (avoirdupois) per minute psi",
            av::pound / (min * pressure::psi)},
        unitD{"K81", "pound (avoirdupois) per second", av::pound / s},
        unitD{
            "K82",
            "pound (avoirdupois) per second degree Fahrenheit",
            av::pound / (s * degF)},
        unitD{
            "K83",
            "pound (avoirdupois) per second psi",
            av::pound / (s * pressure::psi)},
        unitD{"K84", "pound per cubic yard", lb / yd.pow(3)},
        unitD{"K85", "pound-force per square foot", lbf / ft.pow(2)},
        unitD{
            "K86",
            "pound-force per square inch degree Fahrenheit",
            lbf / (in.pow(2) * degF)},
        unitD{"K87", "psi cubic inch per second", pressure::psi* in.pow(3) / s},
        unitD{"K88", "psi litre per second", pressure::psi* L / s},
        unitD{"K89", "psi cubic metre per second", pressure::psi* m.pow(3) / s},
        unitD{"K90", "psi cubic yard per second", pressure::psi* yd.pow(3) / s},
        unitD{"K91", "pound-force second per square foot", lbf* s / ft.pow(2)},
        unitD{"K92", "pound-force second per square inch", lbf* s / in.pow(2)},
        unitD{"K93", "reciprocal psi", pressure::psi.inv()},
        unitD{"K94", "quart (UK liquid) per day", imp::quart / day},
        unitD{"K95", "quart (UK liquid) per hour", imp::quart / hr},
        unitD{"K96", "quart (UK liquid) per minute", imp::quart / min},
        unitD{"K97", "quart (UK liquid) per second", imp::quart / s},
        unitD{"K98", "quart (US liquid) per day", us::quart / day},
        unitD{"K99", "quart (US liquid) per hour", us::quart / hr},
        unitD{"KA", "cake", {1.0, count, commodities::packaging::cake}},
        unitD{"KAT", "katal", katal},
        unitD{
            "KB",
            "kilocharacter",
            {1.0, kilo* count, commodities::character}},
        unitD{"KBA", "kilobar", kilo* bar},
        unitD{
            "KCC",
            "kilogram of choline chloride",
            {1.0, kg, commodities::chemicals::choline_chloride}},
        unitD{"KD", "kilogram decimal", kg},
        unitD{
            "KDW",
            "kilogram drained net weight",
            {1.0, kg, commodities::packaging::net_quantity}},
        unitD{"KEL", "kelvin", K},
        unitD{
            "KF",
            "kilopacket",
            {1000.0, count, commodities::packaging::packet}},
        unitD{"KG", "keg", {1.0, count, commodities::packaging::keg}},
        unitD{"KGM", "kilogram", kg},
        unitD{"KGS", "kilogram per second", kg / s},
        unitD{
            "KHY",
            "kilogram of hydrogen peroxide",
            {1.0, kg, commodities::hydrogen_peroxide}},
        unitD{"KHZ", "kilohertz", kilo* Hz},
        unitD{"KI", "kilogram per millimetre width", kg / mm},
        unitD{"KIC", "kilogram, including container", kg},
        unitD{"KIP", "kilogram, including inner packaging", kg},
        unitD{
            "KJ",
            "kilosegment",
            {1.0, kilo* count, commodities::packaging::segment}},
        unitD{"KJO", "kilojoule", kilo* J},
        unitD{"KL", "kilogram per metre", kg / m},
        unitD{
            "KLK",
            "lactic dry material percentage",
            {1.0, percent, commodities::packaging::dry + commodities::milk}},
        unitD{"KLX", "kilolux", kilo* lux},
        unitD{
            "KMA",
            "kilogram of methylamine",
            {1.0, kg, commodities::chemicals::methylamine}},
        unitD{"KMH", "kilometre per hour", km / hr},
        unitD{"KMK", "square kilometre", km.pow(2)},
        unitD{"KMQ", "kilogram per cubic metre", kg / m.pow(3)},
        unitD{"KMT", "kilometre", km},
        unitD{"KNI", "kilogram of nitrogen", {1.0, kg, commodities::nitrogen}},
        unitD{"KNS", "kilogram named substance", kg},
        unitD{"KNT", "knot", nautical::knot},
        unitD{
            "KO",
            "milliequivalence caustic potash per gram of product",
            {1.0,
             milli / g,
             commodities::packaging::equivalent + commodities::chemicals::KOH}},
        unitD{"KPA", "kilopascal", kilo* Pa},
        unitD{
            "KPH",
            "kilogram of potassium hydroxide (caustic potash)",
            {1.0, kg, commodities::chemicals::KOH}},
        unitD{
            "KPO",
            "kilogram of potassium oxide",
            {1.0, kg, commodities::chemicals::K2O}},
        unitD{
            "KPP",
            "kilogram of phosphorus pentoxide (phosphoric anhydride)",
            {1.0, kg, commodities::chemicals::P2O5}},
        unitD{"KR", "kiloroentgen", kilo* cgs::roentgen},
        unitD{"KS", "thousand pound per square inch", thousand* lb / in.pow(2)},
        unitD{
            "KSD",
            "kilogram of substance 90 % dry",
            {1.0, kg, commodities::packaging::dry90}},
        unitD{
            "KSH",
            "kilogram of sodium hydroxide (caustic soda)",
            {1.0, kg, commodities::chemicals::NaOH}},
        unitD{"KT", "kit", {1.0, count, commodities::packaging::kit}},
        unitD{"KTM", "kilometre", km},
        unitD{"KTN", "kilotonne", kilo* t},
        unitD{"KUR", "kilogram of uranium", {1.0, kg, commodities::uranium}},
        unitD{"KVA", "kilovolt - ampere", kilo* V* A},
        unitD{"KVR", "kilovar", kilo* electrical::VAR},
        unitD{"KVT", "kilovolt", electrical::kV},
        unitD{"KW", "kilogram per millimetre", kg / mm},
        unitD{"KWH", "kilowatt hour", electrical::kW* hr},
        unitD{
            "KWO",
            "kilogram of tungsten trioxide",
            {1.0, kg, commodities::chemicals::W03}},
        unitD{"KWT", "kilowatt", kilo* W},
        unitD{"KX", "millilitre per kilogram", mL / kg},
        unitD{"L10", "quart (US liquid) per minute", us::quart / min},
        unitD{"L11", "quart (US liquid) per second", us::quart / s},
        unitD{"L12", "metre per second kelvin", m / (s * K)},
        unitD{"L13", "metre per second bar", m / (s * bar)},
        unitD{
            "L14",
            "square metre hour degree Celsius per kilocalorie (international table)",
            m.pow(2) * hr* degC / (kilo * energy::cal_it)},
        unitD{"L15", "millipascal second per kelvin", milli* Pa* s / K},
        unitD{"L16", "millipascal second per bar", milli* Pa* s / bar},
        unitD{"L17", "milligram per cubic metre kelvin", mg / (m.pow(3) * K)},
        unitD{"L18", "milligram per cubic metre bar", mg / (m.pow(3) * bar)},
        unitD{"L19", "millilitre per litre", mL / L},
        unitD{"L2", "litre per minute", L / min},
        unitD{"L20", "reciprocal cubic millimetre", mm.pow(-3)},
        unitD{"L21", "cubic millimetre per cubic metre", mm.pow(3) / m.pow(3)},
        unitD{"L23", "mole per hour", mol / hr},
        unitD{"L24", "mole per kilogram kelvin", mol / (kg * K)},
        unitD{"L25", "mole per kilogram bar", mol / (kg * bar)},
        unitD{"L26", "mole per litre kelvin", mol / (L * K)},
        unitD{"L27", "mole per litre bar", mol / (L * bar)},
        unitD{"L28", "mole per cubic metre kelvin", mol / (m.pow(3) * K)},
        unitD{"L29", "mole per cubic metre bar", mol / (m.pow(3) * bar)},
        unitD{"L30", "mole per minute", mol / min},
        unitD{"L31", "milliroentgen aequivalent men", milli* cgs::REM},
        unitD{"L32", "nanogram per kilogram", nano* g / kg},
        unitD{"L33", "ounce (avoirdupois) per day", av::ounce / day},
        unitD{"L34", "ounce (avoirdupois) per hour", av::ounce / hr},
        unitD{"L35", "ounce (avoirdupois) per minute", av::ounce / min},
        unitD{"L36", "ounce (avoirdupois) per second", av::ounce / s},
        unitD{
            "L37",
            "ounce (avoirdupois) per gallon (UK)",
            av::ounce / imp::gallon},
        unitD{
            "L38",
            "ounce (avoirdupois) per gallon (US)",
            av::ounce / us::gallon},
        unitD{
            "L39",
            "ounce (avoirdupois) per cubic inch",
            av::ounce / in.pow(3)},
        unitD{
            "L40",
            "ounce (avoirdupois)-force",
            av::ounce* constants::g0.as_unit()},
        unitD{
            "L41",
            "ounce (avoirdupois)-force inch",
            av::ounce* constants::g0.as_unit() * in},
        unitD{"L42", "picosiemens per metre", pico* S / m},
        unitD{"L43", "peck (UK)", imp::peck},
        unitD{"L44", "peck (UK) per day", imp::peck / day},
        unitD{"L45", "peck (UK) per hour", imp::peck / hr},
        unitD{"L46", "peck (UK) per minute", imp::peck / min},
        unitD{"L47", "peck (UK) per second", imp::peck / s},
        unitD{"L48", "peck (US dry) per day", us::dry::peck / day},
        unitD{"L49", "peck (US dry) per hour", us::dry::peck / hr},
        unitD{"L50", "peck (US dry) per minute", us::dry::peck / min},
        unitD{"L51", "peck (US dry) per second", us::dry::peck / s},
        unitD{"L52", "psi per psi", pu* pressure::psi},
        unitD{"L53", "pint (UK) per day", imp::pint / day},
        unitD{"L54", "pint (UK) per hour", imp::pint / hr},
        unitD{"L55", "pint (UK) per minute", imp::pint / min},
        unitD{"L56", "pint (UK) per second", imp::pint / s},
        unitD{"L57", "pint (US liquid) per day", us::pint / day},
        unitD{"L58", "pint (US liquid) per hour", us::pint / hr},
        unitD{"L59", "pint (US liquid) per minute", us::pint / min},
        unitD{"L60", "pint (US liquid) per second", us::pint / s},
        unitD{"L61", "pint (US dry)", us::dry::pint},
        unitD{"L62", "quart (US dry)", us::dry::quart},
        unitD{"L63", "slug per day", imp::slug / day},
        unitD{"L64", "slug per foot second", imp::slug / (ft * s)},
        unitD{"L65", "slug per cubic foot", imp::slug / ft.pow(3)},
        unitD{"L66", "slug per hour", imp::slug / hr},
        unitD{"L67", "slug per minute", imp::slug / min},
        unitD{"L68", "slug per second", imp::slug / s},
        unitD{"L69", "tonne per kelvin", t / K},
        unitD{"L70", "tonne per bar", t / bar},
        unitD{"L71", "tonne per day", t / day},
        unitD{"L72", "tonne per day kelvin", t / (day * K)},
        unitD{"L73", "tonne per day bar", t / (day * bar)},
        unitD{"L74", "tonne per hour kelvin", t / (hr * K)},
        unitD{"L75", "tonne per hour bar", t / (hr * bar)},
        unitD{"L76", "tonne per cubic metre kelvin", t / (m.pow(3) * K)},
        unitD{"L77", "tonne per cubic metre bar", t / (m.pow(3) * bar)},
        unitD{"L78", "tonne per minute", t / min},
        unitD{"L79", "tonne per minute kelvin", t / (min * K)},
        unitD{"L80", "tonne per minute bar", t / (min * bar)},
        unitD{"L81", "tonne per second", t / s},
        unitD{"L82", "tonne per second kelvin", t / (s * K)},
        unitD{"L83", "tonne per second bar", t / (s * bar)},
        unitD{"L84", "ton (UK shipping)", {42.0, ft.pow(3)}},
        unitD{"L85", "ton long per day", av::longton / day},
        unitD{"L86", "ton (US shipping)", {40.0, ft.pow(3)}},
        unitD{"L87", "ton short per degree Fahrenheit", av::ton / degF},
        unitD{"L88", "ton short per day", av::ton / day},
        unitD{
            "L89",
            "ton short per hour degree Fahrenheit",
            av::ton / (hr * degF)},
        unitD{"L90", "ton short per hour psi", av::ton / (hr * pressure::psi)},
        unitD{"L91", "ton short per psi", av::ton / pressure::psi},
        unitD{"L92", "ton (UK long) per cubic yard", imp::ton / yd.pow(3)},
        unitD{"L93", "ton (US short) per cubic yard", av::ton / yd.pow(3)},
        unitD{"L94", "ton-force", av::ton* constants::g0.as_unit()},
        unitD{"L95", "common year", time::year},
        unitD{"L96", "sidereal year", time::syr},
        unitD{"L98", "yard per degree Fahrenheit", yd / degF},
        unitD{"L99", "yard per psi", yd / pressure::psi},
        unitD{"LA", "pound per cubic inch", lb / in.pow(3)},
        unitD{
            "LAC",
            "lactose excess percentage",
            {1.0,
             percent,
             commodities::packaging::excess + commodities::lactose}},
        unitD{"LBR", "pound", lb},
        unitD{"LBT", "troy pound (US)", troy::pound},
        unitD{
            "LC",
            "linear centimetre",
            {1.0, cm, commodities::packaging::linear}},
        unitD{"LD", "litre per day", L / day},
        unitD{"LE", "lite", {1.0, count, commodities::packaging::lite}},
        unitD{"LEF", "leaf", {1.0, count, commodities::packaging::leaf}},
        unitD{"LF", "linear foot", {1.0, ft, commodities::packaging::linear}},
        unitD{"LH", "labour hour", {1.0, hr, commodities::labor}},
        unitD{"LI", "linear inch", {1.0, in, commodities::packaging::linear}},
        unitD{
            "LJ",
            "large spray",
            {1.0, count, commodities::packaging::large_spray}},
        unitD{"LK", "link", {1.0, count, commodities::packaging::length}},
        unitD{"LM", "linear metre", {1.0, m, commodities::packaging::linear}},
        unitD{
            "LN",
            "length",
            {1.0, count, commodities::packaging::large_spray}},
        unitD{
            "LO",
            "lot  [unit of procurement]",
            {1.0, count, commodities::packaging::lot}},
        unitD{"LP", "liquid pound", {1.0, lb, commodities::packaging::liquid}},
        unitD{
            "LPA",
            "litre of pure alcohol",
            {1.0, L, commodities::chemicals::alcohol}},
        unitD{"LR", "layer", {1.0, count, commodities::packaging::layer}},
        unitD{"LS", "lump sum", {1.0, currency, commodities::packaging::lump}},
        unitD{"LTN", "ton (UK)", av::longton},
        unitD{"LTR", "litre", L},
        unitD{"LUB", "metric ton, lubricating oil", t},
        unitD{"LUM", "lumen", lm},
        unitD{"LUX", "lux", lux},
        unitD{
            "LX",
            "linear yard per pound",
            precise_unit(1.0, yd, commodities::packaging::linear) / lb},
        unitD{"LY", "linear yard", {1.0, yd, commodities::packaging::linear}},
        unitD{
            "M0",
            "magnetic tape",
            {1.0,
             count,
             commodities::generateHarmonizedCode(85, 23, 29, 15, false)}},
        unitD{"M1", "milligram per litre", mg / L},
        unitD{"M10", "reciprocal cubic yard", yd.pow(-3)},
        unitD{"M11", "cubic yard per degree Fahrenheit", yd.pow(3) / degF},
        unitD{"M12", "cubic yard per day", yd.pow(3) / day},
        unitD{"M13", "cubic yard per hour", yd.pow(3) / hr},
        unitD{"M14", "cubic yard per psi", yd.pow(3) / pressure::psi},
        unitD{"M15", "cubic yard per minute", yd.pow(3) / min},
        unitD{"M16", "cubic yard per second", yd.pow(3) / s},
        unitD{"M17", "kilohertz metre", kilo* Hz* m},
        unitD{"M18", "gigahertz metre", giga* Hz* m},
        unitD{"M19", "Beaufort", special::beaufort},
        unitD{"M20", "reciprocal megakelvin", (mega * K).inv()},
        unitD{
            "M21",
            "reciprocal kilovolt - ampere reciprocal hour",
            (kilo * V * A).inv() * hr.inv()},
        unitD{
            "M22",
            "millilitre per square centimetre minute",
            mL / (cm.pow(2) * min)},
        unitD{"M23", "newton per centimetre", N / cm},
        unitD{"M24", "ohm kilometre", ohm* km},
        unitD{"M25", "percent per degree Celsius", percent / degC},
        unitD{"M26", "gigaohm per metre", giga* ohm / m},
        unitD{"M27", "megahertz metre", mega* Hz* m},
        unitD{"M29", "kilogram per kilogram", pu* kg},
        unitD{
            "M30",
            "reciprocal volt - ampere reciprocal second",
            (V * A).inv() * s.inv()},
        unitD{"M31", "kilogram per kilometre", kg / km},
        unitD{"M32", "pascal second per litre", Pa* s / L},
        unitD{"M33", "millimole per litre", milli* mol / L},
        unitD{"M34", "newton metre per square metre", N* m / m.pow(2)},
        unitD{"M35", "millivolt - ampere", electrical::mV* A},
        unitD{"M36", "month{30-day}", {30.0, day}},
        unitD{"M37", "actual/360", {360.0, day}},
        unitD{"M38", "kilometre per second squared", km / s.pow(2)},
        unitD{"M39", "centimetre per second squared", cm / s.pow(2)},
        unitD{"M4", "monetary value", currency},
        unitD{"M40", "yard per second squared", yd / s.pow(2)},
        unitD{"M41", "millimetre per second squared", mm / s.pow(2)},
        unitD{"M42", "mile (statute) per second squared", mile / s.pow(2)},
        unitD{"M43", "mil", i::mil},
        unitD{"M44", "revolution", {constants::tau, precise::rad}},
        unitD{
            "M45",
            "degree [unit of angle] per second squared",
            deg / s.pow(2)},
        unitD{"M46", "revolution per minute", other::rpm},
        unitD{"M47", "circular mil", i::circ_mil},
        unitD{"M48", "square mile (U.S. survey)", us::mile.pow(2)},
        unitD{"M49", "chain (U.S. survey)", us::chain},
        unitD{"M5", "microcurie", micro* cgs::curie},
        unitD{"M50", "furlong", us::furlong},
        unitD{"M51", "foot (U.S. survey)", us::foot},
        unitD{"M52", "mile (U.S. survey)", us::mile},
        unitD{"M53", "metre per pascal", m / Pa},
        unitD{"M55", "metre per radiant", m / rad},
        unitD{"M56", "shake", {1e-8, s}},
        unitD{"M57", "mile per minute", mile / min},
        unitD{"M58", "mile per second", mile / s},
        unitD{"M59", "metre per second pascal", m / (s * Pa)},
        unitD{"M60", "metre per hour", m / hr},
        unitD{"M61", "inch per year", in / yr},
        unitD{"M62", "kilometre per second", km / s},
        unitD{"M63", "inch per minute", in / min},
        unitD{"M64", "yard per second", yd / s},
        unitD{"M65", "yard per minute", yd / min},
        unitD{"M66", "yard per hour", yd / hr},
        unitD{"M67", "acre-foot (U.S. survey)", acre* us::foot},
        unitD{"M68", "cord", us::cord},
        unitD{"M69", "cubic mile UK", imp::mile.pow(3)},
        unitD{"M7", "micro-inch", micro* in},
        unitD{"M70", "ton, register", av::ton},
        unitD{"M71", "cubic metre per pascal", m.pow(3) / Pa},
        unitD{"M72", "bel", log::bel},
        unitD{"M73", "kilogram per cubic metre pascal", kg / (m.pow(3) * Pa)},
        unitD{"M74", "kilogram per pascal", kg / Pa},
        unitD{"M75", "kilopound-force", kilo* lbf},
        unitD{"M76", "poundal", av::poundal},
        unitD{"M77", "kilogram metre per second squared", kg* m / s.pow(2)},
        unitD{"M78", "pond", precise::gm::pond},
        unitD{"M79", "square foot per hour", ft.pow(2) / hr},
        unitD{"M80", "stokes per pascal", cgs::stokes / Pa},
        unitD{"M81", "square centimetre per second", cm.pow(2) / s},
        unitD{"M82", "square metre per second pascal", m.pow(2) / (s * Pa)},
        unitD{"M83", "denier", textile::denier},
        unitD{"M84", "pound per yard", lb / yd},
        unitD{"M85", "ton, assay", {0.02916667, kg}},
        unitD{"M86", "pfund", {0.5, kg}},
        unitD{"M87", "kilogram per second pascal", kg / (s * Pa)},
        unitD{"M88", "tonne per month", mass::tonne / time::moj},
        unitD{"M89", "tonne per year", mass::tonne / time::aj},
        unitD{
            "M9",
            "million Btu per 1000 cubic foot",
            mega* btu / (thousand * ft.pow(3))},
        unitD{"M90", "kilopound per hour", kilo* lbf / hr},
        unitD{"M91", "pound per pound", pu* lb},
        unitD{"M92", "pound-force foot", lbf* ft},
        unitD{"M93", "newton metre per radian", N* m / rad},
        unitD{"M94", "kilogram metre", kg* m},
        unitD{"M95", "poundal foot", av::poundal* ft},
        unitD{"M96", "poundal inch", av::poundal* in},
        unitD{"M97", "dyne metre", cgs::dyn* m},
        unitD{"M98", "kilogram centimetre per second", kg* cm / s},
        unitD{"M99", "gram centimetre per second", g* cm / s},
        unitD{"MA", "machine per unit", {1.0, pu* count, commodities::machine}},
        unitD{"MAH", "megavolt ampere reactive hour", electrical::MVAR* hr},
        unitD{"MAL", "megalitre", mega* L},
        unitD{"MAM", "megametre", mega* m},
        unitD{"MAR", "megavar", electrical::MVAR},
        unitD{"MAW", "megawatt", electrical::MW},
        unitD{
            "MBE",
            "thousand standard brick equivalent",
            {1000.0, count, commodities::brick}},
        unitD{"MBF", "thousand board foot", thousand* i::board_foot},
        unitD{"MBR", "millibar", milli* bar},
        unitD{"MC", "microgram", micro* g},
        unitD{"MCU", "millicurie", milli* cgs::curie},
        unitD{
            "MD",
            "air dry metric ton",
            precise_unit(mass::tonne, commodities::packaging::dry)},
        unitD{
            "MF",
            "milligram per square foot per side",
            {1.0,
             mg / ft.pow(2) / precise_unit(one, commodities::packaging::side)}},
        unitD{"MGM", "milligram", mg},
        unitD{"MHZ", "megahertz", mega* Hz},
        unitD{"MIK", "square mile ", mile.pow(2)},
        unitD{"MIL", "thousand", thousand* count},
        unitD{"MIN", "minute [unit of time]", min},
        unitD{"MIO", "million", mega* count},
        unitD{"MIU", "million international unit", mega* laboratory::IU},
        unitD{"MK", "milligram per square inch", mg / in.pow(2)},
        unitD{"MLD", "milliard", giga* count},
        unitD{"MLT", "millilitre", mL},
        unitD{"MMK", "square millimetre", mm.pow(2)},
        unitD{"MMQ", "cubic millimetre", mm.pow(3)},
        unitD{"MMT", "millimetre", mm},
        unitD{"MND", "kilogram, dry weight", kg},
        unitD{"MON", "month", time::moj},
        unitD{"MPA", "megapascal", mega* Pa},
        unitD{"MQ", "thousand metre", thousand* m},
        unitD{"MQH", "cubic metre per hour", m.pow(3) / hr},
        unitD{"MQS", "cubic metre per second", m.pow(3) / s},
        unitD{"MSK", "metre per second squared", m / s.pow(2)},
        unitD{"MT", "mat", {1.0, count, commodities::packaging::mat}},
        unitD{"MTK", "square metre", m.pow(2)},
        unitD{"MTQ", "cubic metre", m.pow(3)},
        unitD{"MTR", "metre", m},
        unitD{"MTS", "metre per second", m / s},
        unitD{"MV", "number of mults", count},
        unitD{"MVA", "megavolt - ampere", mega* V* A},
        unitD{"MWH", "megawatt hour", mega* W* hr},
        unitD{"N1", "pen calorie", cal},
        unitD{"N10", "pound foot per second", lb* ft / s},
        unitD{"N11", "pound inch per second", lb* in / s},
        unitD{"N12", "Pferdestaerke", power::hpM},
        unitD{"N13", "centimetre of mercury (0 degC)", {1333.22, Pa}},
        unitD{"N14", "centimetre of water (4 degC)", {98.0637795, Pa}},
        unitD{"N15", "foot of water (39.2 degF)", {2988.98400, Pa}},
        unitD{"N16", "inch of mercury (32 degF)", {3383.93102, Pa}},
        unitD{"N17", "inch of mercury (60 degF)", {3376.84789, Pa}},
        unitD{"N18", "inch of water (39.2 degF)", {249.082000, Pa}},
        unitD{"N19", "inch of water (60 degF)", {248.840000, Pa}},
        unitD{
            "N2",
            "number of lines",
            {1.0, count, commodities::linesofservice}},
        unitD{
            "N20",
            "kip per square inch",
            precise::kilo* precise::lbf / in.pow(2)},
        unitD{"N21", "poundal per square foot", av::poundal / ft.pow(2)},
        unitD{
            "N22",
            "ounce (avoirdupois) per square inch",
            av::ounce / in.pow(2)},
        unitD{"N23", "conventional metre of water", thousand* pressure::mmH2O},
        unitD{"N24", "gram per square millimetre", g / mm.pow(2)},
        unitD{"N25", "pound per square yard", lb / yd.pow(2)},
        unitD{"N26", "poundal per square inch", av::poundal / in.pow(2)},
        unitD{"N27", "foot to the fourth power", ft.pow(4)},
        unitD{"N28", "cubic decimetre per kilogram", (deci * m).pow(3) / kg},
        unitD{"N29", "cubic foot per pound", ft.pow(3) / lb},
        unitD{"N3", "print point", typographic::printers::point},
        unitD{"N30", "cubic inch per pound", in.pow(3) / lb},
        unitD{"N31", "kilonewton per metre", kilo* N / m},
        unitD{"N32", "poundal per inch", av::poundal / in},
        unitD{"N33", "pound-force per yard", lbf / yd},
        unitD{
            "N34",
            "poundal second per square foot",
            av::poundal* s / ft.pow(2)},
        unitD{"N35", "poise per pascal", cgs::poise / Pa},
        unitD{"N36", "newton second per square metre", N* s / m.pow(2)},
        unitD{"N37", "kilogram per metre second", kg / (m * s)},
        unitD{"N38", "kilogram per metre minute", kg / (m * min)},
        unitD{"N39", "kilogram per metre day", kg / (m * day)},
        unitD{"N40", "kilogram per metre hour", kg / (m * hr)},
        unitD{"N41", "gram per centimetre second", g / (cm * s)},
        unitD{
            "N42",
            "poundal second per square inch",
            av::poundal* s / in.pow(2)},
        unitD{"N43", "pound per foot minute", lb / (ft * min)},
        unitD{"N44", "pound per foot day", lb / (ft * day)},
        unitD{"N45", "cubic metre per second pascal", m.pow(3) / (s * Pa)},
        unitD{"N46", "foot poundal", ft* av::poundal},
        unitD{"N47", "inch poundal", in* av::poundal},
        unitD{"N48", "watt per square centimetre", W / cm.pow(2)},
        unitD{"N49", "watt per square inch", W / in.pow(2)},
        unitD{
            "N50",
            "British thermal unit (international table) per square foot hour",
            energy::btu_it / (ft.pow(2) * hr)},
        unitD{
            "N51",
            "British thermal unit (thermochemical) per square foot hour",
            energy::btu_th / (ft.pow(2) * hr)},
        unitD{
            "N52",
            "British thermal unit (thermochemical) per square foot minute",
            energy::btu_th / (ft.pow(2) * min)},
        unitD{
            "N53",
            "British thermal unit (international table) per square foot second",
            energy::btu_it / (ft.pow(2) * s)},
        unitD{
            "N54",
            "British thermal unit (thermochemical) per square foot second",
            energy::btu_th / (ft.pow(2) * s)},
        unitD{
            "N55",
            "British thermal unit (international table) per square inch second",
            energy::btu_it / (in.pow(2) * s)},
        unitD{
            "N56",
            "calorie (thermochemical) per square centimetre minute",
            energy::cal_th / (cm.pow(2) * min)},
        unitD{
            "N57",
            "calorie (thermochemical) per square centimetre second",
            energy::cal_th / (cm.pow(2) * s)},
        unitD{
            "N58",
            "British thermal unit (international table) per cubic foot",
            energy::btu_it / ft.pow(3)},
        unitD{
            "N59",
            "British thermal unit (thermochemical) per cubic foot",
            energy::btu_th / ft.pow(3)},
        unitD{
            "N60",
            "British thermal unit (international table) per degree Fahrenheit",
            energy::btu_it / degF},
        unitD{
            "N61",
            "British thermal unit (thermochemical) per degree Fahrenheit",
            energy::btu_th / degF},
        unitD{
            "N62",
            "British thermal unit (international table) per degree Rankine",
            energy::btu_it / temperature::degR},
        unitD{
            "N63",
            "British thermal unit (thermochemical) per degree Rankine",
            energy::btu_th / temperature::degR},
        unitD{
            "N64",
            "British thermal unit (thermochemical) per pound degree Rankine",
            energy::btu_th / (temperature::degR * lb)},
        unitD{
            "N65",
            "kilocalorie (international table) per gram kelvin",
            kilo* energy::cal_it / (g * K)},
        unitD{"N66", "British thermal unit (39 degF)", energy::btu_39},
        unitD{"N67", "British thermal unit (59 degF)", energy::btu_59},
        unitD{"N68", "British thermal unit (60 degF)", energy::btu_60},
        unitD{"N69", "calorie (20 degC)", energy::cal_20},
        unitD{"N70", "quad", {1e15, energy::btu_it}},
        unitD{"N71", "therm (EC)", energy::therm_ec},
        unitD{"N72", "therm (US)", energy::therm_us},
        unitD{
            "N73",
            "British thermal unit (thermochemical) per pound",
            energy::btu_th / lb},
        unitD{
            "N74",
            "British thermal unit (international table) per hour square foot degree Fahrenheit",
            energy::btu_it / (hr * ft.pow(2) * degF)},
        unitD{
            "N75",
            "British thermal unit (thermochemical) per hour square foot degree Fahrenheit",
            energy::btu_th / (hr * ft.pow(2) * degF)},
        unitD{
            "N76",
            "British thermal unit (international table) per second square foot degree Fahrenheit",
            energy::btu_it / (s * ft.pow(2) * degF)},
        unitD{
            "N77",
            "British thermal unit (thermochemical) per second square foot degree Fahrenheit",
            energy::btu_th / (s * ft.pow(2) * degF)},
        unitD{
            "N78",
            "kilowatt per square metre kelvin",
            kilo* W / (m.pow(2) * K)},
        unitD{"N79", "kelvin per pascal", K / Pa},
        unitD{"N80", "watt per metre degree Celsius", W / (m * degC)},
        unitD{"N81", "kilowatt per metre kelvin", kilo* W / (m * K)},
        unitD{
            "N82",
            "kilowatt per metre degree Celsius",
            electrical::kW / (m * degC)},
        unitD{"N83", "metre per degree Celcius metre", m / (degC * m)},
        unitD{
            "N84",
            "degree Fahrenheit hour per British thermal unit (international table)",
            degF* hr / energy::btu_it},
        unitD{
            "N85",
            "degree Fahrenheit hour per British thermal unit (thermochemical)",
            degF* hr / energy::btu_th},
        unitD{
            "N86",
            "degree Fahrenheit second per British thermal unit (international table)",
            degF* s / energy::btu_it},
        unitD{
            "N87",
            "degree Fahrenheit second per British thermal unit (thermochemical)",
            degF* s / energy::btu_th},
        unitD{
            "N88",
            "degree Fahrenheit hour square foot per British thermal unit "
            "(international table) inch",
            degF* hr* ft.pow(2) / energy::btu_it / in},
        unitD{
            "N89",
            "degree Fahrenheit hour square foot per British thermal unit (thermochemical) inch",
            degF* hr* ft.pow(2) / energy::btu_th / in},
        unitD{"N90", "kilofarad", kilo* F},
        unitD{"N91", "reciprocal joule", J.inv()},
        unitD{"N92", "picosiemens", pico* S},
        unitD{"N93", "ampere per pascal", A / Pa},
        unitD{"N94", "franklin", precise::cgs::statC_charge},
        unitD{"N95", "ampere minute", A* min},
        unitD{"N96", "biot", cgs::biot},
        unitD{"N97", "gilbert", cgs::gilbert},
        unitD{"N98", "volt per pascal", V / Pa},
        unitD{"N99", "picovolt", pico* V},
        unitD{"NA", "milligram per kilogram", mg / kg},
        unitD{
            "NAR",
            "number of articles",
            {1.0, count, commodities::packaging::article}},
        unitD{"NB", "barge", {1.0, count, commodities::packaging::barge}},
        unitD{
            "NBB",
            "number of bobbins",
            {1.0, count, commodities::packaging::bobbin}},
        unitD{"NC", "car", {1.0, count, commodities::packaging::car}},
        unitD{"NCL", "number of cells", {1.0, count, commodities::cell}},
        unitD{
            "ND",
            "net barrel",
            {1.0, us::barrel, commodities::packaging::net_quantity}},
        unitD{"NE", "net litre", {1.0, L, commodities::packaging::net}},
        unitD{"NEW", "newton", N},
        unitD{"NF", "message", {1.0, count, commodities::message}},
        unitD{"NG", "net gallon (us)", {1.0, gal, commodities::packaging::net}},
        unitD{"NH", "message hour", {1.0, hr, commodities::message}},
        unitD{
            "NI",
            "net imperial gallon",
            {1.0, imp::gallon, commodities::packaging::net}},
        unitD{"NIL", "nil", {1.0, count, commodities::blank}},
        unitD{"NIU", "international units", laboratory::IU},
        unitD{"NJ", "number of screens", {1.0, count, commodities::screen}},
        unitD{"NL", "load", {1.0, count, commodities::packaging::load}},
        unitD{"NMI", "nautical mile", nautical::mile},
        unitD{
            "NMP",
            "number of packs",
            {1.0, count, commodities::packaging::pack}},
        unitD{"NN", "train", {1.0, count, commodities::packaging::train}},
        unitD{
            "NPL",
            "number of parcels",
            {1.0, count, commodities::packaging::parcel}},
        unitD{"NPR", "number of pairs", {2.0, count}},
        unitD{
            "NPT",
            "number of parts",
            {1.0, count, commodities::packaging::part}},
        unitD{"NQ", "mho", ohm.inv()},
        unitD{"NR", "micromho", micro* ohm.inv()},
        unitD{
            "NRL",
            "number of rolls",
            {1.0, count, commodities::packaging::roll}},
        unitD{
            "NT",
            "net ton",
            {1.0, ton, commodities::packaging::net_quantity}},
        unitD{
            "NTT",
            "net register ton",
            {1.0, ton, commodities::packaging::net_register}},
        unitD{"NU", "newton metre", N* m},
        unitD{"NV", "vehicle", {1.0, count, commodities::vehicle}},
        unitD{"NX", "part per thousand", milli* count},
        unitD{
            "NY",
            "pound per air dry metric ton",
            lb / precise_unit(mass::tonne, commodities::packaging::dry)},
        unitD{"OA", "panel", {1.0, count, commodities::packaging::panel}},
        unitD{"ODE", "ozone depletion equivalent", climate::odp},
        unitD{"OHM", "ohm", ohm},
        unitD{"ON", "ounce per square yard", oz / yd.pow(2)},
        unitD{"ONZ", "ounce (avoirdupois)", av::ounce},
        unitD{"OP", "two pack", {2, count}},
        unitD{"OT", "overtime hour", {1.0, count, commodities::overtime}},
        unitD{"OZ", "ounce av", av::ounce},
        unitD{"OZA", "fluid ounce (US)", us::floz},
        unitD{"OZI", "fluid ounce (UK)", imp::floz},
        unitD{
            "P0",
            "page - electronic",
            {1.0, count, commodities::packaging::page_electronic}},
        unitD{"P1", "percent", percent},
        unitD{"P10", "coulomb per metre", C / m},
        unitD{"P11", "kiloweber", kilo* weber},
        unitD{"P12", "gamma{geo}", precise::nano* precise::T},
        unitD{"P13", "kilotesla", kilo* T},
        unitD{"P14", "joule per second", J / s},
        unitD{"P15", "joule per minute", J / min},
        unitD{"P16", "joule per hour", J / hr},
        unitD{"P17", "joule per day", J / day},
        unitD{"P18", "kilojoule per second", kilo* J / s},
        unitD{"P19", "kilojoule per minute", kilo* J / min},
        unitD{"P2", "pound per foot", lb / ft},
        unitD{"P20", "kilojoule per hour", kilo* J / hr},
        unitD{"P21", "kilojoule per day", kilo* J / day},
        unitD{"P22", "nanoohm", nano* ohm},
        unitD{"P23", "ohm circular-mil per foot", ohm* i::circ_mil / ft},
        unitD{"P24", "kilohenry", kilo* henry},
        unitD{"P25", "lumen per square foot", lm / ft.pow(2)},
        unitD{"P26", "phot", cgs::phot},
        unitD{"P27", "footcandle", precise::lumen / precise::ft.pow(2)},
        unitD{"P28", "candela per square inch", cd / in.pow(2)},
        unitD{
            "P29",
            "footlambert",
            {1.0 / constants::pi, precise::cd / precise::ft.pow(2)}},
        unitD{"P3", "three pack", {3, one}},
        unitD{"P30", "lambert", cgs::lambert},
        unitD{"P31", "stilb", cgs::stilb},
        unitD{"P32", "candela per square foot", cd / ft.pow(2)},
        unitD{"P33", "kilocandela", kilo* cd},
        unitD{"P34", "millicandela", milli* cd},
        unitD{"P35", "Hefner-Kerze", {0.903, cd}},
        unitD{"P36", "international candle", precise::lm / precise::sr},
        unitD{
            "P37",
            "British thermal unit (international table) per square foot",
            energy::btu_it / (ft.pow(2))},
        unitD{
            "P38",
            "British thermal unit (thermochemical) per square foot",
            energy::btu_th / (ft.pow(2))},
        unitD{
            "P39",
            "calorie (thermochemical) per square centimetre",
            energy::cal_th / cm.pow(2)},
        unitD{"P4", "four pack", {4, count}},
        unitD{"P40", "langley", cgs::langley},
        unitD{"P41", "decade (logarithmic)", log::logbase10},
        unitD{"P42", "pascal squared second", Pa.pow(2) * s},
        unitD{"P43", "bel per metre", log::bel / m},
        unitD{"P44", "pound mole", lb* mol},
        unitD{"P45", "pound mole per second", lb* mol / s},
        unitD{"P46", "pound mole per minute", lb* mol / min},
        unitD{"P47", "kilomole per kilogram", kilo* mol / kg},
        unitD{"P48", "pound mole per pound", lb* mol / lb},
        unitD{"P49", "newton square metre per ampere", N* m.pow(2) / A},
        unitD{"P5", "five pack", {5, one}},
        unitD{"P50", "weber metre", Wb* m},
        unitD{"P51", "mol per kilogram pascal", mol / (kg * Pa)},
        unitD{"P52", "mol per cubic metre pascal", mol / (m.pow(3) * Pa)},
        unitD{"P53", "unit pole", cgs::unitpole},
        unitD{"P54", "milligray per second", milli* gray / s},
        unitD{"P55", "microgray per second", micro* gray / s},
        unitD{"P56", "nanogray per second", nano* gray / s},
        unitD{"P57", "gray per minute", gray / min},
        unitD{"P58", "milligray per minute", milli* gray / min},
        unitD{"P59", "microgray per minute", micro* gray / min},
        unitD{"P6", "six pack", {6, one}},
        unitD{"P60", "nanogray per minute", gray / min},
        unitD{"P61", "gray per hour", gray / hr},
        unitD{"P62", "milligray per hour", milli* gray / hr},
        unitD{"P63", "microgray per hour", micro* gray / hr},
        unitD{"P64", "nanogray per hour", nano* gray / hr},
        unitD{"P65", "sievert per second", sievert / s},
        unitD{"P66", "millisievert per second", milli* sievert / s},
        unitD{"P67", "microsievert per second", micro* sievert / s},
        unitD{"P68", "nanosievert per second", nano* sievert / s},
        unitD{"P69", "rem per second", cgs::REM / s},
        unitD{"P7", "seven pack", {7, one}},
        unitD{"P70", "sievert per hour", sievert / hr},
        unitD{"P71", "millisievert per hour", milli* sievert / hr},
        unitD{"P72", "microsievert per hour", micro* sievert / hr},
        unitD{"P73", "nanosievert per hour", nano* sievert / hr},
        unitD{"P74", "sievert per minute", sievert / min},
        unitD{"P75", "millisievert per minute", milli* sievert / min},
        unitD{"P76", "microsievert per minute", micro* sievert / min},
        unitD{"P77", "nanosievert per minute", nano* sievert / min},
        unitD{"P78", "reciprocal square inch", in.pow(2).inv()},
        unitD{"P79", "pascal square metre per kilogram", Pa* m.pow(2) / kg},
        unitD{"P8", "eight pack", {8, one}},
        unitD{"P80", "millipascal per metre", milli* Pa / m},
        unitD{"P81", "kilopascal per metre", kilo* Pa / m},
        unitD{"P82", "hectopascal per metre", hecto* Pa / m},
        unitD{"P83", "standard atmosphere per metre", pressure::atm / m},
        unitD{"P84", "technical atmosphere per metre", pressure::att / m},
        unitD{"P85", "torr per metre", pressure::torr / m},
        unitD{"P86", "psi per inch", pressure::psi / in},
        unitD{
            "P87",
            "cubic metre per second square metre",
            m.pow(3) / (s * m.pow(2))},
        unitD{"P88", "rhe", {10.0, m* s / kg}},
        unitD{"P89", "pound-force foot per inch", lbf* ft / in},
        unitD{"P9", "nine pack", {9, one}},
        unitD{"P90", "pound-force inch per inch", lbf* in / in},
        unitD{"P91", "perm (0 degC)", {5.72135e-11, kg / (m.pow(2) * Pa * s)}},
        unitD{"P92", "perm (23 degC)", {5.74525e-11, kg / (m.pow(2) * Pa * s)}},
        unitD{"P93", "byte per second", data::byte / s},
        unitD{"P94", "kilobyte per second", data::kB / s},
        unitD{"P95", "megabyte per second", data::MB / s},
        unitD{"P96", "reciprocal volt", V.inv()},
        unitD{"P97", "reciprocal radian", rad.inv()},
        unitD{"P98", "pascal to the power sum of stoichiometric numbers", Pa},
        unitD{
            "P99",
            "mole per cubic metre to the power sum of stoichiometric numbers",
            mol / m.pow(3)},
        unitD{"PA", "packet", {1.0, count, commodities::packaging::packet}},
        unitD{"PAL", "pascal", Pa},
        unitD{"PB", "pair inch", {2, count* in}},
        unitD{"PD", "pad", {1.0, count, commodities::packaging::pad}},
        unitD{
            "PE",
            "pound equivalent",
            {10.0, lb, commodities::packaging::equivalent}},
        unitD{
            "PF",
            "pallet (lift)",
            {1.0, count, commodities::packaging::pallet}},
        unitD{
            "PFL",
            "proof litre",
            {1.0,
             L,
             commodities::packaging::equivalent +
                 commodities::chemicals::alcohol}},
        unitD{"PG", "plate", {1.0, count, commodities::packaging::plate}},
        unitD{
            "PGL",
            "proof gallon",
            {1.0,
             gal,
             commodities::packaging::equivalent +
                 commodities::chemicals::alcohol}},
        unitD{"PI", "pitch", {1.0, count / in, commodities::character}},
        unitD{"PK", "pack", {1.0, count, commodities::packaging::pack}},
        unitD{"PL", "pail", {1.0, count, commodities::packaging::pail}},
        unitD{
            "PLA",
            "degree Plato",
            {1.0, generate_custom_unit(112), commodities::sugar}},
        unitD{"PM", "pound percentage", lb* percent},
        unitD{
            "PN",
            "pound net",
            {1.0, lb, commodities::packaging::net_quantity}},
        unitD{"PO", "pound per inch of length", lb / in},
        unitD{
            "PQ",
            "page per inch",
            {1.0, count / in, commodities::packaging::page}},
        unitD{"PR", "pair", {2, count}},
        unitD{"PS", "pound-force per square inch", pressure::psi},
        unitD{"PT", "pint (US)", us::pint},
        unitD{"PTD", "dry pint (US)", us::dry::pint},
        unitD{"PTI", "pint (UK)", imp::pint},
        unitD{"PTL", "liquid pint (US)", us::pint},
        unitD{
            "PU",
            "tray / tray pack",
            {1.0, count, commodities::packaging::tray}},
        unitD{"PV", "half pint (US)", half* us::pint},
        unitD{"PW", "pound per inch of width", lb / in},
        unitD{"PY", "peck dry (US)", us::dry::peck},
        unitD{"PZ", "peck dry (UK)", imp::peck},
        unitD{"Q10", "joule per tesla", J / T},
        unitD{"Q11", "erlang", generate_custom_unit(49)},
        unitD{"Q12", "octet", precise_unit(8.0, precise::count)},
        unitD{"Q13", "octet per second", precise_unit(8.0, precise::count) / s},
        unitD{"Q14", "shannon", data::shannon},
        unitD{"Q15", "hartley", data::hartley},
        unitD{"Q16", "natural unit of information", data::bit_s},
        unitD{"Q17", "shannon per second", data::shannon / s},
        unitD{"Q18", "hartley per second", data::hartley / s},
        unitD{"Q19", "natural unit of information per second", data::bit_s / s},
        unitD{"Q20", "second per kilogramm", s / kg},
        unitD{"Q21", "watt square metre", W* m.pow(2)},
        unitD{"Q22", "second per radian cubic metre", s / (rad * m.pow(3))},
        unitD{"Q23", "weber to the power minus one", weber.inv()},
        unitD{"Q24", "reciprocal inch", in.inv()},
        unitD{"Q25", "dioptre", clinical::diopter},
        unitD{"Q26", "one per one", pu* count},
        unitD{"Q27", "newton metre per metre", N* m / m},
        unitD{
            "Q28",
            "kilogram per square metre pascal second",
            kg / (m.pow(2) * Pa * s)},
        unitD{"Q3", "meal", {1.0, count, commodities::meal}},
        unitD{
            "QA",
            "page - facsimile",
            {1.0, count, commodities::packaging::page}},
        unitD{"QAN", "quarter year", {0.25, yr}},
        unitD{
            "QB",
            "page - hardcopy",
            {1.0, count, commodities::packaging::page + commodities::paper}},
        unitD{"QD", "quarter dozen", {3, one}},
        unitD{"QH", "quarter hour", {0.25, hr}},
        unitD{"QK", "quarter kilogram", {0.25, kg}},
        unitD{
            "QR",
            "quire",
            {25.0, count, commodities::packaging::sheet + commodities::paper}},
        unitD{"QT", "quart (US)", us::quart},
        unitD{"QTD", "dry quart (US)", us::dry::quart},
        unitD{"QTI", "quart (UK)", imp::quart},
        unitD{"QTL", "liquid quart (US)", us::quart},
        unitD{"QTR", "quarter (UK)", {12.70059, kg}},
        unitD{"R1", "printers pica", typographic::printers::pica},
        unitD{"R4", "calorie", cal},
        unitD{"R9", "thousand cubic metre", kilo* m.pow(3)},
        unitD{"RA", "rack", precise_unit(count, commodities::packaging::rack)},
        unitD{"RD", "rod", us::rod},
        unitD{"RG", "ring", precise_unit(count, commodities::packaging::ring)},
        unitD{
            "RH",
            "running or operating hour",
            {1.0, hr, commodities::runtime}},
        unitD{
            "RK",
            "roll metric measure",
            {1.0, count, commodities::packaging::roll}},
        unitD{"RL", "reel", {1.0, count, commodities::packaging::reel}},
        unitD{"RM", "ream", {500, count, commodities::paper}},
        unitD{"RN", "ream metric measure", {500, count, commodities::paper}},
        unitD{"RO", "roll", precise_unit(count, commodities::packaging::roll)},
        unitD{"ROM", "room", {1.0, count, commodities::packaging::room}},
        unitD{
            "RP",
            "pound per ream",
            lb / precise_unit{500, count, commodities::paper}},
        unitD{"RPM", "revolutions per minute", rpm},
        unitD{"RPS", "revolutions per second", {constants::tau, rad* Hz}},
        unitD{"RS", "reset", {1.0, count, commodities::reset}},
        unitD{"RT", "revenue ton mile", currency* ton* mile},
        unitD{"RU", "run", {1.0, count, commodities::packaging::run}},
        unitD{"S3", "square foot per second", ft.pow(2) / s},
        unitD{"S4", "square metre per second", m.pow(2) / s},
        unitD{"S5", "sixty fourths of an inch", {1.0 / 64.0, in}},
        unitD{"S6", "session", {1.0, count, commodities::packaging::session}},
        unitD{
            "S7",
            "storage unit",
            {1.0, count, commodities::packaging::storage_unit}},
        unitD{"S8", "standard advertising unit", generate_custom_unit(380)},
        unitD{"SA", "sack", {1.0, count, commodities::packaging::sack}},
        unitD{"SAN", "half year", {0.5, yr}},
        unitD{"SCO", "score", {20.0, count}},
        unitD{"SCR", "scruple", apothecaries::scruple},
        unitD{"SD", "solid pound", {1.0, lb, commodities::packaging::bulk}},
        unitD{"SE", "section", us::section},
        unitD{"SEC", "second [unit of time]", s},
        unitD{"SET", "set", {1.0, count, commodities::packaging::set}},
        unitD{"SG", "segment", {1.0, count, commodities::packaging::segment}},
        unitD{"SHT", "shipping ton", {40.0, ft.pow(3)}},
        unitD{"SIE", "siemens", S},
        unitD{
            "SK",
            "split tank truck",
            {1.0, count, commodities::packaging::split_tank_truck}},
        unitD{
            "SL",
            "slipsheet",
            {1.0, count, commodities::packaging::slipsheet}},
        unitD{"SMI", "statute mile", mile},
        unitD{"SN", "square rod", us::rod.pow(2)},
        unitD{"SO", "spool", {1.0, count, commodities::packaging::spool}},
        unitD{
            "SP",
            "shelf package",
            {1.0, count, commodities::packaging::shelf_package}},
        unitD{"SQ", "square", {1.0, count, commodities::packaging::square}},
        unitD{
            "SQR",
            "square, roofing",
            {1.0, count, commodities::packaging::square}},
        unitD{"SR", "strip", {1.0, count, commodities::packaging::strip}},
        unitD{
            "SS",
            "sheet metric measure",
            {1.0, count, commodities::packaging::sheet}},
        unitD{
            "SST",
            "short standard (7200 matches)",
            {7200.0, count, commodities::matches}},
        unitD{"ST", "sheet", {1.0, count, commodities::packaging::sheet}},
        unitD{"STC", "stick", {1.0, count, commodities::packaging::stick}},
        unitD{"STI", "stone (UK)", imp::stone},
        unitD{
            "STK",
            "stick, cigarette",
            {1.0,
             count,
             commodities::packaging::stick + commodities::cigarette}},
        unitD{"STL", "standard litre", L},
        unitD{"STN", "ton (US)", av::ton},
        unitD{"STW", "straw", {1.0, count, commodities::packaging::straw}},
        unitD{"SV", "skid", {1.0, count, commodities::packaging::skid}},
        unitD{"SW", "skein", {1.0, count, commodities::packaging::skein}},
        unitD{"SX", "shipment", {1.0, count, commodities::packaging::shipment}},
        unitD{"SYR", "syringe", {1.0, count, commodities::packaging::syringe}},
        unitD{
            "T0",
            "telecommunication line in service",
            {1.0, count, commodities::linesofservice}},
        unitD{
            "T1",
            "thousand pound gross",
            {1000.0, lb, commodities::packaging::gross_quantity}},
        unitD{
            "T3",
            "thousand piece",
            {1000.0, count, commodities::packaging::piece}},
        unitD{
            "T4",
            "thousand bag",
            {1000.0, count, commodities::packaging::bag}},
        unitD{
            "T5",
            "thousand casing",
            {1000.0, count, commodities::packaging::casing}},
        unitD{"T6", "thousand gallon (US)", thousand* us::gallon},
        unitD{
            "T7",
            "thousand impression",
            {1000.0, count, commodities::impression}},
        unitD{
            "T8",
            "thousand linear inch",
            {1000.0, in, commodities::packaging::linear}},
        unitD{"TA", "tenth cubic foot", {0.1, ft.pow(3)}},
        unitD{"TAH", "kiloampere hour", kilo* A* hr},
        unitD{"TAN", "total acid number", generate_custom_unit(213)},
        unitD{"TC", "truckload", {1.0, count, commodities::packaging::truck}},
        unitD{"TD", "therm", energy::therm_ec},
        unitD{"TE", "tote", {1.0, count, commodities::packaging::tote}},
        unitD{"TF", "ten square yard", ten* yd.pow(2)},
        unitD{"TI", "thousand square inch", kilo* in.pow(2)},
        unitD{"TIC", "metric ton, including container", mass::tonne},
        unitD{"TIP", "metric ton, including inner packaging", mass::tonne},
        unitD{"TJ", "thousand square centimetre", kilo* cm.pow(2)},
        unitD{
            "TK",
            "tank, rectangular",
            {1.0, count, commodities::packaging::tank}},
        unitD{"TKM", "tonne kilometre", mass::tonne* km},
        unitD{
            "TL",
            "thousand foot linear",
            {1000.0, ft, commodities::packaging::linear}},
        unitD{
            "TMS",
            "kilogram of imported meat, less offal",
            {1.0, kg, commodities::generateHarmonizedCode(1, 2, 0)}},
        unitD{"TN", "tin", {1.0, count, commodities::packaging::small_tin}},
        unitD{"TNE", "tonne", mass::tonne},
        unitD{"TP", "ten pack", {10.0, count, commodities::packaging::pack}},
        unitD{"TPR", "ten pair", {20, count}},
        unitD{"TQ", "thousand foot", kilo* ft},
        unitD{
            "TQD",
            "thousand cubic metre per day",
            kilo* m.pow(3) / time::day},
        unitD{"TR", "ten square foot", ten* ft.pow(2)},
        unitD{"TRL", "trillion (EUR)", {10e18, one}},
        unitD{"TS", "thousand square foot", kilo* ft.pow(2)},
        unitD{
            "TSD",
            "tonne of substance 90 % dry",
            {1.0, mass::tonne, commodities::packaging::dry90}},
        unitD{
            "TSH",
            "ton of steam per hour",
            precise_unit(1.0, ton, commodities::steam) / hr},
        unitD{"TST", "ten set", {10.0, count, commodities::packaging::set}},
        unitD{
            "TT",
            "thousand linear metre",
            {1000.0, m, commodities::packaging::linear}},
        unitD{
            "TTS",
            "ten thousand sticks",
            precise_unit(10000, count, commodities::packaging::stick)},
        unitD{
            "TU",
            "tube",
            precise_unit(1, count, commodities::packaging::tube)},
        unitD{"TV", "thousand kilogram", thousand* kg},
        unitD{
            "TW",
            "thousand sheet",
            precise_unit(1.0, thousand, commodities::packaging::sheet)},
        unitD{
            "TY",
            "tank, cylindrical",
            precise_unit(1.0, count, commodities::packaging::cylindrical_tank)},
        unitD{
            "U1",
            "treatment",
            {1.0, count, commodities::packaging::treatment}},
        unitD{
            "U2",
            "tablet",
            precise_unit(1.0, count, commodities::packaging::tablet)},
        unitD{"UA", "torr", pressure::torr},
        unitD{
            "UB",
            "telecommunication line in service average",
            precise_unit(1.0, count, commodities::linesofservice)},
        unitD{"UC", "telecommunication port", {1.0, count, commodities::port}},
        unitD{"UD", "tenth minute", {0.1, min}},
        unitD{"UE", "tenth hour", {0.1, hr}},
        unitD{
            "UF",
            "usage per telecommunication line average",
            count / precise_unit(1.0, one, commodities::linesofservice)},
        unitD{"UH", "ten thousand yard", {10000.0, yd}},
        unitD{"UM", "million unit", mega* count},
        unitD{"VA", "volt - ampere per kilogram", V* A / kg},
        unitD{"VI", "vial", precise_unit(count, commodities::packaging::vial)},
        unitD{"VLT", "volt", V},
        unitD{"VP", "percent volume", precise_unit(0.01, pu* m.pow(3))},
        unitD{"VQ", "bulk", precise_unit(count, commodities::packaging::bulk)},
        unitD{"VS", "visit", precise_unit(count, commodities::visit)},
        unitD{"W2", "wet kilo", precise_unit(kg, commodities::packaging::wet)},
        unitD{"W4", "two week", {2.0, time::week}},
        unitD{"WA", "watt per kilogram", W / kg},
        unitD{"WB", "wet pound", {1.0, lb, commodities::packaging::wet}},
        unitD{"WCD", "cord", us::cord},
        unitD{"WE", "wet ton", {1.0, ton, commodities::packaging::wet}},
        unitD{"WEB", "weber", weber},
        unitD{"WEE", "week", time::week},
        unitD{"WG", "wine gallon", us::gallon},
        unitD{"WH", "wheel", {1.0, count, commodities::packaging::wheel}},
        unitD{"WHR", "watt hour", W* hr},
        unitD{"WI", "weight per square inch", lb / in.pow(2)},
        unitD{"WM", "working month", {1.0, time::mog, commodities::labor}},
        unitD{"WR", "wrap", {1.0, count, commodities::packaging::wrap}},
        unitD{"WSD", "standard", {4.672, m.pow(3), commodities::lumber}},
        unitD{"WTT", "watt", W},
        unitD{"WW", "millilitre of water", {1.0, mL, commodities::water}},
        unitD{"X1", "Gunter's chain", us::engineers::chain},
        unitD{"YDK", "square yard", yd.pow(2)},
        unitD{"YDQ", "cubic yard", yd.pow(3)},
        unitD{
            "YL",
            "hundred linear yard",
            {100.0, yd, commodities::packaging::linear}},
        unitD{"YRD", "yard", yd},
        unitD{"YT", "ten yard", {10.0, yd}},
        unitD{
            "Z1",
            "lift van",
            precise_unit(count, commodities::packaging::lift_van)},
        unitD{
            "Z11",
            "hanging container",
            precise_unit(count, commodities::packaging::hanging_container)},
        unitD{
            "Z2",
            "chest",
            precise_unit(count, commodities::packaging::chest)},
        unitD{"Z3", "cask", {1.0, count, commodities::packaging::cask}},
        unitD{"Z4", "hogshead", us::hogshead},
        unitD{"Z5", "lug", {1.0, count, commodities::packaging::lug}},
        unitD{"Z6", "conference point", generate_custom_unit(811)},
        unitD{
            "Z8",
            "newspage agate line",
            precise_unit(1.0 / 14, in) * precise_unit(2.0, in)},
        unitD{"ZP", "page", {1.0, count, commodities::packaging::page}},
        unitD{"ZZ", "mutually defined", generate_custom_unit(262)},
    }};

}  // namespace precise

precise_unit r20_unit(const std::string& r20_string)
{
    // NOLINTNEXTLINE (readability-qualified-auto)
    auto ind = std::lower_bound(
        precise::r20_units.begin(),
        precise::r20_units.end(),
        r20_string,
        [](const precise::unitD& u_set, const std::string& val) {
            return (strcmp(std::get<0>(u_set), val.c_str()) < 0);
        });
    if (ind == precise::r20_units.end()) {
        return precise::invalid;
    }
    if (strcmp(std::get<0>(*ind), r20_string.c_str()) == 0) {
        return std::get<2>(*ind);
    }
    return precise::invalid;
}

#ifdef ENABLE_UNIT_MAP_ACCESS
namespace detail {
    const void* r20rawData(size_t& array_size)
    {
        array_size = precise::r20_units.size();
        return precise::r20_units.data();
    }
}  // namespace detail

#endif

}  // namespace units
