///////////////////////////////////////////////////////////////////////////////
//                                                         
// BigDecimal.cc
// -------------
// BigDecimal implementation
//                                               
// Design and Implementation by Bjoern Lemke               
//                                                         
// (C)opyright 2000-2016 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: BigDecimal
// 
// Description: BigDecimal implementation
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// INCLUDES
#include "Exception.h"
#include "BigInteger.h"
#include "BigDecimal.h"
#include "Tokenizer.h"

BigDecimal::BigDecimal()
{
    _val = Chain(0);
    _scale=0;
    _isPositive = true;
}

BigDecimal::BigDecimal(const Chain& val)
{
    
    if ( val.subChain(1,1) == Chain("+"))
    {
	_isPositive=true;
	_val = val.subChain(2, val.length());
    }
    else if ( val.subChain(1,1) == Chain("-"))
    {
	_isPositive=false;
	_val = val.subChain(2, val.length());
    }
    else
    {
	_isPositive=true;
	_val = val;
    }
    
    if ( ! _val.isDec() )
    {
	Chain msg = Chain("Invalid decimal format for <") + val + Chain(">");
	throw Exception(EXLOC, msg);
    }
	    
    if ( _val.subChain(1,1) == Chain(".") )
	_val = Chain("0") + _val;
    
    Tokenizer tok(_val, Chain("."));
    Chain left, right;
    tok.nextToken(left);
    tok.nextToken(right);

    right = right.truncRight("0");
    if ( right.length() < 2 )
	right = Chain("0");
    
    _scale = right.length() > 0 ? right.length() - 1 : 0;

    left = left.truncLeft("0");
    if ( left.length() < 2 )
	left = Chain("0");
    
    _val = left + right;	 
}

BigDecimal::BigDecimal(const Chain& val, int scale)
{

    if ( val.subChain(1,1) == Chain("+"))
    {
	_isPositive=true;
	_val = val.subChain(2, val.length());
    }
    else if ( val.subChain(1,1) == Chain("-"))
    {
	_isPositive=false;
	_val = val.subChain(2, val.length());
    }
    else
    {
	_isPositive=true;
	_val = val;
    }

    if ( ! _val.isDec() )
    {
	Chain msg = Chain("Invalid decimal format for <") + val + Chain(">");
	throw Exception(EXLOC, msg);
    }

    _scale = scale;
}


BigDecimal::~BigDecimal()
{
}

BigDecimal BigDecimal::abs() const
{
    return BigDecimal(_val);
}

bool BigDecimal::isPositive() const
{
    return _isPositive;
}

void BigDecimal::negate()
{
    _isPositive = false;
}

BigDecimal BigDecimal::add(const BigDecimal& d) const
{
    Chain v1 = _val;
    Chain v2 = d._val;
    int maxScale;

    if ( _scale > d._scale )
    {
	maxScale = _scale;
	v2 = mulDec(v2, _scale - d._scale);
    }
    else if ( _scale < d._scale )
    {
	maxScale = d._scale;
	v1 = mulDec(v1, d._scale - _scale);
    }    
    else
    {
	maxScale = _scale;
    }

    BigInteger i1(v1);
    BigInteger i2(v2);

    if ( _isPositive == false )
	i1.negate();
    if ( d._isPositive == false )
	i2.negate();


    BigInteger i3 = i1 + i2;

    return BigDecimal(i3.toChain(), maxScale); 
    
}

BigDecimal BigDecimal::sub(const BigDecimal& d) const
{

    Chain v1 = _val;
    Chain v2 = d._val;
    int maxScale;

    if ( _scale > d._scale )
    {
	maxScale = _scale;
	v2 = mulDec(v2, _scale - d._scale);
    }
    else if ( _scale < d._scale )
    {
	maxScale = d._scale;
	v1 = mulDec(v1, d._scale - _scale);
    }    
    else
    {
	maxScale = _scale;
    }
    BigInteger i1(v1);
    if ( _isPositive == false )
	i1.negate();
    BigInteger i2(v2);
    if ( d._isPositive == false )
	i2.negate();

    BigInteger i3 =  i1 - i2;

    return BigDecimal(i3.toChain(), maxScale); 

}

BigDecimal BigDecimal::mul(const BigDecimal& d) const
{

    Chain v1 = _val;
    Chain v2 = d._val;

    BigInteger i1(v1);
    if ( _isPositive == false )
	i1.negate();
    BigInteger i2(v2);
    if ( d._isPositive == false )
	i2.negate();
    
    BigInteger i3 =  i1 * i2;

    return BigDecimal(i3.toChain(), _scale + d._scale); 
    
}

BigDecimal BigDecimal::div(const BigDecimal& d) const
{

    Chain v1 = _val;
    Chain v2 = d._val;

    if ( v1 == Chain("00") )
	 return BigDecimal("0");
	 
    int maxScale;
    int corScale;
    
    if ( d._scale > _scale )
    {	
	maxScale = d._scale;
	v1 = mulDec(v1, d._scale - _scale);
    }
    else
    {
	maxScale = _scale;
	v2 = mulDec(v2, _scale - d._scale);
    }

    // for div calculation, we use absolute values
    BigInteger i1(v1);    
    BigInteger i2(v2);
    
    corScale=0;
    while ( i1 < i2 )
    {
	i1 = i1.mul(BigInteger(10));
	corScale++;
    }

    while ( corScale < maxScale )
    {
	i1 = i1.mul(BigInteger(10));
	// cout << i1 << endl;
	corScale++;
    }

    // cout << "i1=" << i1 << " i2=" << i2 << endl;
    BigInteger i3 =  i1 / i2;
    // cout << "i3=" << i3 << endl;

    if ( _isPositive == false && d._isPositive == true
	 || _isPositive == true && d._isPositive == false )
	i3.negate();
    
    Chain v = i3.toChain();
    if ( i3.toChain().length()-1 < maxScale )
    {
	int c = 1 + maxScale - ( i3.toChain().length() -1 );
	while ( c )
	{
	    v = Chain("0") + v;
	    c--;
	}
    }
    
    return BigDecimal(v, maxScale); 
}

int BigDecimal::getScale() const
{
    return _scale;
}

int BigDecimal::length() const
{
    return _val.length();
}

Chain BigDecimal::toChain() const
{
    Chain s;

    if ( _isPositive == false )
	s = Chain("-");

    Chain val = _val;
    
    if ( val.length() - 1  <= _scale )
    {
	int j = val.length() - 1;
	for ( int i = j; i <= _scale ; i++)
	{
	    val = "0" + val;
	}
    }
    
    s +=  val.subChain(1, val.length() - _scale - 1) 
	+ Chain(".")
	+ val.subChain(val.length() - _scale, val.length()); 	

    return s;
}

BigDecimal& BigDecimal::operator = ( const BigDecimal& d)
{
    _val = d._val;
    _isPositive = d._isPositive;
    _scale = d._scale;
    return (*this);
}

BigDecimal& BigDecimal::operator += ( const BigDecimal& d)
{
    this->add(d);
    return (*this);
}

BigDecimal& BigDecimal::operator -= ( const BigDecimal& d)
{
    this->add(d);
    return (*this);
}

bool BigDecimal::operator == ( const BigDecimal& d) const
{
    return ( _val == d._val && _scale == d._scale && _isPositive == d._isPositive );
}

bool BigDecimal::operator != ( const BigDecimal& d) const
{
    return ! ( d == *this );
}

bool BigDecimal::operator < ( const BigDecimal& d) const
{
    if ( ( _val.length() - _scale  ) < ( d._val.length() - d._scale ) )
	return true;
    else if ( ( _val.length() - _scale  ) > ( d._val.length() - d._scale ) )
	return false;
    else
    {
	if ( _scale < d._scale )
	{
	    BigInteger v1( d._val.subChain(1, d._val.length() - ( d._scale - _scale ) - 1));
	    BigInteger v2( _val );
	    return v2 < v1;
	}
	else
	{		    
	    BigInteger v1( d._val );
	    BigInteger v2( _val.subChain(1, _val.length() - ( _scale - d._scale ) - 1));
	    return v2 < v1;
	}
    }		
}

bool BigDecimal::operator > ( const BigDecimal& d) const
{
    if ( ( _val.length() - _scale  ) > ( d._val.length() - d._scale ) )
	return true;
    else if ( ( _val.length() - _scale  ) < ( d._val.length() - d._scale ) )
	return false;
    else
    {
	if ( _scale < d._scale )
	{
	    BigInteger v1( d._val.subChain(1, d._val.length() - ( d._scale - _scale ) - 1));
	    BigInteger v2( _val );
	    return v2 > v1;
	}
	else
	{
	    BigInteger v1( d._val );
	    BigInteger v2( _val.subChain(1, _val.length() - ( _scale - d._scale ) - 1));
	    return v2 > v1;
	}
    }
}

bool BigDecimal::operator <= ( const BigDecimal& d) const
{
    if ( _val == d._val && _isPositive == d._isPositive)
	return true;
    else
	return *this < d;	
}

bool BigDecimal::operator >= ( const BigDecimal& d) const
{
    if ( _val == d._val && _isPositive == d._isPositive)
	return true;
    else
	return *this > d;
}

BigDecimal operator + ( const BigDecimal& d1, const BigDecimal& d2)
{
    return d1.add(d2);
}

BigDecimal operator - ( const BigDecimal& d1, const BigDecimal& d2)
{
    return d1.sub(d2);
}

BigDecimal operator * ( const BigDecimal& d1, const BigDecimal& d2)
{
    return d1.mul(d2);
}

BigDecimal operator / ( const BigDecimal& d1, const BigDecimal& d2)
{
    return d1.div(d2);
}

ostream& operator << (ostream& s, const  BigDecimal& d)
{
    cout << d.toChain();
    return s;
}

Chain BigDecimal::mulDec(const Chain& val, int dim) const
{
    Chain res = val;
    while ( dim > 0 )
    {
	res += Chain("0");
	dim--;
    }
    return res;
}
