/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 */

#ifndef SH4LT_DIR_WATCHER_H_
#define SH4LT_DIR_WATCHER_H_

#include <filesystem>
#include <vector>

#include "../infotree/information-tree.hpp"

namespace fs = std::filesystem;

namespace sh4lt::monitor {

class DirWatcher {
 public:
  /**
   * Construct the DirWatcher object.
   *
   * \param path Directory path to watch
   **/
  DirWatcher(fs::path path);
  DirWatcher() = delete;

  /**
   * Udpate internal state.
   *
   * Updates vector of Sh4lt paths:
   *   - current: all currently active sockets, including new ones
   *   - created: created sockets since last call to update
   *   - removed: removed sockets since last call to update
   *
   * \return The updated state.
   **/
  auto update() -> bool;

  /**
   * Get the vector of current socket paths.
   *
   * \return The list of paths
   **/
  auto get_currents() -> const std::vector<fs::path>& { return currents_; }

  /**
   * Get the vector of socket paths discoved at last call to update.
   *
   * \return The list of paths
   **/
  auto get_created() -> const std::vector<fs::path>& { return created_; }

  /**
   * Get the vector of removed socket paths at last call to update.
   *
   * \return The list of paths
   **/
  auto get_removed() -> const std::vector<fs::path>& { return removed_; }

 private:
  fs::path path_;
  std::vector<fs::path> currents_{};  //< currently active sockets, including new ones
  std::vector<fs::path> created_{};   //< newly created sockets
  std::vector<fs::path> removed_{};   //< removed sockets
};

}  // namespace sh4lt::monitor
#endif
