// =============================================================================
// PROJECT CHRONO - http://projectchrono.org
//
// Copyright (c) 2014 projectchrono.org
// All rights reserved.
//
// Use of this source code is governed by a BSD-style license that can be found
// in the LICENSE file at the top level of the distribution and at
// http://projectchrono.org/license-chrono.txt.
//
// =============================================================================
// Authors: Daniel Melanz, Radu Serban
// =============================================================================
//
// Generic solid axle suspension subsystem.
//
// This concrete suspension subsystem is defined with respect to a right-handed
// frame with X pointing towards the front, Y to the left, and Z up (as imposed
// by the base class ChSolidAxle) and origin at the midpoint between the wheel
// centers.
//
// All point locations are provided for the left half of the suspension.
//
// =============================================================================

#ifndef GENERIC_SOLIDAXLE_H
#define GENERIC_SOLIDAXLE_H

#include "chrono_vehicle/wheeled_vehicle/suspension/ChSolidAxle.h"

#include "chrono_models/ChApiModels.h"

namespace chrono {
namespace vehicle {
namespace generic {

/// @addtogroup vehicle_models_generic
/// @{

/// Solid-axle suspension subsystem for the generic vehicle.
class CH_MODELS_API Generic_SolidAxle : public ChSolidAxle {
  public:
    Generic_SolidAxle(const std::string& name);
    ~Generic_SolidAxle();

    virtual double getCamberAngle() const override { return 0; }
    virtual double getToeAngle() const override { return 0; }

    virtual double getAxleTubeMass() const override { return m_axleTubeMass; }
    virtual double getSpindleMass() const override { return m_spindleMass; }
    virtual double getULMass() const override { return m_ULMass; }
    virtual double getLLMass() const override { return m_LLMass; }
    virtual double getKnuckleMass() const override { return m_knuckleMass; }
    virtual double getTierodMass() const override { return m_tierodMass; }
    virtual double getDraglinkMass() const override { return m_draglinkMass; }
    virtual double getBellCrankMass() const override { return m_bellCrankMass; }
    virtual double getTrackbarMass() const override { return m_trackbarMass; }

    virtual double getAxleTubeRadius() const override { return m_axleTubeRadius; }
    virtual double getSpindleRadius() const override { return m_spindleRadius; }
    virtual double getSpindleWidth() const override { return m_spindleWidth; }
    virtual double getULRadius() const override { return m_ULRadius; }
    virtual double getLLRadius() const override { return m_LLRadius; }
    virtual double getKnuckleRadius() const override { return m_knuckleRadius; }
    virtual double getTierodRadius() const override { return m_tierodRadius; }
    virtual double getDraglinkRadius() const override { return m_draglinkRadius; }
    virtual double getBellCrankRadius() const override { return m_bellCrankRadius; }
    virtual double getTrackbarRadius() const override { return m_trackbarRadius; }

    virtual const ChVector3d& getAxleTubeInertia() const override { return m_axleTubeInertia; }
    virtual const ChVector3d& getSpindleInertia() const override { return m_spindleInertia; }
    virtual const ChVector3d& getULInertia() const override { return m_ULInertia; }
    virtual const ChVector3d& getLLInertia() const override { return m_LLInertia; }
    virtual const ChVector3d& getKnuckleInertia() const override { return m_knuckleInertia; }
    virtual const ChVector3d& getTierodInertia() const override { return m_tierodInertia; }
    virtual const ChVector3d& getDraglinkInertia() const override { return m_draglinkInertia; }
    virtual const ChVector3d& getBellCrankInertia() const override { return m_bellCrankInertia; }
    virtual const ChVector3d& getTrackbarInertia() const override { return m_trackbarInertia; }

    virtual double getAxleInertia() const override { return m_axleInertia; }

    virtual double getSpringRestLength() const override { return m_springRestLength; }
    virtual std::shared_ptr<ChLinkTSDA::ForceFunctor> getSpringForceFunctor() const override { return m_springForceCB; }
    virtual std::shared_ptr<ChLinkTSDA::ForceFunctor> getShockForceFunctor() const override { return m_shockForceCB; }

    virtual const ChVector3d getAxleTubeCOM() const override { return m_axleTubeCOM; }

  private:
    virtual const ChVector3d getLocation(PointId which) override;

    std::shared_ptr<ChLinkTSDA::ForceFunctor> m_springForceCB;
    std::shared_ptr<ChLinkTSDA::ForceFunctor> m_shockForceCB;

    static const double m_axleTubeMass;
    static const double m_spindleMass;
    static const double m_ULMass;
    static const double m_LLMass;
    static const double m_knuckleMass;
    static const double m_tierodMass;
    static const double m_draglinkMass;
    static const double m_bellCrankMass;
    static const double m_trackbarMass;

    static const double m_axleTubeRadius;
    static const double m_spindleRadius;
    static const double m_spindleWidth;
    static const double m_ULRadius;
    static const double m_LLRadius;
    static const double m_knuckleRadius;
    static const double m_tierodRadius;
    static const double m_draglinkRadius;
    static const double m_bellCrankRadius;
    static const double m_trackbarRadius;

    static const ChVector3d m_axleTubeCOM;

    static const ChVector3d m_axleTubeInertia;
    static const ChVector3d m_spindleInertia;
    static const ChVector3d m_ULInertia;
    static const ChVector3d m_LLInertia;
    static const ChVector3d m_knuckleInertia;
    static const ChVector3d m_tierodInertia;
    static const ChVector3d m_draglinkInertia;
    static const ChVector3d m_bellCrankInertia;
    static const ChVector3d m_trackbarInertia;

    static const double m_axleInertia;

    static const double m_springCoefficient;
    static const double m_dampingCoefficient;
    static const double m_springRestLength;
};

/// @} vehicle_models_generic

}  // end namespace generic
}  // end namespace vehicle
}  // end namespace chrono

#endif
