/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// TGUI - Texus' Graphical User Interface
// Copyright (C) 2012-2025 Bruno Van de Velde (vdv_b@tgui.eu)
//
// This software is provided 'as-is', without any express or implied warranty.
// In no event will the authors be held liable for any damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it freely,
// subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented;
//    you must not claim that you wrote the original software.
//    If you use this software in a product, an acknowledgment
//    in the product documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such,
//    and must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef TGUI_CANVAS_RAYLIB_HPP
#define TGUI_CANVAS_RAYLIB_HPP

#include <TGUI/Backend/Renderer/Raylib/BackendTextureRaylib.hpp>

#if !TGUI_BUILD_AS_CXX_MODULE
    #include <TGUI/Backend/Renderer/BackendRenderTarget.hpp>
    #include <TGUI/Widgets/CanvasBase.hpp>
#endif

#include <raylib.h>

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

TGUI_MODULE_EXPORT namespace tgui
{
    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief CanvasRaylib provides a way to directly render raylib contents on a widget
    ///
    /// When gui.draw() is called, all widgets are drawn at once. If you wish to have custom raylib rendering inbetween
    /// TGUI widgets (e.g. draw to the background of a child window) then you need to use a CanvasRaylib widget.
    ///
    /// The canvas widget is essentially just a wrapper around raylib's RenderTexture. You draw your raylib contents on top of
    /// the canvas instead of on the window. The canvas is then added to the gui between the widgets where you want the rendering
    /// to appear.
    ///
    /// You can redraw the contents of the canvas at any time, but make sure to always start by calling BeginTextureMode
    /// with the texture of the canvas and end the rendering by calling EndTextureMode.
    ///
    /// Example:
    /// @code
    /// auto canvas = tgui::CanvasRaylib::create({400, 300});
    /// gui.add(canvas);
    ///
    /// BeginTextureMode(canvas->getTextureTarget());  // Begin drawing to the canvas
    ///   ClearBackground(RAYWHITE);                   // Clear the contents of the canvas with a background color
    ///   DrawRectangle(0, 0, 200, 100, YELLOW);       // Draw a rectangle to the canvas
    /// EndTextureMode();                              // End drawing to the canvas
    /// @endcode
    ///
    /// @since TGUI 1.3
    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    class TGUI_API CanvasRaylib : public CanvasBase
    {
    public:

        using Ptr = std::shared_ptr<CanvasRaylib>; //!< Shared widget pointer
        using ConstPtr = std::shared_ptr<const CanvasRaylib>; //!< Shared constant widget pointer

        static constexpr const char StaticWidgetType[] = "CanvasRaylib"; //!< Type name of the widget

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Destructor
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        ~CanvasRaylib() override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @internal
        /// @brief Constructor
        /// @param typeName     Type of the widget
        /// @param initRenderer Should the renderer be initialized? Should be true unless a derived class initializes it.
        /// @see create
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        CanvasRaylib(const char* typeName = StaticWidgetType, bool initRenderer = true);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Copy constructor
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        CanvasRaylib(const CanvasRaylib& copy);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Default move constructor
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        CanvasRaylib(CanvasRaylib&& copy) = default;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Overload of copy assignment operator
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        CanvasRaylib& operator= (const CanvasRaylib& right);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Default move assignment operator
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        CanvasRaylib& operator= (CanvasRaylib&& right) = default;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Creates a new canvas widget
        ///
        /// @param size  Size of the canvas
        ///
        /// @return The new canvas
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD static CanvasRaylib::Ptr create(const Layout2d& size = {"100%", "100%"});

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Makes a copy of another canvas
        ///
        /// @param canvas  The other canvas
        ///
        /// @return The new canvas
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD static CanvasRaylib::Ptr copy(const CanvasRaylib::ConstPtr& canvas);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the size of the widget
        ///
        /// @param size  The new size of the widget
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setSize(const Layout2d& size) override;
        using Widget::setSize;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Draw the widget to a render target
        ///
        /// @param target Render target to draw to
        /// @param states Current render states
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void draw(BackendRenderTarget& target, RenderStates states) const override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Access the internal render texture to which you can render
        ///
        /// @return Reference to the internal render texture
        ///
        /// The texture has a size that is equal or larger than the size of the canvas widget.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD RenderTexture2D& getTextureTarget()
        {
            return m_textureTarget;
        }

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    protected:

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Makes a copy of the widget
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD Widget::Ptr clone() const override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    protected:

        RenderTexture2D m_textureTarget;
        Vector2u m_textureSize;
        Vector2u m_usedTextureSize;
        std::shared_ptr<BackendTextureRaylib> m_backendTexture;
    };
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#endif // TGUI_CANVAS_RAYLIB_HPP
