// SPDX-License-Identifier: GPL-2.0+
/*
 * Copyright (c) 2024 Philippe Reynes <philippe.reynes@softathome.com>
 *
 * Unit tests for sha256_hmac functions
 */

#include <command.h>
#include <test/lib.h>
#include <test/test.h>
#include <test/ut.h>
#include <u-boot/sha256.h>

struct test_sha256_hmac_s {
	const unsigned char *key;
	int keylen;
	const unsigned char *input;
	int ilen;
	const unsigned char *expected;
	int elen;
};

/*
 * data comes from:
 * https://datatracker.ietf.org/doc/html/rfc4231
 */
static unsigned char key_test1[] = {
	0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b };

static unsigned char input_test1[] = {
	0x48, 0x69, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65 };

static unsigned char expected_test1[] = {
	0xb0, 0x34, 0x4c, 0x61, 0xd8, 0xdb, 0x38, 0x53,
	0x5c, 0xa8, 0xaf, 0xce, 0xaf, 0x0b, 0xf1, 0x2b,
	0x88, 0x1d, 0xc2, 0x00, 0xc9, 0x83, 0x3d, 0xa7,
	0x26, 0xe9, 0x37, 0x6c, 0x2e, 0x32, 0xcf, 0xf7 };

static unsigned char key_test2[] = { 0x4a, 0x65, 0x66, 0x65 };

static unsigned char input_test2[] = {
	0x77, 0x68, 0x61, 0x74, 0x20, 0x64, 0x6f, 0x20,
	0x79, 0x61, 0x20, 0x77, 0x61, 0x6e, 0x74, 0x20,
	0x66, 0x6f, 0x72, 0x20, 0x6e, 0x6f, 0x74, 0x68,
	0x69, 0x6e, 0x67, 0x3f };

static unsigned char expected_test2[] = {
	0x5b, 0xdc, 0xc1, 0x46, 0xbf, 0x60, 0x75, 0x4e,
	0x6a, 0x04, 0x24, 0x26, 0x08, 0x95, 0x75, 0xc7,
	0x5a, 0x00, 0x3f, 0x08, 0x9d, 0x27, 0x39, 0x83,
	0x9d, 0xec, 0x58, 0xb9, 0x64, 0xec, 0x38, 0x43 };

static unsigned char key_test3[] = {
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa };

static unsigned char input_test3[] = {
	0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	0xdd, 0xdd };

static unsigned char expected_test3[] = {
	0x77, 0x3e, 0xa9, 0x1e, 0x36, 0x80, 0x0e, 0x46,
	0x85, 0x4d, 0xb8, 0xeb, 0xd0, 0x91, 0x81, 0xa7,
	0x29, 0x59, 0x09, 0x8b, 0x3e, 0xf8, 0xc1, 0x22,
	0xd9, 0x63, 0x55, 0x14, 0xce, 0xd5, 0x65, 0xfe };

static unsigned char key_test4[] = {
	0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
	0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
	0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
	0x19,
};

static unsigned char input_test4[] = {
	0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	0xcd, 0xcd,
};

static unsigned char expected_test4[] = {
	0x82, 0x55, 0x8a, 0x38, 0x9a, 0x44, 0x3c, 0x0e,
	0xa4, 0xcc, 0x81, 0x98, 0x99, 0xf2, 0x08, 0x3a,
	0x85, 0xf0, 0xfa, 0xa3, 0xe5, 0x78, 0xf8, 0x07,
	0x7a, 0x2e, 0x3f, 0xf4, 0x67, 0x29, 0x66, 0x5b,
};

static unsigned char key_test5[] = {
	0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
	0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
	0x0c, 0x0c, 0x0c, 0x0c,
};

static unsigned char input_test5[] = {
	0x54, 0x65, 0x73, 0x74, 0x20, 0x57, 0x69, 0x74,
	0x68, 0x20, 0x54, 0x72, 0x75, 0x6e, 0x63, 0x61,
	0x74, 0x69, 0x6f, 0x6e,
};

static unsigned char expected_test5[] = {
	0xa3, 0xb6, 0x16, 0x74, 0x73, 0x10, 0x0e, 0xe0,
	0x6e, 0x0c, 0x79, 0x6c, 0x29, 0x55, 0x55, 0x2b,
};

static unsigned char key_test6[] = {
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa };

static unsigned char input_test6[] = {
	0x54, 0x65, 0x73, 0x74, 0x20, 0x55, 0x73, 0x69,
	0x6e, 0x67, 0x20, 0x4c, 0x61, 0x72, 0x67, 0x65,
	0x72, 0x20, 0x54, 0x68, 0x61, 0x6e, 0x20, 0x42,
	0x6c, 0x6f, 0x63, 0x6b, 0x2d, 0x53, 0x69, 0x7a,
	0x65, 0x20, 0x4b, 0x65, 0x79, 0x20, 0x2d, 0x20,
	0x48, 0x61, 0x73, 0x68, 0x20, 0x4b, 0x65, 0x79,
	0x20, 0x46, 0x69, 0x72, 0x73, 0x74 };

static unsigned char expected_test6[] = {
	0x60, 0xe4, 0x31, 0x59, 0x1e, 0xe0, 0xb6, 0x7f,
	0x0d, 0x8a, 0x26, 0xaa, 0xcb, 0xf5, 0xb7, 0x7f,
	0x8e, 0x0b, 0xc6, 0x21, 0x37, 0x28, 0xc5, 0x14,
	0x05, 0x46, 0x04, 0x0f, 0x0e, 0xe3, 0x7f, 0x54 };

static unsigned char key_test7[] = {
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa,
};

static unsigned char input_test7[] = {
	0x54, 0x68, 0x69, 0x73, 0x20, 0x69, 0x73, 0x20,
	0x61, 0x20, 0x74, 0x65, 0x73, 0x74, 0x20, 0x75,
	0x73, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x6c,
	0x61, 0x72, 0x67, 0x65, 0x72, 0x20, 0x74, 0x68,
	0x61, 0x6e, 0x20, 0x62, 0x6c, 0x6f, 0x63, 0x6b,
	0x2d, 0x73, 0x69, 0x7a, 0x65, 0x20, 0x6b, 0x65,
	0x79, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x20,
	0x6c, 0x61, 0x72, 0x67, 0x65, 0x72, 0x20, 0x74,
	0x68, 0x61, 0x6e, 0x20, 0x62, 0x6c, 0x6f, 0x63,
	0x6b, 0x2d, 0x73, 0x69, 0x7a, 0x65, 0x20, 0x64,
	0x61, 0x74, 0x61, 0x2e, 0x20, 0x54, 0x68, 0x65,
	0x20, 0x6b, 0x65, 0x79, 0x20, 0x6e, 0x65, 0x65,
	0x64, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x62, 0x65,
	0x20, 0x68, 0x61, 0x73, 0x68, 0x65, 0x64, 0x20,
	0x62, 0x65, 0x66, 0x6f, 0x72, 0x65, 0x20, 0x62,
	0x65, 0x69, 0x6e, 0x67, 0x20, 0x75, 0x73, 0x65,
	0x64, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68, 0x65,
	0x20, 0x48, 0x4d, 0x41, 0x43, 0x20, 0x61, 0x6c,
	0x67, 0x6f, 0x72, 0x69, 0x74, 0x68, 0x6d, 0x2e,
};

static unsigned char expected_test7[] = {
	0x9b, 0x09, 0xff, 0xa7, 0x1b, 0x94, 0x2f, 0xcb,
	0x27, 0x63, 0x5f, 0xbc, 0xd5, 0xb0, 0xe9, 0x44,
	0xbf, 0xdc, 0x63, 0x64, 0x4f, 0x07, 0x13, 0x93,
	0x8a, 0x7f, 0x51, 0x53, 0x5c, 0x3a, 0x35, 0xe2,
};

static struct test_sha256_hmac_s test_sha256_hmac[] = {
	{
		.key = key_test1,
		.keylen = sizeof(key_test1),
		.input = input_test1,
		.ilen = sizeof(input_test1),
		.expected = expected_test1,
		.elen = sizeof(expected_test1),
	},
	{
		.key = key_test2,
		.keylen = sizeof(key_test2),
		.input = input_test2,
		.ilen = sizeof(input_test2),
		.expected = expected_test2,
		.elen = sizeof(expected_test2),
	},
	{
		.key = key_test3,
		.keylen = sizeof(key_test3),
		.input = input_test3,
		.ilen = sizeof(input_test3),
		.expected = expected_test3,
		.elen = sizeof(expected_test3),
	},
	{
		.key = key_test4,
		.keylen = sizeof(key_test4),
		.input = input_test4,
		.ilen = sizeof(input_test4),
		.expected = expected_test4,
		.elen = sizeof(expected_test4),
	},
	{
		.key = key_test5,
		.keylen = sizeof(key_test5),
		.input = input_test5,
		.ilen = sizeof(input_test5),
		.expected = expected_test5,
		.elen = sizeof(expected_test5),
	},
	{
		.key = key_test6,
		.keylen = sizeof(key_test6),
		.input = input_test6,
		.ilen = sizeof(input_test6),
		.expected = expected_test6,
		.elen = sizeof(expected_test6),
	},
	{
		.key = key_test7,
		.keylen = sizeof(key_test7),
		.input = input_test7,
		.ilen = sizeof(input_test7),
		.expected = expected_test7,
		.elen = sizeof(expected_test7),
	},
};

static int _lib_test_sha256_hmac_run(struct unit_test_state *uts,
				     const unsigned char *key, int keylen,
				     const unsigned char *input, int ilen,
				     const unsigned char *expected, int elen)
{
	unsigned char output[SHA256_SUM_LEN];
	int ret;

	ut_assert(elen <= sizeof(output));
	ret = sha256_hmac(key, keylen, input, ilen, output);
	ut_assert(!ret);
	ut_asserteq_mem(expected, output, elen);

	return 0;
}

static int lib_test_sha256_hmac_run(struct unit_test_state *uts,
				    struct test_sha256_hmac_s *test)
{
	return _lib_test_sha256_hmac_run(uts, test->key, test->keylen,
					 test->input, test->ilen,
					 test->expected, test->elen);
}

static int lib_test_sha256_hmac(struct unit_test_state *uts)
{
	int i, ret = 0;

	for (i = 0; i < ARRAY_SIZE(test_sha256_hmac); i++) {
		ret = lib_test_sha256_hmac_run(uts, &test_sha256_hmac[i]);
		if (ret)
			return ret;
	}

	return 0;
}

LIB_TEST(lib_test_sha256_hmac, 0);
