"use strict";
/**
 * This is the test file for `doCompletionResolve()` function in the completion provider.
 * `doCompletionResolve()` is called to resolve the selected completion item.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const os_1 = require("os");
const completionProvider_1 = require("../../src/providers/completionProvider");
const helper_1 = require("../helper");
function testFQCNEnabled(context) {
    const tests = [
        {
            name: "module name with full FQCN",
            completionItem: {
                label: "module_3",
                data: {
                    documentUri: "dummy/url/for/resolve_completion.yml",
                    moduleFqcn: "org_1.coll_3.module_3",
                    inlineCollections: ["org_1.coll_3", "ansible.builtin"],
                    atEndOfLine: true,
                    firstElementOfList: false,
                },
            },
            completionText: "org_1.coll_3.module_3",
        },
    ];
    tests.forEach(({ name, completionItem, completionText }) => {
        it(`should resolve completion for ${name}`, function () {
            return __awaiter(this, void 0, void 0, function* () {
                const actualCompletionResolveAtLineEnd = yield (0, completionProvider_1.doCompletionResolve)(completionItem, context);
                (0, chai_1.expect)(actualCompletionResolveAtLineEnd.insertText).be.equal(`${completionText}:${os_1.EOL}\t`);
                // Check for completion resolution when asked in between of lines
                completionItem.data.atEndOfLine = false;
                const actualCompletionResolveAtInBetween = yield (0, completionProvider_1.doCompletionResolve)(completionItem, context);
                (0, chai_1.expect)(actualCompletionResolveAtInBetween.insertText).be.equal(`${completionText}`);
            });
        });
    });
}
function testFQCNDisabled(context) {
    const tests = [
        {
            name: "module name with short name since it is present in declared collections in playbook",
            completionItem: {
                label: "module_3",
                data: {
                    documentUri: "dummy/uri/for/resolve_completion.yml",
                    moduleFqcn: "org_1.coll_3.module_3",
                    inlineCollections: ["org_1.coll_3", "ansible.builtin"],
                    atEndOfLine: true,
                    firstElementOfList: false,
                },
            },
            completionText: "module_3",
        },
        {
            name: "module name with full FQCN since it is not present in declared collections in playbook",
            completionItem: {
                label: "module_1",
                data: {
                    documentUri: "dummy/uri/for/resolve_completion.yml",
                    moduleFqcn: "org_1.coll_1.module_1",
                    inlineCollections: ["org_1.coll_3", "ansible.builtin"],
                    atEndOfLine: true,
                    firstElementOfList: false,
                },
            },
            completionText: "org_1.coll_1.module_1",
        },
    ];
    tests.forEach(({ name, completionItem, completionText }) => {
        it(`should resolve completion for ${name}`, function () {
            return __awaiter(this, void 0, void 0, function* () {
                const actualCompletionResolveAtLineEnd = yield (0, completionProvider_1.doCompletionResolve)(completionItem, context);
                (0, chai_1.expect)(actualCompletionResolveAtLineEnd.insertText).be.equal(`${completionText}:${os_1.EOL}\t`);
                // Check for completion resolution when asked in between of lines
                completionItem.data.atEndOfLine = false;
                const actualCompletionResolveAtInBetween = yield (0, completionProvider_1.doCompletionResolve)(completionItem, context);
                (0, chai_1.expect)(actualCompletionResolveAtInBetween.insertText).be.equal(`${completionText}`);
            });
        });
    });
}
function testResolveModuleOptionCompletion(context) {
    const tests = [
        {
            name: "option expecting dictionary with `option: ${EOL}\\t\\t`",
            completionItem: {
                label: "opt_1",
                data: {
                    documentUri: "dummy/uri/for/resolve_completion.yml",
                    type: "dict",
                    atEndOfLine: true,
                    firstElementOfList: true,
                },
            },
            completionText: "opt_1",
        },
        {
            name: "sub option expecting list with `sub_option: ${EOL}\\t- `",
            completionItem: {
                label: "sub_opt_2",
                data: {
                    documentUri: "dummy/uri/for/resolve_completion.yml",
                    type: "list",
                    atEndOfLine: true,
                },
            },
            completionText: "sub_opt_2",
        },
        {
            name: "sub option expecting string or number or boolean with `sub_option: `",
            completionItem: {
                label: "sub_opt_1",
                data: {
                    documentUri: "dummy/uri/for/resolve_completion.yml",
                    type: "string",
                    atEndOfLine: true,
                },
            },
            completionText: "sub_opt_1",
        },
    ];
    tests.forEach(({ name, completionItem, completionText }) => {
        it(`should resolve completion for ${name}`, function () {
            return __awaiter(this, void 0, void 0, function* () {
                const actualCompletionResolveAtLineEnd = yield (0, completionProvider_1.doCompletionResolve)(completionItem, context);
                let returnSuffix;
                switch (completionItem.data.type) {
                    case "list":
                        returnSuffix = completionItem.data.firstElementOfList
                            ? `${os_1.EOL}\t\t- `
                            : `${os_1.EOL}\t- `;
                        break;
                    case "dict":
                        returnSuffix = completionItem.data.firstElementOfList
                            ? `${os_1.EOL}\t\t`
                            : `${os_1.EOL}\t`;
                        break;
                    default:
                        returnSuffix = " ";
                        break;
                }
                (0, chai_1.expect)(actualCompletionResolveAtLineEnd.insertText).be.equal(`${completionText}:${returnSuffix}`);
                // Check for completion resolution when asked in between of lines
                completionItem.data.atEndOfLine = false;
                const actualCompletionResolveAtInBetween = yield (0, completionProvider_1.doCompletionResolve)(completionItem, context);
                (0, chai_1.expect)(actualCompletionResolveAtInBetween.insertText).be.equal(`${completionText}`);
            });
        });
    });
}
describe("doCompletionResolve()", () => {
    const workspaceManager = (0, helper_1.createTestWorkspaceManager)();
    const fixtureFilePath = "completion/resolve_completion.yml";
    const fixtureFileUri = (0, helper_1.resolveDocUri)(fixtureFilePath);
    const context = workspaceManager.getContext(fixtureFileUri);
    const textDoc = (0, helper_1.getDoc)(fixtureFilePath);
    const docSettings = context.documentSettings.get(textDoc.uri);
    describe("Resolve completion for module names", () => {
        describe("With useFQCN enabled and with EE enabled @ee", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)("/home/runner/.ansible/collections:/usr/share/ansible");
                yield (0, helper_1.enableExecutionEnvironmentSettings)(docSettings);
            }));
            testFQCNEnabled(context);
            after(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
        });
        describe("With useFQCN enabled and with EE disabled", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
            testFQCNEnabled(context);
        });
        describe("With useFQCN disabled and with EE enabled @ee", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)("/home/runner/.ansible/collections:/usr/share/ansible");
                yield (0, helper_1.enableExecutionEnvironmentSettings)(docSettings);
                (yield docSettings).ansible.useFullyQualifiedCollectionNames = false;
            }));
            testFQCNDisabled(context);
            after(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
                (yield docSettings).ansible.useFullyQualifiedCollectionNames = true;
            }));
        });
        describe("With useFQCN disabled and with EE disabled", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
                (yield docSettings).ansible.useFullyQualifiedCollectionNames = false;
            }));
            testFQCNDisabled(context);
            after(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                (yield docSettings).ansible.useFullyQualifiedCollectionNames = true;
            }));
        });
    });
    describe("Resolve completion for module options and suboptions", () => {
        describe("with EE enabled @ee", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)("/home/runner/.ansible/collections:/usr/share/ansible");
                yield (0, helper_1.enableExecutionEnvironmentSettings)(docSettings);
            }));
            testResolveModuleOptionCompletion(context);
            after(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
        });
        describe("with EE disabled", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
            testResolveModuleOptionCompletion(context);
        });
    });
});
//# sourceMappingURL=completionResolver.test.js.map