/**
 * deps/ecc/curves.js - Elliptic Curve NIST/SECG/X9.62 Parameters
 * Original Copyright (c) 2003-2005  Tom Wu.
 * Modifications Copyright (c) 2015 Cisco Systems, Inc.  See LICENSE file.
 *
 * Ported from Tom Wu, which is ported from BouncyCastle
 * Modified to reuse existing external NPM modules, restricted to the
 * NIST//SECG/X9.62 prime curves only, and formatted to match project
 * coding styles.
 */
"use strict";

// Named EC curves

var BigInteger = require("../../deps/forge").jsbn.BigInteger,
    ec = require("./math.js");

// ----------------
// X9ECParameters

// constructor
function X9ECParameters(curve, g, n, h) {
  this.curve = curve;
  this.g = g;
  this.n = n;
  this.h = h;
}

function x9getCurve() {
  return this.curve;
}

function x9getG() {
  return this.g;
}

function x9getN() {
  return this.n;
}

function x9getH() {
  return this.h;
}

X9ECParameters.prototype.getCurve = x9getCurve;
X9ECParameters.prototype.getG = x9getG;
X9ECParameters.prototype.getN = x9getN;
X9ECParameters.prototype.getH = x9getH;

// ----------------
// SECNamedCurves

function fromHex(s) { return new BigInteger(s, 16); }

function secp256r1() {
  // p = 2^224 (2^32 - 1) + 2^192 + 2^96 - 1
  var p = fromHex("FFFFFFFF00000001000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFF");
  var a = fromHex("FFFFFFFF00000001000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFC");
  var b = fromHex("5AC635D8AA3A93E7B3EBBD55769886BC651D06B0CC53B0F63BCE3C3E27D2604B");
  var n = fromHex("FFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC632551");
  var h = BigInteger.ONE;
  var curve = new ec.ECCurveFp(p, a, b);
  var G = curve.decodePointHex("04"
              + "6B17D1F2E12C4247F8BCE6E563A440F277037D812DEB33A0F4A13945D898C296"
              + "4FE342E2FE1A7F9B8EE7EB4A7C0F9E162BCE33576B315ECECBB6406837BF51F5");
  return new X9ECParameters(curve, G, n, h);
}

function secp384r1() {
  // p = 2^384 - 2^128 - 2^96 + 2^32 - 1
  var p = fromHex("FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFF0000000000000000FFFFFFFF");
  var a = fromHex("FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFF0000000000000000FFFFFFFC");
  var b = fromHex("B3312FA7E23EE7E4988E056BE3F82D19181D9C6EFE8141120314088F5013875AC656398D8A2ED19D2A85C8EDD3EC2AEF");
  var n = fromHex("FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFC7634D81F4372DDF581A0DB248B0A77AECEC196ACCC52973");
  var h = BigInteger.ONE;
  var curve = new ec.ECCurveFp(p, a, b);
  var G = curve.decodePointHex("04"
              + "AA87CA22BE8B05378EB1C71EF320AD746E1D3B628BA79B9859F741E082542A385502F25DBF55296C3A545E3872760AB7"
              + "3617DE4A96262C6F5D9E98BF9292DC29F8F41DBD289A147CE9DA3113B5F0B8C00A60B1CE1D7E819D7A431D7C90EA0E5F");
  return new X9ECParameters(curve, G, n, h);
}

function secp521r1() {
  // p = 2^521 - 1
  var p = fromHex("01FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF");
  var a = fromHex("01FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFC");
  var b = fromHex("0051953EB9618E1C9A1F929A21A0B68540EEA2DA725B99B315F3B8B489918EF109E156193951EC7E937B1652C0BD3BB1BF073573DF883D2C34F1EF451FD46B503F00");
  var n = fromHex("01FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFA51868783BF2F966B7FCC0148F709A5D03BB5C9B8899C47AEBB6FB71E91386409");
  var h = BigInteger.ONE;
  var curve = new ec.ECCurveFp(p, a, b);
  var G = curve.decodePointHex("04"
                + "00C6858E06B70404E9CD9E3ECB662395B4429C648139053FB521F828AF606B4D3DBAA14B5E77EFE75928FE1DC127A2FFA8DE3348B3C1856A429BF97E7E31C2E5BD66"
                + "011839296A789A3BC0045C8A5FB42C7D1BD998F54449579B446817AFBD17273E662C97EE72995EF42640C550B9013FAD0761353C7086A272C24088BE94769FD16650");
  return new X9ECParameters(curve, G, n, h);
}

// ----------------
// Public API

var CURVES = module.exports = {
  "secp256r1": secp256r1(),
  "secp384r1": secp384r1(),
  "secp521r1": secp521r1()
};

// also export NIST names
CURVES["P-256"] = CURVES.secp256r1;
CURVES["P-384"] = CURVES.secp384r1;
CURVES["P-521"] = CURVES.secp521r1;
