/*
 * Decompiled with CFR 0.152.
 */
package org.apache.helix.common.caches;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import org.apache.helix.HelixDataAccessor;
import org.apache.helix.HelixDefinedState;
import org.apache.helix.PropertyKey;
import org.apache.helix.model.CurrentState;
import org.apache.helix.model.LiveInstance;
import org.apache.helix.model.Message;
import org.apache.helix.util.HelixUtil;
import org.apache.helix.util.RebalanceUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class InstanceMessagesCache {
    private static final Logger LOG = LoggerFactory.getLogger((String)InstanceMessagesCache.class.getName());
    private Map<String, Map<String, Message>> _messageMap;
    private Map<String, Map<String, Message>> _relayMessageMap;
    private Map<String, Map<String, Message>> _messageCache = Maps.newHashMap();
    private Map<String, Map<String, Message>> _relayMessageCache = Maps.newHashMap();
    private Map<String, Message> _relayHostMessageCache = Maps.newHashMap();
    public static final String RELAY_MESSAGE_LIFETIME = "helix.controller.messagecache.relaymessagelifetime";
    private static final long DEFAULT_RELAY_MESSAGE_LIFETIME = TimeUnit.MINUTES.toMillis(60L);
    private final long _relayMessageLifetime;
    private String _clusterName;

    public InstanceMessagesCache(String clusterName) {
        this._clusterName = clusterName;
        this._relayMessageLifetime = HelixUtil.getSystemPropertyAsLong(RELAY_MESSAGE_LIFETIME, DEFAULT_RELAY_MESSAGE_LIFETIME);
    }

    public boolean refresh(HelixDataAccessor accessor, Map<String, LiveInstance> liveInstanceMap) {
        LOG.info("START: InstanceMessagesCache.refresh()");
        long startTime = System.currentTimeMillis();
        PropertyKey.Builder keyBuilder = accessor.keyBuilder();
        HashMap<String, HashMap> msgMap = new HashMap<String, HashMap>();
        LinkedList newMessageKeys = Lists.newLinkedList();
        long purgeSum = 0L;
        for (String instanceName : liveInstanceMap.keySet()) {
            HashMap cachedMap = this._messageCache.get(instanceName);
            if (cachedMap == null) {
                cachedMap = Maps.newHashMap();
                this._messageCache.put(instanceName, cachedMap);
            }
            msgMap.put(instanceName, cachedMap);
            HashSet messageNames = Sets.newHashSet(accessor.getChildNames(keyBuilder.messages(instanceName)));
            long purgeStart = System.currentTimeMillis();
            Iterator cachedNamesIter = cachedMap.keySet().iterator();
            while (cachedNamesIter.hasNext()) {
                String messageName = (String)cachedNamesIter.next();
                if (messageNames.contains(messageName)) continue;
                cachedNamesIter.remove();
            }
            long purgeEnd = System.currentTimeMillis();
            purgeSum += purgeEnd - purgeStart;
            for (String messageName : messageNames) {
                if (cachedMap.containsKey(messageName)) continue;
                newMessageKeys.add(keyBuilder.message(instanceName, messageName));
            }
        }
        if (newMessageKeys.size() > 0) {
            List newMessages = accessor.getProperty(newMessageKeys, true);
            for (Message message : newMessages) {
                if (message == null) continue;
                Map<String, Message> cachedMap = this._messageCache.get(message.getTgtName());
                cachedMap.put(message.getId(), message);
            }
        }
        this._messageMap = Collections.unmodifiableMap(msgMap);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Message purge took: {} ", (Object)purgeSum);
        }
        LOG.info("END: InstanceMessagesCache.refresh(), {} of Messages read from ZooKeeper. took {} ms. ", (Object)newMessageKeys.size(), (Object)(System.currentTimeMillis() - startTime));
        return true;
    }

    public void updateRelayMessages(Map<String, LiveInstance> liveInstanceMap, Map<String, Map<String, Map<String, CurrentState>>> currentStateMap) {
        for (String instance : this._messageMap.keySet()) {
            Map<String, Message> instanceMessages = this._messageMap.get(instance);
            for (Message message : instanceMessages.values()) {
                if (!message.hasRelayMessages()) continue;
                for (Message message2 : message.getRelayMessages().values()) {
                    this.cacheRelayMessage(message2, message);
                }
            }
        }
        long nextRebalanceTime = Long.MAX_VALUE;
        long currentTime = System.currentTimeMillis();
        HashMap<String, HashMap> relayMessageMap = new HashMap<String, HashMap>();
        HashSet<String> targetInstanceToRemove = new HashSet<String>();
        for (String targetInstance : this._relayMessageCache.keySet()) {
            Map<String, Message> relayMessages = this._relayMessageCache.get(targetInstance);
            Iterator<Map.Entry<String, Message>> iterator = relayMessages.entrySet().iterator();
            while (iterator.hasNext()) {
                Message relayMessage = iterator.next().getValue();
                Map<String, Message> instanceMsgMap = this._messageMap.get(targetInstance);
                if (!relayMessage.isValid()) {
                    LOG.warn("Invalid relay message {}, remove it from the cache.", (Object)relayMessage.getId());
                    iterator.remove();
                    this._relayHostMessageCache.remove(relayMessage.getMsgId());
                    continue;
                }
                if (instanceMsgMap != null && instanceMsgMap.containsKey(relayMessage.getMsgId())) {
                    Message committedMessage = instanceMsgMap.get(relayMessage.getMsgId());
                    if (committedMessage.isRelayMessage()) {
                        LOG.info("Relay message already committed, remove relay message {} from the cache.", (Object)relayMessage.getId());
                        iterator.remove();
                        this._relayHostMessageCache.remove(relayMessage.getMsgId());
                        continue;
                    }
                    LOG.info("Controller already sent the message to the target host, set relay message {} to be expired.", (Object)relayMessage.getId());
                    this.setMessageRelayTime(relayMessage, currentTime);
                }
                try {
                    this.checkTargetHost(targetInstance, relayMessage, liveInstanceMap, currentStateMap);
                    Message hostedMessage = this._relayHostMessageCache.get(relayMessage.getMsgId());
                    this.checkRelayHost(relayMessage, liveInstanceMap, currentStateMap, hostedMessage);
                }
                catch (Exception e) {
                    LOG.warn("Failed to check target and relay host and set the relay time. Relay message: {} exception: {}", (Object)relayMessage.getId(), (Object)e);
                }
                if (relayMessage.isExpired()) {
                    LOG.info("relay message {} expired, remove it from cache. relay time {}.", (Object)relayMessage.getId(), (Object)relayMessage.getRelayTime());
                    iterator.remove();
                    this._relayHostMessageCache.remove(relayMessage.getMsgId());
                    continue;
                }
                if (relayMessage.getRelayTime() < 0L && relayMessage.getCreateTimeStamp() + this._relayMessageLifetime < System.currentTimeMillis()) {
                    LOG.info("relay message {} has reached its lifetime, remove it from cache.", (Object)relayMessage.getId());
                    iterator.remove();
                    this._relayHostMessageCache.remove(relayMessage.getMsgId());
                    continue;
                }
                if (!relayMessageMap.containsKey(targetInstance)) {
                    relayMessageMap.put(targetInstance, Maps.newHashMap());
                }
                ((Map)relayMessageMap.get(targetInstance)).put(relayMessage.getMsgId(), relayMessage);
                long expiryTime = relayMessage.getCreateTimeStamp() + this._relayMessageLifetime;
                if (relayMessage.getRelayTime() > 0L) {
                    expiryTime = relayMessage.getRelayTime() + relayMessage.getExpiryPeriod();
                }
                if (expiryTime >= nextRebalanceTime) continue;
                nextRebalanceTime = expiryTime;
            }
            if (!relayMessages.isEmpty()) continue;
            targetInstanceToRemove.add(targetInstance);
        }
        this._relayMessageCache.keySet().removeAll(targetInstanceToRemove);
        if (nextRebalanceTime < Long.MAX_VALUE) {
            this.scheduleFuturePipeline(nextRebalanceTime);
        }
        this._relayMessageMap = Collections.unmodifiableMap(relayMessageMap);
        long l = 0L;
        for (String instance : this._relayMessageMap.keySet()) {
            Map<String, Message> relayMessages = this._relayMessageMap.get(instance);
            if (!this._messageMap.containsKey(instance)) {
                this._messageMap.put(instance, Maps.newHashMap());
            }
            this._messageMap.get(instance).putAll(relayMessages);
            l += (long)relayMessages.size();
        }
        LOG.info("END: updateRelayMessages(), {} of valid relay messages in cache, took {} ms. ", (Object)l, (Object)(System.currentTimeMillis() - currentTime));
    }

    private void checkTargetHost(String targetHost, Message relayMessage, Map<String, LiveInstance> liveInstanceMap, Map<String, Map<String, Map<String, CurrentState>>> currentStateMap) {
        long currentTime = System.currentTimeMillis();
        String resourceName = relayMessage.getResourceName();
        String partitionName = relayMessage.getPartitionName();
        String sessionId = relayMessage.getTgtSessionId();
        if (!liveInstanceMap.containsKey(targetHost)) {
            LOG.info("Target host is not alive anymore, expiring relay message {} immediately.", (Object)relayMessage.getId());
            relayMessage.setExpired(true);
            return;
        }
        String instanceSessionId = liveInstanceMap.get(targetHost).getEphemeralOwner();
        if (!instanceSessionId.equals(sessionId)) {
            LOG.info("Instance SessionId does not match, expiring relay message {} immediately.", (Object)relayMessage.getId());
            relayMessage.setExpired(true);
            return;
        }
        Map<String, Map<String, CurrentState>> instanceCurrentStateMap = currentStateMap.get(targetHost);
        if (instanceCurrentStateMap == null || !instanceCurrentStateMap.containsKey(sessionId)) {
            LOG.warn("CurrentStateMap null for {}, session {}, pending relay message {}", new Object[]{targetHost, sessionId, relayMessage.getId()});
            return;
        }
        Map<String, CurrentState> sessionCurrentStateMap = instanceCurrentStateMap.get(sessionId);
        CurrentState currentState = sessionCurrentStateMap.get(resourceName);
        if (currentState == null) {
            this.setMessageRelayTime(relayMessage, currentTime);
            LOG.warn("CurrentState is null for {} on {}, set relay time {} for message {}", new Object[]{resourceName, targetHost, relayMessage.getRelayTime(), relayMessage.getId()});
            return;
        }
        String partitionCurrentState = currentState.getState(partitionName);
        String targetState = relayMessage.getToState();
        String fromState = relayMessage.getFromState();
        if (targetState.equals(partitionCurrentState) || !fromState.equals(partitionCurrentState)) {
            this.setMessageRelayTime(relayMessage, currentTime);
            LOG.debug("{}'s currentState {} on {} has changed, set relay message {} to be expired.", new Object[]{partitionName, partitionCurrentState, targetHost, relayMessage.getId()});
        }
    }

    private void checkRelayHost(Message relayMessage, Map<String, LiveInstance> liveInstanceMap, Map<String, Map<String, Map<String, CurrentState>>> currentStateMap, Message hostedMessage) {
        long currentTime = System.currentTimeMillis();
        String sessionId = hostedMessage.getTgtSessionId();
        String relayInstance = hostedMessage.getTgtName();
        String resourceName = hostedMessage.getResourceName();
        String partitionName = hostedMessage.getPartitionName();
        if (!liveInstanceMap.containsKey(relayInstance)) {
            this.setMessageRelayTime(relayMessage, currentTime);
            return;
        }
        String instanceSessionId = liveInstanceMap.get(relayInstance).getEphemeralOwner();
        if (!instanceSessionId.equals(sessionId)) {
            LOG.info("Relay instance sessionId {} does not match sessionId {} in hosted message {}, set relay message {} to be expired.", new Object[]{instanceSessionId, sessionId, relayMessage.getId(), hostedMessage.getMsgId()});
            this.setMessageRelayTime(relayMessage, currentTime);
            return;
        }
        Map<String, Map<String, CurrentState>> instanceCurrentStateMap = currentStateMap.get(relayInstance);
        if (instanceCurrentStateMap == null || !instanceCurrentStateMap.containsKey(sessionId)) {
            LOG.warn("CurrentStateMap null for {}, session {}, set relay messages {} to be expired. Hosted message {}.", new Object[]{relayInstance, sessionId, relayMessage.getId(), hostedMessage.getId()});
            this.setMessageRelayTime(relayMessage, currentTime);
            return;
        }
        Map<String, CurrentState> sessionCurrentStateMap = instanceCurrentStateMap.get(sessionId);
        CurrentState currentState = sessionCurrentStateMap.get(resourceName);
        if (currentState == null) {
            LOG.info("No currentState found for {} on {}, set relay message {} to be expired.", new Object[]{resourceName, relayInstance, relayMessage.getId()});
            this.setMessageRelayTime(relayMessage, currentTime);
            return;
        }
        String partitionState = currentState.getState(partitionName);
        String targetState = hostedMessage.getToState();
        String fromState = hostedMessage.getFromState();
        if (!fromState.equals(partitionState)) {
            long completeTime;
            if (HelixDefinedState.ERROR.name().equals(partitionState) && fromState.equals(currentState.getPreviousState(partitionName))) {
                LOG.info("Partition {} got to ERROR from the top state, expiring relay message {} immediately. Hosted message {}.", new Object[]{partitionName, relayMessage.getId(), hostedMessage.getId()});
                relayMessage.setExpired(true);
                return;
            }
            if (targetState.equals(partitionState) && fromState.equals(currentState.getPreviousState(partitionName)) && (completeTime = currentState.getEndTime(partitionName)) > relayMessage.getCreateTimeStamp()) {
                this.setMessageRelayTime(relayMessage, completeTime);
                LOG.error("Target state for partition {} matches the hosted message's target state, set relay message {} to be expired.", (Object)partitionName, (Object)relayMessage.getId());
                return;
            }
            this.setMessageRelayTime(relayMessage, currentTime);
            LOG.info("Current state {} for partition {} does not match hosted message's from state, set relay message {} to be expired.", new Object[]{partitionState, partitionName, relayMessage.getId()});
        }
    }

    private void setMessageRelayTime(Message relayMessage, long relayTime) {
        long currentRelayTime = relayMessage.getRelayTime();
        if (currentRelayTime > relayMessage.getCreateTimeStamp() && currentRelayTime < relayTime) {
            return;
        }
        relayMessage.setRelayTime(relayTime);
        LOG.info("Set relay message {} relay time at {}, to be expired at {}", new Object[]{relayMessage.getId(), relayTime, relayTime + relayMessage.getExpiryPeriod()});
    }

    private void scheduleFuturePipeline(long rebalanceTime) {
        long current = System.currentTimeMillis();
        long delay = rebalanceTime - current;
        RebalanceUtil.scheduleOnDemandPipeline(this._clusterName, delay);
    }

    public Map<String, Message> getMessages(String instanceName) {
        if (this._messageMap.containsKey(instanceName)) {
            return this._messageMap.get(instanceName);
        }
        return Collections.emptyMap();
    }

    public Map<String, Message> getRelayMessages(String instanceName) {
        if (this._relayMessageMap.containsKey(instanceName)) {
            return this._relayMessageMap.get(instanceName);
        }
        return Collections.emptyMap();
    }

    public void cacheMessages(Collection<Message> messages) {
        for (Message message : messages) {
            String instanceName = message.getTgtName();
            if (!this._messageCache.containsKey(instanceName)) {
                this._messageCache.put(instanceName, Maps.newHashMap());
            }
            this._messageCache.get(instanceName).put(message.getId(), message);
            if (!message.hasRelayMessages()) continue;
            for (Message relayMsg : message.getRelayMessages().values()) {
                this.cacheRelayMessage(relayMsg, message);
            }
        }
    }

    private void cacheRelayMessage(Message relayMessage, Message hostMessage) {
        String instanceName = relayMessage.getTgtName();
        if (!this._relayMessageCache.containsKey(instanceName)) {
            this._relayMessageCache.put(instanceName, Maps.newHashMap());
        }
        if (!this._relayMessageCache.get(instanceName).containsKey(relayMessage.getId())) {
            LOG.info("Add relay message to relay cache " + relayMessage.getMsgId() + ", hosted message " + hostMessage.getMsgId());
        }
        this._relayMessageCache.get(instanceName).put(relayMessage.getId(), relayMessage);
        this._relayHostMessageCache.put(relayMessage.getMsgId(), hostMessage);
    }

    public String toString() {
        return "InstanceMessagesCache{_messageMap=" + this._messageMap + ", _messageCache=" + this._messageCache + ", _clusterName='" + this._clusterName + '\'' + '}';
    }
}

