/*
 * Decompiled with CFR 0.152.
 */
package org.apache.helix.rest.server.service;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.helix.ConfigAccessor;
import org.apache.helix.HelixDataAccessor;
import org.apache.helix.HelixException;
import org.apache.helix.model.CurrentState;
import org.apache.helix.model.InstanceConfig;
import org.apache.helix.model.LiveInstance;
import org.apache.helix.model.RESTConfig;
import org.apache.helix.rest.client.CustomRestClient;
import org.apache.helix.rest.client.CustomRestClientFactory;
import org.apache.helix.rest.common.HelixDataAccessorWrapper;
import org.apache.helix.rest.server.json.instance.InstanceInfo;
import org.apache.helix.rest.server.json.instance.StoppableCheck;
import org.apache.helix.rest.server.service.InstanceService;
import org.apache.helix.util.InstanceValidationUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class InstanceServiceImpl
implements InstanceService {
    private static final Logger LOG = LoggerFactory.getLogger(InstanceServiceImpl.class);
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();
    private static final ExecutorService POOL = Executors.newCachedThreadPool();
    private final HelixDataAccessorWrapper _dataAccessor;
    private final ConfigAccessor _configAccessor;
    private final CustomRestClient _customRestClient;

    public InstanceServiceImpl(HelixDataAccessorWrapper dataAccessor, ConfigAccessor configAccessor) {
        this._dataAccessor = dataAccessor;
        this._configAccessor = configAccessor;
        this._customRestClient = CustomRestClientFactory.get();
    }

    @VisibleForTesting
    InstanceServiceImpl(HelixDataAccessorWrapper dataAccessor, ConfigAccessor configAccessor, CustomRestClient customRestClient) {
        this._dataAccessor = dataAccessor;
        this._configAccessor = configAccessor;
        this._customRestClient = customRestClient;
    }

    @Override
    public InstanceInfo getInstanceInfo(String clusterId, String instanceName, List<InstanceService.HealthCheck> healthChecks) {
        InstanceInfo.Builder instanceInfoBuilder = new InstanceInfo.Builder(instanceName);
        InstanceConfig instanceConfig = (InstanceConfig)this._dataAccessor.getProperty(this._dataAccessor.keyBuilder().instanceConfig(instanceName));
        LiveInstance liveInstance = (LiveInstance)this._dataAccessor.getProperty(this._dataAccessor.keyBuilder().liveInstance(instanceName));
        if (instanceConfig != null) {
            instanceInfoBuilder.instanceConfig(instanceConfig.getRecord());
        }
        if (liveInstance != null) {
            instanceInfoBuilder.liveInstance(liveInstance.getRecord());
            String sessionId = liveInstance.getEphemeralOwner();
            List<String> resourceNames = this._dataAccessor.getChildNames(this._dataAccessor.keyBuilder().currentStates(instanceName, sessionId));
            instanceInfoBuilder.resources(resourceNames);
            ArrayList<String> partitions = new ArrayList<String>();
            for (String resourceName : resourceNames) {
                CurrentState currentState = (CurrentState)this._dataAccessor.getProperty(this._dataAccessor.keyBuilder().currentState(instanceName, sessionId, resourceName));
                if (currentState == null || currentState.getPartitionStateMap() == null) continue;
                partitions.addAll(currentState.getPartitionStateMap().keySet());
            }
            instanceInfoBuilder.partitions(partitions);
        }
        try {
            Map<String, Boolean> healthStatus = this.getInstanceHealthStatus(clusterId, instanceName, healthChecks);
            instanceInfoBuilder.healthStatus(healthStatus);
        }
        catch (HelixException ex) {
            LOG.error("Exception while getting health status. Cluster: {}, Instance: {}, reporting health status as unHealth", new Object[]{clusterId, instanceName, ex});
            instanceInfoBuilder.healthStatus(false);
        }
        return instanceInfoBuilder.build();
    }

    @Override
    public StoppableCheck getInstanceStoppableCheck(String clusterId, String instanceName, String jsonContent) throws IOException {
        return this.batchGetInstancesStoppableChecks(clusterId, (List<String>)ImmutableList.of((Object)instanceName), jsonContent).get(instanceName);
    }

    @Override
    public Map<String, StoppableCheck> batchGetInstancesStoppableChecks(String clusterId, List<String> instances, String jsonContent) throws IOException {
        HashMap<String, StoppableCheck> finalStoppableChecks = new HashMap<String, StoppableCheck>();
        Map<String, Future<StoppableCheck>> helixInstanceChecks = instances.stream().collect(Collectors.toMap(Function.identity(), instance -> POOL.submit(() -> this.performHelixOwnInstanceCheck(clusterId, (String)instance))));
        List<String> instancesForCustomInstanceLevelChecks = this.filterInstancesForNextCheck(helixInstanceChecks, finalStoppableChecks);
        if (instancesForCustomInstanceLevelChecks.isEmpty()) {
            return finalStoppableChecks;
        }
        RESTConfig restConfig = this._configAccessor.getRESTConfig(clusterId);
        if (restConfig == null) {
            String errorMessage = String.format("The cluster %s hasn't enabled client side health checks yet, thus the stoppable check result is inaccurate", clusterId);
            LOG.error(errorMessage);
            throw new HelixException(errorMessage);
        }
        Map<String, String> customPayLoads = this.getCustomPayLoads(jsonContent);
        Map<String, Future<StoppableCheck>> customInstanceLevelChecks = instancesForCustomInstanceLevelChecks.stream().collect(Collectors.toMap(Function.identity(), instance -> POOL.submit(() -> this.performCustomInstanceCheck(clusterId, (String)instance, restConfig.getBaseUrl(instance), customPayLoads))));
        List<String> instancesForCustomPartitionLevelChecks = this.filterInstancesForNextCheck(customInstanceLevelChecks, finalStoppableChecks);
        if (!instancesForCustomPartitionLevelChecks.isEmpty()) {
            Map<String, StoppableCheck> instancePartitionLevelChecks = this.performPartitionsCheck(instancesForCustomPartitionLevelChecks, restConfig, customPayLoads);
            for (Map.Entry<String, StoppableCheck> instancePartitionStoppableCheckEntry : instancePartitionLevelChecks.entrySet()) {
                finalStoppableChecks.put(instancePartitionStoppableCheckEntry.getKey(), instancePartitionStoppableCheckEntry.getValue());
            }
        }
        return finalStoppableChecks;
    }

    private List<String> filterInstancesForNextCheck(Map<String, Future<StoppableCheck>> futureStoppableCheckByInstance, Map<String, StoppableCheck> finalStoppableCheckByInstance) {
        ArrayList<String> instancesForNextCheck = new ArrayList<String>();
        for (Map.Entry<String, Future<StoppableCheck>> entry : futureStoppableCheckByInstance.entrySet()) {
            String instance = entry.getKey();
            try {
                StoppableCheck stoppableCheck = entry.getValue().get();
                if (!stoppableCheck.isStoppable()) {
                    finalStoppableCheckByInstance.put(instance, stoppableCheck);
                    continue;
                }
                instancesForNextCheck.add(instance);
            }
            catch (InterruptedException | ExecutionException e) {
                LOG.error("Failed to get StoppableChecks in parallel. Instance: {}", (Object)instance, (Object)e);
            }
        }
        return instancesForNextCheck;
    }

    private StoppableCheck performHelixOwnInstanceCheck(String clusterId, String instanceName) {
        LOG.info("Perform helix own custom health checks for {}/{}", (Object)clusterId, (Object)instanceName);
        Map<String, Boolean> helixStoppableCheck = this.getInstanceHealthStatus(clusterId, instanceName, InstanceService.HealthCheck.STOPPABLE_CHECK_LIST);
        return new StoppableCheck(helixStoppableCheck, StoppableCheck.Category.HELIX_OWN_CHECK);
    }

    private StoppableCheck performCustomInstanceCheck(String clusterId, String instanceName, String baseUrl, Map<String, String> customPayLoads) {
        LOG.info("Perform instance level client side health checks for {}/{}", (Object)clusterId, (Object)instanceName);
        try {
            return new StoppableCheck(this._customRestClient.getInstanceStoppableCheck(baseUrl, customPayLoads), StoppableCheck.Category.CUSTOM_INSTANCE_CHECK);
        }
        catch (IOException ex) {
            LOG.error("Custom client side instance level health check for {}/{} failed.", new Object[]{clusterId, instanceName, ex});
            return new StoppableCheck(false, Arrays.asList(instanceName), StoppableCheck.Category.CUSTOM_INSTANCE_CHECK);
        }
    }

    private Map<String, StoppableCheck> performPartitionsCheck(List<String> instances, RESTConfig restConfig, Map<String, String> customPayLoads) {
        Map<String, Map<String, Boolean>> allPartitionsHealthOnLiveInstance = this._dataAccessor.getAllPartitionsHealthOnLiveInstance(restConfig, customPayLoads);
        List externalViews = this._dataAccessor.getChildValues(this._dataAccessor.keyBuilder().externalViews(), true);
        HashMap<String, StoppableCheck> instanceStoppableChecks = new HashMap<String, StoppableCheck>();
        for (String instanceName : instances) {
            List unHealthyPartitions = InstanceValidationUtil.perPartitionHealthCheck((List)externalViews, allPartitionsHealthOnLiveInstance, (String)instanceName, (HelixDataAccessor)this._dataAccessor);
            StoppableCheck stoppableCheck = new StoppableCheck(unHealthyPartitions.isEmpty(), unHealthyPartitions, StoppableCheck.Category.CUSTOM_PARTITION_CHECK);
            instanceStoppableChecks.put(instanceName, stoppableCheck);
        }
        return instanceStoppableChecks;
    }

    private Map<String, String> getCustomPayLoads(String jsonContent) throws IOException {
        HashMap<String, String> result = new HashMap<String, String>();
        JsonNode jsonNode = OBJECT_MAPPER.readTree(jsonContent);
        jsonNode.fields().forEachRemaining(kv -> result.put((String)kv.getKey(), ((JsonNode)kv.getValue()).asText()));
        return result;
    }

    @VisibleForTesting
    protected Map<String, Boolean> getInstanceHealthStatus(String clusterId, String instanceName, List<InstanceService.HealthCheck> healthChecks) {
        HashMap<String, Boolean> healthStatus = new HashMap<String, Boolean>();
        block10: for (InstanceService.HealthCheck healthCheck : healthChecks) {
            switch (healthCheck) {
                case INVALID_CONFIG: {
                    healthStatus.put(InstanceService.HealthCheck.INVALID_CONFIG.name(), InstanceValidationUtil.hasValidConfig((HelixDataAccessor)this._dataAccessor, (String)clusterId, (String)instanceName));
                    if (!((Boolean)healthStatus.get(InstanceService.HealthCheck.INVALID_CONFIG.name())).booleanValue()) {
                        LOG.error("The instance {} doesn't have valid configuration", (Object)instanceName);
                        return healthStatus;
                    }
                }
                case INSTANCE_NOT_ENABLED: {
                    healthStatus.put(InstanceService.HealthCheck.INSTANCE_NOT_ENABLED.name(), InstanceValidationUtil.isEnabled((HelixDataAccessor)this._dataAccessor, (String)instanceName));
                    continue block10;
                }
                case INSTANCE_NOT_ALIVE: {
                    healthStatus.put(InstanceService.HealthCheck.INSTANCE_NOT_ALIVE.name(), InstanceValidationUtil.isAlive((HelixDataAccessor)this._dataAccessor, (String)instanceName));
                    continue block10;
                }
                case INSTANCE_NOT_STABLE: {
                    boolean isStable = InstanceValidationUtil.isInstanceStable((HelixDataAccessor)this._dataAccessor, (String)instanceName);
                    healthStatus.put(InstanceService.HealthCheck.INSTANCE_NOT_STABLE.name(), isStable);
                    continue block10;
                }
                case HAS_ERROR_PARTITION: {
                    healthStatus.put(InstanceService.HealthCheck.HAS_ERROR_PARTITION.name(), !InstanceValidationUtil.hasErrorPartitions((HelixDataAccessor)this._dataAccessor, (String)clusterId, (String)instanceName));
                    continue block10;
                }
                case HAS_DISABLED_PARTITION: {
                    healthStatus.put(InstanceService.HealthCheck.HAS_DISABLED_PARTITION.name(), !InstanceValidationUtil.hasDisabledPartitions((HelixDataAccessor)this._dataAccessor, (String)clusterId, (String)instanceName));
                    continue block10;
                }
                case EMPTY_RESOURCE_ASSIGNMENT: {
                    healthStatus.put(InstanceService.HealthCheck.EMPTY_RESOURCE_ASSIGNMENT.name(), InstanceValidationUtil.hasResourceAssigned((HelixDataAccessor)this._dataAccessor, (String)clusterId, (String)instanceName));
                    continue block10;
                }
                case MIN_ACTIVE_REPLICA_CHECK_FAILED: {
                    healthStatus.put(InstanceService.HealthCheck.MIN_ACTIVE_REPLICA_CHECK_FAILED.name(), InstanceValidationUtil.siblingNodesActiveReplicaCheck((HelixDataAccessor)this._dataAccessor, (String)instanceName));
                    continue block10;
                }
            }
            LOG.error("Unsupported health check: {}", (Object)healthCheck);
        }
        return healthStatus;
    }
}

