/*
 * Decompiled with CFR 0.152.
 */
package groovyx.net.http;

import groovyx.net.http.NonBlockingCookieStore;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.net.HttpCookie;
import java.net.URI;
import java.net.URISyntaxException;
import java.time.Instant;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.ConcurrentModificationException;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.function.Consumer;

class FileBackedCookieStore
extends NonBlockingCookieStore {
    private static final ConcurrentMap<File, File> inUse = new ConcurrentHashMap<File, File>(5, 0.75f, 1);
    private static final int NUM_LOCKS = 16;
    private static final String SUFFIX = ".properties";
    private final File directory;
    private final Object[] locks;
    private final Executor executor;
    private final Consumer<Throwable> onException;
    private volatile boolean live = true;

    public FileBackedCookieStore(File directory, Executor executor, Consumer<Throwable> onException) {
        this.onException = onException;
        FileBackedCookieStore.ensureUniqueControl(directory);
        this.directory = directory;
        this.locks = new Object[16];
        for (int i = 0; i < 16; ++i) {
            this.locks[i] = new Object();
        }
        this.executor = executor;
        this.readAll();
    }

    public FileBackedCookieStore(File directory, Executor executor) {
        this(directory, executor, t -> {});
    }

    private static void ensureUniqueControl(File directory) {
        if (null != inUse.putIfAbsent(directory, directory)) {
            throw new ConcurrentModificationException(directory + " is already being used by another cookie store in this process");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void withLock(NonBlockingCookieStore.Key key, Runnable runner) {
        Object lock;
        Object object = lock = this.locks[Math.abs(key.hashCode() % 16)];
        synchronized (object) {
            runner.run();
        }
    }

    private void deleteFile(NonBlockingCookieStore.Key key) {
        File file = new File(this.directory, FileBackedCookieStore.fileName(key));
        if (file.exists()) {
            file.delete();
        }
    }

    @Override
    public void add(URI uri, HttpCookie cookie) {
        this.assertLive();
        NonBlockingCookieStore.Key key = NonBlockingCookieStore.Key.make(uri, cookie);
        this.add(key, cookie);
        if (cookie.getMaxAge() != -1L) {
            this.store(key, cookie);
        }
    }

    @Override
    public boolean remove(URI uri, HttpCookie cookie) {
        this.assertLive();
        return this.remove(NonBlockingCookieStore.Key.make(uri, cookie));
    }

    @Override
    public boolean removeAll() {
        this.assertLive();
        boolean ret = this.all.size() > 0;
        for (Map.Entry entry : this.all.entrySet()) {
            this.remove((NonBlockingCookieStore.Key)entry.getKey());
        }
        return ret;
    }

    @Override
    public boolean remove(NonBlockingCookieStore.Key key) {
        this.executor.execute(() -> this.withLock(key, () -> this.deleteFile(key)));
        return super.remove(key);
    }

    private static String clean(String str) {
        if (str == null) {
            return "";
        }
        String ret = str;
        if (ret.indexOf(47) != -1) {
            ret = ret.replace('/', '_');
        }
        if (ret.indexOf(92) != -1) {
            ret = ret.replace('\\', '_');
        }
        return ret;
    }

    private static String fileName(NonBlockingCookieStore.Key key) {
        if (key instanceof NonBlockingCookieStore.UriKey) {
            NonBlockingCookieStore.UriKey uriKey = (NonBlockingCookieStore.UriKey)key;
            return FileBackedCookieStore.clean(uriKey.host) + FileBackedCookieStore.clean(uriKey.name) + SUFFIX;
        }
        NonBlockingCookieStore.DomainKey domainKey = (NonBlockingCookieStore.DomainKey)key;
        return FileBackedCookieStore.clean(domainKey.domain) + FileBackedCookieStore.clean(domainKey.path) + FileBackedCookieStore.clean(domainKey.name) + SUFFIX;
    }

    private void store(NonBlockingCookieStore.Key key, HttpCookie cookie) {
        Runnable runner = () -> {
            File file = new File(this.directory, FileBackedCookieStore.fileName(key));
            try (FileWriter fw = new FileWriter(file);){
                this.toProperties(key, cookie).store(fw, "");
            }
            catch (IOException ioe) {
                this.onException.accept(ioe);
            }
        };
        this.executor.execute(() -> this.withLock(key, runner));
    }

    private void readAll() {
        ArrayList<CompletableFuture<Void>> futures = new ArrayList<CompletableFuture<Void>>();
        for (File file : this.directory.listFiles()) {
            Runnable loadFile = () -> {
                if (file.getName().endsWith(SUFFIX)) {
                    try (FileReader reader = new FileReader(file);){
                        Properties props = new Properties();
                        props.load(reader);
                        Map.Entry<NonBlockingCookieStore.Key, HttpCookie> entry = this.fromProperties(props);
                        if (entry != null) {
                            this.add(entry.getKey(), entry.getValue());
                        } else {
                            file.delete();
                        }
                    }
                    catch (IOException ioe) {
                        throw new RuntimeException(ioe);
                    }
                }
            };
            futures.add(CompletableFuture.runAsync(loadFile, this.executor));
        }
        try {
            CompletableFuture.allOf(futures.toArray(new CompletableFuture[0])).get();
        }
        catch (InterruptedException | ExecutionException e) {
            throw new RuntimeException(e);
        }
    }

    private void ifNotNull(Properties props, String key, String value) {
        if (value != null) {
            props.setProperty(key, value);
        }
    }

    private Properties keyProperties(NonBlockingCookieStore.Key key) {
        Properties props = new Properties();
        props.setProperty("keyType", key.getKeyType());
        if (key instanceof NonBlockingCookieStore.UriKey) {
            props.setProperty("uri", String.format("http://%s", ((NonBlockingCookieStore.UriKey)key).getURI().toString()));
        }
        return props;
    }

    private Properties toProperties(NonBlockingCookieStore.Key key, HttpCookie cookie) {
        Properties props = this.keyProperties(key);
        Instant expires = key.createdAt.plusSeconds(cookie.getMaxAge());
        props.setProperty("expires", expires.toString());
        props.setProperty("name", cookie.getName());
        props.setProperty("value", cookie.getValue());
        if (cookie.getDomain() != null) {
            props.setProperty("domain", cookie.getDomain());
        }
        props.setProperty("discard", Boolean.toString(cookie.getDiscard()));
        props.setProperty("secure", Boolean.toString(cookie.getSecure()));
        props.setProperty("version", Integer.toString(cookie.getVersion()));
        props.setProperty("httpOnly", Boolean.toString(cookie.isHttpOnly()));
        this.ifNotNull(props, "comment", cookie.getComment());
        this.ifNotNull(props, "commentURL", cookie.getCommentURL());
        this.ifNotNull(props, "path", cookie.getPath());
        this.ifNotNull(props, "portlist", cookie.getPortlist());
        return props;
    }

    private Map.Entry<NonBlockingCookieStore.Key, HttpCookie> fromProperties(Properties props, HttpCookie cookie) {
        String keyType = props.getProperty("keyType");
        if (NonBlockingCookieStore.UriKey.uriKey(keyType)) {
            try {
                return new AbstractMap.SimpleImmutableEntry<NonBlockingCookieStore.Key, HttpCookie>(new NonBlockingCookieStore.UriKey(new URI(props.getProperty("uri")), cookie), cookie);
            }
            catch (URISyntaxException e) {
                return null;
            }
        }
        return new AbstractMap.SimpleImmutableEntry<NonBlockingCookieStore.Key, HttpCookie>(new NonBlockingCookieStore.DomainKey(cookie), cookie);
    }

    private Map.Entry<NonBlockingCookieStore.Key, HttpCookie> fromProperties(Properties props) {
        String portlist;
        String path;
        String commentURL;
        String comment;
        Instant expires;
        Instant now = Instant.now();
        if (now.isAfter(expires = Instant.parse(props.getProperty("expires")))) {
            return null;
        }
        long maxAge = (expires.toEpochMilli() - now.toEpochMilli()) / 1000L;
        String name = props.getProperty("name");
        String value = props.getProperty("value");
        HttpCookie cookie = new HttpCookie(name, value);
        cookie.setDiscard(Boolean.valueOf(props.getProperty("discard")));
        cookie.setSecure(Boolean.valueOf(props.getProperty("secure")));
        cookie.setVersion(Integer.valueOf(props.getProperty("version")));
        cookie.setHttpOnly(Boolean.valueOf(props.getProperty("httpOnly")));
        String domain = props.getProperty("domain", null);
        if (null != domain) {
            cookie.setDomain(domain);
        }
        if (null != (comment = props.getProperty("comment", null))) {
            cookie.setComment(comment);
        }
        if (null != (commentURL = props.getProperty("commentURL", null))) {
            cookie.setCommentURL(commentURL);
        }
        if (null != (path = props.getProperty("path", null))) {
            cookie.setPath(path);
        }
        if (null != (portlist = props.getProperty("portlist", null))) {
            cookie.setPortlist(portlist);
        }
        return this.fromProperties(props, cookie);
    }

    public void shutdown() {
        this.live = false;
        inUse.remove(this.directory);
    }

    public void assertLive() {
        if (!this.live) {
            throw new IllegalStateException("You have already called shutdown on this object");
        }
    }
}

