#lang scribble/doc
@(require "ss.rkt" (for-label slideshow/play slideshow/code))

@title[#:tag "play"]{Animations}

@defmodule[slideshow/play]{The @racketmodname[slideshow/play] module
provides tools for generating animations as multiple, automatically
advanced slides.}

Many of the tools are based on a function that takes a number between
@racket[0.0] and @racket[1.0] inclusive and produces a
pict. The pict produced for the input @racket[0.0] is
the starting image of the animation, and the pict produced for
@racket[1.0] is the ending image, while intermediate values produced
intermediate images. For example,

@racketblock[
(lambda (n)
  (cellophane (t "Hello") n))
]

corresponds to an animation that fades in the word ``Hello.''

@; --------------------------------------------------

@section{Generating Animated Slides}

@defproc[(play [gen ((real-in 0.0 1.0) . -> . pict?)]
               [#:steps steps exact-positive-integer? (current-play-steps)]
               [#:delay delay-secs real? 0.05]
               [#:skip-first? skip-first? any/c #f]
               [#:title title (or/c string? pict? #f 
                                    ((real-in 0.0 1.0) . -> . (or/c string? pict? #f)))
                        #f]
               [#:name name (or/c string? #f
                                  ((real-in 0.0 1.0) . -> . (or/c string? #f)))
                       title]
               [#:aspect aspect aspect? #f]
               [#:comment comment (or/c comment? #f) #f]
               [#:layout layout (or/c 'auto 'center 'top 'tall) 'auto])
         void?]{

Generates @math{@racket[steps]+1} slides by calling @racket[gen] on
equally-spaced values from @racket[0.0] (inclusive) to @racket[1.0]
(exclusive). Except for the first of the slides, each slide has a
timeout of @racket[delay-secs], so that the next slide appears
automatically.

Normally, @racket[play] is called via @racket[play-n], which
effectively calls @racket[gen] on @racket[1.0] without a timeout to
complete the animation and stop the auto-advance of slides. The
@racket[play-n] function also manages with multi-step animations.

If @racket[skip-first?] is @racket[#f], then one fewer slide is
generated, because @racket[gen] is not called on @racket[0.0].

The @racket[title], @racket[name], @racket[aspect], and @racket[layout] arguments are
passed on to @racket[slide], at least when @racket[title] and/or
@racket[name] are not functions. When @racket[title] or @racket[name]
is a function, the function is applied to the value used to produce
the slide content, and the resulting title or name is passed on to
@racket[slide].

The @racket[comment] argument is used like a comment argument to
@racket[slide].

In condensed mode (i.e., when @racket[condense?] is @racket[#t]), any
slide that would be registered with a timeout is instead skipped.

@history[#:changed "1.7" @elem{Added the @racket[aspect] argument.}]}


@defproc[(play-n [gen* (and/c (unconstrained-domain-> pict?)
                              (λ (x) (number? (procedure-arity x))))]
                 [#:steps steps (list*of exact-positive-integer?
                                         (or/c exact-positive-integer? '()))
                          (current-play-steps)]
                 [#:delay delay-secs real? 0.05]
                 [#:skip-first? skip-first? any/c #f]
                 [#:skip-last? skip-last? any/c #f]
                 [#:title title (or/c string? pict? #f 
                                      ((real-in 0.0 1.0) . -> . (or/c string? pict? #f)))
                          #f]
                 [#:name name (or/c string? #f
                                    ((real-in 0.0 1.0) . -> . (or/c string? #f)))
                         title]
                 [#:aspect aspect aspect? #f]
                 [#:comments comment (list*of comment? (or/c comment? #f '())) #f]
                 [#:layout layout (or/c 'auto 'center 'top 'tall) 'auto])
          void?]{

Generates a sequence of slides by calling @racket[gen*] with, for each
of its arguments, numbers from @racket[0.0] to @racket[1.0]. If
@racket[gen*] accepts @math{n} arguments, then result is a sequence of
animations with a pause (i.e., not auto-advanced) between each of
@math{n} segments.

If @racket[gen*] accepts a single argument, then @racket[play-n] is
like @racket[play], except that @racket[gen*] is also called with
@racket[1.0] to generate a slide with no timeout. If @racket[gen*]
accepts multiple arguments, then slides are generated by calling
@racket[gen*] with the first argument varying from @racket[0.0] to
@racket[1.0] while all other arguments are @racket[0.0]. Then, the
first argument is held at @racket[1.0] while the second argument varies
from @racket[0.0] to @racket[1.0], and so on.

For example,

@racketblock[
(play-n 
 (lambda (n1 n2)
   (cellophane (t "Hello")
               (* n1 (- 1.0 n2)))))
]

generates an animation to fade in the word ``Hello,'' and then pauses
for a manual advance, and then fades ``Hello'' back out.

If @racket[skip-first?] is @racket[#t], then the very first slide of
the sequence is skipped. Similarly, if @racket[skip-last?] is
@racket[#t], then the last slide of the sequence is skipped.

The @racket[steps] argument controls how many steps happen in each
phase on the animation. If it is a number, then that number is used for
each phase. If it is a pair of two numbers, then the first number is used
for the first phase, and the second number is used for the rest of the
phases. Similarly, if it is @racket[(cons num_1 (cons num_2 num_3))],
@racket[num_1] and @racket[num_2] are used for the first two phases
and @racket[num_3] is used for the rest.

The elements of the @racket[comment] argument are used like the @racket[steps]
argument, except passed as comments instead of used as step counts.

The @racket[delay-secs], @racket[title],
@racket[name], @racket[aspect], and @racket[layout] arguments are passed on to
@racket[play] for each of the @math{n} segments of animation.

@history[#:changed "1.7" @elem{Added the @racket[aspect] argument.}]}


@defproc[(animate-slide [element (flat-rec-contract elem/c
                                   (or/c pict? 'next 'alts
                                        (listof (listof elem/c))))]
                        ...)
         (() (listof (real-in 0.0 1.0)) . ->* . pict?)]{

Accepts slide content similar to @racket[slide] with @racket['next]
and @racket['alts] and produces a procedure suitable for use with
@racket[play-n]. The result is similar to using @racket[slide], but
with fades for @racket['next] and @racket['alts] transitions (to
better fit the style, perhaps, of surrounding animations).}

@defparam[current-play-steps n exact-positive-integer?]{

A parameter that determines the default number of steps used for a
slide animation. The parameter's initial value is @racket[10].

@history[#:added "1.6"]}
