﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iotsitewise/IoTSiteWiseRequest.h>
#include <aws/iotsitewise/IoTSiteWise_EXPORTS.h>

#include <utility>

namespace Aws {
namespace IoTSiteWise {
namespace Model {

/**
 */
class DisassociateAssetsRequest : public IoTSiteWiseRequest {
 public:
  AWS_IOTSITEWISE_API DisassociateAssetsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DisassociateAssets"; }

  AWS_IOTSITEWISE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The ID of the parent asset from which to disassociate the child asset. This
   * can be either the actual ID in UUID format, or else <code>externalId:</code>
   * followed by the external ID, if it has one. For more information, see <a
   * href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/object-ids.html#external-id-references">Referencing
   * objects with external IDs</a> in the <i>IoT SiteWise User Guide</i>.</p>
   */
  inline const Aws::String& GetAssetId() const { return m_assetId; }
  inline bool AssetIdHasBeenSet() const { return m_assetIdHasBeenSet; }
  template <typename AssetIdT = Aws::String>
  void SetAssetId(AssetIdT&& value) {
    m_assetIdHasBeenSet = true;
    m_assetId = std::forward<AssetIdT>(value);
  }
  template <typename AssetIdT = Aws::String>
  DisassociateAssetsRequest& WithAssetId(AssetIdT&& value) {
    SetAssetId(std::forward<AssetIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of a hierarchy in the parent asset's model. (This can be either the
   * actual ID in UUID format, or else <code>externalId:</code> followed by the
   * external ID, if it has one. For more information, see <a
   * href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/object-ids.html#external-id-references">Referencing
   * objects with external IDs</a> in the <i>IoT SiteWise User Guide</i>.)
   * Hierarchies allow different groupings of assets to be formed that all come from
   * the same asset model. You can use the hierarchy ID to identify the correct asset
   * to disassociate. For more information, see <a
   * href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/asset-hierarchies.html">Asset
   * hierarchies</a> in the <i>IoT SiteWise User Guide</i>.</p>
   */
  inline const Aws::String& GetHierarchyId() const { return m_hierarchyId; }
  inline bool HierarchyIdHasBeenSet() const { return m_hierarchyIdHasBeenSet; }
  template <typename HierarchyIdT = Aws::String>
  void SetHierarchyId(HierarchyIdT&& value) {
    m_hierarchyIdHasBeenSet = true;
    m_hierarchyId = std::forward<HierarchyIdT>(value);
  }
  template <typename HierarchyIdT = Aws::String>
  DisassociateAssetsRequest& WithHierarchyId(HierarchyIdT&& value) {
    SetHierarchyId(std::forward<HierarchyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the child asset to disassociate. This can be either the actual ID
   * in UUID format, or else <code>externalId:</code> followed by the external ID, if
   * it has one. For more information, see <a
   * href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/object-ids.html#external-id-references">Referencing
   * objects with external IDs</a> in the <i>IoT SiteWise User Guide</i>.</p>
   */
  inline const Aws::String& GetChildAssetId() const { return m_childAssetId; }
  inline bool ChildAssetIdHasBeenSet() const { return m_childAssetIdHasBeenSet; }
  template <typename ChildAssetIdT = Aws::String>
  void SetChildAssetId(ChildAssetIdT&& value) {
    m_childAssetIdHasBeenSet = true;
    m_childAssetId = std::forward<ChildAssetIdT>(value);
  }
  template <typename ChildAssetIdT = Aws::String>
  DisassociateAssetsRequest& WithChildAssetId(ChildAssetIdT&& value) {
    SetChildAssetId(std::forward<ChildAssetIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique case-sensitive identifier that you can provide to ensure the
   * idempotency of the request. Don't reuse this client token if a new idempotent
   * request is required.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  DisassociateAssetsRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_assetId;

  Aws::String m_hierarchyId;

  Aws::String m_childAssetId;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_assetIdHasBeenSet = false;
  bool m_hierarchyIdHasBeenSet = false;
  bool m_childAssetIdHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace IoTSiteWise
}  // namespace Aws
