unit Streams;
{ Unit to provide enhancements to TV Objects unit streams in the form
  of several filters, i.e. stream clients, and other streams. }

{#Z+}  { These comments don't need to go into the help file. }

{$B-}   { Use fast boolean evaluation. }


{ Load some conditional defines }
{$i STDefine.inc}

{$ifdef overlays}
  {$O-}
  { Don't overlay this unit; it contains code that needs to participate
         in overlay management. }
{$endif}

{  Hierarchy:

   TStream                  (from Objects)
     TFilter                Base type for filters
       TEncryptFilter       Encrypts as it writes; decrypts as it reads
       TConcatFilter        Concatenates two streams
       TSequential          Filter that doesn't allow Seek
         TChksumFilter      Calculates 16 bit checksum for reads and writes
         TCRC16Filter       Calculates XMODEM-style 16 bit CRC
         TCRCARCFilter      Calculates ARC-style 16 bit CRC
         TCRC32Filter       Calculates ZIP/ZModem-style 32 bit CRC
     TNulStream             Eats writes, returns constant on reads
     TRAMStream             Stream in memory
     TXMSStream             Stream in XMS
     TDOSStream             (from Objects)
       TBufStream           (from Objects)
         TNamedBufStream    Buffered file stream that knows its name
           TTempBufStream   Buffered file stream that erases itself when done
     TWorkStream            Stream that grows as needed

   Procedures & functions:

   TempStream      allocates a temporary stream
   OvrInitStream   like OvrInitEMS, but buffers overlays on a stream
                   May be called several times to buffer different
                   segments on different streams.
   OvrDetachStream detaches stream from overlay system
   OvrDisposeStreams detaches all streams from overlay system and disposes of
                   them
   OvrSizeNeeded   Calculates the size needed to load the rest of the segments
                   to a stream
   OvrLoadAll      immediately copies as many overlay segments to the stream
                   as will fit
   UpdateChkSum    updates a 16 bit checksum value
   UpdateCRC16     updates a CRC16 value
   UpdateCRCARC    updates a CRCARC value
   UpdateCRC32     updates a CRC32 value
   ReverseBytes    reverses the byte order within a buffer

}
{#Z-}

interface

uses
  {$ifdef windows}
  wobjects,strings,windos,winprocs,wintypes;
  {$else}
  DOS,objects;
  {$endif}

const
  stBadMode = 1;                  { Bad mode for stream - operation not
                                    supported.  ErrorInfo = mode. }
  stStreamFail = 2;               { Stream init failed }
  stBaseError = 3;                { Error in base stream. ErrorInfo = base error value }
  stMemError = 4;                 { Not enough memory for operation }
  stSigError = 5;                 { Problem with LZ file signature }
  stUsedAll = 6;                  { Used limit of allocation }
  stUnsupported = 7;              { Operation unsupported in this stream }
  stBase2Error = 8;               { Error in second base.  ErrorInfo = base2 error value }
  stMisMatch = 9;                 { Two bases don't match.  ErrorInfo = mismatch position
                                    in current buffer. }
  stIntegrity = 10;               { Stream has detected an integrity error
                                    in a self check.  Info depends on
                                    stream type. }
type
  TOpenMode = $3C00..$3DFF;       { Allowable DOS stream open modes }
  {$ifdef windows}
  FNameStr = PChar;            { To make streams take names as in the manual. }
  {$endif}

  PFilter = ^TFilter;
  TFilter =
    object(TStream)
    { Generic object to filter another stream.  TFilter just passes everything
      through, and mirrors the status of the base stream }

      Base : PStream;
      { Pointer to the base stream. }

      Startofs : LongInt;
      { The offset of the start of the filter in the base stream. }

      OwnsBase : Boolean;
      { Defaults true; if set to false, then #Done# won't dispose of
        the base. }

      constructor Init(ABase : PStream);
        { Initialize the filter with the given base. }

      destructor Done; virtual;
        { Flush filter, then dispose of base if #OwnsBase#. }

      function GetPos : LongInt; virtual;
      function GetSize : LongInt; virtual;
      procedure Read(var Buf; Count : Word); virtual;
      procedure Seek(Pos : LongInt); virtual;
      procedure Truncate; virtual;
      procedure Write(var Buf; Count : Word); virtual;
      procedure Flush; virtual;

      function CheckStatus : Boolean; virtual;
    { Return true if status is stOK.
      If status is stOK, but base is not, then reset the base.  This is a poor
      substitute for a virtual Reset method. }

      procedure CheckBase;
        { Check base stream for error, and copy status using own Error method. }
    end;


type
  PConcatFilter = ^TConcatFilter;
  TConcatFilter =
    object(TFilter)
      { A filter which acts to concatenate two streams (or parts of streams)
        so that they appear as one.}
      Base2 : PStream;
        { Pointer to the second base.  This one logically follows the first.}

      Startofs2 : LongInt;
        { The offset of the start of the filter in the second base. }

      Position : Longint;
        { The current position of the filter.  The corresponding
          base stream is kept synchronized with this }
      Base1Size : Longint;
        { This is used a lot to determine switching. }

      constructor Init(ABase, ABase2 : PStream);        { Initialize the filter with the given bases. }

      destructor Done; virtual;
        { Flush filter, then dispose of both bases. }

      function GetPos:longint; virtual;
      function GetSize:longint; virtual;
      procedure Read(var Buf; Count : Word); virtual;
      procedure Seek(Pos : LongInt); virtual;
      procedure Truncate; virtual;
      procedure Write(var Buf; Count : Word); virtual;
      procedure Flush; virtual;
      { These methods work directly on Base until its size
        is reached, then switch over to Base2.  Base will *never* grow
        from the size at stream initialization. }

      function CheckStatus : Boolean; virtual;

      procedure CheckBase2;
        { Check 2nd base stream for error, and copy status using own Error method. }

    end;

type

  PNulStream = ^TNulStream;
  TNulStream =
    object(TStream)
      Position : LongInt;         { The current position for the stream. }
      Value : Byte;               { The value returned on reads. }

      constructor Init(AValue : Byte);
      function GetPos : LongInt; virtual;
      function GetSize : LongInt; virtual;
      procedure Read(var Buf; Count : Word); virtual;
      procedure Seek(Pos : LongInt); virtual;
      procedure Write(var Buf; Count : Word); virtual;
    end;

  Pbyte_array = ^Tbyte_array;
  Tbyte_array = array[0..65520] of Byte; { Type used as a buffer. }

  PRAMStream = ^TRAMStream;
  TRAMStream =
    object(TStream)
      Position : Word;            { The current position for the stream. }

      Size : Word;                { The current size of the stream. }
      Alloc : Word;               { The size of the allocated block of memory. }

      Buffer : Pbyte_array;       { Points to the stream data. }
      OwnMem : Boolean;           { Whether Done should dispose of data.}

      constructor Init(Asize : Word);
    { Attempt to initialize the stream to a block size of Asize;
       initial stream size and position are 0. }
      constructor UseBuf(ABuffer : Pointer; Asize : Word);
     { Initialize the stream using the specified buffer.  OwnMem is set
       to false, so the buffer won't be disposed of. Initial position is 0,
       size is Asize. }

      destructor Done; virtual;
        { Dispose of the stream. }

      function GetPos : LongInt; virtual;
      function GetSize : LongInt; virtual;
      procedure Read(var Buf; Count : Word); virtual;
      procedure Seek(Pos : LongInt); virtual;
      procedure Truncate; virtual;
      procedure Write(var Buf; Count : Word); virtual;
    end;

  PXMSStream = ^TXMSStream;
  TXMSStream =
    object(TStream)
      Handle : Word;              { XMS handle }
      BlocksUsed : Word;          { Number of 1K blocks used. Always allocates
                                    at least one byte more than Size. }
      Size : LongInt;             { The current size of the stream }
      Position : LongInt;         { Current position }

      constructor Init(MinSize,MaxSize:longint);
      destructor Done; virtual;

      function GetPos : LongInt; virtual;
      function GetSize : LongInt; virtual;
      procedure Read(var Buf; Count : Word); virtual;
      procedure Seek(Pos : LongInt); virtual;
      procedure Truncate; virtual;
      procedure Write(var Buf; Count : Word); virtual;

      procedure NewBlock;         { Internal method to allocate a block }
      procedure FreeBlock;        { Internal method to free one block }
    end;

function xms_MemAvail : Longint;
  { Returns total of available XMS bytes. }
function xms_MaxAvail : Longint;
  { Returns size of largest available XMS block in bytes. }
function ems_MemAvail : Longint;
  { Returns total of available EMS in bytes. }
function ems_MaxAvail : Longint;
  { Returns size of largest available EMS block in bytes. }


type
  PNamedBufStream = ^TNamedBufStream;
  TNamedBufStream =
    object(TBufStream)
      { A simple descendant of TBufStream which knows its own name.}

    {$ifdef windows}
    filename : PChar;
    {$else}
      Filename : PString;
    {$endif}
      { The name of the stream. }

      constructor Init(Name : FNameStr; Mode : TOpenMode; ABufSize : Word);
        { Open the file with the given name, and save the name. }

      destructor Done; virtual;
        { Close the file. }

    end;

{$IFDEF NONO}
  PTempBufStream = ^TTempBufStream;
  TTempBufStream =
    object(TNamedBufStream)
      { A temporary buffered file stream, which deletes itself when done.
        It's allocated on one of the directories specified by #TempEnvVar#.}

      constructor Init(ABufSize : Word;InitSize,MaxSize : Longint);
  { Create a temporary file with a unique name, in the directory
    pointed to by the environment varable named in #TempEnvVar# or in
    the current directory, open it in read/write mode, and try to grow
    it to InitSize bytes.   }

      destructor Done; virtual;
        { Close and delete the temporary file. }

    end;
{$ENDIF}

  TStreamType = (NoStream, RAMStream, EMSStream, XMSStream, FileStream);
  { The type of stream that a tempstream might be. }

const
  NumTypes = Ord(FileStream);

type
  TStreamRanking = array[1..NumTypes] of TStreamType;
  { A ranking of preference for a type of stream, from most to least preferred }

  TAllocator = function (InitSize, MaxSize : LongInt;
                       Preference : TStreamRanking) : PStream;
  { This is a declaration just like the Streams.TempStream function.}

  PWorkStream = ^TWorkStream;
  TWorkStream =
    object(TFilter)
     { This is a stream type that grows as you write to it by allocating new
       blocks according to a specified strategy.  Blocks may be of mixed
       types. It's a descendant of a filter, but it manages its own base. }

     Allocate : TAllocator;
     BlockMin,                     { These fields are passed to Allocate }
     BlockMax : longint;
     Preference : TStreamRanking;
     BlockStart: longint; { The offset in the stream where the
                            last block starts. }

     constructor init(Allocator:TAllocator;ABlockmin,ABlockMax:Longint;
                      APreference : TStreamRanking);
     { ABlockmin to APreference are passed to the allocator to allocate
       a new block whenever the current one gives a write error.
       The TWorkStream will never try to write a single block that crosses
       the ABlockMax boundary, so tests within the stream can be simple.}
     procedure write(var Buf; Count:Word); virtual;
     { The write procedure checks whether the write would make the
       current block grow too large; if so, it splits up the write. }
     procedure Truncate; virtual;
     { ED - required to deal with truncations within the first stream
       in the list - TConcatFilter messes it up otherwise. }
   end;

const
  BufSize : Word = 2048;          { Buffer size if buffered stream is used. }

const ForSpeed : TStreamRanking = (RAMStream, EMSStream, XMSStream, FileStream);
  { Streams ordered for speed }

const ForSize : TStreamRanking = (FileStream, EMSStream, XMSStream, RAMStream);
  { Streams ordered for low impact on the heap }

const ForSizeInMem : TStreamRanking = (EMSStream, XMSStream, RAMStream, NoStream);
  { Streams in memory only, ordered as #ForSize#. }




implementation

{$IFDEF BLA}
function num2str(l: longint): string;
var
  S: string;
begin
  Str(l,S); num2str:=S;
end;

procedure message(S: string);
var
  F: array[0..255] of char;
begin
  StrPCopy(F,S); messagebox(0,F,'',mb_ok);
end;
{$ENDIF}

  function MinLong(x,y:longint):longint;
  begin
    if x<y then
      MinLong := x
    else
      MinLong := y;
  end;

  function MaxLong(x,y:longint):longint;
  begin
    MaxLong := -MinLong(-x,-y);
  end;

  {****** TFilter code *******}

  constructor TFilter.Init(ABase : PStream);
  begin
    TStream.Init;
    Base := ABase;
    CheckBase;
    if Status = stOK then
      Startofs := Base^.GetPos;
    OwnsBase := true;
  end;

  destructor TFilter.Done;
  begin
    if Base <> nil then
    begin
      Flush;
      if OwnsBase then
        Dispose(Base, Done);
    end;
    TStream.Done;
  end;

  function TFilter.GetPos : LongInt;
  begin
    if CheckStatus then
    begin
      GetPos := Base^.GetPos-Startofs;
      CheckBase;
    end else GetPos:=-1;
  end;

  function TFilter.GetSize : LongInt;
  begin
    if CheckStatus then
    begin
      GetSize := Base^.GetSize-Startofs;
      CheckBase;
    end else GetSize:=-1;
  end;

  procedure TFilter.Read(var Buf; Count : Word);
  begin
    if CheckStatus then
    begin
      Base^.Read(Buf, Count);
      CheckBase;
    end;
  end;

  procedure TFilter.Seek(Pos : LongInt);
  begin
    if CheckStatus then
    begin
      Base^.Seek(Pos+Startofs);
      CheckBase;
    end;
  end;

  procedure TFilter.Truncate;
  begin
    if CheckStatus then
    begin
      Base^.Truncate;
      CheckBase;
    end;
  end;

  procedure TFilter.Write(var Buf; Count : Word);
  begin
    if CheckStatus then
    begin
      Base^.Write(Buf, Count);
      CheckBase;
    end;
  end;

  procedure TFilter.Flush;
  begin
    if CheckStatus then
    begin
      Base^.Flush;
      CheckBase;
    end;
  end;

  function TFilter.CheckStatus : Boolean;
  begin
    if (Status = stOK) then
      Base^.Reset;
    CheckStatus := Status = stOK;
  end;

  procedure TFilter.CheckBase;
  begin
    if Base^.Status <> stOK then
      Error(stBaseError, Base^.Status);
  end;



  { ****** Concatenating Filter code ****** }

  constructor TConcatFilter.Init(ABase, ABase2 : PStream);
  { Initialize the filter with the given bases. }
  begin
    if not TFilter.Init(ABase) then
      fail;
    Base2 := ABase2;
    CheckBase2;
    Base1Size := TFilter.GetSize;
    if Status = stOK then
      StartOfs2 := Base2^.GetPos;
    Position := Base1Size;
  end;

  destructor TConcatFilter.done;
  begin
    if Base2 <> nil then
    begin
      Base2^.Flush;
      Dispose(Base2,done);
    end;
    if Base <> nil then
    begin
      Base^.Flush;
      Dispose(Base,Done);   { Can't call TFilter.Done!!!! }
    end;
    TStream.done;
  end;

  function TConcatFilter.GetPos:longint;
  begin
    GetPos := Position;
  end;

  function TConcatFilter.GetSize:longint;
  begin
    if CheckStatus then
    begin
      GetSize := Base1Size + Base2^.GetSize;
      CheckBase2;
    end;
  end;

  procedure TConcatFilter.Read(var Buf; Count : Word);
  var
    Buffer : TByte_array absolute Buf;
    base1part : word;
  begin
    { First read the Base 1 portion }
    if Position < Base1Size then
    begin
      base1part := Count;
      if Position+base1part > Base1Size then
        base1part := Base1Size - Position;
      TFilter.Read(Buf, base1part);
      dec(Count,base1part);
      inc(Position,Base1part);
      if Count > 0 then
        Base2^.Seek(StartOfs2);   { Be sure Base2 agrees with Pos now }
    end
    else
      base1part := 0;
    { Now read the Base 2 portion }
    if (Count > 0) and (status = stOK) then
    begin
      if Position = Base1Size then
        Base2^.Seek(StartOfs2);
      Base2^.Read(Buffer[base1part],Count);
      CheckBase2;
      inc(Position,count);
    end;
  end;

  procedure TConcatFilter.Seek(Pos : LongInt);
  begin
    if Pos < Base1Size then
      TFilter.Seek(Pos)
    else
    begin
      if CheckStatus then
      begin
        Base2^.Seek(Pos-Base1Size+StartOfs2);
        CheckBase2;
      end;
    end;
    if Status = stOK then
      Position := Pos;
  end;

  procedure TConcatFilter.Truncate;
  begin
    if Position < Base1Size then
    begin
{      messagebeep(0);}
      Error(stUnsupported,0);      { We don't allow Base to be truncated, only
                                   Base2 }
    end else
      if CheckStatus then
      begin
        Base2^.Truncate;
        CheckBase2;
      end;
  end;

  procedure TConcatFilter.Write(var Buf; Count : Word);
  var
    Buffer : TByte_array absolute Buf;
    base1part : word;
  begin
    { First write the Base 1 portion }
    if Position < Base1Size then
    begin
      base1part := Count;
      if Position+base1part > Base1Size then
        base1part := Base1Size - Position;
      TFilter.Write(Buf, base1part);
      dec(Count,base1part);
      inc(Position,Base1part);
      if Count > 0 then
        Base2^.Seek(StartOfs2);   { Be sure Base2 agrees with Pos now }
    end
    else
      base1part := 0;
    { Now write the Base 2 portion }
    if (Count > 0) and (status = stOK) then
    begin
      Base2^.Write(Buffer[base1part],Count);
      CheckBase2;
      inc(Position,count);
    end;
  end;

  procedure TConcatFilter.Flush;
  begin
    TFilter.Flush;
    if (status = stOK) and (Base2<>Nil) then
    begin
      Base2^.Flush;
      CheckBase2;
    end;
  end;

  function TConcatFilter.CheckStatus : Boolean;
  begin
    if TFilter.CheckStatus then
      if Base2^.Status <> stOK then
        Base2^.Reset;
    CheckStatus := Status = stOK;
  end;

  procedure TConcatFilter.CheckBase2;
  begin
    if Base2^.status <> stOk then
      Error(stBase2Error,Base2^.status);
  end;


  { ****** Null stream code ****** }

  constructor TNulStream.Init;
  begin
    TStream.Init;
    Position := 0;
    Value := AValue;
  end;

  function TNulStream.GetPos;
  begin
    GetPos := Position;
  end;

  function TNulStream.GetSize;
  begin
    GetSize := Position;
  end;

  procedure TNulStream.Read;
  begin
    FillChar(Buf, Count, Value);
    Inc(Position, Count);
  end;

  procedure TNulStream.Seek;
  begin
    Position := Pos;
  end;

  procedure TNulStream.Write;
  begin
    Inc(Position, Count);
  end;

  { ****** RAM stream code ****** }

  constructor TRAMStream.Init(Asize : Word);
  begin
    TStream.Init;
    Position := 0;
    Size := 0;
    Alloc := Asize;
    if MaxAvail < Alloc then
      Fail;
    GetMem(Buffer, Alloc);
    if Buffer = nil then  { !1.6 }
      Fail;
    OwnMem := True;
    FillChar(Buffer^, Alloc, 0);
  end;

  constructor TRAMStream.UseBuf(ABuffer : Pointer; Asize : Word);
  begin
    TRAMStream.Init(0);
    Alloc := Asize;
    Size  := Asize;
    Buffer := ABuffer;
    OwnMem := False;
  end;

  destructor TRAMStream.Done;
  begin
    if OwnMem then
      FreeMem(Buffer, Alloc);
    TStream.Done;
  end;

  function TRAMStream.GetPos: Longint;
  {     Replaced with assembler for speed.  }
  {
  begin                      
    GetPos := Position;
   end;
  }
  assembler;
  asm
    les di,self
    mov ax,es:di[Position];
    xor dx,dx
  end;

  function TRAMStream.GetSize: Longint;
  {               Replaced with assembler for speed.   }
  {
  begin               
    GetSize := Size;
   end;
  }
   assembler;
   asm
     les di,self
     mov ax,es:di[size]
     xor dx,dx
   end;

  function CheckInc(var pos:word;count,limit:word):boolean; assembler;
  { Increments pos by count, returns false if limit is exceeded }
  asm
    les di,pos
    mov bx,count
    mov al,true
    add bx,es:[di]
    jc  @1            { Carry means error }
    mov es:[di],bx
    sub bx,limit
    jbe @2
  @1:
    dec ax            { Set AX to false }
  @2:
  end;

  procedure TRAMStream.Read(var Buf; Count : Word);
  begin
    Move(Buffer^[Position], Buf, Count);
    if not CheckInc(Position,Count,Size) then
    begin
      Error(stReadError,0);
      Dec(Position,Count);
      FillChar(Buf,Count,0);
    end;
  end;

  procedure TRAMStream.Seek(Pos : LongInt);
  begin
    if Pos > Size then
      Error(stReaderror, 0)
    else
      Position := Pos;
  end;

  procedure TRAMStream.Truncate;
  begin
    Size := Position;
  end;

  procedure TRAMStream.Write(var Buf; Count : Word);
  begin
    if not CheckInc(Position,Count,Alloc) then
      Error(stWriteError, 0)
    else
    begin
      Move(Buf, Buffer^[Position-Count], Count);
      if Position > Size then
        Size := Position;
    end;
  end;

  { ***** XMS stream code ***** }

  {$I xmsstrm.inc}

  { ***** EMS size code ***** }

  function exist_ems:boolean;
  const
    ems_found : boolean = false;  { Used as initialized var }
  var
    S : TEMSStream;
  begin
    if not ems_found then
    begin
      S.init(1,1);
      ems_found := S.status = stOk;
      S.done;
      EMSCurHandle:=$FFFF; EMSCurPage:=$FFFF;
    end;
    exist_ems := ems_found;
  end;

  function ems_maxavail: longint;
  begin
    if not exist_ems then
      ems_maxavail:=0
    else
    asm
      mov ah,$42;
      int $67
      mov ax,16384
      mul bx
      mov word ptr @result,ax
      mov word ptr @result[2],dx
    end;
  end;

  function ems_memavail: longint;
  begin
    ems_memavail := ems_maxavail;
  end;

  { ***** Named Buffered file stream code ***** }

  constructor TNamedBufStream.Init(Name : FNameStr; Mode : TOpenMode; ABufSize : Word);
  begin
    if TBufStream.Init(Name, Mode, ABufSize) then
    {$ifdef windows}
    filename := StrNew(name)
    {$else}
      Filename := NewStr(Name)
    {$endif}
    else
      Fail;
  end;

  destructor TNamedBufStream.Done;
  begin
  {$ifdef windows}
  StrDispose(filename);
  {$else}
    DisposeStr(Filename);
  {$endif}
    TBufStream.Done;
  end;

  {******** TWorkStream code ******* }

  constructor TWorkStream.init(Allocator:TAllocator;ABlockmin,ABlockMax:Longint;
                   APreference : TStreamRanking);
  begin
    TFilter.init(Allocator(ABlockmin,ABlockmax,APreference));
    Allocate := Allocator;
    Blockmin := ABlockmin;
    Blockmax := ABlockmax;
    Preference := APreference;
    BlockStart := 0;
  end;

  procedure TWorkStream.write(var Buf; Count:Word);
  var
    Buffer : TByte_array absolute Buf;
    firstpart : word;
    byteswritten : word;
    pos : longint;
    NewBase : PStream;
    saveStatus, saveInfo : integer;
  begin
    pos := GetPos;
    byteswritten := 0;
    if CheckStatus then
      repeat
        firstpart := Count;
        if (Pos < BlockStart+BlockMax) and (Pos+firstpart > BlockStart+BlockMax) then
          firstpart := BlockStart+BlockMax-Pos;
        TFilter.Write(Buffer[byteswritten], firstpart);

        { **** crummy code to get around problems with TBufStream **** }
        { The test is an efficiency hack - we don't want to flush every
          segment of the stream, just the last one. }
        if typeof(Base^) = typeof(TConcatFilter) then
          PConcatFilter(Base)^.Base2^.Flush
        else
          Base^.Flush;          { Must flush all writes to see TBufStream
                                errors immediately :-( }
        CheckBase;              { 1.6 fix }                        
        { **** end of crummy code :-) ***** }
        if Status = stOK then
        begin
          dec(Count,firstpart);
          inc(Pos,firstpart);
          inc(byteswritten,firstpart);
        end
        else
        begin
          saveStatus := Status;
          saveInfo   := ErrorInfo;
          Reset;
          if Pos = GetSize then
          begin
            { If write failed at eof, allocate a new block }
            Seek(0);
            NewBase := Allocate(BlockMin,BlockMax,Preference);
            if (NewBase = nil) or (NewBase^.Status <> stOK) then
            begin
              error(stBaseError, stWriteError);
              exit;
            end;
            Base := New(PConcatFilter,init(Base,NewBase));
            BlockStart := Pos;
          end
          else  { Some other kind of write failure; restore the error status }
          begin
            error(saveStatus,saveInfo);
            exit;
          end;
        end;
      until count = 0;
  end;

  procedure TWorkStream.Truncate;
  var
    Posit: longint;
    P: PStream;
  begin
    Posit:=GetPos;
    P:=Base;
    while BlockStart>Posit do
    begin
      Base:=PConcatFilter(P)^.Base;
      PConcatFilter(P)^.Base:=Nil; Dispose(P,Done);
      P:=Base;
      if typeof(P^)=typeof(TConcatFilter) then
        BlockStart:=BlockStart-PConcatFilter(P)^.base2^.GetSize
      else BlockStart:=0;
    end;
    TFilter.truncate;
  end;

end.
