/* miktex.c - MiKTeX support routines
   Time-stamp: "97/08/21 20:36:39 mik"

   Copyright (C) 1991, 92, 93, 94, 95, 96, 97
	Christian Schenk  <cschenk@berlin.snafu.de>

   This file is part of MiKTeX.

   MiKTeX is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
   
   MiKTeX is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with MiKTeX; if not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#if defined (_WIN32)
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <signal.h>
#include <process.h>
#include <fcntl.h>
#include <sys/stat.h>

#include "c4plib.h"
#include "miktex.h"

#if defined (_WIN32)
#if defined (IN_MIKTEX_DLL)
#define _MIKTEX_1 __declspec (dllexport)
#endif /* IN_MIKTEX_DLL */
#define _MIKTEX_2 __stdcall
#endif /* _WIN32 */

#if ! defined (_MIKTEX_1)
#define _MIKTEX_1
#endif /* ! _MIKTEX_1 */

#if ! defined (_MIKTEX_2)
#define _MIKTEX_2
#endif /* ! _MIKTEX_2 */

#include "fnamedb.h"
#include "miktex.rc"

#define PATH_SEP_CHAR ';'
#define PATH_SEP_CHAR_AS_STRING ";"
#define RECURSION_INDICATOR "//"

#define isslash(c) ((c) == '/' || (c) == '\\')
#define ispath(filename) (strchr (filename, '/') || strchr (filename, '\\'))

/* Duplicate a string using the stack. */
#define STRDUP(s) strcpy (_alloca (strlen (s) + 1), (s))

static clock_t start_clock;

#define DEBUG_FILE_SEARCH	1
#define DEBUG_FNDB		2

int debug_flags = 0;

static int
lastch (const char *s)

{
  size_t len;

  xassert (s != 0);

  len= strlen (s);
  return (len == 0 ? 0 : s[len - 1]);
}

/* _________________________________________________________________________

   PRINTING ERROR MESSAGES.
   _________________________________________________________________________ */


static void
fixme (const char *	filename,
       int		lineno,
       const char *	fmt,
                        ...)

{
  va_list marker;
  fprintf (stderr, "%s(%d): fixme: ", filename, lineno);
  va_start (marker, fmt);
  vfprintf (stderr, fmt, marker);
  va_end (marker);
  fputc ('\n', stderr);
}

/* Print information about the last system error. */
static void
print_system_error_message ()

{
  LPVOID lpMsgBuf;
  FormatMessage (FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
		 NULL,
		 GetLastError (),
		 MAKELANGID (LANG_NEUTRAL, SUBLANG_DEFAULT),
		 (LPTSTR) &lpMsgBuf,
		 0,
		 NULL);
  fprintf (stderr, "\n%s\n", lpMsgBuf);
  LocalFree (lpMsgBuf);
}

/* Print an error message and exit the program. */
_MIKTEX_1 void _MIKTEX_2
fatal_error (const char *fmt,
	     ...)

{
  va_list marker;
  va_start (marker, fmt);
  fprintf (stderr, "%s: ", c4pargv[0]);
  vfprintf (stderr, fmt, marker);
  va_end (marker);
  fputc ('\n', stderr);
  exit (EXIT_FAILURE);
}

/* _________________________________________________________________________

   MAINTAINING MULTIPLE TEXMF ROOT DIRECTORIES.
   _________________________________________________________________________ */


static struct

{
  char *	root;
  int		fndb_handle;
  int		no_fndb;
} texmf_roots[MAX_TEXMF_ROOT_DIRECTORIES];

size_t number_of_texmf_roots;

static void
add_texmf_root (const char *root_directory)

{
  if (debug_flags & DEBUG_FNDB)
    fprintf (stderr, "Registering TEXMF root %s...\n", root_directory);
  texmf_roots[number_of_texmf_roots].root = strdup (root_directory);
  texmf_roots[number_of_texmf_roots].fndb_handle = -1;
  texmf_roots[number_of_texmf_roots].no_fndb = 0;
  number_of_texmf_roots++;
}

static void
initialize_texmf_roots ()

{
  char *texmf_paths;
  char *cp;
  size_t buf_size;
  char *root;

  /* Get TeXMF root directories. */
  buf_size = (_MAX_PATH + 1) * MAX_TEXMF_ROOT_DIRECTORIES;
  texmf_paths = (char *) malloc (buf_size);
  get_cfg_value ("MiKTeX", "TeXMF Root Directories",
		 texmf_paths, buf_size, "c:\\texmf");
  
  root = strtok (texmf_paths, ";");
  number_of_texmf_roots = 0;
  while (root && number_of_texmf_roots < MAX_TEXMF_ROOT_DIRECTORIES)
    {
      add_texmf_root (root);
      root = strtok (0, ";");
    }

  free (texmf_paths);
}

_MIKTEX_1 size_t _MIKTEX_2
get_number_of_texmf_roots ()

{
  if (number_of_texmf_roots == 0)
    initialize_texmf_roots ();
  return (number_of_texmf_roots);
}

_MIKTEX_1 const char * _MIKTEX_2
get_root_directory (size_t i)

{
  if (get_number_of_texmf_roots () == 0)
    initialize_texmf_roots ();
  
  if (get_number_of_texmf_roots () == 0)
    return (0);

  xassert (i < get_number_of_texmf_roots ());

  return (texmf_roots[i].root);
}

static int
find_texmf_root (char *path,
		 char **root_relative_filename,
		 size_t *handle)

{
  int found;
  
  xassert (root_relative_filename != 0);

  if (! is_absolute_path (path))
    return (0);

  found = 0;

  for (*handle = 0;
       *handle < get_number_of_texmf_roots () && ! found;
       (*handle)++)
    {
      const char *texmf_root = get_root_directory (*handle);
      size_t rootlen = strlen (texmf_root);
      if (_strnicmp (path, texmf_root, rootlen) == 0)
	{
	  found = 1;
	  *root_relative_filename = path + rootlen + 1;
	}
    }

  return (found);
}

static int
get_fndb_handle (unsigned root_idx)

{
  char full_fndb_filename[_MAX_PATH];
  char fndb_filename[_MAX_FNAME];
  char buf[50];
  int fndb_exists;
  char *cp;

  if (get_number_of_texmf_roots () == 0)
    initialize_texmf_roots ();

  if (get_number_of_texmf_roots () == 0)
    return (-1);

  xassert (root_idx < get_number_of_texmf_roots ());

  if (texmf_roots[root_idx].fndb_handle >= 0)
    return (texmf_roots[root_idx].fndb_handle);

  if (texmf_roots[root_idx].no_fndb)
    return (-1);

  strcpy (fndb_filename, "miktex\\config\\texmf.fndb");
  _makepath (full_fndb_filename, 0, texmf_roots[root_idx].root,
	     fndb_filename, 0);
  fndb_exists = (access_regular_file (full_fndb_filename, 0) == 0);
  if (! fndb_exists && root_idx > 0)
    {
      sprintf (fndb_filename, "miktex\\config\\texmf%u.fndb", root_idx);
      _makepath (full_fndb_filename, 0, texmf_roots[0].root, fndb_filename, 0);
      fndb_exists = (access_regular_file (full_fndb_filename, 0) == 0);
    }
  if (! fndb_exists)
    {
      fprintf (stderr, "note: there is no fndb for %s\n",
	       texmf_roots[root_idx].root);
      texmf_roots[root_idx].no_fndb = 1;
      return (-1);
    }
  else
    {
      int fndb_handle;
      if (debug_flags & DEBUG_FNDB)
	printf ("Loading %s...", full_fndb_filename);
      fndb_handle = fndb_new (full_fndb_filename, texmf_roots[root_idx].root);
      if (fndb_handle >= 0)
	{
	  texmf_roots[root_idx].fndb_handle = fndb_handle;
	  if (debug_flags & DEBUG_FNDB)
	    printf ("%u\n", fndb_handle);
	  return (fndb_handle);
	}
      else
	{
	  texmf_roots[root_idx].no_fndb = 1;
	  if (debug_flags & DEBUG_FNDB)
	    printf ("error!\n");
	  return (-1);
	}
    }
}

/* _________________________________________________________________________

   FILE OPERATIONS.
   _________________________________________________________________________ */


/* Open a file for reading / writing. Assume that the file will be accessed
   sequentially. */
_MIKTEX_1 FILE * _MIKTEX_2
open_file (const char *file_name,
	   const char *mode)

{
  int fd;
  FILE *fp;
  int flags = (strchr (mode, 'r')
	       ? _O_RDONLY | _O_SEQUENTIAL
	       : _O_WRONLY | _O_CREAT | _O_TRUNC);
  if (strchr (mode, 'b'))
    flags |= _O_BINARY;
  else
    flags |= _O_TEXT;
  fd = _open (file_name, flags, _S_IREAD | _S_IWRITE);
  if (fd >= 0)
    fp = _fdopen (fd, mode);
  else
    fp = 0;
  if (fp)
    setvbuf (fp, 0, _IOFBF, 1024 * 4);
  return (fp);
}

typedef struct
{
  FILE *_c4p_fp;
  unsigned char _c4p_buf;
} bytefile;

/* Create a directory path ( borrowed from GNU File Utilities ). */
_MIKTEX_1 boolean _MIKTEX_2
make_path_ex (const char *	path,
	      boolean		exit_on_error)

{
#define S_ISDIR(m) (((m) & _S_IFMT) == _S_IFDIR)
  char *dirpath;
  struct _stat stats;
  int retval = 0;
  dirpath = (char *) _alloca (strlen (path) + 1);
  strcpy (dirpath, path);
  if (_stat (dirpath, &stats) != 0)
    {
      char *slash;
      char *cp;
      struct ptr_list
      {
	char *dirname_end;
	struct ptr_list *next;
      };
      struct ptr_list *p, *leading_dirs = NULL;
      slash = dirpath;
      if (slash[0] && slash[1] == ':')
	slash += 2;
      else if (isslash (slash[0]) && slash[1] == slash[0])
	{
	  /* UNC file name. */
	  slash += 2;
	  while (*slash && ! isslash (*slash))
	    slash++;
	  if (isslash (*slash))
	    {
	      slash++;
	      while (*slash && ! isslash (*slash))
		slash++;
	    }
	}
      while (isslash (*slash))
	slash++;
      while ((cp = strchr (slash, '/')) || (cp = strchr (slash, '\\')))
	{
	  slash = cp;
	  *slash = '\0';
	  if (_stat (dirpath, &stats) != 0)
	    {
	      if (_mkdir (dirpath) != 0)
		{
		  if (exit_on_error)
		    fatal_error ("cannot make directory '%s'", dirpath);
		  return (false);
		}
	    }
	  else if (! S_ISDIR (stats.st_mode))
	    {
	      if (exit_on_error)
		fatal_error ("'%s' exists but is not a directory", dirpath);
	      return (true);
	    }
	  *slash++ = '\\';
	  while (*slash == '/' || *slash == '\\')
	    slash++;
	}
      if (_mkdir (dirpath) != 0)
	{
	  if (exit_on_error)
	    fatal_error ("cannot make directory '%s'", dirpath);
	  return (false);
	}
    }
  else
    {
      if (! S_ISDIR (stats.st_mode))
	{
	  if (exit_on_error)
	    fatal_error ("'%s' exists but is not a directory", dirpath);
	  return (false);
	}
    }
  return (true);
}

_MIKTEX_1 boolean _MIKTEX_2
make_path (const char *path)

{
  return (make_path_ex (path, 1));
}

/* _________________________________________________________________________

   MANAGING WORKING DIRECTORIES.
   _________________________________________________________________________ */


static char	current_working_directory[_MAX_PATH];
static int	current_drive;
static char *	temp_dir;
static int	temp_drive;
static int	print_only_p;
static int	in_temporary_directory;

static void
remember_current_working_directory (char *current_working_directory)

{
  char envstring[_MAX_PATH];
  strcpy (envstring, "_MIKTEX_CWD=");
  strcat (envstring, current_working_directory);
  putenv (envstring);
}

_MIKTEX_1 char * _MIKTEX_2
get_current_working_directory (char *	buf,
			       size_t	bufsize)

{
  const char *envstring = getenv ("_MIKTEX_CWD");
  xassert (bufsize > strlen ("\\\\A\\B\\"));
  strcpy (buf, "c:\\");
  if (envstring == 0)
    {
      if (_getcwd (buf, bufsize) == 0)
	perror ("_getcwd");
    }
  else if (strlen (envstring) > bufsize)
    fixme (__FILE__, __LINE__, "buffer too small");
  else
    strcpy (buf, envstring);
  return (buf);
}

_MIKTEX_1 int _MIKTEX_2
enter_temporary_directory (const char *	prefix,
			   int		arg_print_only_p)

{
  char system_temp_dir[_MAX_PATH];

  xassert (! in_temporary_directory);

  print_only_p = arg_print_only_p;

  /* Get current working directory. */
  if (_getcwd (current_working_directory,
	       sizeof (current_working_directory)) == 0)
    {
      perror ("_getcwd");
      return (-1);
    }

  /* Get current drive. */
  current_drive = _getdrive ();

  /* Determine official temporary directory; remember its drive letter. */
  GetTempPath (sizeof (system_temp_dir), system_temp_dir);
  if (tolower (system_temp_dir[0]) >= 'a'
      && tolower (system_temp_dir[0]) <= 'z'
      && system_temp_dir[1] == ':')
    temp_drive = tolower (system_temp_dir[0]) - 'a' + 1;
  else
    temp_drive = 0;

  /* Create sub-directory in official temporary directory. */
  temp_dir = _tempnam (system_temp_dir, prefix);
  if (temp_dir == 0)
    {
      perror ("_tempnam");
      return (-1);
    }
  if (print_only_p)
    printf ("mkdir %s\n", temp_dir);
  else if (_mkdir (temp_dir) != 0)
    {
      perror (temp_dir);
      return (-1);
    }

  /* Remember that we have created the temporary directory. */
  in_temporary_directory = 1;
  remember_current_working_directory (current_working_directory);

  /* Change drive. */
  if (temp_drive != 0 && _chdrive (temp_drive) != 0)
    {
      perror (temp_dir);
      leave_temporary_directory ();
      return (-1);
    }

  /* Change working directory. */
  if (print_only_p)
    printf ("cd /d %s\n", temp_dir);
  else if (_chdir (temp_dir) != 0)
    {
      perror (temp_dir);
      leave_temporary_directory ();
      return (-1);
    }

  return (0);
}

_MIKTEX_1 void _MIKTEX_2
leave_temporary_directory ()

{
  xassert (in_temporary_directory);

  if (print_only_p)
    printf ("cd /d %s\n", current_working_directory);
  else
    {
      WIN32_FIND_DATA fdata;
      HANDLE hnd = FindFirstFile ("*", &fdata);
      if (hnd != INVALID_HANDLE_VALUE)
	{
	  int proceed = 1;
	  while (proceed)
	    {
	      if (! (fdata.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY))
		{
		  if (remove (fdata.cFileName) != 0)
		    perror (fdata.cFileName);
		}
	      proceed = FindNextFile (hnd, &fdata);
	    }
	  FindClose (hnd);
	}
      if (current_drive != 0 && _chdrive (current_drive) != 0)
	perror (current_working_directory);
      if (_chdir (current_working_directory) != 0)
	perror (current_working_directory);
      if (_rmdir (temp_dir) != 0)
	perror (temp_dir);
    }

  in_temporary_directory = 0;
}

/* _________________________________________________________________________

   SEARCHING FOR FILES.
   _________________________________________________________________________ */


/* Test if PATH is absolute. */
static int
is_absolute_path (const char *	path)

{
  if (isslash (path[0]))
    return (1);
  else if (isalpha (path[0])
	   && path[1] == ':'
	   && isslash (path[2]))
    return (1);
  else
    return (0);
}

static void
make_absolute_filename (char *		buf,
			size_t		bufsize,
			const char *	filename)
			

{
  char part[_MAX_PATH];
  get_current_working_directory (part, sizeof (part) - strlen (filename) - 1);
  if (! isslash (part[strlen (part) - 1]))
    strcat (part, "\\");
  strcat (part, filename);
  if (_fullpath (buf, part, bufsize) == 0)
    {
      fixme (__FILE__, __LINE__, "_fullpath failed");
      strcpy (buf, "c:\\fixme");
    }
}

static int
access_file (const char *	filename,
	     int		mode)

{
  struct _stat file_statistics;
  int rc = _stat (filename, &file_statistics);
  if (rc == 0)
    {
      if (file_statistics.st_mode & mode)
	return (0);
      else
	return (-1);
    }
  else
    return (rc);
}

_MIKTEX_1 int _MIKTEX_2
access_regular_file (const char *	filename,
		     int		mode)

{
  return (access_file (filename, mode | _S_IFREG));
}

static int
can_create_file_in_directory (const char *dirname)

{
  return (1);			/* fixme */
}

_MIKTEX_1 int _MIKTEX_2
access_directory (const char *	filename,
		  int		mode)

{
  char myfilename[_MAX_PATH];
  DWORD attr;
  strcpy (myfilename, filename);
  if (! isslash (lastch (myfilename)))
    strcat (myfilename, "\\");
  attr = GetFileAttributes (myfilename);
  return (attr == 0xffffffff
	  ? -1
	  : (! (attr & FILE_ATTRIBUTE_DIRECTORY)
	     ? -1
	     : (mode == 02 || mode == 06
		? (can_create_file_in_directory (filename) ? 0 : -1)
		: 0)));
}

static boolean
slow_find_file_in_domain (const char *, const char *, char *);

static boolean
fast_find_file_in_domain (const char *, const char *, char *);

static void
fndb_out_of_date ()

{
  static int fndb_warning_issued = 0;
  if (! fndb_warning_issued)
    {
      fprintf (stderr,
	       "\nYou should run 'configure -u' to refresh the fndb!\n");
      fndb_warning_issued = 1;
    }
}

/* Find a file. This is the main transaction center of the file finder.

   FILENAME is the name of the wanted file. PATH_LIST is a
   semicolon-seperated ordered (in the sense of priority) list of
   search specifications. If a file is found, then place its name in
   RESULT. */
_MIKTEX_1 boolean _MIKTEX_2
find_file (const char *	filename,
	   const char *	path_list,
	   char *	result)
     
{
  boolean found;

  /* If a path name was given, then don't look out any further. */
  if (ispath (filename))
    {
      if (is_absolute_path (filename))
	{
	  if (access_regular_file (filename, 0) == 0)
	    {
	      strcpy (result, filename);
	      return (1);
	    }
	  else
	    return (0);
	}
      else
	{
	  /* Make filename absolute. */
 	  char absolute_filename[_MAX_PATH];
	  make_absolute_filename (absolute_filename,
				  sizeof (absolute_filename),
				  filename);
	  if (access_regular_file (absolute_filename, 0) == 0)
	    {
	      strcpy (result, absolute_filename);
	      return (1);
	    }
	  else
	    return (0);
	}
    }
  else
    {
      /* If `.' is the first entry in the path list, then try to open
	 the file. */
      if (path_list && path_list[0] == '.'
	  && (path_list[1] == PATH_SEP_CHAR || path_list[1] == 0))
	{
	  char absolute_filename[_MAX_PATH];
	  make_absolute_filename (absolute_filename,
				  sizeof (absolute_filename),
				  filename);
	  if (access_regular_file (absolute_filename, 0) == 0)
	    {
	      strcpy (result, absolute_filename);
	      return (1);
	    }
	}
      
      /* Search the filename data base. */
      found = fast_find_file_in_domain (filename, path_list, result);
      if (found && access_regular_file (result, 0) != 0)
	{
	  if (debug_flags & DEBUG_FNDB)
	    fprintf (stderr,
		     "Found %s in fndb, but the file does not exist!\n",
		     result);
	  fndb_out_of_date ();
	  found = 0;
	}
      
      /* If not found in database, then search the disk. */
      if (! found)
	{
	  found = slow_find_file_in_domain (filename, path_list, result);
	  if (found)
	    {
	      if (debug_flags & DEBUG_FILE_SEARCH)
		fprintf (stderr, "Found %s\n", result);
	    }
	}
      
      return (found);
    }
}

static void
normalize_search_spec (char *s)

{
  size_t i;
  size_t len;

  len = strlen (s);
  for (i = 0; i < len; i++)
    {
      if (s[i] == '/')
	{
	  if (i + 1 < len && s[i + 1] == '/' && i != 0)
	    i++;		/* Preserve RECURSION_INDICATOR */
	  else
	    s[i] = '\\';
	}
    }
}

static int
rsearch_disk (const char *, const char *, const char *, char *);

static int
slow_find_file (const char *, char *, char *);

/* Search for a file in a specified domain using the file system API. */
static boolean
slow_find_file_in_domain (const char *	filename,
			  const char *	path_list,
			  char *	result)

{
  char *my_path_list;
  char *path;
  char *next_path;
  int found;

  if (debug_flags & DEBUG_FILE_SEARCH)
    fprintf (stderr, "Searching %s in domain %s...\n", filename, path_list);

  /* Make a working copy of the path list argument and normalize it. */
  my_path_list = strdup (path_list);
  normalize_search_spec (my_path_list);

  /* Initialize path list search. */
  path = my_path_list;
  next_path = strchr (path, PATH_SEP_CHAR);
  if (next_path)
    *next_path++ = 0;
  found = 0;

  /* For each entry in the path list... */
  while (path && ! found)
    {
      if (path[0] == '%' && path[1] == 'R' && isslash (path[2]))
	{
	  char alt_path[_MAX_PATH];
	  size_t i;
	  
	  for (i = 0; i < get_number_of_texmf_roots () && ! found; i++)
	    {
	      xassert (get_root_directory (i) != 0);
	      if (i > 0)	/* fixme */
		{
		  continue;
		}
	      _makepath (alt_path, 0, get_root_directory (i), &path[3], 0);
	      found = slow_find_file (filename, alt_path, result);
	    }
	}
      else
	found = slow_find_file (filename, path, result);

      /* Get the next entry of the path list. */
      if (next_path && *next_path)
	{
	  path = next_path;
	  next_path = strchr (next_path, PATH_SEP_CHAR);
	  if (next_path)
	    *next_path++ = 0;
	}
      else
	path = 0;
    }

  free (my_path_list);
      
  return (found);
}

static int
slow_find_file (const char *	filename,
		char *		path,
		char *		result)
     
{
  char current_working_directory[_MAX_PATH];
  char absolute_path[_MAX_PATH];
  char home_directory[_MAX_PATH];
  char *buf;
  char *slsl;
  size_t pathlen;
  size_t bufsize;
  int found;

  get_current_working_directory (current_working_directory,
				 sizeof (current_working_directory));

  get_home_directory (home_directory);

  /* Make an absolute search specification. */
  if (! is_absolute_path (path))
    {
      strcpy (absolute_path, current_working_directory);
      strcat (absolute_path, "\\");
      strcat (absolute_path, path);
      path = absolute_path;
    }

  /* Expand "~". */
  if (path[0] == '~')
    {
      strcpy (absolute_path, current_working_directory);
      strcat (absolute_path, "\\");
      strcat (absolute_path, &path[1]);
      path = absolute_path;
    }

  /* Make a search spec: "{PATH}\\{FILENAME}" */
  pathlen = strlen (path);
  bufsize = pathlen + strlen (filename) + 2;
  buf = _alloca (bufsize);
  strcpy (buf, path);
  if (! isslash (buf[pathlen - 1]))
    strcat (buf, "\\");
  strcat (buf, filename);

  found = 0;

  /* If search spec doesn't contain `//', then try to access the file. */
  slsl = strstr (buf, RECURSION_INDICATOR);
  if (slsl == 0)
    {
      strcpy (result, buf);
      if (debug_flags & DEBUG_FILE_SEARCH)
	fprintf (stderr, "Trying %s...\n", result);
      if (access_regular_file (result, 0) == 0)
	found = 1;
    }
  else
    {
      /* Otherwise, do a recursive search. */
      char *curdir;
      char *searchspec;
      curdir = buf;
      *slsl = 0;
      searchspec = slsl + 2;
      if (rsearch_disk (curdir, 0, searchspec, result))
	found = 1;
    }

  return (found);
}

/* Do a recursive file search using the file system API.
   
   CURDIR is the path to the current directory,
   e.g. "\\\\ronin\\texmf\\fonts\\pk\\ljfour".

   SUBDIR is a sub-directory we are looking for, if any.

   SEARCHSPEC is a search specification, e.g. "dpi300\\cmr10.pk".

   */
static int
rsearch_disk (const char *	curdir,
	      const char *	subdir,
	      const char *	searchspec,
	      char *		result)

{
  size_t curdirlen;

  xassert (curdir != 0);
  xassert (*curdir != 0);
  xassert (searchspec != 0);
  xassert (*searchspec != 0);
  xassert (result != 0);

  curdirlen = strlen (curdir);

  /* If we have a sub directory... */
  if (subdir)
    {
      WIN32_FIND_DATA find_file_data;
      HANDLE hnd;
      size_t bufsize;
      char *buf;
      char *newcurdir;

      /* Make a NEWCURDIR by appending the sub directory name to the
	 current directory name. */
      bufsize = curdirlen + strlen (subdir) + 10;
      buf = _alloca (bufsize);
      newcurdir= buf;
      strcpy (newcurdir, curdir);
      newcurdir[curdirlen] = '\\';
      strcpy (newcurdir + curdirlen + 1, subdir);

      /* If the NEWCURDIR exists, then do a recursive search
         here. Return on success. */
      if (access_directory (newcurdir, 0) == 0)
	{
	  if (rsearch_disk (newcurdir, 0, searchspec, result))
	    return (1);
	}

      /* Do a recursive search in each sub-directory of the CURDIR. */
      strcpy (&newcurdir[curdirlen + 1], "*");
      hnd = FindFirstFile (newcurdir, &find_file_data);
      if (hnd != INVALID_HANDLE_VALUE)
	{
	  int proceed = 1;
	  while (proceed)
	    {
	      if (find_file_data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY
		  && strcmp (find_file_data.cFileName, ".") != 0
		  && strcmp (find_file_data.cFileName, "..") != 0)
		{
		  /* Append sub directory name to current directory name. */
		  strcpy (newcurdir + curdirlen + 1, find_file_data.cFileName);

		  /* And do a recursive search. */
		  if (rsearch_disk (newcurdir, subdir, searchspec, result))
		    {
		      FindClose (hnd);
		      return (1);
		    }
		}
	      proceed = FindNextFile (hnd, &find_file_data);
	    }
	  FindClose (hnd);
	}
      return (0);
    }
  else
    {
      /* No sub-directory is given. Check if we must do a recursive
         search. */
      char *slsl = strstr (searchspec, RECURSION_INDICATOR);
      if (slsl)
	{
	  /* Decompose the SEARCHSPEC into a sub-directory and a
	     smaller search spec. */
	  size_t len = slsl - searchspec;
	  char *subdir = _alloca (len + 1);
	  strncpy (subdir, searchspec, len);
	  subdir[len] = 0;

	  /* Do a recursive search with the new sub-directory and
             search spec. */
	  return (rsearch_disk (curdir, subdir, slsl + 2, result));
	}
      else
	{
	  /* The search spec is rather simple: It is a relative
	     filename path. Append the path to the current directory.
	     Return, if the resulting filename is accessible. */
	  char *filename = result;
	  strcpy (filename, curdir);
	  strcat (filename, "\\");
	  strcat (filename, searchspec);
	  if (debug_flags & DEBUG_FILE_SEARCH)
	    fprintf (stderr, "Trying %s...\n", filename);
	  if (access_regular_file (filename, 0) == 0)
	    return (1);
	  else
	    {
	      /* Do a recursive search in each subdirectory of the
		 current directory. */
	      WIN32_FIND_DATA find_file_data;
	      HANDLE hnd;
	      size_t bufsize = strlen (curdir) + _MAX_FNAME;
	      char *buf = _alloca (bufsize);
	      strcpy (buf, curdir);
	      strcat (buf, "\\*");
	      hnd = FindFirstFile (buf, &find_file_data);
	      if (hnd != INVALID_HANDLE_VALUE)
		{
		  int proceed = 1;
		  while (proceed)
		    {
		      if (find_file_data.dwFileAttributes
			  & FILE_ATTRIBUTE_DIRECTORY
			  && strcmp (find_file_data.cFileName, ".") != 0
			  && strcmp (find_file_data.cFileName, "..") != 0)
			{
			  strcpy (buf + curdirlen + 1,
				  find_file_data.cFileName);
			  if (rsearch_disk (buf, 0, searchspec, result))
			    {
			      FindClose (hnd);
			      return (1);
			    }
			}
		      proceed = FindNextFile (hnd, &find_file_data);
		    }
		}
	      FindClose (hnd);
	      return (0);
	    }
	}
    }
}

static int
rsearch_fndb (int,  int, const char *, const char *,  char *);

static int
fast_find_file (const char *, const char *, int, char *);

/* Find a file in a specified domain. Use file name databases. */
static int
fast_find_file_in_domain (const char *	filename,
			  const char *	path_list,
			  char *	result)

{
  char *my_path_list;
  char *path;
  char *next_path;
  int found;

  /* Make a private copy of the path list and normalize it. */
  my_path_list = strdup (path_list);
  normalize_search_spec (my_path_list);

  /* Initialize search. */
  path = my_path_list;
  next_path = strchr (path, PATH_SEP_CHAR);
  if (next_path)
    *next_path++ = 0; 
  found = 0;

  /* For each path list entry... */
  while (path && ! found)
    {
      if (path[0] == '%' && path[1] == 'R' && isslash (path[2]))
	{
	  char alt_path[_MAX_PATH];
	  size_t i;

	  for (i = 0; i < get_number_of_texmf_roots () && ! found; i++)
	    {
	      int fndb_handle;
	      xassert (get_root_directory (i) != 0);
	      _makepath (alt_path, 0, get_root_directory (i), &path[3], 0);
	      fndb_handle = get_fndb_handle (i);
	      if (fndb_handle >= 0)
		{
		  found = fast_find_file (filename, alt_path, fndb_handle,
					  result);
		}
	    }
	}
      else
	{
	  char alt_path[_MAX_PATH];
	  int fndb_handle;
	  char *root_relative_filename;
	  size_t i;

	  if (find_texmf_root (path, &root_relative_filename, &i))
	    {
	      _makepath (alt_path, 0, get_root_directory (i),
			 root_relative_filename, 0);
	      fndb_handle = get_fndb_handle (i);
	      if (fndb_handle >= 0)
		found = fast_find_file (filename, alt_path, fndb_handle,
					result);
	    }
	}

      /* Get next path. */
      if (next_path && *next_path)
	{
	  path = next_path;
	  next_path = strchr (next_path, PATH_SEP_CHAR);
	  if (next_path)
	    *next_path++ = 0;
	}
      else
	path = 0;
    }

  free (my_path_list);

  return (found);
}

/* Find a file in a filename database. */
static int
fast_find_file (const char *	filename,
		const char *	path,
		int		fndb_handle,
		char *		result)

{
  size_t filename_len;
  size_t path_len;
  char *slsl;
  char *search_spec;
  int found;

  filename_len = strlen (filename);
  path_len = strlen (path);

  search_spec = _alloca (path_len + filename_len + 3);
  strcpy (search_spec, path);

  if (! isslash (search_spec[path_len - 1]))
    strcat (search_spec, "\\");
  strcat (search_spec, filename);

  found = 0;

  if (debug_flags & DEBUG_FNDB)
    fprintf (stderr, "Searching %s via fndb %u\n", search_spec, fndb_handle);

  slsl = strstr (search_spec, RECURSION_INDICATOR);
  if (slsl == 0)
    {
      int hnd;
      strcpy (result, search_spec);
      if (fndb_find_file (fndb_handle,
			  fndb_open_directory (fndb_handle, -1, 0),
			  result, 0, 0))
	{
	  if (debug_flags & DEBUG_FNDB)
	    fprintf (stderr, "Found %s in fndb %u\n", result, fndb_handle);
	  found = 1;
	}
    }
  else
    {
      if (rsearch_fndb (fndb_handle,
			fndb_open_directory (fndb_handle, -1, 0),
			0, search_spec, result))
	{
	  if (debug_flags & DEBUG_FNDB)
	    fprintf (stderr, "Found %s in fndb %u\n", result, fndb_handle);
	  found = 1;
	}
    }

  return (found);
}

/* Find a file by recursively searching a filename database. */
static int
rsearch_fndb (int		fndb_handle,
	      int		dir_handle,
	      const char *	subdir,
	      const char *	searchspec,
	      char *		result)

{
  int subdir_handle;
  if (subdir)
    {
      if ((subdir_handle = fndb_open_directory (fndb_handle, dir_handle,
						subdir))
	  >= 0)
	{
	  int found = rsearch_fndb (fndb_handle, subdir_handle,
				    0, searchspec, result);
	  fndb_close_directory (subdir_handle);
	  if (found)
	    return (1);
	}
      while ((subdir_handle = fndb_open_next_sub_directory (dir_handle)) >= 0)
	{
	  int found = rsearch_fndb (fndb_handle, subdir_handle, subdir,
				    searchspec, result);
	  fndb_close_directory (subdir_handle);
	  if (found)
	    return (1);
	}
    }
  else
    {
      char *slsl = strstr (searchspec, RECURSION_INDICATOR);
      if (slsl)
	{
	  size_t len = slsl - searchspec;
	  char *subdir = _alloca (len + 1);
	  strncpy (subdir, searchspec, len);
	  subdir[len] = 0;
	  return (rsearch_fndb (fndb_handle, dir_handle, subdir,
				slsl + 2, result));
	}
      else
	{
	  if (fndb_find_file (fndb_handle, dir_handle, searchspec, result, 0))
	    return (1);
	  else
	    {
	      while ((subdir_handle
		      = fndb_open_next_sub_directory (dir_handle)) >= 0)
		{
		  int found = rsearch_fndb (fndb_handle, subdir_handle, 0,
					    searchspec, result);
		  fndb_close_directory (subdir_handle);
		  if (found)
		    return (1);
		}
	    }
	}
    }
  return (0);
}

/* _________________________________________________________________________

   SEARCHING SPECIAL FILES.
   _________________________________________________________________________ */


/* Find a MiKTeX executable. */
_MIKTEX_1 int _MIKTEX_2
find_executable (const char *	exe_filename,
		 char *		full_exe_filename)

{
  char bin_directories[_MAX_PATH];
  char *bindir;
  char *next_bindir;
  int found;

  get_cfg_value ("MiKTeX", "Bin Directories",
		 bin_directories, sizeof (bin_directories),
		 "%R\\miktex\\bin");

  found = 0;

  /* Initialize search. */
  bindir = bin_directories;
  next_bindir = strchr (bindir, PATH_SEP_CHAR);
  if (next_bindir)
    *next_bindir++ = 0; 

  while (bindir && ! found)
    {
      if (bindir[0] == '%' && bindir[1] == 'R' && isslash (bindir[2]))
	{
	  char alt_bindir[_MAX_PATH];
	  size_t i;

	  for (i = 0; i < get_number_of_texmf_roots () && ! found; i++)
	    {
	      xassert (get_root_directory (i) != 0);
	      _makepath (alt_bindir, 0, get_root_directory (i), &bindir[3], 0);
	      _makepath (full_exe_filename, 0, alt_bindir, exe_filename, 0);
	      if (access_regular_file (full_exe_filename, 0) == 0)
		found = 1;
	    }
	}
      else
	{
	  char partial_exe_filename[_MAX_PATH];
	  _makepath (partial_exe_filename, 0, bindir, exe_filename, 0);
	  _fullpath (full_exe_filename, partial_exe_filename, _MAX_PATH);
	  if (access_regular_file (full_exe_filename, 0) == 0)
	    found = 1;
	}

      /* Get next path. */
      if (next_bindir && *next_bindir)
	{
	  bindir = next_bindir;
	  next_bindir = strchr (next_bindir, PATH_SEP_CHAR);
	  if (next_bindir)
	    *next_bindir++ = 0;
	}
      else
	bindir = 0;
    }

  return (found);
}

#define MAKETEXTFM_EXE "MakeTeXTFM.exe"

/* Open a TFM file. Try to create it if necessary. */
_MIKTEX_1 boolean _MIKTEX_2
mtopentfmfil (bytefile *f,
	      char *name)

{
  FILE *fp;
  char filenamebuf[_MAX_PATH];
  static char buf[2048];
  const char *path_list;
  char *section = "TeX";
  char *key = "Font Metric Dirs";
  char *filename = "fixme.tfm";
  char *save_name = STRDUP (name);
  parse_web_filename (save_name, &section, &key, &filename);
  path_list = get_search_path (section, key, buf, sizeof (buf), 0);
  if (path_list == 0)
    return (false);
  if (find_file (filename, path_list, filenamebuf))
    {
      fp = open_file (filenamebuf, c4prbmode);
      if (fp == 0)
	return (false);
      f->_c4p_fp = fp;
      get (*f);
      return (true);
    }
  else
    {
      char basename[_MAX_PATH];
      char command_line[_MAX_PATH];
      char *argv[10];
      int argc = 0;
      char *ext;
      int rc;

      strcpy (basename, filename);
      ext = strstr (basename, ".tfm");
      if (ext)
	*ext = 0;

      if (! find_executable (MAKETEXTFM_EXE, command_line))
	return (false);

      argv[argc++] = command_line;
      argv[argc++] = "-v";
      argv[argc++] = basename;
      argv[argc++] = 0;
      rc = _spawnv (_P_WAIT, command_line, argv);
      if (rc == 0 && find_file (filename, path_list, filenamebuf))
	{
	  fp = open_file (filenamebuf, c4prbmode);
	  if (fp == 0)
	    return (false);
	  f->_c4p_fp = fp;
	  get (*f);
	  return (true);
	}
    }
  return (false);
}

/* Open a VF file. Called by a DVI driver (e.g. dvips). */
_MIKTEX_1 boolean _MIKTEX_2
mtopenvffile (bytefile *f,
	      char *name)

{
  FILE *fp;
  char filenamebuf[_MAX_PATH];
  static char buf[2048];
  const char *path_list;
  char *section = "dvips";
  char *key = "VFFONTS";
  char *filename = "fixme.vf";
  char *save_name = STRDUP (name);
  parse_web_filename (save_name, &section, &key, &filename);
  path_list = get_search_path (section, key, buf, sizeof (buf), 0);
  if (path_list == 0)
    return (false);
  if (find_file (filename, path_list, filenamebuf))
    {
      fp = open_file (filenamebuf, c4prbmode);
      if (fp == 0)
	return (false);
      f->_c4p_fp = fp;
      get (*f);
      return (true);
    }
  else
    return (false);
}

/* Find a PK file for a given dpi/mode. */
_MIKTEX_1 boolean _MIKTEX_2
find_pk_file (const char *	fontname,
	      const char *	mode,
	      int		dpi,
	      char *		result)

{
  char *search_path;
  char *p;
  const char *q;
  char basefilename[_MAX_PATH];
  char buf[MAX_PATH];
  strcpy (basefilename, fontname);
  strcat (basefilename, ".pk");
  q = get_search_path ("yap", "PK Input Dirs", buf, sizeof (buf), 0);
  if (q == 0)
    q = get_search_path ("dvips", "texpks", buf, sizeof (buf), 0);
  if (q == 0)
    q = "fixme";
  search_path = (char *) _alloca (strlen (q) + _MAX_PATH);
  p = search_path;
  while (*q)
    {
      if (*q == '%')
	{
	  q++;
	  if (*q == 0)
	    break;
	  switch (*q)
	    {
	    case 'R':
	      *p++ = '%';
	      *p++ = 'R';
	      break;
	    case '%':
	      *p++ = '%';
	      break;
	    case 'm':
	      if (mode)
		strcpy (p, mode);
	      else
		strcpy (p, "fixme");
	      p += strlen (p);
	      break;
	    case 'd':
	      sprintf (p, "%ld", (long) dpi);
	      p += strlen (p);
	      break;
	    default:
	      strcpy (p, "fixme");
	      p += strlen (p);
	      break;
	    }
	}
      else
	*p++ = *q;
      q++;
    }
  *p = 0;
  return (find_file (basefilename, search_path, result));
}
/* _________________________________________________________________________

   MAKING PK FONTS.
   _________________________________________________________________________ */


static int
magstep (int	n,
	 int	bdpi)

{
  float t;
  int neg = 0;

  if (n < 0)
    {
      neg = 1;
      n = -n;
    }
  if (n & 1)
    {
      n &= ~1;
      t = 1.095445115;
    }
  else
    t = 1.0;
  while (n > 8)
    {
      n -= 8;
      t = t * 2.0736;
    }
  while (n > 0)
    {
      n -= 2 ;
      t = t * 1.2;
    }
  if (neg)
    return ((int) (0.5 + bdpi / t));
  else
    return ((int) (0.5 + bdpi * t));
}

_MIKTEX_1 int _MIKTEX_2
make_pk_font (const char *	name,
	      int		dpi,
	      int		bdpi,
	      const char *	mode)

{
  char command_line[_MAX_PATH * 3];
  char dpi_string[10];
  char bdpi_string[10];
  char magstep_string[100];
  const char *argv[10];
  int argc;
  int m, n;

  if (! find_executable ("MakeTeXPK.exe", command_line))
    return (-1);

  sprintf (dpi_string, "%d", dpi);
  sprintf (bdpi_string, "%d", bdpi);

  m = 0;
  if (dpi < bdpi)
    {
      while (1)
	{
	  m--;
	  n = magstep (m, bdpi);
	  if (n == dpi)
	    break;
	  if (n < dpi || m < -40)
	    {
	      m = 9999;
	      break;
	    }
	}
    }
  else if (dpi > bdpi)
    {
      while (1)
	{
	  m++;
	  n = magstep (m, bdpi);
	  if (n == dpi)
	    break;
	  if (n > dpi || m > 40)
	    {
	      m = 9999;
	      break;
	    }
	}
    }
  if (m == 9999)
    sprintf (magstep_string, "%d+%d/%d", dpi/bdpi, dpi%bdpi, bdpi);
  else if (m >= 0)
    sprintf (magstep_string, "magstep(%d.%d)", m/2, (m&1)*5);
  else
    sprintf (magstep_string, "magstep(-%d.%d)", (-m)/2, (m&1)*5);
  
  argc = 0;
  argv[argc++] = command_line;
  argv[argc++] = "--verbose";
  argv[argc++] = name;
  argv[argc++] = dpi_string;
  argv[argc++] = bdpi_string;
  argv[argc++] = magstep_string;
  if (mode)
    argv[argc++] = mode;
  argv[argc++] = 0;

  return (_spawnv (_P_WAIT, command_line, argv));
}

/* _________________________________________________________________________

   CONFIGURATION SETTINGS.
   _________________________________________________________________________ */


/* Get a configuration parameter. */
static char *
_get_cfg_value (HKEY		root,
		const char *	section,
		const char *	key,
		char *		buf,
		size_t		bufsize,
		char *		defval)

{
  HKEY hKey;
  LONG res;
  char path[_MAX_PATH];

  sprintf (path, "Software\\MiK\\MiKTeX\\CurrentVersion\\%s", section);
  res = RegOpenKeyEx (root,
		      path,
		      0,
		      KEY_READ,
		      &hKey);
  if (res == ERROR_SUCCESS)
    {
      DWORD type;
      DWORD len = bufsize;
      res = RegQueryValueEx (hKey,
			     key,
			     0,
			     &type,
			     buf,
			     &len);
      RegCloseKey (hKey);
      if (res == ERROR_SUCCESS)
	return (buf);
    }

  if (defval)
    return (strcpy (buf, defval));
 
  return (0);
}

/* Get a personal configuration parameter. */
static char *
get_personal_cfg_value (const char *	section,
			const char *	key,
			char *		buf,
			size_t		bufsize,
			char *		defval)

{
  return (_get_cfg_value (HKEY_CURRENT_USER,
			  section, key, buf, bufsize, defval));
}

/* Get a global configuration parameter. */
static char *
get_global_cfg_value (const char *	section,
		      const char *	key,
		      char *		buf,
		      size_t		bufsize,
		      char *		defval)

{
  return (_get_cfg_value (HKEY_LOCAL_MACHINE,
			  section, key, buf, bufsize, defval));
}

/* Get a configuration parameter. */
_MIKTEX_1 char * _MIKTEX_2
get_cfg_value (const char *	section,
	       const char *	key,
	       char *		buf,
	       size_t		bufsize,
	       char *		defval)

{
  char *value;

  /* First try to get a personal value. */
  value = get_personal_cfg_value (section, key, buf, bufsize, 0);
  if (value)
    return (value);

  /* Then try to get a global value. */
  value = get_global_cfg_value (section, key, buf, bufsize, 0);
  if (value)
    return (value);

  /* Return default value. */
  return (strcpy (buf, defval));
}

/* Parse a search path; store directory name pointers in DIRS. */
static size_t
decompose_search_path (char *	search_path,
		       char **	dirs)

{
  size_t i;
  char *cp;

  i = 0;
  cp = search_path;
  while (cp && *cp)
    {
      if (*cp != PATH_SEP_CHAR)
	dirs[i++] = cp;
      cp = strchr (cp, PATH_SEP_CHAR);
      if (cp)
	{
	  *cp = 0;
	  cp++;
	}
    }

  return (i);
}

/* Concatenate PATH1 with PATH2; eliminate duplicate entries. */
static char *
combine_search_paths (char *	path1,
		      char *	path2,
		      char *	result)

{
  char *dirs1[20];
  char *dirs2[20];
  size_t n_dirs1;
  size_t n_dirs2;
  size_t i;
  size_t j;

  n_dirs1 = decompose_search_path (path1, dirs1);
  n_dirs2 = decompose_search_path (path2, dirs2);

  /* First PATH1 elements. */
  *result = 0;
  for (i = 0; i < n_dirs1; i++)
    {
      if (i > 0)
	strcat (result, ";");
      strcat (result, dirs1[i]);
    }

  /* Then PATH2 elements. */
  for (j = 0; j < n_dirs2; j++)
    {
      for (i = 0; i < n_dirs1; i++)
	{
	  if (_strcmpi (dirs1[i], dirs2[j]) == 0)
	    break;
	}
      if (i == n_dirs1)
	{
	  strcat (result, ";");
	  strcat (result, dirs2[j]);
	}
    }

  return (result);
}

/* Get a search path. */
_MIKTEX_1 char * _MIKTEX_2
get_search_path (const char *	section,
		 const char *	key,
		 char *		buf,
		 size_t		bufsize,
		 char *		defval)

{
  char * personal_path;
  char * global_path;
  char * path1;
  char * path2;
  char * ret;
  static char section_cache[_MAX_FNAME];
  static char key_cache[_MAX_FNAME];
  static char value_cache[_MAX_PATH * 5];

  if (_strcmpi (section, section_cache) == 0
      && _strcmpi (key, key_cache) == 0)
    return (strcpy (buf, value_cache));

  /* Get personal path. */
  personal_path = (char *) _alloca (bufsize);
  path1 = get_personal_cfg_value (section, key, personal_path, bufsize, 0);

  /* Get global path. */
  global_path = (char *) _alloca (bufsize);
  path2 = get_global_cfg_value (section, key, global_path, bufsize, 0);

  /* Return default if both are undefined. */
  if (path1 == 0 && path2 == 0)
    {
      if (defval)
	return (strcpy (buf, defval));
      else
	return (0);
    }

  /* Combine both paths. */
  if (path1 && path2)
    ret = combine_search_paths (path1, path2, buf);
  else if (path1)
    ret = strcpy (buf, path1);
  else
    ret = strcpy (buf, path2);

  strcpy (section_cache, section);
  strcpy (key_cache, key);
  strcpy (value_cache, ret);

  return (ret);
}

/* Determine the name of user's home directory. */
_MIKTEX_1 char * _MIKTEX_2
get_home_directory (char *buf)

{
  const char *homedrive = getenv ("HOMEDRIVE");
  const char *homepath = getenv ("HOMEPATH");
  const char *home = getenv ("HOME");
  static char mybuf[_MAX_PATH];
  if (buf == 0)
    buf = mybuf;
  if (homedrive && homepath)
    {
      strcpy (buf, homedrive);
      if (! isslash (homepath[0]))
	strcat (buf, "\\");
      strcat (buf, homepath);
    }
  else if (home)
    strcpy (buf, home);
  else
    strcpy (buf, "c:\\");
  /* fixme: we should check the existence */
  return (buf);
}

/* _________________________________________________________________________

   READING MAP FILES
   _________________________________________________________________________ */


static int
is_prefix_of (const char *	s1,
	      const char *	s2)

{
  size_t l1;

  xassert (s1 != 0);
  xassert (s2 != 0);

  l1 = strlen (s1);
  return (l1 <= strlen (s2) && strncmp (s1, s2, l1) == 0);
}

static int
find_in_typeface_map (const char *	font_name,
		      const char *	supplier_arg,
		      char *		typeface_arg)

{
  char filenamebuf[_MAX_PATH];
  int found;

  found = 0;

  if (strlen (font_name) >= 3
      && find_file ("typeface.map", "%R\\fontname", filenamebuf))
    {
      FILE *map_file = fopen (filenamebuf, "r");
      if (map_file)
	{
	  char linebuf[200];	/* fixme */
	  char typeface_abbrev[3];

	  typeface_abbrev[0] = font_name[1];
	  typeface_abbrev[1] = font_name[2];
	  typeface_abbrev[2] = 0;

	  while (! found && fgets (linebuf, sizeof (linebuf), map_file) != 0)
	    {
	      if (strncmp (linebuf, "@c", 2) != 0)
		{
		  char abbrev[_MAX_FNAME];
		  char supplier[_MAX_FNAME];
		  char typeface[_MAX_FNAME];

		  if ((sscanf (linebuf, "%s %s %s", abbrev, supplier, typeface)
		       == 3)
		      && strcmp (abbrev, typeface_abbrev) == 0)
		    {
		      strcpy (typeface_arg, typeface);
		      found = 1;
		    }
		}
	    }
	  fclose (map_file);
	}
    }

  return (found);
}

static int
find_in_supplier_map (const char *	font_name,
		      char *		supplier_arg,
		      char *		typeface_arg)

{
  char filenamebuf[_MAX_PATH];
  int found;

  found = 0;

  if (find_file ("supplier.map", "%R\\fontname", filenamebuf))
    {
      FILE *map_file = fopen (filenamebuf, "r");
      if (map_file)
	{
	  char linebuf[200];	/* fixme */
	  char supplier_abbrev = font_name[1];
	  while (! found && fgets (linebuf, sizeof (linebuf), map_file) != 0)
	    {
	      if (strncmp (linebuf, "@c", 2) != 0)
		{
		  char abbrev;
		  char supplier[_MAX_FNAME];
		  if (sscanf (linebuf, "%c %s", &abbrev, supplier) == 2
		      && supplier_abbrev == abbrev)
		    {
		      found = 1;
		      strcpy (supplier_arg, supplier);
		    }
		}
	    }
	  fclose (map_file);
	}
    }

  if (found && supplier_arg && typeface_arg)
    found = find_in_typeface_map (font_name, supplier_arg, typeface_arg);

  return (found);
}

static int
find_in_special_map (const char *	font_name,
		     char *		supplier_arg,
		     char *		typeface_arg)

{
  char filenamebuf[_MAX_PATH];
  int found;

  found = 0;

  if (find_file ("special.map", "%R\\fontname", filenamebuf))
    {
      FILE *map_file = fopen (filenamebuf, "r");
      if (map_file)
	{
	  char linebuf[200];	/* fixme */
	  while (! found && fgets (linebuf, sizeof (linebuf), map_file) != 0)
	    {
	      if (strncmp (linebuf, "@c", 2) != 0)
		{
		  char name[_MAX_FNAME];
		  char supplier[_MAX_FNAME];
		  char typeface[_MAX_FNAME];
		  if (sscanf (linebuf, "%s %s %s",
			      name, supplier, typeface) == 3)
		    {
		      if (strcmp (name, font_name) == 0
			  || (is_prefix_of (name, font_name)
			      && (isdigit (lastch (font_name)))
			      && (! isdigit (lastch (name)))))
			{
			  strcpy (supplier_arg, supplier);
			  strcpy (typeface_arg, typeface);
			  found = 1;
			}
		    }
		}
	    }
	  fclose (map_file);
	}
    }

  return (found);
}

_MIKTEX_1 int _MIKTEX_2
get_font_info (const char *	font_name,
	       char *		supplier_arg,
	       char *		typeface_arg)

{
  xassert (font_name != 0);
  xassert (supplier_arg != 0);
  xassert (typeface_arg != 0);

  return (find_in_special_map (font_name, supplier_arg, typeface_arg)
	  || find_in_supplier_map (font_name, supplier_arg, typeface_arg));
}

/* _________________________________________________________________________

   HELPERS FOR TEX & FRIENDS.
   _________________________________________________________________________ */


/* Print the MiKTeX version number. */
_MIKTEX_1 void _MIKTEX_2
miktexbanner (_c4p_textfile_t *f)

{
  fprintf (f->_c4p_fp,
	   " (%s %s)",
	   VER_PRODUCTNAME_STR, VER_PRODUCTVERSION_STR);
}

/* Pre-[TeX|MF|MP]-init routine. */
_MIKTEX_1 void _MIKTEX_2
mtfirstword (void)

{
  start_clock = clock ();
}

/* Post-init routine. */
_MIKTEX_1 void _MIKTEX_2
mtafterinit (void)

{
  ;
}

static int
is_windows_nt ()

{
  OSVERSIONINFO osver;
  osver.dwOSVersionInfoSize = sizeof (OSVERSIONINFO);
  GetVersionEx (&osver);
  return (osver.dwPlatformId == VER_PLATFORM_WIN32_NT);
}

/* Post-job routine. */
_MIKTEX_1 void _MIKTEX_2
mtlastword (void)

{
  char buf[10];
  if (atoi (get_cfg_value ("miktex", "showexecutiontime",
			   buf, sizeof (buf), "0")))
    {
      clock_t ellapsed_time = clock () - start_clock;;
      printf ("\nGross execution time: %u ms\n", ellapsed_time);
      if (is_windows_nt ())
	{
	  HINSTANCE lib;
	  lib = LoadLibrary ("kernel32.dll");
	  if (lib != 0)
	    {
	      FARPROC get_process_times;
	      get_process_times = GetProcAddress (lib, "GetProcessTimes");
	      if (get_process_times != 0)
		{
		  FILETIME ftCreate, ftExit, ftKernel, ftUser;
    		  if (GetProcessTimes (GetCurrentProcess (),
				       &ftCreate,
				       &ftExit,
				       &ftKernel,
				       &ftUser))
		    {
		      LARGE_INTEGER tUser64;
		      LARGE_INTEGER tKernel64;
		      DWORD tUser, tKernel;
		      
		      tUser64.LowPart = ftUser.dwLowDateTime;
		      tUser64.HighPart = ftUser.dwHighDateTime;
		      tKernel64.LowPart = ftKernel.dwLowDateTime;
		      tKernel64.HighPart = ftKernel.dwHighDateTime;

		      tUser = (DWORD) (tUser64.QuadPart / 10000);
		      tKernel = (DWORD) (tKernel64.QuadPart / 10000);

		      printf ("User mode: %u ms, ", tUser);
		      printf ("Kernel mode: %u ms, ", tKernel);
		      printf ("Total: %u\n", tUser + tKernel);
		    }
		}
	      FreeLibrary (lib);
	    }
	}
    }
}

/* Parse a system-independend filename (as defined in tex.web). */
_MIKTEX_1 void _MIKTEX_2
parse_web_filename (char *web_filename,
		    char **area,
		    char **directory,
		    char **filename)

{
  if (*web_filename == '[')
    {
      web_filename = strtok (web_filename + 1, "]");
      if (web_filename)
	{
	  if (area)
	    *area = web_filename;
	  web_filename = strtok (0, ":");
	  if (web_filename)
	    {
	      if (*directory)
		*directory = web_filename;
	      if (filename)
		*filename = web_filename + strlen (web_filename) + 1;
	    }
	}
    }
  else if (filename)
    *filename = web_filename;
}

_MIKTEX_1 int _MIKTEX_2
set_metafont_mode (const mf_mode *mode)

{
  const char *home = get_home_directory (0);
  char dvisrc[_MAX_PATH];
  sprintf (dvisrc, "%s\\.dvipsrc", home);
  if (access_regular_file (dvisrc, 0))
    {
      char buf[1000];
      FILE *fp1, *fp2;
      char *tname = _tempnam ("c:\\tmp", "mik");
      fp1 = fopen (dvisrc, "r");
      if (fp1 == 0)
	return (-1);
      fp2 = fopen (tname, "w");
      if (fp2 == 0)
	{
	  fclose (fp1);
	  return (-1);
	}
      while (fgets (buf, sizeof (buf), fp1))
	{
	  if (*buf == 'M')
	    fprintf (fp2, "M %s\n", mode->mnemonic);
	  else if (*buf == 'D')
	    fprintf (fp2, "D %d", mode->mnemonic);
	  else
	    fputs (buf, fp2);
	}
      fclose (fp1);
      fclose (fp2);
      if (_unlink (dvisrc) == 0)
	rename (tname, dvisrc);
    }
  else
    {
      FILE *fp = fopen (dvisrc, "w");
      fprintf (fp, "M %s\n", mode);
      fclose (fp);
    }
}

_MIKTEX_1 int _MIKTEX_2
get_metafont_modes (mf_mode *modes,
		    size_t array_size)

{
  FILE *fp;
  char filenamebuf[_MAX_PATH];
  static char buf[2048];
  const char *path_list;
  char *section = "METAFONT";
  char *key = "Input Dirs";
  char *filename = "modes.mf";
  int count = 0;
  int in_mode_def = 0;
  path_list = get_cfg_value (section, key, buf, sizeof (buf), 0);
  if (path_list == 0)
    return (-1);
  if (! find_file (filename, path_list, filenamebuf))
    return (-1);
  fp = open_file (filenamebuf, "r");
  if (fp == 0)
    return (-1);
  while (fgets (buf, sizeof (buf), fp) && count < array_size)
    {
      if (in_mode_def)
	{
	  char *cp;
	  if (strncmp (buf, "enddef;", 7) == 0)
	    {
	      count++;
	      in_mode_def = 0;
	    }
	  else if (modes[count].hor_res == 0
		   && (cp = strstr (buf, "pixels_per_inch")))
	    {
	      while (*cp && ! isdigit (*cp))
		cp++;
	      modes[count].hor_res = atoi (cp);
	      modes[count].vert_res = atoi (cp);
	    }
	  else if (cp = strstr (buf, "aspect_ratio"))
	    {
	      while (*cp && ! isdigit (*cp))
		cp++;
	      modes[count].vert_res = atoi (cp);
	    }
	}
      else if (strncmp (buf, "mode_def", 8) == 0)
	{
	  char *cp = buf + 8;
	  char *mode_name;
	  char *printer_name;
	  while (! isalpha (*cp) && *cp != '\n')
	    cp++;
	  if (! isalpha (*cp))
	    continue;
	  mode_name = cp;
	  while (isalpha (*cp))
	    cp++;
	  if (*cp == 0)
	    continue;
	  *cp++ = 0;
	  if (strcmp (mode_name, "help") == 0)
	    continue;
	  cp = strstr (cp, "%\\[");
	  if (cp == 0)
	    continue;
	  cp += 3;
	  while (isspace (*cp))
	    *cp++;
	  if (cp == 0)
	    continue;
	  printer_name = cp;
	  while (*cp && *cp != '\n')
	    cp++;
	  *cp = 0;
	  in_mode_def++;
	  modes[count].mnemonic = strdup (mode_name);
	  modes[count].description = strdup (printer_name);
	  modes[count].hor_res = 0;
	}
    }
  return (count);
}

/* _________________________________________________________________________

   DLL ENTRY / EXIT.
   _________________________________________________________________________ */


static void
get_debug_options ()

{
  char buf[_MAX_PATH];
  char *cp;

  get_cfg_value ("miktex", "debug", buf, sizeof (buf), "ndebug");
  
  debug_flags = 0;
  cp = strtok (buf, ",");
  while (cp && *cp)
    {
      if (strcmp (cp, "ndebug") == 0)
	debug_flags = 0;
      else if (strcmp (cp, "filesearch") == 0)
	debug_flags |= DEBUG_FILE_SEARCH;
      else if (strcmp (cp, "fndb") == 0)
	debug_flags |= DEBUG_FNDB;
      cp = strtok (0, ",");
    }
}

/* DLL entry/exit routine. */
#ifdef MIKTEX_DLL
int APIENTRY 
DllMain (HINSTANCE	ignore1,
	 DWORD		reason,
	 LPVOID		ignore2)
     
{
  if (reason == DLL_PROCESS_ATTACH)
    get_debug_options ();
  return (1);
}
#endif /* MIKTEX_DLL */

/* miktex.c ends here */
