/*
 * LaTeD Version 1.1
 * (c) Gene Ressler 1993, 94, 97
 *   de8827@trotter.usma.edu
 *
 * LaTeD is a graphical editor for drawings in the LaTeX "picture" 
 * environment.  It runs under MSDOS or in a Windows DOS box.  The
 * distribution includes full sources, including LaTeX source for 
 * its documentation.
 *
 * No warranty of this software is expressed or implied by the author.
 *
 * Copy and use this program freely for any purpose except for sale
 * of the program (including the source code) itself.  That is, 
 * no one can copy this program for the purpose of providing it to 
 * another person in exchange for money or other compensation, even 
 * if this program is only part of the exchange.
 *
 * All copies of computer source code in this distribution, whether
 * copies in whole or in part, must have this notice attached.
 */

#include <stdlib.h>
#include <alloc.h>
#include <mem.h>
#include <assert.h>
#include <graphics.h>
#include "window.h"
#include "wintern.h"

/* Id of a destroyed window. */
#define DEAD 0xffff

int root_bg_color = cDARKGRAY;

/* Check consistency of window hierarchy. */
#ifndef NDEBUG

static int pn, nn;

LOCAL(void) pwalk(WINDOW w)
{
  ++pn;
  assert(pn < 300);
  if (w->kids != NULL) {
    WINDOW k, end;
    k = end = w->kids->prev;
    do {
      pwalk(k);
    } while ((k = k->prev) != end);
  }
}

LOCAL(void) nwalk(WINDOW w)
{
  ++nn;
  assert(nn < 300);
  if (w->kids != NULL) {
    WINDOW k = w->kids;
    do {
      nwalk(k);
    } while ((k = k->next) != w->kids);
  }
}

int window_hierarchy_ok_p(void)
{
  /* Don't worry if root window isn't init'ed yet. */
  if (root_window == NULL)
    return 1;
  pn = nn = 0;
  pwalk(root_window);
  nwalk(root_window);
  return (pn == nn);
}

#endif

/* Put a window on a doubly linked loop of them. */
LOCAL(void) push_window(WINDOW win, WINDOW *wloop)
{
  WINDOW h = *wloop;
  if (h != NULL) {
    win->next = h;
    win->prev = h->prev;
    h->prev->next = win;
    h->prev = win;
  }
  *wloop = win;
  assert(window_hierarchy_ok_p());
}

/* Remove a window from a loop. */
LOCAL(void) delete_window(WINDOW win, WINDOW *wloop)
{
  WINDOW next, prev;
  if (win->next == win) 
    *wloop = NULL;
  else {
    next = win->next;
    prev = win->prev;
    next->prev = prev;
    prev->next = next;
    if (*wloop == win)
     *wloop = next;
  }
  win->next = win->prev = win;
  assert(window_hierarchy_ok_p());
}

/* Allocate storage for a window or window superclass. */
void walloc(void *p, unsigned n)
{
  WINDOW w = *(WINDOW*)p;

  if (w == NULL) {
    w = malloc(n);
    memset(w, 0, n);
    w->status_mask = bit(wsDYNAMIC);
    *(WINDOW*)p = w;
  }
  else 
    w->status_mask = 0;
}

/* Make a new window. */
WINDOW open_window(WINDOW w, WINDOW parent, 
		   int x, int y, int width, int height, 
		   int border_width, int border_color, int bg_color, 
		   unsigned event_mask)
{
  static unsigned id = 0;
  assert(w != NULL);
  w->id = ++id;
  if (x == CENTER_WINDOW)
    x = (parent->width - width) / 2;
  if (y == CENTER_WINDOW)
    y = (parent->height - height) / 2;
  if (x < 0) x = 0;
  if (y < 0) y = 0;
  w->x = x + parent->x;
  w->y = y + parent->y;
  w->width = min(width, parent->width - x);
  w->height = min(height, parent->height - y);
  assert(w->width > 0 && w->height > 0);
  w->parent = parent;
  w->bg_color = bg_color;
  w->border_color = border_color;
  w->border_width = border_width;
  w->event_mask = event_mask;
  w->prev = w->next = w;
  w->kids = NULL;
  w->save = NULL;
  w->handler = null_handler_code;
  push_window(w, &parent->kids);
  assert(window_hierarchy_ok_p());
  return(w);
}

/* Set the save under property of a window. */
void set_window_save_under(WINDOW w)
{
  assert(!w_status_p(w, wsVISIBLE));
  w->status_mask |= bit(wsSAVE_UNDER);
}

LOCAL(void) draw_border(WINDOW w)
{
  int i, p0, x1, y1;

  if (w->border_width 
      && w->border_color != TRANSPARENT 
      && w->border_color != NO_COLOR) {
    setcolor(w->border_color);
    for (i = 0, p0 = -1, x1 = w->width, y1 = w->height;
      i < w->border_width;
      ++i, --p0, ++x1, ++y1)
      rectangle(p0, p0, x1, y1);
  }
}

/* Paint a whole window with its background color/pattern. */
void clear_window(WINDOW w, int map_event_p)
{
  if (w_status_p(w, wsVISIBLE)) {
    push_graphics_state(w, 0);
    protect_cursor(w);
    if (!colorless_p(w->bg_color)) {
      setfillstyle(SOLID_FILL, w->bg_color);
      bar(0, 0, w->width-1, w->height-1);
    }
    draw_border(w);
    unprotect_cursor();
    pop_graphics_state();
    if (map_event_p)
      push_map_event(eMAP, w);
  }
  assert(window_hierarchy_ok_p());
}

/* Change color of window border (if requested
   color is different than old one. */
void set_window_border_color(WINDOW w, int color)
{
  if (color == w->border_color)
    return;

  w->border_color = color;

  if (w_status_p(w, wsVISIBLE)) {
    push_graphics_state(w, 0);
    protect_cursor(w);
    draw_border(w);
    unprotect_cursor();
    pop_graphics_state();
  }
}

/* Mark window w and its descendents visible. */
LOCAL(void) set_visible(WINDOW w)
{
  if (!w_status_p(w, wsMAPPED))
    return;
  w->status_mask |= bit(wsVISIBLE);
  if (w->kids != NULL) {
    WINDOW k = w->kids;
    do {
      set_visible(k);
    } while ((k = k->next) != w->kids);
  }
}

/* Paint a window and recursively paint
   its mapped descendents.  Also
   mark it visible. */
LOCAL(void) paint_window(WINDOW w)
{
  if (!w_status_p(w, wsMAPPED))
    return;
  if (w_status_p(w, wsSAVE_UNDER) && w->save == NULL) {
    /* Deal with stupid get/putimage by using absolute coords
       and checking, else they go die if border is off screen. */
    int bw = w->border_width,
	x0 = max(0, w->x - bw),
	y0 = max(0, w->y - bw),
	x1 = min(getmaxx(), w->x + w->width + bw - 1),
	y1 = min(getmaxy(), w->y + w->height + bw - 1);
    w->save = malloc(sizeof(SAVE_HEADER) + imagesize(x0, y0, x1, y1));
    w->save->x0 = x0;
    w->save->y0 = y0;
    push_graphics_state(root_window, 0);
    protect_cursor(w);
    getimage(x0, y0, x1, y1, w->save->bits);
    unprotect_cursor();
    pop_graphics_state();
  }
  clear_window(w, 1);
  if (w->kids != NULL) {
    WINDOW k = w->kids;
    do {
      paint_window(k);
    } while ((k = k->next) != w->kids);
  }
}

/* Map a window onto the screen. */
void map_window(WINDOW w)
{
  if (w_status_p(w, wsMAPPED))
    return;

  delete_window(w, &w->parent->kids);
  push_window(w, &w->parent->kids);
  w->status_mask |= bit(wsMAPPED);
  if (w_status_p(w->parent, wsVISIBLE)) {
    WINDOW pw;
    set_visible(w);
    pw = begin_update_ptr_window();
    paint_window(w);
    end_update_ptr_window(pw);
  }
  assert(window_hierarchy_ok_p());
}

/* Mark window w and its descendents invisible. */
LOCAL(void) set_invisible(WINDOW w)
{
  if (!w_status_p(w, wsMAPPED))
    return;
  if (w->kids != NULL) {
    WINDOW k, end;
    k = end = w->kids->prev;
    do {
      set_invisible(k);
    } while ((k = k->prev) != end);
  }
  w->status_mask &= notbit(wsVISIBLE);
  push_map_event(eUNMAP, w);
}

/* Unmap a window, clearing the area behind it. */
void unmap_window(WINDOW w)
{
  WINDOW pw;

  assert(w != root_window);
  if (w_status_p(w, wsVISIBLE)) {
    set_invisible(w);
    update_state_for_unmap();
    pw = begin_update_ptr_window(); 
    protect_cursor(w);
    if (w_status_p(w, wsSAVE_UNDER) && w->save != NULL) {
      push_graphics_state(root_window, 0);
      putimage(w->save->x0, w->save->y0, w->save->bits, COPY_PUT);
      free(w->save);
      w->save = NULL;
      pop_graphics_state();
    }
    else if (w->bg_color != TRANSPARENT) {
      int bw = w->border_width;
      push_graphics_state(w, 0);
      setfillstyle(SOLID_FILL, w->parent->bg_color);
      bar(-bw, -bw, w->width + bw - 1,  w->height + bw - 1);
      pop_graphics_state();
    }
    unprotect_cursor();
    w->status_mask &= notbit(wsMAPPED);
    end_update_ptr_window(pw);
  }
  assert(window_hierarchy_ok_p());
}

/* Free window storage and recursively
   all of its children's. */
LOCAL(void) free_window(WINDOW w)
{
  if (w->kids != NULL) {
    WINDOW k = w->kids;
    do {
      free_window(k);
    } while ((k = k->next) != w->kids);
  }
  w->id = DEAD;

  assert(w->save == NULL);

  if (w_status_p(w, wsDYNAMIC))
    free(w);
  assert(window_hierarchy_ok_p());
}

/* Destroy a window and all its children. */
void destroy_window(WINDOW w)
{
  assert(w->id != DEAD);
  unmap_window(w);
  delete_window(w, &w->parent->kids);
  free_window(w);
}

/* Walk a window subtree moving all windows
   by a relative amount. */
LOCAL(void) move_window(WINDOW w, int dx, int dy)
{
  w->x += dx;
  w->y += dy;
  if (w->kids != NULL) {
    WINDOW k = w->kids;
    do {
      move_window(k, dx, dy);
    } while ((k = k->next) != w->kids);
  }
  assert(window_hierarchy_ok_p());
}


/* Locate a window.  If the location is
   changed while the window is mapped,
   erase and repaint it.  The location
   is modified if necessary to ensure
   the new location is inside the parent,
   with the border optionally inside the
   parent. */
void locate_window(WINDOW w, int x, int y, int protect_border_p)
{
  int dx, dy, bw, do_paint_p;
  WINDOW parent;

  assert(w != root_window);
  parent = w->parent;
  if (x == CENTER_WINDOW)
    x = (parent->width - w->width) / 2;
  if (y == CENTER_WINDOW)
    y = (parent->height - w->height) / 2;
  do_paint_p = w_status_p(w, wsVISIBLE);
  bw = protect_border_p ? w->border_width : 0;
  if (x < bw)
    x = bw;
  if (y < bw)
    y = bw;
  dx = x + w->width + bw - parent->width;
  if (dx > 0)
    x -= dx;
  dy = y + w->height + bw - parent->height;
  if (dy > 0)
    y -= dy;
  dx = x + parent->x - w->x;
  dy = y + parent->y - w->y;
  if (dx || dy) {
    if (do_paint_p)
      unmap_window(w);
    move_window(w, dx, dy);
    if (do_paint_p)
      map_window(w);
  }
  assert(window_hierarchy_ok_p());
}

/* Set the background color of a window.
   It it's currently visible and there's
   a change, repaint it. */
int set_window_bg_color(WINDOW w, int color, int map_event_p)
{
  if (color == w->bg_color)
    return 0;
  w->bg_color = color;
  if (w_status_p(w, wsVISIBLE))
    clear_window(w, map_event_p);
  assert(window_hierarchy_ok_p());
  return 1;
}

/* Build up the root window. */
WINDOW make_root_window(int x0, int y0, int x1, int y1)
{
  WINDOW w = malloc(sizeof(WINDOW_REC));
  w->id = 0;
  w->x = x0;
  w->y = y0;
  w->width = x1 - x0 + 1;
  w->height = y1 - y0 + 1;
  w->parent = NULL;
  w->bg_color = root_bg_color;
  w->border_width = 0;
  w->event_mask = 0;
  w->prev = w->next = w;
  w->kids = NULL;
  w->status_mask = bit(wsMAPPED)|bit(wsVISIBLE);
  w->save = NULL;
  paint_window(w);
  return(w);
}

