// Copyright 2001 by
// DANTE e.V. and any individual authors listed elsewhere in this file. 
// 
// This file is part of the NTS system.
// ------------------------------------
// 
// It may be distributed and/or modified under the
// conditions of the NTS Public License (NTSPL), either version 1.0
// of this license or (at your option) any later version.
// The latest version of this license is in
//    http://www.dante.de/projects/nts/ntspl.txt
// and version 1.0 or later is part of all distributions of NTS 
// version 1.0-beta or later.
// 
// The list of all files belonging to the NTS distribution is given in
// the file `manifest.txt'. 
// 
// Filename: nts/tex/TeXErrorPool.java
// $Id: TeXErrorPool.java,v 1.1.1.1 2000/10/24 14:14:41 ksk Exp $
package	nts.tex;

import	java.util.HashMap;
import	java.util.Iterator;
import	nts.io.Log;
import	nts.io.Loggable;
import	nts.command.TokenList;

public class	TeXErrorPool {

    private HashMap		pool = new HashMap(199);

    public TeXError	get(String ident) {
        Object		err = pool.get(ident);
	if (err != null) return (TeXError) err;
	throw new RuntimeException("TeX error `" + ident
				   + "' not available in TeXErrorPool");
    }

    private void	put(String ident, TeXError err) {
        if (pool.containsKey(ident))
	    throw new RuntimeException("Redefinition of TeX error `" + ident
				       + "' in TeXErrorPool");
	pool.put(ident, err);
    }

    public Iterator	idents() { return pool.keySet().iterator(); }

    /* TeXtp[1283] */
    private String	errMessageHelp =
	"This error message was generated by an \\errmessage\n"
	+ "command, so I can't give any explicit help.\n"
	+ "Pretend that you're Hercule Poirot: Examine all clues,\n"
	+ "and deduce the truth by order and method.";

    public TeXError	get(final TokenList text, final TokenList help) {
	return new TeXError() {

	    public void		addText(Log log, Loggable[] params)
		{ log.add(text); }

	    public void		addHelp(Log log, Loggable[] params) {
		if (!help.isEmpty()) log.add(help);
	        else { print(log, errMessageHelp, params); log.endLine(); }
	    }

	    public void		addDesc(Log log, Loggable[] params) {
		if (!help.isEmpty()) log.endLine().add(help);
		else {
		    print(log.startLine(), errMessageHelp, params);
		    errMessageHelp = "(That was another \\errmessage.)";
		}
	    }

	};
    }

    private String	defaultHelp =
	"Sorry, I don't know how to help in this situation.\n"
	+ "Maybe you should try asking a human?";

    private void	defineError(String ident, final String text,
						  final String help) {
	put(ident, new TeXError() {

		public void	addText(Log log, Loggable[] params)
		    { if (text != null) print(log, text, params); }

		public void	addHelp(Log log, Loggable[] params) {
		    if (help != null) print(log, help, params);
		    else print(log, defaultHelp, null);
		    log.endLine();
		}

		public void	addDesc(Log log, Loggable[] params)
		    { if (help != null) print(log.startLine(), help, params); }

	    } );
    }

    private void	defineFatalError(String ident, final String desc) {
	put(ident, new TeXError() {

		public void	addText(Log log, Loggable[] params)
		    { log.add("Emergency stop"); }

		public void	addHelp(Log log, Loggable[] params)
		    { print(log, defaultHelp, null); log.endLine(); }

		public void	addDesc(Log log, Loggable[] params)
		    { log.startLine().add(desc); }

	    } );
    }

    /* TeXtp[89] */
    private String	helpAfterHelp =
	"Sorry, I already gave what help I could...\n"
	+ "Maybe you should try asking a human?\n"
	+ "An error might have occurred before I noticed any problems.\n"
	+ "``If all else fails, read the instructions.''";

    public void		addHelpAfterHelp(Log log)
	{ print(log, helpAfterHelp, null); log.endLine(); }

    /* TeXtp[88] */
    private String	helpAfterDel =
	"I have just deleted some text, as you asked.\n"
	+ "You can now delete more, or insert, or whatever.";

    public void		addHelpAfterDel(Log log)
	{ print(log, helpAfterDel, null); log.endLine(); }

    private void	print(Log log, String str, Loggable[] params) {
        for (int i = 0; i < str.length();) {
	    char	ch = str.charAt(i++);
	    if (ch == '\n') log.endLine();
	    else if (ch != '%') log.add(ch);
	    else if (str.charAt(i) == '%') log.add(str.charAt(i++));
	    else {
		int	j = i;
		while (  j < str.length()
		      && Character.isDigit(str.charAt(j))  ) j++;
		if (j > i) {
		    int		n = Integer.parseInt(str.substring(i, j));
		    if (params != null && 0 < n && n <= params.length)
			log.add(params[n - 1]);
		    else throw new RuntimeException("Error parameter %" + n
				    + " not specified (for:\n" + str + ')');
		    i = j;
		}
	    }
	}
    }

    {

	/* Void()						  */
	defineError("Void", null, null);

	/* Confusion(String)				TeXtp[95] */
	defineError("Confusion",
	  "This can't happen (%1)",
	  "I'm broken. Please show this to someone who can fix can fix"
	);

	/* NextConfusion()				TeXtp[95] */
	defineError("NextConfusion",
	  "I can't go on meeting you like this",
	  "One of your faux pas seems to have wounded me deeply...\n"
	  + "in fact, I'm barely conscious. Please fix it and try again."
	);

	/* Interrupt()					TeXtp[98] */
	defineError("Interrupt",
	  "Interruption",
	  "You rang?\n"
	  + "Try to insert some instructions for me (e.g.,`I\\showlists'),\n"
	  + "unless you just want to quit by typing `X'."
	);

	/* IncompatMag(number, number)			TeXtp[288] */
	defineError("IncompatMag",
	  "Incompatible magnification (%1);\n"
	  + " the previous value will be retained (%2)",
	  "I can handle only one magnification ratio per job. So I've\n"
	  + "reverted to the magnification you used earlier on this run."
	);

	/* IllegalMag(number)				TeXtp[288,1258] */
	defineError("IllegalMag",
	  "Illegal magnification has been changed to 1000 (%1)",	//XXX
	  "The magnification ratio must be between 1 and 32768."	//XXX
	);

    }

    private void	defineInSkipped(String ident, String what) {
	defineError(ident,
	  "Incomplete %1; all text was ignored after line %2",
	  what + " text.\n"
	  + "This kind of error happens when you say `\\if...' and forget\n"
	  + "the matching `\\fi'. I've inserted a `\\fi'; this might work."
	);
    }


    {

	/* OuterInSkipped(Command, line)		TeXtp[336] */
	defineInSkipped("OuterInSkipped",
	    "A forbidden control sequence occurred in skipped"
	);

	/* EOFinSkipped(Command, line)			TeXtp[336] */
	defineInSkipped("EOFinSkipped",
	    "The file ended while I was skipping conditional"
	);

    }

    private void	defineRunaway(String ident, String w1, String w2) {
	defineError(ident,
	  w1 + " while scanning " + w2 + " of %1",
	  "I suspect you have forgotten a `}', causing me\n"
	  + "to read past where you wanted me to stop.\n"
	  + "I'll try to recover; but if the error is serious,\n"
	  + "you'd better type `E' or `X' now and fix your file."
	);
    }

    {

	/* OuterInDef(Token)				TeXtp[338] */
	defineRunaway("OuterInDef",
	    "Forbidden control sequence found", "definition"
	);

	/* EOFinDef(Token)				TeXtp[338] */
	defineRunaway("EOFinDef",
	    "File ended", "definition"
	);

	/* OuterInMatch(Token)				TeXtp[338] */
	defineRunaway("OuterInMatch",
	    "Forbidden control sequence found", "use"
	);

	/* EOFinMatch(Token)				TeXtp[338] */
	defineRunaway("EOFinMatch",
	    "File ended", "use"
	);

	/* OuterInPreamble(Token)			TeXtp[338] */
	defineRunaway("OuterInPreamble",
	    "Forbidden control sequence found", "preamble"
	);

	/* EOFinPreamble(Token)				TeXtp[338] */
	defineRunaway("EOFinPreamble",
	    "File ended", "preamble"
	);

	/* OuterInToks(Token)				TeXtp[338] */
	defineRunaway("OuterInToks",
	    "Forbidden control sequence found", "text"
	);

	/* EOFinToks(Token)				TeXtp[338] */
	defineRunaway("EOFinToks",
	    "File ended", "text"
	);

	/* InvalidChar()				TeXtp[346] */
	defineError("InvalidChar",
	  "Text line contains an invalid character",
	  "A funny symbol that I can't read has just been input.\n"
	  + "Continue, and I'll forget that it ever happened."
	);

	/* UndefinedToken()				TeXtp[370] */
	defineError("UndefinedToken",
	  "Undefined control sequence",
	  "The control sequence at the end of the top line\n"
	  + "of your error message was never \\def'ed. If you have\n"
	  + "misspelled it (e.g., `\\hobx'), type `I' and the correct\n"
	  + "spelling (e.g., `I\\hbox'). Otherwise just continue,\n"
	  + "and I'll forget about whatever was undefined."
	);

	/* MissingEndcsname(escape)			TeXtp[373] */
	defineError("MissingEndcsname",
	  "Missing %1 inserted",
	  "The control sequence marked <to be read again> should\n"
	  + "not appear between \\csname and \\endcsname."
	);

	/* ExtraRightBrace(Token)			TeXtp[395] */
	defineError("ExtraRightBrace",
	  "Argument of %1 has an extra }",
	  "I've run across a `}' that doesn't seem to match anything.\n"
	  + "For example, `\\def\\a#1{...}' and `\\a}' would produce\n"
	  + "this error. If you simply proceed now, the `\\par' that\n"
	  + "I've just inserted will cause me to report a runaway\n"
	  + "argument that might be the root of the problem. But if\n"
	  + "your `}' was spurious, just type `2' and it will go away."
	);

	/* RunawayArg(Token)				TeXtp[396] */
	defineError("RunawayArg",
	  "Paragraph ended before %1 was complete",
	  "I suspect you've forgotten a `}', causing me to apply this\n"
	  + "control sequence to too much text. How can we recover?\n"
	  + "My plan is to forget the whole thing and hope for the best."
	);

	/* UseDoesntMatch(Token)			TeXtp[398] */
	defineError("UseDoesntMatch",
	  "Use of %1 doesn't match its definition",
	  "If you say, e.g., `\\def\\a1{...}', then you must always\n"
	  + "put `1' after `\\a', since control sequence names are\n"
	  + "made up of letters only. The macro here has not been\n"
	  + "followed by the required stuff, so I'm ignoring it."
	);

	/* MissingLeftBrace()				TeXtp[403] */
	defineError("MissingLeftBrace",
	  "Missing { inserted",
	  "A left brace was mandatory here, so I've put one in.\n"
	  + "You might want to delete and/or insert some corrections\n"
	  + "so that I will find a matching right brace soon.\n"
	  + "(If you're confused by all this, try typing `I}' now.)"
	);

	/* MixedGlueUnits()				TeXtp[408] */
	defineError("MixedGlueUnits",
	  "Incompatible glue units",
	  "I'm going to assume that 1mu=1pt when they're mixed."
	);

	/* MissingNumber()				TeXtp[415,446] */
	defineError("MissingNumber",
	  "Missing number, treated as zero",
	  "A number should have been here; I inserted `0'.\n"	//XXX
	  + "(If you can't figure out why I needed to see a number,\n"
	  + "look up `weird error' in the index to The TeXbook.)"
	);

	/* ImproperSForPD(Command)			TeXtp[418] */
	defineError("ImproperSForPD",
	  "Improper %1",
	  "You can refer to \\spacefactor only in horizontal mode;\n"
	  + "you can refer to \\prevdepth only in vertical mode; and\n"
	  + "neither of these is meaningful inside \\write. So\n"
	  + "I'm forgetting what you said and using zero instead."
	);

	/* CantAfterThe(Command, Command)		TeXtp[428] */
	defineError("CantAfterThe",
	  "You can't use `%1' after %2",
	  "I'm forgetting what you said and using zero instead."
	);

	/* BadRegister(number, number, number)		TeXtp[433] */
	defineError("BadRegister",
	  "Bad register code (%1)",
	  "A register number must be between %2 and %3.\n"
	  + "I changed this one to zero."
	);

	/* BadCharCode(number, number, number)		TeXtp[434] */
	defineError("BadCharCode",
	  "Bad character code (%1)",
	  "A character number must be between %2 and %3.\n"
	  + "I changed this one to zero."
	);

	/* BadFourBit(number, number, number)		TeXtp[435] */
	defineError("BadFileNum",
	  "Bad number (%1)",
	  "Since I expected to read a number between %2 and %3,\n"
	  + "I changed this one to zero."
	);

	/* BadMathcharCode(number, number, number)	TeXtp[436] */
	defineError("BadMathCharCode",
	  "Bad mathchar (%1)",
	  "A mathchar number must be between %2 and %3.\n"
	  + "I changed this one to zero."
	);

	/* BadDelimiterCode(number, number)			TeXtp[437] */
	defineError("BadDelimiterCode",
	  "Bad delimiter code (%1)",
	  "A numeric delimiter code must be between %2 and 2^{27}-1.\n"	//XXX
	  + "I changed this one to zero."
	);

	/* NonNumericToken()				TeXtp[442] */
	defineError("NonNumericToken",
	  "Improper alphabetic constant",
	  "A one-character control sequence belongs after a ` mark.\n"
	  + "So I'm essentially inserting \\0 here."
	);

	/* NumberTooBig()				TeXtp[445] */
	defineError("NumberTooBig",
	  "Number too big",
	  "I can only go up to 2147483647='17777777777=\"7FFFFFFF,\n"	//XXX
	  + "so I'm using that number instead of yours."
	);

	/* IllegalFil()					TeXtp[454] */
	defineError("IllegalFil",
	  "Illegal unit of measure (replaced by filll)",
	  "I dddon't go any higher than filll."
	);

	/* IllegalMu()					TeXtp[456] */
	defineError("IllegalMu",
	  "Illegal unit of measure (mu inserted)",
	  "The unit of measurement in math glue must be mu.\n"
	  + "To recover gracefully from this error, it's best to\n"
	  + "delete the erroneous units; e.g., type `2' to delete\n"
	  + "two letters. (See Chapter 27 of The TeXbook.)"
	);

	/* IllegalUnit()				TeXtp[459] */
	defineError("IllegalUnit",
	  "Illegal unit of measure (pt inserted)",
	  "Dimensions can be in units of em, ex, in, pt, pc,\n"
	  + "cm, mm, dd, cc, bp, or sp; but yours is a new one!\n"
	  + "I'll assume that you meant to say pt, for printer's points.\n"
	  + "To recover gracefully from this error, it's best to\n"
	  + "delete the erroneous units; e.g., type `2' to delete\n"
	  + "two letters. (See Chapter 27 of The TeXbook.)"
	);

	/* DimenTooLarge()				TeXtp[460] */
	defineError("DimenTooLarge",
	  "Dimension too large",
	  "I can't work with sizes bigger than about 19 feet.\n"
	  + "Continue and I'll use the largest value I can."
	);

	/* MissingLeftDefBrace()			TeXtp[475] */
	defineError("MissingLeftDefBrace",
	  "Missing { inserted",
	  "Where was the left brace? You said something like `\\def\\a}',\n"
	  + "which I'm going to interpret as `\\def\\a{}'."
	);

	/* TooManyParams()				TeXtp[476] */
	defineError("TooManyParams",
	  "You already have nine parameters",
	  "I'm going to ignore the # sign you just used."
	);

	/* NonConseqParams()				TeXtp[476] */
	defineError("NonConseqParams",
	  "Parameters must be numbered consecutively",
	  "I've inserted the digit you should have used after the #.\n"
	  + "Type `1' to delete what you did use."
	);

	/* IllegalParamNum(Token)			TeXtp[479] */
	defineError("IllegalParamNum",
	  "Illegal parameter number in definition of %1",
	  "You meant to type ## instead of #, right?\n"
	  + "Or maybe a } was forgotten somewhere earlier, and things\n"
	  + "are all screwed up? I'm going to assume that you meant ##."
	);

	/* EOFinRead(escape)				TeXtp[486] */
	defineError("EOFinRead",
	  "File ended within %1",
	  "This \\read has unbalanced braces."
	);

	/* ExtraOrElseFi(escape | Command)		TeXtp[500,510] */
	defineError("ExtraOrElseFi",
	  "Extra %1",
	  "I'm ignoring this; it doesn't match any \\if."
	);

	/* IllegalIfnumOp(Command)			TeXtp[503] */
	defineError("IllegalIfnumOp",
	  "Missing = inserted for %1",
	  "I was expecting to see `<', `=', or `>'. Didn't."
	);

    }

    private void	defineBadTFM(String ident, String what) {
	defineError(ident,
	  "Font %1 not loadable: " + what,
	  "I wasn't able to read the size data for this font,\n"
	  + "so I will ignore the font specification.\n"
	  + "[Wizards can fix TFM files using TFtoPL/PLtoTF.]\n"
	  + "You might try inserting a different font spec;\n"
	  + "e.g., type `I\\font<same font id>=<substitute font name>'."
	);
    }

    {

	/* TFMisBad(Font)				TeXtp[561] */
	defineBadTFM("TFMisBad",
	    "Bad metric (TFM) file"
	);

	/* TFMnotFound(Font)				TeXtp[561] */
	defineBadTFM("TFMnotFound",
	    "Metric (TFM) file not found"
	);

	/* NotEnoughRoomForTFM(Font)			TeXtp[567] */
	defineError("NotEnoughRoomForTFM",
	  "Font %1 not loaded: Not enough room left",
	  "I'm afraid I won't be able to make use of this font,\n"
	  + "because my memory for character-size data is too small.\n"
	  + "If you're really stuck, ask a wizard to enlarge me.\n"
	  + "Or maybe try `I\\font<same font id>=<name of loaded font>'."
	);

	/* MissingFontIdent()				TeXtp[577] */
	defineError("MissingFontIdent",
	  "Missing font identifier",
	  "I was looking for a control sequence whose\n"
	  + "current meaning has been defined by \\font."
	);

	/* TooBigFontdimenNum(escape, number)		TeXtp[579] */
	defineError("TooBigFontdimenNum",
	  "Font %1 has only %2 fontdimen parameters",
	  "To increase the number of font parameters, you must\n"
	  + "use \\fontdimen immediately after the \\font is loaded."
	);

	/* PageTooLarge()				TeXtp[641] */
	defineError("PageTooLarge",
	  "Huge page cannot be shipped out",
	  "The page just created is more than 18 feet tall or\n"
	  + "more than 18 feet wide, so I suspect something went wrong."
	);

	/* UndefFamily(escape, number, ascii)		TeXtp[723] */
	defineError("UndefFamily",
	  "%1 %2 is undefined (character %3)",
	  "Somewhere in the math formula just ended, you used the\n"
	  + "stated character from an undefined font family. For example,\n"
	  + "plain TeX doesn't allow \\it or \\sl in subscripts. Proceed,\n"
	  + "and I'll try to forget that I needed that character."
	);

	/* ImproperAlignInFormula(escape)		TeXtp[776] */
	defineError("ImproperAlignInFormula",
	  "Improper %1 inside $$'s",
	  "Displays can use special alignments (like \\eqalignno)\n"
	  + "only if nothing but the alignment itself is between $$'s.\n"
	  + "So I've deleted the formulas that preceded this alignment."
	);

	/* MissingSharp()				TeXtp[783] */
	defineError("MissingSharp",
	  "Missing # inserted in alignment preamble",
	  "There should be exactly one # between &'s, when an\n"
	  + "\\halign or \\valign is being set up. In this case you had\n"
	  + "none, so I've put one in; maybe that will work."
	);

	/* SecondSharpInTab()				TeXtp[784] */
	defineError("SecondSharpInTab",
	  "Only one # is allowed per tab",
	  "There should be exactly one # between &'s, when an\n"
	  + "\\halign or \\valign is being set up. In this case you had\n"
	  + "more than one, so I'm ignoring all but the first."
	);

	/* ExtraAlignTab(escape)			TeXtp[792] */
	defineError("ExtraAlignTab",
	  "Extra alignment tab has been changed to %1",
	  "You have given more \\span or & marks than there were\n"
	  + "in the preamble to the \\halign or \\valign now in progress.\n"
	  + "So I'll assume that you meant to type \\cr instead."
	);

	/* InfShringInPar()				TeXtp[826] */
	defineError("InfShringInPar",
	  "Infinite glue shrinkage found in a paragraph",
	  "The paragraph just ended includes some glue that has\n"
	  + "infinite shrinkability, e.g., `\\hskip 0pt minus 1fil'.\n"
	  + "Such glue doesn't belong there---it allows a paragraph\n"
	  + "of any length to fit on one line. But it's safe to proceed,\n"
	  + "since the offensive shrinkability has been made finite."
	);

	/* ImproperHyphen(escape)			TeXtp[936] */
	defineError("ImproperHyphen",
	  "Improper %1 will be flushed",
	  "Hyphenation exceptions must contain only letters\n"
	  + "and hyphens. But continue; I'll forgive and forget."
	);

	/* NonLetterInHyph()				TeXtp[937] */
	defineError("NonLetterInHyph",
	  "Not a letter",
	  "Letters in \\hyphenation words must have \\lccode>0.\n"
	  + "Proceed; I'll ignore the character I just read."
	);

	/* LatePatterns(escape)				TeXtp[960] */
	defineError("LatePatterns",
	  "Too late for %1",
	  "All patterns must be given before typesetting begins."
	);

	/* BadPatterns(escape)				TeXtp[961] */
	defineError("BadPatterns",
	  "Bad %1",
	  "(See Appendix H.)"
	);

	/* NonLetter()					TeXtp[962] */
	defineError("NonLetter",
	  "Nonletter",
	  "(See Appendix H.)"
	);

	/* DupPattern()					TeXtp[963] */
	defineError("DupPattern",
	  "Duplicate pattern",
	  "(See Appendix H.)"
	);

	/* InfShrinkInVsplit()				TeXtp[976] */
	defineError("InfShrinkInVsplit",
	  "Infinite glue shrinkage found in box being split",
	  "The box you are \\vsplitting contains some infinitely\n"
	  + "shrinkable glue, e.g., `\\vss' or `\\vskip 0pt minus 1fil'.\n"
	  + "Such glue doesn't belong there; but you can safely proceed,\n"
	  + "since the offensive shrinkability has been made finite."
	);

	/* SplittingNonVbox(Command, escape)		TeXtp[978] */
	defineError("SplittingNonVbox",
	  "%1 needs a %2",
	  "The box you are trying to split is an \\hbox.\n"
	  + "I can't split such a box, so I'll leave it alone."
	);

	/* MisplacedInsert()				TeXtp[993] */
	defineError("MisplacedInsert",
	  "Insertions can only be added to a vbox",
	  "Tut tut: You're trying to \\insert into a\n"
	  + "\\box register that now contains an \\hbox.\n"
	  + "Proceed, and I'll discard its present contents."
	);

	/* InfShrinkInPage()				TeXtp[1004] */
	defineError("InfShrinkInPage",
	  "Infinite glue shrinkage found on current page",
	  "The page about to be output contains some infinitely\n"
	  + "shrinkable glue, e.g., `\\vss' or `\\vskip 0pt minus 1fil'.\n"
	  + "Such glue doesn't belong there; but you can safely proceed,\n"
	  + "since the offensive shrinkability has been made finite."
	);

	/* InfShrinkInsert(escape, number)		TeXtp[1009] */
	defineError("InfShrinkInsert",
	  "Infinite glue shrinkage inserted from %1%2",
	  "The correction glue for page breaking with insertions\n"
	  + "must have finite shrinkability. But you may proceed,\n"
	  + "since the offensive shrinkability has been made finite."
	);

	/* NonEmptyOutbox(escape, number)		TeXtp[1015] */
	defineError("NonEmptyOutBox",
	  "%1%2 is not void",
	  "You shouldn't use \\box255 except in \\output routines.\n"
	  + "Proceed, and I'll discard its present contents."
	);

	/* TooMuchDead(number)				TeXtp[1024] */
	defineError("TooMuchDead",
	  "Output loop---%1 consecutive dead cycles",
	  "I've concluded that your \\output is awry; it never does a\n"
	  + "\\shipout, so I'm shipping \\box255 out myself. Next time\n"
	  + "increase \\maxdeadcycles if you want me to be more patient!"
	);

	/* UnbalancedOutput()				TeXtp[1027] */
	defineError("UnbalancedOutput",
	  "Unbalanced output routine",
	  "Your sneaky output routine has problematic {'s and/or }'s.\n"
	  + "I can't handle that very well; good luck."
	);

	/* NonEmptyOutboxAfter(escape, number)		TeXtp[1028] */
	defineError("NonEmptyOutBoxAfter",
	  "Output routine didn't use all of %1%2",
	  "Your \\output commands should empty \\box255,\n"
	  + "e.g., by saying `\\shipout\\box255'.\n"
	  + "Proceed; I'll discard its present contents."
	);

	/* MissingDollar()				TeXtp[1047] */
	defineError("MissingDollar",
	  "Missing $ inserted",
	  "I've inserted a begin-math/end-math symbol since I think\n"
	  + "you left one out. Proceed, with fingers crossed."
	);

	/* CantUseIn(Command, Mode)			TeXtp[1050] */
	defineError("CantUseIn",
	  "You can't use `%1' in %2",
	  "Sorry, but I'm not programmed to handle this case;\n"
	  + "I'll just pretend that you didn't ask for it.\n"
	  + "If you're in the wrong mode, you might be able to\n"
	  + "return to the right one by typing `I}' or `I$' or `I\\par'."
	);

	/* MissingInserted(escape | char)		TeXtp[1064] */
	defineError("MissingInserted",
	  "Missing %1 inserted",
	  "I've inserted something that you may have forgotten.\n"
	  + "(See the <inserted text> above.)\n"
	  + "With luck, this will get me unwedged. But if you\n"
	  + "really didn't forget anything, try typing `2' now; then\n"
	  + "my insertion and my current dilemma will both disappear."
	);

	/* ExtraToken(Command)				TeXtp[1066] */
	defineError("ExtraCommand",
	  "Extra %1",
	  "Things are pretty mixed up, but I think the worst is over."
	);

	/* TooManyRightBraces()				TeXtp[1068] */
	defineError("TooManyRightBraces",
	  "Too many }'s",
	  "You've closed more groups than you opened.\n"
	  + "Such booboos are generally harmless, so keep going."
	);

	/* ExtraOrForgotten(escape | char)		TeXtp[1069] */
	defineError("ExtraOrForgotten",
	  "Extra }, or forgotten %1",
	  "I've deleted a group-closing symbol because it seems to be\n"
	  + "spurious, as in `$x}$'. But perhaps the } is legitimate and\n"
	  + "you forgot something else, as in `\\hbox{$x}'. In such cases\n"
	  + "the way to recover is to insert both the forgotten and the\n"
	  + "deleted material, e.g., by typing `I$}'."
	);

	/* BadGlueAfterLeaders()			TeXtp[1078] */
	defineError("BadGlueAfterLeaders",
	  "Leaders not followed by proper glue",
	  "You should say `\\leaders <box or rule><hskip or vskip>'.\n"
	  + "I found the <box or rule>, but there's no suitable\n"
	  + "<hskip or vskip>, so I'm ignoring these leaders."
	);

	/* LastBoxIn(Command, Mode)			TeXtp[1080] */
	defineError("LastBoxIn",
	  "You can't use `%1' in %2",
	  "Sorry; this \\lastbox will be void.\n"
	);

	/* CantTakeFromPage(Command, Mode)		TeXtp[1080] */
	defineError("CantTakeFromPage",
	  "You can't use `%1' in %2",
	  "Sorry...I usually can't take things from the current page.\n"
	  + "This \\lastbox will therefore be void."
	);

	/* MissingToForVsplit()				TeXtp[1082] */
	defineError("MissingToForVsplit",
	  "Missing `to' inserted",
	  "I'm working on `\\vsplit<box number> to <dimen>';\n"
	  + "will look for the <dimen> next."
	);

	/* BoxExpected()				TeXtp[1084] */
	defineError("BoxExpected",
	  "A <box> was supposed to be here",
	  "I was expecting to see \\hbox or \\vbox or \\copy or \\box or\n"
	  + "something like that. So you might find something missing in\n"
	  + "your output. But keep trying; you can fix this later."
	);

	/* CantUseHrule(escape)				TeXtp[1095] */
	defineError("CantUseHrule",
	  "You can't use `%1' here except with leaders",
	  "To put a horizontal rule in an hbox or an alignment,\n"
	  + "you should use \\leaders or \\hrulefill (see The TeXbook)."
	);

	/* CantInsertOutbox(command, number, number)	TeXtp[1099] */
	defineError("CantInsertOutbox",
	  "You can't %1%2",
	  "I'm changing to %1%3; box %2 is special."
	);

    }

    private void	defineCantDeleteLastItem(String ident, String what) {
	defineError(ident,
	  "You can't use `%1' in %2",
	  "Sorry...I usually can't take things from the current page.\n"
	  + what + "."
	);
    }

    {

	/* CantDeleteLastSkip(Command, Mode)		TeXtp[1106] */
	defineCantDeleteLastItem("CantDeleteLastSkip",
	  "Try `I\\vskip-\\lastskip' instead"
	);

	/* CantDeleteLastKern(Command, Mode)		TeXtp[1106] */
	defineCantDeleteLastItem("CantDeleteLastKern",
	  "Try `I\\kern-\\lastkern' instead"
	);

	/* CantDeleteLastGlue(Command, Mode)		TeXtp[1106] */
	defineCantDeleteLastItem("CantDeleteLastPenalty",
	  "Perhaps you can make the output routine do it"
	);

	/* IncompatibleUnbox()				TeXtp[1110] */
	defineError("IncompatibleUnbox",
	  "Incompatible list can't be unboxed",
	  "Sorry, Pandora. (You sneaky devil.)\n"
	  + "I refuse to unbox an \\hbox in vertical mode or vice versa.\n"
	  + "And I can't open any boxes in math mode."
	);

	/* IllegalMathDisc(escape)			TeXtp[1120] */
	defineError("IllegalMathDisc",
	  "Illegal math %1",
	  "Sorry: The third part of a discretionary break must be\n"
	  + "empty, in math formulas. I had to delete your third part."
	);

	/* TooLongDisc()				TeXtp[1120] */
	defineError("TooLongDisc",
	  "Discretionary list is too long",
	  "Wow---I never thought anybody would tweak me here.\n"
	  + "You can't seriously need such a huge discretionary list?"
	);

	/* ImproperDisc()				TeXtp[1121] */
	defineError("ImproperDisc",
	  "Improper discretionary list",
	  "Discretionary lists must contain only boxes and kerns."
	);

    }

    private void	defineAllignError(String ident, String what) {
	defineError(ident,
	  "Missing " + what + " inserted",
	  "I've put in what seems to be necessary to fix\n"
	  + "the current column of the current alignment.\n"
	  + "Try to go on, since this might almost work."
	);
    }

    {

	/* AllignLeftError()				TeXtp[1127] */
	defineAllignError("AllignLeftError",
	    "{"
	);

	/* AllignRightError()				TeXtp[1127] */
	defineAllignError("AllignRightError",
	    "}"
	);

	/* MisplacedTabMark(Command)			TeXtp[1128] */
	defineError("MisplacedTabMark",
	  "Misplaced %1",
	  "I can't figure out why you would want to use a tab mark\n"
	  + "here. If you just want an ampersand, the remedy is\n"
	  + "simple: Just type `I\\&' now. But if some right brace\n"
	  + "up above has ended a previous alignment prematurely,\n"
	  + "you're probably due for more error messages, and you\n"
	  + "might try typing `S' now just to see what is salvageable."
	);

	/* MisplacedCrSpan(Command)			TeXtp[1128] */
	defineError("MisplacedCrSpan",
	  "Misplaced %1",
	  "I can't figure out why you would want to use a tab mark\n"
	  + "or \\cr or \\span just now. If something like a right brace\n"
	  + "up above has ended a previous alignment prematurely,\n"
	  + "you're probably due for more error messages, and you\n"
	  + "might try typing `S' now just to see what is salvageable."
	);

	/* MisplacedNoalign(escape)			TeXtp[1129] */
	defineError("MisplacedNoalign",
	  "Misplaced %1",
	  "I expect to see \\noalign only after the \\cr of\n"
	  + "an alignment. Proceed, and I'll ignore this case."
	);

	/* MisplacedOmit(escape)			TeXtp[1129] */
	defineError("MisplacedOmit",
	  "Misplaced %1",
	  "I expect to see \\omit only after tab marks or the \\cr of\n"
	  + "an alignment. Proceed, and I'll ignore this case."
	);

	/* MissingCr(escape)				TeXtp[1132] */
	defineError("MissingCr",
	  "Missing %1 inserted",
	  "I'm guessing that you meant to end an alignment here."
	);

	/* ExtraEndcsname(escape)			TeXtp[1135] */
	defineError("ExtraEndcsname",
	  "Extra %1",
	  "I'm ignoring this, since I wasn't doing a \\csname."
	);

	/* MisplacedLimits()				TeXtp[1159] */
	defineError("MisplacedLimits",
	  "Limit controls must follow a math operator",
	  "I'm ignoring this misplaced \\limits or \\nolimits command."
	);

	/* MissingDelim()				TeXtp[1161] */
	defineError("MissingDelim",
	  "Missing delimiter (. inserted)",
	  "I was expecting to see something like `(' or `\\{' or\n"
	  + "`\\}' here. If you typed, e.g., `{' instead of `\\{', you\n"
	  + "should probably delete the `{' by typing `1' now, so that\n"
	  + "braces don't get unbalanced. Otherwise just proceed.\n"
	  + "Acceptable delimiters are characters whose \\delcode is\n"
	  + "nonnegative, or you can use `\\delimiter <delimiter code>'."
	);

	/* AccentInMathMode(escape)			TeXtp[1166] */
	defineError("AccentInMathMode",
	  "Please use %1 for accents in math mode",
	  "I'm changing \\accent to \\mathaccent here; wish me luck.\n"
	  + "(Accents are not the same in formulas as they are in text.)"
	);

	/* DoubleSuperscript()				TeXtp[1177] */
	defineError("DoubleSuperscript",
	  "Double superscript",
	  "I treat `x^1^2' essentially like `x^1{}^2'."
	);

	/* DoubleSubscript()				TeXtp[1177] */
	defineError("DoubleSubscript",
	  "Double subscript",
	  "I treat `x_1_2' essentially like `x_1{}_2'."
	);

	/* AmbiguousFraction()				TeXtp[1183] */
	defineError("AmbiguousFraction",
	  "Ambiguous; you need another { and }",
	  "I'm ignoring this fraction specification, since I don't\n"
	  + "know whether a construction like `x \\over y \\over z'\n"
	  + "means `{x \\over y} \\over z' or `x \\over {y \\over z}'."
	);

	/* ExtraRight(Token)				TeXtp[1192] */
	defineError("ExtraRight",
	  "Extra %1",
	  "I'm ignoring a \\right that had no matching \\left."
	);

	/* InsufSymFonts()				TeXtp[1195] */
	defineError("InsufSymFonts",
	  "Math formula deleted: Insufficient symbol fonts",
	  "Sorry, but I can't typeset math unless \\textfont 2\n"
	  + "and \\scriptfont 2 and \\scriptscriptfont 2 have all\n"
	  + "the \\fontdimen values needed in math symbol fonts."
	);

	/* InsufExtFonts()				TeXtp[1195] */
	defineError("InsufExtFonts",
	  "Math formula deleted: Insufficient extension fonts",
	  "Sorry, but I can't typeset math unless \\textfont 3\n"
	  + "and \\scriptfont 3 and \\scriptscriptfont 3 have all\n"
	  + "the \\fontdimen values needed in math extension fonts."
	);

	/* BadFormulaEnd()				TeXtp[1197] */
	defineError("BadFormulaEnd",
	  "Display math should end with $$",
	  "The `$' that I just saw supposedly matches a previous `$$'.\n"
	  + "So I shall assume that you typed `$$' both times."
	);

	/* MissingFormulaEnd()				TeXtp[1207] */
	defineError("MissingFormulaEnd",
	  "Missing $$ inserted",
	  "Displays can use special alignments (like \\eqalignno)\n"
	  + "only if nothing but the alignment itself is between $$'s."
	);

	/* NonPrefixCommand(Command)			TeXtp[1212] */
	defineError("NonPrefixCommand",
	  "You can't use a prefix with `%1'",
	  "I'll pretend you didn't say \\long or \\outer or \\global."
	);

	/* NonDefineCommand(escape, escape, Command)	TeXtp[1213] */
	defineError("NonDefineCommand",
	  "You can't use `%1' or `%2' with `%3'",
	  "I'll pretend you didn't say \\long or \\outer here."
	);

	/* MissingCtrlSeq()				TeXtp[1215] */
	defineError("MissingCtrlSeq",
	  "Missing control sequence inserted",
	  "Please don't say `\\def cs{...}', say `\\def\\cs{...}'.\n"
	  + "I've inserted an inaccessible control sequence so that your\n"
	  + "definition will be completed without mixing me up too badly.\n"
	  + "You can recover graciously from this error, if you're\n"
	  + "careful; see exercise 27.2 in The TeXbook."
	);

	/* MissingToForRead()				TeXtp[1225] */
	defineError("MissingToForRead",
	  "Missing `to' inserted",
	  "You should have said `\\read<number> to \\cs'.\n"
	  + "I'm going to look for the \\cs now."
	);

    }

    private void	defineCodeOut(String ident, String what) {
	defineError(ident,
	  "Invalid code (%1), should be " + what + "%2",
	  "I'm going to use 0 instead of that illegal code value."	//XXX
	);
    }

    {

	/* CodeOutOfRange(number, number)		TeXtp[1232] */
	defineCodeOut("CodeOutOfRange",
	    "in the range 0.."	//XXX
	);

	/* CodeGreater(number, number)			TeXtp[1232] */
	defineCodeOut("CodeGreater",
	    "at most "
	);

	/* ArithOverflow()				TeXtp[1236] */
	defineError("ArithOverflow",
	  "Arithmetic overflow",
	  "I can't carry out that multiplication or division,\n"
	  + "since the result is out of range."
	);

	/* CantUseAfter(Command, Command)		TeXtp[1237] */
	defineError("CantUseAfter",
	  "You can't use `%1' after %2",
	  "I'm forgetting what you said and not changing anything."
	);

	/* ImproperSetbox(Command)			TeXtp[1241] */
	defineError("ImproperSetbox",
	  "Improper %1",
	  "Sorry, \\setbox is not allowed after \\halign in a display,\n"
	  + "or between \\accent and an accented character."
	);

	/* BadSpaceFactor(number, number, number)	TeXtp[1243] */
	defineError("BadSpaceFactor",
	  "Bad space factor (%1)",
	  "I allow only values in the range %2..%3 here."
	);

	/* BadPrevGraf(escape, number)			TeXtp[1244] */
	defineError("BadPrevGraf",
	  "Bad %1 (%2)",
	  "I allow only nonnegative values here."
	);

	/* CantLoadPatterns()				TeXtp[1252] */
	defineError("CantLoadPatterns",
	  "Patterns can be loaded only by INITEX",
	  null
	);

	/* ImproperAt(Dimen)				TeXtp[1259] */
	defineError("ImproperAt",
	  "Improper `at' size (%1pt), replaced by 10pt",	//XXX
	  "I can only handle fonts at positive sizes that are\n"
	  + "less than 2048pt, so I've changed what you said to 10pt."
	);

	/* ShortShow()					TeXtp[1293] */
	defineError("ShortShow", null,
	  "This isn't an error message; I'm just \\showing something.\n"
	  + "Type `I\\show...' to show more (e.g., \\show\\cs,\n"
	  + "\\showthe\\count10, \\showbox255, \\showlists)."
	);

	/* LongShow()					TeXtp[1293] */
	defineError("LongShow", null,
	  "This isn't an error message; I'm just \\showing something.\n"
	  + "Type `I\\show...' to show more (e.g., \\show\\cs,\n"
	  + "\\showthe\\count10, \\showbox255, \\showlists).\n"
	  + "And type `I\\tracingonline=1\\show...' to show boxes and\n"
	  + "lists on your terminal as well as in the transcript file."
	);

	/* DumpInGroup()				TeXtp[1304] */
	defineError("DumpInGroup",
	  "You can't dump inside a group",
	  "`{...\\dump}' is a no-no."
	);

	/* UnbalancedWrite()				TeXtp[1372] */
	defineError("UnbalancedWrite",
	  "Unbalanced write command",
	  "On this page there's a \\write with fewer real {'s than }'s.\n"
	  + "I can't handle that very well; good luck."
	);

	/* EOFonTerm()					TeXtp[71] */
	defineFatalError("EOFonTerm",
	    "End of file on the terminal!");

	/* NoEnd()					TeXtp[360] */
	defineFatalError("NoEnd",
	    "*** (job aborted, no legal \\end found)");

	/* NoTermRead()				TeXtp[484] */
	defineFatalError("NoTermRead",
	    "*** (cannot \\read from terminal in nonstop modes)");

	/* MissingFile()				TeXtp[530] */
	defineFatalError("MissingFile",
	    "*** (job aborted, file error in nonstop mode)");

	/* IntAlignPre()			TeXtp[324,782,789,791] */
	defineFatalError("IntAlignPre",
	    "(interwoven alignment preambles are not allowed)");

    }

    public static void		main(String[] args) {
        TeXErrorPool	pl = new TeXErrorPool();
        Iterator	id = pl.idents();
	while (id.hasNext())
	    System.out.println(id.next());
    }

}
