//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2024, 2026 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.bdd.conversion.preconditions;

import org.eclipse.escet.cif.checkers.CifPreconditionChecker;
import org.eclipse.escet.cif.checkers.checks.AutOnlySpecificSupKindsCheck;
import org.eclipse.escet.cif.checkers.checks.AutReqNoChannelCheck;
import org.eclipse.escet.cif.checkers.checks.EdgeNoMultiAssignCheck;
import org.eclipse.escet.cif.checkers.checks.EdgeNoPartialVarAssignCheck;
import org.eclipse.escet.cif.checkers.checks.EventOnlyWithControllabilityCheck;
import org.eclipse.escet.cif.checkers.checks.ExprNoSpecificExprsCheck;
import org.eclipse.escet.cif.checkers.checks.ExprNoSpecificExprsCheck.NoSpecificExpr;
import org.eclipse.escet.cif.checkers.checks.FuncNoSpecificIntUserDefFuncStatsCheck;
import org.eclipse.escet.cif.checkers.checks.FuncNoSpecificIntUserDefFuncStatsCheck.NoSpecificStatement;
import org.eclipse.escet.cif.checkers.checks.FuncNoSpecificUserDefCheck;
import org.eclipse.escet.cif.checkers.checks.FuncNoSpecificUserDefCheck.NoSpecificUserDefFunc;
import org.eclipse.escet.cif.checkers.checks.InvNoSpecificInvsCheck;
import org.eclipse.escet.cif.checkers.checks.SpecAutomataCountsCheck;
import org.eclipse.escet.cif.checkers.checks.TypeNoSpecificTypesCheck;
import org.eclipse.escet.cif.checkers.checks.TypeNoSpecificTypesCheck.NoSpecificType;
import org.eclipse.escet.cif.checkers.checks.VarNoContinuousCheck;
import org.eclipse.escet.cif.checkers.checks.invcheck.NoInvariantKind;
import org.eclipse.escet.cif.checkers.checks.invcheck.NoInvariantPlaceKind;
import org.eclipse.escet.cif.checkers.checks.invcheck.NoInvariantSupKind;
import org.eclipse.escet.cif.metamodel.cif.SupKind;
import org.eclipse.escet.common.java.Termination;

/** CIF to BDD conversion precondition checker. */
public class CifToBddConverterPreChecker extends CifPreconditionChecker {
    /**
     * Constructor for the {@link CifToBddConverterPreChecker} class.
     *
     * @param termination Cooperative termination query function.
     */
    public CifToBddConverterPreChecker(Termination termination) {
        super(termination,

                // Only plant and requirement automata are supported.
                new AutOnlySpecificSupKindsCheck(SupKind.PLANT, SupKind.REQUIREMENT),

                // Specifications without plant automata are not supported.
                new SpecAutomataCountsCheck().setMinMaxPlantAuts(1, SpecAutomataCountsCheck.NO_CHANGE),

                // Events not declared as controllable or uncontrollable are not supported.
                new EventOnlyWithControllabilityCheck(),

                // Requirement automata that are senders or receivers for channels are not supported.
                new AutReqNoChannelCheck(),

                // Only plant and requirement invariants are supported.
                new InvNoSpecificInvsCheck()
                        .disallow(NoInvariantSupKind.KINDLESS, NoInvariantKind.ALL_KINDS,
                                NoInvariantPlaceKind.ALL_PLACES)
                        .disallow(NoInvariantSupKind.SUPERVISOR, NoInvariantKind.ALL_KINDS,
                                NoInvariantPlaceKind.ALL_PLACES),

                // Continuous variables are not supported.
                new VarNoContinuousCheck(),

                // Only allow supported types for places where users enter them, for supported language features, such
                // as types of variables.
                new CifToBddVarOnlySpecificTypesCheck(),

                // Only allow supported expressions (including predicates), for all places where expressions in
                // supported language features.
                new CifToBddExprOnlySupportedExprsCheck(),

                // Multi-assignments and partial variable assignments are not supported.
                new EdgeNoMultiAssignCheck(),
                new EdgeNoPartialVarAssignCheck(),

                // No external user-defined functions. Internal user-defined functions must not have direct or indirect
                // recursive function calls, nor multiple return values.
                new FuncNoSpecificUserDefCheck(
                        NoSpecificUserDefFunc.EXTERNAL,
                        NoSpecificUserDefFunc.INTERNAL_RECURSIVE)
                                .internalFuncReturnValuesLimits(null, 1),

                // No functions as data.
                new ExprNoSpecificExprsCheck(NoSpecificExpr.FUNC_REFS_USER_DEF_AS_DATA),
                new TypeNoSpecificTypesCheck(NoSpecificType.FUNC_TYPES_AS_DATA),

                // No assignments in internal user-defined functions that may result in out-of-range errors. Also no
                // 'while' statements in them. Implies no 'break' and 'continue' statements.
                new FuncNoSpecificIntUserDefFuncStatsCheck(
                        NoSpecificStatement.ASSIGN_OUT_OF_RANGE,
                        NoSpecificStatement.WHILE)

        );
    }
}
