//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2010, 2026 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.bdd.varorder.hyperedges;

import static org.eclipse.escet.common.java.Sets.set;

import java.util.Set;

import org.eclipse.escet.cif.common.CifEquationUtils;
import org.eclipse.escet.cif.common.CifLocationUtils;
import org.eclipse.escet.cif.metamodel.cif.automata.Automaton;
import org.eclipse.escet.cif.metamodel.cif.automata.Location;
import org.eclipse.escet.cif.metamodel.cif.automata.Update;
import org.eclipse.escet.cif.metamodel.cif.declarations.AlgVariable;
import org.eclipse.escet.cif.metamodel.cif.expressions.AlgVariableExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.DiscVariableExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.Expression;
import org.eclipse.escet.cif.metamodel.cif.expressions.InputVariableExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.LocationExpression;
import org.eclipse.escet.cif.metamodel.java.CifWithArgWalker;
import org.eclipse.escet.common.position.metamodel.position.PositionObject;

/**
 * CIF variable object collector.
 *
 * <p>
 * Each referenced algebraic variable is only considered once.
 * </p>
 */
class VariableCollector extends CifWithArgWalker<Set<PositionObject>> {
    /** The already considered objects that don't have to be considered again. */
    private final Set<AlgVariable> alreadyConsidered = set();

    /**
     * Collect CIF variable objects in the given update, recursively.
     *
     * @param update The update.
     * @param cifVarObjs The CIF variable objects collected so far. Is extended in-place.
     * @return The collected CIF variable objects, i.e., {@code cifVarObjs}.
     */
    Set<PositionObject> collectCifVarObjs(Update update, Set<PositionObject> cifVarObjs) {
        walkUpdate(update, cifVarObjs);
        return cifVarObjs;
    }

    /**
     * Collect CIF variable objects in the given expression, recursively.
     *
     * @param expr The expression.
     * @param cifVarObjs The CIF variable objects collected so far. Is extended in-place.
     * @return The collected CIF variable objects, i.e., {@code cifVarObjs}.
     */
    Set<PositionObject> collectCifVarObjs(Expression expr, Set<PositionObject> cifVarObjs) {
        walkExpression(expr, cifVarObjs);
        return cifVarObjs;
    }

    @Override
    protected void preprocessDiscVariableExpression(DiscVariableExpression expr, Set<PositionObject> cifVarObjs) {
        cifVarObjs.add(expr.getVariable());
    }

    @Override
    protected void preprocessInputVariableExpression(InputVariableExpression expr, Set<PositionObject> cifVarObjs) {
        cifVarObjs.add(expr.getVariable());
    }

    @Override
    protected void preprocessLocationExpression(LocationExpression expr, Set<PositionObject> cifVarObjs) {
        // Only add automaton if location pointer variable will be created for it.
        Location loc = expr.getLocation();
        Automaton aut = CifLocationUtils.getAutomaton(loc);
        if (aut.getLocations().size() > 1) {
            cifVarObjs.add(aut);
        }
    }

    @Override
    protected void preprocessAlgVariableExpression(AlgVariableExpression expr, Set<PositionObject> cifVarObjs) {
        // Skip already processed algebraic variables.
        AlgVariable algVar = expr.getVariable();
        if (alreadyConsidered.contains(algVar)) {
            return;
        }
        alreadyConsidered.add(algVar);

        // Obtain single value expression, to get 'if' expression over locations, if equations per location are
        // used. That way, the location pointer variable (for the automaton) is also collected.
        Expression value = CifEquationUtils.getSingleValueForAlgVar(algVar);

        // Collect in the value expression.
        walkExpression(value, cifVarObjs);
    }
}
