//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2025, 2026 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.bdd.varorder.orderers;

import static org.eclipse.escet.common.java.Strings.fmt;

import org.eclipse.escet.cif.bdd.varorder.helper.VarOrdererData;
import org.eclipse.escet.cif.bdd.varorder.helper.VarOrdererEffect;
import org.eclipse.escet.common.java.Assert;

/** Variable orderer that repeatedly applies another orderer. */
public class RepeatVarOrderer extends VarOrderer {
    /** The variable orderer to repeatedly apply. */
    private final VarOrderer orderer;

    /** The number of times to repeat the {@link #orderer}. */
    private final int count;

    /** The effect of applying the {@link #orderer}, for all but the final time it is applied. */
    private final VarOrdererEffect intermediateEffect;

    /** The effect of applying the {@link #orderer}, for the final time it is applied. */
    private final VarOrdererEffect finalEffect;

    /**
     * Constructor for the {@link RepeatVarOrderer} class.
     *
     * @param orderer The variable orderer to repeatedly apply.
     * @param count The number of times to repeat the {@code orderer}.
     * @param intermediateEffect The effect of applying the {@code orderer}, for all but the final time it is applied.
     * @param finalEffect The effect of applying the {@code orderer}, for the final time it is applied.
     */
    public RepeatVarOrderer(VarOrderer orderer, int count, VarOrdererEffect intermediateEffect,
            VarOrdererEffect finalEffect)
    {
        this.orderer = orderer;
        this.count = count;
        this.intermediateEffect = intermediateEffect;
        this.finalEffect = finalEffect;
        Assert.check(count >= 1);
    }

    @Override
    public VarOrdererData order(VarOrdererData inputData, boolean dbgEnabled, int dbgLevel) {
        // Debug output before applying the orderer.
        if (dbgEnabled) {
            inputData.helper.dbg(dbgLevel, "Applying an orderer repeatedly:");
            inputData.helper.dbg(dbgLevel + 1, "Count: %s", count);
            inputData.helper.dbg(dbgLevel + 1, "Intermediate effect: %s", enumValueToParserArg(intermediateEffect));
            inputData.helper.dbg(dbgLevel + 1, "Final effect: %s", enumValueToParserArg(finalEffect));
            inputData.helper.dbg();
        }

        // Repeatedly apply the orderer.
        VarOrdererData currentData = inputData;
        for (int i = 0; i < count; i++) {
            // Separate different applications of the orderer by an empty line of debug output.
            if (i > 0 && dbgEnabled) {
                inputData.helper.dbg();
            }

            // Indicate the number of the current application, in debug output.
            if (dbgEnabled) {
                inputData.helper.dbg(dbgLevel + 1, "Repeated application %,d/%,d:", i + 1, count);
            }

            // Apply the orderer.
            VarOrdererData ordererData = orderer.order(currentData, dbgEnabled, dbgLevel + 2);

            // Update the variable order using the result from the orderer, based on the configured effect.
            VarOrdererEffect effect = (i == count - 1) ? finalEffect : intermediateEffect;
            currentData = new VarOrdererData(currentData, ordererData.varOrder, effect);
        }

        // Return the final variable order.
        return currentData;
    }

    @Override
    public String toString() {
        return fmt("repeat(orderer=%s, count=%s, intermediate-effect=%s, final-effect=%s)", orderer.toString(), count,
                enumValueToParserArg(intermediateEffect), enumValueToParserArg(finalEffect));
    }
}
