//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2025, 2026 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.common;

import java.util.List;

import org.eclipse.escet.cif.metamodel.cif.annotations.AnnotatedObject;
import org.eclipse.escet.cif.metamodel.cif.annotations.Annotation;
import org.eclipse.escet.cif.metamodel.cif.expressions.Expression;

/** CIF reachability requirement annotations utilities. */
public class CifReachabilityRequirementAnnotationUtils {
    /** The name of the reachability requirement annotation. */
    public static final String NAME = "requirement:reachable";

    /** Constructor for the {@link CifReachabilityRequirementAnnotationUtils} class. */
    private CifReachabilityRequirementAnnotationUtils() {
        // Static class.
    }

    /**
     * Returns the reachability requirement predicates of the reachability requirement annotations of an annotated
     * object.
     *
     * @param obj The annotated object.
     * @return The reachability requirement predicates.
     */
    public static List<Expression> getPredicates(AnnotatedObject obj) {
        return CifAnnotationUtils.getAnnotations(obj, NAME).map(anno -> getPredicate(anno)).toList();
    }

    /**
     * Returns the reachability requirement predicate of a reachability requirement annotation.
     *
     * @param anno The reachability requirement annotation.
     * @return The reachability requirement predicate.
     */
    public static Expression getPredicate(Annotation anno) {
        return anno.getArguments().getFirst().getValue();
    }

    /**
     * Remove all reachability requirement annotations from the given object.
     *
     * @param obj The object from which to remove the reachability requirement annotations.
     */
    public static void remove(AnnotatedObject obj) {
        CifAnnotationUtils.removeAnnotations(obj, NAME);
    }
}
