//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2026 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.relations;

import static org.eclipse.escet.cif.common.CifTextUtils.getAbsName;
import static org.eclipse.escet.cif.common.CifTextUtils.invToStr;
import static org.eclipse.escet.common.app.framework.output.OutputProvider.warn;
import static org.eclipse.escet.common.java.Lists.list;

import java.util.List;

import org.eclipse.escet.cif.cif2cif.ElimComponentDefInst;
import org.eclipse.escet.cif.cif2cif.ElimSelf;
import org.eclipse.escet.cif.cif2cif.RemoveAnnotations;
import org.eclipse.escet.cif.cif2cif.RemoveIoDecls;
import org.eclipse.escet.cif.cif2cif.SimplifyValuesOptimized;
import org.eclipse.escet.cif.common.CifReachabilityRequirementAnnotationUtils;
import org.eclipse.escet.cif.io.CifReader;
import org.eclipse.escet.cif.metamodel.cif.Invariant;
import org.eclipse.escet.cif.metamodel.cif.Specification;
import org.eclipse.escet.cif.metamodel.cif.automata.Automaton;
import org.eclipse.escet.cif.relations.ciftodmm.CifRelations;
import org.eclipse.escet.cif.relations.ciftodmm.CifToDmm;
import org.eclipse.escet.cif.relations.options.PlantGroupsDmmOutputFileOption;
import org.eclipse.escet.cif.relations.options.PlantGroupsRequirementGroupsDmmOutputFileOption;
import org.eclipse.escet.cif.relations.options.RequirementGroupsDmmOutputFileOption;
import org.eclipse.escet.cif.relations.options.Rfc1480ComplianceOption;
import org.eclipse.escet.common.app.framework.Application;
import org.eclipse.escet.common.app.framework.Paths;
import org.eclipse.escet.common.app.framework.io.AppStream;
import org.eclipse.escet.common.app.framework.io.AppStreams;
import org.eclipse.escet.common.app.framework.io.FileAppStream;
import org.eclipse.escet.common.app.framework.options.InputFileOption;
import org.eclipse.escet.common.app.framework.options.Option;
import org.eclipse.escet.common.app.framework.options.OptionCategory;
import org.eclipse.escet.common.app.framework.options.Options;
import org.eclipse.escet.common.app.framework.output.IOutputComponent;
import org.eclipse.escet.common.app.framework.output.OutputProvider;
import org.eclipse.escet.common.dsm.Dmm;
import org.eclipse.escet.common.java.PathPair;
import org.eclipse.escet.common.position.metamodel.position.PositionObject;

/**
 * CIF relations collector application.
 *
 * <p>
 * This implementation is based on the paper Goorden 2020:
 *
 * M. Goorden, J. v. d. Mortel-Fronczak, M. Reniers, W. Fokkink and J. Rooda, "Structuring Multilevel Discrete-Event
 * Systems With Dependence Structure Matrices", IEEE Transactions on Automatic Control, volume 65, issue 4, pages
 * 1625-1639, 2020, doi:<a href="https://doi.org/10.1109/TAC.2019.2928119">10.1109/TAC.2019.2928119</a>.
 * </p>
 */
public class CifRelationsApp extends Application<IOutputComponent> {
    /**
     * Application main method.
     *
     * @param args The command line arguments supplied to the application.
     */
    public static void main(String[] args) {
        CifRelationsApp app = new CifRelationsApp();
        app.run(args, true);
    }

    /** Constructor for the {@link CifRelationsApp} class. */
    public CifRelationsApp() {
        // Nothing to do.
    }

    /**
     * Constructor for the {@link CifRelationsApp} class.
     *
     * @param streams The streams to use for input, output, warning, and error streams.
     */
    public CifRelationsApp(AppStreams streams) {
        super(streams);
    }

    @Override
    protected int runInternal() {
        // Load the provided CIF specification.
        CifReader cifReader = new CifReader().init();
        Specification inputSpec = cifReader.read();
        String inputSpecPath = InputFileOption.getPath();
        String absInputSpecPath = Paths.resolve(inputSpecPath);
        if (isTerminationRequested()) {
            return 0;
        }

        // Specification transformations.
        //
        // Keep the reachability annotations, so they can be reported in the pre-checker.
        new RemoveAnnotations().except(CifReachabilityRequirementAnnotationUtils.NAME).transform(inputSpec);
        new ElimComponentDefInst().transform(inputSpec);
        new ElimSelf().transform(inputSpec);
        new SimplifyValuesOptimized().transform(inputSpec);
        if (isTerminationRequested()) {
            return 0;
        }

        // Remove I/O declarations and complain about SVG inputs because that may indicate a simulation model may have
        // been supplied.
        RemoveIoDecls removeIoDecls = new RemoveIoDecls();
        removeIoDecls.transform(inputSpec);
        removeIoDecls.warnAboutIgnoredSvgInputDecsIfRemoved(getAppEnvData().getProvider().getWarningOutputStream());
        if (isTerminationRequested()) {
            return 0;
        }

        // Check pre-conditions.
        CifToDmm.checkSpec(inputSpec, absInputSpecPath, () -> isTerminationRequested());
        if (isTerminationRequested()) {
            return 0;
        }

        // Get the relations in the specification.
        CifRelations relations = CifToDmm.transformToDmms(inputSpec);
        if (isTerminationRequested()) {
            return 0;
        }

        // Write the DMM files (for as far as requested).
        boolean rfc4180Compliant = Rfc1480ComplianceOption.isEnabled();
        writeDmmFile(relations.plantGroups, PlantGroupsDmmOutputFileOption.getFilePath(inputSpecPath),
                rfc4180Compliant);
        if (isTerminationRequested()) {
            return 0;
        }
        writeDmmFile(relations.requirementGroups, RequirementGroupsDmmOutputFileOption.getFilePath(inputSpecPath),
                rfc4180Compliant);
        if (isTerminationRequested()) {
            return 0;
        }
        writeDmmFile(relations.relations, PlantGroupsRequirementGroupsDmmOutputFileOption.getFilePath(inputSpecPath),
                rfc4180Compliant);
        if (isTerminationRequested()) {
            return 0;
        }

        // Warn about requirements of requirement groups that don't do anything useful since they are not related to
        // any plant group.
        for (PositionObject posObj: relations.getUselessRequirements()) {
            if (posObj instanceof Automaton) {
                warn("Requirement automaton \"%s\" has no relation to any plant element and does not affect behavior.",
                        getAbsName(posObj, false));
            } else if (posObj instanceof Invariant inv) {
                warn("Requirement invariant \"%s\" has no relation to any plant element and does not affect behavior.",
                        invToStr(inv, true));
            } else {
                throw new AssertionError("Unexpected kind of requirement found: \"" + posObj + "\".");
            }
        }

        return 0;
    }

    /**
     * (Try to) write the DMM file to the file system if the user requests it.
     *
     * @param dmmData Data of the DMM to write.
     * @param outputFilePath Absolute or relative local file system path of the file to write. May be {@code null} to
     *     not write the file.
     * @param rfc4180Compliance Whether to use CRLF line-endings in order to be RFC-4180 compliant.
     */
    private void writeDmmFile(Dmm dmmData, String outputFilePath, boolean rfc4180Compliance) {
        if (outputFilePath == null) {
            return;
        }
        PathPair outputPair = new PathPair(outputFilePath, Paths.resolve(outputFilePath));
        try (AppStream stream = new FileAppStream(outputPair)) {
            stream.print(dmmData.toString(rfc4180Compliance));
        }
    }

    @SuppressWarnings("rawtypes")
    @Override
    protected OptionCategory getAllOptions() {
        OptionCategory generalCat = getGeneralOptionCategory();

        List<Option> appOpts = list();
        appOpts.add(Options.getInstance(InputFileOption.class));
        appOpts.add(Options.getInstance(PlantGroupsDmmOutputFileOption.class));
        appOpts.add(Options.getInstance(RequirementGroupsDmmOutputFileOption.class));
        appOpts.add(Options.getInstance(PlantGroupsRequirementGroupsDmmOutputFileOption.class));
        appOpts.add(Options.getInstance(Rfc1480ComplianceOption.class));
        List<OptionCategory> appSubCats = list();
        OptionCategory appCat = new OptionCategory("Collector", "Collector options.", appSubCats, appOpts);

        List<OptionCategory> cats = list(generalCat, appCat);
        return new OptionCategory("CIF Relations Collector Options",
                "All options for the CIF relations collector.", cats, list());
    }

    @Override
    protected OutputProvider<IOutputComponent> createProvider() {
        return new OutputProvider<>();
    }

    @Override
    public String getAppName() {
        return "CIF relations collector";
    }

    @Override
    public String getAppDescription() {
        return "Collects relations between plants and requirements in a CIF specification.";
    }

    @Override
    public String getAppToolDefLibName() {
        return "cif";
    }

    @Override
    public String getAppToolDefToolName() {
        return "cifrelations";
    }
}
