/*
 * Decompiled with CFR 0.152.
 */
package com.unboundid.ldap.sdk;

import com.unboundid.asn1.ASN1Buffer;
import com.unboundid.ldap.protocol.LDAPMessage;
import com.unboundid.ldap.sdk.ConnectThread;
import com.unboundid.ldap.sdk.DisconnectInfo;
import com.unboundid.ldap.sdk.DisconnectType;
import com.unboundid.ldap.sdk.LDAPConnection;
import com.unboundid.ldap.sdk.LDAPConnectionOptions;
import com.unboundid.ldap.sdk.LDAPConnectionReader;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.LDAPMessages;
import com.unboundid.ldap.sdk.LDAPRuntimeException;
import com.unboundid.ldap.sdk.ResponseAcceptor;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.Debug;
import com.unboundid.util.DebugType;
import com.unboundid.util.InternalUseOnly;
import com.unboundid.util.StaticUtils;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.InetAddress;
import java.net.Socket;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import javax.net.SocketFactory;
import javax.net.ssl.SSLSocket;
import javax.net.ssl.SSLSocketFactory;
import javax.security.sasl.SaslClient;

@InternalUseOnly
final class LDAPConnectionInternals {
    private static final AtomicLong ACTIVE_CONNECTION_COUNT = new AtomicLong(0L);
    private static final AtomicReference<ThreadLocal<ASN1Buffer>> ASN1_BUFFERS = new AtomicReference(new ThreadLocal());
    private final AtomicInteger nextMessageID;
    private final boolean synchronousMode;
    private final InetAddress inetAddress;
    private final int port;
    private final long connectTime;
    private final LDAPConnection connection;
    private final LDAPConnectionReader connectionReader;
    private volatile OutputStream outputStream;
    private volatile SaslClient saslClient;
    private volatile Socket socket;
    private final String host;

    LDAPConnectionInternals(LDAPConnection connection, LDAPConnectionOptions options, SocketFactory socketFactory, String host, InetAddress inetAddress, int port, int timeout) throws IOException {
        this.connection = connection;
        this.host = host;
        this.inetAddress = inetAddress;
        this.port = port;
        if (options.captureConnectStackTrace()) {
            connection.setConnectStackTrace(Thread.currentThread().getStackTrace());
        }
        this.connectTime = System.currentTimeMillis();
        this.nextMessageID = new AtomicInteger(0);
        this.synchronousMode = options.useSynchronousMode();
        this.saslClient = null;
        try {
            ConnectThread connectThread = new ConnectThread(socketFactory, inetAddress, port, timeout);
            connectThread.start();
            this.socket = connectThread.getConnectedSocket();
            if (this.socket instanceof SSLSocket) {
                SSLSocket sslSocket = (SSLSocket)this.socket;
                options.getSSLSocketVerifier().verifySSLSocket(host, port, sslSocket);
            }
        }
        catch (LDAPException le) {
            Debug.debugException(le);
            throw new IOException(le);
        }
        if (options.getReceiveBufferSize() > 0) {
            this.socket.setReceiveBufferSize(options.getReceiveBufferSize());
        }
        if (options.getSendBufferSize() > 0) {
            this.socket.setSendBufferSize(options.getSendBufferSize());
        }
        try {
            Debug.debugConnect(host, port, connection);
            this.socket.setKeepAlive(options.useKeepAlive());
            this.socket.setReuseAddress(options.useReuseAddress());
            this.socket.setSoLinger(options.useLinger(), options.getLingerTimeoutSeconds());
            this.socket.setTcpNoDelay(options.useTCPNoDelay());
            this.outputStream = new BufferedOutputStream(this.socket.getOutputStream());
            this.connectionReader = new LDAPConnectionReader(connection, this);
        }
        catch (IOException ioe) {
            Debug.debugException(ioe);
            try {
                this.socket.close();
            }
            catch (Exception e) {
                Debug.debugException(e);
            }
            throw ioe;
        }
        ACTIVE_CONNECTION_COUNT.incrementAndGet();
    }

    void startConnectionReader() {
        if (!this.synchronousMode) {
            this.connectionReader.start();
        }
    }

    LDAPConnection getConnection() {
        return this.connection;
    }

    LDAPConnectionReader getConnectionReader() {
        return this.connectionReader;
    }

    InetAddress getInetAddress() {
        return this.inetAddress;
    }

    String getHost() {
        return this.host;
    }

    int getPort() {
        return this.port;
    }

    Socket getSocket() {
        return this.socket;
    }

    void setSocket(Socket socket) {
        this.socket = socket;
    }

    OutputStream getOutputStream() {
        return this.outputStream;
    }

    boolean isConnected() {
        return this.socket.isConnected();
    }

    boolean synchronousMode() {
        return this.synchronousMode;
    }

    void convertToTLS(SSLSocketFactory sslSocketFactory) throws LDAPException {
        this.outputStream = this.connectionReader.doStartTLS(sslSocketFactory);
    }

    void applySASLQoP(SaslClient saslClient) throws LDAPException {
        this.saslClient = saslClient;
        this.connectionReader.applySASLQoP(saslClient);
    }

    int nextMessageID() {
        int msgID = this.nextMessageID.incrementAndGet();
        if (msgID > 0) {
            return msgID;
        }
        do {
            if (!this.nextMessageID.compareAndSet(msgID, 1)) continue;
            return 1;
        } while ((msgID = this.nextMessageID.incrementAndGet()) <= 0);
        return msgID;
    }

    void registerResponseAcceptor(int messageID, ResponseAcceptor responseAcceptor) throws LDAPException {
        if (!this.isConnected()) {
            LDAPConnectionOptions connectionOptions = this.connection.getConnectionOptions();
            boolean autoReconnect = connectionOptions.autoReconnect();
            boolean closeRequested = this.connection.closeRequested();
            if (autoReconnect && !closeRequested) {
                this.connection.reconnect();
                this.connection.registerResponseAcceptor(messageID, responseAcceptor);
            } else {
                throw new LDAPException(ResultCode.SERVER_DOWN, LDAPMessages.ERR_CONN_NOT_ESTABLISHED.get());
            }
        }
        this.connectionReader.registerResponseAcceptor(messageID, responseAcceptor);
    }

    void deregisterResponseAcceptor(int messageID) {
        this.connectionReader.deregisterResponseAcceptor(messageID);
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    void sendMessage(LDAPMessage message, boolean allowRetry) throws LDAPException {
        if (!this.isConnected()) {
            throw new LDAPException(ResultCode.SERVER_DOWN, LDAPMessages.ERR_CONN_NOT_ESTABLISHED.get());
        }
        ASN1Buffer buffer = ASN1_BUFFERS.get().get();
        if (buffer == null) {
            buffer = new ASN1Buffer();
            ASN1_BUFFERS.get().set(buffer);
        }
        buffer.clear();
        try {
            message.writeTo(buffer);
        }
        catch (LDAPRuntimeException lre) {
            Debug.debugException(lre);
            lre.throwLDAPException();
        }
        try {
            OutputStream os = this.outputStream;
            if (this.saslClient == null) {
                buffer.writeTo(os);
            } else {
                byte[] clearBytes = buffer.toByteArray();
                byte[] saslBytes = this.saslClient.wrap(clearBytes, 0, clearBytes.length);
                byte[] lengthBytes = new byte[]{(byte)(saslBytes.length >> 24 & 0xFF), (byte)(saslBytes.length >> 16 & 0xFF), (byte)(saslBytes.length >> 8 & 0xFF), (byte)(saslBytes.length & 0xFF)};
                os.write(lengthBytes);
                os.write(saslBytes);
            }
            os.flush();
            return;
        }
        catch (IOException ioe) {
            Debug.debugException(ioe);
            if (message.getProtocolOpType() == 66) {
                return;
            }
            boolean closeRequested = this.connection.closeRequested();
            if (!allowRetry || closeRequested || this.connection.synchronousMode()) throw new LDAPException(ResultCode.SERVER_DOWN, LDAPMessages.ERR_CONN_SEND_ERROR.get(this.host + ':' + this.port, StaticUtils.getExceptionMessage(ioe)), ioe);
            this.connection.reconnect();
            try {
                this.sendMessage(message, false);
                return;
            }
            catch (Exception e) {
                Debug.debugException(e);
                throw new LDAPException(ResultCode.SERVER_DOWN, LDAPMessages.ERR_CONN_SEND_ERROR.get(this.host + ':' + this.port, StaticUtils.getExceptionMessage(ioe)), ioe);
            }
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new LDAPException(ResultCode.LOCAL_ERROR, LDAPMessages.ERR_CONN_ENCODE_ERROR.get(this.host + ':' + this.port, StaticUtils.getExceptionMessage(e)), e);
        }
        finally {
            if (buffer.zeroBufferOnClear()) {
                buffer.clear();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void close() {
        DisconnectInfo disconnectInfo = this.connection.getDisconnectInfo();
        if (disconnectInfo == null) {
            disconnectInfo = this.connection.setDisconnectInfo(new DisconnectInfo(this.connection, DisconnectType.UNKNOWN, null, null));
        }
        boolean closedByFinalizer = disconnectInfo.getType() == DisconnectType.CLOSED_BY_FINALIZER && this.socket.isConnected();
        try {
            this.connectionReader.close(false);
        }
        catch (Exception e) {
            Debug.debugException(e);
        }
        try {
            this.outputStream.close();
        }
        catch (Exception e) {
            Debug.debugException(e);
        }
        try {
            this.socket.close();
        }
        catch (Exception e) {
            Debug.debugException(e);
        }
        if (this.saslClient != null) {
            try {
                this.saslClient.dispose();
            }
            catch (Exception e) {
                Debug.debugException(e);
            }
            finally {
                this.saslClient = null;
            }
        }
        Debug.debugDisconnect(this.host, this.port, this.connection, disconnectInfo.getType(), disconnectInfo.getMessage(), disconnectInfo.getCause());
        if (closedByFinalizer && Debug.debugEnabled(DebugType.LDAP)) {
            Debug.debug(Level.WARNING, DebugType.LDAP, "Connection closed by LDAP SDK finalizer:  " + this.toString());
        }
        disconnectInfo.notifyDisconnectHandler();
        long remainingActiveConnections = ACTIVE_CONNECTION_COUNT.decrementAndGet();
        if (remainingActiveConnections <= 0L) {
            ASN1_BUFFERS.set(new ThreadLocal());
            if (remainingActiveConnections < 0L) {
                ACTIVE_CONNECTION_COUNT.compareAndSet(remainingActiveConnections, 0L);
            }
        }
    }

    public long getConnectTime() {
        if (this.isConnected()) {
            return this.connectTime;
        }
        return -1L;
    }

    static long getActiveConnectionCount() {
        return ACTIVE_CONNECTION_COUNT.get();
    }

    public String toString() {
        StringBuilder buffer = new StringBuilder();
        this.toString(buffer);
        return buffer.toString();
    }

    public void toString(StringBuilder buffer) {
        buffer.append("LDAPConnectionInternals(host='");
        buffer.append(this.host);
        buffer.append("', port=");
        buffer.append(this.port);
        buffer.append(", connected=");
        buffer.append(this.socket.isConnected());
        buffer.append(", nextMessageID=");
        buffer.append(this.nextMessageID.get());
        buffer.append(')');
    }
}

