/*
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.activity.diagram.design;

import java.util.HashSet;
import java.util.Optional;
import java.util.Set;

import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.EContentAdapter;
import org.eclipse.emf.transaction.RecordingCommand;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.emf.transaction.util.TransactionUtil;
import org.eclipse.lsat.activity.diagram.services.ActivityServices;
import org.eclipse.sirius.business.api.componentization.ViewpointRegistry;
import org.eclipse.sirius.business.api.dialect.DialectManager;
import org.eclipse.sirius.business.api.session.Session;
import org.eclipse.sirius.business.api.session.SessionListener;
import org.eclipse.sirius.business.api.session.SessionManager;
import org.eclipse.sirius.business.api.session.SessionManagerListener;
import org.eclipse.sirius.diagram.DDiagram;
import org.eclipse.sirius.diagram.DDiagramElement;
import org.eclipse.sirius.diagram.description.filter.FilterDescription;
import org.eclipse.sirius.viewpoint.description.Viewpoint;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.BundleContext;

import activity.Activity;
import activity.PeripheralAction;

/**
 * The activator class controls the plug-in life cycle
 */
public class Activator extends AbstractUIPlugin {
    // The plug-in ID
    public static final String PLUGIN_ID = "org.eclipse.lsat.activity.diagram.design";
    private final String timeFilterID = "Show Activity Time";

    // The shared instance
    private static Activator plugin;

    private static Set<Viewpoint> viewpoints;

    private SessionManagerListener sessionListener;

    /**
     * The constructor
     */
    public Activator() {
    }

    /*
     * (non-Javadoc)
     *
     * @see org.eclipse.ui.plugin.AbstractUIPlugin#start(org.osgi.framework.BundleContext)
     */
    @Override
    public void start(BundleContext context) throws Exception {
        super.start(context);
        plugin = this;
        viewpoints = new HashSet<Viewpoint>();
        viewpoints
                .addAll(ViewpointRegistry.getInstance().registerFromPlugin(PLUGIN_ID + "/description/diagram.odesign"));

        sessionListener = new SessionManagerListener.Stub() {
            @Override
            public void notify(Session session, int notification) {
                if (notification == SessionListener.OPENED) {
                    registerTimeFilterEvents(session);
                }
            }
        };

        SessionManager.INSTANCE.addSessionsListener(sessionListener);
    }

    /*
     * (non-Javadoc)
     *
     * @see org.eclipse.ui.plugin.AbstractUIPlugin#stop(org.osgi.framework.BundleContext)
     */
    @Override
    public void stop(BundleContext context) throws Exception {
        plugin = null;
        if (viewpoints != null) {
            for (final Viewpoint viewpoint: viewpoints) {
                ViewpointRegistry.getInstance().disposeFromPlugin(viewpoint);
            }
            viewpoints.clear();
            viewpoints = null;
        }

        if (sessionListener != null) {
            SessionManager.INSTANCE.removeSessionsListener(sessionListener);
        }
        super.stop(context);
    }

    /**
     * Returns the shared instance
     *
     * @return the shared instance
     */
    public static Activator getDefault() {
        return plugin;
    }

    public void registerTimeFilterEvents(Session session) {
        EContentAdapter adapter = new EContentAdapter() {
            @Override
            public void notifyChanged(Notification notification) {
                super.notifyChanged(notification);

                if (!(notification.getFeature() instanceof EStructuralFeature feature)
                        || !"activatedFilters".equals(feature.getName())
                        || !(notification.getNotifier() instanceof DDiagram diagram))
                {
                    return;
                }

                // Get target filter
                Optional<FilterDescription> targetFilter = diagram.getDescription()
                        .getFilters()
                        .stream()
                        .filter(f -> timeFilterID.equals(f.getName()))
                        .findFirst();

                if (targetFilter.isEmpty()) {
                    return;
                }

                boolean active = diagram.getActivatedFilters().contains(targetFilter.get());
                updateTimingLabelVisibility(diagram, active);

                // Refresh diagram
                DialectManager.INSTANCE.refresh(diagram, new NullProgressMonitor());
            }
        };

        session.getTransactionalEditingDomain()
               .getResourceSet()
               .eAdapters()
               .add(adapter);
    }

    private void updateTimingLabelVisibility(DDiagram diagram, boolean showTimingLabels) {
        TransactionalEditingDomain domain = TransactionUtil.getEditingDomain(diagram);
        if (domain == null) {
            return;
        }

        domain.getCommandStack().execute(new RecordingCommand(domain) {
            @Override
            protected void doExecute() {
                for (DDiagramElement element : diagram.getDiagramElements()) {
                    EObject target = element.getTarget();
                    if (target instanceof PeripheralAction action && action.getGraph() instanceof Activity) {
                        ActivityServices.setShowTimeLabels(diagram, showTimingLabels);
                    }
                }
            }
        });
    }
}
