/**
 * Copyright (c) 2016 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.di;

import static com.google.common.base.Preconditions.checkNotNull;
import static org.eclipse.epp.internal.logging.aeri.ide.l10n.LogMessages.WARN_EXTENSION_FAILED;
import static org.eclipse.epp.logging.aeri.core.util.Logs.log;

import java.util.List;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.e4.core.contexts.ContextFunction;
import org.eclipse.e4.core.contexts.IEclipseContext;
import org.eclipse.epp.internal.logging.aeri.ide.IDEWorkflow;
import org.eclipse.epp.internal.logging.aeri.ide.IIdeFactory;
import org.eclipse.epp.internal.logging.aeri.ide.IProcessorDescriptor;
import org.eclipse.epp.internal.logging.aeri.ide.handlers.CreateReportProcessorHandler;
import org.eclipse.epp.internal.logging.aeri.ide.processors.ReportProcessorSafeWrapper;
import org.eclipse.epp.logging.aeri.core.SystemControl;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableList.Builder;

public class ExtensionPointReportProcessorsCreationFunction extends ContextFunction {

    public static final String EXT_REPORT_PROCESSORS = IDEWorkflow.CTX_REPORT_PROCESSORS;

    @Override
    public Object compute(IEclipseContext context, String contextKey) {
        checkNotNull(contextKey);
        IEclipseContext systemContext = SystemControl.getSystemContext();

        IExtensionRegistry registry = systemContext.get(IExtensionRegistry.class);
        IConfigurationElement[] elements = registry.getConfigurationElementsFor(EXT_REPORT_PROCESSORS);
        Builder<IProcessorDescriptor> builder = ImmutableList.builder();
        for (final IConfigurationElement element : elements) {
            IProcessorDescriptor descriptor = IIdeFactory.eINSTANCE.createProcessorDescriptor();
            descriptor.setConfigurationElement(element);
            descriptor.setDirective(element.getAttribute("directive"));
            descriptor.setName(element.getAttribute("name"));
            descriptor.setDescription(element.getAttribute("description"));
            descriptor.setIcon16(element.getAttribute("icon16"));
            descriptor.setContributor(element.getContributor().getName());
            descriptor.setAutomatic(Boolean.valueOf(element.getAttribute("isAutomatic")));
            try {
                IEclipseContext childContext = systemContext.createChild("Report processor context of '" + descriptor.getDirective() + "'"); //$NON-NLS-1$ //$NON-NLS-2$
                childContext.set(IProcessorDescriptor.class, descriptor);
                String providerClass = element.getAttribute("class");
                String contributor = element.getContributor().getName();
                ReportProcessorSafeWrapper processor = new CreateReportProcessorHandler().execute(providerClass, childContext, contributor);
                descriptor.setProcessor(processor);
            } catch (Exception e) {
                log(WARN_EXTENSION_FAILED, e, descriptor.getDirective(), e.getMessage());
            }
            builder.add(descriptor);
        }
        List<IProcessorDescriptor> descriptors = builder.build();
        systemContext.set(contextKey, descriptors);
        return descriptors;
    }
}
