/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.utils;

import static org.apache.commons.lang3.StringUtils.trimToNull;
import static org.eclipse.epp.internal.logging.aeri.ide.l10n.LogMessages.ERROR_FAILED_TO_PARSE_PREFERENCE_VALUE;
import static org.eclipse.epp.logging.aeri.core.util.Logs.log;

import java.util.List;
import java.util.Set;

import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.ui.preferences.ScopedPreferenceStore;

import com.google.common.base.Joiner;
import com.google.common.base.Splitter;

public class EmfPreferences {

    @SuppressWarnings("unchecked")
    public static <T> T listen(String nodeQualifier, EObject eObject, Set<EAttribute> attributes) {
        ScopedPreferenceStore instanceStore = new ScopedPreferenceStore(InstanceScope.INSTANCE, nodeQualifier);
        loadFromPreferences(instanceStore, eObject, attributes);
        registerPreferenceStore2EObjectChangeListener(instanceStore, eObject, attributes);
        registerEObjectToPreferencesChangeListener(instanceStore, eObject, attributes);
        return (T) eObject;
    }

    private static void loadFromPreferences(final ScopedPreferenceStore store, final EObject eObject, Set<EAttribute> attributes) {
        eObject.eSetDeliver(false);
        for (EAttribute attr : attributes) {
            EDataType type = attr.getEAttributeType();
            String key = attr.getName();
            String value = trimToNull(store.getString(key));
            try {
                if (value == null) {
                    eObject.eUnset(attr);
                    continue;
                }
                if (attr.isMany()) {
                    @SuppressWarnings("unchecked")
                    List<Object> list = (List<Object>) eObject.eGet(attr);
                    for (String s : convert(value)) {
                        Object data = EcoreUtil.createFromString(type, s);
                        list.add(data);
                    }
                } else {
                    Object data = EcoreUtil.createFromString(type, value);
                    if (data == null) {
                        eObject.eUnset(attr);
                    } else {
                        eObject.eSet(attr, data);
                    }
                }
            } catch (Exception e) {
                log(ERROR_FAILED_TO_PARSE_PREFERENCE_VALUE, e, attr, value);
            }
        }
        eObject.eSetDeliver(true);
    }

    private static void registerEObjectToPreferencesChangeListener(final ScopedPreferenceStore store, final EObject eObject,
            final Set<EAttribute> attributes) {
        eObject.eAdapters().add(new AdapterImpl() {
            @Override
            public void notifyChanged(Notification msg) {
                Object feature = msg.getFeature();
                if (!(feature instanceof EAttribute)) {
                    return;
                }
                EAttribute attr = (EAttribute) feature;
                if (!attributes.contains(attr)) {
                    return;
                }
                String key = attr.getName();
                EDataType type = attr.getEAttributeType();
                Object value = msg.getNewValue();
                String data = EcoreUtil.convertToString(type, value);
                if (data == null) {
                    store.setToDefault(key);
                } else {
                    store.putValue(key, data);
                }
            }
        });
    }

    private static void registerPreferenceStore2EObjectChangeListener(final ScopedPreferenceStore store, final EObject eObject,
            final Set<EAttribute> attributes) {
        store.addPropertyChangeListener(new IPropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent event) {
                String property = event.getProperty();
                EStructuralFeature feature = eObject.eClass().getEStructuralFeature(property);
                if (feature != null && feature instanceof EAttribute) {
                    EAttribute attr = (EAttribute) feature;
                    if (!attributes.contains(attr)) {
                        return;
                    }
                    EDataType type = attr.getEAttributeType();
                    String string = EcoreUtil.convertToString(type, event.getNewValue());
                    Object value = EcoreUtil.createFromString(type, string);
                    eObject.eSet(feature, value);
                }
            }
        });
    }

    static List<String> convert(String string) {
        return Splitter.on(";#;").omitEmptyStrings().trimResults().splitToList(string); //$NON-NLS-1$

    }

    static String convert(List<String> strings) {
        return Joiner.on(";#;").skipNulls().join(strings); //$NON-NLS-1$
    }
}
