/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.logging.aeri.core.filters;

import java.text.MessageFormat;

import org.eclipse.core.runtime.ILog;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.epp.logging.aeri.core.Constants;
import org.eclipse.epp.logging.aeri.core.IReport;
import org.eclipse.epp.logging.aeri.core.SystemControl;
import org.eclipse.jdt.annotation.Nullable;
import org.osgi.framework.FrameworkUtil;

import com.google.common.base.Predicate;

public class DecoratingDebugFilter<T> implements Predicate<T> {

    private static final int DEBUG_STATUS_SEVERITY = IStatus.INFO;
    private static final String DEBUG_STATUS_PLUGIN_ID = Constants.BUNDLE_ID;

    @SuppressWarnings("unchecked")
    public static <T> Predicate<? super T>[] decorate(Predicate<? super T>... components) {
        ILog log = Platform.getLog(FrameworkUtil.getBundle(DecoratingDebugFilter.class));
        Predicate<? super T>[] predicates = new Predicate[components.length];
        for (int i = 0; i < components.length; i++) {
            predicates[i] = new DecoratingDebugFilter<>(components[i], log);
        }
        return predicates;
    }

    private Predicate<T> predicate;
    private ILog log;

    public DecoratingDebugFilter(Predicate<T> predicate, ILog log) {
        this.predicate = predicate;
        this.log = log;
    }

    @Override
    public boolean apply(@Nullable T input) {
        boolean apply = predicate.apply(input);
        if (!apply && SystemControl.isDebug()) {
            if (input instanceof IStatus) {
                debugStatus((IStatus) input);
            } else if (input instanceof IReport) {
                debugReport((IReport) input);
            }
        }
        return apply;
    }

    private void debugStatus(IStatus inputStatus) {
        // avoid recursion with own debug status logging
        if (inputStatus.getSeverity() == DEBUG_STATUS_SEVERITY && inputStatus.getPlugin().equals(DEBUG_STATUS_PLUGIN_ID)) {
            return;
        }
        String message = MessageFormat.format("Debug: predicate {0} rejected status: {1}", predicate.getClass().getSimpleName(),
                inputStatus.getMessage());
        logDebugMessageStatus(message);
    }

    private void debugReport(IReport report) {
        String message = MessageFormat.format("Debug: predicate {0} rejected report: {1}", predicate.getClass().getSimpleName(),
                report.getStatus().getMessage());
        logDebugMessageStatus(message);
    }

    private void logDebugMessageStatus(String message) {
        IStatus status = new Status(DEBUG_STATUS_SEVERITY, DEBUG_STATUS_PLUGIN_ID, message);
        log.log(status);
    }

    @Override
    public boolean equals(@Nullable Object object) {
        return predicate.equals(object);
    }

    @Override
    public int hashCode() {
        return predicate.hashCode();
    }

    @Override
    public String toString() {
        return predicate.toString();
    }

}
