/*
 * Copyright 2015-2019 the original author or authors.
 *
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License v2.0 which
 * accompanies this distribution and is available at
 *
 * https://www.eclipse.org/legal/epl-v20.html
 */

package org.junit.jupiter.engine.support;

import java.util.function.Predicate;

import org.junit.platform.commons.util.ReflectionUtils;
import org.junit.platform.engine.support.hierarchical.ThrowableCollector;
import org.opentest4j.TestAbortedException;

/**
 * Specialization of {@link ThrowableCollector} that treats instances of the
 * OTA's {@link org.opentest4j.TestAbortedException} and JUnit 4's
 * {@code org.junit.AssumptionViolatedException} as <em>aborting</em>.
 *
 * @since 5.4
 * @see ThrowableCollector
 * @see org.junit.platform.engine.support.hierarchical.OpenTest4JAwareThrowableCollector
 */
class OpenTest4JAndJUnit4AwareThrowableCollector extends ThrowableCollector {

	private static final Predicate<? super Throwable> abortedExecutionPredicate = createAbortedExecutionPredicate();

	OpenTest4JAndJUnit4AwareThrowableCollector() {
		super(abortedExecutionPredicate);
	}

	private static Predicate<? super Throwable> createAbortedExecutionPredicate() {
		Predicate<Throwable> otaPredicate = TestAbortedException.class::isInstance;

		// Additionally support JUnit 4's AssumptionViolatedException?
		Class<?> clazz = ReflectionUtils.tryToLoadClass(
			"org.junit.internal.AssumptionViolatedException").toOptional().orElse(null);
		if (clazz != null) {
			return otaPredicate.or(clazz::isInstance);
		}

		// Else just OTA's TestAbortedException
		return otaPredicate;
	}

}
