/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide;

import static java.util.concurrent.TimeUnit.MINUTES;
import static org.eclipse.epp.internal.logging.aeri.ide.l10n.LogMessages.WARN_REPORTING_ERROR;
import static org.eclipse.epp.internal.logging.aeri.ide.utils.IDEConstants.BUNDLE_ID;
import static org.eclipse.epp.internal.logging.aeri.ide.utils.Servers.copyServerDefaultsIfUnset;
import static org.eclipse.epp.internal.logging.aeri.ide.utils.Servers.copySystemDefaultsIfUnset;
import static org.eclipse.epp.internal.logging.aeri.ide.utils.Servers.viewOfEnabled;
import static org.eclipse.epp.logging.aeri.core.ProblemStatus.FAILURE;
import static org.eclipse.epp.logging.aeri.core.ProblemStatus.IGNORED;
import static org.eclipse.epp.logging.aeri.core.ProblemStatus.INVALID;
import static org.eclipse.epp.logging.aeri.core.SendMode.BACKGROUND;
import static org.eclipse.epp.logging.aeri.core.util.Logs.log;

import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import javax.inject.Inject;
import javax.inject.Named;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.e4.core.contexts.IEclipseContext;
import org.eclipse.e4.core.di.annotations.Optional;
import org.eclipse.e4.core.di.extensions.EventTopic;
//import org.eclipse.e4.core.di.extensions.EventTopic;
import org.eclipse.e4.core.services.events.IEventBroker;
import org.eclipse.emf.common.util.EList;
import org.eclipse.epp.internal.logging.aeri.ide.dialogs.ReviewDialog;
import org.eclipse.epp.internal.logging.aeri.ide.dialogs.SetupWizard;
import org.eclipse.epp.internal.logging.aeri.ide.l10n.LogMessages;
import org.eclipse.epp.internal.logging.aeri.ide.l10n.Messages;
import org.eclipse.epp.internal.logging.aeri.ide.notifications.MylynNotificationsSupport;
import org.eclipse.epp.internal.logging.aeri.ide.processors.Processors;
import org.eclipse.epp.internal.logging.aeri.ide.processors.WiringErrorAnalyzerProcessor;
import org.eclipse.epp.internal.logging.aeri.ide.utils.Formats;
import org.eclipse.epp.internal.logging.aeri.ide.utils.Servers;
import org.eclipse.epp.internal.logging.aeri.ide.utils.UploadReportsScheduler;
import org.eclipse.epp.logging.aeri.core.IModelFactory;
import org.eclipse.epp.logging.aeri.core.IProblemState;
import org.eclipse.epp.logging.aeri.core.IReportProcessor;
import org.eclipse.epp.logging.aeri.core.ISendOptions;
import org.eclipse.epp.logging.aeri.core.ISystemSettings;
import org.eclipse.epp.logging.aeri.core.ProblemStatus;
import org.eclipse.epp.logging.aeri.core.SendMode;
import org.eclipse.epp.logging.aeri.core.util.Logs;
import org.osgi.service.event.Event;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

@SuppressWarnings("restriction")
public class IDEWorkflow {

    public static final String CTX_LOG_EVENT_LAST = BUNDLE_ID + ".di.event.last"; //$NON-NLS-1$
    public static final String CTX_LOG_EVENT_LAST_NOTIFIED = BUNDLE_ID + ".di.event.last-notified"; //$NON-NLS-1$
    public static final String CTX_LOG_EVENT_LAST_DELAYED = BUNDLE_ID + ".di.event.last-delayed"; //$NON-NLS-1$
    public static final String CTX_SERVERS = BUNDLE_ID + ".servers"; //$NON-NLS-1$
    public static final String CTX_REPORT_PROCESSORS = BUNDLE_ID + ".processors"; //$NON-NLS-1$
    public static final String CTX_BUNDLE_STATE_LOCATION = BUNDLE_ID + ".stateLocation"; //$NON-NLS-1$

    public static final String TOPIC_BASE = "org/eclipse/epp/internal/logging/aeri/ui"; //$NON-NLS-1$
    public static final String TOPIC_NEW_STATUS_LOGGED = TOPIC_BASE + "/status/new"; //$NON-NLS-1$
    public static final String TOPIC_NEW_EVENT_LOGGED = TOPIC_BASE + "/event/new"; //$NON-NLS-1$
    public static final String TOPIC_RESCHEDULE_NOTIFCATION_LAST_LOG_EVENT = TOPIC_BASE + "/event/reschedule"; //$NON-NLS-1$
    public static final String TOPIC_USER_REQUESTS_DISABLE_NEW_SERVERS = TOPIC_BASE + "/servers/disabled"; //$NON-NLS-1$
    public static final String TOPIC_USER_REQUESTS_CLEAR_QUEUE = TOPIC_BASE + "/events/clear"; //$NON-NLS-1$
    public static final String TOPIC_NEW_SERVER_RESPONSES = TOPIC_BASE + "/events/responses"; //$NON-NLS-1$
    public static final String TOPIC_USER_REQUESTS_SEND_ONE_GROUP = TOPIC_BASE + "/event/send/group"; //$NON-NLS-1$
    public static final String TOPIC_USER_REQUESTS_SEND_ALL_GROUPS = TOPIC_BASE + "/event/send/groups"; //$NON-NLS-1$

    private final List<IServerDescriptor> servers;
    private final ILogEventsQueue queue;
    private final ISystemSettings settings;
    private final IEventBroker broker;
    private final MylynNotificationsSupport notificationSupport;
    private final UploadReportsScheduler uploadService;
    private final IEclipseContext context;
    private final List<IProcessorDescriptor> descriptors;

    private boolean systemStateSetupInProgress;
    private boolean systemStateReviewInProgress;
    private boolean systemStateNotificationInProgress;
    private long setupTimeout;

    @Inject
    public IDEWorkflow(@Named(CTX_SERVERS) List<IServerDescriptor> endpoints, ILogEventsQueue queue, ISystemSettings settings,
            IEventBroker broker, MylynNotificationsSupport notificationSupport, UploadReportsScheduler uploadService,
            IEclipseContext context, @Named(IDEWorkflow.CTX_REPORT_PROCESSORS) List<IProcessorDescriptor> descriptors) {
        this.servers = endpoints;
        this.queue = queue;
        this.settings = settings;
        this.broker = broker;
        this.notificationSupport = notificationSupport;
        this.uploadService = uploadService;
        this.context = context;
        this.descriptors = descriptors;
    }

    @Inject
    @Optional
    protected void onNewStatusLogged(@EventTopic(TOPIC_NEW_STATUS_LOGGED) IStatus status) {
        try {
            doNewStatusLogged(status);
        } catch (Exception e) {
            log(WARN_REPORTING_ERROR, e);
        }
    }

    private void doNewStatusLogged(IStatus status) {
        List<ILogEvent> events = Lists.newLinkedList();
        for (IServerDescriptor server : viewOfEnabled(servers)) {
            ISendOptions options = IModelFactory.eINSTANCE.createSendOptions();
            copyServerDefaultsIfUnset(server, options);
            copySystemDefaultsIfUnset(settings, options);

            IEclipseContext eventContext = context.createChild("Log event context for '" + server.getId() + "'"); //$NON-NLS-1$ //$NON-NLS-2$
            eventContext.set(IStatus.class, status);
            eventContext.set(IServerDescriptor.class, server);
            eventContext.declareModifiable(ISendOptions.class);
            eventContext.set(ISendOptions.class, options);

            IProblemState interest = getServerInterest(server, status, eventContext);
            if (!isInterested(interest)) {
                continue;
            }

            ILogEvent event = IIdeFactory.eINSTANCE.createLogEvent();
            event.setStatus(status);
            event.setContext(eventContext);
            event.setServer(server);
            event.setInterest(interest);
            event.setOptions(options);
            event.setLabel(status.getMessage());
            events.add(event);
        }

        if (!events.isEmpty()) {
            ILogEventGroup group = IIdeFactory.eINSTANCE.createLogEventGroup();
            group.setStatus(status);
            group.getEvents().addAll(events);
            broker.post(TOPIC_NEW_EVENT_LOGGED, group);
        }
    }

    @Inject
    @Optional
    protected synchronized void onNewEventLogged(@EventTopic(TOPIC_NEW_EVENT_LOGGED) ILogEventGroup group) {
        try {
            doNewEventLogged(group);
        } catch (Exception e) {
            log(WARN_REPORTING_ERROR, e);
        }
    }

    private void doNewEventLogged(ILogEventGroup group) {
        context.modify(CTX_LOG_EVENT_LAST, group);

        EList<ILogEventGroup> groups = queue.getGroups();
        groups.add(group);

        if (!settings.isConfigured()) {
            if (isAfterWelcomeTimeout()) {
                setupTimeout = in(15, MINUTES);
                notificationSupport.showConfigureSystemNotification();
            }
            context.modify(CTX_LOG_EVENT_LAST_DELAYED, group);
            return;
        }

        List<IServerDescriptor> unconfiguredServers = Lists.newLinkedList();
        for (ILogEvent event : group.getEvents()) {
            IServerDescriptor server = event.getServer();
            if (!server.isConfigured()) {
                unconfiguredServers.add(server);
            }
        }
        if (!unconfiguredServers.isEmpty()) {
            if (canShowNotification()) {
                notificationSupport.showConfigureNewServersNotification(unconfiguredServers);
            }
            // TODO this is not exactly correct but...
            context.modify(CTX_LOG_EVENT_LAST_DELAYED, group);
            return;
        }

        for (ILogEvent event : group.getEvents()) {
            checkAutomaticProcessors(event);
        }

        // TODO we may have unauthorized events in the queue!
        if (settings.getSendMode() == BACKGROUND) {
            groups.remove(group);
            broker.post(TOPIC_USER_REQUESTS_SEND_ONE_GROUP, group);
        } else {
            if (canShowNotification()) {
                context.modify(CTX_LOG_EVENT_LAST_NOTIFIED, group);
                notificationSupport.showNewEventLoggedNotification(group);
            }
        }
    }

    private void checkAutomaticProcessors(ILogEvent event) {
        for (IProcessorDescriptor descriptor : descriptors) {
            if (descriptor.isAutomatic() && Processors.shouldProcess(descriptor, event)) {
                if (descriptor.getProcessor().getWrapped() instanceof WiringErrorAnalyzerProcessor
                        && settings.isDisableAutomaticWiringAnalysis()) {
                    continue;
                }
                EList<IReportProcessor> enabledProcessors = event.getOptions().getEnabledProcessors();
                IReportProcessor processor = descriptor.getProcessor();
                if (!enabledProcessors.contains(processor)) {
                    enabledProcessors.add(processor);
                }
            }
        }
    }

    @Inject
    @Optional
    protected void onRescheduleDelayedLogEvent(@EventTopic(TOPIC_RESCHEDULE_NOTIFCATION_LAST_LOG_EVENT) Event busEvent) {
        for (ILogEventGroup group : queue.getGroups()) {
            Set<ILogEvent> remove = Sets.newLinkedHashSet();
            for (ILogEvent event : group.getEvents()) {
                // remove all events for not-yet enabled servers:
                if (!event.getServer().isActive()) {
                    remove.add(event);
                    continue;
                }
                // we need to reset the send options when rescheduling - the defaults may have been modified.
                ISendOptions options = IModelFactory.eINSTANCE.createSendOptions();
                copyServerDefaultsIfUnset(event.getServer(), options);
                copySystemDefaultsIfUnset(settings, options);
                // TODO setOptions should set the value in the context as well...
                event.setOptions(options);
                event.getContext().set(ISendOptions.class, options);
                // enabled processors may be lost, check again...
                checkAutomaticProcessors(event);
            }

            // remove from the queue
            group.getEvents().removeAll(remove);
        }

        // send mode may have changed:
        if (settings.getSendMode() == SendMode.BACKGROUND) {
            broker.post(TOPIC_USER_REQUESTS_SEND_ALL_GROUPS, queue);
            return;
        }

        ILogEventGroup delayed = (ILogEventGroup) context.get(CTX_LOG_EVENT_LAST);

        if (!canShowNotification() || delayed == null) {
            return;
        }

        context.modify(CTX_LOG_EVENT_LAST_NOTIFIED, delayed);
        notificationSupport.showNewEventLoggedNotification(delayed);
        context.remove(CTX_LOG_EVENT_LAST_DELAYED);
    }

    @Inject
    @Optional
    protected void onUserDisablesNewEndpoints(@EventTopic(TOPIC_USER_REQUESTS_DISABLE_NEW_SERVERS) Event busEvent) {
        try {
            doUserDisablesNewEndpoints();
        } catch (Exception e) {
            log(WARN_REPORTING_ERROR, e);
        }
    }

    private void doUserDisablesNewEndpoints() {
        for (IServerDescriptor server : Servers.viewOfUnconfigured(servers)) {
            server.setEnabled(false);
            server.setConfigured(true);
        }
    }

    private IProblemState getServerInterest(IServerDescriptor server, IStatus status, IEclipseContext eventContext) {
        IProblemState res;
        try {
            res = server.getConnection().interested(status, eventContext, new NullProgressMonitor());
        } catch (Exception e) {
            Logs.log(LogMessages.WARN_SERVER_FAILURE, e, server.getId(), e.getMessage());
            res = IModelFactory.eINSTANCE.createProblemState();
            res.setStatus(FAILURE);
            res.setMessage(Formats.format(Messages.NOTIFY_INTEREST_MESSAGE_ENPOINT_FAILED, e.getMessage()));
        }
        eventContext.set(IProblemState.class, res);
        return res;
    }

    private boolean isInterested(IProblemState predictedState) {
        ProblemStatus status = predictedState.getStatus();
        return !ImmutableSet.of(IGNORED, INVALID).contains(status);
    }

    private boolean canShowNotification() {
        return !(systemStateSetupInProgress || systemStateReviewInProgress || systemStateNotificationInProgress);
    }

    private boolean isAfterWelcomeTimeout() {
        return System.currentTimeMillis() > setupTimeout;
    }

    private long in(int i, TimeUnit minutes) {
        return System.currentTimeMillis() + minutes.toMillis(i);
    }

    @Inject
    @Optional
    protected void onSendLogEventGroup(@EventTopic(TOPIC_USER_REQUESTS_SEND_ONE_GROUP) ILogEventGroup logEventGroup) {
        uploadService.schedule(logEventGroup);
        queue.getGroups().remove(logEventGroup);
    }

    @Inject
    @Optional
    void onSendLogEventQueue(@EventTopic(TOPIC_USER_REQUESTS_SEND_ALL_GROUPS) Event busEvent) {
        EList<ILogEventGroup> groups = queue.getGroups();
        uploadService.schedule(queue.getGroups());
        groups.clear();
    }

    @Inject
    @Optional
    void onClearEventQueue(@EventTopic(TOPIC_USER_REQUESTS_CLEAR_QUEUE) Event busEvent) {
        try {
            doClearEventQueue();
        } catch (Exception e) {
            log(WARN_REPORTING_ERROR, e);
        }
    }

    private void doClearEventQueue() {
        EList<ILogEventGroup> events = queue.getGroups();
        ImmutableList<ILogEventGroup> copy = ImmutableList.copyOf(events);
        events.clear();
        for (ILogEventGroup group : copy) {
            for (ILogEvent logEvent : group.getEvents()) {
                IServerDescriptor server = logEvent.getServer();
                server.getConnection().discarded(logEvent.getStatus(), logEvent.getContext());
            }
        }
    }

    @Inject
    @Optional
    protected void onServerResponsesCompleted(@EventTopic(TOPIC_NEW_SERVER_RESPONSES) ILogEventGroup group) {
        if (canShowNotification()) {
            notificationSupport.showServerResponsesNotification(group);
        }
    }

    @Inject
    protected void setConfigureInProgress(@Named(SetupWizard.CTX_STATE_SETUP_IN_PROGRESS) boolean newState) {
        systemStateSetupInProgress = newState;
    }

    @Inject
    protected void setReviewInProgress(@Named(ReviewDialog.CTX_STATE_REVIEW_IN_PROGRESS) boolean newState) {
        systemStateReviewInProgress = newState;
    }

    @Inject
    protected void setNotificationInProgress(@Named(MylynNotificationsSupport.CTX_STATE_NOTIFICATION_IN_PROGRESS) boolean newState) {
        systemStateSetupInProgress = newState;
    }
}
