/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.di;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.ImmutableSet.of;
import static org.eclipse.epp.internal.logging.aeri.ide.IIdePackage.Literals.*;
import static org.eclipse.epp.internal.logging.aeri.ide.l10n.LogMessages.WARN_EXTENSION_FAILED;
import static org.eclipse.epp.internal.logging.aeri.ide.utils.Servers.sort;
import static org.eclipse.epp.logging.aeri.core.IModelPackage.Literals.*;
import static org.eclipse.epp.logging.aeri.core.util.Logs.log;

import java.io.File;
import java.util.List;
import java.util.Set;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.e4.core.contexts.ContextFunction;
import org.eclipse.e4.core.contexts.IEclipseContext;
import org.eclipse.emf.common.util.EMap;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.epp.internal.logging.aeri.ide.IIdeFactory;
import org.eclipse.epp.internal.logging.aeri.ide.IServerDescriptor;
import org.eclipse.epp.internal.logging.aeri.ide.handlers.CreateServerConnectionHandler;
import org.eclipse.epp.logging.aeri.core.ILink;
import org.eclipse.epp.logging.aeri.core.IModelFactory;
import org.eclipse.epp.logging.aeri.core.IServerConnection;
import org.eclipse.epp.logging.aeri.core.SystemControl;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableList.Builder;

/**
 * Reads the list of available servers from the servers extension point.
 */
public class ExtensionPointServersCreationFunction extends ContextFunction {
    // values for extension point
    public static final String EXT_SERVERS_ID = "org.eclipse.epp.logging.aeri.ide.servers"; //$NON-NLS-1$

    @Override
    public Object compute(IEclipseContext notused, String contextKey) {
        contextKey = checkNotNull(contextKey);
        IEclipseContext systemContext = SystemControl.getSystemContext();

        IExtensionRegistry registry = systemContext.get(IExtensionRegistry.class);
        IConfigurationElement[] elements = registry.getConfigurationElementsFor(EXT_SERVERS_ID);

        Builder<IServerDescriptor> b = ImmutableList.builder();
        for (final IConfigurationElement element : elements) {
            IServerDescriptor server = IIdeFactory.eINSTANCE.createServerDescriptor();
            server.setConfigurationElement(element);
            server.setClazz(element.getAttribute("class")); //$NON-NLS-1$

            String contributor = element.getContributor().getName();
            server.setContributor(contributor);

            copyFromRegistry(element, server, of(SERVER_DESCRIPTOR__ID, SERVER_DESCRIPTOR__NAME, SERVER_DESCRIPTOR__DESCRIPTION,
                    SERVER_DESCRIPTOR__ICON16, SERVER_DESCRIPTOR__ICON32));
            readLinks(server, element);
            try {
                IEclipseContext childContext = systemContext.createChild("Server Context of '" + server.getId() + "'"); //$NON-NLS-1$ //$NON-NLS-2$
                childContext.set(IServerDescriptor.class, server);
                childContext.set(File.class.getName(), new ServerStateLocationCreationFunction());
                IServerConnection object = new CreateServerConnectionHandler().execute(server, childContext);
                server.setConnection(object);
            } catch (Exception e) {
                log(WARN_EXTENSION_FAILED, e, server.getId(), e.getMessage());
            }
            b.add(server);
        }
        List<IServerDescriptor> servers = sort(b.build());
        systemContext.set(contextKey, servers);
        return servers;
    }

    private void readLinks(IServerDescriptor server, IConfigurationElement element) {
        EMap<String, ILink> links = server.getLinks();
        IConfigurationElement[] children = element.getChildren("link"); //$NON-NLS-1$
        for (IConfigurationElement child : children) {
            ILink link = IModelFactory.eINSTANCE.createLink();
            copyFromRegistry(child, link, of(LINK__REL, LINK__HREF, LINK__TITLE));
            links.put(link.getRel(), link);
        }
    }

    private void copyFromRegistry(IConfigurationElement child, EObject link, Set<EAttribute> attrs) {
        for (EAttribute attr : attrs) {
            String value = child.getAttribute(attr.getName());
            link.eSet(attr, value);
        }
    }
}
