/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.server;

import static com.google.common.base.Charsets.UTF_8;
import static org.apache.commons.lang3.StringUtils.stripToEmpty;
import static org.eclipse.epp.logging.aeri.core.util.Reports.visit;

import org.eclipse.epp.logging.aeri.core.IBundle;
import org.eclipse.epp.logging.aeri.core.IReport;
import org.eclipse.epp.logging.aeri.core.IStackTraceElement;
import org.eclipse.epp.logging.aeri.core.IStatus;
import org.eclipse.epp.logging.aeri.core.IThrowable;
import org.eclipse.epp.logging.aeri.core.util.ModelSwitch;
import org.eclipse.epp.logging.aeri.core.util.Reports;

import com.google.common.hash.Hasher;

public class ReportFingerprints {

    public static String exactIdentityHash(IReport report) {
        final Hasher hasher = Reports.newHasher();
        ModelSwitch<Hasher> s = new ModelSwitch<Hasher>() {
            @Override
            public Hasher caseReport(IReport object) {
                hasher.putString(stripToEmpty(object.getEclipseProduct()), UTF_8);
                hasher.putString(stripToEmpty(object.getEclipseBuildId()), UTF_8);
                hasher.putString(stripToEmpty(object.getJavaRuntimeVersion()), UTF_8);
                hasher.putString(stripToEmpty(object.getOsgiOs()), UTF_8);
                hasher.putString(stripToEmpty(object.getOsgiOsVersion()), UTF_8);
                hasher.putString(stripToEmpty(object.getOsgiArch()), UTF_8);
                hasher.putString(stripToEmpty(object.getOsgiWs()), UTF_8);
                return null;
            };

            @Override
            public Hasher caseStatus(IStatus object) {
                hasher.putString(stripToEmpty(object.getPluginId()), UTF_8);
                hasher.putString(stripToEmpty(object.getPluginVersion()), UTF_8);
                hasher.putString(stripToEmpty(object.getMessage()), UTF_8);
                hasher.putInt(object.getSeverity());
                hasher.putInt(object.getCode());
                return null;
            }

            @Override
            public Hasher caseBundle(IBundle object) {
                hasher.putString(stripToEmpty(object.getName()), UTF_8);
                hasher.putString(stripToEmpty(object.getVersion()), UTF_8);
                return null;
            }

            @Override
            public Hasher caseStackTraceElement(IStackTraceElement object) {
                hasher.putString(stripToEmpty(object.getClassName()), UTF_8);
                hasher.putString(stripToEmpty(object.getMethodName()), UTF_8);
                hasher.putInt(object.getLineNumber());
                return null;
            }

            @Override
            public Hasher caseThrowable(IThrowable object) {
                hasher.putString(stripToEmpty(object.getClassName()), UTF_8);
                hasher.putString(stripToEmpty(object.getMessage()), UTF_8);
                return null;
            }
        };

        visit(report, s);
        String hash = hasher.hash().toString();
        return hash;
    }

    public static String traceIdentityHash(IReport report) {
        final Hasher hasher = Reports.newHasher();
        visit(report, new ModelSwitch<Hasher>() {

            @Override
            public Hasher caseStackTraceElement(IStackTraceElement element) {
                hasher.putString(element.getClassName(), UTF_8);
                hasher.putString(element.getMethodName(), UTF_8);
                hasher.putInt(element.getLineNumber());
                return null;
            }
        });
        String hash = hasher.hash().toString();
        return hash;
    }

}
